/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#define	PRODUCT_SHORT_NAME		"mwmail.nlm"
#define CREA_SENT   "CREA Sent\r\n"
#define CHECK_SENT  "CHECK Sent\r\n"
#define NMLOGID_H_NEED_LOGGING_KEY
#define NMLOGID_H_NEED_LOGGING_CERT
#include <xpl.h>

#include <mdb.h>
#include <msgapi.h>
#include <nmap.h>
#include <libical.h>
#include <modweb.h>		/* APIs */
#include <mwtempl.h>		/* Token definition */


MWAPIDefine;

#include "mwmail.h"

/* Globals */
BOOL MwMailUnloadOK = TRUE;
MwMailGlobal MwMail;

/*
	These are the functions this module implements
*/
BOOL
MWMAILInitSession(SessionStruct *Session, void **ModuleData)
{
	MailSessionStruct	*MailSession;
	MDBValueStruct		*V;
	long				i;

	ModDebug("MWMAILInitSession() called\n");

	MailSession=MemMalloc(sizeof(MailSessionStruct));

	if (!MailSession) {
		XplConsolePrintf("NWFOLDER Module out of memory!\n");
		return(FALSE);
	}

	memset(MailSession, 0, sizeof(MailSessionStruct));
	
	/* Prepare */
	V = MDBCreateValueStruct(MwMail.DirectoryHandle, NULL);

	/* This is tricky - we share the struct so we don't get a DS context, etc.., but we only use it for storage, thus the original can go away before our shared copy */
	MailSession->Value=MDBShareContext(V);

	/* Maximum Number of recipients */
	if (MsgGetUserFeature(Session->UserDN, FEATURE_MODWEB, MSGSRV_A_RECIPIENT_LIMIT, V)) {
		MailSession->MaxRecipients = atol(V->Value[0]);
	} else {
		MailSession->MaxRecipients = MwMail.MaxRecipients;
	}
	MDBFreeValues(V);

	if (MsgGetUserFeature(Session->UserDN, FEATURE_MODWEB, MSGSRV_A_MESSAGE_LIMIT, V)) {
		MailSession->MaxSize = atol(V->Value[0])*1024*1024;
	} else {
		MailSession->MaxSize = MwMail.MaxMessageSize;
	}
	MDBFreeValues(V);

	/* Preferences */
	if (MsgGetUserFeature(Session->UserDN, FEATURE_MODWEB, MSGSRV_A_PREFERENCES, V)) {
	    unsigned long index;
		for (index = 0; index < V->Used; index++) {
			if (MWQuickNCmp(V->Value[index], "Webmail:Options:", 16)) {
				i = atol(V->Value[index] + 16);
				MailSession->PurgeDefault		= (i & PREFS_PURGE) >> PREFS_PURGE_SHIFT;
			} else if (MWQuickNCmp(V->Value[index], "Webmail:SentFolderName:", 23)) {
				if (V->Value[index] + 23 != '\0') {
					MailSession->SentFolder = MemStrdup(V->Value[index] + 23);
					if (MailSession->SentFolder) {
						unsigned char	*ptr=MailSession->SentFolder;

						while ((ptr=strchr(ptr, ' '))!=NULL) {
							*ptr=0x7f;
						}
					} else {
						LoggerEvent(MwMail.LogHandle, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_OUT_OF_MEMORY, LOG_ERROR, 0, __FILE__, NULL, (strlen(V->Value[index] + 23)), __LINE__, NULL, 0);
						//syslog(LOG_WARNING, "Out of memory, loading sent folder failed");
					}
				}
			}
		}
		MDBFreeValues(V);
	}

	if(MailSession->SentFolder != NULL) {
	    ;
	} else {
	    unsigned long retVal;    
	    unsigned char *buffer;

	    /* Sets the folder to store sent mail in the Sent folder */
	    MDBAddValue("Webmail:SentFolderName:Sent", V);
	    MDBWrite(Session->UserDN, MSGSRV_A_PREFERENCES, V);
	    MDBFreeValues(V);

	    /* Create the Sent folder if it doesn't exist */
	    buffer = MemMalloc(BUFSIZE);
	    if (buffer) {
		retVal = MWSendNMAPServer(Session, CHECK_SENT, strlen(CHECK_SENT));
		if(retVal) {
		    retVal = MWGetNMAPAnswer(Session, buffer, BUFSIZE, TRUE);
		    if(retVal == 4224) {
			retVal = MWSendNMAPServer(Session, CREA_SENT, strlen(CREA_SENT));
			if(retVal) {
			    retVal = MWGetNMAPAnswer(Session, buffer, BUFSIZE, TRUE);
			    if (retVal != 1000) {
				MailSession->Error = ERROR_CREATING_FOLDER__CREA;
			    }
			}
		    }
		}
		MemFree(buffer);
	    }
	}
   
	/* Address Book Search Scope */
	/* We haven't added an attribute in DS for this, but it is ready if we want to */
	MailSession->SearchScope = SEARCH_PERSONAL | SEARCH_SYSTEM;

	MDBDestroyValueStruct(V);

	*ModuleData=(void *)MailSession;
	MwMailComposeCleanUp(Session, MailSession);

	return(TRUE);
}


BOOL
MWMAILDestroySession(SessionStruct *Session, void *ModuleData)
{
	MailSessionStruct	*MailSession=(MailSessionStruct	*)ModuleData;

	ModDebug("MWMAILDestroySession() called (ModuleData:%x)\n", (unsigned int)ModuleData);

	if (MailSession) {
		MwMailComposeCleanUp(Session, MailSession);
		MwMailFreeMessageSessionData(Session, MailSession);
		MwMailClearMIMECache(MailSession);

		if (MailSession->SentFolder) {
			MemFree(MailSession->SentFolder);
		}

		if (MailSession->AttachPath) {
			MemFree(MailSession->AttachPath);
		}

		if (MailSession->SearchString) {
			MemFree(MailSession->SearchString);
		}		

		if (MailSession->MIMEParamStream) {
			MWReleaseStreams(MailSession->MIMEParamStream);
		}

		MDBDestroyValueStruct(MailSession->Value);

		MemFree(MailSession);
	}

	return(TRUE);
}


BOOL
MWMAILHandleTemplate(ConnectionStruct *Client, SessionStruct *Session, unsigned long Page, TokenOverlayStruct *Token, unsigned long *GotoToken, void *ModuleData)
{
	MailSessionStruct	*MailSession=(MailSessionStruct	*)ModuleData;
	unsigned char		URL[256];
	unsigned char		Answer[BUFSIZE+1];
	int					ReplyInt = FALSE;
	unsigned char		*ptr;

	//ModDebug("MWMAILHandleTemplate() called (ModuleData:%x)\n", ModuleData);

	/* This will just return if the folder is already selected */
	MwMailSelectFolder(Session->CurrentFolder, Session, MailSession);

	/* Is there a dirty message that we need to take care of? */
	if (Session->DirtyMessage && (Session->DirtyMessageFolder < Session->FolderList->Used) && (Session->FolderList->Value[Session->DirtyMessageFolder][0] == 'M')) {
		/*
			Yup, there is a message to be deleted, and it is indeed a mail message
		*/

		if (MwMailSelectFolder(Session->DirtyMessageFolder, Session, MailSession)) {
			MwMailSetMessageFlags(Client, Session, MailSession, UIDtoID(Session, MailSession, Session->DirtyMessageID), FLAG_DELETE);
		}

		Session->DirtyMessage = FALSE;
	}

	switch(Token->TokenID) {
		/******************************
				General Commands
		*******************************/
		case T_ACTIVEBG: {
			if (MailSession->CurrentColor) {
				MWSendClient(Client, MailSession->CurrentColor, MailSession->CurrentColorLen);
			}
			return(TRUE);
		}

	    case T_MAIL_CONDITION: {
		BOOL condition;

		if (Token->ArgumentOffsetOrID[0] == AA_ENDIF) {
		    return(TRUE);
		}

		switch (Token->ArgumentOffsetOrID[1]) {		 
		    case AA_HASPREVPAGE: {
			if (!MailSession->MailboxSelected || Session->NumOfMessages == 0) {
			    *GotoToken=T_MAIL_CONDITION;
			    return(TOKEN_MOVE_FORWARD);
			}

			condition = (MailSession->CurrentMsgListMessage < Session->NumOfMessages);
			break;

		    }

		    case AA_HASNEXTPAGE: {
			if (!MailSession->MailboxSelected || Session->NumOfMessages == 0) {
			    *GotoToken=T_MSGLISTNEXTEND;
			    return(TOKEN_MOVE_FORWARD);
			}

			condition = (MailSession->CurrentMsgListMessage > Session->MessagesPerPage);
			break;
		    }

		    case AA_HASPREVMESSAGE: {
			if (MailSession->CurrentMsgViewMessage < Session->NumOfMessages) {
			    condition = TRUE;
			    break;
			}
			
			condition = FALSE;
			break;
		    }

		    case AA_HASNEXTMESSAGE: {
			if (MailSession->CurrentMsgViewMessage > 1) {
			    condition = TRUE;
			    break;
			}
			
			condition = FALSE;
			break;
		    }

		    default: {
			*GotoToken=T_MAIL_CONDITION;
			return(TOKEN_MOVE_FORWARD);
		    }
		}

		if (((Token->ArgumentOffsetOrID[0] == AA_IF) && !condition) || ((Token->ArgumentOffsetOrID[0] == AA_ELSE) && condition)) {
		    *GotoToken=T_MAIL_CONDITION;
		    return(TOKEN_MOVE_FORWARD);
		}

		return(TRUE);			
	    }

		case T_STORAGE: {
			switch(Token->ArgumentOffsetOrID[0]) {
				case AA_MB: {
					unsigned long	Used=0, Total=0;

					MWSendNMAPServer(Session, "SPACE\r\n", 7);
					ReplyInt=MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);
					if (ReplyInt==1000) {
						sscanf(Answer, "%lu %lu", &Used, &Total);
						ReplyInt=snprintf(Answer, sizeof(Answer), "%d.%1d", (int)Used/1000,(int)(Used % 1000)/100);
						MWSendClient(Client, Answer, ReplyInt);
					} else {
						MWSendClient(Client, "0", 1);
					}
					return(TRUE);
				}

				case AA_PERCENT: {
					unsigned long	Used=0, Total=0, Percent=0;

					MWSendNMAPServer(Session, "SPACE\r\n", 7);
					ReplyInt=MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);
					if (ReplyInt==1000) {
						sscanf(Answer, "%lu %lu", &Used, &Total);
						if (Total!=0) {
							if (Used>Total) {
								Percent=100;
							} else {
								Percent=(Used*100)/Total;
								if (!Percent) {
									Percent=1;                    /* Avoid reporting 0% */
								}
							}
						} else {
							Percent=0;
						}
						if (Used<101) {
							Used=101;                       /* Avoid reporting zero size */
						}

						ReplyInt=snprintf(Answer, sizeof(Answer), "%d", (int)Percent);
						MWSendClient(Client, Answer, ReplyInt);
					} else {
						MWSendClient(Client, "0", 1);
					}
					return(TRUE);
				}
			}
			return(TRUE);
		}


		/******************************
				Folderlist Components
		*******************************/
		case T_FOLDERLISTSTART: {
			if (Session->FolderList->Used == 0) {
				*GotoToken=T_FOLDERLISTEND;
				return(TOKEN_MOVE_FORWARD);
			}

			if (MailSession->InFolderList == 0) {
				MailSession->InFolderList = 1;
			}

			if ((MailSession->InFolderList % 2) == 0) {
				MailSession->CurrentColor = Token->Data+Token->ArgumentOffsetOrID[0];
				MailSession->CurrentColorLen = Token->ArgumentSize[0];
			} else {
				MailSession->CurrentColor = Token->Data+Token->ArgumentOffsetOrID[1];
				MailSession->CurrentColorLen = Token->ArgumentSize[1];
			}
			return(TRUE);
		}

		case T_FOLDERLISTEND: {
			do {
				MailSession->InFolderList++;
			} while (MailSession->InFolderList <= Session->FolderList->Used && Session->FolderList->Value[MailSession->InFolderList - 1][0] != 'M');
			
			if (MailSession->InFolderList <= Session->FolderList->Used) {
				*GotoToken = T_FOLDERLISTSTART;
				return(TOKEN_MOVE_BACKWARD);
			} else {
				MailSession->InFolderList = 0;
				return(TRUE);
			}
		}

		case T_FOLDERLISTREFRESH: {	/* Refresh the folder list */
			if (Token->ArgumentOffsetOrID[0] != 0) {
				MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_REFRESH_FOLDERLIST, Token->ArgumentOffsetOrID[0], 0, 0, 0);
				MWSendClient(Client, URL, strlen(URL));
			} else {
				Session->FolderListIsDirty = TRUE;
			}
			return(TRUE);
		}

		case T_MSGLIST: {	/* Create URL to display currently displayed folder or calendar */
			if (MailSession->InFolderList <= Session->FolderList->Used) {
				switch (Session->FolderList->Value[MailSession->InFolderList - 1][0]) {
					case 'M': {	/* Mailbox */
						if (MailSession->SentFolder && MWQuickCmp(MailSession->SentFolder, Session->FolderList->Value[MailSession->InFolderList - 1] + 3)) {
							MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MSGLIST, MailSession->InFolderList - 1, Token->ArgumentOffsetOrID[1], 0, 0);
						} else {
							MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MSGLIST, MailSession->InFolderList - 1, Token->ArgumentOffsetOrID[0], 0, 0);
						}
						break;
					}

					case 'C': {	/* Calendar */
						MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_DISPLAY_CALENDAR, MailSession->InFolderList - 1, Token->ArgumentOffsetOrID[2], 0, 0);
						break;
					}

					default: { /* Anything else.  This should never happen, but in case the user mucks with a URL, just display the current folder */
						MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MSGLIST, MailSession->CurrentFolder, Token->ArgumentOffsetOrID[1], 0, 0);
					}
				}
				MWSendClient(Client, URL, strlen(URL));
				return(TRUE);
			}	else {
				return(FALSE);
			}
		}

		case T_SELECTFOLDER: {	/* Create URL to display named folder */
			unsigned long	i;

			for (i = 0; i < Session->FolderList->Used; i++) {
				if (MWQuickNCmp(Token->Data + Token->ArgumentOffsetOrID[0], Session->FolderDisplayNames->Value[i], Token->ArgumentSize[0])) {
					if (MailSession->SentFolder && MWQuickCmp(MailSession->SentFolder, Session->FolderList->Value[i] + 3)) {
						MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MSGLIST, i, Token->ArgumentOffsetOrID[1], 0, 0);
					} else {
						MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MSGLIST, i, Token->ArgumentOffsetOrID[0], 0, 0);
					}
					MWSendClient(Client, URL, strlen(URL));
					return(TRUE);
				}
			}
			return(FALSE);
		}

		case T_CURRENTFOLDER_INDENT: {
			if (MailSession->InFolderList > 0 && MailSession->InFolderList <= Session->FolderList->Used) {
				ptr = Session->FolderList->Value[MailSession->InFolderList - 1] + 3;

				do {
					ptr = strchr(++ptr, '/');
					if (ptr) {
						MWSendClient(Client, "&nbsp;", 6);
					}
				} while (ptr);
			}
			return(TRUE);
		}

		case T_CURRENTFOLDER_TYPE: {
			if (MailSession->InFolderList > 0 && MailSession->InFolderList <= Session->FolderList->Used) {
				switch (Session->FolderList->Value[MailSession->InFolderList - 1][0]) {
					case 'M': {	/* MailBox */
						MWSendClient(Client, Token->Data+Token->ArgumentOffsetOrID[0], Token->ArgumentSize[0]);
						break;
					}
					case 'C': {	/* Calendar */
						MWSendClient(Client, Token->Data+Token->ArgumentOffsetOrID[0], Token->ArgumentSize[1]);
						break;
					}
					case 'F': {	/* Folder */
						MWSendClient(Client, Token->Data+Token->ArgumentOffsetOrID[0], Token->ArgumentSize[2]);
						break;
					}
				}
			}
			return(TRUE);
		}

		case T_CURRENTFOLDER_IMAGE: {
			if (MailSession->InFolderList > 0 && MailSession->InFolderList <= Session->FolderList->Used) {
				switch (Session->FolderList->Value[MailSession->InFolderList - 1][0]) {
					case 'M': {	/* MailBox */
						MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[0], 0);
						break;
					}
					case 'C': {	/* Calendar */
						MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[1], 0);
						break;
					}
					case 'F': {	/* Folder */
						MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[2], 0);
						break;
					}
				}
			}
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_CURRENTFOLDER_INFO: {
			if (MailSession->InFolderList > 0 && MailSession->InFolderList <= Session->FolderList->Used) {
				switch (Token->ArgumentOffsetOrID[0]) {
					case AA_FOLDERID: {
						ReplyInt = snprintf(Answer, sizeof(Answer), "%lu", MailSession->InFolderList);
						MWSendClient(Client, Answer, ReplyInt);
						return(TRUE);
					}		

					case AA_FOLDERNAME: {
						ptr = Session->FolderDisplayNames->Value[MailSession->InFolderList - 1];
						while (*ptr && isspace(*ptr)) {
							ptr++;
						}
						MWSendClient(Client, ptr, strlen(ptr));
						return(TRUE);
					}		

					case AA_FULLFOLDERNAME: {
						MwMailFolderUTF7toUTF8(Session->FolderList->Value[MailSession->InFolderList - 1] + 3, Answer);

						MWSendClient(Client, Answer, strlen(Answer));
						return(TRUE);
					}		

					case AA_FOLDERINDENT: {
						ptr = Session->FolderDisplayNames->Value[MailSession->InFolderList - 1];
						while (*ptr && isspace(*ptr)) {
							MWSendClient(Client, "&nbsp;&nbsp;&nbsp;", 18);
							ptr++;
						}
						return(TRUE);
					}		

					case AA_FOLDERPREVNESTINGCOUNT: {
						unsigned long	IndentLevel;
						unsigned long	ParentLevel;
						unsigned long	i;

						/* Find current indentation level; then count back to the first item with less indentation */
						if (MailSession->InFolderList > 1) {
							IndentLevel = 1;
							ptr = Session->FolderDisplayNames->Value[MailSession->InFolderList - 1];
							while (*ptr && isspace(*ptr)) {
								IndentLevel++;
								ptr++;
							}

							ParentLevel = 0;
							for (i = MailSession->InFolderList - 2; i >= 0; i--) {
							    unsigned long index;
								index = 1;
								ptr = Session->FolderDisplayNames->Value[i];
								while (*ptr && isspace(*ptr)) {
									index++;
									ptr++;
								}
								if (index < IndentLevel) {
									ParentLevel = index;
									break;
								}
							}

							ReplyInt = snprintf(Answer, sizeof(Answer), "%d", (int)ParentLevel);
							MWSendClient(Client, Answer, ReplyInt);
						} else {
							MWSendClient(Client, "0", 1);
						}
						return(TRUE);
					}		

					case AA_FOLDERNESTINGCOUNT: {
						ReplyInt = 1;
						ptr = Session->FolderDisplayNames->Value[MailSession->InFolderList - 1];
						while (*ptr && isspace(*ptr)) {
							ReplyInt++;
							ptr++;
						}
						ReplyInt = snprintf(Answer, sizeof(Answer), "%d", ReplyInt);
						MWSendClient(Client, Answer, ReplyInt);
						return(TRUE);
					}		

					case AA_MESSAGESTOTAL: {
						if (MailSession->InFolderList <= Session->FolderList->Used) {
							ReplyInt = snprintf(Answer, sizeof(Answer), "%lu", Session->FolderTotalCount[MailSession->InFolderList - 1]);
							MWSendClient(Client, Answer, ReplyInt);
						}
						return(TRUE);
					}		

					case AA_MESSAGESUNREAD: {
						if (MailSession->InFolderList <= Session->FolderList->Used) {
							ReplyInt = snprintf(Answer, sizeof(Answer), "%lu", Session->FolderUnreadCount[MailSession->InFolderList - 1]);
							MWSendClient(Client, Answer, ReplyInt);
						}
						return(TRUE);
					}		

					case AA_MESSAGESNEW: {
						return(TRUE);
					}
				}
			}
			return(TRUE);
		}

		case T_CURRENTFOLDER_SKIP: {
			if (MailSession->InFolderList > 0 && MailSession->InFolderList <= Session->FolderList->Used) {
				switch (Token->ArgumentOffsetOrID[0]) {
					case AA_MESSAGESTOTAL: {
						if (MailSession->InFolderList <= Session->FolderList->Used && Session->FolderTotalCount[MailSession->InFolderList - 1] == 0) {
							*GotoToken=T_CURRENTFOLDER_SKIP_END;
							return(TOKEN_MOVE_FORWARD);
						}
						return(0);
					}

					case AA_MESSAGESUNREAD: {
						if (MailSession->InFolderList <= Session->FolderList->Used && Session->FolderUnreadCount[MailSession->InFolderList - 1] == 0) {
							*GotoToken=T_CURRENTFOLDER_SKIP_END;
							return(TOKEN_MOVE_FORWARD);
						}
						return(0);
					}

					case AA_MESSAGESNEW: {
						*GotoToken=T_CURRENTFOLDER_SKIP_END;
						return(TOKEN_MOVE_FORWARD);
						return(0);
					}
				}
			}
			return(TRUE);
		}

		/*********************************
			Folder management commands
		**********************************/

		case T_FOLDERLISTFORM: {
			unsigned long	i;

			MWSendClient(Client, "<SELECT NAME=\"FolderName\">", 26);
			ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION VALUE=\"%d\">[%s]", 0, Session->User);
			MWSendClient(Client, Answer, ReplyInt);
			for (i = 1; i <= Session->FolderList->Used; i++) {
				ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION VALUE=\"%d\">&nbsp;&nbsp;", (int)i);
				MWSendClient(Client, Answer, ReplyInt);
				ptr=Session->FolderDisplayNames->Value[i-1];
				while (*ptr && isspace(*ptr)) {
					MWSendClient(Client, "&nbsp;", 6);
					ptr++;
				}
				MWSendClient(Client, Session->FolderDisplayNames->Value[i-1], strlen(Session->FolderDisplayNames->Value[i-1]));
			}
			MWSendClient(Client, "</SELECT>", 9);
			return(TRUE);
		}

		case T_FOLDER_MANAGE: {
			MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_FOLDER_MANAGE, Token->ArgumentOffsetOrID[1], Token->ArgumentOffsetOrID[0], 0, 0);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_FORM_MOVEMSG_FOLDER: {
			MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_FOLDER_MOVEMSG, Token->ArgumentOffsetOrID[1], Token->ArgumentOffsetOrID[0], MailSession->CurrentFolder, 0);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		/******************************
				MessageList Components
		*******************************/

		case T_MSGLISTSTART: {
			if (!MailSession->InMsgList) {
				if (!MailSession->MailboxSelected) {
					MwMailSelectFolder(Session->CurrentFolder, Session, MailSession);

					MailSession->FirstMessageOnPage = MailSession->CurrentMsgListMessage = Session->NumOfMessages;

					if (MailSession->FirstMessageOnPage > Session->MessagesPerPage) {
						MailSession->LastMessageOnPage = (MailSession->FirstMessageOnPage - Session->MessagesPerPage);
					} else {
						MailSession->LastMessageOnPage = 0;
					}

				}

				if (!MailSession->MailboxSelected || Session->NumOfMessages==0) {
					*GotoToken=T_MSGLISTEND;
					return(TOKEN_MOVE_FORWARD);
				}
				MailSession->InMsgList=TRUE;
				if (MailSession->CachedHeader) {
					MWReleaseStreamData(MailSession->CachedHeader);
					MailSession->CachedHeader=NULL;
				}
				if (Token->ArgumentOffsetOrID[2]>0) {
					MailSession->LastMessageOnPage=Session->NumOfMessages-Token->ArgumentOffsetOrID[2];
				}
			}

			if (!MwMailLoadMessage(Client, MailSession->CurrentMsgListMessage, Session, MailSession)) {
				*GotoToken=T_MSGLISTEND;
				return(TOKEN_MOVE_FORWARD);
			}

			if ((MailSession->CurrentMsgListMessage % 2)==0) {
				MailSession->CurrentColor=Token->Data+Token->ArgumentOffsetOrID[0];
				MailSession->CurrentColorLen=Token->ArgumentSize[0];
			} else {
				MailSession->CurrentColor=Token->Data+Token->ArgumentOffsetOrID[1];
				MailSession->CurrentColorLen=Token->ArgumentSize[1];
			}
			return(TRUE);
		}

		case T_MSGLISTEND: {
			if (MailSession->InMsgList && MailSession->CurrentMsgListMessage>(MailSession->LastMessageOnPage+1)) {
				*GotoToken=T_MSGLISTSTART;
				MailSession->CurrentMsgListMessage--;
				return(TOKEN_MOVE_BACKWARD);
			} else {
				if (MailSession->MailboxSelected) {
					MailSession->InMsgList=FALSE;

					/* reset to first message on page */
					MailSession->CurrentMsgListMessage=MailSession->FirstMessageOnPage;

				}
			}
			return(TRUE);
		}

		case T_CURMSG_LIST_ID: {
			if (MailSession->CachedHeader) {
				ReplyInt=snprintf(Answer, sizeof(Answer), "%lu", Session->NumOfMessages-MailSession->CurrentMsgListMessage+1);
				MWSendClient(Client, Answer, ReplyInt);
			}
			return(TRUE);
		}

		case T_CURMSG_SIZE: {
			if (MailSession->CachedHeader) {
				ReplyInt=MailSession->CachedMessageHeaderSize+MailSession->CachedMessageBodySize;
				ReplyInt=MwMailPrintFormatted(((ReplyInt+1023)/1024)*1024, Answer, sizeof(Answer));
				MWSendClient(Client, Answer, ReplyInt);
			}
			return(TRUE);
		}

		case T_CURMSG_STATE_STRING: {
			if (MailSession->CachedHeader) {
				if (MailSession->CachedMessageState & MSG_STATE_DELETED) {
					MWSendClient(Client, Session->Strings[Token->ArgumentOffsetOrID[1]], strlen(Session->Strings[Token->ArgumentOffsetOrID[1]]));
				} else if (!(MailSession->CachedMessageState & MSG_STATE_READ)) {
					MWSendClient(Client, Session->Strings[Token->ArgumentOffsetOrID[0]], strlen(Session->Strings[Token->ArgumentOffsetOrID[0]]));
				} else {
					MWSendClient(Client, Session->Strings[Token->ArgumentOffsetOrID[2]], strlen(Session->Strings[Token->ArgumentOffsetOrID[2]]));
				}
			}
			return(TRUE);
		}

		case T_CURMSG_STATE_TEXT: {
			if (MailSession->CachedHeader) {
				if (MailSession->CachedMessageState & MSG_STATE_DELETED) {
					ptr=Token->Data+Token->ArgumentOffsetOrID[1];
					ReplyInt=Token->ArgumentSize[1];
				} else if (!(MailSession->CachedMessageState & MSG_STATE_READ)) {
					ptr=Token->Data+Token->ArgumentOffsetOrID[0];
					ReplyInt=Token->ArgumentSize[0];
				} else {
					ptr=Token->Data+Token->ArgumentOffsetOrID[2];
					ReplyInt=Token->ArgumentSize[2];
				}
				MWSendClient(Client, ptr, ReplyInt);
			}
			return(TRUE);
		}

		case T_CURMSG_FLAG_TEXT: {
			if (MailSession->CachedHeader) {
				switch(Token->ArgumentOffsetOrID[0]) {
					case AA_READ: {
						if (MailSession->CachedMessageState & MSG_STATE_READ) {
							MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
						} else {
							MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
						}

						break;
					}

					case AA_READHIGHPRIORITY: {
						if ((MailSession->CachedMessageState & MSG_STATE_PRIOHIGH)
							&& (MailSession->CachedMessageState & MSG_STATE_READ)) {

							MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
						} else {
							MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
						}

						break;
					}

					case AA_READLOWPRIORITY: {
						if ((MailSession->CachedMessageState & MSG_STATE_PRIOLOW)
							&& (MailSession->CachedMessageState & MSG_STATE_READ)) {

							MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
						} else {
							MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
						}

						break;
					}

					case AA_UNREAD: {
						if (MailSession->CachedMessageState & MSG_STATE_READ) {
							MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
						} else {
							MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
						}

						break;
					}

					case AA_UNREADHIGHPRIORITY: {
						if ((MailSession->CachedMessageState & MSG_STATE_PRIOHIGH)
							&& !(MailSession->CachedMessageState & MSG_STATE_READ)) {

							MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
						} else {
							MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
						}

						break;
					}

					case AA_UNREADLOWPRIORITY: {
						if ((MailSession->CachedMessageState & MSG_STATE_PRIOLOW)
							&& !(MailSession->CachedMessageState & MSG_STATE_READ)) {

							MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
						} else {
							MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
						}

						break;
					}

					case AA_DELETED: {
						if (MailSession->CachedMessageState & MSG_STATE_DELETED) {
							MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
						} else {
							MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
						}

						break;
					}

					case AA_PRIVATE: {
						if (MailSession->CachedMessageState & MSG_STATE_PRIVATE) {
							MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
						} else {
							MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
						}

						break;
					}

					case AA_RECENT: {
						if (MailSession->CachedMessageState & MSG_STATE_RECENT) {
							MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
						} else {
							MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
						}

						break;
					}

					case AA_HIGHPRIORITY: {
						if (MailSession->CachedMessageState & MSG_STATE_PRIOHIGH) {
							MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
						} else {
							MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
						}

						break;
					}

					case AA_LOWPRIORITY: {
						if (MailSession->CachedMessageState & MSG_STATE_PRIOLOW) {
							MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
						} else {
							MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
						}

						break;
					}

					case AA_NORMALPRIORITY: {
						if (!(MailSession->CachedMessageState & MSG_STATE_PRIOHIGH)
							&& !(MailSession->CachedMessageState & MSG_STATE_PRIOLOW)) {

							MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
						} else {
							MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
						}

						break;
					}

					default: {
						MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);

						break;
					}
				}
			}

			return(TRUE);
		}

		case T_MSG_SET_ICON: {
			switch (Token->ArgumentOffsetOrID[0]) {
				case AA_READ: {
					MailSession->ReadImage = Token->ArgumentOffsetOrID[1];
					break;
				}
				case AA_READHIGHPRIORITY: {
					MailSession->ReadHighImage = Token->ArgumentOffsetOrID[1];
					break;
				}
				case AA_READLOWPRIORITY: {
					MailSession->ReadLowImage = Token->ArgumentOffsetOrID[1];
					break;
				}
				case AA_UNREAD: {
					MailSession->UnreadImage = Token->ArgumentOffsetOrID[1];
					break;
				}
				case AA_UNREADHIGHPRIORITY: {
					MailSession->UnreadHighImage = Token->ArgumentOffsetOrID[1];
					break;
				}
				case AA_UNREADLOWPRIORITY: {
					MailSession->UnreadLowImage = Token->ArgumentOffsetOrID[1];
					break;
				}
				case AA_DELETED: {
					MailSession->DeletedImage = Token->ArgumentOffsetOrID[1];
					break;
				}
				default: {
					break;
				}
			}
			return(TRUE);
		}

		case T_CURMSG_STATE_IMAGE:{
			if (MailSession->CachedHeader) {
				unsigned long		State = MailSession->CachedMessageState;

				if (!(State & MSG_STATE_DELETED)) {
					if (State & MSG_STATE_READ) {
						if (MailSession->ReadHighImage != 0 && (State & MSG_STATE_PRIOHIGH)) {
							MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, MailSession->ReadHighImage, 0);
						} else if (MailSession->ReadLowImage != 0 && (State & MSG_STATE_PRIOLOW)) {
							MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, MailSession->ReadLowImage, 0);
						} else if (MailSession->ReadImage != 0) {
							MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, MailSession->ReadImage, 0);
						} else {
							MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[0], 0);
						}
					} else {
						if (MailSession->UnreadHighImage != 0 && (State & MSG_STATE_PRIOHIGH)) {
							MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, MailSession->UnreadHighImage, 0);
						} else if (MailSession->UnreadLowImage != 0 && (State & MSG_STATE_PRIOLOW)) {
							MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, MailSession->UnreadLowImage, 0);
						} else if (MailSession->UnreadImage != 0) {
							MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, MailSession->UnreadImage, 0);
						} else {
							MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[0], 0);
						}
					}
				} else {
					/* Deleted */
					if (MailSession->DeletedImage != 0) {
						MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, MailSession->DeletedImage, 0);
					} else {
						MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[1], 0);
					}
				}
			}
			MWSendClient(Client, URL, strlen(URL));

			return(TRUE);
		}

		case T_CURMSG_SUBJECT: {
			if (MailSession->CachedHeader) {
				if ((ReplyInt=MwMailFindRFC822HeaderLine(MailSession->CachedHeader, "Subject:", &ptr))>0) {
					if (*ptr) {
						MWHTMLSendClient(Client, ptr, ReplyInt);
						return(TRUE);
					} else {
						MWSendClient(Client, Session->Strings[Token->ArgumentOffsetOrID[0]], strlen(Session->Strings[Token->ArgumentOffsetOrID[0]]));
						return(TRUE);
					}
				} else {
					MWSendClient(Client, Session->Strings[Token->ArgumentOffsetOrID[0]], strlen(Session->Strings[Token->ArgumentOffsetOrID[0]]));
					return(TRUE);
				}
			}
			return(TRUE);
		}

		case T_CURMSG_FROM_NAME: {
			if (MailSession->CachedHeader) {
				if ((ReplyInt=MwMailFindRFC822HeaderLine(MailSession->CachedHeader, "From:", &ptr))>0) {
					unsigned char	ch;
					unsigned char	*p2;

					ch=ptr[ReplyInt];
					ptr[ReplyInt]='\0';
					p2=ptr;
					do {
						p2=MwMailParseRFC822Address(p2, Answer, BUFSIZE + 1, NULL, 0);
						MWHTMLSendClient(Client, Answer, strlen(Answer));
					} while (p2);
					ptr[ReplyInt]=ch;
				}
			}
			return(TRUE);
		}

		case T_CURMSG_RECEIVED: {
			if (MailSession->CachedHeader) {
				snprintf(Answer, sizeof(Answer), "%s&nbsp;&nbsp;%s", Session->DateFormatShort, Session->TimeFormat);
				ReplyInt=MsgPrint(Client->Temp, sizeof (Client->Temp), Answer, MailSession->CachedMessageInternaldate, &Session->DateFormat);
				MWSendClient(Client, Client->Temp, ReplyInt);
			}
			return(TRUE);
		}

		case T_CURMSG_FROM_ADDRESS: {
			if (MailSession->CachedHeader) {
				if ((ReplyInt=MwMailFindRFC822HeaderLine(MailSession->CachedHeader, "From:", &ptr))>0) {
					unsigned char	ch;
					unsigned char	*p2;

					ch=ptr[ReplyInt];
					ptr[ReplyInt]='\0';
					p2=ptr;
					do {
						p2=MwMailParseRFC822Address(p2, NULL, 0, Answer, BUFSIZE+1);
						MWSendClient(Client, Answer, strlen(Answer));
					} while (p2);
					ptr[ReplyInt]=ch;
				}
			}
			return(TRUE);
		}

		case T_CURMSG_TO_NAME: {
			/* arg 1 contains a text delimiter that should be inserted between names					*/
			/* arg 2, if non-zero, is the maximum number of bytes that can be sent by this token	*/
			if (MailSession->CachedHeader) {
				if ((ReplyInt=MwMailFindRFC822HeaderLine(MailSession->CachedHeader, "To:", &ptr))>0) {
					unsigned char	ch;
					unsigned char	*p2;

					ch=ptr[ReplyInt];
					ptr[ReplyInt]='\0';
					p2=ptr;
					if (Token->ArgumentOffsetOrID[1] == 0) {
						do {
							p2=MwMailParseRFC822Address(p2, NULL, 0, Answer, BUFSIZE+1);
							MWHTMLSendClient(Client, Answer, strlen(Answer));
							if (p2) {
								MWSendClient(Client, Token->Data+Token->ArgumentOffsetOrID[0], Token->ArgumentSize[0]);
							}
						} while (p2);
					} else {					
						unsigned long bytesSent = 0;
						unsigned long byteLimit = Token->ArgumentOffsetOrID[1];
						unsigned long len;

						do {
							p2=MwMailParseRFC822Address(p2, NULL, 0, Answer, BUFSIZE+1);
							len = strlen(Answer);
							if	((bytesSent + len) < byteLimit) {
								MWHTMLSendClient(Client, Answer, len);
								if (p2) {
									MWSendClient(Client, Token->Data+Token->ArgumentOffsetOrID[0], Token->ArgumentSize[0]);
								}
								bytesSent += (len + Token->ArgumentSize[0]);
							} else {
								break;
							}
						} while (p2);
					}

					ptr[ReplyInt]=ch;
				}
			}
			return(TRUE);
		}
		case T_CURMSG_TO_ADDRESS: {
			/* arg 1 contains a text delimiter that should be inserted between addresses.				*/
			/* arg 2, if non-zero, is the maximum number of bytes that can be sent by this token	*/
			if (MailSession->CachedHeader) {
				if ((ReplyInt=MwMailFindRFC822HeaderLine(MailSession->CachedHeader, "To:", &ptr))>0) {
					unsigned char	ch;
					unsigned char	*p2;

					ch=ptr[ReplyInt];
					ptr[ReplyInt]='\0';
					p2=ptr;
					if (Token->ArgumentOffsetOrID[1] == 0) {
						do {
							p2=MwMailParseRFC822Address(p2, NULL, 0, Answer, BUFSIZE+1);
							MWSendClient(Client, Answer, strlen(Answer));
							if (p2) {
								MWSendClient(Client, Token->Data+Token->ArgumentOffsetOrID[0], Token->ArgumentSize[0]);
							}
						} while (p2);
					} else {					
						unsigned long bytesSent = 0;
						unsigned long byteLimit = Token->ArgumentOffsetOrID[1];
						unsigned long len;

						do {
							p2=MwMailParseRFC822Address(p2, NULL, 0, Answer, BUFSIZE+1);
							len = strlen(Answer);
							if ((bytesSent + len) < byteLimit) {
								MWSendClient(Client, Answer, len);
								if (p2) {
									MWSendClient(Client, Token->Data+Token->ArgumentOffsetOrID[0], Token->ArgumentSize[0]);
								}
								bytesSent += len + Token->ArgumentSize[0];
							} else {
								break;
							}
						} while (p2);
					}
					ptr[ReplyInt]=ch;
				}
			}
			return(TRUE);
		}

		case T_CURMSG_ID: {
			if (MailSession->CachedHeader) {
				ReplyInt=snprintf(Answer, sizeof(Answer), "%lu", IDtoUID(Session, MailSession, MailSession->CurrentMsgListMessage));
				MWSendClient(Client, Answer, ReplyInt);
			}
			return(TRUE);
		}

		case T_CURMSG_VIEW_ID: {
			if (MailSession->CachedHeader) {
				ReplyInt=snprintf(Answer, sizeof(Answer), "%lu", Session->NumOfMessages-MailSession->CurrentMsgViewMessage+1);
				MWSendClient(Client, Answer, ReplyInt);
			}
			return(TRUE);
		}

		case T_CURMSG_CC_NAME: {
			/* arg 1 contains a text delimiter that should be inserted between names					*/
			/* arg 2, if non-zero, is the maximum number of bytes that can be sent by this token	*/
			if (MailSession->CachedHeader) {
				if ((ReplyInt=MwMailFindRFC822HeaderLine(MailSession->CachedHeader, "Cc:", &ptr))>0) {
					unsigned char	ch;
					unsigned char	*p2;

					ch=ptr[ReplyInt];
					ptr[ReplyInt]='\0';
					p2=ptr;
					if (Token->ArgumentOffsetOrID[1]	== 0) {
						do {
							p2=MwMailParseRFC822Address(p2, NULL, 0, Answer, BUFSIZE+1);
							MWHTMLSendClient(Client, Answer, strlen(Answer));
							if (p2) {
								MWSendClient(Client, Token->Data+Token->ArgumentOffsetOrID[0], Token->ArgumentSize[0]);
							}
						} while (p2);
					} else {
						unsigned long bytesSent = 0;
						unsigned long byteLimit = Token->ArgumentOffsetOrID[1];
						unsigned long len;

						do {
							p2=MwMailParseRFC822Address(p2, NULL, 0, Answer, BUFSIZE+1);
							len = strlen(Answer);
							if ((bytesSent + len) < byteLimit) {
								MWHTMLSendClient(Client, Answer, len);
								if (p2) {
									MWSendClient(Client, Token->Data+Token->ArgumentOffsetOrID[0], Token->ArgumentSize[0]);
								}
								bytesSent += (len + Token->ArgumentSize[0]);
							} else {
								break;
							}
						} while (p2);
					}
					ptr[ReplyInt]=ch;
				}
			}
			return(TRUE);
		}

		case T_CURMSG_CC_ADDRESS: {
			/* arg 1 contains a text delimiter that should be inserted between addresses				*/
			/* arg 2, if non-zero, is the maximum number of bytes that can be sent by this token	*/
			if (MailSession->CachedHeader) {
				if ((ReplyInt=MwMailFindRFC822HeaderLine(MailSession->CachedHeader, "Cc:", &ptr))>0) {
					unsigned char	ch;
					unsigned char	*p2;

					ch=ptr[ReplyInt];
					ptr[ReplyInt]='\0';
					p2=ptr;

					if (Token->ArgumentOffsetOrID[1]	== 0) {
						do {
							p2=MwMailParseRFC822Address(p2, NULL, 0, Answer, BUFSIZE+1);
							MWSendClient(Client, Answer, strlen(Answer));
							if (p2) {
								MWSendClient(Client, Token->Data+Token->ArgumentOffsetOrID[0], Token->ArgumentSize[0]);
							}
						} while (p2);
					} else {
						unsigned long bytesSent = 0;
						unsigned long byteLimit = Token->ArgumentOffsetOrID[1];
						unsigned long len;

						do {
							p2=MwMailParseRFC822Address(p2, NULL, 0, Answer, BUFSIZE+1);
							len = strlen(Answer);
							if ((bytesSent + len) < byteLimit) {
								MWSendClient(Client, Answer, len);
								if (p2) {
									MWSendClient(Client, Token->Data+Token->ArgumentOffsetOrID[0], Token->ArgumentSize[0]);
								}
								bytesSent += (len + Token->ArgumentSize[0]);
							} else {
								break;
							}
						} while (p2);
					}
						
					ptr[ReplyInt]=ch;
				}
			}
			return(TRUE);
		}

		case T_MSGLISTPREVSTART: {
			if (!MailSession->MailboxSelected) {
				MwMailSelectFolder(Session->CurrentFolder, Session, MailSession);
				MailSession->FirstMessageOnPage = MailSession->CurrentMsgListMessage = Session->NumOfMessages;

				if (MailSession->FirstMessageOnPage > Session->MessagesPerPage) {
					MailSession->LastMessageOnPage = (MailSession->FirstMessageOnPage - Session->MessagesPerPage);
				}
				else {

					MailSession->LastMessageOnPage = 0;
				}
			}

			if (!MailSession->MailboxSelected || Session->NumOfMessages==0 || (MailSession->CurrentMsgListMessage>=Session->NumOfMessages)) {
				*GotoToken=T_MSGLISTPREVEND;
				ModDebug("T_MSGLISTPREVSTART:No more data, no messages or no mailbox \n");
				return(TOKEN_MOVE_FORWARD);
			}
			return(TRUE);
		}

		case T_MSGLISTPREV: {
			MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MSGLIST_PREV, Token->ArgumentOffsetOrID[0], Session->CurrentFolder, 0, 0);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_MSGLISTPREVEND: {
			return(TRUE);
		}

		case T_MSGLISTNEXTSTART: {
			if (!MailSession->MailboxSelected) {
				MwMailSelectFolder(Session->CurrentFolder, Session, MailSession);
				MailSession->FirstMessageOnPage = MailSession->CurrentMsgListMessage = Session->NumOfMessages;

				if (MailSession->FirstMessageOnPage > Session->MessagesPerPage) {
					MailSession->LastMessageOnPage = (MailSession->FirstMessageOnPage - Session->MessagesPerPage);
				}
				else {

					MailSession->LastMessageOnPage = 0;
				}


			}
			if (!MailSession->MailboxSelected || Session->NumOfMessages==0 || MailSession->CurrentMsgListMessage<=Session->MessagesPerPage) {
				*GotoToken=T_MSGLISTNEXTEND;
				ModDebug("T_MSGLISTNEXTSTART:No more data, no messages or no mailbox \n");
				return(TOKEN_MOVE_FORWARD);
			}
			return(TRUE);
		}

		case T_MSGLISTNEXT: {
			MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MSGLIST_NEXT, Token->ArgumentOffsetOrID[0], Session->CurrentFolder, 0, 0);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_MSGLISTNEXTEND: {
			return(TRUE);
		}

		case T_MSGPERPAGE: {
			ReplyInt=snprintf(Answer, sizeof(Answer), "%lu", Session->MessagesPerPage);
			MWSendClient(Client, Answer, ReplyInt);
			return(TRUE);
		}

		case T_MSGINMBOX: {
			if (!MailSession->MailboxSelected) {
				MwMailSelectFolder(Session->CurrentFolder, Session, MailSession);
				MailSession->FirstMessageOnPage = MailSession->CurrentMsgListMessage = Session->NumOfMessages;

				if (MailSession->FirstMessageOnPage > Session->MessagesPerPage) {
					MailSession->LastMessageOnPage = (MailSession->FirstMessageOnPage - Session->MessagesPerPage);
				}
				else {

					MailSession->LastMessageOnPage = 0;
				}

			}

			ReplyInt=snprintf(Answer, sizeof(Answer), "%lu", Session->NumOfMessages);
			MWSendClient(Client, Answer, ReplyInt);
			return(TRUE);
		}

		case T_FIRSTMESSAGEONPAGE: {
			if (!MailSession->MailboxSelected) {
				MwMailSelectFolder(Session->CurrentFolder, Session, MailSession);
				MailSession->FirstMessageOnPage = MailSession->CurrentMsgListMessage = Session->NumOfMessages;

				if (MailSession->FirstMessageOnPage > Session->MessagesPerPage) {
					MailSession->LastMessageOnPage = (MailSession->FirstMessageOnPage - Session->MessagesPerPage);
				}
				else {

					MailSession->LastMessageOnPage = 0;
				}


			}

			if (Session->NumOfMessages!=0) {
					ReplyInt=snprintf(Answer, sizeof(Answer), "%lu", Session->NumOfMessages-MailSession->FirstMessageOnPage+1);
					MWSendClient(Client, Answer, ReplyInt);
			} else {
				MWSendClient(Client, "0", 1);
			}
			return(TRUE);
		}

		case T_LASTMESSAGEONPAGE: {
			if (!MailSession->MailboxSelected) {
				MwMailSelectFolder(Session->CurrentFolder, Session, MailSession);
				MailSession->FirstMessageOnPage = MailSession->CurrentMsgListMessage = Session->NumOfMessages;

				if (MailSession->FirstMessageOnPage > Session->MessagesPerPage) {
					MailSession->LastMessageOnPage = (MailSession->FirstMessageOnPage - Session->MessagesPerPage);
				}
				else {

					MailSession->LastMessageOnPage = 0;
				}

			}

			if (Session->NumOfMessages!=0) {
				ReplyInt=snprintf(Answer, sizeof(Answer), "%lu", Session->NumOfMessages-MailSession->LastMessageOnPage);
				MWSendClient(Client, Answer, ReplyInt);
			} else {
				MWSendClient(Client, "0", 1);
			}
			return(TRUE);
		}

		case T_DISPLAYED_MBOX: {
			ModDebug("Have T_DISPLAYED_MBOX %d\n", (int)Session->CurrentFolder);

			if (MailSession->CurrentFolder < Session->FolderDisplayNames->Used) {
				ptr = Session->FolderDisplayNames->Value[MailSession->CurrentFolder];
				MWSendClient(Client, ptr, strlen(ptr));
			}

			return(TRUE);
		}

		case T_MSGLIST_REFRESH: {
			MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MSGLIST_REFRESH, Token->ArgumentOffsetOrID[0], 0, 0, 0);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_PROCESS_MSGLIST: {
			MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MSGLIST_PROCESS, Token->ArgumentOffsetOrID[0], Token->ArgumentOffsetOrID[1], Token->ArgumentOffsetOrID[2], Session->CurrentFolder);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_VIEW_CURMSG: {
			MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MESSAGE_DISPLAY, Token->ArgumentOffsetOrID[0], IDtoUID(Session, MailSession, MailSession->CurrentMsgListMessage), MailSession->CurrentFolder, 0);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		/******************************
				View Message Template
		*******************************/

		case T_CURMSG_SELECT_PREV: {
			if (MailSession->CurrentMsgViewMessage<Session->NumOfMessages) {
				MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MESSAGE_DISPLAY, Token->ArgumentOffsetOrID[0], IDtoUID(Session, MailSession, MailSession->CurrentMsgViewMessage+1), MailSession->CurrentFolder, 0);
			} else {
				MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MESSAGE_DISPLAY, Token->ArgumentOffsetOrID[0], IDtoUID(Session, MailSession, MailSession->CurrentMsgViewMessage), MailSession->CurrentFolder, 0);
			}

			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_CURMSG_SELECT_NEXT: {
			if (MailSession->CurrentMsgViewMessage>1) {
				MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MESSAGE_DISPLAY, Token->ArgumentOffsetOrID[0], IDtoUID(Session, MailSession, MailSession->CurrentMsgViewMessage-1), MailSession->CurrentFolder, 0);
			} else {
				MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MESSAGE_DISPLAY, Token->ArgumentOffsetOrID[0], IDtoUID(Session, MailSession, MailSession->CurrentMsgViewMessage), MailSession->CurrentFolder, 0);
			}

			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_CURMSG_FORWARD: {
			MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_COMPOSE_ANSWER, Token->ArgumentOffsetOrID[0], IDtoUID(Session, MailSession, MailSession->CurrentMsgViewMessage), COMPOSE_FORWARD, MailSession->CurrentFolder);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_CURMSG_REPLY: {
			if (Token->ArgumentOffsetOrID[1] != AA_DROPORIGINALMESSAGE) {
				MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_COMPOSE_ANSWER, Token->ArgumentOffsetOrID[0], IDtoUID(Session, MailSession, MailSession->CurrentMsgViewMessage), COMPOSE_REPLY, MailSession->CurrentFolder);
			} else {
				MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_COMPOSE_ANSWER_SIMPLE, Token->ArgumentOffsetOrID[0], IDtoUID(Session, MailSession, MailSession->CurrentMsgViewMessage), COMPOSE_REPLY, MailSession->CurrentFolder);
			}
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_CURMSG_REPLYALL: {
			if (Token->ArgumentOffsetOrID[1] != AA_DROPORIGINALMESSAGE) {
				MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_COMPOSE_ANSWER, Token->ArgumentOffsetOrID[0], IDtoUID(Session, MailSession, MailSession->CurrentMsgViewMessage), COMPOSE_REPLYALL, MailSession->CurrentFolder);
			} else {
				MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_COMPOSE_ANSWER_SIMPLE, Token->ArgumentOffsetOrID[0], IDtoUID(Session, MailSession, MailSession->CurrentMsgViewMessage), COMPOSE_REPLYALL, MailSession->CurrentFolder);
			}
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_CURMSG_REDIRECT: {
			MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_COMPOSE_ANSWER, Token->ArgumentOffsetOrID[0], IDtoUID(Session, MailSession, MailSession->CurrentMsgViewMessage), COMPOSE_REDIRECT, MailSession->CurrentFolder);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_CURMSG_DELETE: {
			if (MailSession->CurrentMsgViewMessage==MailSession->CachedMessageID && MailSession->CachedMessageState & MSG_STATE_DELETED) {
				MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MESSAGE_UNSETFLAG, Token->ArgumentOffsetOrID[0], IDtoUID(Session, MailSession, MailSession->CurrentMsgViewMessage), FLAG_DELETE, MailSession->CurrentFolder);
			} else {
				MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MESSAGE_SETFLAG, Token->ArgumentOffsetOrID[0], IDtoUID(Session, MailSession, MailSession->CurrentMsgViewMessage), FLAG_DELETE, MailSession->CurrentFolder);
			}
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_CURMSG_UNDELETE: {
			MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MESSAGE_UNSETFLAG, Token->ArgumentOffsetOrID[0], IDtoUID(Session, MailSession, MailSession->CurrentMsgViewMessage), FLAG_DELETE, MailSession->CurrentFolder);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_CURMSG_MARK_UNREAD: {
			MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MESSAGE_UNSETFLAG, Token->ArgumentOffsetOrID[0], IDtoUID(Session, MailSession, MailSession->CurrentMsgViewMessage), FLAG_READ, MailSession->CurrentFolder);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_CURMSG_MARK_READ: {
			MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MESSAGE_SETFLAG, Token->ArgumentOffsetOrID[0], IDtoUID(Session, MailSession, MailSession->CurrentMsgViewMessage), FLAG_READ, MailSession->CurrentFolder);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_CURMSG_SETFLAG: {
			signed long			flag;

			switch (Token->ArgumentOffsetOrID[1]) {
				case AA_READ:						flag = FLAG_READ;			break;
				case AA_DELETED:					flag = FLAG_DELETE;		break;
				case AA_PRIVATE:					flag = FLAG_PRIVATE;		break;
				case AA_RECENT:					flag = FLAG_RECENT;		break;
				case AA_HIGHPRIORITY:			flag = FLAG_HIPRIO;		break;
				case AA_LOWPRIORITY:				flag = FLAG_LOWPRIO;		break;
				case AA_NORMALPRIORITY:			flag = FLAG_NORMPRIO;	break;

				default:								flag = -1;					break;
			}

			if (flag != -1) {
				MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MESSAGE_SETFLAG, Token->ArgumentOffsetOrID[0], IDtoUID(Session, MailSession, MailSession->CurrentMsgViewMessage), flag, MailSession->CurrentFolder);
				MWSendClient(Client, URL, strlen(URL));
			}

			return(TRUE);
		}

		case T_CURMSG_UNSETFLAG: {
			signed long			flag;

			switch (Token->ArgumentOffsetOrID[1]) {
				case AA_READ:						flag = FLAG_READ;			break;
				case AA_DELETED:					flag = FLAG_DELETE;		break;
				case AA_PRIVATE:					flag = FLAG_PRIVATE;		break;
				case AA_RECENT:					flag = FLAG_RECENT;		break;
				case AA_HIGHPRIORITY:			flag = FLAG_HIPRIO;		break;
				case AA_LOWPRIORITY:				flag = FLAG_LOWPRIO;		break;
				case AA_NORMALPRIORITY:			flag = FLAG_NORMPRIO;	break;

				default:								flag = -1;					break;
			}

			if (flag != -1) {
				MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MESSAGE_UNSETFLAG, Token->ArgumentOffsetOrID[0], IDtoUID(Session, MailSession, MailSession->CurrentMsgViewMessage), flag, MailSession->CurrentFolder);
				MWSendClient(Client, URL, strlen(URL));
			}

			return(TRUE);
		}

		case T_CURMSG_MOVE: {
			MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MESSAGE_MOVE, Token->ArgumentOffsetOrID[1], IDtoUID(Session, MailSession, MailSession->CurrentMsgViewMessage), Token->ArgumentOffsetOrID[0], MailSession->CurrentFolder);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_CURMSG_ACCEPT: {
			MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MESSAGE_ACCEPT, Token->ArgumentOffsetOrID[0], IDtoUID(Session, MailSession, MailSession->CurrentMsgViewMessage), MailSession->CurrentFolder, 0);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_CURMSG_DECLINE: {
			MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MESSAGE_DECLINE, Token->ArgumentOffsetOrID[0], IDtoUID(Session, MailSession, MailSession->CurrentMsgViewMessage), MailSession->CurrentFolder, 0);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_CURMSG_DECLINE_COMMENT: {
			/* Is there is not a return page value or we are not the organizer for this appointment ? */
			if (!Token->ArgumentSize[1] || !MwMailSessionUserIsICalOrganizer(Client, Session, MailSession)){
				/* Token arguments - 0 decline comment  page, 1 subject string, 2 decline page */
				MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MESSAGE_DECLINE_COMMENT, Token->ArgumentOffsetOrID[0], IDtoUID(Session, MailSession, MailSession->CurrentMsgViewMessage), MailSession->CurrentFolder, Token->ArgumentOffsetOrID[2]);
			} else {
				/* Treat this like a normal decline */
				MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MESSAGE_DECLINE, Token->ArgumentOffsetOrID[1], IDtoUID(Session, MailSession, MailSession->CurrentMsgViewMessage), MailSession->CurrentFolder, 0);
			}

			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_CURMSG_DELEGATE: {
			MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MESSAGE_DELEGATE, Token->ArgumentOffsetOrID[0], IDtoUID(Session, MailSession, MailSession->CurrentMsgViewMessage), MailSession->CurrentFolder, Token->ArgumentOffsetOrID[1]);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_CURMSG_BODY: {
			MwMailDisplayBody(Client, Session, MailSession, Token->ArgumentOffsetOrID[0], Token->ArgumentOffsetOrID[1], Token->ArgumentSize[2]>0 ? Token->Data+Token->ArgumentOffsetOrID[2] : NULL, MailSession->CurrentMsgViewMessage, 0, 0);
			return(TRUE);
		}

		case T_CURMSG_BODY_SOURCE: {
			MwMailDisplayBodySource(Client, Session, MailSession, Token->ArgumentOffsetOrID[0], MailSession->CurrentMsgViewMessage);
			return(TRUE);
		}

		case T_CURMSG_BDY_TXT: {
			if (Token->ArgumentOffsetOrID[0] == AA_FIRST) {
				MailSession->NextChunkStart = 0;
			} 

			MwMailDisplayBodyText(Client, Session, MailSession, Token->ArgumentOffsetOrID[2], Token->ArgumentOffsetOrID[3], Token->ArgumentSize[4]>0 ? Token->Data+Token->ArgumentOffsetOrID[4] : NULL, MailSession->CurrentMsgViewMessage, 0, 0, &MailSession->NextChunkStart, Token->ArgumentOffsetOrID[1], &MailSession->MoreBodyText);
			return(TRUE);
		}

		case T_CURMSG_BDY_TXT_MORE_STRT: {
			if (!MailSession->MoreBodyText) {
				*GotoToken=T_CURMSG_BDY_TXT_MORE_END;
				return(TOKEN_MOVE_FORWARD);
			}
			return(TRUE);
		}

		case T_CURMSG_BDY_TXT_MORE_END: {
			return(TRUE);
		}

		case T_CURMSG_BDY_TXT_DONE_STRT: {
			if (MailSession->MoreBodyText) {
				*GotoToken=T_CURMSG_BDY_TXT_DONE_END;
				return(TOKEN_MOVE_FORWARD);
			}
			return(TRUE);
		}

		case T_CURMSG_BDY_TXT_DONE_END: {
			return(TRUE);
		}

		case T_RFCMSG_SUBJECT: {
			if (MailSession->CachedPartHeader) {
				if ((ReplyInt=MwMailFindRFC822HeaderLine(MailSession->CachedPartHeader, "Subject:", &ptr))>0) {
					MWHTMLSendClient(Client, ptr, ReplyInt);
				} else {
					MWSendClient(Client, Session->Strings[Token->ArgumentOffsetOrID[0]], strlen(Session->Strings[Token->ArgumentOffsetOrID[0]]));
				}
			} else {
				MWSendClient(Client, Session->Strings[Token->ArgumentOffsetOrID[0]], strlen(Session->Strings[Token->ArgumentOffsetOrID[0]]));
			}
			return(TRUE);
		}

		case T_RFCMSG_FROM_NAME: {
			if (MailSession->CachedPartHeader) {
				if ((ReplyInt=MwMailFindRFC822HeaderLine(MailSession->CachedPartHeader, "From:", &ptr))>0) {
					unsigned char	ch;
					unsigned char	*p2;

					ch=ptr[ReplyInt];
					ptr[ReplyInt]='\0';
					p2=ptr;
					do {
						p2=MwMailParseRFC822Address(p2, Answer, BUFSIZE+1, NULL, 0);
						MWHTMLSendClient(Client, Answer, strlen(Answer));
						if (p2) {
							MWSendClient(Client, "&nbsp;", 6);
						}
					} while (p2);
					ptr[ReplyInt]=ch;
				}
			}
			return(TRUE);
		}

		case T_RFCMSG_FROM_ADDRESS: {
			if (MailSession->CachedPartHeader) {
				if ((ReplyInt=MwMailFindRFC822HeaderLine(MailSession->CachedPartHeader, "From:", &ptr))>0) {
					unsigned char	ch;
					unsigned char	*p2;

					ch=ptr[ReplyInt];
					ptr[ReplyInt]='\0';
					p2=ptr;
					do {
						p2=MwMailParseRFC822Address(p2, NULL, 0, Answer, BUFSIZE+1);
						MWSendClient(Client, Answer, strlen(Answer));
						if (p2) {
							MWSendClient(Client, "&nbsp;", 6);
						}
					} while (p2);
					ptr[ReplyInt]=ch;
				}
			}
			return(TRUE);
		}
		case T_RFCMSG_TO_NAME: {
			if (MailSession->CachedPartHeader) {
				if ((ReplyInt=MwMailFindRFC822HeaderLine(MailSession->CachedPartHeader, "To:", &ptr))>0) {
					unsigned char	ch;
					unsigned char	*p2;

					ch=ptr[ReplyInt];
					ptr[ReplyInt]='\0';
					p2=ptr;
					do {
						p2=MwMailParseRFC822Address(p2, NULL, 0, Answer, BUFSIZE+1);
						MWHTMLSendClient(Client, Answer, strlen(Answer));
						if (p2) {
							MWSendClient(Client, "&nbsp;", 6);
						}
					} while (p2);
					ptr[ReplyInt]=ch;
				}
			}
			return(TRUE);
		}

		case T_RFCMSG_TO_ADDRESS: {
			if (MailSession->CachedPartHeader) {
				if ((ReplyInt=MwMailFindRFC822HeaderLine(MailSession->CachedPartHeader, "To:", &ptr))>0) {
					unsigned char	ch;
					unsigned char	*p2;

					ch=ptr[ReplyInt];
					ptr[ReplyInt]='\0';
					p2=ptr;
					do {
						p2=MwMailParseRFC822Address(p2, NULL, 0, Answer, BUFSIZE+1);
						MWSendClient(Client, Answer, strlen(Answer));
						if (p2) {
							MWSendClient(Client, "&nbsp;", 6);
						}
					} while (p2);
					ptr[ReplyInt]=ch;
				}
			}
			return(TRUE);
		}

		case T_RFCMSG_CC_NAME: {
			if (MailSession->CachedPartHeader) {
				if ((ReplyInt=MwMailFindRFC822HeaderLine(MailSession->CachedPartHeader, "Cc:", &ptr))>0) {
					unsigned char	ch;
					unsigned char	*p2;

					ch=ptr[ReplyInt];
					ptr[ReplyInt]='\0';
					p2=ptr;
					do {
						p2=MwMailParseRFC822Address(p2, NULL, 0, Answer, BUFSIZE+1);
						MWSendClient(Client, Answer, strlen(Answer));
						if (p2) {
							MWSendClient(Client, "&nbsp;", 6);
						}
					} while (p2);
					ptr[ReplyInt]=ch;
				}
			}
			return(TRUE);
		}

		case T_RFCMSG_CC_ADDRESS: {
			if (MailSession->CachedPartHeader) {
				if ((ReplyInt=MwMailFindRFC822HeaderLine(MailSession->CachedPartHeader, "Cc:", &ptr))>0) {
					unsigned char	ch;
					unsigned char	*p2;

					ch=ptr[ReplyInt];
					ptr[ReplyInt]='\0';
					p2=ptr;
					do {
						p2=MwMailParseRFC822Address(p2, NULL, 0, Answer, BUFSIZE+1);
						MWSendClient(Client, Answer, strlen(Answer));
						if (p2) {
							MWSendClient(Client, "&nbsp;", 6);
						}
					} while (p2);
					ptr[ReplyInt]=ch;
				}
			}
			return(TRUE);
		}

		case T_CURMSG_ATTACH_START: {
			if (!MwMailLoadMIMECache(MailSession->CurrentMsgViewMessage, Client, Session, MailSession)) {
				*GotoToken=T_CURMSG_ATTACH_END;
				ModDebug("Skipping to T_CURMSG_ATTACH_END\n");
				return(TOKEN_MOVE_FORWARD);
			}

			/* Go through the list of body parts and decide if something needs to be displayed */
			if (MailSession->MIMEUsed==0 || !MailSession->MIME) {
				*GotoToken=T_CURMSG_ATTACH_END;
				ModDebug("Skipping to T_CURMSG_ATTACH_END\n");
				return(TOKEN_MOVE_FORWARD);
			}

			/* Initialize, if neccessary */
			if (MailSession->InAttachmentList==0) {
				MailSession->InAttachmentList=1;
			}

			/* Find the first "attachment" we want to display */
			while (MailSession->InAttachmentList <= MailSession->MIMEUsed) {
				/* We want to display anything we don't have a handler for */
				if (MailSession->MIME[MailSession->InAttachmentList-1].FileName[0]=='\0') {
					if (MWFindCodec(MailSession->MIME[MailSession->InAttachmentList-1].Type, FALSE)==NULL) {
						if (MWQuickNCmp(MailSession->MIME[MailSession->InAttachmentList-1].Type, "multipart/", 10)==FALSE) {
							return(TRUE);
						}
					}
				} else {
					if (MWQuickNCmp(MailSession->MIME[MailSession->InAttachmentList-1].Type, "multipart/", 10)==FALSE) {
						return(TRUE);
					}
				}
				MailSession->InAttachmentList++;
			};

			/* Didn't find any attachments */
			*GotoToken=T_CURMSG_ATTACH_END;
			ModDebug("Skipping to T_CURMSG_ATTACH_END (%d)\n", (int)MailSession->InAttachmentList);
			/* Do not reset InAttachmentList, need to do that at the _END token, otherwise we'll loop endless */
			return(TOKEN_MOVE_FORWARD);
		}

		case T_CURMSG_ATTACH_END: {
			/* Select the next item; _START will check if it's valid and skip to the next if not */
			if (++MailSession->InAttachmentList<=MailSession->MIMEUsed) {
				*GotoToken=T_CURMSG_ATTACH_START;
				ModDebug("Skipping to T_CURMSG_ATTACH_START\n");

				return(TOKEN_MOVE_BACKWARD);
			} else {
				if (!MailSession->MIMEParamStream) {
					MailSession->InAttachmentList=0;
				} else {
					MWReleaseStreams(MailSession->MIMEParamStream);
					MailSession->MIMEParamStream = NULL;
					MailSession->InAttachmentList=0;
				}
			}
			return(TRUE);
		}

		case T_CURMSG_ATTACH_INFO: {
			switch (Token->ArgumentOffsetOrID[0]) {
				case AA_FILELINK: {
					if (MailSession->InAttachmentList<=MailSession->MIMEUsed) {
						MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_MESSAGE_PART, IDtoUID(Session, MailSession, MailSession->CurrentMsgViewMessage), MailSession->InAttachmentList-1, MailSession->CurrentFolder, 0);
						MWSendClient(Client, URL, strlen(URL));
					}
					return(TRUE);
				}
			
				case AA_FILENAME: {

					/* Find the first "attachment" */
					if (MailSession->InAttachmentList<=MailSession->MIMEUsed) {
						MWDisplayMIMEParam(Client, MailSession->MIME[MailSession->InAttachmentList-1].FileName, &MailSession->MIMEParamStream);
					}
					return(TRUE);
				}
				
				case AA_FILESIZE: {
					if (MailSession->InAttachmentList<=MailSession->MIMEUsed) {
						ReplyInt=MwMailPrintFormatted(MailSession->MIME[MailSession->InAttachmentList-1].PartSize, Answer, sizeof(Answer));
						MWSendClient(Client, Answer, ReplyInt);
					}
					return(TRUE);
				}
			}
			return(TRUE);
		}

		/******************************
				Compose Template
		*******************************/

		case T_COMPOSE_FORM: {
			MWEncodeURLEx(Session, URL, URL_TYPE_LINK, COMPOSE_FORM, Token->ArgumentOffsetOrID[0], Token->ArgumentOffsetOrID[1], Token->ArgumentOffsetOrID[2], 0, 1);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_COMPOSE_CANCEL: {
			MWEncodeURL(Session, URL, URL_TYPE_LINK, COMPOSE_CANCEL, Token->ArgumentOffsetOrID[0], 0, 0, 0);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		} 

		case T_COMPOSE_VALUE: {
			FILE				*FileHandle;
			unsigned long	count=0;

			switch (Token->ArgumentOffsetOrID[0]) {
				case AA_ATTACHMENTS: {
					ReplyInt=snprintf(Answer, sizeof(Answer), "%d", (int)MailSession->Attachments);
					MWSendClient(Client, Answer, ReplyInt);
					return(TRUE);
				}	

				case AA_TO: {
					MwMailSendAddressValues(Client, Session, COMPOSE_EXT_TO_LIST[COMPOSE_TO]);
					return(TRUE);
				}	

				case AA_CC: {
					MwMailSendAddressValues(Client, Session, COMPOSE_EXT_TO_LIST[COMPOSE_CC]);
					return(TRUE);
				}	

				case AA_BCC: {
					MwMailSendAddressValues(Client, Session, COMPOSE_EXT_TO_LIST[COMPOSE_BCC]);
					return(TRUE);
				}	

				case AA_LOCATION: {
					snprintf(Client->Temp, sizeof(Client->Temp), "%s/%x."COMPOSE_EXT_LOCATION, MwMail.WorkDir, (unsigned int)Session->SessionID);			
					break;
				}

				case AA_SUBJECT: {
					snprintf(Client->Temp, sizeof(Client->Temp), "%s/%x."COMPOSE_EXT_SUBJECT, MwMail.WorkDir, (unsigned int)Session->SessionID);			
					break;
				}

				case AA_BODY: {
					snprintf(Client->Temp, sizeof(Client->Temp), "%s/%x."COMPOSE_EXT_BODY, MwMail.WorkDir, (unsigned int)Session->SessionID);			
					break;
				}

				case AA_PRIORITY: {
					if (MailSession->Priority == 0) {
						/* 3 == Normal */
						MailSession->Priority = 3;
					}

					if (Token->ArgumentOffsetOrID[1] == MailSession->Priority) {
						MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
					} else {
						MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[3], Token->ArgumentSize[3]);
					}
					return(TRUE);
				}

				case AA_DSN: {
					if (MailSession->DSNFlags == 0) {
						MailSession->DSNFlags = DSN_DEFAULT;
					}

					switch (Token->ArgumentOffsetOrID[1]) {
						case 1: {		/* Success */
							if (!(MailSession->DSNFlags & DSN_FAILURE) && (MailSession->DSNFlags & DSN_SUCCESS)) {
								MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
							} else {
								MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[3], Token->ArgumentSize[3]);
							}
							break;
						}

						case 2: {		/* Failure and Success */
							if ((MailSession->DSNFlags & DSN_FAILURE) && (MailSession->DSNFlags & DSN_SUCCESS)) {
								MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
							} else {
								MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[3], Token->ArgumentSize[3]);
							}
							break;
						}

						default:
						case 0: {		/* Default: Failure */
							if ((MailSession->DSNFlags & DSN_FAILURE) && !(MailSession->DSNFlags & DSN_SUCCESS)) {
								MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
							} else {
								MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[3], Token->ArgumentSize[3]);
							}
							break;
						}
					}
					return(TRUE);
				}

				default: {
					return(TRUE);
				}						
			}			

			MWProtectNMAP(Client, TRUE);
			FileHandle=fopen(Client->Temp,"rb");	

			if (FileHandle) {
				while (!feof(FileHandle) && !ferror(FileHandle)) {
					count=fread(Client->Temp, 1, BUFSIZE, FileHandle);
					if (count>0) {
						MWSendClient(Client, Client->Temp, count);
					} 
				}
				fclose(FileHandle);	
			}
			MWProtectNMAP(Client, FALSE);

			return(TRUE);
		}

		case T_ERR_MESSAGE: {
			if (MailSession->Error) {

#if 0
				/* Should we log this */
				ReplyInt = snprintf(Answer, sizeof(Answer), "(Error: %lu) ", MailSession->Error); 
				MWSendClient(Client, Answer, ReplyInt);
#endif


				MWSendClient(Client, Session->Strings[Token->ArgumentOffsetOrID[0] + MailSession->Error], strlen(Session->Strings[Token->ArgumentOffsetOrID[0] + MailSession->Error]));
				MailSession->Error = 0;		
			}
			return(TRUE);
		}

		/******************************
				Attach Template
		*******************************/

		case T_ATTACH_LIST_START: {
			if (MailSession->Attachments==0) {
				*GotoToken = T_ATTACH_LIST_END;
				return(TOKEN_MOVE_FORWARD);
			}

			if (!MailSession->InFileList) {
				MailSession->InFileList=TRUE;
				MailSession->CurrentFile=0;
			}

			if ((MailSession->CurrentFile % 2)==0) {
				MailSession->CurrentColor=Token->Data+Token->ArgumentOffsetOrID[0];
				MailSession->CurrentColorLen=Token->ArgumentSize[0];
			} else {
				MailSession->CurrentColor=Token->Data+Token->ArgumentOffsetOrID[1];
				MailSession->CurrentColorLen=Token->ArgumentSize[1];
			}

			return(TRUE);
		}

		case T_ATTACH_LIST_END: {
			if (MailSession->Attachments>0) {
				if (MailSession->InFileList && ((MailSession->CurrentFile+1) < MailSession->Attachments)) {
					MailSession->CurrentFile++;
					*GotoToken = T_ATTACH_LIST_START;
					return(TOKEN_MOVE_BACKWARD);
				} else {
					if (!MailSession->MIMEParamStream) {
						MailSession->InFileList=FALSE;
					} else {
						MWReleaseStreams(MailSession->MIMEParamStream);
						MailSession->MIMEParamStream = NULL;
						MailSession->InFileList=FALSE;
					}
				}
			}
			return(TRUE);
		}

		case T_ATTACH_CURRENT_INFO: {
			if (MailSession->InFileList) {
				switch (Token->ArgumentOffsetOrID[0]) {
					case AA_FILEID: {
						ReplyInt=snprintf(Answer, sizeof(Answer), "%d", (int)MailSession->CurrentFile);
						MWSendClient(Client, Answer, ReplyInt);
						return(TRUE);
					}
				
					case AA_FILENAME: {
						FILE						*FileHandle;	

						snprintf(Answer, sizeof(Answer), "%s/%x.%d", MwMail.WorkDir, (unsigned int)Session->SessionID, (int)MailSession->CurrentFile);		
						FileHandle=fopen(Answer, "rb");

						if (FileHandle) {
							if (fgets(Answer, BUFSIZE, FileHandle)) {
								fclose(FileHandle);
								ChopNL(Answer);
								MWDisplayMIMEParam(Client, Answer, &MailSession->MIMEParamStream);
								return(TRUE);
							}
							fclose(FileHandle);		
						}
						return(FALSE);
					}
					
					case AA_FILESIZE: {
						struct stat				FileInfo;

						snprintf(Answer, sizeof(Answer), "%s/%x.%d", MwMail.WorkDir, (unsigned int)Session->SessionID, (int)MailSession->CurrentFile);		
						if (stat(Answer, &FileInfo) == -1) {		
							return (TRUE);	
						}

						ReplyInt = snprintf(Answer, sizeof(Answer), "%d", (int)(FileInfo.st_size / 1024) + 1);
						MWSendClient(Client, Answer, ReplyInt);

						return(TRUE);;
					}
				}
			}
			return(FALSE);
		}

		/******************************
			Address Book Tokens
		*******************************/

#define FindField(PTR, BUFFER, FIELD) {unsigned long COUNTER = 0; PTR = BUFFER; while (COUNTER < FIELD) { if (*PTR == '\0') { COUNTER++;} PTR++;}}

		case T_ADDRESS_SEARCH_NOW: {
			if (Token->ArgumentOffsetOrID[0] == AA_CLEARSEARCHSTRING) {
				if (MailSession->SearchString) {
					MemFree(MailSession->SearchString);
					MailSession->SearchString = NULL;
				}		
			}
			MwMailAddrBookSearch(Client, Session, MailSession);
			return(TRUE);
		}

		case T_ADDRESS_SEARCH_FORM: {
			MWEncodeURL(Session, URL, URL_TYPE_LINK, ADDRESS_SEARCH_FORM, Token->ArgumentOffsetOrID[0], Token->ArgumentOffsetOrID[1], 0, 0);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_ADDRESS_SEARCH_CANCEL: {
			MWEncodeURL(Session, URL, URL_TYPE_LINK, ADDRESS_SEARCH_CANCEL, Token->ArgumentOffsetOrID[0], 0, 0, 0);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		} 

		case T_ADDRESS_SEARCH_STRING: {
			if (MailSession->SearchString) {
				MWSendClient(Client, MailSession->SearchString, strlen(MailSession->SearchString));			
			}	
			return(TRUE);
		} 

		case T_ADDRESS_SEARCH_SCOPE: {
			switch (Token->ArgumentOffsetOrID[0]) {
				case AA_PERSONAL: {
					if ((MailSession->SearchScope & SEARCH_PERSONAL) == 0) {
						return(TRUE);
					}
					break;
				}
				case AA_SYSTEM: {
					if ((MailSession->SearchScope & SEARCH_SYSTEM) == 0) {
						return(TRUE);
					}
					break;
				}
				case AA_PUBLIC: {
					if ((MailSession->SearchScope & SEARCH_PUBLIC) == 0) {
						return(TRUE);
					}
					break;
				}
				default: {
					return(FALSE);
				}
			}		
			MWSendClient(Client, Token->Data+Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);				
			return(TRUE);
		} 

		case T_ADDRESS_RESULT_FORM: {
			if (MailSession->ValueType != SESSION_VALUE_TYPE_ADDRESSES)  {
				/* The ValueStruct must have used for something else; redo the search */
				MwMailAddrBookSearch(Client, Session, MailSession);
			}

			MWEncodeURL(Session, URL, URL_TYPE_LINK, ADDRESS_RESULT_FORM, Token->ArgumentOffsetOrID[0], Token->ArgumentOffsetOrID[1], Token->ArgumentOffsetOrID[2], Token->ArgumentOffsetOrID[3]);
			MWSendClient(Client, URL, strlen(URL));

			return(TRUE);
		}

		case T_ADDRESS_PERSONAL_FORM: {
			unsigned long	PersonalAddressID = 0;
			unsigned char	Current;
			unsigned long	i;									 

			if ((MailSession->InAddressList == PERSONAL) && (MailSession->Index < MailSession->Value->Used)) {
				/* This for loop converts the contents of Value from a HEX string to an unsigned long */
				for (i = 0; i < ADDR_ENTRY_ID_LEN; i++) {
					Current = toupper(*(MailSession->Value->Value[MailSession->Index] + 2 + i));
					switch (Current) {
						case 'A':
						case 'B':
						case 'C':
						case 'D':
						case 'E':
						case 'F': {
							PersonalAddressID += ((Current - 55) * (1 << (4*((ADDR_ENTRY_ID_LEN-1)-i))));
							break;  
						}
						default: {
							PersonalAddressID += ((Current - 48) * (1 << (4*((ADDR_ENTRY_ID_LEN-1)-i))));
							break;  				
						}
					}
				}
			}														 

			MWEncodeURL(Session, URL, URL_TYPE_LINK, ADDRESS_PERSONAL_FORM, PersonalAddressID, Token->ArgumentOffsetOrID[0], Token->ArgumentOffsetOrID[1], 0);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_ADDRESS_PERSONAL_TITLE: {
			if (MailSession->InAddressList != PERSONAL) {
				MWSendClient(Client, Session->Strings[Token->ArgumentOffsetOrID[0]], strlen(Session->Strings[Token->ArgumentOffsetOrID[0]]));
			} else {
				MWSendClient(Client, Session->Strings[Token->ArgumentOffsetOrID[1]], strlen(Session->Strings[Token->ArgumentOffsetOrID[1]]));
			}
			return(TRUE);
		}

		case T_ADDRESS_PERSONAL_LINK: {
			unsigned long	PersonalAddressID = 0;
			unsigned char	Current;
			unsigned long	i;

			if ((Token->ArgumentOffsetOrID[0] != AA_CREATE) && (MailSession->InAddressList == PERSONAL) && (MailSession->Index < MailSession->Value->Used)) {

				/* This for loop converts the contents of Value from a HEX string to an unsigned long */
				for (i = 0; i < ADDR_ENTRY_ID_LEN; i++) {
					Current = toupper(*(MailSession->Value->Value[MailSession->Index] + 2 + i));
					switch (Current) {
						case 'A':
						case 'B':
						case 'C':
						case 'D':
						case 'E':
						case 'F': {
							PersonalAddressID += ((Current - 55) * (1 << (4*((ADDR_ENTRY_ID_LEN-1)-i))));
							break;  
						}
						default: {
							PersonalAddressID += ((Current - 48) * (1 << (4*((ADDR_ENTRY_ID_LEN-1)-i))));
							break;  				
						}
					}
				}
			}			

			MWEncodeURL(Session, URL, URL_TYPE_LINK, ADDRESS_PERSONAL_VIEW, Token->ArgumentOffsetOrID[1], MailSession->Index, PersonalAddressID, 0);
			MWSendClient(Client, URL, strlen(URL));

			return(TRUE);
		}

		case T_ADDRESS_FIELD_STRING: {
			unsigned char *FieldPtr;

			if ((!MailSession->InAddressList) || (MailSession->Index >= MailSession->Value->Used)) {
				return(TRUE);
			}

			FindField(FieldPtr, MailSession->Value->Value[MailSession->Index]+2, Token->ArgumentOffsetOrID[0]);
			ReplyInt = strlen(FieldPtr);
			if (ReplyInt != 0) {
				MWHTMLSendClient(Client, FieldPtr, ReplyInt);
			}

			return(TRUE);
		}

		case T_ADDRESS_LIST_ID: {
			if (MailSession->InAddressList) {
				ReplyInt = snprintf(Answer, sizeof(Answer),"%d", (int)(MailSession->Index+1));
				MWSendClient(Client, Answer, ReplyInt);
			}
			
			return(TRUE);
		}

		case T_ADDRESS_FIELD_ID: {
			if (MailSession->InAddressList) {
				if (MailSession->InAddressList == PERSONAL) {
					strncpy(Answer, MailSession->Value->Value[MailSession->Index]+2, 8);
				} 	
				MWSendClient(Client, Answer, 8);
			}
			
			return(TRUE);
		}

		case T_ADDRESS_FIELD_BIRTHDAY: {
			unsigned long	i;
			unsigned long	SetValue;
			unsigned char	*ptr;
			unsigned char	*FieldPtr;
			BOOL				UseDefault = TRUE;

			if ((MailSession->InAddressList) && (MailSession->Index < MailSession->Value->Used)) {
				/* We have real data */
				UseDefault = FALSE;
			}							 

			ptr = Session->DateFormatLong;			

			ModDebug("T_INSERTBIRTHDATESEL: Date Format String = %s\n", ptr);

			while (*ptr != '\0') {
				switch (*ptr) {
					case '%': {								
						ptr++;
						switch (*ptr) {
							case 'm': {
								MWSendClient(Client, "<SELECT NAME=\"BIRTHMONTH\">", 26);

								if (UseDefault) {
									SetValue = 0;
								} else {
									FindField(FieldPtr, MailSession->Value->Value[MailSession->Index]+2, AA_BIRTHMONTH);
									SetValue = atol(FieldPtr);
								}

								MWSendClient(Client, "<OPTION VALUE=\"0\"></OPTION>", 27);
								for (i=1; i<13; i++) {
									if (i != SetValue) {
										ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION VALUE=\"%d\">%02d</OPTION>", (int)i, (int)i);
									} else {
										ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION SELECTED VALUE=\"%d\">%02d</OPTION>", (int)i, (int)i);
									}
									MWSendClient(Client, Answer, ReplyInt);
								}

								MWSendClient(Client, "</SELECT>", 9);
							
								break;
							}

							case 'B':
							case 'b': {

								MWSendClient(Client, "<SELECT NAME=\"BIRTHMONTH\">", 26);

								if (UseDefault) {
									SetValue = 0;
									
								} else {
									FindField(FieldPtr, MailSession->Value->Value[MailSession->Index]+2, AA_BIRTHMONTH);
									SetValue = atol(FieldPtr);
								}

								MWSendClient(Client, "<OPTION VALUE=\"0\"></OPTION>", 27);
								for (i=1; i<13; i++) {
									if (i != SetValue) {
										ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION VALUE=\"%d\">%s</OPTION>", (int)i, Session->DateFormat.monthLong[i-1]);
									} else {
										ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION SELECTED VALUE=\"%d\">%s</OPTION>", (int)i, Session->DateFormat.monthLong[i-1]);
									}
									MWSendClient(Client, Answer, ReplyInt);
								}

								MWSendClient(Client, "</SELECT>", 9);

								break;
							}

							case 'd': {
								MWSendClient(Client, "<SELECT NAME=\"BIRTHDAY\">", 24);		

								if (UseDefault) {
									SetValue = 0;
								} else {
									FindField(FieldPtr, MailSession->Value->Value[MailSession->Index]+2, AA_BIRTHDAY);
									SetValue = atol(FieldPtr);
								}
				
								MWSendClient(Client, "<OPTION VALUE=\"0\"></OPTION>", 27);
								for (i=1; i<32; i++) {
									if (i != SetValue) {
										ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION VALUE=\"%d\">%d</OPTION>", (int)i, (int)i);
									} else {
										ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION SELECTED VALUE=\"%d\">%d</OPTION>", (int)i, (int)i);
									}
									MWSendClient(Client, Answer, ReplyInt);
								}
								MWSendClient(Client, "</SELECT>", 9);

								break;
							}

							case 'y': {
								BOOL RealYear = FALSE;							
	
								if (UseDefault) {
									SetValue = Session->CurrentYear;
								} else {
									FindField(FieldPtr, MailSession->Value->Value[MailSession->Index]+2, AA_BIRTHYEAR);
									SetValue = atol(FieldPtr);
									if ((SetValue > 1899) && (SetValue < 2051)) {
										RealYear = TRUE;	
									} else {
										SetValue = Session->CurrentYear;
									}
								}
							
								MWSendClient(Client, "<SELECT NAME=\"BIRTHYEAR\">", 25);
								for (i=1900; i<2051; i++) {
									if (i != SetValue) {
										ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION VALUE=\"%d\">%02d</OPTION>", (int)i, (i > 1999) ?  (int)(i - 2000) : (int)(i - 1900));
									} else {
										if (UseDefault) {
											ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION SELECTED VALUE=\"0\"></OPTION><OPTION VALUE=\"%d\">%02d</OPTION>", (int)i, (i > 1999) ?  (int)(i - 2000) : (int)(i - 1900));
										} else {
											ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION VALUE=\"0\"></OPTION><OPTION SELECTED VALUE=\"%d\">%02d</OPTION>", (int)i, (i > 1999) ?  (int)(i - 2000) : (int)(i - 1900));
										}
									}
									MWSendClient(Client, Answer, ReplyInt);
								}
								MWSendClient(Client, "</SELECT>", 9);
							
								break;
							}

							case 'Y': {
								BOOL RealYear = FALSE;								

 								if (UseDefault) {
									SetValue = Session->CurrentYear;
								} else {
									FindField(FieldPtr, MailSession->Value->Value[MailSession->Index]+2, AA_BIRTHYEAR);
									SetValue = atol(FieldPtr);
									if ((SetValue > 1899) && (SetValue < 2051)) {
										RealYear = TRUE;	
									} else {
										SetValue = Session->CurrentYear;
									}
								}

								MWSendClient(Client, "<SELECT NAME=\"BIRTHYEAR\">", 25);
								for (i=1900; i<2051; i++) {
									if (i != SetValue) {
										ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION VALUE=\"%d\">%d</OPTION>", (int)i, (int)i);
									} else {
										if (!RealYear) {
											ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION SELECTED VALUE=\"0\"></OPTION><OPTION VALUE=\"%d\">%d</OPTION>", (int)i, (int)i);
										} else {
											ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION VALUE=\"0\"></OPTION><OPTION SELECTED VALUE=\"%d\">%d</OPTION>", (int)i, (int)i);
										}
									}
									MWSendClient(Client, Answer, ReplyInt);
								}
								MWSendClient(Client, "</SELECT>", 9);
						
								break;
							}

							case '%': {
								MWSendClient(Client, ptr, 1);

								break;
							}

							default: {
								/* This little bit of ugly skips the unknown token and whitespace up to next token */
								/* skip unknown token  */
								while((*ptr != ' ') && (*ptr != '\t') && (*ptr != '\0'))
								{
									ptr++;
								}
								if (*ptr == '\0')
								{
									/* if we have reached the end break already */
									ptr--;
									break;
								}

								/* skip whitespace */
								while((*ptr == ' ') || (*ptr == '\t'))
								{
									ptr++;
								}

								/* back up one space for auto-increment below */
								ptr--;
								break;

							}

						}	
						break;
					}

					default: {
						MWSendClient(Client, ptr, 1);
						break;
					}	
				}
				ptr++;	
			}	
			return(TRUE);
		}

		case T_ADDRESS_FIELD_PHONE_TYPE: {
			unsigned long	i;
			unsigned long	SetValue = 0;
			unsigned char	*FieldPtr;
			
			ModDebug("T_ADDRESS_FIELD_PHONE_TYPE\n");

			if ((Token->ArgumentOffsetOrID[0] != AA_PHONETYPE1)  && (Token->ArgumentOffsetOrID[0] != AA_PHONETYPE2)) {
				return(FALSE);
			}

			if ((MailSession->InAddressList) && (MailSession->Index < MailSession->Value->Used)) {
				/* We have real data */
				FindField(FieldPtr, MailSession->Value->Value[MailSession->Index]+2, Token->ArgumentOffsetOrID[0]);
				SetValue = atol(FieldPtr);
			}							 	

			for (i=0; i<5; i++) {
				if (i != SetValue) {
					ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION VALUE=\"%d\">%s</OPTION>", (int)i, Session->Strings[Token->ArgumentOffsetOrID[1] + i]);
				} else {
					ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION SELECTED VALUE=\"%d\">%s</OPTION>", (int)i, Session->Strings[Token->ArgumentOffsetOrID[1] + i]);
				}
				MWSendClient(Client, Answer, ReplyInt);
			}

			return(TRUE);
		}

		case T_ADDRESS_DISPLAY_START: {
			switch (Token->ArgumentOffsetOrID[0]) {
				case AA_PERSONAL: {
					if (MailSession->SearchScope & SEARCH_PERSONAL) {
						return(TRUE);
					}
					break;						
				}

				case AA_SYSTEM: {
					if (MailSession->SearchScope & SEARCH_SYSTEM) {
						return(TRUE);
					}		
					break;				
				}

				case AA_PUBLIC: {
					if (MailSession->SearchScope & SEARCH_PUBLIC) {
						return(TRUE);
					}		
					break;				
				}								  
			}

			*GotoToken = T_ADDRESS_DISPLAY_END;
			return(TOKEN_MOVE_FORWARD);				
		} 

		case T_ADDRESS_DISPLAY_END: {
			return (TRUE);

		} 

		case T_ADDRESS_RESULTS_PER_PAGE: {
			if (Token->ArgumentOffsetOrID[0] > 0) {
				MailSession->ResultsPerPage = Token->ArgumentOffsetOrID[0];
			}	
			return(TRUE);
		}

		case T_ADDRESS_RESULTS_NEXT_START: {
			if ((MailSession->FirstResultInList + MailSession->ResultsPerPage) >= MailSession->Value->Used) {
				*GotoToken = T_ADDRESS_RESULTS_NEXT_END;
				return(TOKEN_MOVE_FORWARD);				
			}
			return(TRUE);			
		}

		case T_ADDRESS_RESULTS_NEXT_END: {
			return(TRUE);				
		}

		case T_ADDRESS_RESULTS_PREV_START: {
			if (MailSession->FirstResultInList == 0) {
				*GotoToken = T_ADDRESS_RESULTS_PREV_END;
				return(TOKEN_MOVE_FORWARD);				
			}
			return(TRUE);			
		}

		case T_ADDRESS_RESULTS_PREV_END: {
			return(TRUE);				
		}

		case T_ADDRESS_PRS_START: {

			if (MailSession->ValueType != SESSION_VALUE_TYPE_ADDRESSES)  {
				/* Bail, this token doesn't understand the data in the ValueStruct */
				*GotoToken = T_ADDRESS_PRS_END;
				return(TOKEN_MOVE_FORWARD);				
			}		

			switch (MailSession->InAddressList) {
				case PERSONAL: {
					return(TRUE);
				}

				case NO: {
					for (MailSession->Index=0; MailSession->Index < MailSession->Value->Used; MailSession->Index++) {
						if (*MailSession->Value->Value[MailSession->Index] == PERSONAL) {
							/* Found the first one */
							MailSession->InAddressList = PERSONAL;			
							MailSession->FirstResultInList = MailSession->Index;
							return(TRUE);
						}
					}
					/* There are no results of this type; fall through */
				}

				default: {
					*GotoToken = T_ADDRESS_PRS_END;
					return(TOKEN_MOVE_FORWARD);				
				}
			}
		}

		case T_ADDRESS_PRS_END: {
			if ((MailSession->ValueType == SESSION_VALUE_TYPE_ADDRESSES) && (MailSession->InAddressList == PERSONAL) && (MailSession->Index < MailSession->Value->Used)) {

				MailSession->Index++;

				if (MailSession->Index == MailSession->Value->Used) {
					/* We've reached the end of the list, Bail */
					MailSession->InAddressList = NO;
					return(TRUE);
				}

				if ((MailSession->ResultsPerPage > 0) && (MailSession->Index >= (MailSession->FirstResultInList + MailSession->ResultsPerPage))) {
					/* We've displayed everything we need to on this page */
					return(TRUE);
				}			

				if (*MailSession->Value->Value[MailSession->Index] == PERSONAL) {
					/* We have more personal entries; Loop */
					*GotoToken = T_ADDRESS_PRS_START;
					return(TOKEN_MOVE_BACKWARD);					
				} else {
					/* No more personal entries; Bail */
					MailSession->InAddressList = NO;
					return(TRUE);
				}
			}
			return(TRUE);
		}

		case T_ADDRESS_SYS_START: {

			if (MailSession->ValueType != SESSION_VALUE_TYPE_ADDRESSES)  {
				/* Bail, this token doesn't understand the data in the ValueStruct */
				*GotoToken = T_ADDRESS_SYS_END;
				return(TOKEN_MOVE_FORWARD);				
			}		

			switch (MailSession->InAddressList) {
				case SYSTEM: {
					return(TRUE);
				}

				case NO: {
					for (MailSession->Index=0; MailSession->Index < MailSession->Value->Used; MailSession->Index++) {
						if (*MailSession->Value->Value[MailSession->Index] == SYSTEM) {
							/* Found the first one */
							MailSession->InAddressList = SYSTEM;			
							MailSession->FirstResultInList = MailSession->Index;
							return(TRUE);
						}
					}
					/* There are no results of this type; fall through */
				}

				default: {
					*GotoToken = T_ADDRESS_SYS_END;
					return(TOKEN_MOVE_FORWARD);				
				}
			}
		}

		case T_ADDRESS_SYS_END: {
			if ((MailSession->ValueType == SESSION_VALUE_TYPE_ADDRESSES) && (MailSession->InAddressList == SYSTEM) && (MailSession->Index < MailSession->Value->Used)) {

				MailSession->Index++;
				if (MailSession->Index == MailSession->Value->Used) {
					/* We've reached the end of the list, Bail */
					MailSession->InAddressList = NO;
					return(TRUE);
				}
				if ((MailSession->ResultsPerPage > 0) && (MailSession->Index >= (MailSession->FirstResultInList + MailSession->ResultsPerPage))) {
					/* We've displayed everything we need to on this page */
					return(TRUE);
				}			
				if (*MailSession->Value->Value[MailSession->Index] == SYSTEM) {
					/* We have more system entries; Loop */
					*GotoToken = T_ADDRESS_SYS_START;
					return(TOKEN_MOVE_BACKWARD);					
				} else {
					/* No more system entries; Bail */
					MailSession->InAddressList = NO;
				}				
			} 
			return(TRUE);
		}

		case T_ADDRESS_PUB_START: {

			if (MailSession->ValueType != SESSION_VALUE_TYPE_ADDRESSES)  {
				/* Bail, this token doesn't understand the data in the ValueStruct */
				*GotoToken = T_ADDRESS_PUB_END;
				return(TOKEN_MOVE_FORWARD);				
			}		

			switch (MailSession->InAddressList) {
				case PUBLIC: {
					return(TRUE);
				}

				case NO: {
					for (MailSession->Index=0; MailSession->Index < MailSession->Value->Used; MailSession->Index++) {
						if (*MailSession->Value->Value[MailSession->Index] == PUBLIC) {
							/* Found the first one */
							MailSession->InAddressList = PUBLIC;			
							MailSession->FirstResultInList = MailSession->Index;
							return(TRUE);
						}
					}
					/* There are no results of this type; fall through */
				}

				default: {
					*GotoToken = T_ADDRESS_PUB_END;
					return(TOKEN_MOVE_FORWARD);				
				}
			}
		}

		case T_ADDRESS_PUB_END: {
			if ((MailSession->ValueType == SESSION_VALUE_TYPE_ADDRESSES) && (MailSession->InAddressList == PUBLIC) && (MailSession->Index < MailSession->Value->Used)) {

				MailSession->Index++;

				if (MailSession->Index == MailSession->Value->Used) {
					/* We've reached the end of the list, Bail */
					MailSession->InAddressList = NO;
					return(TRUE);
				}

				if ((MailSession->ResultsPerPage > 0) && (MailSession->Index >= (MailSession->FirstResultInList + MailSession->ResultsPerPage))) {
					/* We've displayed everything we need to on this page */
					return(TRUE);
				}			

				if ((MailSession->Index < MailSession->Value->Used) && (*MailSession->Value->Value[MailSession->Index] == PUBLIC)) {
					/* We have more public entries; Loop */
					*GotoToken = T_ADDRESS_PUB_START;
					return(TOKEN_MOVE_BACKWARD);					
				} else {
					/* No more public entries; Bail */
					MailSession->InAddressList = NO;
				}				
			}
			return(TRUE);
		}

		case T_ADDRESS_SELECT_FORM: {
			MWEncodeURL(Session, URL, URL_TYPE_LINK, ADDRESS_SELECT_FORM, Token->ArgumentOffsetOrID[0], 0, 0, 0);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_ADDRESS_SELECT_LIST: {
			FILE	*FileHandle;
			unsigned char FieldID;

			switch (Token->ArgumentOffsetOrID[0]) {
				case AA_TO: {
					snprintf(Answer, sizeof(Answer), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_TO]);
					FieldID = 'T';
					break;
				}
				case AA_CC: {
					snprintf(Answer, sizeof(Answer), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_CC]);
					FieldID = 'C';
					break;
				}
				case AA_BCC: {
					snprintf(Answer, sizeof(Answer), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_BCC]);
					FieldID = 'B';
					break;
				}
				default: {
					return(TRUE);
				}
			}

			MWProtectNMAP(Client, TRUE);
			FileHandle=fopen(Answer,"rb");

			if (FileHandle) {
				while (!feof(FileHandle) && !ferror(FileHandle)) {
					if (fgets(Client->Temp, BUFSIZE, FileHandle)) {
						ChopNL(Client->Temp);
						Client->Temp[MAXEMAILNAMESIZE]='\0';
						ReplyInt=snprintf(Answer, sizeof(Answer), "<TR><TD width=\"5%%\"><INPUT Type=\"CheckBox\" Name=\"%c\" value=\"%s\"></TD><TD width=\"95%%\" NOWRAP nowrap>&nbsp;&nbsp;<FONT size=\"%d\">%s</FONT></TD></TR>", 
											  FieldID, Client->Temp, (int)Token->ArgumentOffsetOrID[1], Client->Temp);
						MWSendClient(Client, Answer, ReplyInt);
					}
				}
				fclose(FileHandle);
			}
			MWProtectNMAP(Client, FALSE);
			return(TRUE);
		}

		case T_PURGE_DISABLED_START: {
			if (MailSession->PurgeDefault) {
				*GotoToken = T_PURGE_DISABLED_END;
				return(TOKEN_MOVE_FORWARD);
			}
			return(TRUE);
		}

		case T_PURGE_DISABLED_END: {
			return(TRUE);
		}

		case T_SERVER_TIME: {
			ReplyInt=snprintf(Answer, sizeof(Answer), "%d", (int)time(NULL));
			MWSendClient(Client, Answer, ReplyInt);				
			return(TRUE);
		} 

		case T_ISBCC_BEGIN: {
			/* This token returns true if we are a bcc recipient */
			/* This is discovered by not finding our name or address in to or cc fields. */
			/* search through to: field, go to end token if we find our address or name */
			if (MailSession->CachedHeader) {
				if ((ReplyInt=MwMailFindRFC822HeaderLine(MailSession->CachedHeader, "To:", &ptr))>0) {
					unsigned char	ch;
					unsigned char	*p2;

					ch=ptr[ReplyInt];
					ptr[ReplyInt]='\0';
					p2=ptr;

					do {
						Answer[0] = '\0';
						p2=MwMailParseRFC822Address(p2, NULL, 0, Answer, BUFSIZE+1);

						/* See if this matches our email address or user name */
						if ((MWQuickNCmp(Answer, Session->User, strlen(Session->User))) || 
							 (MWQuickNCmp(Answer, Session->EMailAddress, strlen(Session->User)))){
							/* found myself in to field skip to end */
							*GotoToken = T_ISBCC_END;
							ptr[ReplyInt] = ch;
							return(TOKEN_MOVE_FORWARD);
						}
					} while (p2);
					ptr[ReplyInt] = ch;

				}

				if ((ReplyInt=MwMailFindRFC822HeaderLine(MailSession->CachedHeader, "Cc:", &ptr))>0) {
					unsigned char	ch;
					unsigned char	*p2;

					ch=ptr[ReplyInt];
					ptr[ReplyInt]='\0';
					p2=ptr;

					do {
							p2=MwMailParseRFC822Address(p2, NULL, 0, Answer, BUFSIZE+1);

							/* See if this matches our email address or user name */
							if ((MWQuickNCmp(Answer, Session->User, strlen(Session->User))) || 
								 (MWQuickNCmp(Answer, Session->EMailAddress, strlen(Session->User)))){
								/* found myself in to field skip to end */
								*GotoToken = T_ISBCC_END;
								ptr[ReplyInt] = ch;
								return(TOKEN_MOVE_FORWARD);
							}
					} while (p2);
					ptr[ReplyInt] = ch;
				}

				return(TRUE);
			}
		}

		case T_ISBCC_END: {
			return(TRUE);
		}

	}
	return(FALSE);
}

BOOL
MWMAILHandleURL(ConnectionStruct *Client, SessionStruct *Session, URLStruct *URL, void *ModuleData)
{
	MailSessionStruct	*MailSession=(MailSessionStruct	*)ModuleData;
	ModDebug("MWMAILHandleURL called (ModuleData:%x)\n", (unsigned int)ModuleData);

	switch(URL->Request) {
		case REQUEST_REFRESH_FOLDERLIST: {
			Session->FolderListIsDirty = TRUE;
		}

		case REQUEST_MSGLIST: {
			MwMailSelectFolder(URL->Argument[0], Session, MailSession);
			MailSession->CurrentMsgListMessage=Session->NumOfMessages;
			MailSession->FirstMessageOnPage=MailSession->CurrentMsgListMessage;
			if (MailSession->CurrentMsgListMessage>Session->MessagesPerPage) {
				MailSession->LastMessageOnPage=MailSession->CurrentMsgListMessage-Session->MessagesPerPage;
			} else {
				MailSession->LastMessageOnPage=0;
			}
			MailSession->InMsgList=FALSE;
			MWHandleTemplate(Client, Session, URL->Argument[1]);
			return(TRUE);
		}

		case COMPOSE_FORM: {
			ModDebug("Received COMPOSE_FORM Request\n");
			MwMailProcessComposeForm(Client, Session, MailSession, &URL->Argument[0], URL->Argument[1]);		  /* New Template returned in &URL->Argument[0] */
			MWHandleTemplate(Client, Session, URL->Argument[0]);	 
			return(TRUE);
		}

		case COMPOSE_CANCEL: {
			ModDebug("Received COMPOSE_CANCEL Request\n");
			MwMailComposeCleanUp(Session, MailSession);
			MWHandleTemplate(Client, Session, URL->Argument[0]);	 
			return(TRUE);
		}

		case REQUEST_MSGLIST_PREV: {
			MwMailSelectFolder(URL->Argument[1], Session, MailSession);

			/* Can we back up another page */
			if (Session->NumOfMessages > Session->MessagesPerPage) {
				if (MailSession->FirstMessageOnPage < Session->NumOfMessages) {
					if ((Session->NumOfMessages - MailSession->FirstMessageOnPage) > Session->MessagesPerPage) {
						MailSession->FirstMessageOnPage += Session->MessagesPerPage;
						MailSession->CurrentMsgListMessage = MailSession->FirstMessageOnPage;
						MailSession->LastMessageOnPage = (MailSession->FirstMessageOnPage - Session->MessagesPerPage);
					} else {
						MailSession->FirstMessageOnPage = Session->NumOfMessages;
						MailSession->CurrentMsgListMessage = MailSession->FirstMessageOnPage;
						MailSession->LastMessageOnPage = (Session->NumOfMessages - Session->MessagesPerPage);
					}
				}
			}

			MailSession->InMsgList=FALSE;
			MWHandleTemplate(Client, Session, URL->Argument[0]);
			return(TRUE);
		}

		case REQUEST_MSGLIST_LAST: {
			MwMailSelectFolder(URL->Argument[1], Session, MailSession);
			MailSession->CurrentMsgListMessage = MailSession->FirstMessageOnPage;

			if (MailSession->FirstMessageOnPage>Session->MessagesPerPage) {
				MailSession->LastMessageOnPage=MailSession->FirstMessageOnPage-Session->MessagesPerPage;
			} else {
				MailSession->LastMessageOnPage=0;
			}

			MailSession->InMsgList=FALSE;
			MWHandleTemplate(Client, Session, URL->Argument[0]);
			return(TRUE);
		}

		case REQUEST_MSGLIST_NEXT: {
			MwMailSelectFolder(URL->Argument[1], Session, MailSession);

			/* is there room to go forward another page */
			if (Session->NumOfMessages > Session->MessagesPerPage) {
				
				if (MailSession->FirstMessageOnPage > 0) {
					if (MailSession->FirstMessageOnPage > 2 * Session->MessagesPerPage) {
						MailSession->FirstMessageOnPage -= Session->MessagesPerPage;
					} else {
						MailSession->FirstMessageOnPage = Session->MessagesPerPage;

					}
				}
			}

			MailSession->CurrentMsgListMessage = MailSession->FirstMessageOnPage;

			if (MailSession->FirstMessageOnPage>Session->MessagesPerPage) {
				MailSession->LastMessageOnPage = MailSession->FirstMessageOnPage - Session->MessagesPerPage;
			} else {
				MailSession->LastMessageOnPage=0;
			}
			MailSession->InMsgList=FALSE;
			MWHandleTemplate(Client, Session, URL->Argument[0]);
			return(TRUE);
		}

		case REQUEST_MSGLIST_REFRESH: {
			Session->FolderListIsDirty = TRUE;

			MwMailRefreshFolder(Session, MailSession);
			MWHandleTemplate(Client, Session, URL->Argument[0]);	 
			return(TRUE);
		}

		case REQUEST_MSGLIST_PROCESS: {
			MwMailSelectFolder(URL->Argument[3], Session, MailSession);
			MwMailProcessMsgListForm(Client, Session, MailSession, URL->Argument[0], URL->Argument[1], URL->Argument[2], &URL->Argument[0]);		  /* New Template returned in &URL->Argument[0] */
			MWHandleTemplate(Client, Session, URL->Argument[0]);	 
			return(TRUE);
		}

		case REQUEST_MESSAGE_DISPLAY: {
			MwMailSelectFolder(URL->Argument[2], Session, MailSession);
			MailSession->CurrentMsgViewMessage=UIDtoID(Session, MailSession, URL->Argument[1]);
			MwMailLoadMessage(Client, MailSession->CurrentMsgViewMessage, Session, MailSession);
			MWHandleTemplate(Client, Session, URL->Argument[0]);	 
			return(TRUE);
		}

		case REQUEST_COMPOSE_ANSWER: {
			unsigned long	ID;

			MwMailSelectFolder(URL->Argument[3], Session, MailSession);
			ID=UIDtoID(Session, MailSession, URL->Argument[1]);

			/* Temp workaround to avoid replying to wrong message */
			MailSession->CachedMessageID=0;
			if (MailSession->CachedHeader) {
				MemFree(MailSession->CachedHeader);
				MailSession->CachedHeader=NULL;
				MailSession->CurrentMsgViewMessage=0;
			}

			if (MailSession->CurrentMsgViewMessage!=ID) {
				MailSession->CurrentMsgViewMessage=ID;
				MwMailLoadMessage(Client, MailSession->CurrentMsgViewMessage, Session, MailSession);
			}

			MwMailPrepareComposeAnswer(Client, Session, MailSession, URL->Argument[2], AA_INCLUDEORIGINALMESSAGE);
			MWHandleTemplate(Client, Session, URL->Argument[0]);	 
			return(TRUE);
		}

		case REQUEST_COMPOSE_ANSWER_SIMPLE: {
			unsigned long	ID;

			MwMailSelectFolder(URL->Argument[3], Session, MailSession);
			ID=UIDtoID(Session, MailSession, URL->Argument[1]);
			if (MailSession->CurrentMsgViewMessage!=ID) {
				MailSession->CurrentMsgViewMessage=ID;
				MwMailLoadMessage(Client, MailSession->CurrentMsgViewMessage, Session, MailSession);
			}

			MwMailPrepareComposeAnswer(Client, Session, MailSession, URL->Argument[2], AA_DROPORIGINALMESSAGE);
			MWHandleTemplate(Client, Session, URL->Argument[0]);	 
			return(TRUE);
		}

		case REQUEST_COMPOSE_VIA_URL: {
			FILE				*Handle;
			unsigned char	*ptr;

			if (Client->URLExtra && Client->URLExtra[0]!='\0') {
				snprintf(Client->Temp, sizeof(Client->Temp), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_TO]);
				Handle=fopen(Client->Temp, "ab");
				if (Handle) {
					/* We could look for a subject... */
					ptr=strchr(Client->URLExtra, '&');
					if (ptr) {
						*ptr='\0';
					}
					if (Client->URLExtra[0]=='<') {
						ptr=strchr(Client->URLExtra, '>');
						if (ptr) {
							*ptr='\0';
						}
						fprintf(Handle, "%s\r\n", Client->URLExtra+1);
					} else {
						fprintf(Handle, "%s\r\n", Client->URLExtra);
					}
					fclose(Handle);
				}	
			}
			MWHandleTemplate(Client, Session, URL->Argument[0]);
			return(TRUE);
		}

		case REQUEST_MESSAGE_PART: {
			MwMailSelectFolder(URL->Argument[2], Session, MailSession);
			MwMailSendBodyPart(Client, Session, MailSession, UIDtoID(Session, MailSession, URL->Argument[0]), URL->Argument[1]);
			return(TRUE);
		}

		case REQUEST_MESSAGE_SETFLAG: {
			MwMailSelectFolder(URL->Argument[3], Session, MailSession);

			MwMailSetMessageFlags(Client, Session, MailSession, UIDtoID(Session, MailSession, URL->Argument[1]), URL->Argument[2]);

			if (URL->Argument[2] == FLAG_DELETE) {
				if (MailSession->CurrentMsgViewMessage > 1) {
					MailSession->CurrentMsgViewMessage--;

					MwMailLoadMessage(Client, MailSession->CurrentMsgViewMessage, Session, MailSession);
				}
			}

			MWHandleTemplate(Client, Session, URL->Argument[0]);
			return(TRUE);
		}

		case REQUEST_MESSAGE_UNSETFLAG: {
			MwMailSelectFolder(URL->Argument[3], Session, MailSession);

			MwMailUnSetMessageFlags(Client, Session, MailSession, UIDtoID(Session, MailSession, URL->Argument[1]), URL->Argument[2]);
			MWHandleTemplate(Client, Session, URL->Argument[0]);

			return(TRUE);
		}

		case REQUEST_MESSAGE_ACCEPT: {
			URLStruct urlData ={0};
			MwMailSelectFolder(URL->Argument[2], Session, MailSession);
			MwMailHandleCalendarMessage(Client, Session, MailSession, UIDtoID(Session, MailSession, URL->Argument[1]), TRUE);
			urlData.Request = NM_REFRESH_CALENDAR;
			MWHandleNamedURL(Client,Session,&urlData);
			MWHandleTemplate(Client, Session, URL->Argument[0]);
			return(TRUE);
		}

		case REQUEST_MESSAGE_DECLINE: {
			URLStruct urlData ={0};
			MwMailSelectFolder(URL->Argument[2], Session, MailSession);
			MwMailHandleCalendarMessage(Client, Session, MailSession, UIDtoID(Session, MailSession, URL->Argument[1]), FALSE);

			/* keep calendar in sync with accepted messages */
			urlData.Request = NM_REFRESH_CALENDAR;
			MWHandleNamedURL(Client,Session,&urlData);
			MWHandleTemplate(Client, Session, URL->Argument[0]);
			return(TRUE);
		}

		case REQUEST_MESSAGE_DECLINE_COMMENT: {
			unsigned long	id;
			URLStruct urlData ={0};

			MwMailSelectFolder(URL->Argument[2], Session, MailSession);
			id = UIDtoID(Session, MailSession, URL->Argument[1]);

			/* Temp workaround to avoid replying to wrong message */
			MailSession->CachedMessageID = 0;
			if (MailSession->CachedHeader) {
				MemFree(MailSession->CachedHeader);
				MailSession->CachedHeader = NULL;
				MailSession->CurrentMsgViewMessage = 0;
			}

			if (MailSession->CurrentMsgViewMessage != id) {
				MailSession->CurrentMsgViewMessage = id;
				MwMailLoadMessage(Client, MailSession->CurrentMsgViewMessage, Session, MailSession);
			}

			/*
				Store the folder and the message id in the session so that we can delete it, if the user
				actually declines
			*/
			Session->DirtyMessageFolder = URL->Argument[2];
			Session->DirtyMessageID = URL->Argument[1];
			Session->DirtyMessage = FALSE;

			/* Set the subject prefix */
			MailSession->PrefixString = URL->Argument[3];

			/*
				We now need to pull the iCal object out of the message and store it in a compose file.
			*/
			MwMailPrepareComposeAnswer(Client, Session, MailSession, COMPOSE_DECLINE, AA_DROPORIGINALMESSAGE);

			/* keep calendar in sync with mail messages */
			urlData.Request = NM_REFRESH_CALENDAR;
			MWHandleNamedURL(Client,Session,&urlData);
			MWHandleTemplate(Client, Session, URL->Argument[0]);	 
			return(TRUE);
		}

		case REQUEST_MESSAGE_DELEGATE: {
			unsigned long	id;

			MwMailSelectFolder(URL->Argument[3], Session, MailSession);
			id = UIDtoID(Session, MailSession, URL->Argument[1]);

			/* Temp workaround to avoid replying to wrong message */
			MailSession->CachedMessageID = 0;
			if (MailSession->CachedHeader) {
				MemFree(MailSession->CachedHeader);
				MailSession->CachedHeader = NULL;
				MailSession->CurrentMsgViewMessage = 0;
			}

			if (MailSession->CurrentMsgViewMessage != id) {
				MailSession->CurrentMsgViewMessage = id;
				MwMailLoadMessage(Client, MailSession->CurrentMsgViewMessage, Session, MailSession);
			}

			/*
				Store the folder and the message id in the session so that we can delete it, if the user
				actually delegates
			*/
			Session->DirtyMessageFolder = URL->Argument[3];
			Session->DirtyMessageID = URL->Argument[1];
			Session->DirtyMessage = FALSE;

			/* Set the subject prefix */
			MailSession->PrefixString = URL->Argument[4];

			/*
				We now need to pull the iCal object out of the message and store it in a compose file.
			*/
			MwMailPrepareComposeAnswer(Client, Session, MailSession, COMPOSE_DELEGATE, AA_DROPORIGINALMESSAGE);

			MWHandleTemplate(Client, Session, URL->Argument[0]);	 
			return(TRUE);
		}

		case ADDRESS_SEARCH_FORM: {
			ModDebug("Received ADDRESS_SEARCH_FORM Request\n");

			MwMailProcessAddrSearchForm(Client, Session, MailSession, &URL->Argument[0], URL->Argument[1]);				/* New Template returned in &URL->Argument[0] */
			MWHandleTemplate(Client, Session, URL->Argument[0]);	 

			return(TRUE);
		}

		case ADDRESS_SEARCH_CANCEL: {
			ModDebug("Received ADDRESS_SEARCH_CANCEL Request\n");

			MwMailAddrBookCleanUp(Session, MailSession);

			MWHandleTemplate(Client, Session, URL->Argument[0]);	 

			return(TRUE);
		}

		case ADDRESS_RESULT_FORM: {
			ModDebug("Received ADDRESS_RESULT_FORM Request\n");

			MwMailProcessAddrResultForm(Client, Session, MailSession, &URL->Argument[0], URL->Argument[1], URL->Argument[2], URL->Argument[3]);					
			MWHandleTemplate(Client, Session, URL->Argument[0]);	 

			return(TRUE);
		}

		case ADDRESS_PERSONAL_VIEW: {
			ModDebug("Received ADDRESS_PERSONAL_VIEW Request\n");

			/* Arg 2 (PersonalEntryID) will be zero if this is a request to create a new entry */
			if ((URL->Argument[2]!=0) && (MwMailGetPersonalAddressBookEntry(Client, Session, MailSession, URL->Argument[1], URL->Argument[2]))) {
				MailSession->InAddressList = PERSONAL;	
			} else {
				MailSession->InAddressList = NO;
			}

			MWHandleTemplate(Client, Session, URL->Argument[0]);	 
			MailSession->InAddressList = NO;

			return(TRUE);
		}

		case ADDRESS_PERSONAL_FORM: {
			ModDebug("Received ADDRESS_PERSONAL_FORM Request\n");

			MwMailProcessAddrPersonalForm(Client, Session, MailSession, URL->Argument[0], &URL->Argument[1], URL->Argument[2]);
			MWHandleTemplate(Client, Session, URL->Argument[1]);

			return(TRUE);
		}

		case ADDRESS_SELECT_FORM: {
			ModDebug("Received ADDRESS_SELECT_FORM Request\n");

			MwMailProcessAddrSelectForm(Client, Session, MailSession, &URL->Argument[0]);
			MWHandleTemplate(Client, Session, URL->Argument[0]);

			return(TRUE);
		}

		case REQUEST_FOLDER_MANAGE: {
			MwMailProcessFolderManagementForm(Client, Session, MailSession, URL->Argument[1]);
			MWHandleTemplate(Client, Session, URL->Argument[0]);	 
			return(TRUE);
		}

		case REQUEST_FOLDER_MOVEMSG: {
			MwMailSelectFolder(URL->Argument[2], Session, MailSession);
			MwMailProcessFolderMessageMove(Client, Session, MailSession, URL->Argument[1]);
			MWHandleTemplate(Client, Session, URL->Argument[0]);
			return(TRUE);
		}

		case REQUEST_MESSAGE_MOVE: {
			MwMailSelectFolder(URL->Argument[3], Session, MailSession);
			MDBFreeValues(MailSession->Value);
			snprintf(Client->Temp, sizeof(Client->Temp), "%lu", URL->Argument[1]);
			MDBAddValue(Client->Temp, MailSession->Value);
			MailSession->ValueType=SESSION_VALUE_TYPE_MSGLIST;
			MwMailProcessFolderMessageMove(Client, Session, MailSession, URL->Argument[2]);
			MWHandleTemplate(Client, Session, URL->Argument[0]);
			return(TRUE);
		}

		default: {
			break;
		}
	}
	
	return(FALSE);
}

/*
	This are the two required functions for a module. The naming
	must be the binary name plus "Init" / "Shutdown"
	Init is supposed to call ModWeb and register any modules that
	are contained within the binary.
	Shutdown must clear the module for unloading.
*/

/********** new *************************/

static BOOL 
ReadMailConfiguration(void)
{
	MDBValueStruct	*V;
	XplDir			*Dir;
	XplDir			*DirEnt;
	unsigned char	Path[XPL_MAX_PATH + 1];
	unsigned char	*ptr;
	unsigned long	i;

	V = MDBCreateValueStruct(MwMail.DirectoryHandle, MsgGetServerDN(NULL));

	/* Queue server */
	if (MsgReadIP(MSGSRV_AGENT_MODWEB"\\"MSGSRV_AGENT_MWMAIL, MSGSRV_A_NMAP_QUEUE_SERVER, V)) {
		MwMail.NmapQAddress = inet_addr(V->Value[0]);
		if (MwMail.NmapQAddress == (unsigned long)-1) {
			MwMail.NmapQAddress = 0;
		}
	} else {
		MwMail.NmapQAddress = 0;
	}
	MDBFreeValues(V);

	/* Maximum Number of recipients */
	if (MDBRead(MSGSRV_AGENT_MODWEB"\\"MSGSRV_AGENT_MWMAIL, MSGSRV_A_RECIPIENT_LIMIT, V)) {
		MwMail.MaxRecipients = atol(V->Value[0]);
	}

	/* Maximum Message-size */
	if (MDBRead(MSGSRV_AGENT_MODWEB"\\"MSGSRV_AGENT_MWMAIL, MSGSRV_A_MESSAGE_LIMIT, V)) {
		MwMail.MaxMessageSize = atol(V->Value[0])*1024*1024;
	}
	MDBFreeValues(V);

	if (MDBRead(MSGSRV_AGENT_MODWEB"\\"MSGSRV_AGENT_MWMAIL, MSGSRV_A_CONFIGURATION, V)) {
		for (i=0; i<V->Used; i++) {
			if (MWQuickNCmp(V->Value[i], "SortAddressbook=", 16)) {
				if (V->Value[i][16]=='1') {
					MwMail.SortAddressBook=TRUE;
				} else {
					MwMail.SortAddressBook=FALSE;
				}
				continue;
			}

			if (MWQuickNCmp(V->Value[i], "SortKey=", 8)) {
				if (toupper(V->Value[i][8])=='F') {
					MwMail.SortFirstName=TRUE;
				} else {
					MwMail.SortFirstName=FALSE;
				}
				continue;
			}

			if (MWQuickNCmp(V->Value[i], "DisplayTextAttachments=", 23)) {
				if (V->Value[i][23]=='0') {
					MwMail.DisplayTextAttachments = FALSE;
					continue;
				}
				continue;
			}

			if (MWQuickNCmp(V->Value[i], "ANESend=", 8)) {
				MwMail.AttachmentNameEncodingSend = atol(V->Value[i] + 8);
				continue;
			}
		}
	}
	MDBDestroyValueStruct(V);

	MsgGetWorkDir(MwMail.WorkDir);
	strcat(MwMail.WorkDir, "/modweb");
	MsgMakePath(MwMail.WorkDir);

	/* Clean up our temp dir */
	DirEnt = XplOpenDir(MwMail.WorkDir);
	if (DirEnt) {
		while((Dir = XplReadDir(DirEnt)) != NULL) {
			ptr=strchr(Dir->d_name, '.');
			if (ptr) {
				ptr++;
				if (MWQuickCmp(ptr, COMPOSE_EXT_SUBJECT) || MWQuickCmp(ptr, COMPOSE_EXT_BODY) || MWQuickCmp(ptr, COMPOSE_EXT_MESSAGE) ||
				MWQuickCmp(ptr, COMPOSE_EXT_TO_LIST[COMPOSE_TO]) || MWQuickCmp(ptr, COMPOSE_EXT_TO_LIST[COMPOSE_CC]) || MWQuickCmp(ptr, COMPOSE_EXT_TO_LIST[COMPOSE_BCC]) ||
				isdigit(*ptr)) {
					sprintf(Path, "%s/%s", MwMail.WorkDir, Dir->d_name);
					unlink(Path);
				}
					
			}
		}
		XplCloseDir(DirEnt);
	}

	return(TRUE);
}

static void MWMAILMonitorThread(void)
{
	int	i;

	XplRenameThread(XplGetThreadID(), "MWMAIL Monitor");

	do {
		for (i = 0; (i < 12) && (MwMailUnloadOK == FALSE); i++) {
			XplDelay(5 * 1000);
		}

		MwMail.TimeStamp = time(NULL);
	} while (MwMailUnloadOK == FALSE);

	XplConsolePrintf("\rMWMAIL: Monitor thread done.\r\n");

	XplSafeDecrement(MwMail.LibraryUserCount);

	return;
}

EXPORT BOOL 
MWMAILInit(MWAPIArg)
{
	int						ccode;
	XplThreadID				ID;
	ModuleRegisterStruct	Register;

	if (MwMailUnloadOK == TRUE) {
	    MwMail.WorkDir[0] = '\0';
	    MwMail.LogHandle = NULL;
	    MwMail.DirectoryHandle = NULL;
	    MwMail.SortAddressBook = FALSE;
	    MwMail.SortFirstName = FALSE;
	    MwMail.DisplayTextAttachments = TRUE;
	    MwMail.AttachmentNameEncodingSend = ANE_RESTRICT_TO_7BIT;
	    MwMail.MaxMessageSize = 0;
	    MwMail.MaxRecipients = 0;
	    MwMail.NmapQAddress = 0;
	    MwMail.TimeStamp = 0;

		XplSafeWrite(MwMail.LibraryUserCount, 0);

		MwMail.DirectoryHandle = (MDBHandle)MsgInit();
		if (MwMail.DirectoryHandle) {
			MwMailUnloadOK = FALSE;

			MwMail.LogHandle = LoggerOpen("mwmail" /* DefaultNSureCertificate, DefaultNSurePrivateKey, 0, NULL */);
			if (MwMail.LogHandle == NULL) {
				XplConsolePrintf("MWMAIL: Unable to initialize Nsure Audit.  Logging disabled.\r\n");
			}

			MWAPISet;

			MwMail.TimeStamp = time(NULL);

			ReadMailConfiguration();

			Register.ModuleType = MODULE_TEMPLATE;
			Register.Module.Template.InitSession = MWMAILInitSession;
			Register.Module.Template.DestroySession = MWMAILDestroySession;
			Register.Module.Template.HandleURL = MWMAILHandleURL;
			Register.Module.Template.HandleTemplate = MWMAILHandleTemplate;
			Register.Module.Template.TokenRangeStart = MWMAIL_TOKEN_START;
			Register.Module.Template.TokenRangeEnd = MWMAIL_TOKEN_END;

			MWRegisterModule(&Register);

			XplBeginCountedThread(&ID, MWMAILMonitorThread, 16 * 1024, NULL, ccode, MwMail.LibraryUserCount);

			XplSafeIncrement(MwMail.LibraryUserCount);

			return(TRUE);
		}
	}

	return(FALSE);
}

EXPORT BOOL 
MWMAILShutdown(void)
{
	XplSafeDecrement(MwMail.LibraryUserCount);

	if (MwMailUnloadOK == FALSE) {
		MwMailUnloadOK = TRUE;

		/*	Make sure the library users are gone before beginning to 
			shutdown.	*/
		while (XplSafeRead(MwMail.LibraryUserCount)) {
			XplDelay(33);
		}

		/* Do any required cleanup */
		LoggerClose(MwMail.LogHandle);

#if defined(NETWARE) || defined(LIBC)
		XplSignalLocalSemaphore(MwMail.ShutdownSemaphore);	/* The signal will release main() */
		XplWaitOnLocalSemaphore(MwMail.ShutdownSemaphore);	/* The wait will wait until main() is gone */

		XplCloseLocalSemaphore(MwMail.ShutdownSemaphore);
#endif
	}

	return(FALSE);
}

/*
	Below are "stock" functions that are basically infrastructure.
	However, one might want to add initialization code to main
	and takedown code to the signal handler
*/
void 
MWMAILShutdownSigHandler(int Signal)
{
	int	oldTgid;

	oldTgid = XplSetThreadGroupID(MwMail.Tgid);

	if (MwMailUnloadOK == FALSE) {
		MwMailUnloadOK	= TRUE;

		/*	Make sure the library users are gone before beginning to 
			shutdown.	*/
		while (XplSafeRead(MwMail.LibraryUserCount) > 1) {
			XplDelay(33);
		}

		/* Do any required cleanup */
		LoggerClose(MwMail.LogHandle);

#if defined(NETWARE) || defined(LIBC)
		XplSignalLocalSemaphore(MwMail.ShutdownSemaphore);	/* The signal will release main() */
		XplWaitOnLocalSemaphore(MwMail.ShutdownSemaphore);	/* The wait will wait until main() is gone */

		XplCloseLocalSemaphore(MwMail.ShutdownSemaphore);
#endif
	}

	XplSetThreadGroupID(oldTgid);

	return;
}

int 
_NonAppCheckUnload(void)
{
	if (MwMailUnloadOK == FALSE) {
		XplConsolePrintf("\rThis NLM will automatically be unloaded by the thread that loaded it.\n");
		XplConsolePrintf("\rIt does not allow manual unloading.\n");
#if !defined(LIBC)	/* fixme - NetWare requirement */
		XplSetCurrentScreen(XplCreateScreen("System Console", 0));
#endif

		return(1);
	}

	return(0);
}

int main(int argc, char *argv[])
{
 /* init globals */
 MwMail.Tgid = XplGetThreadGroupID();

 XplRenameThread(GetThreadID(), "ModWeb Mail Module");
 XplOpenLocalSemaphore(MwMail.ShutdownSemaphore, 0);
 XplSignalHandler(MWMAILShutdownSigHandler);

	/*
		This will "park" the module 'til we get unloaded; 
		it would not be neccessary to do this on NetWare, 
		but to prevent from automatically exiting on Unix
		we need to keep main around...
	*/
	XplWaitOnLocalSemaphore(MwMail.ShutdownSemaphore);
	XplSignalLocalSemaphore(MwMail.ShutdownSemaphore);

	return(0);
}
