/****************************************************************************
 *
 * Copyright (c) 2001 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>

#include <mdb.h>
#include <msgapi.h>
#include <nmap.h>
#include <hulautil.h>

#include <modweb.h>		/* APIs */
#include <mwtempl.h>		/* Token definition */

#include "mwmail.h"


#define	MSGLIST_MESSAGE_ID			0
#define	MSGLIST_ACTION_DELETE		1
#define	MSGLIST_ACTION_MOVE			2
#define	MSGLIST_ACTION_COPY			3
#define	MSGLIST_ACTION_MARK_READ	4
#define	MSGLIST_ACTION_MARK_UNREAD	5
#define	MSGLIST_ACTION_PURGE			6
#define	MSGLIST_ACTION_ACCEPT		7
#define	MSGLIST_ACTION_DECLINE		8
#define	MSGLIST_ACTION_COMPLETE		9
#define	MSGLIST_ACTION_PURGEALL		10
#define	MSGLIST_ACTION_UNDELETE		11

FormCommandStruct MsgListCommands[]	= {
	{"Item.ID",				TRUE,		MSGLIST_MESSAGE_ID},
	{"Item.Delete",			FALSE,	MSGLIST_ACTION_DELETE},
	{"Item.UnDelete",		FALSE,	MSGLIST_ACTION_UNDELETE},
	{"Item.Move",			FALSE,	MSGLIST_ACTION_MOVE},
	{"Item.Copy",			FALSE,	MSGLIST_ACTION_COPY},
	{"Item.MarkRead",		FALSE,	MSGLIST_ACTION_MARK_READ},
	{"Item.MarkUnread",	FALSE,	MSGLIST_ACTION_MARK_UNREAD},
	{"Item.Purge",			FALSE,	MSGLIST_ACTION_PURGE},
	{"Item.Accept",		FALSE,	MSGLIST_ACTION_ACCEPT},
	{"Item.Decline",		FALSE,	MSGLIST_ACTION_DECLINE},
	{"Item.Complete",		FALSE,	MSGLIST_ACTION_COMPLETE},
	{"Item.PurgeAll",		FALSE,	MSGLIST_ACTION_PURGEALL},
	{NULL,						0, 0}
};

BOOL
MwMailFreeMessageSessionData(SessionStruct *Session, MailSessionStruct *MailSession)
{
	if (MailSession->IDList) {
		MemFree(MailSession->IDList);
		MailSession->IDList=NULL;
	}

	if (MailSession->UIDList) {
		MemFree(MailSession->UIDList);
		MailSession->UIDList=NULL;
	}

	if (MailSession->CachedHeader) {
		MWReleaseStreamData(MailSession->CachedHeader);
		MailSession->CachedHeader=NULL;
	}

	if (MailSession->CachedPartHeader) {
		MemFree(MailSession->CachedPartHeader);
		MailSession->CachedPartHeader=NULL;
	}

	return(TRUE);
}

static BOOL
LoadIDList(SessionStruct *Session, MailSessionStruct *MailSession)
{
	unsigned long	NMAPCount;
	unsigned long	State;
	unsigned long	ID;
	unsigned long	UID;
	int				ReplyInt;
	unsigned char	Reply[1024];

	MWSendNMAPServer(Session, "INFO\r\n", 6);
	ReplyInt=MWGetNMAPAnswer(Session, Reply, sizeof(Reply), TRUE);
	if (ReplyInt!=2002) {
		return(FALSE);
	}

	NMAPCount=atol(Reply);

	if (MailSession->IDList!=NULL) {
		MemFree(MailSession->IDList);
	}

	if (MailSession->UIDList!=NULL) {
		MemFree(MailSession->UIDList);
	}

	Session->NumOfMessages=0;
	MailSession->IDList=MemMalloc(sizeof(unsigned long)*(NMAPCount+1));
	MailSession->UIDList=MemMalloc(sizeof(unsigned long)*(NMAPCount+1));
	if (!MailSession->IDList || !MailSession->UIDList) {
		if (MailSession->IDList) {
			MemFree(MailSession->IDList);
		}
		if (MailSession->UIDList) {
			MemFree(MailSession->UIDList);
		}
		return(FALSE);
	}

	for (ID=1; ID<=NMAPCount; ID++) {
		MWGetNMAPAnswer(Session, Reply, sizeof(Reply), FALSE);
		sscanf(Reply, "%*u %*u %*u %*u %lu %*s %lu", &State, &UID);
		if (!(State & MSG_STATE_PURGED)) {
			MailSession->IDList[Session->NumOfMessages]=ID;
			MailSession->UIDList[Session->NumOfMessages]=UID;
			Session->NumOfMessages++;
		}
	}

	ReplyInt=MWGetNMAPAnswer(Session, Reply, sizeof(Reply), TRUE);
	if (ReplyInt!=1000) {
		Session->NumOfMessages=0;
		if (MailSession->IDList!=NULL) {
			MemFree(MailSession->IDList);
			MailSession->IDList = NULL;
		}
		if (MailSession->UIDList!=NULL) {
			MemFree(MailSession->UIDList);
			MailSession->UIDList = NULL;
		}
		return(FALSE);
	}
	return(TRUE);
}


unsigned long
MwMailUIDtoIDFunc(SessionStruct *Session, MailSessionStruct *MailSession, unsigned long UID)
{
	register unsigned long	Start;
	register unsigned long	End;
	register unsigned long	Middle;

	if (Session->NumOfMessages==0) {
		return(NO_UID);		/* Error */
	}

	Start=0;
	End=Session->NumOfMessages;
	Middle=End/2;

	do {
		if (MailSession->UIDList[Middle]>UID) {
			End=Middle;
		} else {
			Start=Middle;
		}
		Middle=Start+(End-Start)/2;
	} while ((End-Start)>1);

	if (UID==MailSession->UIDList[Start]) {
		return(Start+1);
	}

	return(NO_UID);		/* Error */
}

unsigned long
MwMailIDtoUIDFunc(SessionStruct *Session, MailSessionStruct *MailSession, unsigned long ID)
{
	if (ID>Session->NumOfMessages) {
		return(NO_UID);
	}
	return(MailSession->UIDList[ID-1]);
}



static unsigned char
*FindNonQuotedChar(unsigned char *String, unsigned char Ch)
{
	unsigned char	*p1;
	BOOL				inquote=FALSE;
	
	/* Now find if we have an address in <>, skip quoted strings */
	p1=String;
	while ((*p1!='\0' && *p1!=Ch && !inquote) || (*p1!='\0' && inquote)) {
		if (*p1=='"') {
			if (inquote) {
				inquote=0;
			} else {
				inquote=1;
			}
		} else if (*p1=='\\') {
			switch(*(p1+1)) {
				case '(':
				case ')': if (inquote) *p1=' '; break;
				case '\\':
				case '"': p1+=1; break;
				default: *p1=' '; break;
			}
		}
		p1++;
	}
	if (*p1!='\0') {
		return(p1);
	} else {
		return(NULL);
	}
}

static unsigned char
*FindNonQuotedEndChar(unsigned char *String, unsigned char Ch)
{
	unsigned char	*p1;
	BOOL				inquote=FALSE;
	
	/* Now find if we have an address in <>, skip quoted strings */
	p1=String;
	while ((*p1!='\0' && *p1!=Ch && !inquote) || (*p1!='\0' && inquote)) {
		if (*p1=='"') {
			if (inquote) {
				inquote=0;
			} else {
				inquote=1;
			}
		} else if (*p1=='\\') {
			if (*(p1+1)==Ch) {
				*p1=' ';
				p1+=1;
			} else {
				switch(*(p1+1)) {
					case '(': *p1=' '; break;
					case '\\':
					case '"': break;
					default: *p1=' '; break;
				}
			}
		}
		p1++;
	}
	if (*p1!='\0') {
		return(p1);
	} else {
		return(NULL);
	}
}

/* 
	Parse the addresses in a list; breaks an rfc822 address into name and address;
	The function either returns NULL, which means there are no more addresses
	or a pointer that should be fed to it as AddressLine next time around
*/

unsigned char *
MwMailParseRFC822Address(unsigned char *AddressLine, unsigned char *Name, unsigned int NameLen, unsigned char *Address, unsigned int AddressLen)
{
	unsigned char	*ptr, *s, *e, *p1, *p2;
	unsigned char	*start, *next;
	unsigned char	*inquote = NULL;
	unsigned char	ch;

	/* Name can be four forms: */
	/* 1: peter@novonyx.com									just an address */
	/* 2: Peter Dennis Bartok <peter@novonyx.com>	Name and address in angle brackets */
	/* 3: peter@novonyx.com (Peter Dennis Bartok)	Address and name in parenthesis */
	/* 4: "Peter \"Dennis\" Bartok" <peter@novonyx.com>	Name and address in angle brackets, possibly with escaped quotes*/

	start=AddressLine;
	next=NULL;

	do {
		p1 = start;

		while ((*p1!='\0' && *p1!=';' && *p1!=',' && inquote==NULL) || (*p1!='\0' && inquote!=NULL)) {
			if (*p1=='"') {
				if (inquote) {
					inquote=NULL;
				} else {
					inquote=p1;
				}
			} else if (*p1=='\\') {
				switch(*(p1+1)) {
					case '(':
					case ')': if (inquote) *p1=' '; break;
					case '\\':
					case '"': p1+=1; break;

					default: *p1=' '; break;
				}
			}
			p1++;
		}

		if (inquote!=NULL) {
			*inquote=' ';
			inquote=NULL;
			continue;
		}

		break;
	} while (TRUE);

//	inquote=NULL;
	if (*p1) {
		next=p1;
		*next='\0';
	} else {
		next=NULL;
	}

//	ptr=strchr(start, '<');
	ptr=FindNonQuotedChar(start, '<');

	if (!ptr) {
//		ptr=strchr(start, '(');
		ptr=FindNonQuotedChar(start, '(');
		if (!ptr) {				/* version one, email address only */
			if (Name) {
				/* Just display it.  Pull out non-escaped quotes, and escape chars.
					We don't care if the address is legal here, just make it look good */
				
				ptr = start;
				p1 = Name;
				while (ptr && *ptr && ((unsigned long)(p1 - Name + 1) < NameLen) ) {
					switch (*ptr) {
						case '"':	ptr++;	break;
						case '\\':	ptr++;	/* fall through */
						default:	{
							*p1 = *ptr;
							ptr++;
							p1++;
							break;
						}
					}
				}
				*p1 = '\0';
			}
			if (Address) {
				/* Just display it.  Pull out non-escaped quotes, and escape chars.
					We don't care if the address is legal here, just make it look good */

				ptr = start;
				p1 = Address;
				while (ptr && *ptr && ((unsigned long)(p1 - Address + 1) < AddressLen)) {
					switch (*ptr) {
						case '"':	ptr++;	break;
						case '\\':	ptr++;	/* fall through */
						default:	{
							*p1 = *ptr;
							ptr++;
							p1++;
							break;
						}
					}
				}
				*p1 = '\0';
			}
		} else {					/* version three, address (name) */
			s=ptr+1;
			e=FindNonQuotedEndChar(s, ')');
			p2=NULL;
			if (e) {
				*e='\0';
			}
			ptr=s-2;
			while (ptr>start && isspace(*ptr)) {
				ptr--;
			}
			ch=*(ptr+1);
			*(ptr+1)='\0';
			if (*s=='"') {
				p1=s+1;
				p2=strrchr(p1, '"');
				if (p2) {
					*p2='\0';
				}
			} else {
				p1=s;
			}
			if (Name) {
				HulaStrNCpy(Name, p1, NameLen);
			}
			if (Address) {
				HulaStrNCpy(Address, start, AddressLen);
			}
			if (e) {
				*e=')';
			}
			*(ptr+1)=ch;
			if (p2) {
				*p2='"';
			}
		}
	} else {						/* version two, name <address> */
		s=ptr+1;					/* s points to start of address, e to the end and p1 to the name */
		e=strchr(s, '>');
		if (e) {
			*e='\0';
		}
		p2=NULL;
		ptr=s-2;
		while (ptr>start && isspace(*ptr))
			ptr--;
		ch=*(ptr+1);
		*(ptr+1)='\0';
		if (*start=='"') {
			p1=start+1;
			p2=strrchr(p1, '"');
			if (p2)
				*p2='\0';
		} else {
			p1=start;
		}
		if (Name) {
			HulaStrNCpy(Name, p1, NameLen);
		}
		if (Address) {
			HulaStrNCpy(Address, s, AddressLen);
		}
		if (e) {
			*e='>';
		}
		*(ptr+1)=ch;
		if (p2) {
			*p2='"';
		}
	}

	if (next) {
		*next=',';
		next++;
		while (isspace(*next)) {
			next++;
		}
	}

	if (next && *next) {
		return(next);
	} else {
		return(NULL);
	}
}

/*
	Turn multi-line header fields into single lines
 */

unsigned long
MwMailMakeRFC822Header(unsigned char *Header)
{
	unsigned char	*ptr, *dest, *src;
	unsigned long	len;

	ptr = Header;
	
	while (*ptr != '\0') {

		if ((*ptr != '\n') || !isspace(*(ptr + 1))) {
			ptr++;
			continue;
		}

		/* We have our first fold */
		if ((ptr > Header) && *(ptr - 1) == '\r') {
			dest = ptr - 1;			
		} else {
			dest = ptr;
		}

		ptr += 2;
		while (*ptr && isspace(*ptr)) {
			ptr++;
		}

		/* the first character in any moved string needs to be a space to replace the fold */
		src = ptr - 1;
		*src = ' ';

		while (*ptr != '\0') {

			if ((*ptr != '\n') || !isspace(*(ptr + 1))) {
				ptr++;
				continue;
			}

			/* We have a fold */
			len = (ptr - src);
			
			if (len && *(ptr - 1) == '\r') {
				 len--;
			}

			memmove(dest, src, len);
			dest += len;

			ptr += 2;
			while (*ptr && isspace(*ptr)) {
				ptr++;
			}
			/* the first character in any moved string needs to be a space to replace the fold */
			src = ptr - 1;
			*src = ' ';
		}
		
		/* We hit the end of the string. Do the last memmove here */
		len = (ptr - src);
		memmove(dest, src, len);

		dest[len] = '\0';
		return((dest - Header) + len);
	}

	/* We only get here if there are no folds and we have done nothing to change the string */

	return(ptr - Header);
}


unsigned long
MwMailFindRFC822HeaderLine(unsigned char *Header, unsigned char *SearchFor, unsigned char **Result)
{
	unsigned char	*ptr;
	unsigned char	*NL;
	unsigned long	Len;

	if (!Header) {
		return(0);
	}

	ptr=Header;
	Len=strlen(SearchFor);

	while (TRUE) {
		NL=strchr(ptr, '\n');
		if (NL) {
			*NL='\0';
		}

		if (MWQuickNCmp(ptr, SearchFor, Len)) {
			ptr+=Len;
			while (*ptr!='\0' && isspace(*ptr)) {
				ptr++;
			}
			*Result=ptr;

			while (*ptr!='\0' && *ptr!='\r') {
				ptr++;
			}
			Len=ptr-*Result;
			if (NL) {
				*NL='\n';
			}
			return(Len);
		}
		
		if (NL) {
			*NL='\n';
			ptr=NL+1;
		} else {
			break;
		}
	}

	return(0);
}


/* Recalculate the view based on FirstMessageOnPage and the total number of messages.*/
/* Reverse logic accomodates NMAP ordering. Messages arriving to NMAP 0 - N          */
/* must be reversed (ie. N -> 0  (N-1) -> 1 (N - 2) -> 2 ...) for display purposes.  */

static void 
RecalculateMsgListNumbers(SessionStruct *Session, MailSessionStruct *MailSession)
{

	/* Recalculate the view based on FirstMessageOnPage and the total number of messages.*/
	/* Reverse logic accomodates NMAP ordering. Messages arriving to NMAP 0 - N */
	/* must be reversed (ie. N -> 0  (N-1) -> 1 (N - 2) -> 2 ...) for display purposes. */
	if (Session->NumOfMessages > Session->MessagesPerPage) {

		/* if first message on page can fit either in the first or last page */
		/* Handle this special case below separately */
		if (!((MailSession->FirstMessageOnPage <= Session->MessagesPerPage) &&
				(MailSession->FirstMessageOnPage >= (Session->NumOfMessages - Session->MessagesPerPage)))) {
				 
			if (MailSession->FirstMessageOnPage <= Session->MessagesPerPage){
				/* last page i.e 0 - 9 */
				MailSession->FirstMessageOnPage = Session->MessagesPerPage;
				MailSession->CurrentMsgListMessage = MailSession->FirstMessageOnPage;
				MailSession->LastMessageOnPage = 0;
			}  else {
				/* middle page */

				/* see if FirstMessageOnPage and LastMessageOnPage is now beyond the number of messages */
				if(MailSession->FirstMessageOnPage > Session->NumOfMessages){
					/* need to change one or more values */
					MailSession->LastMessageOnPage = 0;
					MailSession->FirstMessageOnPage = Session->MessagesPerPage;
				}
					
				MailSession->CurrentMsgListMessage = MailSession->FirstMessageOnPage;

			}
		} else {
			/* if first message on page can fit either in the first or last page */
			/* break tie based on last selection. */
			if (MailSession->LastMessageOnPage == Session->NumOfMessages){
				/* user was previously viewing first page */
				MailSession->LastMessageOnPage = Session->NumOfMessages;
				MailSession->FirstMessageOnPage = Session->NumOfMessages - Session->MessagesPerPage;

			} else {
				/* default to last page otherwise */
				MailSession->LastMessageOnPage = 0;
				MailSession->FirstMessageOnPage = Session->MessagesPerPage;
			}

		} 

	} else {
		/* everything fits on one page */
		MailSession->FirstMessageOnPage = MailSession->CurrentMsgListMessage = Session->NumOfMessages;
		MailSession->LastMessageOnPage = 0;
	}

	MailSession->CurrentMsgListMessage = MailSession->FirstMessageOnPage;

}


BOOL
MwMailRefreshFolder(SessionStruct *Session, MailSessionStruct *MailSession)
{
	unsigned char	Answer[BUFSIZE+1];
	int				ReplyInt;

	/* We should have the mailbox selected, let's just get the stats */

	if (!MailSession->MailboxSelected) {
		return(FALSE);
	}

	/* Give NMAP a chance to let us know the index has changed */
	if (!Session->NMAPChanged) {
		MWSendNMAPServer(Session, "NOOP\r\n", 6);
		ReplyInt = MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);
	}

	if (Session->NMAPChanged) {
		MwMailClearMIMECache(MailSession);

		MWSendNMAPServer(Session, "UPDA\r\n", 6);
		ReplyInt = MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);
		while (ReplyInt != 1000 && ReplyInt != -1) {
			ReplyInt=MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);
		}

		Session->NMAPChanged = FALSE;

		/* Force a cached header reset */
		MailSession->CachedMessageID = 0;
	}

	MWSendNMAPServer(Session, "STAT\r\n", 6);
	ReplyInt = MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);
	if (ReplyInt != 1000) {
		/* Get back into a known state */
		ModDebug("MwMailRefreshFolder(): NMAP error %s\n", Answer);
		MWSendNMAPServer(Session, "RSET MBOX\r\n", 11);

		ReplyInt = MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);
		Session->NumOfMessages = 0;
		MailSession->MailboxSelected = FALSE;
		return(FALSE);
	} else {
		unsigned long msgsBeforeUpdate = Session->NumOfMessages;

		/* need to check return code */
		if (!LoadIDList(Session, MailSession)) {
			/* Get back into a known state */
			ModDebug("MwMailRefreshFolder(): NMAP error %s\n", Answer);
			MWSendNMAPServer(Session, "RSET MBOX\r\n", 11);

			ReplyInt = MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);
			Session->NumOfMessages = 0;
			MailSession->MailboxSelected = FALSE;
			return(FALSE);
		}
		
	   if (Session->NumOfMessages < msgsBeforeUpdate) {
			/* provides the illusion that messages are in acsending order */
			/* makes more sense to user when deleting */
			if (MailSession->FirstMessageOnPage > (msgsBeforeUpdate - Session->NumOfMessages)) {
				MailSession->FirstMessageOnPage -= (msgsBeforeUpdate - Session->NumOfMessages);
				MailSession->LastMessageOnPage -= (msgsBeforeUpdate - Session->NumOfMessages);
			}
		}
		else if (Session->NumOfMessages > msgsBeforeUpdate) {
			/* If we were previously on the first page when new messages arrive */
			/* then we wish to stay on the first page to view the new messages */
			if (MailSession->FirstMessageOnPage == msgsBeforeUpdate) {
				MailSession->CurrentMsgListMessage = MailSession->FirstMessageOnPage = Session->NumOfMessages;
				if (Session->NumOfMessages > Session->MessagesPerPage) {
					MailSession->LastMessageOnPage = (Session->NumOfMessages - Session->MessagesPerPage);
				} else {
					MailSession->LastMessageOnPage = 0;
				}
			}
		}
			
		RecalculateMsgListNumbers(Session, MailSession);

	}

	return(TRUE);

}


BOOL
MwMailSelectFolder(unsigned long FolderID, SessionStruct *Session, MailSessionStruct *MailSession)
{
	unsigned char	Answer[BUFSIZE+1];
	int				ReplyInt;

	if (MailSession->CurrentFolder == FolderID && MailSession->MailboxSelected) {
		return(TRUE);
	}

	if (FolderID >= Session->FolderList->Used) {
		MailSession->MailboxSelected = FALSE;
		Session->NumOfMessages = 0;
		return(FALSE);
	}

	ReplyInt=snprintf(Answer, sizeof(Answer),"RSET MBOX\r\n");
	MWSendNMAPServer(Session, Answer, ReplyInt);
	ReplyInt = MWGetNMAPAnswer(Session, Answer, sizeof(Answer ), TRUE);

	ReplyInt = snprintf(Answer, sizeof(Answer), "MBOX %s\r\n", Session->FolderList->Value[FolderID] + 3);

	MWSendNMAPServer(Session, Answer, ReplyInt);
	ReplyInt = MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);

	if (ReplyInt == 1000 || ReplyInt == 1020) {
		/* We've got the mailbox selected, get the stats */
		MWSendNMAPServer(Session, "STAT\r\n", 6);
		ReplyInt = MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);
		if (ReplyInt!=1000) {
			/* Get back into a known state */
			ModDebug("MwMailSelectFolder(): NMAP error %s\n", Answer);
			MWSendNMAPServer(Session, "RSET MBOX\r\n", 11);
			ReplyInt = MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);
			Session->NumOfMessages = 0;
			MailSession->MailboxSelected = FALSE;
			return(FALSE);
		} else {
			if (MailSession->CachedHeader) {
				MWReleaseStreamData(MailSession->CachedHeader);
				MailSession->CachedHeader = NULL;
			}

			if (MailSession->CachedPartHeader) {
				MemFree(MailSession->CachedPartHeader);
				MailSession->CachedPartHeader = NULL;
			}
			MailSession->CachedMessageID = 0;
			MwMailClearMIMECacheData(MailSession);

		/* JNT need to check return code */
			if (!LoadIDList(Session, MailSession)) {
				/* Get back into a known state */
				ModDebug("MwMailSelectFolder(): NMAP error %s\n", Answer);
				MWSendNMAPServer(Session, "RSET MBOX\r\n", 11);
				ReplyInt = MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);
				Session->NumOfMessages = 0;
				MailSession->MailboxSelected = FALSE;
				return(FALSE);
			}
			MailSession->CurrentMsgListMessage=Session->NumOfMessages;

			MailSession->FirstMessageOnPage = MailSession->CurrentMsgListMessage;
			if (MailSession->CurrentMsgListMessage > Session->MessagesPerPage) {
				MailSession->LastMessageOnPage = MailSession->CurrentMsgListMessage - Session->MessagesPerPage;
			} else {
				MailSession->LastMessageOnPage = 0;
			}
			MailSession->MailboxSelected = TRUE;

			/* Set both CurrentFolders */
			MailSession->CurrentFolder = FolderID;
			Session->CurrentFolder = FolderID;
		}
	} else {
		MailSession->Error = ERROR_SELECTING_FOLDER__MBOX;
		ModDebug("MwMailSelectFolder(): NMAP error %s\n", Answer);
		Session->NumOfMessages = 0;
		MailSession->MailboxSelected = FALSE;
		return(FALSE);
	}

	return(TRUE);
}


BOOL
MwMailLoadMessage(ConnectionStruct *Client, unsigned long MessageID, SessionStruct *Session, MailSessionStruct *MailSession)
{
	unsigned char	Answer[BUFSIZE+1];
	int				ReplyInt;
	StreamStruct	NetworkInCodec;
	StreamStruct	*RFC1522Codec;
#ifdef HOTMAIL_CHARSET_FIX
	StreamStruct	*HotmailCodec;
#endif
	StreamStruct	*MemoryOutCodec;

	/* Is it already cached? */
	if (MessageID==MailSession->CachedMessageID && MailSession->CachedHeader) {
		ModDebug("MwMailLoadMessage(): Message %d already cached\n", (int)MessageID);
		return(TRUE);
	}

	if (MailSession->CachedHeader) {
		MWReleaseStreamData(MailSession->CachedHeader);
		MailSession->CachedHeader=NULL;
	}

	MailSession->CachedMessageID=MessageID;
	if (MessageID>Session->NumOfMessages) {
		return(FALSE);
	}

	MessageID--;		/* Our index is 0-based */

	/* Cache sizes & state */
	ReplyInt=snprintf(Answer, sizeof(Answer), "INFO %lu\r\n", MailSession->IDList[MessageID]);
	MWSendNMAPServer(Session, Answer, ReplyInt);
	ReplyInt=MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);
	if (ReplyInt==2001) {
		sscanf(Answer, "%*u %*u %lu %lu %lu %*s %*u %*u %lu",
														&MailSession->CachedMessageHeaderSize, 
														&MailSession->CachedMessageBodySize, 
														&MailSession->CachedMessageState,
														&MailSession->CachedMessageInternaldate);
	} else {
		MailSession->CachedHeader=NULL;
		ModDebug("NMAP error: %s\n", Answer);
		return(FALSE);		
	}
	MailSession->CachedMessageInternaldate+=Session->TimezoneOffset;

	/* Request the header */
	ReplyInt=snprintf(Answer, sizeof(Answer), "HEAD %lu\r\n", MailSession->IDList[MessageID]);
	MWSendNMAPServer(Session, Answer, ReplyInt);
	ReplyInt=MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);

	if (ReplyInt!=2020) {
		MailSession->CachedHeader=NULL;
		ModDebug("NMAP error: %s\n", Answer);
		return(FALSE);		
	}

	/* Read & Decode any RFC1522 header stuff */
	memset(&NetworkInCodec, 0, sizeof(StreamStruct));
	RFC1522Codec=MWGetStream(NULL, "RFC1522", FALSE);
#ifdef HOTMAIL_CHARSET_FIX
	HotmailCodec=MWGetStream(NULL, Session->Charset, FALSE);
#endif

	NetworkInCodec.StreamData=Session;
	NetworkInCodec.StreamLength=atoi(Answer);
	NetworkInCodec.Codec=MWStreamFromMWNMAP;

#ifdef HOTMAIL_CHARSET_FIX
	if (HotmailCodec) {
		NetworkInCodec.Next=HotmailCodec;
		HotmailCodec->Next=RFC1522Codec;
	} else 
#endif
	{
		NetworkInCodec.Next=RFC1522Codec;
	}

	MemoryOutCodec=MWGetStream(RFC1522Codec, NULL, FALSE);
	MemoryOutCodec->Codec=MWStreamToMemory;

	Client->NMAPProtected=TRUE;
#ifdef HOTMAIL_CHARSET_FIX
	NetworkInCodec.Codec(&NetworkInCodec, HotmailCodec);
#else
	NetworkInCodec.Codec(&NetworkInCodec, RFC1522Codec);
#endif
	Client->NMAPProtected=FALSE;

	if (MemoryOutCodec->StreamData == NULL) {
		MWReleaseStream(RFC1522Codec);
#ifdef HOTMAIL_CHARSET_FIX
		MWReleaseStream(HotmailCodec);
#endif
		MWReleaseStream(MemoryOutCodec);
		return(FALSE);
	}

	MailSession->CachedHeader=MemoryOutCodec->StreamData;
	MailSession->CachedMessageHeaderSize=MemoryOutCodec->StreamLength;
	MailSession->CachedHeader[MemoryOutCodec->StreamLength]='\0';

	MWReleaseStream(RFC1522Codec);
#ifdef HOTMAIL_CHARSET_FIX
	MWReleaseStream(HotmailCodec);
#endif
	MWReleaseStream(MemoryOutCodec);

	/* Grab confirmation of HEAD */
	ReplyInt=MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);
	if (ReplyInt!=1000) {
		if (MailSession->CachedHeader) {
			MWReleaseStreamData(MailSession->CachedHeader);
		}
		MailSession->CachedHeader=NULL;
		ModDebug("NMAP error: %s\n", Answer);
		return(FALSE);
	}
	MailSession->CachedMessageHeaderSize = MwMailMakeRFC822Header(MailSession->CachedHeader);
	MailSession->CachedSent=0;

	return(TRUE);
}

BOOL
MwMailProcessMsgListForm(ConnectionStruct *Client, SessionStruct *Session, MailSessionStruct *MailSession, unsigned long MsgListTemplate, unsigned long MsgMoveTemplate, unsigned long MsgCopyTemplate, unsigned long *NextTemplate)
{
	unsigned char	buffer[128];
	unsigned long	RetVal;
	unsigned long	Action = 0;
	unsigned long	Task = 0;
	unsigned long	i;
	BOOL				AddFlag = TRUE;
#if !defined(ABSOLUTE_ID)
	unsigned long	ID;
#endif

	Task = MwMailProcessListForm(Client, Session, MailSession, SESSION_VALUE_TYPE_MSGLIST, MsgListCommands);

	if (Session->ReadOnly) {
		return(FALSE);
	}

	if ((Task != 0 && MailSession->Value->Used > 0) || (Task == MSGLIST_ACTION_PURGEALL)) {
		switch (Task) {
			case MSGLIST_ACTION_MOVE: {
				/* Don't free the MDB Values; it contains the messages we're moving */
				*NextTemplate = MsgMoveTemplate;
				return(TRUE);
			}

			case MSGLIST_ACTION_COPY: {
				/* Don't free the MDB Values; it contains the messages we're moving */
				*NextTemplate = MsgCopyTemplate;
				return(TRUE);
			}

			case MSGLIST_ACTION_MARK_READ: {
				AddFlag = TRUE;
				Action = MSG_STATE_READ;
				break;
			}

			case MSGLIST_ACTION_MARK_UNREAD: {
				AddFlag = FALSE;
				Action = MSG_STATE_READ;
				break;
			}

			case MSGLIST_ACTION_PURGEALL:
			case MSGLIST_ACTION_PURGE: {
				AddFlag = TRUE;
				Action = MSG_STATE_PURGED;
				break;
			}

			case MSGLIST_ACTION_DELETE: {
				AddFlag = TRUE;
				if (!MailSession->PurgeDefault) {
					Action = MSG_STATE_DELETED;
				} else {
					Action = MSG_STATE_PURGED;
				}
				break;
			}

			case MSGLIST_ACTION_UNDELETE: {
				AddFlag = FALSE;
				Action = MSG_STATE_DELETED;
				break;
			}

			case MSGLIST_ACTION_ACCEPT: {
				URLStruct urlData ={0};

				for (i = 0; i < MailSession->Value->Used; i++) {
#if defined(ABSOLUTE_ID)
					MwMailHandleCalendarMessage(Client, Session, MailSession, MailSession->IDList[atol(MailSession->Value.Value[i]) - 1], TRUE);
#else
					if ((ID = UIDtoID(Session, MailSession, atol(MailSession->Value->Value[i]))) != NO_UID) {
//JNT - how did this work before!						MwMailHandleCalendarMessage(Client, Session, MailSession, MailSession->IDList[ID - 1], TRUE);
						MwMailHandleCalendarMessage(Client, Session, MailSession, ID, TRUE);
					}
#endif
				}

				MDBFreeValues(MailSession->Value);	/* Just to be nice */

				/* keep calendar in sync with accepted messages */
				urlData.Request = NM_REFRESH_CALENDAR;
				MWHandleNamedURL(Client,Session,&urlData);
				if (MwMailRefreshFolder(Session, MailSession)) {
					*NextTemplate = MsgListTemplate;
					return(TRUE);
				}
				return(FALSE);
			}

			case MSGLIST_ACTION_DECLINE: {
				URLStruct urlData ={0};

				for (i = 0; i < MailSession->Value->Used; i++) {
#if defined(ABSOLUTE_ID)
					MwMailHandleCalendarMessage(Client, Session, MailSession, MailSession->IDList[atol(MailSession->Value.Value[i]) - 1], FALSE);
#else
					if ((ID = UIDtoID(Session, MailSession, atol(MailSession->Value->Value[i]))) != NO_UID) {
//How did this work before !						MwMailHandleCalendarMessage(Client, Session, MailSession, MailSession->IDList[ID - 1], FALSE);
						MwMailHandleCalendarMessage(Client, Session, MailSession, ID, FALSE);
					}
#endif
				}

				MDBFreeValues(MailSession->Value);	/* Just to be nice */

				/* keep calendar in sync with accepted messages */
				urlData.Request = NM_REFRESH_CALENDAR;
				MWHandleNamedURL(Client,Session,&urlData);
				if(MwMailRefreshFolder(Session, MailSession)){
					*NextTemplate = MsgListTemplate;
					return(TRUE);
				}

				return(FALSE);

			}
		}

		for (i = 0; i < MailSession->Value->Used; i++) {
#if defined(ABSOLUTE_ID)
			RetVal = sprintf(buffer, sizeof(buffer), "%cFLG %lu %lu\r\n", AddFlag ? 'A' : 'D', MailSession->IDList[atol(MailSession->Value.Value[i]) - 1], Action);
			MWSendNMAPServer(Session, buffer, RetVal);
			/* We grab the response, but we wouldn't know how to handle an error, so we just continue... */
			MWGetNMAPAnswer(Session, buffer, sizeof(buffer), TRUE);
#else
			if ((ID = UIDtoID(Session, MailSession, atol(MailSession->Value->Value[i]))) != NO_UID) {
				RetVal = snprintf(buffer, sizeof(buffer), "%cFLG %lu %lu\r\n", AddFlag ? 'A' : 'D', MailSession->IDList[ID - 1], Action);
				MWSendNMAPServer(Session, buffer, RetVal);
				/* We grab the response, but we wouldn't know how to handle an error, so we just continue... */
				MWGetNMAPAnswer(Session, buffer, sizeof(buffer), TRUE);
			}
#endif
		}

		if (Task == MSGLIST_ACTION_PURGEALL) {
			MWSendNMAPServer(Session, "PURG\r\n", 6);
			/* We grab the response, but we wouldn't know how to handle an error, so we just continue... */
			MWGetNMAPAnswer(Session, buffer, sizeof(buffer), TRUE);
			Session->NMAPChanged = TRUE;   /* We need to reset the msglist because a PURGE has just occured! */
		}

		/* We need to let the cache know that things have changed */
		if (MwMailRefreshFolder(Session, MailSession)) {
			;
		} else {
			return(FALSE);
		}
			
	}

	MDBFreeValues(MailSession->Value);

	*NextTemplate = MsgListTemplate;
	return(TRUE);
}

BOOL
MwMailProcessFolderMessageMove(ConnectionStruct *Client, SessionStruct *Session, MailSessionStruct *MailSession, BOOL Move)
{
	unsigned long	ValueSize;
	unsigned char	buffer[128];
	unsigned long	Offset=0;
	unsigned long	i;
	unsigned long	ReplyInt;
	unsigned long	SelectedFolder = 0;
	unsigned long	ID;

	if ((MailSession->ValueType != SESSION_VALUE_TYPE_MSGLIST) || (MailSession->Value->Used == 0) || (Session->ReadOnly)) {
		return(FALSE);
	}

	/* First, read which folder we're moving stuff to */
	while (MWGetFormName(Client, buffer, 128)) {
		ValueSize=BUFSIZE-Offset;
		while (MWGetFormValue(Client, Client->Temp+Offset, &ValueSize)!=FORMFIELD_NEXT) {
			ValueSize+=Offset;
			if (MWQuickCmp("FolderName", buffer)) {
				SelectedFolder=atol(Client->Temp);
			}
			ValueSize=BUFSIZE-Offset;
		}
		Offset=0;
	}

	if (SelectedFolder==0) {
		MailSession->Error = ERROR_NO_FOLDER_SELECTED;
		return(FALSE);
	}
	MailSession->Error=0;

	if (SelectedFolder > Session->FolderList->Used) {
		return(FALSE);
	}

	for (i=0; i<MailSession->Value->Used; i++) {
#if defined(ABSOLUTE_ID)
		if ((ID=atol(MailSession->Value.Value[i])-1)<Session->NumOfMessages) {		/* } */
#else
		if ((ID=UIDtoID(Session, MailSession, atol(MailSession->Value->Value[i])))!=NO_UID) {
			ID--;
#endif
			ReplyInt=snprintf(Client->Temp, sizeof(Client->Temp), "COPY %lu %s\r\n", MailSession->IDList[ID], Session->FolderList->Value[SelectedFolder-1] + 3);
			MWSendNMAPServer(Session, Client->Temp, ReplyInt);
			ReplyInt=MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);
			if (ReplyInt==1000 && Move) {
				ReplyInt=snprintf(Client->Temp, sizeof(Client->Temp), "AFLG %lu %lu\r\n", MailSession->IDList[ID], (long unsigned int)MSG_STATE_PURGED);
				MWSendNMAPServer(Session, Client->Temp, ReplyInt);
				ReplyInt=MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);
				if (ReplyInt != 1000) {
					MailSession->Error = ERROR_DELETING_MESSAGE__AFLG;
				}
				Session->NMAPChanged = TRUE;   /* We need to reset the msglist because a PURGE has just occured! */
			} else {
				if (ReplyInt != 1000) {
					MailSession->Error = ERROR_COPYING_MESSAGE__COPY;
				}
			}
		}
	}
	MDBFreeValues(MailSession->Value);	/* Just to be nice */
	if (MwMailRefreshFolder(Session, MailSession)) {
		return(TRUE);
	}
	return(FALSE);
}
