/****************************************************************************
 *
 * Copyright (c) 1997-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>
#include <logger.h>
#include <mdb.h>
#include <nmap.h>
#include <nmlib.h>

#include "nmapdp.h"

/* Validate line
 * Accepts: pointer to candidate string to validate as a From header
 *          return pointer to end of date/time field
 *          return pointer to offset from t of time (hours of ``mmm dd hh
 *          return pointer to offset from t of time zone (if non-zero)
 * Returns: t,ti,zn set if valid From string, else ti is NIL
 */

#define VALID(s, x, ti, zn) \
            { \
                ti = 0; \
                if ((*s == 'F') && (s[1] == 'r') && (s[2] == 'o') && (s[3] == 'm') && \
                (s[4] == ' ')) { \
                    for (x = s + 5; *x && *x != '\n' && *x != '\r'; x++){}; \
                    if (x) { \
                        if (x - s >= 41) { \
                            for (zn = -1; x[zn] != ' '; zn--) {}; \
                            if ((x[zn-1] == 'm') && (x[zn-2] == 'o') && (x[zn-3] == 'r') \
                                    && (x[zn-4] == 'f') && (x[zn-5] == ' ') && (x[zn-6] == 'e') \
                                    && (x[zn-7] == 't') && (x[zn-8] == 'o') && (x[zn-9] == 'm') \
                                    && (x[zn-10] == 'e') && (x[zn-11] == 'r') && (x[zn-12] == ' ')) { \
                                x += zn - 12; \
                            } \
                        } \
                        if (x - s >= 27) { \
                            if (x[-5] == ' ') { \
                                if (x[-8] == ':') { \
                                    zn = 0,ti = -5; \
                                } else if (x[-9] == ' ') { \
                                    ti = zn = -9; \
                                } else if ((x[-11] == ' ') && ((x[-10]=='+') || (x[-10]=='-'))) { \
                                    ti = zn = -11; \
                                } \
                            } else if (x[-4] == ' ') { \
                                if (x[-9] == ' ') zn = -4,ti = -9; \
                            } else if (x[-6] == ' ') { \
                                if ((x[-11] == ' ') && ((x[-5] == '+') || (x[-5] == '-'))) { \
                                    zn = -6,ti = -11; \
                                } \
                            } \
                            if (ti && !((x[ti - 3] == ':') \
                                    && (x[ti -= ((x[ti - 6] == ':') ? 9 : 6)] == ' ') \
                                    && (x[ti - 3] == ' ') && (x[ti - 7] == ' ') \
                                    && (x[ti - 11] == ' '))) { \
                                ti = 0; \
                            } \
                        } \
                    } \
                } \
            }

#pragma pack(push, 4)
typedef struct {
    long MSize; /* Size of Message [Header+Body] */
    long SSize; /* Size of message to be sent */
    long HeadPos; /* Offset of Header-start */
    long BodyPos; /* Offset of Body-start */
    long AuthPos; /* Offset of Auth-string-start */
    unsigned int AuthLen; /* Length of Auth-string */
    long BodyLines; /* Lines in body */
    unsigned long State; /* State of message, see below */
    char UIDL[UIDLLEN]; /* Hash of message */
    long UID; /* folder unique id */
    time_t DateSent; /* receive time, seconds since '70 */
    time_t DateReceived; /* receive time, seconds since '70 */
    unsigned long UseSCMS; /* Use Single Copy Message Store */
    unsigned long SCMSID; /* ID of Single Copy */
    unsigned long RealStart; /* Maildrop message start pos */
    unsigned long RealSize; /* maildrop message size */
} MessageInfoStructV2;
#pragma pack(pop)

BOOL 
AddMsgInfo(NMAPClient *client, unsigned long increment)
{
    MessageInfoStruct *tmp;

    tmp = MemRealloc(client->mailbox.message.info, ((client->mailbox.message.allocated + increment) * sizeof(MessageInfoStruct)));
    if (tmp != NULL) {
        client->mailbox.message.info = tmp;
        client->mailbox.message.allocated += increment;

        return(TRUE);
    }

    LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_NMAP_OUT_OF_MEMORY, LOG_CRITICAL, 0, client->user, client->mailbox.name, (client->mailbox.message.allocated + increment) * sizeof(MessageInfoStruct), 0, NULL, 0);

    XplConsolePrintf("NMAPD: FATAL ERROR - Out of memory processing user %s, mailbox %s; request size: %lu bytes\r\n", client->user, client->mailbox.name, (client->mailbox.message.allocated + increment) * sizeof(MessageInfoStruct));

    return(FALSE);
}

static int
IsFromLine(NMAPClient *client, unsigned char *cp)
{
    int ti;
    int zn;
    unsigned char *x;

    if (strcmp(cp,"\r\n")==0) {
        client->mailbox.newline = TRUE;
        return(FALSE);
    } else if (!client->mailbox.newline) {
        return(FALSE);
    } else
        client->mailbox.newline = FALSE;

    VALID(cp, x, ti, zn);

    return(ti != 0);
}

static BOOL
CreateMBoxPath(const unsigned char *mailboxPath, unsigned char *user, unsigned char *mailbox, unsigned char *buffer1, unsigned char *buffer2) 
{
    unsigned char *ptr;
    unsigned char *name;
    unsigned char *buf1;
    unsigned char *buf2;
    BOOL retVal = TRUE;

    /* fixme - clean this up! */
    if (buffer1) {
        buf1 = NULL;
        buf2 = NULL;
    } else {
        buf2 = NULL;
        buf1 = MemMalloc(CONN_BUFSIZE + 1);
        buffer1 = buf1;
    }

    /* Make sure user directory exists */
    sprintf(buffer1, "%s/%s", mailboxPath, user);
    if (access(buffer1, 0) == 0) {
        ;
    } else {
        if (XplMakeDir(buffer1) == 0) {
            if (buffer2) {
                SetLongName(buffer1, buffer2);
            } else {
                buf2 = MemMalloc(CONN_BUFSIZE + 1);
                buffer2 = buf2;
                SetLongName(buffer1, buffer2);
            }
        } else {
            LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_FILE_OPEN_FAILURE, LOG_CRITICAL, 0, buffer1, __FILE__, __LINE__, 0, NULL, 0);
            retVal = FALSE;
            goto CleanUp;
        }
    }

    name = strrchr(mailbox, '/');
    if ((name != NULL) && (name != mailbox)) {
        BOOL More = TRUE;

        *name = '\0';

        ptr = strchr(mailbox, '/');

        while (More) {
            if (ptr) {
                *ptr = '\0';
            } else {
                More = FALSE;
            }

            sprintf(buffer1, "%s/%s/%s", mailboxPath, user, mailbox);
            if (access(buffer1, 0) != 0) {

                if (XplMakeDir(buffer1) != 0) {

                    LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_FILE_OPEN_FAILURE, LOG_CRITICAL, 0, buffer1, __FILE__, __LINE__, 0, NULL, 0);
                    retVal = FALSE;
                    goto CleanUp;
                }
                if (buffer2) {
                    SetLongName(buffer1, buffer2);
                } else {
                    buf2 = MemMalloc(CONN_BUFSIZE + 1);
                    buffer2 = buf2;
                    SetLongName(buffer1, buffer2);
                }
            }

            if (ptr) {
                *ptr = '/';
                ptr = strchr(ptr + 1, '/');
            }
        }

        *name = '/';
    }

CleanUp:
    if (buf1) {
        buffer1 = NULL;
        MemFree(buf1);
    }
    if (buf2) {
        buffer2 = NULL;
        MemFree(buf2);
    }

    return(retVal);
}

static void
PrintResources(NMAPClient *client, MDBValueStruct *vs, unsigned char *path, unsigned char *scratch, int skip, unsigned char *convertedPath, unsigned long recurse, NMAPClient *sclient)
{
    int len;
    unsigned long used;
    unsigned char *cPath;
    unsigned char *ptr;
    unsigned char *lBuffer;
    XplDir *dirEntryP;
    XplDir *dirP;
    BOOL result;
    FILE *index;

    lBuffer = MemMalloc(strlen(path) + 7);
    if (lBuffer != NULL) {
        strcpy(lBuffer, path);
    } else {
        return;
    }

    XplSetThreadGroupID(NMAP.id.load);
    cPath = convertedPath;
    *(cPath++) = '2';
    *(cPath++) = '0';
    *(cPath++) = '0';
    *(cPath++) = '2';
    *(cPath++) = '-';

    dirEntryP = XplOpenDir(lBuffer);
    if (dirEntryP != NULL) {
        while ((dirP = XplReadDir(dirEntryP)) != NULL) {
            cPath = convertedPath + 5;

            if ((dirP->d_attr & XPL_A_SUBDIR) && (dirP->d_name[0] != '.')) {
                /* Directory Entry */
                len = sprintf(scratch, "%s/%s.box", lBuffer, dirP->d_name);
                if (access(scratch, 0) != 0) {
                    ptr = scratch + len;
                    *(--ptr) = 'l';
                    *(--ptr) = 'a';
                    *(--ptr) = 'c';

                    if (access(scratch, 0) != 0) {
                        *(scratch + len - 4) = '\0';

                        *(cPath++) = 'H';
                        *(cPath++) = '0';
                        *(cPath++) = '0';

                        len = XPLDosToLong(scratch, skip, cPath, XPL_MAX_PATH - 8) + 4;
                        result = FALSE;

                        /*    Do we have a share with this name?    */
                        for (used = 0; used < vs->Used; used++) {
                            ptr = vs->Value[used];

                            if ((*(ptr++) != 'M') && (*ptr != 'C')) {
                                continue;
                            }

                            ptr = strrchr(++ptr, '\r');
                            if ((ptr == NULL) || (XplStrCaseCmp(++ptr, cPath) != 0)) {
                                continue;
                            }

                            result = TRUE;
                            break;
                        }

                        if (result == FALSE) {
                            len += 4;
                            convertedPath[len++] = '/';
                            convertedPath[len++] = '\r';
                            convertedPath[len++] = '\n';
                            convertedPath[len] = '\0';

                            ConnWrite(client->conn, convertedPath, len);
                        }
                    } else {
                        *(scratch + len - 4) = '\0';
                    }
                } else {
                    *(scratch + len - 4) = '\0';
                }

                if (recurse) {
                    PrintResources(client, vs, scratch, scratch, skip, convertedPath, recurse - 1, sclient);

                    /* We have to set it after the recursive call, since it gets reset at the end of printdir    */
                    XplSetThreadGroupID(NMAP.id.load);
                }

                continue;
            } else if (QuickCmp(dirP->d_name + strlen(dirP->d_name) - 4, ".box")) {
                sclient->mailbox.message.count = 0;
                sclient->mailbox.message.used = 0;

                sprintf(scratch, "%s/%s", lBuffer, dirP->d_name);
                len = XPLDosToLong(scratch, skip, sclient->mailbox.name, sizeof(sclient->mailbox.name)) - 4;
                sclient->mailbox.name[len] = '\0';

                XplSetThreadGroupID(NMAP.id.group);

                sprintf(scratch, "%s/%s/%s.idx", client->store, client->user, sclient->mailbox.name);

                if (ReadNLockAquire(&sclient->mailbox.lock, &client->userHash, client->user, sclient->mailbox.name, 0) == NLOCK_AQUIRED) {
                    index = fopen(scratch, "rb");
                    if (index != NULL) {
                        fseek(index, 0, SEEK_SET);
                        if (fgets(scratch, XPL_MAX_PATH, index)) {
                            if ((strncmp(scratch, NMAP_IDX_VERSION, 7) == 0) && (scratch[7] == '\r')) {
                                fgets(scratch, XPL_MAX_PATH, index);
                                sclient->mailbox.flags = atol(scratch);
                                fclose(index);

                                result = TRUE;
                            } else {
                                fclose(index);
                                result = ParseMaildrop(sclient);
                            }
                        } else {
                            fclose(index);
                            result = FALSE;
                        }
                    } else {
                        result = FALSE;
                    }
                    ReadNLockRelease(sclient->mailbox.lock);
                    sclient->mailbox.lock = NULL;                    
                } else {
                    result = FALSE;
                }

                if (result == FALSE) {
                    sclient->mailbox.flags = 0;
                }

                XplSetThreadGroupID(NMAP.id.load);

                *(cPath++) = 'M';

                if (!(sclient->mailbox.flags & RESOURCE_FLAG_SHARED)) {
                    *(cPath++) = '0';
                } else if (!(sclient->mailbox.flags & RESOURCE_FLAG_SHARED_PUBLIC)) {
                    *(cPath++) = '1';
                } else {
                    *(cPath++) = '3';
                }

                if (!(sclient->mailbox.flags & RESOURCE_FLAG_UNSUBSCRIBED)) {
                    *(cPath++) = '1';
                } else {
                    *(cPath++) = '0';
                }

                sprintf(scratch, "%s/%s", lBuffer, dirP->d_name);
                len = XPLDosToLong(scratch, skip, cPath, XPL_MAX_PATH - 8) + 4;
                convertedPath[len++] = '\r';
                convertedPath[len++] = '\n';
                convertedPath[len] = '\0';

                ConnWrite(client->conn, convertedPath, len);

                continue;
            } else if (QuickCmp(dirP->d_name + strlen(dirP->d_name) - 4, ".cal")) {
                sclient->calendar.entries.count = 0;
                sclient->calendar.entries.used = 0;

                sprintf(scratch, "%s/%s", lBuffer, dirP->d_name);
                len = XPLDosToLong(scratch, skip, sclient->calendar.name, sizeof(sclient->calendar.name)) - 4;
                sclient->calendar.name[len] = '\0';

                XplSetThreadGroupID(NMAP.id.group);

                sprintf(scratch, "%s/%s/%s.idc", client->store, client->user, sclient->calendar.name);

                if (ReadNLockAquire(&sclient->calendar.lock, &client->userHash, client->user, sclient->calendar.name, 0) == NLOCK_AQUIRED) {
                    index = fopen(scratch, "rb");
                    if (index != NULL) {
                        fseek(index, 0, SEEK_SET);
                        if (fgets(scratch, XPL_MAX_PATH, index)) {
                            if ((strncmp(scratch, NMAP_CAL_IDX_VERSION, 8) == 0) && (scratch[8] == '\r')){
                                fgets(scratch, XPL_MAX_PATH, index);
                                sclient->calendar.flags = atol(scratch);
                                fclose(index);

                                result = TRUE;
                            } else {
                                fclose(index);
                                result = ParseCalendar(sclient);
                            }
                        } else {
                            fclose(index);
                            result = FALSE;
                        }
                    } else {
                        result = FALSE;
                    }
                    ReadNLockRelease(sclient->calendar.lock);
                    sclient->calendar.lock = NULL;
                } else {
                    result = FALSE;
                }

                if (result == FALSE) {
                    sclient->calendar.flags = 0;
                }

                XplSetThreadGroupID(NMAP.id.load);

                *(cPath++) = 'C';

                if (!(sclient->calendar.flags & RESOURCE_FLAG_SHARED)) {
                    *(cPath++) = '0';
                } else if (!(sclient->calendar.flags & RESOURCE_FLAG_SHARED)) {
                    *(cPath++) = '1';
                } else {
                    *(cPath++) = '3';
                }

                if (!(sclient->calendar.flags & RESOURCE_FLAG_UNSUBSCRIBED)) {
                    *(cPath++) = '1';
                } else {
                    *(cPath++) = '0';
                }

                sprintf(scratch, "%s/%s", lBuffer, dirP->d_name);
                len = XPLDosToLong(scratch, skip, cPath, XPL_MAX_PATH - 8) + 4;
                convertedPath[len++] = '\r';
                convertedPath[len++] = '\n';
                convertedPath[len] = '\0';

                ConnWrite(client->conn, convertedPath, len);

                continue;
            }
        }                                                    

        XplCloseDir(dirEntryP);                            
    }                                                        

    XplSetThreadGroupID(NMAP.id.group);

    MemFree(lBuffer);

    return;
}

static void
PrintDir(NMAPClient *client, MDBValueStruct *vs, unsigned char *path, unsigned char *scratch, unsigned char *pattern, int skip, unsigned char *convertedPath, unsigned long recurse, NMAPClient *sclient)
{
    int len;
    int len2;
    unsigned long used;
    unsigned char *ptr;
    unsigned char *lBuffer;
    FILE *index;
    XplDir *dirEntryP;
    XplDir *dirP;
    BOOL result;

    lBuffer = MemMalloc(strlen(path) + 7);    /* "\*.*ck\0"    */
    if (lBuffer != NULL) {
        len = sprintf(lBuffer, "%s", path);
    } else {
        return;
    }

    XplSetThreadGroupID(NMAP.id.load);

    len2 = strlen(pattern);

    dirEntryP = XplOpenDir(lBuffer);
    if (dirEntryP != NULL) {
        while ((dirP = XplReadDir(dirEntryP)) != NULL) {
            if ((dirP->d_attr & XPL_A_SUBDIR) && (dirP->d_name[0] != '.')) {
                sprintf(scratch, "%s/%s.box", lBuffer, dirP->d_name);
                if (access(scratch, 0) != 0) {
                    result = FALSE;

                    sprintf(scratch, "%s/%s", lBuffer, dirP->d_name);
                    len = XPLDosToLong(scratch, skip, convertedPath, XPL_MAX_PATH + 1);

                    /*    Do we have a share with this name?    */
                    for (used = 0; used < vs->Used; used++) {
                        ptr = vs->Value[used];

                        if ((*(ptr++) != 'M') || (*ptr != 'B')) {
                            continue;
                        }

                        ptr = strrchr(++ptr, '\r');
                        if ((ptr == NULL) || (XplStrCaseCmp(++ptr, convertedPath) != 0)) {
                            continue;
                        }

                        result = TRUE;
                        break;
                    }

                    if (result == FALSE) {
                        if ((pattern[0] == '\0') || (XplStrNCaseCmp(convertedPath, pattern, len2) == 0)) {
                            convertedPath[len++] = '/';
                            convertedPath[len++] = '\r';
                            convertedPath[len++] = '\n';
                            convertedPath[len] = '\0';

                            ConnWrite(client->conn, "2002-", 5);
                            ConnWrite(client->conn, convertedPath, len);
                        }
                    }
                }

                if (recurse) {
                    sprintf(scratch, "%s/%s", lBuffer, dirP->d_name);
                    PrintDir(client, vs, scratch, scratch, pattern, skip, convertedPath, recurse - 1, sclient);

                    /* We have to set it after the recursive call, since it gets reset at the end of printdir    */
                    XplSetThreadGroupID(NMAP.id.load);
                }

                continue;
            } else if (QuickCmp(dirP->d_name + strlen(dirP->d_name) - 4, ".box")) {
                if (sclient == NULL) {
                    sprintf(scratch, "%s/%s", lBuffer, dirP->d_name);
                    len = XPLDosToLong(scratch, skip, convertedPath, XPL_MAX_PATH + 1) - 4;
                    convertedPath[len] = '\0';
                    if ((pattern[0] == '\0') || (XplStrNCaseCmp(convertedPath, pattern, len2) == 0)) { 
                        convertedPath[len++] = '\r';
                        convertedPath[len++] = '\n';
                        convertedPath[len] = '\0';

                        ConnWrite(client->conn, "2002-", 5);
                        ConnWrite(client->conn, convertedPath, len);
                    }

                    continue;
                } else {
                    sclient->mailbox.message.count = 0;
                    sclient->mailbox.message.used = 0;

                    sprintf(scratch, "%s/%s", lBuffer, dirP->d_name);
                    len = XPLDosToLong(scratch, skip, sclient->mailbox.name, sizeof(sclient->mailbox.name)) - 4;
                    sclient->mailbox.name[len] = '\0';

                    XplSetThreadGroupID(NMAP.id.group);

                    sprintf(scratch, "%s/%s/%s.idx", client->store, client->user, sclient->mailbox.name);

                    if (ReadNLockAquire(&sclient->mailbox.lock, &client->userHash, client->user, sclient->mailbox.name, 0) == NLOCK_AQUIRED) {
                        index = fopen(scratch, "rb");
                        if (index != NULL) {
                            fseek(index, 0, SEEK_SET);
                            if (fgets(scratch, XPL_MAX_PATH, index)) {
                                if ((strncmp(scratch, NMAP_IDX_VERSION, 7) == 0) && (scratch[7] == '\r')) {
                                    fgets(scratch, XPL_MAX_PATH, index);
                                    sclient->mailbox.flags = atol(scratch);
                                    fclose(index);
                                } else {
                                    fclose(index);
                                    if (ParseMaildrop(sclient) == FALSE) {
                                        sclient->mailbox.flags = 0;
                                    }
                                }
                            } else {
                                fclose(index);
                                if (ParseMaildrop(sclient) == FALSE) {
                                    sclient->mailbox.flags = 0;
                                }
                            }
                        } else {
                            sclient->mailbox.flags = 0;
                        }
                        ReadNLockRelease(sclient->mailbox.lock);
                        sclient->mailbox.lock = NULL;
                    } else {
                        sclient->mailbox.flags = 0;
                    }

                    XplSetThreadGroupID(NMAP.id.load);

                    if (!(sclient->mailbox.flags & RESOURCE_FLAG_UNSUBSCRIBED)) {
                        sprintf(scratch, "%s/%s", lBuffer, dirP->d_name);
                        len = XPLDosToLong(scratch, skip, convertedPath, XPL_MAX_PATH + 1) - 4;
                        convertedPath[len] = '\0';
                        if ((pattern[0] == '\0') || (XplStrNCaseCmp(convertedPath, pattern, len2) == 0)) { 
                            convertedPath[len++] = '\r';
                            convertedPath[len++] = '\n';
                            convertedPath[len] = '\0';

                            ConnWrite(client->conn, "2002-", 5);
                            ConnWrite(client->conn, convertedPath, len);
                        }

                        continue;
                    }

                    ptr = strrchr(sclient->mailbox.name, '/');
                    if (ptr == NULL) {                    
                        sprintf(scratch, "%s/%s", lBuffer, sclient->mailbox.name);
                    } else {
                        sprintf(scratch, "%s%s", lBuffer, ptr);
                    }

                    if ((access(scratch, 0) == 0) 
                            && ((pattern[0] == '\0') || (XplStrNCaseCmp(sclient->mailbox.name, pattern, len2) == 0))) {
                        sclient->mailbox.name[len++] = '/';
                        sclient->mailbox.name[len++] = '\r';
                        sclient->mailbox.name[len++] = '\n';
                        sclient->mailbox.name[len] = '\0';

                        ConnWrite(client->conn, "2002-", 5);
                        ConnWrite(client->conn, sclient->mailbox.name, len);
                    }

                    continue;
                }
            }
        }                                                    

        XplCloseDir(dirEntryP);                            
    }                                                        

    XplSetThreadGroupID(NMAP.id.group);
    MemFree(lBuffer);

    return;
}

__inline static long
CopyNonPurgedMessages(FILE *source, FILE *destination, MessageInfoStruct *msgInfo, unsigned long msgCount, unsigned char *buffer, unsigned long bufferSize)
{
    unsigned long i;
    unsigned long len;
    unsigned long count;
    unsigned long headPos;
    unsigned long overHead;
    long newCount;

    newCount = 0;

    for (i=0; i < msgCount; i++) {
        if (!msgInfo[i].UseSCMS) {
            if (!(msgInfo[i].State & MSG_STATE_PURGED)) {        /* Msg is deleted, throw it away */
                newCount++;
                count = msgInfo[i].MSize;
                if (fseek(source, msgInfo[i].HeadPos, SEEK_SET) == 0) {
                    /* Update index */
                     headPos = msgInfo[i].BodyPos-msgInfo[i].HeadPos;
                    overHead = msgInfo[i].AuthPos-msgInfo[i].HeadPos;
                    msgInfo[i].State &= ~MSG_STATE_RECENT;
                    msgInfo[i].HeadPos = ftell(destination);
                    msgInfo[i].BodyPos = msgInfo[i].HeadPos + headPos;
                    msgInfo[i].AuthPos = msgInfo[i].HeadPos + overHead;
                    while (count > 0) {
                        len = min(count, bufferSize);

                        if (fread(buffer, sizeof(unsigned char), len, source) == len) {
                            fwrite(buffer, sizeof(unsigned char), len, destination);
                            count -= len;
                            continue;
                        }

                        return(-1);
                    }
                    continue;
                }

                return(-1);
            }
        } else {
            if (!(msgInfo[i].State & MSG_STATE_PURGED)) {        /* Msg is deleted, throw it away */
                newCount++;
                count = msgInfo[i].RealSize;
                if (fseek(source, msgInfo[i].RealStart, SEEK_SET) == 0)    {
                    /* Update index */
                    msgInfo[i].State &= ~MSG_STATE_RECENT;
                    msgInfo[i].RealStart = ftell(destination);
                    while (count > 0) {
                        len = min(count, bufferSize);

                        if (fread(buffer, sizeof(unsigned char), len, source) == len) {
                            fwrite(buffer, sizeof(unsigned char), len, destination);
                            count -= len;
                            continue;
                        }
                        
                        return(-1);
                    }
                    continue;
                }

                return(-1);
            } else {
                /* QuotaDecrementUserValue(Client->user, Client->msgInfo[i].MSize); */
                SCMSDeleteMessage(msgInfo[i].SCMSID);
            }
        }
    }
    return(newCount);
}

BOOL
StoreMaildrop(NMAPClient *client)
{
    unsigned long i;
    unsigned long j;
    unsigned char pathIDX[XPL_MAX_PATH+1];
    unsigned char pathBOX[XPL_MAX_PATH+1];
    unsigned char pathTMP[XPL_MAX_PATH+1];
    FILE *index;
    FILE *fh;
    struct stat sb;

    if ((client->flags & NMAP_READONLY) == 0) {
        if ((client->user[0] != '\0') && (client->mailbox.name[0] != '\0')) {
            if (PurgeNLockAquire(client->mailbox.lock, 0) == NLOCK_AQUIRED) {

                sprintf(pathIDX, "%s/%s/%s.idx",client->store, client->user, client->mailbox.name);

                /* synchronize idx in memory with the idx on disk */
                if (client->flags & NMAP_OOBMESSAGES) {
                    GET_CHANGED_FLAGS(pathIDX, TAKE_PURGES, SILENT);
                }

                if (client->mailbox.changed) {
                    /* We have to save the maildrop and our index file */
                    sprintf(pathBOX, "%s/%s/%s.box", client->store, client->user, client->mailbox.name);
                    client->mailbox.fh = fopen(pathBOX, "rb");
                    if (client->mailbox.fh) {
                        sprintf(pathTMP, "%s/%s/%s.tmp",client->store, client->user, client->mailbox.name);
                        fh = fopen(pathTMP, "wb");
                        if (fh) {
                            unsigned long newCount;
                            unsigned long newSize;

                            SetLongName(pathTMP, client->path);

                            newCount = CopyNonPurgedMessages(client->mailbox.fh, fh, client->mailbox.message.info, client->mailbox.message.used, client->path, XPL_MAX_PATH);
                            if (newCount != 0xFFFFFFFF) {
                                /* Put the new mailbox in place */
                                fseek(fh, 0, SEEK_END);
                                newSize = ftell(fh);
                                if (!(NMAP.flushFlags & FLUSH_BOX_ON_PURGE)) {
                                    fclose(fh);
                                } else {
                                    XplFlushWrites(fh);
                                    fclose(fh);
                                }

                                fclose(client->mailbox.fh);
                                client->mailbox.fh = NULL;

                                if (unlink(pathBOX) == 0) {
                                    
                                    SetLongName(pathBOX, client->path);

                                    index = fopen(pathIDX, "wb");
                                    if (index) {

                                        SetLongName(pathIDX, client->path);
                                        fprintf(index, NMAP_IDX_VERSION"\r\n");
                                        fprintf(index, "%010lu\r\n", client->mailbox.flags);
                                        fprintf(index, "%010lu\r\n", newSize);
                                        fprintf(index, "%010lu\r\n", newCount);
                                        fprintf(index, "%010lu\r\n", client->mailbox.id);
                                        fprintf(index, "%010lu\r\n", client->mailbox.nextUID);

                                        for (i=0; i<client->mailbox.message.used; i++) {
                                            if (!(client->mailbox.message.info[i].State & MSG_STATE_PURGED))
                                                fwrite(&client->mailbox.message.info[i], sizeof(MessageInfoStruct), 1, index);
                                        }

                                        if (!(NMAP.flushFlags & FLUSH_BOX_ON_PURGE)) {
                                            fclose(index);
                                        } else {
                                            XplFlushWrites(index);
                                            fclose(index);
                                        }

                                        stat(pathIDX, &sb);
                                        client->mailbox.indexTime = sb.st_mtime;
                                        
                                        for (i = 0; i < 5; i++) {
                                            if (rename(pathTMP, pathBOX) == 0) {
                                                PurgeNLockRelease(client->mailbox.lock);
                                                client->mailbox.changed = FALSE;
                                                return(TRUE);
                                            }

                                            /* Add LogEvent */
                                            XplDelay(60);
                                        }

                                        /* Add LogEvent */
                                    } 

                                    /* Add LogEvent */        
                                    PurgeNLockRelease(client->mailbox.lock);
                                    return(FALSE);
                                }
                            }

                            unlink(pathTMP);
                        }

                        if (client->mailbox.fh) {
                            fclose(client->mailbox.fh);
                            client->mailbox.fh = NULL;
                        }
                    }
                } else {
                    /* Are there any recent flags that need cleared? */
                    for (i = 0; i < client->mailbox.message.used; i++) {
                        if ((client->mailbox.message.info[i].State & MSG_STATE_RECENT) == 0) {
                            continue;
                        }

                        /* We have some flags to clear */
                        index = fopen(pathIDX, "r+b");
                        if (index) {
                            SetLongName(pathIDX, client->path);

                            fseek(index, NMAP_IDX_HEADERSIZE + (i * sizeof(MessageInfoStruct)), SEEK_SET);

                            for (j = i; j < client->mailbox.message.used; j++) {
                                client->mailbox.message.info[j].State &= ~MSG_STATE_RECENT;
                                fwrite(&client->mailbox.message.info[j], sizeof(MessageInfoStruct), 1, index);
                            }

                            fclose(index);
                            stat(pathIDX, &sb);
                            client->mailbox.indexTime = sb.st_mtime;
                            PurgeNLockRelease(client->mailbox.lock);
                            return(TRUE);
                        }
                        break;
                    }

                    
                }

                PurgeNLockRelease(client->mailbox.lock);
                LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_WRITE_ERROR, LOG_WARNING, 0, client->mailbox.name, client->user, 0, 0, NULL, 0);
                return(FALSE);
            }

            /* could not get an exclusive lock on the folder; let the other guy do the purge */
            return(TRUE);
        }

        /* need user and resource name to purge */
        LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_WRITE_ERROR, LOG_WARNING, 0, client->mailbox.name, client->user, 0, 0, NULL, 0);
        return(FALSE);
    }

    /* read only mode can not purge */
    return(TRUE);
}

__inline static BOOL
WriteIndexHeader(NMAPClient *client, FILE *indexFH)
{
    if (fseek(client->mailbox.fh, 0, SEEK_END) == 0) {
        if (fprintf(indexFH, NMAP_IDX_VERSION"\r\n") > 0) {
            if (fprintf(indexFH, "%010lu\r\n", client->mailbox.flags) > 0) {
                if (fprintf(indexFH, "%010lu\r\n", ftell(client->mailbox.fh)) > 0) {
                    if (fprintf(indexFH, "%010lu\r\n", client->mailbox.message.used) > 0) {
                        if (fprintf(indexFH, "%010lu\r\n", client->mailbox.id) > 0) {
                            if (fprintf(indexFH, "%010lu\r\n", client->mailbox.nextUID) > 0) {
                                return(TRUE);
                            }
                        }
                    }
                }
            }
        }
    }
    return(FALSE);
} 

__inline static BOOL
WriteIndexFile(NMAPClient *client, unsigned char *path, unsigned long startRead)
{
    unsigned long i;
    MessageInfoStruct *msgInfo;
    FILE *index;

    if ((index = fopen(path, "r+b")) != NULL) {
        if (WriteIndexHeader(client, index)) {
            if (startRead > 0) {
                msgInfo = &client->mailbox.message.info[startRead];
                fseek(index, startRead * sizeof(MessageInfoStruct), SEEK_CUR);
            } else {
                msgInfo = client->mailbox.message.info;
            }
        } else {
            fclose(index);
            return(FALSE);
        }
    } else if ((index = fopen(path,"wb")) != NULL) {
        if (WriteIndexHeader(client, index)) {
            msgInfo = client->mailbox.message.info;
            startRead = 0;
        } else {
            fclose(index);
            return(FALSE);
        }
    } else {
        return(FALSE);
    }

    i = client->mailbox.message.used;

    while (i-- > startRead) {
        /* This is quite ugly, but so is IMAP which defines the "recent" behaviour" */
        if (!(msgInfo->State & MSG_STATE_RECENT)) {
            fwrite(msgInfo, sizeof(MessageInfoStruct), 1, index);
        } else {
            msgInfo->State &= ~MSG_STATE_RECENT;
            fwrite(msgInfo, sizeof(MessageInfoStruct), 1, index);
            msgInfo->State |= MSG_STATE_RECENT;
        }

        msgInfo++;
    }

    if (!(NMAP.flushFlags & FLUSH_IDX_ON_PARSE)) {
        fclose(index);
    } else {
        XplFlushWrites(index);
        fclose(index);
    }
    return(TRUE);
}

/* fixme - consolidation?

    Parse the old (NIMS 2.6 and older) index file
    This can only be called from ParseMaildrop
    A version 4 idx file will be written                */
static BOOL
ParseV2Maildrop(NMAPClient *client)
{
    int len;
    unsigned int AuthLen;
    unsigned long i;
    unsigned long HeadPos;
    unsigned long BodyPos;
    unsigned long MSize;
    unsigned long SSize;
    unsigned long BodyLines;
    unsigned long AuthPos;
    unsigned long StartRead;
    unsigned long IndexStartOffset;
    unsigned long RealStart;
    unsigned long RealSize;
    unsigned long previous = 0xFFFFFFFF;
    unsigned char *ptr;
    unsigned int month;
    unsigned char szMonth[20];
    unsigned char Path[XPL_MAX_PATH];
    unsigned char Buffer[CONN_BUFSIZE + 1];
    unsigned char UIDL[UIDLLEN];
    unsigned char HexDigest[33];
    unsigned char digest[16];
    struct tm datesent;
    struct tm datereceived;
    MD5_CTX mdContext;
    BOOL Escape;
    BOOL UseSCMS;
    BOOL Valid;
    BOOL InHeader = FALSE;
    unsigned long SCMSID;
    struct stat sb;
    FILE *Index;
    MDBValueStruct *UID;

    XplConsolePrintf("NMAPD: Updating user %s, mailbox %s, to NIMSv4 format\r\n", client->user, client->mailbox.name);

    if (client->mailbox.fh != NULL) {
        fclose(client->mailbox.fh);
    }

    sprintf(Path, "%s/%s/%s.box", client->store, client->user, client->mailbox.name);
    client->mailbox.fh = fopen(Path, "rb");
    if (client->mailbox.fh == NULL) {
        sprintf(Path, "%s/%s/%s.idx", client->store, client->user, client->mailbox.name);
        unlink(Path);

        return(TRUE);
    }

    stat(Path, &sb);
    client->mailbox.size = sb.st_size;

V2StartOver:
    sprintf(Path, "%s/%s/%s.idx", client->store, client->user, client->mailbox.name);
    Index = fopen(Path, "rb");

    client->mailbox.newSize = 0;
    client->mailbox.newCount = 0;

    if (Index) {
        fgets(Buffer, sizeof(Buffer), Index);
        XplConsolePrintf("NMAPD: First index line %s\r\n", Buffer);
        if (strncmp(Buffer, "NIMSv03", 7) == 0) {
            /*    Version 3 to Version 4 Update
            
                We are missing the mailbox flags
                We are missing the signature on each message information block
                
                The line in the buffer is the version string.
                The next line in the buffer is the mailbox size.    */
            client->mailbox.flags = 0;

            fgets(Buffer, sizeof(Buffer), Index);
            MSize = atol(Buffer);
        } else {
            ptr = Buffer;
            Valid = TRUE;

            /*    Version 2 to Version 4 Update
            
                We are missing the mailbox flags
                We are missing the signature on each message information block

                This line in the buffer is the mailbox size.    */
            while ((*ptr != '\0') && (*ptr != '\r') && (*ptr != '\n') && (ptr < (Buffer + 10))) {
                if (isdigit(*ptr)) {
                    ptr++;
                    continue;
                }

                Valid = FALSE;
                break;
            }

            if ((Valid) && ((MSize = atol(Buffer)) != 0)) {
                client->mailbox.flags = 0;
            } else {
                /*    Corrupt index file.    */
                fclose(Index);

                sprintf(Path, "%s/%s/%s.idx", client->store, client->user, client->mailbox.name);
                unlink(Path);

                LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_INDEX_CORRUPTION, LOG_ERROR, 0, client->user, client->mailbox.name, -3, 0, NULL, 0);

                goto V2StartOver;
            }
        }
        
        fgets(Buffer, sizeof(Buffer),Index);
        client->mailbox.message.count = atol(Buffer);

        fgets(Buffer, sizeof(Buffer), Index);
        client->mailbox.id = atol(Buffer);

        fgets(Buffer, sizeof(Buffer), Index);
        client->mailbox.nextUID = atol(Buffer);

        IndexStartOffset = ftell(Index);
        if (MSize > client->mailbox.size) {
            MemFree(client->mailbox.message.info);
            client->mailbox.message.info = NULL;

            client->mailbox.flags = 0;
            client->mailbox.message.count = 0;
            client->mailbox.message.allocated = 0;
            client->mailbox.message.used = 0;
        }

        if (client->mailbox.message.info == NULL) {
            client->mailbox.message.allocated = client->mailbox.message.count + 1;
            client->mailbox.message.info = MemMalloc(client->mailbox.message.allocated * sizeof(MessageInfoStruct));
            if (client->mailbox.message.info == NULL) {
                fclose(Index);
                fclose(client->mailbox.fh);
                client->mailbox.fh = NULL;

                return(FALSE);
            }

            client->mailbox.message.used = client->mailbox.message.count;
            StartRead = 0;
        } else {
            StartRead = client->mailbox.message.used;
            if (client->mailbox.message.count > client->mailbox.message.allocated) {
                if (AddMsgInfo(client, (client->mailbox.message.count - client->mailbox.message.allocated) + 128)) {
                    ;
                } else {
                    fclose(Index);
                    fclose(client->mailbox.fh);
                    client->mailbox.fh = NULL;

                    return(FALSE);
                }
            }

            client->mailbox.message.used = client->mailbox.message.count;
        }

        fseek(Index, IndexStartOffset + (StartRead * sizeof(MessageInfoStructV2)), SEEK_SET);
        for (i = StartRead; i < client->mailbox.message.count; i++) {
            client->mailbox.message.info[i].TailSig = NMAP_IDX_LINE_TAIL_SIGNATURE;

            if (fread(&(client->mailbox.message.info[i].MSize), sizeof(MessageInfoStructV2), 1, Index)) {
                if (!(client->mailbox.message.info[i].State & MSG_STATE_DELETED) && (client->mailbox.message.info[i].State & MSG_STATE_RECENT)) {
                    client->mailbox.newCount++;
                    client->mailbox.newSize += client->mailbox.message.info[i].SSize;
                }
            }
        }

        fclose(Index);
        
        if (MSize == client->mailbox.size) {        /* Box hasn't changed, still need to rewrite the .IDX file to newer version */
            goto RewriteV4IndexFile;
        }

        if (client->mailbox.message.count) {
            if (client->mailbox.message.info[client->mailbox.message.count-1].UseSCMS) {
                fseek(client->mailbox.fh, client->mailbox.message.info[client->mailbox.message.count-1].RealStart + client->mailbox.message.info[client->mailbox.message.count-1].RealSize, SEEK_SET);
            } else {
                fseek(client->mailbox.fh, client->mailbox.message.info[client->mailbox.message.count-1].HeadPos + client->mailbox.message.info[client->mailbox.message.count-1].MSize, SEEK_SET);
            }
            /* client->mailbox.nextUID=client->mailbox.message.info[client->mailbox.message.count-1].UID+1; */
        } else {
            fseek(client->mailbox.fh, 0, SEEK_SET);
        }

        client->mailbox.newline = TRUE;
    } else {
        UID = MDBCreateValueStruct(NMAP.handle.directory, NMAP.server.dn);

        fseek(client->mailbox.fh, 0, SEEK_SET);
        client->mailbox.flags = 0;
        client->mailbox.newSize = 0;
        client->mailbox.newCount = 0;
        client->mailbox.message.count = 0;
        client->mailbox.message.used = 0;
        client->mailbox.id = NMAP.universalCounter++;
        client->mailbox.nextUID = NMAP.universalCounter++;
        sprintf(Path, "%lu", NMAP.universalCounter);

        MDBAddValue(Path, UID);
        MDBWrite(MSGSRV_AGENT_NMAP, MSGSRV_A_UID, UID);
        MDBDestroyValueStruct(UID);
    }

    while (!feof(client->mailbox.fh)) {
        Buffer[0]='\0';
        if (fgets(Buffer, sizeof(Buffer), client->mailbox.fh)) {
            len = strlen(Buffer);
            if (IsFromLine(client, Buffer)) {
                if (previous != 0xFFFFFFFF) {
                    client->mailbox.message.info[previous].TailSig = NMAP_IDX_LINE_TAIL_SIGNATURE;

                    client->mailbox.message.info[previous].DateSent = mktime( &datesent);
                    client->mailbox.message.info[previous].DateReceived = mktime( &datereceived);
                    client->mailbox.message.info[previous].UseSCMS = UseSCMS;
                    client->mailbox.message.info[previous].SCMSID = SCMSID;
                    client->mailbox.message.info[previous].RealStart = RealStart;
                    client->mailbox.message.info[previous].RealSize = RealSize;

                    client->mailbox.message.info[previous].MSize = MSize;
                    client->mailbox.message.info[previous].SSize = SSize;
                    client->mailbox.message.info[previous].HeadPos = HeadPos;

                    if (BodyPos) {
                        client->mailbox.message.info[previous].BodyPos = BodyPos;
                    } else {
                        client->mailbox.message.info[previous].BodyPos = HeadPos + MSize;
                    }

                    client->mailbox.message.info[previous].BodyLines = BodyLines;
                    client->mailbox.message.info[previous].AuthPos = AuthPos;
                    client->mailbox.message.info[previous].AuthLen = AuthLen;
                    client->mailbox.message.info[previous].State = MSG_STATE_RECENT;
                    client->mailbox.message.info[previous].UID = client->mailbox.nextUID++;

                    if (Escape)
                        client->mailbox.message.info[previous].State |= MSG_STATE_ESCAPE;
                    if (UIDL[0]=='\0') {
                        client->mailbox.message.info[previous].State |= MSG_STATE_NEWUIDL;
                        strcpy(client->mailbox.message.info[previous].UIDL, HexDigest);
                    } else {
                        strcpy(client->mailbox.message.info[previous].UIDL, UIDL);
                    }

                    client->mailbox.newSize += SSize;
                    client->mailbox.newCount++;
                    client->mailbox.message.count++;
                }
                MD5_Init(&mdContext);
                MD5_Update(&mdContext, Buffer, len);
                InHeader=TRUE;

                if ((previous = client->mailbox.message.used++) < client->mailbox.message.allocated) {
                    HeadPos = ftell(client->mailbox.fh) - len;
                } else {
                    if (AddMsgInfo(client, 128)) {
                        ;
                    } else {
                        fclose(client->mailbox.fh);
                        client->mailbox.fh = NULL;

                        return(FALSE);
                    }

                    HeadPos = ftell(client->mailbox.fh) - len;
                }

                HeadPos=ftell(client->mailbox.fh)-len;
                RealStart=HeadPos;
                BodyPos=0;
                RealSize=0;
                BodyLines=0;
                SSize=0;
                Escape=FALSE;
                UseSCMS=FALSE;
                SCMSID=0;
                UIDL[0]='\0';

                /* Get the Date Received (From line) e.g. From jared@hula-spot.com Sat Feb 12 20:47 2005 */
                memset( &datereceived, 0, sizeof( struct tm));
                sscanf( 
                    Buffer, 
                    "%*s %*s %*s %s %d %*s %d", 
                    szMonth, &datereceived.tm_mday, &datereceived.tm_year);
                datereceived.tm_isdst = -1;
                for( month = 0; month < 12; month++ ) 
                {
                    if( 0 == XplStrNCaseCmp( szMonth, MonthsOfTheYear[ month], 3))
                    {
                        datereceived.tm_mon = month;
                        break;
                    }
                }
                datereceived.tm_min  = 0;
                datereceived.tm_hour = 0;
                datereceived.tm_sec  = 0;
                datereceived.tm_year -= 1900; /* adjust for mktime */

                MSize=len;
                RealSize=len;

                /* Check the Authenticated thingy    */
                AuthPos=ftell(client->mailbox.fh);
                fgets(Buffer, sizeof(Buffer), client->mailbox.fh);
                len=strlen(Buffer);
                AuthLen=len-13;            /* X-Auth-OK:_ / X-Auth-NO:_, 2 less for CR/LF  */
                RealSize+=len;
                if (Buffer[6]=='+') {
                    /* Check the SCMS reference            */
                    MSize+=len;
                    SSize+=len;
                    fgets(Buffer, sizeof(Buffer), client->mailbox.fh);
                    /* Get the ID, the line looks like this:    X-SCMS-ID: xxxxxxx */
                    if (Buffer[0]=='X' && Buffer[2]=='S' && Buffer[3]=='C') {
                        SCMSID=atol(Buffer+11);
                    }
                    len=strlen(Buffer);
                    MSize+=len;
                    SSize+=len;
                    UseSCMS=TRUE;
                }
            }

            if (!UseSCMS) {
                MSize+=len;
                SSize+=len;
            } else {
                RealSize+=len;
            }

            if (InHeader) {
                if (Buffer[0]==CR && Buffer[1]==LF) {
                    MD5_Final(digest, &mdContext);
                    for (i=0; i<16; i++)
                        sprintf(HexDigest+(i*2),"%02x",digest[i]);

                    InHeader=FALSE;
                    if (!UseSCMS) {
                        BodyPos=ftell(client->mailbox.fh);
                    } else {
                        FILE    *SCMS;
                        /* Since we use SCMS, we have to read the message from there and count it all up */
                        sprintf(Path, "%s/%x/%lx", NMAP.path.scms, (unsigned int)(SCMSID % 16), SCMSID);
                        SCMS=fopen(Path, "rb");
                        if (SCMS) {
                            HeadPos=0;
                            AuthPos=0;

                            MSize=0;
                            SSize=0;
                            while (!feof(SCMS)) {
                                if (fgets(Buffer, sizeof(Buffer), SCMS)) {
                                    len=strlen(Buffer);
                                    MSize+=len;
                                    SSize+=len;
                                    if (Buffer[0]==CR && Buffer[1]==LF) {
                                        break;
                                    }
                                }
                            }
                            /* Now in the body    */
                            BodyPos=ftell(SCMS);

                            while (!feof(SCMS)) {
                                if (fgets(Buffer, sizeof(Buffer), SCMS)) {
                                    if (Buffer[0]=='.')
                                        Escape=TRUE;
                                    len=strlen(Buffer);
                                    BodyLines++;
                                    MSize+=len;
                                    SSize+=len;
                                }
                            }
                            fclose(SCMS);
                        } else {
                            /* Behave as if it wasn't SCMS */
                            RealSize-=len;
                            MSize+=len;
                            SSize+=len;
                            UseSCMS=FALSE;
                            BodyPos=ftell(client->mailbox.fh);
                        }
                    }
                } else {
                    /* Date sent - from Date: line e.g. Date: Tue, 30 Sep 1997 10:20:49 -0600 */
                    if(XplStrNCaseCmp(Buffer, "date:", 5) == 0) {
                        memset( &datesent, 0, sizeof( struct tm));
                        sscanf( 
                            Buffer, 
                            "%*s %*s %d %s %d", /* %d %s %d",  */
                            &datesent.tm_mday, szMonth, &datesent.tm_year);
                        datesent.tm_isdst = -1;
                        for(month = 0; month < 12; month++) {
                            if(XplStrNCaseCmp(szMonth, MonthsOfTheYear[month], 3) == 0) {
                                datesent.tm_mon = month;
                                break;
                            }
                        }
                        datesent.tm_year -= 1900; /* adjust for mktime */
                    }
                    MD5_Update(&mdContext, Buffer, len);
                }
            } else {
                if (!UseSCMS) {
                    if (Buffer[0]=='.')
                        Escape=TRUE;
                    BodyLines++;
                }
            }
        }
    }
    if (previous != 0xFFFFFFFF) {
        client->mailbox.message.info[previous].TailSig = NMAP_IDX_LINE_TAIL_SIGNATURE;

        client->mailbox.message.info[previous].DateSent = mktime( &datesent);
        client->mailbox.message.info[previous].DateReceived = mktime( &datereceived);
        client->mailbox.message.info[previous].UseSCMS = UseSCMS;
        client->mailbox.message.info[previous].SCMSID = SCMSID;
        client->mailbox.message.info[previous].RealStart = RealStart;
        client->mailbox.message.info[previous].RealSize = RealSize;

        client->mailbox.message.info[previous].MSize=MSize;
        client->mailbox.message.info[previous].SSize=SSize;
        client->mailbox.message.info[previous].HeadPos=HeadPos;
        if (BodyPos) {
            client->mailbox.message.info[previous].BodyPos=BodyPos;
        } else {
            client->mailbox.message.info[previous].BodyPos=HeadPos+MSize;
        }
        client->mailbox.message.info[previous].BodyLines=BodyLines;
        client->mailbox.message.info[previous].AuthPos=AuthPos;
        client->mailbox.message.info[previous].AuthLen=AuthLen;
        client->mailbox.message.info[previous].State=MSG_STATE_RECENT;
        client->mailbox.message.info[previous].UID=client->mailbox.nextUID++;
        if (Escape)
            client->mailbox.message.info[previous].State |= MSG_STATE_ESCAPE;
        if (UIDL[0]=='\0') {
            client->mailbox.message.info[previous].State |= MSG_STATE_NEWUIDL;
            strcpy(client->mailbox.message.info[previous].UIDL,HexDigest);
        } else {
            strcpy(client->mailbox.message.info[previous].UIDL,UIDL);
        }
        client->mailbox.newSize+=SSize;
        client->mailbox.newCount++;
        client->mailbox.message.count++;
    }

RewriteV4IndexFile:
    /* Write changed index file */
    if (!(client->flags & NMAP_READONLY)) {
        sprintf(Path, "%s/%s/%s.idx", client->store, client->user, client->mailbox.name);
        Index = fopen(Path,"wb");
        if (Index != NULL) {
            fseek(client->mailbox.fh, 0, SEEK_END);
            fprintf(Index, NMAP_IDX_VERSION"\r\n");
            fprintf(Index, "%010lu\r\n", client->mailbox.flags);
            fprintf(Index, "%010lu\r\n", ftell(client->mailbox.fh));
            fprintf(Index, "%010lu\r\n", client->mailbox.message.used);
            fprintf(Index, "%010lu\r\n", client->mailbox.id);
            fprintf(Index, "%010lu\r\n", client->mailbox.nextUID);

            for (i = 0; i < client->mailbox.message.used; i++) {
                fwrite(&client->mailbox.message.info[i], sizeof(MessageInfoStruct), 1, Index);
            }
            fclose(Index);
        } else {
            fclose(client->mailbox.fh);
            client->mailbox.fh = NULL;

            return(FALSE);
        }
    }

    fclose(client->mailbox.fh);
    client->mailbox.fh = NULL;

    return(TRUE);    
}

/* fixme - consolidation? */
BOOL  
ParseMaildrop(NMAPClient *client)
{
    int len;
    unsigned int authLen;
    unsigned long i;
    unsigned long lastParsedDropSize;
    unsigned long scmsID;
    unsigned long headPos;
    unsigned long bodyPos;
    unsigned long bodyLines;
    unsigned long authPos;
    unsigned long startRead;
    unsigned long indexStartOffset;
    unsigned long realStart;
    unsigned long realSize;
    unsigned long scmsSize;
    unsigned long privateLineLen;
    unsigned long msgState;
    unsigned long msgTime;
    unsigned long estCount;
    unsigned long previous;
    unsigned char path[XPL_MAX_PATH];
    unsigned char buffer[CONN_BUFSIZE+1];
    unsigned char uidL[UIDLLEN];
    unsigned char hexDigest[33];
    struct tm datesent;
    MD5_CTX mdContext;
    BOOL escape;
    BOOL useSCMS;
    BOOL inHeader = FALSE;
    BOOL flagsSynced = TRUE;
    struct stat sb;
    FILE *index;

    if ((client->user[0] != '\0') && (client->mailbox.name[0] != '\0'))    {
        previous = 0xFFFFFFFF;
    } else {
        return(FALSE);
    }

V4StartOver:
    if (client->mailbox.fh) {
        fclose(client->mailbox.fh);
        client->mailbox.fh = NULL;
    }

    sprintf(path, "%s/%s/%s.box", client->store, client->user, client->mailbox.name);
    client->mailbox.fh = fopen(path, "rb");
    if (client->mailbox.fh != NULL) {
        /* Store new size; remember old size for index check */
        stat(path, &sb);
        client->mailbox.size = sb.st_size;
        client->mailbox.newSize = 0;
        client->mailbox.newCount = 0;
    } else {
        /*    May not have a mailbox yet.    */
        return(FALSE);
    }

    sprintf(path, "%s/%s/%s.idx", client->store, client->user, client->mailbox.name);
    index = fopen(path, "rb");
    if (index != NULL) {
        buffer[6] = '\0';
        fgets(buffer, sizeof(buffer), index);
        if ((buffer[0] == 'N') && (buffer[6] == '4')) {
            fgets(buffer, sizeof(buffer), index);
            client->mailbox.flags = atol(buffer);

            fgets(buffer, sizeof(buffer), index);
            lastParsedDropSize = atol(buffer);

            fgets(buffer, sizeof(buffer), index);
            client->mailbox.message.count = atol(buffer);

            fgets(buffer, sizeof(buffer), index);
            client->mailbox.id = atol(buffer);

            fgets(buffer, sizeof(buffer), index);
            client->mailbox.nextUID = atol(buffer);

            if (ftell(index) == NMAP_IDX_HEADERSIZE) {
                indexStartOffset = NMAP_IDX_HEADERSIZE;
            } else {
                fclose(index);
                index = NULL;

                unlink(path);

                LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_INDEX_CORRUPTION, LOG_ERROR, 0, client->user, client->mailbox.name, -1, 0, NULL, 0);

                goto V4StartOver;
            }
        } else {
            /* We didn't find our current IDX format, parse somewhere else and start over */
            if (feof(index)) {
                /* Probably 0 byte index file; remove it and start over */
                fclose(index);

                unlink(path);

                goto V4StartOver;
            }

            fclose(index);
            fclose(client->mailbox.fh);
            client->mailbox.fh = NULL;

            if (ParseV2Maildrop(client)) {
                client->mailbox.message.count = 0;
                client->mailbox.message.used = 0;

                goto V4StartOver;
            }

            /*    We could not process, or recreate, the index file.  Help!    */
            ConnWrite(client->conn, MSG5004INTERNALERR, sizeof(MSG5004INTERNALERR) - 1);

            XplConsolePrintf("NMAPD: Unable to use index file %s for user %s; header corrupted.\r\n", client->mailbox.name, client->user);
            LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_INDEX_CORRUPTION, LOG_ERROR, 0, client->user, client->mailbox.name, -1, 0, NULL, 0);

            FreeClientData(client);
            return(FALSE);
        }

        if (client->mailbox.size >= lastParsedDropSize) {
            if (client->mailbox.message.info == NULL) {
                estCount = client->mailbox.message.count + ((client->mailbox.size - lastParsedDropSize) >> 12) + 36;

                client->mailbox.message.info = MemMalloc(estCount * sizeof(MessageInfoStruct));
                if (client->mailbox.message.info != NULL) {
                    client->mailbox.message.allocated = estCount;
                    client->mailbox.message.used = client->mailbox.message.count;
                    startRead = 0;
                } else {
                    ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1);

                    fclose(index);
                    fclose(client->mailbox.fh);
                    client->mailbox.fh = NULL;

                    FreeClientData(client);
                    return(FALSE);
                }
            } else {
                startRead = client->mailbox.message.used;

                if (client->mailbox.message.count <= client->mailbox.message.allocated) {
                    estCount = ((client->mailbox.size - lastParsedDropSize) >> 12) + 36;
                } else {
                    estCount = (client->mailbox.message.count - client->mailbox.message.allocated) + ((client->mailbox.size - lastParsedDropSize) >> 12) + 36;
                    if (AddMsgInfo(client, estCount)) {
                        ;
                    } else {
                        ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1);

                        fclose(index);
                        fclose(client->mailbox.fh);
                        client->mailbox.fh = NULL;

                        FreeClientData(client);
                        return(FALSE);
                    }
                }

                client->mailbox.message.used = client->mailbox.message.count;
            }
        } else {
            estCount = (client->mailbox.size >> 12) + 36;

            if (client->mailbox.message.info != NULL) {
                MemFree(client->mailbox.message.info);
            }

            client->mailbox.message.count = 0;
            client->mailbox.message.used = 0;

            client->mailbox.message.info = MemMalloc(estCount * sizeof(MessageInfoStruct));
            if (client->mailbox.message.info != NULL) {
                client->mailbox.message.allocated = estCount;
                startRead = 0;
            } else {
                ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1);

                fclose(index);
                fclose(client->mailbox.fh);
                client->mailbox.fh = NULL;

                FreeClientData(client);
                return(FALSE);
            }
        }

        stat(path, &sb);
        /* client->mailbox.indexTime will only be zero when no index file has not been read yet.  */
        /* In this case, the entire thing will be    read and disk and RAM will be in sync */
        if ((client->mailbox.indexTime != 0) && (client->mailbox.indexTime != sb.st_mtime)) {
            flagsSynced = FALSE;
        }

        fseek(index, indexStartOffset + (startRead * sizeof(MessageInfoStruct)), SEEK_SET);
        if (sb.st_mtime >= NMAP.time.startup) {
            MessageInfoStruct    *msgInfo;
            
            for (i = startRead, msgInfo = &(client->mailbox.message.info[startRead]); i < client->mailbox.message.count; i++, msgInfo++) {
                fread(msgInfo, sizeof(MessageInfoStruct), 1, index);
            }
        } else {
            MessageInfoStruct    *msgInfo;
            
            for (i = startRead, msgInfo = &(client->mailbox.message.info[startRead]); i < client->mailbox.message.count; i++, msgInfo++) {
                fread(msgInfo, sizeof(MessageInfoStruct), 1, index);

                if (msgInfo->State & MSG_STATE_PURGED) {
                    client->mailbox.changed = TRUE;
                }

                if (msgInfo->TailSig == NMAP_IDX_LINE_TAIL_SIGNATURE) {
                    continue;
                }

                /*    This message information block is invalid; toss the index and restart.    */
                client->mailbox.message.count = 0;
                client->mailbox.message.used = 0;

                fclose(index);

                unlink(path);

                XplConsolePrintf("NMAPD: Unable to use index file %s for user %s; entry %d corrupted.\r\n", client->mailbox.name, client->user, (int)i);
                LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_INDEX_CORRUPTION, LOG_ERROR, 0, client->user, client->mailbox.name, i, 0, NULL, 0);

                goto V4StartOver;
            }
        }

        fclose(index);
        
        if (lastParsedDropSize == client->mailbox.size) {
            /* Box hasn't changed, just read the file and return */
            fclose(client->mailbox.fh);
            client->mailbox.fh = NULL;
            return(TRUE);
        }

        if (client->mailbox.message.count) {
            if (client->mailbox.message.info[client->mailbox.message.count - 1].UseSCMS) {
                fseek(client->mailbox.fh, client->mailbox.message.info[client->mailbox.message.count - 1].RealStart + client->mailbox.message.info[client->mailbox.message.count - 1].RealSize, SEEK_SET);
            } else {
                fseek(client->mailbox.fh, client->mailbox.message.info[client->mailbox.message.count - 1].HeadPos + client->mailbox.message.info[client->mailbox.message.count - 1].MSize, SEEK_SET);
            }

            /* client->mailbox.nextUID = client->mailbox.message.info[client->mailbox.message.count-1].UID + 1; */
        } else {
            fseek(client->mailbox.fh, 0, SEEK_SET);
        }

        client->mailbox.newline = TRUE;
    } else {
        MDBValueStruct    *UID;

        fseek(client->mailbox.fh, 0, SEEK_SET);

        estCount = (client->mailbox.size >> 12) + 36;

        /*    Mailboxes are subscribed by default, if no index file is found for the 
            mailbox, it is considered subscribed.    */
        client->mailbox.flags = 0;

        startRead = 0;
        client->mailbox.newSize = 0;
        client->mailbox.newCount = 0;
        client->mailbox.message.count = 0;
        client->mailbox.message.used = 0;
        client->mailbox.id = NMAP.universalCounter++;
        client->mailbox.nextUID = NMAP.universalCounter++;

        sprintf(buffer, "%lu", NMAP.universalCounter);

        UID = MDBCreateValueStruct(NMAP.handle.directory, NMAP.server.dn);
        MDBAddValue(buffer, UID);
        MDBWrite(MSGSRV_AGENT_NMAP, MSGSRV_A_UID, UID);
        MDBDestroyValueStruct(UID);
    }

    while (!feof(client->mailbox.fh) && (NMAP.state < NMAP_STOPPING)) {
        unsigned int    month;
        unsigned char    szMonth[20];

        if (fgets(buffer, sizeof(buffer), client->mailbox.fh)) {
            len = strlen(buffer);
            if (len && buffer[len - 1] == '\n') {
                ;
            } else {
                LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_MBOX_CORRUPTION, LOG_ERROR, 0, client->user, client->mailbox.name, previous + 1, 0, NULL, 0);
                /*
                XplConsolePrintf("User %s has an invalid header in mailbox: %s message: %d\n", client->user, client->mailbox.name, previous);
                */
            }
            if (IsFromLine(client, buffer)) {
                if (previous != 0xFFFFFFFF) {
                    client->mailbox.message.info[previous].TailSig = NMAP_IDX_LINE_TAIL_SIGNATURE;

                    client->mailbox.message.info[previous].DateSent = mktime(&datesent);
                    client->mailbox.message.info[previous].DateReceived = msgTime;
                    client->mailbox.message.info[previous].UseSCMS = useSCMS;
                    client->mailbox.message.info[previous].SCMSID = scmsID;
                    client->mailbox.message.info[previous].RealStart = realStart;
                    client->mailbox.message.info[previous].RealSize = realSize;
                    if (!useSCMS) {
                        client->mailbox.message.info[previous].MSize                            = realSize;
                        client->mailbox.newSize += client->mailbox.message.info[previous].SSize    = realSize - privateLineLen;
                        if (bodyPos) {
                            client->mailbox.message.info[previous].BodyPos                        = bodyPos;
                        } else {
                            client->mailbox.message.info[previous].BodyPos                        = headPos + realSize;
                        }
                    } else {
                        client->mailbox.message.info[previous].MSize                            = scmsSize;
                        client->mailbox.newSize += client->mailbox.message.info[previous].SSize    = scmsSize;
                        if (bodyPos) {
                            client->mailbox.message.info[previous].BodyPos                        = bodyPos;
                        } else {
                            client->mailbox.message.info[previous].BodyPos                        = scmsSize;
                        }
                    }
                    client->mailbox.message.info[previous].HeadPos = headPos;
                    client->mailbox.message.info[previous].BodyLines = bodyLines;
                    client->mailbox.message.info[previous].AuthPos = authPos;
                    client->mailbox.message.info[previous].AuthLen = authLen;
                    client->mailbox.message.info[previous].State = msgState;
                    client->mailbox.message.info[previous].UID = client->mailbox.nextUID++;

                    if (escape) {
                        client->mailbox.message.info[previous].State |= MSG_STATE_ESCAPE;
                    }

                    if (uidL[0] == '\0') {
                        client->mailbox.message.info[previous].State |= MSG_STATE_NEWUIDL;
                        strcpy(client->mailbox.message.info[previous].UIDL, hexDigest);
                    } else {
                        strcpy(client->mailbox.message.info[previous].UIDL, uidL);
                    }

                    
                    client->mailbox.newCount++;
                    client->mailbox.message.count++;
                }

                /* Generate the "internal" date for the message; UIDL & MsgTime ab-used */
                memset(&datesent, 0, sizeof(struct tm));
                sscanf(buffer, "From %*s %*s %s %d %d:%d %d", uidL, &datesent.tm_mday, &datesent.tm_hour, &datesent.tm_min, &datesent.tm_year);
                for (msgTime = 0; msgTime < 12; msgTime++) {
                    if (QuickNCmp(uidL, MonthsOfTheYear[msgTime], 3)) {
                        datesent.tm_mon = msgTime;
                        break;
                    }
                }

                msgTime = MsgGetUTC(datesent.tm_mday, datesent.tm_mon+1, datesent.tm_year, datesent.tm_hour, datesent.tm_min, 0);


                MD5_Init(&mdContext);
                MD5_Update(&mdContext, buffer, len);
                inHeader = TRUE;

                if ((previous = client->mailbox.message.used++) < client->mailbox.message.allocated) {
                    headPos = ftell(client->mailbox.fh) - len;
                } else {
                    if (AddMsgInfo(client, estCount)) {
                        ;
                    } else {
                        ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1);

                        fclose(client->mailbox.fh);
                        client->mailbox.fh = NULL;

                        FreeClientData(client);
                        return(FALSE);
                    }
                    estCount <<= 1;

                    headPos = ftell(client->mailbox.fh) - len;
                }

                realStart = headPos;
                bodyPos = 0;
                bodyLines = 0;
                msgState = MSG_STATE_RECENT;
                escape = FALSE;
                useSCMS = FALSE;
                scmsID = 0;
                uidL[0] = '\0';
                realSize = privateLineLen = len;

                /* Check the Authenticated thingy    */
                authPos = ftell(client->mailbox.fh);
                fgets(buffer, sizeof(buffer), client->mailbox.fh);
                len = strlen(buffer);
                authLen = len - 13;            /* X-Auth-OK:_ / X-Auth-NO:_, 2 less for CR/LF  */
                realSize += len;
                if (buffer[6] == '+') {
                    /* Check the SCMS reference            */
                    fgets(buffer, sizeof(buffer), client->mailbox.fh);

                    /* Get the ID, the line looks like this:    X-SCMS-ID: xxxxxxx */
                    if (buffer[0] == 'X' && QuickNCmp(buffer, "X-SCMS-ID:", 10)) {
                        scmsID = atol(buffer + 11);
                        useSCMS = TRUE;
                    }

                    len = strlen(buffer);

                    realSize += len;
                }
            } else {
                realSize += len;
            }

            if (inHeader) {
                i = 0;
                while (buffer[i] == CR) {
                    i++;
                }

                if (buffer[i] == LF) {
                    unsigned char    digest[16];

                    MD5_Final(digest, &mdContext);
                    for (i = 0; i < 16; i++) {
                        sprintf(hexDigest + (i * 2), "%02x", digest[i]);
                    }

                    inHeader = FALSE;
                    if (useSCMS == FALSE) {
                        bodyPos = ftell(client->mailbox.fh);
                    } else {
                        unsigned char    scmsPath[XPL_MAX_PATH];
                        FILE                *scms;

                        /* Since we use SCMS, we have to read the message from there and count it all up */
                        sprintf(scmsPath, "%s/%x/%lx", NMAP.path.scms, (unsigned int)(scmsID % 16), scmsID);
                        scms = fopen(scmsPath, "rb");
                        if (scms) {
                            headPos = 0;
                            authPos = 0;

                            scmsSize = 0;

                            while (!feof(scms)) {
                                if (fgets(buffer, sizeof(buffer), scms)) {
                                    len = strlen(buffer);
                                    scmsSize += len;

                                    if (buffer[0] == CR && buffer[1] == LF) {
                                        break;
                                    }
                                }
                            }

                            /* Now in the body    */
                            bodyPos = ftell(scms);

                            while (!feof(scms)) {
                                if (fgets(buffer, sizeof(buffer), scms)) {
                                    if (buffer[0]=='.') {
                                        escape = TRUE;
                                    }

                                    len = strlen(buffer);
                                    bodyLines++;
                                    scmsSize += len;
                                }
                            }

                            fclose(scms);
                        } else {
                            /* Behave as if it wasn't SCMS */
                            useSCMS = FALSE;
                            bodyPos = ftell(client->mailbox.fh);
                        }
                    }
                } else {
                    /* Date sent - from Date: line e.g. Date: Tue, 30 Sep 1997 10:20:49 -0600 */
                    if (QuickNCmp(buffer, "date:", 5)) {
                        memset(&datesent, 0, sizeof(struct tm));
                        sscanf(buffer, "%*s %*s %d %s %d" /* "%d %s %d"    */, &datesent.tm_mday, szMonth, &datesent.tm_year);
                        datesent.tm_isdst = -1;
                        for(month = 0; month < 12; month++) {
                            if (QuickNCmp(szMonth, MonthsOfTheYear[month], 3)) {
                                datesent.tm_mon = month;
                                break;
                            }
                        }

                        datesent.tm_year -= 1900; /* adjust for mktime */
                    } else if (buffer[0] == 'X') {
                        int                priority;
                        unsigned char    *ptr;

                        if (QuickNCmp(buffer, "X-NIMS-Flags:", 13)) {
                            ptr = buffer + 14;

                            /* Grab any potential message flags */
                            msgState |= atol(ptr);

                            ptr = strchr(ptr, ' ');
                            if (ptr) {
                                msgTime = atol(ptr + 1);
                            }
                        } else if (QuickNCmp(buffer, "X-Priority:", 11)) {
                            ptr = buffer + 12;

                            while (isspace(*ptr)) {
                                ptr++;
                            }

                            priority = atol(ptr);
                            if (priority == 3) {
                                ;
                            } else if (priority < 3) {
                                msgState |= MSG_STATE_PRIOHIGH;
                            } else if (priority > 3) {
                                msgState |= MSG_STATE_PRIOLOW;
                            }
                        } else if (QuickNCmp(buffer, "X-MSMail-Priority:", 18)) {
                            ptr = buffer + 18;

                            while (isspace(*ptr)) {
                                ptr++;
                            }

                            if (QuickNCmp(buffer, "normal", 6)) {
                                ;
                            } else if (QuickNCmp(buffer, "high", 4)) {
                                msgState |= MSG_STATE_PRIOHIGH;
                            } else if (QuickNCmp(buffer, "low", 4)) {
                                msgState |= MSG_STATE_PRIOLOW;
                            }
                        }
                    }

                    /* Count the line for the hash */
                    MD5_Update(&mdContext, buffer, len);
                }
            } else {
                if (useSCMS == FALSE) {
                    if (buffer[0] == '.') {
                        escape = TRUE;
                    }

                    bodyLines++;
                }
            }
        }
    }

    if (NMAP.state == NMAP_STOPPING) {
        return(FALSE);
    }

    if (previous != 0xFFFFFFFF) {
        client->mailbox.message.info[previous].TailSig = NMAP_IDX_LINE_TAIL_SIGNATURE;

        client->mailbox.message.info[previous].DateSent = mktime(&datesent);
        client->mailbox.message.info[previous].DateReceived = msgTime;
        client->mailbox.message.info[previous].UseSCMS = useSCMS;
        client->mailbox.message.info[previous].SCMSID = scmsID;
        client->mailbox.message.info[previous].RealStart = realStart;
        client->mailbox.message.info[previous].RealSize = realSize;
        if (!useSCMS) {
            client->mailbox.message.info[previous].MSize                            = realSize;
            client->mailbox.newSize += client->mailbox.message.info[previous].SSize    = realSize - privateLineLen;
            if (bodyPos) {
                client->mailbox.message.info[previous].BodyPos                        = bodyPos;
            } else {
                client->mailbox.message.info[previous].BodyPos                        = headPos + realSize;
            }
        } else {
            client->mailbox.message.info[previous].MSize                            = scmsSize;
            client->mailbox.newSize += client->mailbox.message.info[previous].SSize    = scmsSize;
            if (bodyPos) {
                client->mailbox.message.info[previous].BodyPos                        = bodyPos;
            } else {
                client->mailbox.message.info[previous].BodyPos                        = scmsSize;
            }
        }
        client->mailbox.message.info[previous].HeadPos = headPos;
        client->mailbox.message.info[previous].BodyLines = bodyLines;
        client->mailbox.message.info[previous].AuthPos = authPos;
        client->mailbox.message.info[previous].AuthLen = authLen;
        client->mailbox.message.info[previous].State = msgState;
        client->mailbox.message.info[previous].UID = client->mailbox.nextUID++;

        if (escape) {
            client->mailbox.message.info[previous].State |= MSG_STATE_ESCAPE;
        }

        if (uidL[0] == '\0') {
            client->mailbox.message.info[previous].State |= MSG_STATE_NEWUIDL;
            strcpy(client->mailbox.message.info[previous].UIDL, hexDigest);
        } else {
            strcpy(client->mailbox.message.info[previous].UIDL, uidL);
        }

        client->mailbox.newCount++;
        client->mailbox.message.count++;
    }

    /* Write changed index file */
    if (!(client->flags & NMAP_READONLY)) {
        if (WriteNLockAquire(client->mailbox.lock, 0) == NLOCK_AQUIRED) {
            if (WriteIndexFile(client, path, startRead)) {
                if (flagsSynced && (stat(path, &sb) == 0)) {
                    /* Index file and MSGInfo are in synch */
                    /* ParseMailDrop is not a good place to pick up new flags because the client can't always be notified. */
                    /* Every flag oriented command will pick up new flags if mailbox.indexTime does not match the time on the file */
                    client->mailbox.indexTime = sb.st_mtime;
                }
            } else {
                /* Add log event here */
            }
            WriteNLockRelease(client->mailbox.lock);
        }
    }

    fclose(client->mailbox.fh);

    client->mailbox.fh = NULL;
    return(TRUE);    
}

__inline static BOOL
UserUnderQuota(unsigned char *recipient, unsigned char *userStore, unsigned long messageSize, unsigned char *buffer, BOOL *postQuotaWarning, unsigned long *spaceUsed, unsigned long *userQuota)
{
    unsigned long used;
    unsigned long enforcedQuota;
    unsigned char dn[MDB_MAX_OBJECT_CHARS + 1];
    MDBValueStruct *vs;

    if (NMAP.quota.useUser || NMAP.quota.useSystem) {
        /* Determine Space Used */
        used = XplGetDiskspaceUsed(userStore);
        used += messageSize / 1024;

        /* Determine Space Allowed */
        if (!NMAP.quota.useUser) {
            enforcedQuota = NMAP.quota.defaultValue;    
        } else {
            vs = MDBCreateValueStruct(NMAP.handle.directory, NULL);

            if (MsgFindObject(recipient, dn, NULL, NULL, vs)) {
                MDBFreeValues(vs);

                if ((MsgGetUserFeature(dn, FEATURE_NMAP_QUOTA, MSGSRV_A_USE_QUOTA, vs) > 0) && (vs->Value[0][0] == '1')) {
                    MDBFreeValues(vs);
                    if (MsgGetUserFeature(dn, FEATURE_NMAP_QUOTA, MSGSRV_A_QUOTA_VALUE, vs) > 0) {
                        enforcedQuota = atol(vs->Value[0]);            
                    } else if (NMAP.quota.useSystem) {
                        enforcedQuota = NMAP.quota.defaultValue;
                    } else {
                        /* The recipient exists and has no configured value and we are not configured to use the system quota. */
                        enforcedQuota = 0;
                    }
                } else if (NMAP.quota.useSystem) {
                    enforcedQuota = NMAP.quota.defaultValue;
                } else {
                    /* The recipient doesn't have a configured quota limit and we are not configured to use the system quota. */
                    enforcedQuota = 0;
                }
            } else if (NMAP.quota.useSystem) {
                enforcedQuota = NMAP.quota.defaultValue;
            } else {
                /* The recipient wasn't found and we are not configured to use the system quota. */
                enforcedQuota = 0;
            }

            MDBDestroyValueStruct(vs); 
        } 
        
        /* Determine Compliance */
        if (enforcedQuota) {
            if (used <= (enforcedQuota * 9 / 10)) {
                ;
            } else {
                if ((recipient[0] != NMAP.postMaster[0]) || (XplStrCaseCmp(recipient, NMAP.postMaster) != 0)) {
                    if (used <= enforcedQuota) {
                        if (postQuotaWarning) {
                            *postQuotaWarning = TRUE;
                        }

                        return(TRUE);
                    } else {
                        if (spaceUsed) {
                            *spaceUsed = used;
                        }

                        if (userQuota) {
                            *userQuota = enforcedQuota;
                        }

                        
                        return(FALSE);
                    }
                }
            }
        }
    }

    *postQuotaWarning = FALSE;

    return(TRUE);
}

__inline static int
CopyMessageToMailbox(FILE *messageFH, FILE *boxFH, unsigned char *recipient, unsigned char *sender, unsigned char *authSender, unsigned long scmsID, unsigned long messageSize, unsigned long flags, BOOL postQuotaWarning, unsigned char *buffer, unsigned long bufferSize)
{
    size_t count;
    size_t overhead;
    unsigned char timeBuffer[80];
    time_t time_of_day;

    time(&time_of_day);

    strftime(timeBuffer, 80, "%a %b %d %H:%M %Y",gmtime(&time_of_day));
    GENERATE_FROM_DELIMITER(timeBuffer, sender, buffer, overhead);

    if (fwrite(buffer, sizeof(unsigned char), overhead, boxFH) == overhead) {
        if (scmsID == 0) {
            if (authSender && authSender[0] != '-') {
                overhead = sprintf(buffer, "X-Auth-OK: %s\r\n", authSender);
            } else {
                overhead = sprintf(buffer, "X-Auth-No: \r\n");
            }

            if (fwrite(buffer, sizeof(unsigned char), overhead, boxFH) == overhead) {
                if (flags) {
                    overhead = sprintf(buffer, "X-NIMS-flags: %lu\r\n", flags);
                    if (fwrite(buffer, sizeof(unsigned char), overhead, boxFH) == overhead) {
                        ;
                    } else {
                        goto StoreIOError;
                    }
                }

                overhead = sprintf(buffer, "Return-Path: <%s>\r\n", sender);
                if (fwrite(buffer, sizeof(unsigned char), overhead, boxFH) == overhead) {
                    while(!ferror(messageFH) && !feof(messageFH)) {
                        count = fread(buffer, sizeof(unsigned char), bufferSize, messageFH);
                        if (count > 0) {
                            if (fwrite(buffer, sizeof(unsigned char), count, boxFH) == count) {
                                continue;
                            } else {
                                goto StoreIOError;
                            }
                        }
                    }

                    XplSafeAdd(NMAP.stats.storedLocal.bytes, (messageSize + 1023) / 1024);
                    XplSafeIncrement(NMAP.stats.storedLocal.count);
                } else {
                    goto StoreIOError;
                }
            } else {
                goto StoreIOError;
            }
        } else {
            /* SCMS Reference */
            if (authSender && authSender[0]!='-') {
                overhead = sprintf(buffer, "X-Auth+OK: %s\r\n", authSender);
            } else {
                overhead = sprintf(buffer, "X-Auth+No: \r\n");
            }

            if (fwrite(buffer, sizeof(unsigned char), overhead, boxFH) == overhead) {
                overhead = sprintf(buffer, "X-SCMS-ID: %lu\r\n", scmsID);
                if (fwrite(buffer, sizeof(unsigned char), overhead, boxFH) == overhead) {
                    if (flags) {
                        overhead = sprintf(buffer, "X-NIMS-flags: %lu\r\n", flags);
                        if (fwrite(buffer, sizeof(unsigned char), overhead, boxFH) == overhead) {
                            ;
                        } else {
                            goto StoreIOError;
                        }
                    }

                    /*    We're using SCMS, only write our minimal header, stop at the body    */
                    overhead = sprintf(buffer, "\r\nThis message is stored in SCMS\r\n");
                    if (fwrite(buffer, sizeof(unsigned char), overhead, boxFH) == overhead) {
                        ;
                    } else {
                        goto StoreIOError;
                    }
                } else {
                    goto StoreIOError;
                }
            } else {
                goto StoreIOError;
            }
        }
    } else {
        goto StoreIOError;
    }

    if (fwrite("\r\n", sizeof(unsigned char), 2, boxFH) == 2) {
        if (!postQuotaWarning) {
            ;
        } else {
            time(&time_of_day);
            strftime(timeBuffer, 80, "%a %b %d %H:%M %Y", gmtime(&time_of_day));
            GENERATE_FROM_DELIMITER(timeBuffer, NMAP.postMaster, buffer, overhead);
            overhead += sprintf(buffer + overhead, "X-Auth-OK: %s\r\nReturn-Path: <%s>\r\n", NMAP.postMaster, NMAP.postMaster);
            if (fwrite(buffer, sizeof(unsigned char), overhead, boxFH) == overhead) {
                XplSafeAdd(NMAP.stats.storedLocal.bytes, (overhead + 1023) / 1024);

                time(&time_of_day);
                strftime(timeBuffer, 80, "%a, %d %b %Y %H:%M:%S %Z", gmtime(&time_of_day));
                overhead = sprintf(buffer, "Subject: mailbox almost full\r\nDate: %s\r\nFrom: %s@%s\r\nTo: %s\r\n\r\n", timeBuffer, NMAP.postMaster, NMAP.officialName, recipient);
                if (fwrite(buffer, sizeof(unsigned char), overhead, boxFH) == overhead) {
                    XplSafeAdd(NMAP.stats.storedLocal.bytes, (overhead + 1023) / 1024);

                    if (NMAP.quota.warning) {
                        overhead = sprintf(buffer, "%s\r\n\r\n", NMAP.quota.warning);
                        fwrite(buffer, sizeof(unsigned char), overhead, boxFH);
                    } else {
                        overhead = sprintf(buffer, "%s\r\n", DEFAULT_QUOTA_WARNING);
                        fwrite(buffer, sizeof(unsigned char), overhead, boxFH);
                    }

                    XplSafeAdd(NMAP.stats.storedLocal.bytes, (overhead + 1023) / 1024);
                }
            }

            XplSafeIncrement(NMAP.stats.storedLocal.count);
        }

        XplSafeIncrement(NMAP.stats.storedLocal.recipients);

        return(DELIVER_SUCCESS);
    }

StoreIOError:

    if (errno != ENOSPC) {
        return(DELIVER_TRY_LATER);
    }

    return(DELIVER_QUOTA_EXCEEDED);
}

int
StoreMessageInMailbox(unsigned char *sender, unsigned char *authSender, unsigned char *recipient, unsigned char *mBoxName, FILE *messageFH, unsigned long scmsID, unsigned long messageSize, unsigned long flags)
{
    int retVal;
    unsigned long spaceUsed;
    unsigned long userQuota;
    unsigned char *mailboxName = mBoxName;
    unsigned char path[CONN_BUFSIZE + 1];
    unsigned char buffer[CONN_BUFSIZE + 1];
    const unsigned char *mailboxPath;
    BOOL postQuotaWarning;
    FILE *mailboxFH;
    NLockStruct *mailboxLock;

    mailboxLock = NULL;
    mailboxPath = MsgFindUserStore(recipient, NMAP.path.mail);
    sprintf(path, "%s/%s", mailboxPath, recipient);

    /*    First, check quota enforcement.    */
    if (UserUnderQuota(recipient, path, messageSize, buffer, &postQuotaWarning, &spaceUsed, &userQuota)) {
        sprintf(path, "%s/%s/%s.box", mailboxPath, recipient, mailboxName);
        if (ReadNLockAquire(&mailboxLock, NULL, recipient, mailboxName, 3000) == NLOCK_AQUIRED) {
            if (WriteNLockAquire(mailboxLock, 2000) == NLOCK_AQUIRED) {
                mailboxFH = fopen(path, "ab");
                if (mailboxFH != NULL) {
                    retVal = CopyMessageToMailbox(messageFH, mailboxFH, recipient, sender, authSender, scmsID, messageSize, flags, postQuotaWarning, buffer, sizeof(buffer));
                    if ((retVal == DELIVER_SUCCESS) && (NMAP.flushFlags & FLUSH_BOX_ON_STORE)) {
                        XplFlushWrites(mailboxFH);
                    }
                    fclose(mailboxFH);
                    WriteNLockRelease(mailboxLock);
                    ReadNLockRelease(mailboxLock);
                    mailboxLock = NULL;
                    /* LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_MAIL, LOGGER_EVENT_ITEM_STORED, LOG_INFO, 0, recipient, authSender? authSender: sender, messageSize, 0, MIME_TEXT_PLAIN, strlen(mailboxName) + 1, mailboxName, NULL, 0); */
                    return(retVal);
                } else {
                    /* Maybe the Path doesn't exist yet; try to create it */
                    if (CreateMBoxPath(mailboxPath, recipient, mailboxName, path, buffer)) {
                        sprintf(path, "%s/%s/%s.box", mailboxPath, recipient, mailboxName);
                        mailboxFH = fopen(path, "ab");
                    } 
                    
                    if (mailboxFH) {
                        retVal = CopyMessageToMailbox(messageFH, mailboxFH, recipient, sender, authSender, scmsID, messageSize, flags, postQuotaWarning, buffer, sizeof(buffer));
                        if ((retVal == DELIVER_SUCCESS) && (NMAP.flushFlags & FLUSH_BOX_ON_STORE)) {
                            XplFlushWrites(mailboxFH);
                        }
                        fclose(mailboxFH);
                        WriteNLockRelease(mailboxLock);
                        ReadNLockRelease(mailboxLock);
                        mailboxLock = NULL;
                        /* LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_MAIL, LOGGER_EVENT_ITEM_STORED, LOG_INFO, 0, recipient, authSender? authSender: sender, messageSize, 0, MIME_TEXT_PLAIN, strlen(mailboxName) + 1, mailboxName, NULL, 0); */
                        return(retVal);
                    } else if ((toupper(mailboxName[0] != 'I')) || (XplStrCaseCmp("INBOX", mailboxName) != 0)) {
                        /* We just can't open the requested box file; try inbox */
                        WriteNLockRelease(mailboxLock);
                        ReadNLockRelease(mailboxLock);
                        mailboxLock = NULL;
                        mailboxName = "INBOX";

                        sprintf(path, "%s/%s/%s.box", mailboxPath, recipient, mailboxName);

                        if ((ReadNLockAquire(&mailboxLock, NULL, recipient, mailboxName, 3000) == NLOCK_AQUIRED) ) {
                            if (WriteNLockAquire(mailboxLock, 2000) == NLOCK_AQUIRED) {
                                mailboxFH = fopen(path, "ab");
                                if (mailboxFH) {
                                    retVal = CopyMessageToMailbox(messageFH, mailboxFH, recipient, sender, authSender, scmsID, messageSize, flags, postQuotaWarning, buffer, sizeof(buffer));
                                    if ((retVal == DELIVER_SUCCESS) && (NMAP.flushFlags & FLUSH_BOX_ON_STORE)) {
                                        XplFlushWrites(mailboxFH);
                                    }
                                    fclose(mailboxFH);
                                    WriteNLockRelease(mailboxLock);
                                    ReadNLockRelease(mailboxLock);
                                    mailboxLock = NULL;
                                    /* LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_MAIL, LOGGER_EVENT_ITEM_STORED, LOG_INFO, 0, recipient, authSender? authSender: sender, messageSize, 0, MIME_TEXT_PLAIN, strlen(mailboxName) + 1, mailboxName, NULL, 0); */
                                    return(retVal);
                                }
                                WriteNLockRelease(mailboxLock);
                            }
                            ReadNLockRelease(mailboxLock);
                            mailboxLock = NULL;
                        }

                        return(DELIVER_TRY_LATER);
                    }

                    WriteNLockRelease(mailboxLock);
                    ReadNLockRelease(mailboxLock);
                    mailboxLock = NULL;
                    return(DELIVER_TRY_LATER);
                }
                
                WriteNLockRelease(mailboxLock);
            }

            ReadNLockRelease(mailboxLock);
            mailboxLock = NULL;
        }

        return(DELIVER_TRY_LATER);
    }
    
    LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_OVER_QUOTA, LOG_INFO, 0, recipient, sender, spaceUsed, userQuota, NULL, 0);
     return(DELIVER_QUOTA_EXCEEDED);
}

__inline static int 
MailboxOOBNotifications(NMAPClient *client, struct stat *sb)
{
    register NMAPClient *c = client;

    if (!(c->share.flags & NMAP_SHARE_MAILBOX)) {
        /* Check if new mail arrived or if the state of any message changed */
        sprintf(c->path, "%s/%s/%s.box", c->store, c->user, c->mailbox.name);
        if ((stat(c->path, sb) == 0) && (c->mailbox.size != (unsigned long)sb->st_size)) {
            ParseMaildrop(c);

            if (c->flags & NMAP_OOBMESSAGES) {
                return(ConnWriteF(c->conn, "6000 %d New Messages\r\n", EVENT_MBOX_NEW_MESSAGES));
            }
        }
    } else if (IsAvailableShare(c->user, c->mailbox.name, NMAP_SHARE_MAILBOX, NULL, NULL)) {
        BOOL changed = FALSE;

        /* Synchronizing the shared resource may change the value for c->mailbox.indexTime. */
        if (SyncSharedMaildrop(c, &changed) && (changed) && (c->flags & NMAP_OOBMESSAGES)) {
            return(ConnWriteF(c->conn, "6000 %d New Messages\r\n", EVENT_MBOX_NEW_MESSAGES));
        }
    } else {
        /* This user no longer has permissions to the shared mailbox. */
        c->share.flags |= NMAP_SHARE_MAILBOX_DENIED;
    }

    return(0);
}

int 
NmapCommandAflg(void *param)
{
    int ccode;
    unsigned long id;
    unsigned long flags;
    unsigned long newFlags;
    unsigned char *ptr;
    unsigned char *ptr2;
    struct stat sb;
    FILE *index;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_MBOX) {
        if (!(client->flags & NMAP_READONLY)) {
            ptr = client->buffer + 4;
            if ((*ptr++ == ' ') 
                    && (!isspace(*ptr)) 
                    && ((ptr2 = strchr(ptr, ' ')) != NULL)) {
                *ptr2++ = '\0';
            } else {
                return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
            }
        } else {
            return(ConnWrite(client->conn, MSG4243READONLY, sizeof(MSG4243READONLY) - 1));
        }
    } else if (!(client->states & NMAP_CLIENT_USER)) {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    } else {
        return(ConnWrite(client->conn, MSG4225NOMBOX, sizeof(MSG4225NOMBOX) - 1));
    }

    ccode = MailboxOOBNotifications(client, &sb);
    if (ccode != -1) {
        id = atol(ptr);
    } else {
        return(ccode);
    }

    if (id && (id <= client->mailbox.message.used)) {
        id--;
    } else {
        return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
    }

    newFlags = atol(ptr2);

    if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
        sprintf(client->path, "%s/%s/%s.idx", client->store, client->user, client->mailbox.name);

        if (client->flags & NMAP_OOBMESSAGES) {
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 

            flags = client->mailbox.message.info[id].State;
            if ((flags & newFlags) == newFlags) {
                return(ConnWriteF(client->conn, "1000 %lu %s\r\n", flags, MSG1000FLAGSET));
            }
        } else {
            flags = client->mailbox.message.info[id].State;
            if ((flags & newFlags) == newFlags) {
                return(ConnWriteF(client->conn, "1000 %lu %s\r\n", flags, MSG1000FLAGSET));
            }
        }
    } else if (!(client->share.flags & NMAP_SHARE_MAILBOX_DENIED)) {
        sprintf(client->path, "%s/%s/%s.sdx", client->store, client->user, client->mailbox.name);

        if (client->flags & NMAP_OOBMESSAGES) {
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 

            flags = client->mailbox.message.info[id].State;
            if ((flags & newFlags) == newFlags) {
                return(ConnWriteF(client->conn, "1000 %lu %s\r\n", flags, MSG1000FLAGSET));
            }
        } else {
            flags = client->mailbox.message.info[id].State;
            if ((flags & newFlags) == newFlags) {
                return(ConnWriteF(client->conn, "1000 %lu %s\r\n", flags, MSG1000FLAGSET));
            }
        }

        if ((newFlags & MSG_STATE_READ) && !(client->share.mailbox.permissions & NMAP_SHARE_SEEN)) {
            newFlags &= ~MSG_STATE_READ;
        }

        if ((newFlags & (MSG_STATE_DELETED | MSG_STATE_PURGED)) && !(client->share.mailbox.permissions & NMAP_SHARE_DELETE)) {
            newFlags &= ~(MSG_STATE_DELETED | MSG_STATE_PURGED);
        }
    } else {
        return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
    }

    if (WriteNLockAquire(client->mailbox.lock, 3000) == NLOCK_AQUIRED) {
        index = fopen(client->path, "r+b");
        if (index != NULL) {
            if ((newFlags & MSG_STATE_DELETED) && !(flags & MSG_STATE_DELETED)) {
                client->mailbox.message.count--;
            }

            if (newFlags & MSG_STATE_PURGED) {
                client->mailbox.changed = TRUE;
            }

            flags |= newFlags;

            fseek(index, NMAP_IDX_HEADERSIZE + (id * sizeof(MessageInfoStruct)) + offsetof(MessageInfoStruct, State), SEEK_SET);

            if (!(flags & MSG_STATE_RECENT)) {
                fwrite(&flags, sizeof(unsigned long), 1, index);
            } else {
                flags &= ~MSG_STATE_RECENT;
                fwrite(&flags, sizeof(unsigned long), 1, index);
                flags |= MSG_STATE_RECENT;
            }

            fclose(index);
            
            WriteNLockRelease(client->mailbox.lock);

            if (stat(client->path, &sb) == 0) {
                client->mailbox.indexTime = sb.st_mtime;    
            }

            client->mailbox.message.info[id].State = flags;

            return(ConnWriteF(client->conn, "1000 %lu %s\r\n", flags, MSG1000FLAGSET));
        }

        WriteNLockRelease(client->mailbox.lock);
    }

    return(ConnWrite(client->conn, MSG4228CANTWRITEMBOX, sizeof(MSG4228CANTWRITEMBOX) - 1));
}

int 
NmapCommandAinfo(void *param)
{
    int ccode;
    unsigned long id;
    unsigned long msgUsed;
    unsigned char *ptr;
    unsigned char authSender[MAXEMAILNAMESIZE + 1];
    struct stat sb;
    NMAPClient *client = (NMAPClient *)param;
    MessageInfoStruct *msgInfo;
    FILE *scms;

    if (client->states & NMAP_CLIENT_MBOX) {
        ccode = MailboxOOBNotifications(client, &sb);
        if (ccode != -1) {
            ptr = client->buffer + 5;
        } else {
            return(ccode);
        }
    } else if (!(client->states & NMAP_CLIENT_USER)) {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    } else {
        return(ConnWrite(client->conn, MSG4225NOMBOX, sizeof(MSG4225NOMBOX) - 1));
    }

    if (*ptr == '\0') {
        if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
            if (client->flags & NMAP_OOBMESSAGES) {
                sprintf(client->path, "%s/%s/%s.idx", client->store, client->user, client->mailbox.name);
                GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
            }

            sprintf(client->path, "%s/%s/%s.box", client->store, client->user, client->mailbox.name);
            client->mailbox.fh = fopen(client->path, "rb");
            if (client->mailbox.fh != NULL) {
                msgUsed = client->mailbox.message.used;

                ccode = ConnWriteF(client->conn, "2002 %lu\r\n", msgUsed);
            } else {
                return(ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1));
            }
        } else if (!(client->share.flags & NMAP_SHARE_MAILBOX_DENIED)) {
            if (client->flags & NMAP_OOBMESSAGES) {
                sprintf(client->path, "%s/%s/%s.sdx", client->store, client->user, client->mailbox.name);
                GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
            }

            if (NMAPSendCommandF(client->share.mailbox.conn, "AINFO\r\n", 7) != -1) {
                ccode = NMAPReadAnswerLine(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE);
                if (ccode == 2002) {
                    msgUsed = atol(client->line + 5);
                    if (msgUsed == client->mailbox.message.used) {
                        ccode = ConnWriteF(client->conn, "2002 %lu\r\n", msgUsed);
                    } else {
                        return(ConnWriteF(client->conn, MSG3014REQNOTALLOWED, "  Shared resource not synchronized."));
                    }
                } else if (ccode != -1) {
                    return(ccode = ConnWrite(client->conn, client->line, strlen(client->line)));
                } else {
                    return(ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1));
                }
            } else {
                return(ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1));
            }
        } else {
            return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
        }

        for (id = 0, msgInfo = client->mailbox.message.info; (ccode != -1) && (id < msgUsed); id++, msgInfo++) {
            if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
                if ((msgInfo->AuthLen > 0) && (msgInfo->AuthLen < MAXEMAILNAMESIZE)) {
                    if (!msgInfo->UseSCMS) {
                        fseek(client->mailbox.fh, msgInfo->AuthPos + 11, SEEK_SET);
                        fread(authSender, sizeof(unsigned char), msgInfo->AuthLen, client->mailbox.fh);
                        authSender[msgInfo->AuthLen] = '\0';
                    } else {
                        sprintf(client->path, "%s/%x/%lx", NMAP.path.scms, (unsigned int)(msgInfo->SCMSID % 16), msgInfo->SCMSID);
                        if (SCMSWaitforAndLockMessage(msgInfo->SCMSID)) {
                            scms = fopen(client->path, "rb");
                            if (scms != NULL) {
                                fseek(scms, msgInfo->AuthPos + 11, SEEK_SET);
                                fread(authSender, sizeof(unsigned char), msgInfo->AuthLen, scms);
                                authSender[msgInfo->AuthLen] = '\0';
                                fclose(scms);
                            } else {
                                authSender[0] = '-';
                                authSender[1] = '\0';
                            }

                            SCMSUnlockMessage(msgInfo->SCMSID);
                        } else {
                            authSender[0] = '-';
                            authSender[1] = '\0';
                        }
                    }
                } else {
                    authSender[0] = '-';
                    authSender[1] = '\0';
                }
            } else {
                NMAPReadAnswer(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE);

                ptr = strrchr(client->line, ' ');
                if (ptr != NULL) {
                    strcpy(authSender, ++ptr);
                } else {
                    authSender[0] = '-';
                    authSender[1] = '\0';
                }
            }

            ccode = ConnWriteF(client->conn, "%lu %lu %lu %lu %lu %s %lu %lu %lu %s\r\n",
                id + 1, 
                msgInfo->SSize, 
                msgInfo->BodyPos - msgInfo->AuthPos, /* HeadSize */
                msgInfo->SSize + msgInfo->AuthPos - msgInfo->BodyPos, /* BodySize */
                msgInfo->State, 
                msgInfo->UIDL, 
                msgInfo->UID, 
                client->mailbox.id, 
                msgInfo->DateReceived, 
                authSender);
        }

        if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
            if (client->mailbox.fh != NULL) {
                fclose(client->mailbox.fh);
                client->mailbox.fh = NULL;
            }

            if (ccode != -1) {
                ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
            }
        } else {
            if ((ccode != -1) 
                    && ((ccode = NMAPReadAnswer(client->share.mailbox.conn, client->line, CONN_BUFSIZE, TRUE)) == 1000)) {
                ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
            } else {
                ConnWrite(client->conn, MSG5004INTERNALERR, sizeof(MSG5004INTERNALERR) - 1);

                ccode = -1;
            }
        }

        return(ccode);
    } else if ((*ptr != ' ') || !(isdigit(ptr[1]))) {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    /* fixme - result return behaviour inconsistent - should always send 1000 if successful */
    id = atoi(++ptr);
    if (id && (id <= client->mailbox.message.used)) {
        id--;
    } else {
        return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
    }

    msgInfo = &(client->mailbox.message.info[id]);
    if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
        if (client->flags & NMAP_OOBMESSAGES) {
            sprintf(client->path, "%s/%s/%s.idx", client->store, client->user, client->mailbox.name);
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
        }

        sprintf(client->path, "%s/%s/%s.box", client->store, client->user, client->mailbox.name);
        client->mailbox.fh = fopen(client->path, "rb");
        if (client->mailbox.fh != NULL) {
            if ((msgInfo->AuthLen > 0) && (msgInfo->AuthLen < MAXEMAILNAMESIZE)) {
                if (!msgInfo->UseSCMS) {
                    fseek(client->mailbox.fh, msgInfo->AuthPos + 11, SEEK_SET);
                    fread(authSender, sizeof(unsigned char), msgInfo->AuthLen, client->mailbox.fh);
                    authSender[msgInfo->AuthLen] = '\0';
                } else {
                    sprintf(client->path, "%s/%x/%lx",NMAP.path.scms, (unsigned int)(msgInfo->SCMSID % 16), msgInfo->SCMSID);
                    if (SCMSWaitforAndLockMessage(msgInfo->SCMSID)) {
                        scms = fopen(client->path, "rb");
                        if (scms != NULL) {
                            fseek(scms, msgInfo->AuthPos + 11, SEEK_SET);
                            fread(authSender, sizeof(unsigned char), msgInfo->AuthLen, scms);
                            
                            authSender[msgInfo->AuthLen] = '\0';
                            
                            fclose(scms);
                        } else {
                            authSender[0] = '-';
                            authSender[1] = '\0';
                        }

                        SCMSUnlockMessage(msgInfo->SCMSID);
                    } else {
                        authSender[0] = '-';
                        authSender[1] = '\0';
                    }
                }
            } else {
                authSender[0] = '-';
                authSender[1] = '\0';
            }
        } else {
            return(ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1));
        }
    } else if (!(client->share.flags & NMAP_SHARE_MAILBOX_DENIED)) {
        if (client->flags & NMAP_OOBMESSAGES) {
            sprintf(client->path, "%s/%s/%s.sdx", client->store, client->user, client->mailbox.name);
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
        }

        if ((ccode = NMAPSendCommandF(client->share.mailbox.conn, "AINFO %lu\r\n", id + 1)) != -1) {
            ccode = NMAPReadAnswer(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE);
            if (ccode == 2001) {
                ptr = strrchr(client->line + 5, ' ');
                if (ptr != NULL) {
                    strcpy(authSender, ++ptr);
                } else {
                    authSender[0] = '-';
                    authSender[1] = '\0';
                }
            } else if (ccode != -1) {
                return(ConnWrite(client->conn, client->line, strlen(client->line)));
            } else {
                return(ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1));
            }
        } else {
            return(ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
    }

    if (ccode != -1) {
        ccode = ConnWriteF(client->conn, "2001 %lu %lu %lu %lu %lu %s %lu %lu %lu %s\r\n",
            id + 1, 
            msgInfo->SSize,
            msgInfo->BodyPos - msgInfo->AuthPos, /* HeadSize */
            msgInfo->SSize + msgInfo->AuthPos - msgInfo->BodyPos, /* BodySize */
            msgInfo->State, 
            msgInfo->UIDL, 
            msgInfo->UID, 
            client->mailbox.id, 
            msgInfo->DateReceived, 
            authSender);
    }

    if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
        if (client->mailbox.fh) {
            fclose(client->mailbox.fh);
        }

        client->mailbox.fh = NULL;
    }

    return(ccode);
}

int 
NmapCommandBody(void *param)
{
    int count;
    int ccode;
    unsigned long id;
    unsigned char *ptr;
    struct stat sb;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_MBOX) {
        ccode = MailboxOOBNotifications(client, &sb);
        if (ccode != -1) {
            ptr = client->buffer + 4;
        } else {
            return(ccode);
        }

        if ((*ptr == ' ') && (isdigit(*(++ptr)))) {
            id = atoi(ptr);
        } else {
            return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
        }
    } else if (!(client->states & NMAP_CLIENT_USER)) {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    } else {
        return(ConnWrite(client->conn, MSG4225NOMBOX, sizeof(MSG4225NOMBOX) - 1));
    }

    if (id && (id <= client->mailbox.message.used)) {
        id--;
    } else {
        return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
    }

    if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
        if (client->flags & NMAP_OOBMESSAGES) {
            sprintf(client->path, "%s/%s/%s.idx", client->store, client->user, client->mailbox.name);
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
        }

        /* make sure the mailbox can be opened */
        if (!client->mailbox.message.info[id].UseSCMS) {
            sprintf(client->path, "%s/%s/%s.box", client->store, client->user, client->mailbox.name);
        } else {
            sprintf(client->path, "%s/%x/%lx", NMAP.path.scms, (unsigned int)(client->mailbox.message.info[id].SCMSID % 16), client->mailbox.message.info[id].SCMSID);
        }

        client->mailbox.fh = fopen(client->path, "rb");
        if (client->mailbox.fh != NULL) {
            count = client->mailbox.message.info[id].SSize - client->mailbox.message.info[id].BodyPos + client->mailbox.message.info[id].AuthPos;

            if (!(client->mailbox.message.info[id].State & MSG_STATE_ESCAPE)) {
                ccode = ConnWriteF(client->conn, "2021 %lu\r\n", count);
            } else {
                ccode = ConnWriteF(client->conn, "2025 %lu\r\n", count);
            }
        } else {
            ConnWrite(client->conn, "<Message not available>\r\n", 25);
            return(ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1));
        }

        fseek(client->mailbox.fh, client->mailbox.message.info[id].BodyPos, SEEK_SET);

        ccode = ConnWriteFromFile(client->conn, client->mailbox.fh, count);

        fclose(client->mailbox.fh);
        client->mailbox.fh = NULL;

        if (ccode != -1) {
            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
        }
    } else if (!(client->share.flags & NMAP_SHARE_MAILBOX_DENIED)) {
        if (client->flags & NMAP_OOBMESSAGES) {
            sprintf(client->path, "%s/%s/%s.sdx",client->store, client->user, client->mailbox.name);
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
        }

        if (NMAPSendCommandF(client->share.mailbox.conn, "BODY %lu\r\n", id + 1) != -1) {
            ccode = NMAPReadAnswerLine(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE);
            if ((ccode == 2021) || (ccode == 2025)) {
                count = atol(client->line + 5);
            } else if (ccode != -1) {
                ConnWrite(client->conn, client->line, strlen(client->line));
                return(ccode);
            } else {
                return(-1);
            }

            ccode = ConnWriteF(client->conn, "%lu %lu\r\n", (long unsigned int)ccode, (long unsigned int)count);

            if (ccode != -1) {
                ccode = ConnReadToConn(client->share.mailbox.conn, client->conn, count);
            }

            if (ccode != -1) {
                ccode = NMAPReadAnswerLine(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE);
                if (ccode == 1000) {
                    ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
                } else if (ccode != -1) {
                    ConnWrite(client->conn, client->line, strlen(client->line));
                }
            }
        } else {
            ccode = ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1);
        }
    } else {
        ccode = ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1);
    }

    return(ccode);;
}

int 
NmapCommandBraw(void *param)
{
    int count;
    int size;
    int ccode;
    unsigned long id;
    unsigned long start;
    unsigned char *ptr;
    struct stat sb;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_MBOX) {
        ccode = MailboxOOBNotifications(client, &sb);
        if (ccode != -1) {
            ptr = client->buffer + 4;
        } else {
            return(ccode);
        }

        if ((*ptr == ' ') && (isdigit(*(++ptr)))) {
            id = atol(ptr);
        } else {
            return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
        }
    } else if (!(client->states & NMAP_CLIENT_USER)) {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    } else {
        return(ConnWrite(client->conn, MSG4225NOMBOX, sizeof(MSG4225NOMBOX) - 1));
    }

    if (id && (id <= client->mailbox.message.used)) {
        id--;
    } else {
        return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
    }

    ptr = strchr(ptr, ' ');
    if (ptr != NULL) {
        start = atol(++ptr);
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    ptr = strchr(ptr, ' ');
    if (ptr != NULL) {
        size = atol(++ptr);
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
        if (client->flags & NMAP_OOBMESSAGES) {
            sprintf(client->path, "%s/%s/%s.idx",client->store, client->user, client->mailbox.name);
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
        }

        if (size > (client->mailbox.message.info[id].SSize + client->mailbox.message.info[id].AuthPos - client->mailbox.message.info[id].BodyPos)) {
            size = client->mailbox.message.info[id].SSize + client->mailbox.message.info[id].AuthPos - client->mailbox.message.info[id].BodyPos;
        }

        if (!client->mailbox.message.info[id].UseSCMS) {
            sprintf(client->path, "%s/%s/%s.box", client->store, client->user, client->mailbox.name);
        } else {
            sprintf(client->path, "%s/%x/%lx", NMAP.path.scms, (unsigned int)(client->mailbox.message.info[id].SCMSID % 16), client->mailbox.message.info[id].SCMSID);
        }

        client->mailbox.fh = fopen(client->path,"rb");

        if (client->mailbox.fh != NULL) {
            count = size;
            if (!(client->mailbox.message.info[id].State & MSG_STATE_ESCAPE)) {
                ccode = ConnWriteF(client->conn, "2021 %lu\r\n", size);
            } else {
                ccode = ConnWriteF(client->conn, "2025 %lu\r\n", size);
            }

            fseek(client->mailbox.fh, client->mailbox.message.info[id].BodyPos + start, SEEK_SET);

            ccode = ConnWriteFromFile(client->conn, client->mailbox.fh, count);
        } else {
            ccode = ConnWrite(client->conn, "<Message not available>\r\n", 25);
        }

        fclose(client->mailbox.fh);
        client->mailbox.fh = NULL;

        ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
    } else if (!(client->share.flags & NMAP_SHARE_MAILBOX_DENIED)) {
        if (client->flags & NMAP_OOBMESSAGES) {
            sprintf(client->path, "%s/%s/%s.sdx",client->store, client->user, client->mailbox.name);
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
        }

        if (((ccode = NMAPSendCommandF(client->share.mailbox.conn, "BRAW %lu %lu %lu\r\n", id + 1, start, size)) != -1) 
                && ((ccode = NMAPReadAnswerLine(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                && ((ccode == 2021) || (ccode == 2025))) {
            count = atol(client->line + 5);
            if (((ccode = ConnWriteF(client->conn, "%lu %lu\r\n", (long unsigned int)ccode, (long unsigned int)count)) != -1) 
                    && ((ccode = ConnReadToConn(client->share.mailbox.conn, client->conn, count)) != -1) 
                    && ((ccode = NMAPReadAnswerLine(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                    && (ccode == 1000)) {
                ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
            } else if (ccode != -1) {
                ccode = ConnWrite(client->conn, client->line, strlen(client->line));
            } else {
                ccode = ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1);
            }
        } else {
            ccode = ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1);
        }
    } else {
        ccode = ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1);
    }

    return(ccode);
}

int 
NmapCommandCheck(void *param)
{
    int ccode;
    unsigned long i;
    unsigned long totalSize = 0;
    unsigned long totalCount = 0;
    unsigned long purgedSize = 0;
    unsigned long purgedCount = 0;
    unsigned long readCount = 0;
    unsigned long readSize = 0;
    unsigned char *ptr;
    unsigned char *resourceOwner;
    unsigned char *resourceName;
    BOOL result;
    struct stat sb;
    MessageInfoStruct *msgInfo;
    NMAPClient *client = (NMAPClient *)param;
    NMAPClient *sclient;

    if (client->states & NMAP_CLIENT_USER) {
        ptr = client->buffer + 5;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if ((*ptr++ == ' ') && (strchr(ptr, ' ') == NULL)) {
        sclient = NMAPClientAlloc();
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (sclient != NULL) {
        strcpy(sclient->user, client->user);
        strcpy(sclient->mailbox.name, ptr);

        sclient->userHash = client->userHash;
        sclient->store = client->store;
        sclient->mailbox.newline = TRUE;
    } else {
        return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
    }

    if (client->states & NMAP_CLIENT_MBOX) {
	ccode = MailboxOOBNotifications(client, &sb);
	if (ccode == -1) {
	    return(ccode);
	}
    }

    sprintf(sclient->path, "%s/%s/%s.box", sclient->store, sclient->user, sclient->mailbox.name);
    if ((access(sclient->path, 0) == 0) 
            && (ReadNLockAquire(&sclient->mailbox.lock, &sclient->userHash, sclient->user, sclient->mailbox.name, 0) == NLOCK_AQUIRED)) {
        ParseMaildrop(sclient);
        ReadNLockRelease(sclient->mailbox.lock);
        sclient->mailbox.lock = NULL;
    } else if (XplStrCaseCmp(sclient->mailbox.name, "INBOX") == 0) {
        memcpy(sclient->mailbox.name, "INBOX", 5);

        sprintf(sclient->path, "%s/%s/INBOX.box", sclient->store, sclient->user);
        if ((access(sclient->path, 0) == 0) 
                && (ReadNLockAquire(&sclient->mailbox.lock, &sclient->userHash, sclient->user, sclient->mailbox.name, 0) == NLOCK_AQUIRED)) {
            ParseMaildrop(sclient);
            ReadNLockRelease(sclient->mailbox.lock);
            sclient->mailbox.lock = NULL;
        } else {
            NMAPClientFree(sclient);

            return(ConnWrite(client->conn, MSG4224NOMBOX, sizeof(MSG4224NOMBOX) - 1));
        }
    } else if (IsAvailableShare(sclient->user, sclient->mailbox.name, NMAP_SHARE_MAILBOX, &resourceOwner, &resourceName)) {
        ccode = ConnectToNMAPShare(sclient, resourceOwner, resourceName, NMAP_SHARE_MAILBOX, &(sclient->share.mailbox.permissions));

        MemFree(resourceOwner);
        MemFree(resourceName);

        if ((ccode == 1000) && (sclient->share.mailbox.permissions & NMAP_SHARE_READ)){
            if (ReadNLockAquire(&sclient->mailbox.lock, &sclient->userHash, sclient->user, sclient->mailbox.name, 0) == NLOCK_AQUIRED) {
                result = SelectNMAPShare(sclient, resourceOwner, resourceName, NMAP_SHARE_MAILBOX);
                ReadNLockRelease(sclient->mailbox.lock);
                sclient->mailbox.lock = NULL;
            } else {
                result = FALSE;
            }
         } else if (ccode == 1000) {
            NMAPQuit(sclient->share.mailbox.conn);
            ConnFree(sclient->share.mailbox.conn);
            sclient->share.mailbox.conn = NULL;

            sclient->share.mailbox.permissions = 0;
            sclient->share.flags &= ~(NMAP_SHARE_MAILBOX | NMAP_SHARE_MAILBOX_DENIED);

            NMAPClientFree(sclient);
            return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
        } else {
            NMAPClientFree(sclient);
            return(ConnWrite(client->conn, MSG5004INTERNALERR, sizeof(MSG5004INTERNALERR) - 1));
        }

        if (sclient->share.mailbox.conn) {
            NMAPQuit(sclient->share.mailbox.conn);
            ConnFree(sclient->share.mailbox.conn);
            sclient->share.mailbox.conn = NULL;
        }

        if (result == FALSE) {
            NMAPClientFree(sclient);
            return(ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1));
        }
    } else {
        NMAPClientFree(sclient);
        return(ConnWrite(client->conn, MSG4224NOMBOX, sizeof(MSG4224NOMBOX) - 1));
    }

    i = sclient->mailbox.message.used;
    msgInfo = sclient->mailbox.message.info;

    while (i-- != 0) {
        totalSize += msgInfo->SSize;
        totalCount++;

        if (msgInfo->State & MSG_STATE_PURGED) {
            purgedSize += msgInfo->SSize;
            purgedCount++;
        } else {
            if (msgInfo->State & MSG_STATE_READ) {
                readSize += msgInfo->SSize;
                readCount++;
            }
        }

        msgInfo++;
    }

    if (sclient->mailbox.message.used > 0) {
        ccode = ConnWriteF(client->conn, "1000 %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu\r\n",
            totalCount, totalSize, sclient->mailbox.newCount, sclient->mailbox.newSize, purgedCount, purgedSize,
            readCount, readSize,sclient->mailbox.message.info[sclient->mailbox.message.used - 1].UID + 1, 
            sclient->mailbox.id);
    } else {
        ccode = ConnWriteF(client->conn, "1000 0 0 0 0 0 0 0 0 %lu %lu\r\n", sclient->mailbox.nextUID, sclient->mailbox.id);
    }

    if (sclient->mailbox.message.info) {
        MemFree(sclient->mailbox.message.info);
        sclient->mailbox.message.info = NULL;
    }

    NMAPClientFree(sclient);

    return(ccode);
}

static void 
RemoveDir(unsigned char *path, unsigned char *scratch, unsigned char *pattern, int skip, BOOL cleanFiles, unsigned long recurse)
{
    int len;
    int len2;
    unsigned char *lBuffer;
    XplDir *dirEntryP;
    XplDir *dirP;

    lBuffer = MemMalloc(strlen(path) + 7); /* "\*.*ck\0" */
    if (!lBuffer) {
        return;
    }

    len = sprintf(lBuffer, "%s", path);

    XplSetThreadGroupID(NMAP.id.load);

    len2 = strlen(pattern);

    if ((dirEntryP = XplOpenDir(lBuffer)) != NULL) {
        while ((dirP = XplReadDir(dirEntryP)) != NULL) {
            if ((dirP->d_attr & XPL_A_SUBDIR) && (dirP->d_name[0] != '.')) {
                sprintf(scratch, "%s/%s", lBuffer, dirP->d_name);
                rmdir(scratch);
                if (recurse) {
                    RemoveDir(scratch, scratch, pattern, skip, cleanFiles, recurse-1);

                    /* We have to set it after the recursive call, since it gets reset at the end of printdir. */
                    XplSetThreadGroupID(NMAP.id.load);
                }
            } else {
                if (cleanFiles) {
                    sprintf(scratch, "%s/%s", lBuffer, dirP->d_name);
                    unlink(scratch);
                }
            }
        }

        XplCloseDir(dirEntryP);
    }

    rmdir(lBuffer);

    MemFree(lBuffer);

    XplSetThreadGroupID(NMAP.id.group);

    return;
}

int 
NmapCommandClr(void *param)
{
    int ccode;
    unsigned long count;
    unsigned char *ptr;
    unsigned char pattern[64] = "";
    unsigned char dn[MDB_MAX_OBJECT_CHARS + 1];
    unsigned char scratch[(XPL_MAX_PATH * 2) + 1];
    const unsigned char *storePath;
    BOOL recurse = 99999;
    struct stat sb;
    MDBValueStruct *vs;
    NMAPClient *client = (NMAPClient *)param;

    /* fixme - using a BOOL as an int! see recurse above */
    if (client->states & NMAP_CLIENT_AUTHORIZED) {
        if (client->states & NMAP_CLIENT_MBOX) {
            ccode = MailboxOOBNotifications(client, &sb);
            if (ccode == -1) {
                return(ccode);
            }
        }

        ptr = client->buffer + 3;
        if ((isspace(*ptr)) 
                && (!(isspace(*(++ptr)))) 
                && (*ptr != '\0') 
                && (strchr(ptr, '/') == NULL) 
                && (strchr(ptr, '\\') == NULL)) {
            storePath = MsgFindUserStore(ptr, NMAP.path.mail);
        } else {
            return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
        }

    } else {
        if (client->states & NMAP_CLIENT_USER) {
            return(ConnWrite(client->conn, MSG4242NOTALLOWED, sizeof(MSG4242NOTALLOWED) - 1));
        }
        return(ConnWrite(client->conn, MSG3240NOAUTH, sizeof(MSG3240NOAUTH) - 1));
    }

    /* Security check */
    if (*ptr == '\0' || strchr(ptr, '/') || strchr(ptr, '\\')) {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    sprintf(scratch, "%s/%s", storePath, ptr);
    if ((access(scratch, 0) == 0) 
            && (MsgFindObject(ptr, dn, NULL, NULL, NULL))) {
        vs = MDBCreateValueStruct(NMAP.handle.directory, NULL);
        if (vs) {
            count = MDBRead(dn, MSGSRV_A_OWNED_SHARES, vs);
            MDBDestroyValueStruct(vs);

            if (!count) {
                if (!XPLLongToDos(scratch, client->path, XPL_MAX_PATH)) {
                    return(ConnWrite(client->conn, MSG4224NOSTORE, sizeof(MSG4224NOSTORE) - 1));
                }
            } else {
                return(ConnWrite(client->conn, MSG4220USERSHARES, sizeof(MSG4220USERSHARES) - 1));
            }
        } else {
            return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    RemoveAvailableShares(client);
    RemoveDir(client->path, scratch, pattern, strlen(client->path), TRUE, recurse);

    return(ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1));
}

__inline static int 
CopyMessageFromLocalToLocal(NMAPClient *client, unsigned long id, unsigned char *target)
{
    int count;
    unsigned long len;
    unsigned char *ptr;
    FILE *targetBox;
    NLockStruct *ShortTermMBoxLock;

    ShortTermMBoxLock = NULL;
    if ((ReadNLockAquire(&ShortTermMBoxLock, &client->userHash, client->user, target, 3000) == NLOCK_AQUIRED) 
            && (WriteNLockAquire(ShortTermMBoxLock, 2000) == NLOCK_AQUIRED)) {
        targetBox = fopen(client->path, "ab");
    } else {
        if (ShortTermMBoxLock) {
            ReadNLockRelease(ShortTermMBoxLock);
        }

        return(ConnWrite(client->conn, MSG4228CANTWRITEMBOX, sizeof(MSG4228CANTWRITEMBOX) - 1));
    }

    if (targetBox != NULL) {
        sprintf(client->path, "%s/%s/%s.box", client->store, client->user, client->mailbox.name);

        client->mailbox.fh = fopen(client->path, "rb");
    } else {
        WriteNLockRelease(ShortTermMBoxLock);
        ReadNLockRelease(ShortTermMBoxLock);

        return(ConnWrite(client->conn, MSG4228CANTWRITEMBOX, sizeof(MSG4228CANTWRITEMBOX) - 1));
    }

    if (client->mailbox.fh != NULL) {
        if (!client->mailbox.message.info[id].UseSCMS) {
            count = client->mailbox.message.info[id].MSize;
            fseek(client->mailbox.fh, client->mailbox.message.info[id].HeadPos, SEEK_SET);
        } else {
            count = client->mailbox.message.info[id].RealSize;
            fseek(client->mailbox.fh, client->mailbox.message.info[id].RealStart, SEEK_SET);

            SCMSStoreMessageReference(client->mailbox.message.info[id].SCMSID);
        }

        /* We need to copy the header line by line, so we can insert the flags */
        while (!feof(client->mailbox.fh) && !ferror(client->mailbox.fh)) {
            if (fgets(client->line, CONN_BUFSIZE, client->mailbox.fh) != NULL) {
                len = strlen(client->line);
                count -= len;

                if ((client->line[0] == '\r') && (client->line[1] == '\n')) {
                    len = sprintf(client->line, "X-NIMS-Flags: %lu %lu\r\n\r\n", client->mailbox.message.info[id].State, client->mailbox.message.info[id].DateReceived);
                    fwrite(client->line, sizeof(unsigned char), len, targetBox);
                    break;
                }

                if (!QuickNCmp(client->line, "X-NIMS-Flags:", 13)) {
                    fwrite(client->line, sizeof(unsigned char), len, targetBox);
                }
            }
        }
        
        while (count > 0) {
            if (count < CONN_BUFSIZE) {
                len = fread(client->line, sizeof(unsigned char), count, client->mailbox.fh);
            } else {
                len = fread(client->line, sizeof(unsigned char), CONN_BUFSIZE, client->mailbox.fh);
            }

            if (len > 0) {
                fwrite(client->line, sizeof(unsigned char), len, targetBox);
                count -= len;
            } else {
                /* fixme - correct error handling */
                break;
            }
        }

        fclose(client->mailbox.fh);
        client->mailbox.fh = NULL;

        if (NMAP.flushFlags & FLUSH_BOX_ON_STORE) {
            XplFlushWrites(targetBox);
        }

        ptr = MSG1000OK;
        count = sizeof(MSG1000OK) - 1;
    } else {
        ptr = MSG4224CANTREADMBOX;
        count = sizeof(MSG4224CANTREADMBOX) - 1;
    }

    fclose(targetBox);

    WriteNLockRelease(ShortTermMBoxLock);

    ReadNLockRelease(ShortTermMBoxLock);
    ShortTermMBoxLock = NULL;

    return(ConnWrite(client->conn, ptr, count));
}

__inline static int 
CopyMessageFromLocalToShare(NMAPClient *client, unsigned long id, unsigned char *target)
{
    int count;
    int ccode;
    unsigned long len;
    unsigned char *owner = NULL;
    unsigned char *mailbox = NULL;
    NMAPClient *sclient = NULL;

    do {
        if (IsAvailableShare(client->user, target, NMAP_SHARE_MAILBOX, &owner, &mailbox)) {
            sclient = NMAPClientAlloc();
        } else {
            ccode = ConnWrite(client->conn, MSG4224NOMBOX, sizeof(MSG4224NOMBOX) - 1);
            break;
        }

        if (sclient != NULL) {
            strcpy(sclient->user, client->user);

            sclient->userHash = client->userHash; 
            sclient->mailbox.newline = TRUE;

            sprintf(client->path, "%s/%s/%s.box", client->store, client->user, client->mailbox.name);
            client->mailbox.fh = fopen(client->path, "rb");
        } else {
            ccode = ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1);
        }

        if (client->mailbox.fh != NULL) {
            ccode = ConnectToNMAPShare(sclient, owner, mailbox, NMAP_SHARE_MAILBOX, &(sclient->share.mailbox.permissions));
        } else {
            ccode = ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1);
            break;
        }

        if ((ccode == 1000) && (sclient->share.mailbox.permissions & NMAP_SHARE_INSERT)) {
            ccode = NMAPSendCommandF(sclient->share.mailbox.conn, "USER %s\r\n", owner);
        } else if (ccode == 1000) {
            ccode = ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1);
            break;
        } else {
            ccode = ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1);
            break;
        }

        if (ccode != -1) {
            ccode = NMAPReadAnswerLine(sclient->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE);
        } else {
            ccode = ConnWrite(client->conn, MSG5004INTERNALERR, sizeof(MSG5004INTERNALERR) - 1);
            break;
        }

        if (ccode == 1000) {
            if (!client->mailbox.message.info[id].UseSCMS) {
                count = client->mailbox.message.info[id].MSize;
                fseek(client->mailbox.fh, client->mailbox.message.info[id].HeadPos, SEEK_SET);
            } else {
                count = client->mailbox.message.info[id].RealSize;
                fseek(client->mailbox.fh, client->mailbox.message.info[id].RealStart, SEEK_SET);

                SCMSStoreMessageReference(client->mailbox.message.info[id].SCMSID);
            }

            ccode = NMAPSendCommandF(sclient->share.mailbox.conn, "STRAW %s 0\r\n", mailbox);
        } else if (ccode != -1) {
            ccode = ConnWrite(client->conn, client->line, strlen(client->line));
            break;
        } else {
            ccode = ConnWrite(client->conn, MSG5004INTERNALERR, sizeof(MSG5004INTERNALERR) - 1);
            break;
        }

        if ((ccode != -1) 
                && ((ccode = NMAPReadAnswerLine(sclient->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                && (ccode == 2002)) {
            /* Copy the header line by line to insert the flags */
            while (!feof(client->mailbox.fh) && !ferror(client->mailbox.fh) && (ccode != -1)) {
                if (fgets(client->line, CONN_BUFSIZE, client->mailbox.fh) != NULL) {
                    len = strlen(client->line);
                    count -= len;

                    if ((client->line[0] == '\r') && (client->line[1] == '\n')) {
                        ccode = ConnWriteF(sclient->share.mailbox.conn, "X-NIMS-Flags: %lu %lu\r\n\r\n", client->mailbox.message.info[id].State, client->mailbox.message.info[id].DateReceived);
                        break;
                    }

                    if (!QuickNCmp(client->line, "X-NIMS-Flags:", 13)) {
                        ccode = ConnWrite(sclient->share.mailbox.conn, client->line, len);
                    }
                }
            }

            while ((count > 0) && (ccode != -1)) {
                if (count < CONN_BUFSIZE) {
                    len = fread(client->line, sizeof(unsigned char), count, client->mailbox.fh);
                } else {
                    len = fread(client->line, sizeof(unsigned char), CONN_BUFSIZE, client->mailbox.fh);
                }

                if (len > 0) {
                    if (client->line[0] != '.') {
                        ccode = ConnWrite(sclient->share.mailbox.conn, client->line, len);
                    } else {
                        ccode = ConnWrite(sclient->share.mailbox.conn, ".", 1);
                        if (ccode != -1) {
                            ccode = ConnWrite(sclient->share.mailbox.conn, client->line, len);
                        }
                    }

                    count -= len;
                } else {
                    break;
                }
            }
        } else if (ccode != -1) {
            ccode = ConnWrite(client->conn, client->line, strlen(client->line));
            break;
        } else {
            ccode = ConnWrite(client->conn, MSG4228CANTWRITEMBOX, sizeof(MSG4228CANTWRITEMBOX) - 1);
            break;
        }

        if ((ccode != -1) 
                && ((ccode = NMAPSendCommand(sclient->share.mailbox.conn, ".\r\n", 3)) != -1) 
                && ((ccode = NMAPReadAnswerLine(sclient->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE)) != -1)) {
            ccode = ConnWrite(client->conn, client->line, strlen(client->line));
        }

        fclose(client->mailbox.fh);
        client->mailbox.fh = NULL;
    } while (TRUE);

    if (mailbox) {
        MemFree(mailbox);
    }

    if (owner) {
        MemFree(owner);
    }

    if (sclient) {
        if (sclient->mailbox.message.info != NULL) {
            MemFree(sclient->mailbox.message.info);
            sclient->mailbox.message.info = NULL;
        }

        NMAPQuit(sclient->share.mailbox.conn);
        ConnFree(sclient->share.mailbox.conn);

        NMAPClientFree(sclient);
    }

    return(ccode);
}

__inline static int 
CopyMessageFromShareToLocal(NMAPClient *client, unsigned long id, unsigned char *target)
{
    int count;
    int ccode;
    unsigned long len;
    unsigned char *ptr;
    unsigned char authSender[MAXEMAILNAMESIZE + 1];
    unsigned char timeBuf[80];
    FILE *targetBox = NULL;
    NLockStruct *ShortTermMBoxLock = NULL;

    do {
        if ((ReadNLockAquire(&ShortTermMBoxLock, &client->userHash, client->user, target, 3000) == NLOCK_AQUIRED) 
                && (WriteNLockAquire(ShortTermMBoxLock, 2000) == NLOCK_AQUIRED)) {
            targetBox = fopen(client->path, "ab");
        } else if (ShortTermMBoxLock) {
            ReadNLockRelease(ShortTermMBoxLock);
            ShortTermMBoxLock = NULL;

            ccode = ConnWrite(client->conn, MSG4228CANTWRITEMBOX, sizeof(MSG4228CANTWRITEMBOX) - 1);
            break;
        } else {
            ccode = ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1);
            break;
        }

        if (targetBox) {
            ccode = NMAPSendCommandF(client->share.mailbox.conn, "AINFO %lu\r\n", id + 1);
        } else {
            ccode = ConnWrite(client->conn, MSG4228CANTWRITEMBOX, sizeof(MSG4228CANTWRITEMBOX) - 1);
            break;
        }

        if ((ccode != -1) 
                && ((ccode = NMAPReadAnswerLine(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE)) != -1)
                && (ccode == 2001)) {
            ptr = strrchr(client->line, ' ');
            if (ptr != NULL) {
                strcpy(authSender, ++ptr);
            } else {
                authSender[0] = '-';
                authSender[1] = '\0';
            }

            strftime(timeBuf, sizeof(timeBuf), "%a %b %d %H:%M %Y", gmtime((time_t *)&(client->mailbox.message.info[id].DateReceived)));
            GENERATE_FROM_DELIMITER(timeBuf, authSender, client->line, len);

            fwrite(client->line, sizeof(unsigned char), len, targetBox);                                    
        } else if (ccode != -1) {
            ccode = ConnWrite(client->conn, client->line, strlen(client->line));
            break;
        } else {
            ccode = ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1);
            break;
        }

        if (((ccode = NMAPSendCommandF(client->share.mailbox.conn, "HEAD %lu\r\n", id + 1)) != -1) 
                && ((ccode = NMAPReadAnswer(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                && (ccode == 2020)) {
            count = atol(client->line + 5);
            while ((count > 0) && (ccode != -1)) {
                ccode = ConnReadLine(client->share.mailbox.conn, client->line, CONN_BUFSIZE);
                if (ccode != -1) {
                    count -= ccode;

                    if ((client->line[0] == '\r') && (client->line[1] == '\n')) {
                        ccode = sprintf(client->line, "X-NIMS-Flags: %lu %lu\r\n\r\n", client->mailbox.message.info[id].State, client->mailbox.message.info[id].DateReceived);
                        fwrite(client->line, sizeof(unsigned char), ccode, targetBox);
                        break;
                    }

                    if (!QuickNCmp(client->line, "X-NIMS-Flags:", 13)) {
                        fwrite(client->line, sizeof(unsigned char), ccode, targetBox);
                    }
                }
            }
        } else if (ccode != -1) {
            ccode = ConnWrite(client->conn, client->line, strlen(client->line));
            break;
        } else {
            ccode = ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1);
            break;
        } 

        if ((ccode != -1) 
                && ((ccode = NMAPReadAnswerLine(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                && (ccode == 1000) 
                && ((ccode = NMAPSendCommandF(client->share.mailbox.conn, "BODY %lu\r\n", id + 1)) != -1) 
                && ((ccode = NMAPReadAnswerLine(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                && ((ccode == 2021) || (ccode == 2025))) {
            count = atol(client->line + 5);
        } else if (ccode != -1) {
            ccode = ConnWrite(client->conn, client->line, strlen(client->line));
            break;
        } else {
            ccode = ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1);
            break;
        }

        if (((ccode = ConnReadToFile(client->share.mailbox.conn, targetBox, count)) != -1) 
                && ((ccode = NMAPReadAnswer(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                && (ccode == 1000)) {
            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
        } else if (ccode != -1) {
            ccode = ConnWrite(client->conn, client->line, strlen(client->line));
        } else {
            ccode = ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1);
        } 

        break;
    } while (TRUE);

    if (targetBox) {
        fclose(targetBox);
    }

    if (ShortTermMBoxLock) {
        WriteNLockRelease(ShortTermMBoxLock);

        ReadNLockRelease(ShortTermMBoxLock);
    }

    return(ccode);
}

__inline static int 
CopyMessageFromShareToShare(NMAPClient *client, unsigned long id, unsigned char *target)
{
    int count;
    int ccode;
    unsigned long len;
    unsigned char *ptr;
    unsigned char *owner = NULL;
    unsigned char *mailbox = NULL;
    unsigned char authSender[MAXEMAILNAMESIZE + 1];
    unsigned char timeBuf[80];
    BOOL escaped;
    NMAPClient *sclient = NULL;

    do {
        if (IsAvailableShare(client->user, target, NMAP_SHARE_MAILBOX, &owner, &mailbox)) {
            sclient = NMAPClientAlloc();
        } else {
            ccode = ConnWrite(client->conn, MSG4224NOMBOX, sizeof(MSG4224NOMBOX) - 1);
            break;
        }

        if (sclient != NULL) {
            strcpy(sclient->user, client->user);

            sclient->userHash = client->userHash; 
            sclient->mailbox.newline = TRUE;

            ccode = ConnectToNMAPShare(sclient, owner, mailbox, NMAP_SHARE_MAILBOX, &(sclient->share.mailbox.permissions));
        } else {
            ccode = ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1);
            break;
        }

        if ((ccode == 1000) && (sclient->share.mailbox.permissions & NMAP_SHARE_INSERT)) {
            ccode = NMAPSendCommandF(sclient->share.mailbox.conn, "USER %s\r\n", owner);
        } else if (ccode == 1000) {
            ccode = ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1);
            break;
        } else {
            ccode = ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1);
            break;
        }

        if ((ccode != -1) 
                && ((ccode = NMAPReadAnswerLine(sclient->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                && (ccode == 1000) 
                && ((ccode = NMAPSendCommandF(client->share.mailbox.conn, "AINFO %lu\r\n", id + 1)) != -1) 
                && ((ccode = NMAPReadAnswer(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                && (ccode == 2001)) {
            ptr = strrchr(client->line, ' ');
            if (ptr != NULL) {
                strcpy(authSender, ++ptr);
            } else {
                authSender[0] = '-';
                authSender[1] = '\0';
            }
        } else if (ccode == -1) {
            ccode = ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1);
            break;
        } else {
            ccode = ConnWrite(client->conn, client->line, strlen(client->line));
            break;
        }

        if (((ccode = NMAPSendCommandF(sclient->share.mailbox.conn, "STRAW %s 0\r\n", mailbox)) != -1)
                && ((ccode = NMAPReadAnswer(sclient->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                && (ccode == 2002)) {
            strftime(timeBuf, sizeof(timeBuf), "%a %b %d %H:%M %Y", gmtime((time_t *)&(client->mailbox.message.info[id].DateReceived)));
            GENERATE_FROM_DELIMITER(timeBuf, authSender, client->line, len);
        } else if (ccode == -1) {
            ccode = ConnWrite(client->conn, MSG4228CANTWRITEMBOX, sizeof(MSG4228CANTWRITEMBOX) - 1);
            break;
        } else {
            ccode = ConnWrite(client->conn, client->line, strlen(client->line));
            break;
        }

        if ((ccode = ConnWrite(sclient->share.mailbox.conn, client->line, len)) != -1) {
            if (((ccode = NMAPSendCommandF(client->share.mailbox.conn, "HEAD %lu\r\n", id + 1)) != -1) 
                    && ((ccode = NMAPReadAnswer(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE)) != -1)
                    && (ccode == 2020)) {
                count = atol(client->line + 5);
            } else if (ccode == -1) {
                ccode = ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1);
                break;
            } else {
                ccode = ConnWrite(client->conn, client->line, strlen(client->line));
                break;
            }
        } else if (ccode == -1) {
            ccode = ConnWrite(client->conn, MSG4228CANTWRITEMBOX, sizeof(MSG4228CANTWRITEMBOX) - 1);
            break;
        } else {
            ccode = ConnWrite(client->conn, client->line, strlen(client->line));
            break;
        }

        /* Transfer the header to the destination share. */
        while ((ccode != -1) && (count > 0)) {
            ccode = ConnReadLine(client->share.mailbox.conn, client->line, CONN_BUFSIZE);
            if (ccode != -1) {
                count -= ccode;

                if ((client->line[0] == '\r') && (client->line[1] == '\n')) {
                    ccode = ConnWriteF(sclient->share.mailbox.conn, "X-NIMS-Flags: %lu %lu\r\n\r\n", client->mailbox.message.info[id].State, client->mailbox.message.info[id].DateReceived);
                    break;
                }

                if (!QuickNCmp(client->line, "X-NIMS-Flags:", 13)) {
                    ccode = ConnWrite(sclient->share.mailbox.conn, client->line, ccode);
                }
            }
        }

        if (ccode != -1) {
            if (((ccode = NMAPReadAnswer(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                    && (ccode == 1000) 
                    && ((ccode = NMAPSendCommandF(client->share.mailbox.conn, "BODY %lu\r\n", id + 1)) != -1) 
                    && ((ccode = NMAPReadAnswer(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                    && ((ccode == 2021) || (ccode == 2025))) {
                if (ccode == 2021) {
                    escaped = FALSE;
                } else {
                    escaped = TRUE;
                }

                count = atol(client->line + 5);
            } else if (ccode != -1) {
                ccode = ConnWrite(client->conn, client->line, strlen(client->line));
                break;
            } else {
                ccode = ConnWrite(client->conn, MSG4228CANTWRITEMBOX, sizeof(MSG4228CANTWRITEMBOX) - 1);
                break;
            }
        } else {
            ccode = ConnWrite(client->conn, MSG4228CANTWRITEMBOX, sizeof(MSG4228CANTWRITEMBOX) - 1);
            break;
        }

        /* Transfer the body to the destination share.    */
        if (!escaped) {
            ccode = ConnReadToConn(client->share.mailbox.conn, sclient->share.mailbox.conn, count);
        } else {
            while ((ccode != -1) && (count > 0)) {
                ccode = ConnReadLine(client->share.mailbox.conn, client->line, CONN_BUFSIZE);
                if (ccode != -1) {
                    count -= len;

                    if (client->line[0] != '.') {
                        ccode = ConnWrite(sclient->share.mailbox.conn, client->line, ccode);
                    } else {
                        if (ConnWrite(sclient->share.mailbox.conn, ".", 1) != -1) {
                            ccode = ConnWrite(sclient->share.mailbox.conn, client->line, ccode);
                            continue;
                        }

                        ccode = -1;
                        break;
                    }
                }
            }
        }

        if ((ccode != -1) 
                && ((ccode = NMAPSendCommand(sclient->share.mailbox.conn, ".\r\n", 3)) != -1)) {
            if (((ccode = NMAPReadAnswer(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                    && (ccode == 1000) 
                    && ((ccode = NMAPReadAnswer(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                    && (ccode == 1000)) {
                ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
            } else if (ccode == -1) {
                ccode = ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1);
            } else {
                ccode = ConnWrite(client->conn, client->line, strlen(client->line));
            }
        } else {
            ccode = ConnWrite(client->conn, MSG4228CANTWRITEMBOX, sizeof(MSG4228CANTWRITEMBOX) - 1);
        }

        break;
    } while (TRUE);

    if (owner) {
        MemFree(owner);
    }

    if (mailbox) {
        MemFree(mailbox);
    }

    if (sclient) {
        if (sclient->mailbox.message.info != NULL) {
            MemFree(sclient->mailbox.message.info);
            sclient->mailbox.message.info = NULL;
        }

        if (sclient->share.mailbox.conn) {
            NMAPQuit(sclient->share.mailbox.conn);
            ConnFree(sclient->share.mailbox.conn);
        }

        NMAPClientFree(sclient);
    }

    return(ccode);
}

int 
NmapCommandCopy(void *param)
{
    int ccode;
    unsigned long id;
    unsigned char *ptr;
    unsigned char *target;
    struct stat sb;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_MBOX) {
        if (!(NMAP.flags & NMAP_FLAG_DISK_SPACE_LOW)) {
            ccode = MailboxOOBNotifications(client, &sb);
            if (ccode != -1) {
                ptr = client->buffer + 4;
            } else {
                return(ccode);
            }

            ptr = client->buffer + 4;
            if ((*ptr == ' ') && ((ptr = strchr(++ptr, ' ')) != NULL)) {
                *ptr++ = '\0';
            } else {
                return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
            }
        } else {
            return(ConnWrite(client->conn, MSG5221SPACELOW, sizeof(MSG5221SPACELOW) - 1));
        }
    } else if (!(client->states & NMAP_CLIENT_USER)) {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    } else {
        return(ConnWrite(client->conn, MSG4225NOMBOX, sizeof(MSG4225NOMBOX) - 1));
    }

    target = ptr;
    if (XplStrCaseCmp(target, client->mailbox.name) != 0) {
        id = atoi(client->buffer + 5);
        if (id && (id <= client->mailbox.message.used)) {
            id--;
        } else {
            return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG4227TGTMBOXSELECTED, sizeof(MSG4227TGTMBOXSELECTED) - 1));
    }

    if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
        if (client->flags & NMAP_OOBMESSAGES) {
            sprintf(client->path, "%s/%s/%s.idx", client->store, client->user, client->mailbox.name);
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
        }

        sprintf(client->path,"%s/%s/%s.box", client->store, client->user, target);
        if (access(client->path, 0) == 0) {
            ccode = CopyMessageFromLocalToLocal(client, id, target);
        } else {
            ccode = CopyMessageFromLocalToShare(client, id, target);
        }
    } else if (!(client->share.flags & NMAP_SHARE_MAILBOX_DENIED)) {
        if (client->flags & NMAP_OOBMESSAGES) {
            sprintf(client->path, "%s/%s/%s.sdx", client->store, client->user, client->mailbox.name);
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
        }

        sprintf(client->path,"%s/%s/%s.box", client->store, client->user, target);
        if (access(client->path, 0) == 0) {
            ccode = CopyMessageFromShareToLocal(client, id, target);
        } else {
            ccode = CopyMessageFromShareToShare(client, id, target);
        }
    } else {
        ccode = ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1);
    }

    return(ccode);
}

int 
NmapCommandCrea(void *param)
{
    int ccode;
    unsigned char *ptr;
    unsigned char *ptr2;
    unsigned char *path;
    unsigned char *name;
    unsigned char scratch[(2 * XPL_MAX_PATH) + 1];
    struct stat sb;
    FILE *handle;
    NMAPClient *client = (NMAPClient *)param;
    MDBValueStruct *vs;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->states & NMAP_CLIENT_MBOX) {
            ccode = MailboxOOBNotifications(client, &sb);
            if (ccode == -1) {
                return(ccode);
            }
        }

        ptr = client->buffer + 4;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    } 

    if ((*ptr == ' ') && !(ptr2 = strchr(++ptr, ' ')) 
            && ((strchr(ptr, '.') == NULL) || ((strstr(ptr, "../") == NULL) && (strstr(ptr, "..\\") == NULL)))) {
        path = ptr;
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if ((strlen(client->store) + strlen(client->user) + strlen(path) + 4) <= XPL_MAX_PATH) {
    } else {
        return(ConnWrite(client->conn, MSG4230PATHTOOLONG, sizeof(MSG4230PATHTOOLONG) - 1));
    }

    name = strrchr(path, '/');
    if (name != NULL) {
        ptr = strchr(path, '/');

        while (ptr) {
            *ptr = '\0';

            sprintf(scratch, "%s/%s/%s", client->store, client->user, path);
            if (stat(scratch, &sb) != 0) {
                XplMakeDir(scratch);
                SetLongName(scratch, client->path);
            }

            *ptr = '/';
            ptr2 = ptr;
            ptr = strchr(ptr2 + 1, '/');
        }

        *name = '/';
    }

    sprintf(scratch, "%s/%s/%s.box", client->store, client->user, path);
    handle = fopen(scratch, "r");
    if ((handle == NULL) 
            && (IsAvailableShare(client->user, path, NMAP_SHARE_MAILBOX, NULL, NULL) == FALSE)) {
        handle = fopen(scratch, "wb");
        if (handle != NULL) {
            fclose(handle);

            SetLongName(scratch, client->path);

            sprintf(scratch, "%s/%s/%s.sdx", client->store, client->user, path);
            unlink(scratch);

            sprintf(scratch, "%s/%s/%s.sdc", client->store, client->user, path);
            unlink(scratch);

            sprintf(scratch, "%s/%s/%s.idx", client->store, client->user, path);
            handle = fopen(scratch, "wb");

            SetLongName(scratch, client->path);
            fprintf(handle, "%s\r\n0000000000\r\n0000000000\r\n0000000000\r\n%010lu\r\n%010lu\r\n", NMAP_IDX_VERSION, NMAP.universalCounter++, NMAP.universalCounter++);

            sprintf(client->line, "%lu", NMAP.universalCounter);
            vs = MDBCreateValueStruct(NMAP.handle.directory, NMAP.server.dn);
            MDBAddValue(client->line, vs);
            MDBWrite(MSGSRV_AGENT_NMAP, MSGSRV_A_UID, vs);
            MDBDestroyValueStruct(vs);
            fclose(handle);

            ccode = ConnWrite(client->conn, MSG1000MBOXMADE, sizeof(MSG1000MBOXMADE) - 1);
        } else {
            ccode = ConnWriteF(client->conn, MSG3014REQNOTALLOWED, " Illegal mailbox name.");
        }
    } else {
        if (handle != NULL) {
            fclose(handle);
        }

        ccode = ConnWrite(client->conn, MSG4226MBOXEXISTS, sizeof(MSG4226MBOXEXISTS) - 1);
    }

    return(ccode);
}

int 
NmapCommandDele(void *param)
{
    int ccode;
    unsigned long id;
    unsigned long delCount = 0;
    unsigned char *ptr;
    struct stat sb;
    FILE *index;
    MessageInfoStruct *msgInfo;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_MBOX) {
        if (!(client->flags & NMAP_READONLY)) {
            ccode = MailboxOOBNotifications(client, &sb);
            if (ccode != -1) {
                ptr = client->buffer + 4;
            } else {
                return(ccode);
            }
        } else {
            return(ConnWrite(client->conn, MSG4243READONLY, sizeof(MSG4243READONLY) - 1));
        }
    } else if (!(client->states & NMAP_CLIENT_USER)) {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    } else {
        return(ConnWrite(client->conn, MSG4225NOMBOX, sizeof(MSG4225NOMBOX) - 1));
    }

    if (*ptr == '\0') {
        if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
            sprintf(client->path, "%s/%s/%s.idx", client->store, client->user, client->mailbox.name);

            if (client->flags & NMAP_OOBMESSAGES) {
                GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
            }
        } else if (!(client->share.flags & NMAP_SHARE_MAILBOX_DENIED) && (client->share.mailbox.permissions & NMAP_SHARE_DELETE)) {
            sprintf(client->path, "%s/%s/%s.sdx", client->store, client->user, client->mailbox.name);

            if (client->flags & NMAP_OOBMESSAGES) {
                GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
            }
        } else {
            return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
        }

        id = client->mailbox.message.used;
        msgInfo = client->mailbox.message.info;

        while (id-- != 0) {
            if (msgInfo->State & MSG_STATE_DELETED) {
                delCount++;
            }

            msgInfo++;
        }

        ccode = ConnWriteF(client->conn, "2002 %lu %s\r\n", delCount, MSG2002DELETED);

        id = client->mailbox.message.used;
        msgInfo = client->mailbox.message.info;

        while ((ccode != -1) && (id-- != 0)) {
            if (msgInfo->State & MSG_STATE_DELETED) {
                ccode = ConnWriteF(client->conn, "%lu\r\n", id + 1);
            }

            msgInfo++;
        }

        if (ccode != -1) {
            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
        }

        return(ccode);
    } else if ((*ptr++ == ' ') && (isdigit(*ptr))) {
        id = atoi(ptr);
        if (id && (id <= client->mailbox.message.used)) {
            id--;
        } else {
            return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (!(client->mailbox.message.info[id].State & MSG_STATE_DELETED)) {
        if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
            if (client->flags & NMAP_OOBMESSAGES) {
                sprintf(client->path, "%s/%s/%s.idx", client->store, client->user, client->mailbox.name);

                GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 

                if (client->mailbox.message.info[id].State & MSG_STATE_DELETED) {
                    return(ConnWrite(client->conn, MSG4222MSGDELETED, sizeof(MSG4222MSGDELETED) - 1));
                }
            } else {
                if (client->mailbox.message.info[id].State & MSG_STATE_DELETED) {
                    return(ConnWrite(client->conn, MSG4222MSGDELETED, sizeof(MSG4222MSGDELETED) - 1));
                }

                sprintf(client->path, "%s/%s/%s.idx", client->store, client->user, client->mailbox.name);
            }
        } else if (!(client->share.flags & NMAP_SHARE_MAILBOX_DENIED) && (client->share.mailbox.permissions & NMAP_SHARE_DELETE)) {
            if (client->flags & NMAP_OOBMESSAGES) {
                sprintf(client->path, "%s/%s/%s.sdx", client->store, client->user, client->mailbox.name);

                GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 

                if (client->mailbox.message.info[id].State & MSG_STATE_DELETED) {
                    return(ConnWrite(client->conn, MSG4222MSGDELETED, sizeof(MSG4222MSGDELETED) - 1));
                }
            } else {
                if (client->mailbox.message.info[id].State & MSG_STATE_DELETED) {
                    return(ConnWrite(client->conn, MSG4222MSGDELETED, sizeof(MSG4222MSGDELETED) - 1));
                }

                sprintf(client->path, "%s/%s/%s.sdx", client->store, client->user, client->mailbox.name);
            }
        } else {
            return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
        }

        if (WriteNLockAquire(client->mailbox.lock, 3000) == NLOCK_AQUIRED) {
            index = fopen(client->path, "r+b");
            if (index != NULL) {
                fseek(index, NMAP_IDX_HEADERSIZE + (id * sizeof(MessageInfoStruct)) + offsetof(MessageInfoStruct, State), SEEK_SET);

                client->mailbox.message.info[id].State |= MSG_STATE_DELETED;
                client->mailbox.message.count--;

                if (!(client->mailbox.message.info[id].State & MSG_STATE_RECENT)) {
                    fwrite(&client->mailbox.message.info[id].State, sizeof(unsigned long), 1, index);
                } else {
                    client->mailbox.message.info[id].State &= ~MSG_STATE_RECENT;
                    fwrite(&client->mailbox.message.info[id].State, sizeof(unsigned long), 1, index);
                    client->mailbox.message.info[id].State |= MSG_STATE_RECENT;
                }

                fclose(index);
                WriteNLockRelease(client->mailbox.lock);

                return(ConnWriteF(client->conn, "1000 %s %lu\r\n", MSG1000DELETED, id + 1));
            }
        }

        ccode = ConnWrite(client->conn, MSG4228CANTWRITEMBOX, sizeof(MSG4228CANTWRITEMBOX) - 1);
    } else {
        ccode = ConnWriteF(client->conn, "1000 %s %lu\r\n", MSG1000DELETED, id + 1);
    }

    return(ccode);
}

int 
NmapCommandDflg(void *param)
{
    int ccode;
    unsigned long id;
    unsigned long newFlags;
    unsigned long flags;
    unsigned char *ptr;
    unsigned char *ptr2;
    struct stat sb;
    FILE *index;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_MBOX) {
        if (!(client->flags & NMAP_READONLY)) {
            ccode = MailboxOOBNotifications(client, &sb);
            if (ccode != -1) {
                ptr = client->buffer + 4;
            } else {
                return(ccode);
            }
        } else {
            return(ConnWrite(client->conn, MSG4243READONLY, sizeof(MSG4243READONLY) - 1));
        }
    } else if (!(client->states & NMAP_CLIENT_USER)) {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    } else {
        return(ConnWrite(client->conn, MSG4225NOMBOX, sizeof(MSG4225NOMBOX) - 1));
    }

    if ((*ptr++ == ' ') && (*ptr != ' ') && ((ptr2 = strchr(ptr, ' ')) != NULL)) {
        *ptr2++ = '\0';
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    id = atol(ptr);
    newFlags = atol(ptr2);

    if (id && (id <= client->mailbox.message.used)) {
        id--;
    } else {
        return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
    }

    if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
        if (client->flags & NMAP_OOBMESSAGES) {
            sprintf(client->path, "%s/%s/%s.idx", client->store, client->user, client->mailbox.name);
            
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 

            flags = client->mailbox.message.info[id].State;
            if ((flags & ~newFlags) == flags) {
                return(ConnWriteF(client->conn, "1000 %lu %s\r\n", flags, MSG1000FLAGSET));
            }
        } else {
            flags = client->mailbox.message.info[id].State;
            if ((flags & ~newFlags) == flags) {
                return(ConnWriteF(client->conn, "1000 %lu %s\r\n", flags, MSG1000FLAGSET));
            }

            sprintf(client->path, "%s/%s/%s.idx", client->store, client->user, client->mailbox.name);
        }
    } else if (!(client->share.flags & NMAP_SHARE_MAILBOX_DENIED)) {
        if (client->flags & NMAP_OOBMESSAGES) {
            sprintf(client->path, "%s/%s/%s.sdx", client->store, client->user, client->mailbox.name);
            
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 

            flags = client->mailbox.message.info[id].State;
            if ((flags & ~newFlags) == flags) {
                return(ConnWriteF(client->conn, "1000 %lu %s\r\n", flags, MSG1000FLAGSET));
            }
        } else {
            flags = client->mailbox.message.info[id].State;
            if ((flags & ~newFlags) == flags) {
                return(ConnWriteF(client->conn, "1000 %lu %s\r\n", flags, MSG1000FLAGSET));
            }

            sprintf(client->path, "%s/%s/%s.sdx", client->store, client->user, client->mailbox.name);
        }

        if ((newFlags & MSG_STATE_READ) && !(client->share.mailbox.permissions & NMAP_SHARE_SEEN) && (flags & MSG_STATE_READ)) {
            newFlags &= ~MSG_STATE_READ;                                
        }
    } else {
        return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
    }

    if (WriteNLockAquire(client->mailbox.lock, 3000) == NLOCK_AQUIRED) {
        index = fopen(client->path, "r+b");
        if (index != NULL) {
            if ((newFlags & MSG_STATE_PURGED) && (flags & MSG_STATE_PURGED)) {
                newFlags &= ~MSG_STATE_PURGED;
            } else if ((newFlags & MSG_STATE_DELETED) && (flags & MSG_STATE_DELETED)) {
                client->mailbox.message.count++;
            }

            fseek(index, NMAP_IDX_HEADERSIZE + (id * sizeof(MessageInfoStruct)) + offsetof(MessageInfoStruct, State), SEEK_SET);

            flags &= ~newFlags;
            if (!(flags & MSG_STATE_RECENT)) {
                fwrite(&flags, sizeof(unsigned long), 1, index);
            } else {
                flags &= ~MSG_STATE_RECENT;
                fwrite(&flags, sizeof(unsigned long), 1, index);
                flags |= MSG_STATE_RECENT;
            }

            fclose(index);
            WriteNLockRelease(client->mailbox.lock);

            if (stat(client->path, &sb) == 0) {
                client->mailbox.indexTime = sb.st_mtime;    
            }

            client->mailbox.message.info[id].State = flags;

            return(ConnWriteF(client->conn, "1000 %lu %s\r\n", flags, MSG1000FLAGSET));
        }

        WriteNLockRelease(client->mailbox.lock);
    }

    return(ConnWrite(client->conn, MSG4228CANTWRITEMBOX, sizeof(MSG4228CANTWRITEMBOX) - 1));
}

int 
NmapCommandFlag(void *param)
{
    unsigned long mask = 0;
    unsigned char *ptr;
    NMAPClient *client = (NMAPClient *)param;

    ptr = client->buffer + 4;
    if ((*ptr++ == ' ') && (*ptr != ' ')) {
        if (client->flags & NMAP_KEEPUSER) {
            mask |= NMAP_KEEPUSER;
        }

        if (client->flags & NMAP_READONLY) {
            mask |= NMAP_READONLY;
        }

        client->flags = atol(ptr) | mask;
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    return(ConnWrite(client->conn, MSG1000PARAMSET, sizeof(MSG1000PARAMSET) - 1));
}

int 
NmapCommandGflg(void *param)
{
    int ccode;
    unsigned long id;
    unsigned char *ptr;
    struct stat sb;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_MBOX) {
        ccode = MailboxOOBNotifications(client, &sb);
        if (ccode != -1) {
            ptr = client->buffer + 4;
        } else {
            return(ccode);
        }
    } else if (!(client->states & NMAP_CLIENT_USER)) {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    } else {
        return(ConnWrite(client->conn, MSG4225NOMBOX, sizeof(MSG4225NOMBOX) - 1));
    }

    if ((*ptr++ == ' ') && (isdigit(*ptr))) {
        id = atol(ptr);
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (id && (id <= client->mailbox.message.used)) {
        id--;
    } else {
        return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
    }

    if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
        if (client->flags & NMAP_OOBMESSAGES) {
            sprintf(client->path, "%s/%s/%s.idx",client->store, client->user, client->mailbox.name);

            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
        }
    } else if (!(client->share.flags & NMAP_SHARE_MAILBOX_DENIED)) {
        if (client->flags & NMAP_OOBMESSAGES) {
            sprintf(client->path, "%s/%s/%s.sdx",client->store, client->user, client->mailbox.name);

            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
        }
    } else {
        return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
    }

    return(ConnWriteF(client->conn, "1000 %lu %s\r\n", client->mailbox.message.info[id].State, MSG1000FLAGINFO));
}

int 
NmapCommandGinfo(void *param)
{
    int ccode;
    int length;
    unsigned long start;
    unsigned long stop;
    unsigned char *request;
    unsigned char *field = NULL;
    unsigned char reply[CONN_BUFSIZE + 1];
    BOOL found = FALSE;
    struct stat sb;
    FILE *scms;
    MessageInfoStruct *msgInfo;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_MBOX) {
        ccode = MailboxOOBNotifications(client, &sb);
        if (ccode != -1) {
            request = client->buffer + 5;
        } else {
            return(ccode);
        }
    } else if (!(client->states & NMAP_CLIENT_USER)) {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    } else {
        return(ConnWrite(client->conn, MSG4225NOMBOX, sizeof(MSG4225NOMBOX) - 1));
    }

    /* GINFO[ <id>] <field> */
    if (*request++ == ' ') {
        if (isdigit(*request)) {
            field = strchr(request + 1, ' ');
            if (field) {
                *field = '\0';

                start = atoi(request);
                stop = start;

                *field++ = ' ';
            }
        } else {
            field = request;

            start = 1;
            stop = client->mailbox.message.used;
        }
    }

    if (field && *field) {
        if (start && (start <= client->mailbox.message.used)) {
            start--;
        } else {
            return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    length = strlen(field);
    if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
        sprintf(client->path, "%s/%s/%s.box", client->store, client->user, client->mailbox.name);
        client->mailbox.fh = fopen(client->path,"rb");
        if (client->mailbox.fh) {
            if (client->flags & NMAP_OOBMESSAGES) {
                sprintf(client->path, "%s/%s/%s.idx",client->store, client->user, client->mailbox.name);
                GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
            }
        } else {
            return(ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1));
        }

        ccode = 0;
        for (msgInfo = &(client->mailbox.message.info[start]); (ccode != -1) && (start < stop); start++, msgInfo++) {
            if (msgInfo->UseSCMS) {
                sprintf(client->path, "%s/%x/%lx", NMAP.path.scms, (unsigned int)(msgInfo->SCMSID % 16), msgInfo->SCMSID);
                scms = fopen(client->path, "rb");
                if (!scms) {
                    continue;
                }
            } else {
                scms = client->mailbox.fh;
            }

            fseek(scms, msgInfo->AuthPos, SEEK_SET);

            found = FALSE;

            while (!feof(scms) && !ferror(scms) && (ccode != -1)) {
                if (fgets(client->line, CONN_BUFSIZE, scms) != NULL) {
                    if ((client->line[0] != CR) || client->line[1] != LF) {
                        if (!found) {
                            if (XplStrNCaseCmp(client->line, field, length) == 0) {
                                ccode = ConnWriteF(client->conn, "2002-%lu %lu %lu %lu %lu %s %lu %lu %lu %s",
                                    start + 1, 
                                    msgInfo->SSize, /* Size */
                                    msgInfo->BodyPos - msgInfo->AuthPos, /* HeadSize */
                                    msgInfo->SSize + msgInfo->AuthPos - msgInfo->BodyPos, /* BodySize */
                                    msgInfo->State,
                                    msgInfo->UIDL,
                                    msgInfo->UID,
                                    client->mailbox.id,
                                    msgInfo->DateReceived, 
                                    client->line);

                                found = TRUE;
                            }
                        } else if (isspace(client->line[0])) {
                            ccode = ConnWriteF(client->conn, "2003-%s", client->line);
                        } else {
                            /* Found the field, and no additional header lines belong to it
                               Don't break yet as the search should continue for multiple lines 
                               with the same field */
                            found = FALSE;
                        }
                    } else {
                        break;
                    }
                }
            }

            if (scms != client->mailbox.fh) {
                fclose(scms);
            }
        }

        fclose(client->mailbox.fh);
        client->mailbox.fh = NULL;
    } else if (!(client->share.flags & NMAP_SHARE_MAILBOX_DENIED)) {
        if (client->flags & NMAP_OOBMESSAGES) {
            sprintf(client->path, "%s/%s/%s.sdx",client->store, client->user, client->mailbox.name);
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
        }

        if (((ccode = NMAPSendCommandF(client->share.mailbox.conn, "GINFO %s\r\n", request)) != -1) 
                && ((ccode = NMAPReadAnswerLine(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                && (ccode == 2002)) {
            do {
                if (ccode == 2002) {
                    if (sscanf(client->line, "2002-%lu %*u %*u %*u %*u %*s %*u %*u %*u %s", &start, reply) == 2) {
                        if (--start < client->mailbox.message.used) {
                            msgInfo = &(client->mailbox.message.info[start]);

                            ccode = ConnWriteF(client->conn, "2002-%lu %lu %lu %lu %lu %s %lu %lu %lu %s", 
                                start + 1,
                                msgInfo->SSize, /* Size */
                                msgInfo->BodyPos - msgInfo->AuthPos, /* HeadSize */
                                msgInfo->SSize + msgInfo->AuthPos - msgInfo->BodyPos, /* BodySize */
                                msgInfo->State, 
                                msgInfo->UIDL, 
                                msgInfo->UID, 
                                client->mailbox.id, 
                                msgInfo->DateReceived, 
                                reply);
                        }
                    }
                } else {
                    ccode = ConnWrite(client->conn, client->line, strlen(client->line));
                }

                if (ccode != -1) {
                    ccode = NMAPReadAnswerLine(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE);
                }
            } while ((ccode != -1) && ((ccode == 2002) || (ccode == 2003)));
        } else if (ccode != 1000) {
            return(ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
    }

    if (ccode != -1) {
        ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
    }

    return(ccode);
}

int 
NmapCommandHead(void *param)
{
    int count;
    int ccode;
    unsigned long id;
    unsigned char *ptr;
    struct stat sb;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_MBOX) {
        ccode = MailboxOOBNotifications(client, &sb);
        if (ccode != -1) {
            ptr = client->buffer + 4;
        } else {
            return(ccode);
        }
    } else if (!(client->states & NMAP_CLIENT_USER)) {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    } else {
        return(ConnWrite(client->conn, MSG4225NOMBOX, sizeof(MSG4225NOMBOX) - 1));
    }

    if ((*ptr++ == ' ') && (isdigit(*ptr))) {
        id = atoi(ptr);
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (id && (id <= client->mailbox.message.used)) {
        id--;
    } else {
        return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
    }

    if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
        if (client->flags & NMAP_OOBMESSAGES) {
            sprintf(client->path, "%s/%s/%s.idx", client->store, client->user, client->mailbox.name);
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
        }

        if (!client->mailbox.message.info[id].UseSCMS) {
            sprintf(client->path, "%s/%s/%s.box", client->store, client->user, client->mailbox.name);
        } else {
            sprintf(client->path, "%s/%x/%lx", NMAP.path.scms, (unsigned int)(client->mailbox.message.info[id].SCMSID % 16), client->mailbox.message.info[id].SCMSID);
        }

        client->mailbox.fh = fopen(client->path,"rb");
        if (client->mailbox.fh != NULL) {
            count = client->mailbox.message.info[id].BodyPos - client->mailbox.message.info[id].AuthPos;
        } else {
            /* fixme - this should be MSG4224CANTREADMBOX instead of a fake out. */
            return(ConnWriteF(client->conn, "2020 40\r\nSubject: <Message not available>\r\n  \r\n\r\n%s", MSG1000OK));
        }

        fseek(client->mailbox.fh, client->mailbox.message.info[id].AuthPos, SEEK_SET);

        ccode = ConnWriteF(client->conn, "2020 %lu\r\n", (long unsigned int)count);
        if (ccode != -1) {
            ccode = ConnWriteFromFile(client->conn, client->mailbox.fh, count);
        }

        fclose(client->mailbox.fh);
        client->mailbox.fh = NULL;

        if (ccode != -1) {
            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
        }
    } else if (!(client->share.flags & NMAP_SHARE_MAILBOX_DENIED)) {
        if (client->flags & NMAP_OOBMESSAGES) {
            sprintf(client->path, "%s/%s/%s.sdx", client->store, client->user, client->mailbox.name);
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
        }

        if (((ccode = NMAPSendCommandF(client->share.mailbox.conn, "HEAD %lu\r\n", id + 1)) != -1) 
                && ((ccode = NMAPReadAnswer(client->share.mailbox.conn, client->line, CONN_BUFSIZE, TRUE)) != -1) 
                && (ccode == 2020)) {
            count = atol(client->line);

            if (((ccode = ConnWriteF(client->conn, "2020 %lu\r\n", (long unsigned int)count)) != -1) 
                    && ((ccode = ConnReadToConn(client->share.mailbox.conn, client->conn, count)) != -1) 
                    && ((ccode = NMAPReadAnswerLine(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                    && (ccode == 1000)) {
                ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
            } else if (ccode != -1) {
                ccode = ConnWrite(client->conn, client->line, strlen(client->line));
            } else {
                ccode = ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1);
            }
        } else if (ccode != -1) {
            ccode = ConnWrite(client->conn, client->line, strlen(client->line));
        }
    } else {
        ccode = ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1);
    }

    return(ccode);
}

int 
NmapCommandInfo(void *param)
{
    int ccode;
    unsigned long id;
    unsigned long clientID;
    unsigned char *ptr;
    struct stat sb;
    MessageInfoStruct *msgInfo;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_MBOX) {
        ccode = MailboxOOBNotifications(client, &sb);
        if (ccode != -1) {
            ptr = client->buffer + 4;
        } else {
            return(ccode);
        }
    } else if (!(client->states & NMAP_CLIENT_USER)) {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    } else {
        return(ConnWrite(client->conn, MSG4225NOMBOX, sizeof(MSG4225NOMBOX) - 1));
    }

    if (*ptr == '\0') {
        id = 0;
    } else if ((*ptr++ == ' ') && (isdigit(*ptr))) {
        id = atol(ptr);
        if (id && (id <= client->mailbox.message.used)) {
            id--;
        } else {
            return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
        if (client->flags & NMAP_OOBMESSAGES) {
            sprintf(client->path, "%s/%s/%s.idx", client->store, client->user, client->mailbox.name);
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
        }
    } else if (!(client->share.flags & NMAP_SHARE_MAILBOX_DENIED)) {
        if (client->flags & NMAP_OOBMESSAGES) {
            sprintf(client->path, "%s/%s/%s.sdx", client->store, client->user, client->mailbox.name);
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
        }
    } else {
        return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
    }

    if (*ptr != '\0') {
        msgInfo = &(client->mailbox.message.info[id]);
        return(ConnWriteF(client->conn, "2001 %lu %lu %lu %lu %lu %s %lu %lu %lu\r\n",
            id + 1, 
            msgInfo->SSize, /* Size        */
            msgInfo->BodyPos - msgInfo->AuthPos, /* HeadSize */
            msgInfo->SSize + msgInfo->AuthPos - msgInfo->BodyPos, /* BodySize */
            msgInfo->State, 
            msgInfo->UIDL, 
            msgInfo->UID, 
            client->mailbox.id, 
            msgInfo->DateReceived));
    }

    clientID = 1;
    id = client->mailbox.message.used;
    msgInfo = client->mailbox.message.info;

    ccode = ConnWriteF(client->conn, "2002 %lu\r\n", client->mailbox.message.used);
    while ((ccode != -1) && (id-- != 0)) {
        ccode = ConnWriteF(client->conn, "%lu %lu %lu %lu %lu %s %lu %lu %lu\r\n",
            clientID++, 
            msgInfo->SSize, /* Size        */
            msgInfo->BodyPos - msgInfo->AuthPos, /* HeadSize */
            msgInfo->SSize + msgInfo->AuthPos - msgInfo->BodyPos, /* BodySize */
            msgInfo->State, 
            msgInfo->UIDL, 
            msgInfo->UID, 
            client->mailbox.id, 
            msgInfo->DateReceived);

        msgInfo++;
    }

    if (ccode != -1) {
        ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
    }

    return(ccode);
}

int 
NmapCommandList(void *param)
{
    int count;
    int ccode;
    unsigned long id;
    unsigned long len;
    unsigned char *ptr;
    struct stat sb;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_MBOX) {
        ccode = MailboxOOBNotifications(client, &sb);
        if (ccode != -1) {
            ptr = client->buffer + 4;
        } else {
            return(ccode);
        }
    } else if (!(client->states & NMAP_CLIENT_USER)) {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    } else {
        return(ConnWrite(client->conn, MSG4225NOMBOX, sizeof(MSG4225NOMBOX) - 1));
    }

    if ((*ptr++ = ' ') && (isdigit(*ptr))) {
        id = atoi(client->buffer + 5);
        if (id && (id <= client->mailbox.message.used)) {
            id--;
        } else {
            return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
        if (client->flags & NMAP_OOBMESSAGES) {
            sprintf(client->path, "%s/%s/%s.idx", client->store, client->user, client->mailbox.name);
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
        }

        /* make sure the mailbox can be opened */
        if (!client->mailbox.message.info[id].UseSCMS) {
            sprintf(client->path, "%s/%s/%s.box", client->store, client->user, client->mailbox.name);
        } else {
            sprintf(client->path, "%s/%x/%lx", NMAP.path.scms, (unsigned int)(client->mailbox.message.info[id].SCMSID % 16), client->mailbox.message.info[id].SCMSID);
        }

        client->mailbox.fh = fopen(client->path, "rb");
        if (client->mailbox.fh != NULL) {
            if (!(client->mailbox.message.info[id].State & MSG_STATE_ESCAPE)) {
                ccode = ConnWriteF(client->conn, "2023 %lu %lu\r\n", 
                        client->mailbox.message.info[id].BodyPos - client->mailbox.message.info[id].AuthPos, 
                        client->mailbox.message.info[id].SSize + client->mailbox.message.info[id].AuthPos - client->mailbox.message.info[id].BodyPos);
            } else {
                ccode = ConnWriteF(client->conn, "2024 %lu %lu\r\n", 
                        client->mailbox.message.info[id].BodyPos - client->mailbox.message.info[id].AuthPos, 
                        client->mailbox.message.info[id].SSize + client->mailbox.message.info[id].AuthPos - client->mailbox.message.info[id].BodyPos);
            }
        } else {
            return(ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1));
        }

        if (ccode != -1) {
            fseek(client->mailbox.fh, client->mailbox.message.info[id].AuthPos, SEEK_SET);

            count = client->mailbox.message.info[id].SSize;

            ccode = ConnWriteFromFile(client->conn, client->mailbox.fh, count);
        }

        fclose(client->mailbox.fh);
        client->mailbox.fh = NULL;

        if (ccode != -1) {
            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
        }
    } else if (!(client->share.flags & NMAP_SHARE_MAILBOX_DENIED)) {
        if (client->flags & NMAP_OOBMESSAGES) {
            sprintf(client->path, "%s/%s/%s.sdx", client->store, client->user, client->mailbox.name);
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
        }

        if (((ccode = NMAPSendCommandF(client->share.mailbox.conn, "LIST %lu\r\n", id + 1)) != -1) 
                && ((ccode = NMAPReadAnswer(client->share.mailbox.conn, client->line, CONN_BUFSIZE, TRUE)) != -1) 
                && ((ccode == 2023) || (ccode == 2024))) {
            ptr = strchr(client->line + 1, ' ');
            if (ptr != NULL) {
                *ptr = '\0';

                len = atol(client->line + 5);
                count = atol(++ptr);

                if (((ccode = ConnWriteF(client->conn, "%lu %lu %lu\r\n", (long unsigned int)ccode, len, (long unsigned int)count)) != -1) 
                        && ((ccode = ConnReadToConn(client->share.mailbox.conn, client->conn, count + len)) != -1) 
                        && ((ccode = NMAPReadAnswer(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                        && (ccode == 1000)) {
                    ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
                } else if (ccode != -1) {
                    ccode = ConnWrite(client->conn, client->line, strlen(client->line));
                } else {
                    ccode = ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1);
                }
            } else {
                ccode = ConnWrite(client->conn, MSG5004INTERNALERR, sizeof(MSG5004INTERNALERR) - 1);
            }
        } else if (ccode != -1) {
            ccode = ConnWrite(client->conn, client->line, strlen(client->line));
        } else {
            ccode = ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1);
        }
    } else {
        ccode = ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1);
    }

    return(ccode);
}

int 
NmapCommandMbox(void *param)
{
    int ccode;
    unsigned char *ptr;
    unsigned char *ptr2;
    unsigned char *resourceOwner;
    unsigned char *resourceName;
    BOOL result;
    struct stat sb;
    FILE *box;
    NMAPClient *client = (NMAPClient *)param;

    
    if (client->states & NMAP_CLIENT_USER) {
        ptr = client->buffer + 4;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if (!(client->states & NMAP_CLIENT_MBOX)) {
        if ((*ptr++ == ' ') 
                && !(ptr2 = strchr(ptr, ' ')) 
                && ((strchr(ptr, '.') == NULL) || ((strstr(ptr, "../") == NULL) && (strstr(ptr, "..\\") == NULL)))) {
            if (ReadNLockAquire(&client->mailbox.lock, &client->userHash, client->user, ptr, 0) == NLOCK_AQUIRED) {
                sprintf(client->path,"%s/%s/%s.box", client->store, client->user, ptr);
                box = fopen(client->path, "rb");
            } else {
                return(ConnWrite(client->conn,MSG4221MBOXLOCKED, sizeof(MSG4221MBOXLOCKED) - 1));
            }
        } else {
            return(ConnWrite(client->conn, MSG4224NOMBOX, sizeof(MSG4224NOMBOX) - 1));
        }
    } else {
        ccode = MailboxOOBNotifications(client, &sb);
        if (ccode != -1) {
            return(ConnWrite(client->conn, MSG4221MBOXDONE, sizeof(MSG4221MBOXDONE) - 1));
        }

        return(ccode);
    }

    if (box) {
        fclose(box);
        strcpy(client->mailbox.name, ptr);

        if (ParseMaildrop(client)) {
            client->states |= NMAP_CLIENT_MBOX;
            if (!(client->flags & NMAP_READONLY)) {
                ccode = ConnWriteF(client->conn, "1000 %lu %s\r\n", client->mailbox.id, MSG1000MBOX);
            } else {
                ccode = ConnWriteF(client->conn, "1020 %lu %s\r\n", client->mailbox.id, MSG1020MBOX);
            }

            return(ccode);
        }

        ccode = ConnWrite(client->conn,MSG4221MBOXLOCKED, sizeof(MSG4221MBOXLOCKED) - 1);
    } else if (XplStrCaseCmp(ptr, "INBOX") == 0) {
        sprintf(client->path, "%s/%s/INBOX.box", client->store, client->user);
        box = fopen(client->path, "rb");
        if (box) {
            fclose(box);
            strcpy(client->mailbox.name, "INBOX");

            if (ParseMaildrop(client)) {
                client->states |= NMAP_CLIENT_MBOX;
                if (!(client->flags & NMAP_READONLY)) {
                    ccode = ConnWriteF(client->conn, "1000 %lu %s\r\n", client->mailbox.id, MSG1000MBOX);
                } else {
                    ccode = ConnWriteF(client->conn, "1020 %lu %s\r\n", client->mailbox.id, MSG1020MBOX);
                }

                return(ccode);
            }
        }

        ccode = ConnWrite(client->conn,MSG4221MBOXLOCKED, sizeof(MSG4221MBOXLOCKED) - 1);
    } else if (IsAvailableShare(client->user, ptr, NMAP_SHARE_MAILBOX, &resourceOwner, &resourceName)) {
        ccode = ConnectToNMAPShare(client, resourceOwner, resourceName, NMAP_SHARE_MAILBOX, &(client->share.mailbox.permissions));
        if ((ccode == 1000) && (client->share.mailbox.permissions & NMAP_SHARE_READ)) {
            strcpy(client->mailbox.name, ptr);

#if 0
            /*    To determine if a shared mailbox is read-only or read-write, we need to 
                examine the permissions.

                NMAP_SHARE_READ        MBOX, INFO, SEARCH, GFLG ...
                NMAP_SHARE_WRITE        SFLAG and AFLG for all flags other than MSG_STATE_DELETED and MSG_STATE_PURGED
                NMAP_SHARE_INSERT        STOR and COPY
            */
            if (!(client->share.mailbox.permissions & NMAP_SHARE_INSERT)) {
                client->flags |= NMAP_READONLY;
            }
#endif

            result = SelectNMAPShare(client, resourceOwner, resourceName, NMAP_SHARE_MAILBOX);

            MemFree(resourceOwner);
            MemFree(resourceName);

            if (result) {
                client->states |= NMAP_CLIENT_MBOX;
                client->share.flags |= NMAP_SHARE_MAILBOX;

                if (!(client->flags & NMAP_READONLY)) {
                    ccode = ConnWriteF(client->conn, "1000 %lu %s\r\n", client->mailbox.id, MSG1000MBOX);
                } else {
                    ccode = ConnWriteF(client->conn, "1020 %lu %s\r\n", client->mailbox.id, MSG1020MBOX);
                }

                return(ccode);
            } else {
                client->mailbox.name[0] = '\0';

                NMAPQuit(client->share.mailbox.conn);
                ConnFree(client->share.mailbox.conn);
                client->share.mailbox.conn = NULL;

                if (client->mailbox.message.info != NULL) {
                    MemFree(client->mailbox.message.info);
                    client->mailbox.message.info = NULL;
                }

                client->mailbox.message.allocated = 0;
                client->mailbox.message.count = 0;
                client->mailbox.message.used = 0;

                ccode = ConnWrite(client->conn, MSG4221MBOXLOCKED, sizeof(MSG4221MBOXLOCKED) - 1);
            }
        } else if (ccode == 1000) {
            NMAPQuit(client->share.mailbox.conn);
            ConnFree(client->share.mailbox.conn);
            client->share.mailbox.conn = NULL;

            client->share.mailbox.permissions = 0;
            client->share.flags &= ~(NMAP_SHARE_MAILBOX | NMAP_SHARE_MAILBOX_DENIED);

            ccode = ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1);
        } else {
            ccode = ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1);
        }
    } else {
        ccode = ConnWrite(client->conn, MSG4224NOMBOX, sizeof(MSG4224NOMBOX) - 1);
    }

    ReadNLockRelease(client->mailbox.lock);
    client->mailbox.lock = NULL;

    return(ccode);
}

int 
NmapCommandMime(void *param)
{
    int ccode;
    unsigned long id;
    unsigned char *ptr;
    struct stat sb;
    NMAPClient *client = (NMAPClient *)param;
    MIMEReportStruct *report;

    if (client->states & NMAP_CLIENT_MBOX) {
        ccode = MailboxOOBNotifications(client, &sb);
        if (ccode != -1) {
            ptr = client->buffer + 4;
        } else {
            return(ccode);
        }
    } else if (!(client->states & NMAP_CLIENT_USER)) {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    } else {
        return(ConnWrite(client->conn, MSG4225NOMBOX, sizeof(MSG4225NOMBOX) - 1));
    }

    if ((*ptr++ == ' ') && (isdigit(*ptr))) {
        id = atoi(ptr);
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (id && (id <= client->mailbox.message.used)) {
        id--;
    } else {
        return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
    }

    if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
        if (client->flags & NMAP_OOBMESSAGES) {
            sprintf(client->path, "%s/%s/%s.idx", client->store, client->user, client->mailbox.name);
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
        }

        /* 2002 <Type> <Subtype> <Charset> <Encoding> <Name> <Part-start-pos> <Size> <Lines> */
        if (!client->mailbox.message.info[id].UseSCMS) {
            sprintf(client->path, "%s/%s/%s.box",client->store, client->user,client->mailbox.name);
        } else {
            sprintf(client->path, "%s/%x/%lx",NMAP.path.scms, (unsigned int)(client->mailbox.message.info[id].SCMSID % 16), client->mailbox.message.info[id].SCMSID);
        }
        client->mailbox.fh = fopen(client->path,"rb");
        if (client->mailbox.fh) {
            fseek(client->mailbox.fh, client->mailbox.message.info[id].AuthPos, SEEK_SET);

            report = ParseMIME(client->mailbox.fh, client->mailbox.message.info[id].BodyPos - client->mailbox.message.info[id].AuthPos, client->mailbox.message.info[id].SSize, client->mailbox.message.info[id].BodyLines, client->line, CONN_BUFSIZE);
            if (report) {
                ccode = SendMIME(client, report);

                FreeMIME(report);
            } else {
                /* fixme - make a Parsing error message */
                ccode = ConnWrite(client->conn, MSG5230NOMEMORYERR, sizeof(MSG5230NOMEMORYERR) - 1);
            }

            fclose(client->mailbox.fh);
            client->mailbox.fh=NULL;
        } else {
            /* fixme - make a Parsing error message */
            if ((ccode = ConnWriteF(client->conn, "2002-text plain US-ASCII - \"\" 0  %lu 0 %lu\r\n", client->mailbox.message.info[id].SSize+client->mailbox.message.info[id].AuthPos-client->mailbox.message.info[id].BodyPos, ((client->mailbox.message.info[id].SSize+client->mailbox.message.info[id].AuthPos-client->mailbox.message.info[id].BodyPos) / 70) + 1)) != -1) {
                ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
            }
        }
    } else if (!(client->share.flags & NMAP_SHARE_MAILBOX_DENIED)) {
        if (client->flags & NMAP_OOBMESSAGES) {
            sprintf(client->path, "%s/%s/%s.sdx", client->store, client->user, client->mailbox.name);
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
        }

        if (((ccode = NMAPSendCommandF(client->share.mailbox.conn, "MIME %lu\r\n", id + 1)) != -1) 
                && ((ccode = NMAPReadAnswerLine(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE)) != -1)) {
            while ((ccode != -1) && (ccode != 1000)) {
                ccode = ConnWrite(client->conn, client->line, strlen(client->line));
                if ((ccode != -1) && (ccode >= 2002) && (ccode <= 2003)) {
                    ccode = NMAPReadAnswerLine(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE);
                    continue;
                }

                break;
            }

            if (ccode == 1000) {
                ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
            }
        } else {
            ccode = ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1);
        }
    } else {
        ccode = ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1);
    }

    return(ccode);
}

int 
NmapCommandPurg(void *param)
{
    int ccode;
    unsigned long delCount = 0;
    unsigned long startID;
    unsigned long stopID;
    unsigned long state;
    unsigned char *ptr;
    BOOL verbose = FALSE;
    BOOL markedOnly = FALSE;
    struct stat sb;
    FILE *index;
    NMAPClient *client = (NMAPClient *)param;
    MessageInfoStruct *msgInfo;

    if (client->states & NMAP_CLIENT_MBOX) {
        ccode = MailboxOOBNotifications(client, &sb);
        if (ccode != -1) {
            ptr = client->buffer + 4;
        } else {
            return(ccode);
        }
    } else if (!(client->states & NMAP_CLIENT_USER)) {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    } else {
        return(ConnWrite(client->conn, MSG4225NOMBOX, sizeof(MSG4225NOMBOX) - 1));
    }

    if (client->flags & NMAP_READONLY) {
        return(ConnWrite(client->conn, MSG4243READONLY, sizeof(MSG4243READONLY) - 1));
    }

    if (toupper(ptr[-1]) == 'V') {
        verbose = TRUE;
    } else if (toupper(ptr[-1] == 'M')) {
        markedOnly = TRUE;
    }

    if (!markedOnly) {
        if (*ptr == '\0') {
            startID = 0;
            stopID = client->mailbox.message.used;
        } else if ((*ptr++ == ' ') && (isdigit(*ptr))) {
            stopID = startID = atoi(ptr);
            if (startID && (startID <= client->mailbox.message.used)) {
                startID--;
            } else {
                return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
            }
        } else {
            return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
        }

        if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
            sprintf(client->path, "%s/%s/%s.idx", client->store, client->user, client->mailbox.name);
        } else if (!(client->share.flags & NMAP_SHARE_MAILBOX_DENIED) && (client->share.mailbox.permissions & NMAP_SHARE_DELETE)) {
            sprintf(client->path, "%s/%s/%s.sdx", client->store, client->user, client->mailbox.name);
        } else {
            return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
        }

        if (client->flags & NMAP_OOBMESSAGES) {
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
        }

        if (WriteNLockAquire(client->mailbox.lock, 3000) == NLOCK_AQUIRED) {
            index = fopen(client->path, "r+b");
            if (index) {
                for (ccode = 0, msgInfo = &(client->mailbox.message.info[startID]); (ccode != -1) && (startID < stopID); startID++, msgInfo++) {
                    state = msgInfo->State;

                    if (state & (MSG_STATE_DELETED | MSG_STATE_PURGED)) {
                        if (!(state & MSG_STATE_PURGED)) {
                            state |= MSG_STATE_PURGED;
                            state &= ~MSG_STATE_DELETED;

                            fseek(index, NMAP_IDX_HEADERSIZE + (startID * sizeof(MessageInfoStruct)) + offsetof(MessageInfoStruct, State), SEEK_SET);

                            if (!(state & MSG_STATE_RECENT)) {
                                fwrite(&state, sizeof(unsigned long), 1, index);
                            } else {
                                state &= ~MSG_STATE_RECENT;
                                fwrite(&state, sizeof(unsigned long), 1, index);
                                state |= MSG_STATE_RECENT;
                            }

                            msgInfo->State = state;

                            if (verbose) {
                                ccode = ConnWriteF(client->conn, "2002-%lu marked purged\r\n", startID + 1);
                            }
                        }

                        delCount++;
                    }
                }

                fclose(index);
            } else {
                WriteNLockRelease(client->mailbox.lock);
                return(ConnWrite(client->conn, MSG4228CANTWRITEMBOX, sizeof(MSG4228CANTWRITEMBOX) - 1));
            }

            WriteNLockRelease(client->mailbox.lock);
        } else {
            return(ConnWrite(client->conn, MSG4228CANTWRITEMBOX, sizeof(MSG4228CANTWRITEMBOX) - 1));
        }
    } else {
        for (startID = 0, msgInfo = client->mailbox.message.info; startID < client->mailbox.message.used; startID++, msgInfo++) {
            if (msgInfo->State & MSG_STATE_PURGED) {
                delCount++;
            }
        }
    }

    if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
        if (delCount > 0) {
            client->mailbox.changed = TRUE;
        }

        StoreMaildrop(client);

        client->mailbox.message.used = 0;
        client->mailbox.message.count = 0;
        client->mailbox.newCount = 0;
        client->mailbox.newline = TRUE;
        client->mailbox.size = 0;
        client->mailbox.newSize = 0;
        client->mailbox.id = 0;
        client->mailbox.nextUID = 0;

        ParseMaildrop(client);
    }

    return(ConnWriteF(client->conn, "1000 %lu %s\r\n", delCount, MSG1000PURGED));
}

int 
NmapCommandRnam(void *param)
{
    unsigned char *oldName;
    unsigned char *newName;
    unsigned char path[XPL_MAX_PATH + 1];
    FILE *index;
    NMAPClient *client = (NMAPClient *)param;
    MDBValueStruct *uid;

    if (client->states & NMAP_CLIENT_USER) {
        oldName = client->buffer + 4;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    } 

    /* RNAM <old mailbox name> <new mailbox name> */
    if ((*oldName++ == ' ') 
            && (*oldName) 
            && (!isspace(*oldName)) 
            && ((newName = strchr(oldName, ' ')) != NULL) 
            && (newName[1]) 
            && (!isspace(newName[1]))) {
        *newName++ = '\0';
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if ((strchr(oldName, '.')) && ((strstr(oldName, "../") != NULL) || (strstr(oldName, "..\\") != NULL))) {
        return(ConnWrite(client->conn, MSG4224NOMBOX, sizeof(MSG4224NOMBOX) - 1));
    }

    if ((strchr(newName, '.')) && ((strstr(newName, "../") != NULL) || (strstr(newName, "..\\") != NULL))) {
        return(ConnWrite(client->conn, MSG4224NOMBOX, sizeof(MSG4224NOMBOX) - 1));
    }

    if (XplStrCaseCmp(client->mailbox.name, oldName) == 0) {
        return(ConnWrite(client->conn, MSG4227TGTMBOXSELECTED, sizeof(MSG4227TGTMBOXSELECTED) - 1));
    }

    if (((strlen(client->store) + strlen(client->user) + strlen(oldName) + 4) > XPL_MAX_PATH) 
            || ((strlen(client->store) + strlen(client->user) + strlen(newName) + 4) > XPL_MAX_PATH)) {
        return(ConnWrite(client->conn, MSG4230PATHTOOLONG, sizeof(MSG4230PATHTOOLONG) - 1));
    }

    sprintf(client->path, "%s/%s/%s.box", client->store, client->user, oldName);
    if (access(client->path, 0) == 0) {
        sprintf(path, "%s/%s/%s.idx", client->store, client->user, oldName);
        index = fopen(path, "rb");
        if (index != NULL) {
            fseek(index, 9, SEEK_SET);
            fgets(client->line, CONN_BUFSIZE, index);
            fclose(index);

            if (atol(client->line) & RESOURCE_FLAG_SHARED) {
                return(ConnWrite(client->conn, MSG4221MBOXSHARED, sizeof(MSG4221MBOXSHARED) - 1));
            }
        }

        sprintf(path, "%s/%s/%s.box", client->store, client->user, newName);
    } else if (IsAvailableShare(client->user, oldName, NMAP_SHARE_MAILBOX, NULL, NULL)) {
        /* fixme - enhancment (support renaming shared resources)?
        RenameAvailableShare(client, NMAP_SHARE_MAILBOX, oldName, newName); */

        return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
    } else {
        return(ConnWrite(client->conn, MSG4224NOMBOX, sizeof(MSG4224NOMBOX) - 1));
    }

    if (access(path, 0) == 0) {
        return(ConnWrite(client->conn, MSG4226MBOXEXISTS, sizeof(MSG4226MBOXEXISTS) - 1));
    }

    /* fixme - why are we not locking the mailbox to be renamed? */
    if (rename(client->path, path)) {
        return(ConnWrite(client->conn, MSG4224NOMBOX, sizeof(MSG4224NOMBOX) - 1));
    }

    SetLongPathName(path);

    sprintf(client->path, "%s/%s/%s.idx",client->store, client->user, oldName);
    sprintf(path, "%s/%s/%s.idx",client->store,client->user, newName);
    rename(client->path, path);

    SetLongPathName(path);

    if (XplStrCaseCmp(oldName, "INBOX") == 0) {
        sprintf(client->path, "%s/%s/INBOX.box", client->store, client->user);
        fclose(fopen(client->path, "wb"));

        sprintf(client->path, "%s/%s/INBOX.idx",client->store, client->user);
        index = fopen(client->path, "wb");
        fprintf(index, "%s\r\n0000000000\r\n0000000000\r\n0000000000\r\n%010lu\r\n%010lu\r\n", NMAP_IDX_VERSION, NMAP.universalCounter++, NMAP.universalCounter++);

        sprintf(client->line, "%lu", NMAP.universalCounter);
        uid = MDBCreateValueStruct(NMAP.handle.directory, NMAP.server.dn);
        MDBAddValue(client->line, uid);
        MDBWrite(MSGSRV_AGENT_NMAP, MSGSRV_A_UID, uid);
        MDBDestroyValueStruct(uid);
        fclose(index);
    } else {
        sprintf(client->path, "%s/%s/%s", client->store, client->user, oldName);

        if (access(client->path, 0) == 0) {
            sprintf(path, "%s/%s/%s", client->store,client->user, newName);
            rename(client->path, path);
            SetLongPathName(path);
        }
    }

    return(ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1));
}

int 
NmapCommandRights(void *param)
{
    int count;
    int ccode;
    unsigned long used;
    unsigned char *mailbox;
    unsigned char *name;
    unsigned char *owner;
    unsigned char dn[MDB_MAX_OBJECT_CHARS + 1];
    BOOL result;
    struct stat sb;
    NMAPClient *client = (NMAPClient *)param;
    MDBValueStruct *vs;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->states & NMAP_CLIENT_MBOX) {
            ccode = MailboxOOBNotifications(client, &sb);
            if (ccode == -1) {
                return(ccode);
            }
        }

        mailbox = client->buffer + 6;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    } 

    /* RIGHTS <mailbox> */
    if ((*mailbox++ == ' ') && (!isspace(*mailbox))) {
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (strchr(mailbox, '.')) {
        if ((strstr(mailbox, "../") != NULL) || (strstr(mailbox, "..\\") != NULL)) {
            return(ConnWrite(client->conn, MSG4224NOMBOX, sizeof(MSG4224NOMBOX) - 1));
        }
    }

    sprintf(client->path, "%s/%s/%s.box", client->store, client->user, mailbox);
    if (access(client->path, 0) == 0) {
        if ((ccode = ConnWrite(client->conn, "2002-Coming up\r\n", 16)) != -1) {
            /* Send the owner's permissions. */
            ccode = ConnWriteF(client->conn, "2002-%s %010lu\r\n", client->user, 
                   (NMAP_SHARE_VISIBLE | NMAP_SHARE_SEEN | NMAP_SHARE_READ | NMAP_SHARE_WRITE | NMAP_SHARE_INSERT | NMAP_SHARE_CREATE | NMAP_SHARE_DELETE | NMAP_SHARE_ADMINISTER | NMAP_SHARE_POST));

            if (ccode != -1) {
                result = MsgFindObject(client->user, client->dn, NULL, NULL, NULL);
            } else {
                result = FALSE;
            }

            if (result) {
                vs = MDBCreateValueStruct(NMAP.handle.directory, NULL);
                if (vs) {
                    MDBRead(client->dn, MSGSRV_A_OWNED_SHARES, vs);

                    count = sprintf(client->line, "MB%s\r", mailbox);
                    for (used = 0; (ccode != -1) && (used < vs->Used); used++) {
                        if (XplStrNCaseCmp(vs->Value[used], client->line, count) != 0) {
                            continue;
                        }

                        /* Replace the delimter between grantee name and grantee permissions. */
                        vs->Value[used][strlen(vs->Value[used]) - 11] = ' ';

                        ccode = ConnWriteF(client->conn, "2002-%s\r\n", vs->Value[used] + count);
                    }

                    MDBDestroyValueStruct(vs);
                    vs = NULL;
                }
            }

            if (ccode != -1) {
                ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
            }
        }

        return(ccode);
    }
    
    if (IsAvailableShare(client->user, mailbox, NMAP_SHARE_MAILBOX, &owner, &name)) {
        if ((ccode = ConnWrite(client->conn, "2002-Coming up\r\n", 16)) != -1) {
            ccode = ConnWriteF(client->conn, "2002-%s %010lu\r\n", client->user, 
                   (NMAP_SHARE_VISIBLE | NMAP_SHARE_SEEN | NMAP_SHARE_READ | NMAP_SHARE_WRITE | NMAP_SHARE_INSERT | NMAP_SHARE_CREATE | NMAP_SHARE_DELETE | NMAP_SHARE_ADMINISTER | NMAP_SHARE_POST));

            if (ccode != -1) {
                result = MsgFindObject(owner, dn, NULL, NULL, NULL);
                if (!result) {
                    ccode = ConnWriteF(client->conn, "2002-%s 0\r\n", client->user);
                }
            } else {
                result = FALSE;
            }

            if (result) {
                vs = MDBCreateValueStruct(NMAP.handle.directory, NULL);
                if (vs) {
                    MDBRead(dn, MSGSRV_A_OWNED_SHARES, vs);

                    count = sprintf(client->line, "MB%s\r%s\r", name, client->user);
                    for (used = 0; used < vs->Used; used++) {
                        if (XplStrNCaseCmp(vs->Value[used], client->line, count) != 0) {
                            continue;
                        }

                        break;
                    }

                    if (used < vs->Used) {
                        if (atol(&(vs->Value[used][strlen(vs->Value[used]) - 10])) & NMAP_SHARE_ADMINISTER) {
                            /* Send all grantee permissions. */
                            count = strlen(name) + 3;
                            for (used = 0; (ccode != -1) && (used < vs->Used); used++) {
                                /* Replace the delimter between grantee name and grantee permissions. */
                                vs->Value[used][strlen(vs->Value[used]) - 11] = ' ';

                                ccode = ConnWriteF(client->conn, "2002-%s\r\n", vs->Value[used] + count);
                            }
                        } else {
                            /*    Send only this grantee's permissions.    */
                            vs->Value[used][count - 1] = ' ';

                            ccode = ConnWriteF(client->conn, "2002-%s\r\n", vs->Value[used] + strlen(name) + 3);
                        }
                    } else {
                        /* This grantee has no permissions!

                           Since we just returned from IsAvailableShare successfully, this shouldn't happen.

                           However, a small window does exist for the permissions to be pulled, so we
                           cover it. */
                        ccode = ConnWriteF(client->conn, "2002-%s 0\r\n", client->user);
                    }

                    MDBDestroyValueStruct(vs);
                    vs = NULL;
                }
            }

            if (ccode != -1) {
                ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
            }
        }

        MemFree(name);
        MemFree(owner);

        if (ccode != -1) {
            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
        }
    } else {
        ccode = ConnWrite(client->conn, MSG4224NOMBOX, sizeof(MSG4224NOMBOX) - 1);
    }

    return(ccode);
}

int 
NmapCommandRmov(void *param)
{
    int ccode;
    unsigned long id;
    unsigned char *ptr;
    unsigned char scratch[(XPL_MAX_PATH * 2) + 1];
    struct stat sb;
    FILE *handle;
    MessageInfoStruct *msgInfo;
    NMAPClient *client = (NMAPClient *)param;
    NMAPClient *sclient;
    MDBValueStruct *uid;

    /* validate state */
    if (client->states & NMAP_CLIENT_USER) {
        if (client->states & NMAP_CLIENT_MBOX) {
            ccode = MailboxOOBNotifications(client, &sb);
            if (ccode == -1) {
                return(ccode);
            }
        }

        ptr = client->buffer + 4;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    } 

    /* RMOV <mailbox> */
    if ((*ptr++ == ' ') && (!isspace(*ptr))) {
        if (XplStrCaseCmp(client->mailbox.name, ptr) == 0) {
            return(ConnWrite(client->conn, MSG4227TGTMBOXSELECTED, sizeof(MSG4227TGTMBOXSELECTED) - 1));
        }

        if (strchr(ptr, '.')) {
            if ((strstr(ptr, "../") != NULL) || (strstr(ptr, "..\\") != NULL)) {
                return(ConnWrite(client->conn, MSG4224NOMBOX, sizeof(MSG4224NOMBOX) - 1));
            }
        }

        if ((strlen(client->store) + strlen(client->user) + strlen(ptr) + 4) >= XPL_MAX_PATH) {
            return(ConnWrite(client->conn, MSG4230PATHTOOLONG, sizeof(MSG4230PATHTOOLONG) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    sprintf(client->path, "%s/%s/%s.box", client->store, client->user, ptr);
    if (access(client->path, 0) == 0) {
        sclient = NMAPClientAlloc();
        if (sclient != NULL) {
            strcpy(sclient->user, client->user);
            sclient->userHash = client->userHash; 
            strcpy(sclient->mailbox.name, ptr);

            sclient->store = client->store;
            sclient->mailbox.newline = TRUE;
        } else {
            return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
        }

        /* We open the mailbox, and then delete everything in it; this takes care of SCMS */
        if (ReadNLockAquire(&sclient->mailbox.lock, &sclient->userHash, sclient->user, sclient->mailbox.name, 0) == NLOCK_AQUIRED) {
            ParseMaildrop(sclient);
        } else {
            NMAPClientFree(sclient);

            return(ConnWrite(client->conn, MSG4221MBOXLOCKED, sizeof(MSG4221MBOXLOCKED) - 1));
        }

        if (!(sclient->mailbox.flags & RESOURCE_FLAG_SHARED)) {
            id = sclient->mailbox.message.used;
            msgInfo = sclient->mailbox.message.info;

            while (id-- != 0) {
                msgInfo->State |= MSG_STATE_DELETED | MSG_STATE_PURGED;
                msgInfo++;
            }

            sclient->mailbox.changed = TRUE;
            StoreMaildrop(sclient);

            if (sclient->mailbox.message.info) {
                MemFree(sclient->mailbox.message.info);
                sclient->mailbox.message.info = NULL;
            }

            if (PurgeNLockAquire(sclient->mailbox.lock, 0) == NLOCK_AQUIRED) {
                if (unlink(client->path) == 0) {
                    sprintf(client->path, "%s/%s/%s.idx", client->store, client->user, ptr);
                    unlink(client->path);

                    CollapseSubTree(client->store, client->user, ptr, scratch, sizeof(scratch) - 1);

                    if (XplStrCaseCmp(ptr, "INBOX") == 0) {
                        sprintf(client->path, "%s/%s/INBOX.box", client->store, client->user);
                        fclose(fopen(client->path, "wb"));

                        sprintf(client->path, "%s/%s/INBOX.idx",client->store, client->user);
                        handle = fopen(client->path, "wb");
                        fprintf(handle, "%s\r\n0000000000\r\n0000000000\r\n0000000000\r\n%010lu\r\n%010lu\r\n", NMAP_IDX_VERSION, NMAP.universalCounter++, NMAP.universalCounter++);

                        sprintf(client->line, "%lu", NMAP.universalCounter);
                        uid = MDBCreateValueStruct(NMAP.handle.directory, NMAP.server.dn);
                        MDBAddValue(client->line, uid);
                        MDBWrite(MSGSRV_AGENT_NMAP, MSGSRV_A_UID, uid);

                        MDBDestroyValueStruct(uid);

                        fclose(handle);
                    }

                    ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
                } else {
                    ccode = ConnWrite(client->conn, MSG4224NOMBOX, sizeof(MSG4224NOMBOX) - 1);
                }

                PurgeNLockRelease(sclient->mailbox.lock);
            } else {
                ccode = ConnWrite(client->conn, MSG4221MBOXLOCKED, sizeof(MSG4221MBOXLOCKED) - 1);
            }
        } else {
            ccode = ConnWrite(client->conn, MSG4221MBOXSHARED, sizeof(MSG4221MBOXSHARED) - 1);
        }

        ReadNLockRelease(sclient->mailbox.lock);
        sclient->mailbox.lock = NULL;

        if (sclient->mailbox.message.info) {
            MemFree(sclient->mailbox.message.info);
            sclient->mailbox.message.info = NULL;
        }

        NMAPClientFree(sclient);

        return(ccode);
    }

    if (IsAvailableShare(client->user, ptr, NMAP_SHARE_MAILBOX, NULL, NULL)) {
        if (RemoveAvailableShare(client, NMAP_SHARE_MAILBOX, ptr)) {
            sprintf(client->path, "%s/%s/%s.sdx", client->store, client->user, ptr);
            unlink(client->path);

            CollapseSubTree(client->store, client->user, ptr, scratch, sizeof(scratch) - 1);

            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
        } else {
            ccode = ConnWrite(client->conn, MSG4221MBOXLOCKED, sizeof(MSG4221MBOXLOCKED) - 1);
        }

        return(ccode);
    }

    /* May be a dangling hierarchical element. */
    client->path[strlen(client->path) - 4] = '\0';
    if (access(client->path, 0) == 0) {
        if (rmdir(client->path) == 0) {
            CollapseSubTree(client->store, client->user, ptr, scratch, sizeof(scratch) - 1);

            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
        } else {
            ccode = ConnWrite(client->conn, MSG4221MBOXLOCKED, sizeof(MSG4221MBOXLOCKED) - 1);
        }
    } else {
        ccode = ConnWrite(client->conn, MSG4224NOMBOX, sizeof(MSG4224NOMBOX) - 1);
    }

    return(ccode);
}

int 
NmapCommandRepair(void *param)
{
    NMAPClient *client = (NMAPClient *)param;

    return(ConnWrite(client->conn, MSG1000NOTIMP, sizeof(MSG1000NOTIMP) - 1));
}

int 
NmapCommandResource(void *param)
{
    int ccode;
    unsigned long used;
    unsigned char *ptr;
    unsigned char *reply;
    unsigned char *name;
    unsigned char scratch[(XPL_MAX_PATH * 2) + 1];
    struct stat sb;
    FILE *index;
    NMAPClient *client = (NMAPClient *)param;
    NMAPClient *sclient;
    MDBValueStruct *vs;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->states & NMAP_CLIENT_MBOX) {
            ccode = MailboxOOBNotifications(client, &sb);
            if (ccode == -1) {
                return(ccode);
            }
        }

        ptr = client->buffer + 8;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    /* RESOURCE */
    if (*ptr == '\0') {
        sclient = NMAPClientAlloc();
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (sclient != NULL) {
        strcpy(sclient->user, client->user);
        sclient->userHash = client->userHash; 
        sclient->store = client->store;
        sclient->mailbox.newline = TRUE;
    } else {
        return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
    }

    if (MsgFindObject(sclient->user, sclient->dn, NULL, NULL, NULL)) {
        ccode = ConnWrite(client->conn, "2002-Coming up\r\n", 16);
    } else {
        NMAPClientFree(sclient);

        return(ConnWriteF(client->conn, MSG5244USERLOOKUPFAILURE, client->user));
    }

    if (ccode != -1) {
        vs = MDBCreateValueStruct(NMAP.handle.directory, NULL);
        if (vs) {
            MDBRead(sclient->dn, MSGSRV_A_AVAILABLE_SHARES, vs);
            MDBRead(MSGSRV_ROOT, MSGSRV_A_AVAILABLE_SHARES, vs);
        } else {
            NMAPClientFree(sclient);

            return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
        }

        sprintf(scratch, "%s/%s", client->store, client->user);
        if (!XPLLongToDos(scratch, client->path, XPL_MAX_PATH)) {
            MDBDestroyValueStruct(vs);

            NMAPClientFree(sclient);

            return(ConnWrite(client->conn, MSG4224NOSTORE, sizeof(MSG4224NOSTORE) - 1));
        }

        PrintResources(client, vs, client->path, client->path, strlen(client->path), scratch, 256, sclient);

        reply = client->line;
        *reply++ = '2';
        *reply++ = '0';
        *reply++ = '0';
        *reply++ = '2';
        *reply++ = '-';
    } else {
        vs = NULL;
    }

    for (used = 0; (ccode != -1) && (used < vs->Used); used++) {
        ptr = vs->Value[used];
        if (*ptr == 'M') {
            ptr = strrchr(++ptr, '\r');
            if (ptr != NULL) {
                sclient->mailbox.message.count = 0;
                sclient->mailbox.message.used = 0;

                strcpy(sclient->mailbox.name, ++ptr);
            } else {
                continue;
            }

            /* We need to send non-existant hiearchichal elements. */
            name = strrchr(sclient->mailbox.name, '/');
            if (name != NULL) {
                ptr = strchr(sclient->mailbox.name, '/');

                while (ptr) {
                    *ptr = '\0';

                    sprintf(client->path, "%s/%s/%s", client->store, client->user, sclient->mailbox.name);
                    if (stat(client->path, &sb) == 0) {
                        *ptr = '/';
                        ptr = strchr(ptr + 1, '/');
                        continue;
                    }

                    XplMakeDir(client->path);
                    SetLongName(client->path, scratch);

                    reply = client->line + 5;
                    *reply++ = 'H';
                    *reply++ = '0';
                    *reply++ = '0';
                    sprintf(reply, "%s\r\n", sclient->mailbox.name);

                    ccode = ConnWrite(client->conn, client->line, strlen(client->line));

                    *ptr = '/';
                    ptr = strchr(ptr + 1, '/');
                }

                *name = '/';
            }

            reply = client->line + 5;
            *(reply++) = 'M';
            *(reply++) = '2';

            sprintf(scratch, "%s/%s/%s.sdx", client->store, client->user, sclient->mailbox.name);
            if (ReadNLockAquire(&sclient->mailbox.lock, &client->userHash, client->user, sclient->mailbox.name, 0) == NLOCK_AQUIRED) {
                index = fopen(scratch, "rb");
                if (index != NULL) {
                    fseek(index, 0, SEEK_SET);
                    if (fgets(scratch, sizeof(scratch), index)) {
                        if ((strncmp(scratch, NMAP_IDX_VERSION, 7) == 0) && (scratch[7] == '\r')) {
                            fgets(scratch, sizeof(scratch), index);
                            sclient->mailbox.flags = atol(scratch);
                        } else {
                            sclient->mailbox.flags = RESOURCE_FLAG_UNSUBSCRIBED;
                        }
                    } else {
                        sclient->mailbox.flags = RESOURCE_FLAG_UNSUBSCRIBED;
                    }

                    fclose(index);
                } else {
                    sclient->mailbox.flags = RESOURCE_FLAG_UNSUBSCRIBED;
                }

                ReadNLockRelease(sclient->mailbox.lock);
                sclient->mailbox.lock = NULL;
            } else {
                sclient->mailbox.flags = RESOURCE_FLAG_UNSUBSCRIBED;
            }
            
            if (!(sclient->mailbox.flags & RESOURCE_FLAG_UNSUBSCRIBED)) {
                *(reply++) = '1';
            } else {
                *(reply++) = '0';
            }

            sprintf(reply, "%s\r\n", sclient->mailbox.name);
        } else if (*ptr == 'C') {
            ptr = strrchr(++ptr, '\r');
            if (ptr != NULL) {
                sclient->calendar.entries.count = 0;
                sclient->calendar.entries.used = 0;

                strcpy(sclient->calendar.name, ++ptr);
            } else {
                continue;
            }

            /* We need to send non-existant hiearchichal elements. */
            name = strrchr(sclient->calendar.name, '/');
            if (name != NULL) {
                ptr = strchr(sclient->calendar.name, '/');

                while (ptr) {
                    *ptr = '\0';

                    sprintf(client->path, "%s/%s/%s", client->store, client->user, sclient->calendar.name);
                    if (stat(client->path, &sb) == 0) {
                        *ptr = '/';
                        ptr = strchr(ptr + 1, '/');
                        continue;
                    }

                    XplMakeDir(client->path);
                    SetLongName(client->path, scratch);

                    reply = client->line + 5;
                    *(reply++) = 'H';
                    *(reply++) = '0';
                    *(reply++) = '0';
                    sprintf(reply, "%s\r\n", sclient->calendar.name);

                    ccode = ConnWrite(client->conn, client->line, strlen(client->line));

                    *ptr = '/';
                    ptr = strchr(ptr + 1, '/');
                }

                *name = '/';
            }

            reply = client->line + 5;
            *(reply++) = 'C';
            *(reply++) = '2';
            
            sprintf(scratch, "%s/%s/%s.sdc", client->store, client->user, sclient->calendar.name);
            if (ReadNLockAquire(&sclient->calendar.lock, &client->userHash, client->user, sclient->calendar.name, 0) == NLOCK_AQUIRED) {
                index = fopen(scratch, "rb");
                if (index != NULL) {
                    fseek(index, 0, SEEK_SET);
                    if (fgets(scratch, sizeof(scratch), index)) {
                        if ((strncmp(scratch, NMAP_CAL_IDX_VERSION, 8) == 0) && (scratch[8] == '\r')) {
                            fgets(scratch, sizeof(scratch), index);
                            sclient->calendar.flags = atol(scratch);
                        } else {
                            sclient->calendar.flags = RESOURCE_FLAG_UNSUBSCRIBED;
                        }
                    } else {
                        sclient->calendar.flags = RESOURCE_FLAG_UNSUBSCRIBED;
                    }

                    fclose(index);
                } else {
                    sclient->calendar.flags = RESOURCE_FLAG_UNSUBSCRIBED;
                }

                ReadNLockRelease(sclient->calendar.lock);
                sclient->calendar.lock = NULL;
            } else {
                sclient->calendar.flags = RESOURCE_FLAG_UNSUBSCRIBED;
            }

            if (!(sclient->calendar.flags & RESOURCE_FLAG_UNSUBSCRIBED)) {
                *(reply++) = '1';
            } else {
                *(reply++) = '0';
            }

            sprintf(reply, "%s\r\n", sclient->calendar.name);
        } else if (*ptr == 'A') {
            reply = client->line + 5;
            *(reply++) = 'A';
            *(reply++) = '2';
            *(reply++) = '1';

            ptr = strrchr(++ptr, '\r');
            if (ptr != NULL) {
                sprintf(reply, "%s\r\n", ++ptr);
            } else {
                continue;
            }
        } else {
            continue;
        }

        if (ccode != -1) {
            ccode = ConnWrite(client->conn, client->line, strlen(client->line));
        }
    }

    if (ccode != -1) {
        ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
    }

    MDBDestroyValueStruct(vs);

    if (sclient->mailbox.message.info != NULL) {
        MemFree(sclient->mailbox.message.info);
    }

    if (sclient->calendar.entries.info != NULL) {
        MemFree(sclient->calendar.entries.info);
    }

    NMAPClientFree(sclient);

    return(ccode);
}

int 
NmapCommandSalv(void *param)
{
    int ccode;
    unsigned long startID;
    unsigned long stopID;
    unsigned long delCount = 0;
    unsigned char *ptr;
    BOOL all;
    struct stat sb;
    FILE *index;
    NMAPClient *client = (NMAPClient *)param;
    MessageInfoStruct *msgInfo;

    if (client->states & NMAP_CLIENT_MBOX) {
        ccode = MailboxOOBNotifications(client, &sb);
        if (ccode != -1) {
            ptr = client->buffer + 4;
        } else {
            return(ccode);
        }
    } else if (!(client->states & NMAP_CLIENT_USER)) {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    } else {
        return(ConnWrite(client->conn, MSG4225NOMBOX, sizeof(MSG4225NOMBOX) - 1));
    }

    /* SALV[ <message id>] */
    if (*ptr == '\0') {
        all = TRUE;

        startID = 0;
        stopID = client->mailbox.message.used;
    } else if ((*ptr++ == ' ') && (isdigit(*ptr))) {
        all = FALSE;

        startID = stopID = atoi(client->buffer + 5);
        if (startID && startID <= client->mailbox.message.used) {
            startID--;
        } else {
            return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
        sprintf(client->path, "%s/%s/%s.idx", client->store, client->user, client->mailbox.name);
    } else if (!(client->share.flags & NMAP_SHARE_MAILBOX_DENIED) && (client->share.mailbox.permissions & NMAP_SHARE_DELETE)) {
        sprintf(client->path, "%s/%s/%s.sdx", client->store, client->user, client->mailbox.name);
    } else {
        return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
    }

    if (client->flags & NMAP_OOBMESSAGES) {
        GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
    }

    if (WriteNLockAquire(client->mailbox.lock, 3000) == NLOCK_AQUIRED) {
        index = fopen(client->path, "r+b");
        if (index) {
            for (msgInfo = &(client->mailbox.message.info[startID]); startID < stopID; startID++, msgInfo++) {
                if (msgInfo->State & MSG_STATE_DELETED) {
                    msgInfo->State &= ~MSG_STATE_DELETED;

                    delCount++;

                    fseek(index, NMAP_IDX_HEADERSIZE + (startID * sizeof(MessageInfoStruct)) + offsetof(MessageInfoStruct, State), SEEK_SET);

                    if (!(msgInfo->State & MSG_STATE_RECENT)) {
                        fwrite(&(msgInfo->State), sizeof(unsigned long), 1, index);
                    } else {
                        msgInfo->State &= ~MSG_STATE_DELETED;
                        fwrite(&(msgInfo->State), sizeof(unsigned long), 1, index);
                        msgInfo->State |= MSG_STATE_DELETED;
                    }
                }
            }

            fclose(index);

            WriteNLockRelease(client->mailbox.lock);

            if (all) {
                ccode = ConnWriteF(client->conn, "1000 %lu %s\r\n", delCount, MSG1000SALVAGED);
            } else {
                ccode = ConnWriteF(client->conn, "1000 %s\r\n", MSG1000SALVAGED);
            }

            return(ccode);
        }

        WriteNLockRelease(client->mailbox.lock);
    }

    return(ConnWrite(client->conn, MSG4228CANTWRITEMBOX, sizeof(MSG4228CANTWRITEMBOX) - 1));
}

int 
NmapCommandSearch(void *param)
{
    int ccode;
    struct stat sb;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_MBOX) {
        ccode = MailboxOOBNotifications(client, &sb);
        if (ccode == -1) {
            return(ccode);
        }
    } else if (!(client->states & NMAP_CLIENT_USER)) {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    } else {
        return(ConnWrite(client->conn, MSG4225NOMBOX, sizeof(MSG4225NOMBOX) - 1));
    }

    if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
        if (client->flags & NMAP_OOBMESSAGES) {
            sprintf(client->path, "%s/%s/%s.idx", client->store, client->user, client->mailbox.name);
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
        }

        return(NmapSearch(client));
    } else if (!(client->share.flags & NMAP_SHARE_MAILBOX_DENIED)) {
        if (client->flags & NMAP_OOBMESSAGES) {
            sprintf(client->path, "%s/%s/%s.sdx", client->store, client->user, client->mailbox.name);
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
        }

        if (((ccode = NMAPSendCommandF(client->share.mailbox.conn, "%s\r\n", client->buffer)) != -1) 
                && ((ccode = NMAPReadAnswerLine(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE)) != -1)) {
            while (ccode != 1000) {
                if ((ConnWrite(client->conn, client->line, strlen(client->line))) != -1) {
                    if (ccode == 2002) {
                        ccode = NMAPReadAnswer(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE);
                        continue;
                    }
                }

                ccode = -1;
                break;
            }

            if (ccode == 1000) {
                ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
            }
        } else {
            return(ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1));
        }
    } else {
        ccode = ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1);
    }

    return(ccode);
}

int 
NmapCommandSflg(void *param)
{
    int ccode;
    unsigned long id;
    unsigned long flags;
    unsigned long newFlags;
    unsigned char *ptr;
    unsigned char *ptr2;
    struct stat sb;
    FILE *index;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_MBOX) {
        if (!(client->flags & NMAP_READONLY)) {
            ccode = MailboxOOBNotifications(client, &sb);
            if (ccode != -1) {
                ptr = client->buffer + 4;
            } else {
                return(ccode);
            }
        } else {
            return(ConnWrite(client->conn, MSG4243READONLY, sizeof(MSG4243READONLY) - 1));
        }
    } else if (!(client->states & NMAP_CLIENT_USER)) {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    } else {
        return(ConnWrite(client->conn, MSG4225NOMBOX, sizeof(MSG4225NOMBOX) - 1));
    }

    /* SFLG <message id> <flags> */
    if ((*ptr++ == ' ') && (isdigit(*ptr)) && ((ptr2 = strchr(ptr, ' ')) != NULL)) {
        *ptr2++ = '\0';
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    id = atol(ptr);
    newFlags = atol(ptr2);
    if (id && (id <= client->mailbox.message.used)) {
        id--;
    } else {
        return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
    }

    if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
        sprintf(client->path, "%s/%s/%s.idx", client->store, client->user, client->mailbox.name);

        if (client->flags & NMAP_OOBMESSAGES) {
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
        }

        flags = client->mailbox.message.info[id].State;
        if (flags == newFlags) {
            return(ConnWriteF(client->conn, "1000 %lu %s\r\n", flags, MSG1000FLAGSET));
        }
    } else if (!(client->share.flags & NMAP_SHARE_MAILBOX_DENIED)) {
        sprintf(client->path, "%s/%s/%s.sdx", client->store, client->user, client->mailbox.name);

        if (client->flags & NMAP_OOBMESSAGES) {
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
        }

        flags = client->mailbox.message.info[id].State;
        if (flags == newFlags) {
            return(ConnWriteF(client->conn, "1000 %lu %s\r\n", flags, MSG1000FLAGSET));
        }

        if ((newFlags & MSG_STATE_READ) && !(client->share.mailbox.permissions & NMAP_SHARE_SEEN) && !(flags & MSG_STATE_READ)) {
            newFlags &= ~MSG_STATE_READ;
        }

        if ((newFlags & (MSG_STATE_DELETED | MSG_STATE_PURGED)) && !(client->share.mailbox.permissions & NMAP_SHARE_DELETE)) {
            if ((newFlags & MSG_STATE_PURGED) && (!(flags & MSG_STATE_PURGED))) {
                newFlags &= ~MSG_STATE_PURGED;
            }

            if ((newFlags & MSG_STATE_DELETED) && (!(flags & MSG_STATE_DELETED))) {
                newFlags &= ~MSG_STATE_DELETED;
            }
        }
    } else {
        return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
    }

    if (WriteNLockAquire(client->mailbox.lock, 3000) == NLOCK_AQUIRED) {
        index = fopen(client->path, "r+b");
        if (index != NULL) {
            if ((newFlags & MSG_STATE_DELETED) && !(flags & MSG_STATE_DELETED)) {
                client->mailbox.message.count--;
            }

            if (!(newFlags & MSG_STATE_DELETED) && (flags & MSG_STATE_DELETED)) {
                client->mailbox.message.count++;
            }

            if (newFlags & MSG_STATE_PURGED) {
                client->mailbox.changed = TRUE;
            }

            fseek(index, NMAP_IDX_HEADERSIZE + (id * sizeof(MessageInfoStruct)) + offsetof(MessageInfoStruct, State), SEEK_SET);

            if (!(flags & MSG_STATE_RECENT)) {
                fwrite(&newFlags, sizeof(unsigned long), 1, index);
            } else {
                newFlags &= ~MSG_STATE_RECENT;
                fwrite(&newFlags, sizeof(unsigned long), 1, index);
                newFlags |= MSG_STATE_RECENT;
            }

            fclose(index);
            WriteNLockRelease(client->mailbox.lock);

            client->mailbox.message.info[id].State = newFlags;

            return(ConnWriteF(client->conn, "1000 %lu %s\r\n", newFlags, MSG1000FLAGSET));
        }

        WriteNLockRelease(client->mailbox.lock);
    }

    return(ConnWrite(client->conn, MSG4228CANTWRITEMBOX, sizeof(MSG4228CANTWRITEMBOX) - 1));
}

int 
NmapCommandShow(void *param)
{
    int ccode;
    unsigned long len;
    unsigned long used;
    unsigned char *ptr;
    unsigned char scratch[(XPL_MAX_PATH * 2) + 1];
    unsigned char pattern[64];
    struct stat sb;
    BOOL recurse = 256;
    NMAPClient *client = (NMAPClient *)param;
    MDBValueStruct *vs;

    /* SHOW[ <pattern>] */
    if (client->states & NMAP_CLIENT_USER) {
        ptr = client->buffer + 4;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if (*ptr == '\0') {
        pattern[0] = '\0';
    } else if ((*ptr++ == ' ') && (!isspace(*ptr)) && (strlen(ptr) < (sizeof(pattern) - 1))) {
        strcpy(pattern, ptr);

        ptr = strchr(pattern, '*');
        if (ptr) {
            *ptr = '\0';
        }

        ptr = strchr(pattern, '%');
        if (ptr) {
            *ptr = '\0';
            ptr = pattern;

            recurse = 0;
            while (*ptr) {
                if (*ptr == '/') {
                    recurse++;
                }
                ptr++;
            }
        }
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (client->states & NMAP_CLIENT_MBOX) {
        ccode = MailboxOOBNotifications(client, &sb);
        if (ccode == -1) {
            return(ccode);
        }
    }

    sprintf(scratch, "%s/%s", client->store, client->user);
    if (!XPLLongToDos(scratch, client->path, XPL_MAX_PATH)) {
        return(ConnWrite(client->conn, MSG4224NOSTORE, sizeof(MSG4224NOSTORE) - 1));
    }

    vs = MDBCreateValueStruct(NMAP.handle.directory, NULL);
    if (vs) {
        if (MsgFindObject(client->user, client->dn, NULL, NULL, NULL)) {
            ccode = ConnWrite(client->conn, "2002-Coming up\r\n", 16);
        } else {
            MDBDestroyValueStruct(vs);

            return(ConnWriteF(client->conn, MSG5244USERLOOKUPFAILURE, client->user));
        }
    } else {
        return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
    }

    if (ccode != -1) {
        MDBRead(client->dn, MSGSRV_A_AVAILABLE_SHARES, vs);
        MDBRead(MSGSRV_ROOT, MSGSRV_A_AVAILABLE_SHARES, vs);

        PrintDir(client, vs, client->path, scratch, pattern, strlen(client->path), scratch, recurse, NULL);

        len = strlen(pattern);
        for (used = 0; used < vs->Used; used++) {
            ptr = vs->Value[used];

            if ((*(ptr++) != 'M') || (*ptr != 'B')) {
                continue;
            }

            ptr = strrchr(++ptr, '\r');
            if ((ptr++ == NULL) || ((len > 0) && (XplStrNCaseCmp(ptr, pattern, len) != 0))) {
                continue;
            }

            ccode = ConnWriteF(client->conn, "2002-%s\r\n", ptr);
        }
    }

    if (ccode != -1) {
        ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
    }

    MDBDestroyValueStruct(vs);

    return(ccode);
}

int 
NmapCommandShowsub(void *param)
{
    int ccode;
    unsigned long len;
    unsigned long used;
    unsigned char *ptr;
    unsigned char scratch[(XPL_MAX_PATH * 2) + 1];
    unsigned char pattern[64];
    struct stat sb;
    FILE *index;
    BOOL recurse = 256;
    BOOL result;
    NMAPClient *client = (NMAPClient *)param;
    NMAPClient *sclient;
    MDBValueStruct *vs;

    /* SHOWSUB[ <pattern>] */
    if (client->states & NMAP_CLIENT_USER) {
        ptr = client->buffer + 7;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if (*ptr == '\0') {
        pattern[0] = '\0';
    } else if ((*ptr++ == ' ') && (!isspace(*ptr)) && (strlen(ptr) < (sizeof(pattern) - 1))) {
        strcpy(pattern, ptr);

        ptr = strchr(pattern, '*');
        if (ptr) {
            *ptr = '\0';
        }

        ptr = strchr(pattern, '%');
        if (ptr) {
            *ptr = '\0';
            ptr = pattern;

            recurse = 0;
            while (*ptr) {
                if (*ptr == '/') {
                    recurse++;
                }
                ptr++;
            }
        }
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (client->states & NMAP_CLIENT_MBOX) {
        ccode = MailboxOOBNotifications(client, &sb);
        if (ccode == -1) {
            return(ccode);
        }
    }

    sprintf(scratch, "%s/%s", client->store, client->user);
    if (!XPLLongToDos(scratch, client->path, XPL_MAX_PATH)) {
        return(ConnWrite(client->conn, MSG4224NOSTORE, sizeof(MSG4224NOSTORE) - 1));
    }

    result = MsgFindObject(client->user, client->dn, NULL, NULL, NULL);
    if (result) {
        sclient = NMAPClientAlloc();
    } else {
        return(ConnWriteF(client->conn, MSG5244USERLOOKUPFAILURE, client->user));
    }

    if (sclient != NULL) {
        strcpy(sclient->user, client->user);
        sclient->userHash = client->userHash; 
        sclient->store = client->store;
        sclient->mailbox.newline = TRUE;

        vs = MDBCreateValueStruct(NMAP.handle.directory, NULL);
    } else {
        return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
    }

    if (vs) {
        ConnWrite(client->conn, "2002-Coming up\r\n", 16);
    } else {
        NMAPClientFree(sclient);

        return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
    }

    MDBRead(client->dn, MSGSRV_A_AVAILABLE_SHARES, vs);
    MDBRead(MSGSRV_ROOT, MSGSRV_A_AVAILABLE_SHARES, vs);

    PrintDir(client, vs, client->path, scratch, pattern, strlen(client->path), scratch, recurse, sclient);

    ccode = 0;
    len = strlen(pattern);
    for (used = 0; used < vs->Used; used++) {
        ptr = vs->Value[used];

        if ((*(ptr++) != 'M') || (*ptr != 'B')) {
            continue;
        }

        ptr = strrchr(++ptr, '\r');
        if ((ptr++ == NULL) || ((len > 0) && (XplStrNCaseCmp(ptr, pattern, len) != 0))) {
            continue;
        }

        sprintf(client->path, "%s/%s/%s.sdx", client->store, client->user, ptr);
        if (ReadNLockAquire(&sclient->mailbox.lock, &client->userHash, client->user, ptr, 0) == NLOCK_AQUIRED) {
            index = fopen(client->path, "rb");
            if (index != NULL) {
                fseek(index, 0, SEEK_SET);
                if (fgets(client->line, CONN_BUFSIZE, index)) {
                    if ((strncmp(client->line, NMAP_IDX_VERSION, 7) == 0) && (client->line[7] == '\r')) {
                        fgets(client->line, CONN_BUFSIZE, index);
                        sclient->mailbox.flags = atol(client->line);
                    } else {
                        sclient->mailbox.flags = RESOURCE_FLAG_UNSUBSCRIBED;
                    }
                } else {
                    sclient->mailbox.flags = RESOURCE_FLAG_UNSUBSCRIBED;
                }

                fclose(index);
            } else {
                sclient->mailbox.flags = RESOURCE_FLAG_UNSUBSCRIBED;
            }

            ReadNLockRelease(sclient->mailbox.lock);
            sclient->mailbox.lock = NULL;
        } else {
            sclient->mailbox.flags = RESOURCE_FLAG_UNSUBSCRIBED;
        }

        if (!(sclient->mailbox.flags & RESOURCE_FLAG_UNSUBSCRIBED)) {
            ccode = ConnWriteF(client->conn, "2002-%s\r\n", ptr);
        } else {
            sprintf(client->path, "%s/%s/%s", client->store, client->user, ptr);
            if (access(client->path, 0) == 0) {
                ccode = ConnWriteF(client->conn, "2002-%s\r\n", ptr);
            }
        }
    }

    if (ccode != -1) {
        ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
    }

    MDBDestroyValueStruct(vs);

    if (sclient->mailbox.message.info != NULL) {
        MemFree(sclient->mailbox.message.info);
        sclient->mailbox.message.info = NULL;
    }

    NMAPClientFree(sclient);

    return(ccode);
}

int 
NmapCommandSinfo(void *param)
{
    int ccode;
    unsigned long stopID;
    unsigned long startID;
    unsigned char *ptr;
    struct stat sb;
    NMAPClient *client = (NMAPClient *)param;
    MessageInfoStruct *msgInfo;

    if (client->states & NMAP_CLIENT_MBOX) {
        ccode = MailboxOOBNotifications(client, &sb);
        if (ccode != -1) {
            ptr = client->buffer + 5;
        } else {
            return(ccode);
        }
    } else if (!(client->states & NMAP_CLIENT_USER)) {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    } else {
        return(ConnWrite(client->conn, MSG4225NOMBOX, sizeof(MSG4225NOMBOX) - 1));
    }

    if (*ptr == '\0') {
        startID = 0;
        stopID = client->mailbox.message.used;

        msgInfo = client->mailbox.message.info;
    } else if ((*ptr++ == ' ') && (isdigit(*ptr))) {
        startID = stopID = atol(ptr);
        if (startID && startID <= client->mailbox.message.used) {
            startID--;
        } else {
            return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
        }

        msgInfo = &(client->mailbox.message.info[startID]);
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
        if (client->flags & NMAP_OOBMESSAGES) {
            sprintf(client->path, "%s/%s/%s.idx", client->store, client->user, client->mailbox.name);
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
        }
    } else if (!(client->share.flags & NMAP_SHARE_MAILBOX_DENIED)) {
        if (client->flags & NMAP_OOBMESSAGES) {
            sprintf(client->path, "%s/%s/%s.sdx", client->store, client->user, client->mailbox.name);
            GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
        }
    } else {
        return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
    }

    ccode = ConnWriteF(client->conn, "2002 %lu %lu\r\n", stopID - startID, client->mailbox.id);
    for (; (ccode != -1) && (startID < stopID); startID++, msgInfo++) {
        ccode = ConnWriteF(client->conn, "%lu %lu %lu %lu %lu %lu %lu %lu %lu %s %lu %lu %lu %lu %lu %lu %lu\r\n", 
            startID + 1, 
            msgInfo->MSize, 
            msgInfo->SSize, 
            msgInfo->HeadPos, 
            msgInfo->BodyPos, 
            msgInfo->AuthPos, 
            (long unsigned int)msgInfo->AuthLen, 
            msgInfo->BodyLines, 
            msgInfo->State, 
            msgInfo->UIDL, 
            msgInfo->UID, 
            msgInfo->DateSent, 
            msgInfo->DateReceived, 
            msgInfo->UseSCMS, 
            msgInfo->SCMSID, 
            msgInfo->RealStart, 
            msgInfo->RealSize);
    }

    if (ccode != -1) {
        ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
    }

    return(ccode);
}

int 
NmapCommandStat(void *param)
{
    int ccode;
    unsigned long used;
    unsigned long totalSize = 0;
    unsigned long totalCount = 0;
    unsigned long purgedSize = 0;
    unsigned long purgedCount = 0;
    struct stat sb;
    NMAPClient *client = (NMAPClient *)param;

    MessageInfoStruct *msgInfo;

    /* fixme - verify command buffer; no arguments means no delimiter. */
    if (client->states & NMAP_CLIENT_MBOX) {
        ccode = MailboxOOBNotifications(client, &sb);
        if (ccode != -1) {
            if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
                if (client->flags & NMAP_OOBMESSAGES) {
                    sprintf(client->path, "%s/%s/%s.idx", client->store, client->user, client->mailbox.name);
                    GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
                }
            } else if (!(client->share.flags & NMAP_SHARE_MAILBOX_DENIED)) {
                if (client->flags & NMAP_OOBMESSAGES) {
                    sprintf(client->path, "%s/%s/%s.sdx", client->store, client->user, client->mailbox.name);
                    GET_CHANGED_FLAGS(client->path, IGNORE_PURGES, NOTIFY_CLIENT_ASYNC); 
                }
            } else {
                return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
            }
        } else {
            return(ccode);
        }
    } else if (!(client->states & NMAP_CLIENT_USER)) {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    } else {
        return(ConnWrite(client->conn, MSG4225NOMBOX, sizeof(MSG4225NOMBOX) - 1));
    }

    used = client->mailbox.message.used;
    msgInfo = client->mailbox.message.info;

    while (used-- != 0) {
        totalSize += msgInfo->SSize;
        totalCount++;

        if (msgInfo->State & MSG_STATE_PURGED) {
            purgedSize += msgInfo->SSize;
            purgedCount++;
        }

        msgInfo++;
    }

    return(ConnWriteF(client->conn, "1000 %s %lu %lu %lu %lu %lu %lu\r\n",
        client->user, totalCount, totalSize, client->mailbox.newCount, 
        client->mailbox.newSize, purgedCount, purgedSize));
}

static int 
StorDeliveryHeaders(NMAPClient *client, FILE *fh, BOOL noAuth, unsigned long flags, unsigned long date)
{
    size_t count;
    unsigned char delimiter[80];
    time_t time_of_day;

    time(&time_of_day);
    strftime(delimiter, 80, "%a %b %d %H:%M %Y", gmtime(&time_of_day));

    GENERATE_FROM_DELIMITER(delimiter, client->user, client->line, count);
    if (fwrite(client->line, sizeof(unsigned char), count, fh) == count) {
        if (noAuth == FALSE) {
            count = sprintf(client->line, "X-Auth-OK: %s@%s\r\n", client->user, NMAP.officialName);
            if (fwrite(client->line, sizeof(unsigned char), count, fh) == count) {
                ;
            } else {
                return(-1);
            }
        }

        if (flags || date) {
            if (date) {
                if (!flags) {
                    flags = MSG_STATE_RECENT;
                }

                count = sprintf(client->line, "X-NIMS-Flags: %lu %lu\r\n", flags, date);
            } else {
                count = sprintf(client->line, "X-NIMS-Flags: %lu\r\n", flags);
            }

            if (fwrite(client->line, sizeof(unsigned char), count, fh) == count) {
                ;
            } else {
                return(-1);
            }
        }

        return(0);
    }

    return(-1);
}

int 
NmapCommandStor(void *param)
{
    int ccode;
    long size = 0;
    unsigned long lcode;
    unsigned long flags;
    unsigned long date;
    unsigned char *ptr;
    unsigned char *ptr2;
    unsigned char *owner;
    unsigned char *mailbox;
    unsigned char *resource;
    BOOL noAuth;
    struct stat sb;
    FILE *fh;
    NMAPClient *client = (NMAPClient *)param;
    NLockStruct *lock;

    if (client->states & NMAP_CLIENT_USER) {
        if (!(NMAP.flags & NMAP_FLAG_DISK_SPACE_LOW)) {
            mailbox = client->buffer + 4;
        } else {
            return(ConnWrite(client->conn, MSG5221SPACELOW, sizeof(MSG5221SPACELOW) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if (client->states & NMAP_CLIENT_MBOX) {
        MailboxOOBNotifications(client, &sb);
    }

    /* STOR <mailbox>[ D<date>] <flags>[ NOAUTH][ <size>] */
    if ((*mailbox++ == ' ') 
            && (!isspace(*mailbox)) 
            && ((ptr = strchr(mailbox, ' ')) != NULL)) { 
        *ptr++ = '\0';

        if (*ptr == 'D') {
            ptr2 = strchr(ptr, ' ');
            if (ptr2) {
                *ptr2++ = '\0';
            } else {
                return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
            }

            date = atol(++ptr);

            ptr = ptr2;
        } else {
            date = 0;
        }

        ptr2 = strchr(ptr, ' ');
        if (ptr2) {
            *ptr2++ = '\0';
        }

        flags = atol(ptr);

        ptr = ptr2;

        noAuth = FALSE;
        if (ptr) {
            ptr2 = strchr(ptr, ' ');
            if (ptr2) {
                *ptr2++ = '\0';
            }

            if (XplStrNCaseCmp(ptr, "NOAUTH", 6) == 0) {
                noAuth = TRUE;

                ptr = ptr2;
            }
        }

        if (ptr) {
            size = atol(ptr);
        } else {
            size = 0;
        }
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    sprintf(client->path, "%s/%s/%s.box", client->store, client->user, mailbox);
    if (access(client->path, 0) == 0) {
        lock = NULL;
        lcode = ReadNLockAquire(&lock, &client->userHash, client->user, mailbox, 3000);
        if (lcode == NLOCK_AQUIRED) {
            lcode = WriteNLockAquire(lock, 2000);
        } else {
            return(ConnWrite(client->conn, MSG4120BOXLOCKED, sizeof(MSG4120BOXLOCKED) - 1));
        }

        if (lcode == NLOCK_AQUIRED) {
            fh = fopen(client->path, "ab");
            if (fh != NULL) {
                if ((ccode = StorDeliveryHeaders(client, fh, noAuth, flags, date)) != -1) {
                    if (((ccode = ConnWrite(client->conn, "2002 Send message\r\n", 19)) != -1) 
                            && ((ccode = ConnFlush(client->conn)) != -1)) {
                        if (size) {
                            ccode = ConnReadToFile(client->conn, fh, size);
                        } else {
                            ccode = ConnReadToFileUntilEOS(client->conn, fh);
                        }

                        if (ccode != -1) {
                            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
                        }
                    }
                } else {
                    ccode = ConnWrite(client->conn, MSG4229CANTWRITEMBOX, sizeof(MSG4229CANTWRITEMBOX) - 1);
                }

                if (!(NMAP.flushFlags & FLUSH_BOX_ON_STORE)) {
                    fclose(fh);
                } else {
                    XplFlushWrites(fh);
                    fclose(fh);
                }
            } else if (errno == ENOSPC) {
                ccode = ConnWrite(client->conn, MSG5220QUOTAERR, sizeof(MSG5220QUOTAERR) - 1);
            } else {
                ccode = ConnWrite(client->conn, MSG4229CANTWRITEMBOX, sizeof(MSG4229CANTWRITEMBOX) - 1);
            }

            WriteNLockRelease(lock);
        } else {
            ccode = ConnWrite(client->conn, MSG4120BOXLOCKED, sizeof(MSG4120BOXLOCKED) - 1);
        }

        ReadNLockRelease(lock);
        lock = NULL;
    } else if (IsAvailableShare(client->user, mailbox, NMAP_SHARE_MAILBOX, &owner, &resource)) {
        ccode = HandleShareStor(client, mailbox, date, flags, noAuth, size, owner, resource);

        MemFree(owner);
        MemFree(resource);
    } else {
        ccode = ConnWrite(client->conn, MSG4224NOMBOX, sizeof(MSG4224NOMBOX) - 1);
    }

    return(ccode);
}

int 
NmapCommandStraw(void *param)
{
    int ccode;
    long size;
    long lcode;
    unsigned char *ptr;
    unsigned char *owner;
    unsigned char *mailbox;
    unsigned char *resource;
    struct stat sb;
    FILE *fh;
    NMAPClient *client = (NMAPClient *)param;
    NLockStruct *lock;


    if (client->states & NMAP_CLIENT_USER) {
        if (!(NMAP.flags & NMAP_FLAG_DISK_SPACE_LOW)) {
            mailbox = client->buffer + 5;
        } else {
            return(ConnWrite(client->conn, MSG5221SPACELOW, sizeof(MSG5221SPACELOW) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if (client->states & NMAP_CLIENT_MBOX) {
        MailboxOOBNotifications(client, &sb);
    }

    /* STRAW <mailbox>[ <count>] */
    if ((*mailbox++ == ' ') && (!isspace(*mailbox))) {
        ptr = strchr(mailbox, ' ');
        if (ptr) {
            *ptr++ = '\0';
            size = atol(ptr);
        } else {
            size = 0;
        }
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    sprintf(client->path, "%s/%s/%s.box", client->store, client->user, mailbox);
    if (access(client->path, 0) == 0) {
        lock = NULL;
        lcode = ReadNLockAquire(&lock, &client->userHash, client->user, mailbox, 3000);
        if (lcode == NLOCK_AQUIRED) {
            lcode = WriteNLockAquire(lock, 2000);
        } else {
            return(ConnWrite(client->conn, MSG4120BOXLOCKED, sizeof(MSG4120BOXLOCKED) - 1));
        }

        if (lcode == NLOCK_AQUIRED) {
            fh = fopen(client->path, "ab");
            if (fh != NULL) {
                if (((ccode = ConnWrite(client->conn, "2002 Send message\r\n", 19)) != -1) 
                        && ((ccode = ConnFlush(client->conn)) != -1)) {
                    if (size) {
                        ccode = ConnReadToFile(client->conn, fh, size);
                    } else {
                        ccode = ConnReadToFileUntilEOS(client->conn, fh);
                    }

                    if (ccode != -1) {
                        ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
                    }
                }

                if (!(NMAP.flushFlags & FLUSH_BOX_ON_STORE)) {
                    fclose(fh);
                } else {
                    XplFlushWrites(fh);
                    fclose(fh);
                }

            } else if (errno == ENOSPC) {
                ccode = ConnWrite(client->conn, MSG5220QUOTAERR, sizeof(MSG5220QUOTAERR) - 1);
            } else {
                ccode = ConnWrite(client->conn, MSG4229CANTWRITEMBOX, sizeof(MSG4229CANTWRITEMBOX) - 1);
            }

            WriteNLockRelease(lock);
        } else {
            ccode = ConnWrite(client->conn, MSG4120BOXLOCKED, sizeof(MSG4120BOXLOCKED) - 1);
        }

        ReadNLockRelease(lock);
        lock = NULL;
    } else if (IsAvailableShare(client->user, mailbox, NMAP_SHARE_MAILBOX, &owner, &resource)) {
        ccode = HandleShareStraw(client, mailbox, size, owner, resource);

        MemFree(owner);
        MemFree(resource);
    } else {
        ccode = ConnWrite(client->conn, MSG4224NOMBOX, sizeof(MSG4224NOMBOX) - 1);
    }

    return(ccode);
}

int 
NmapCommandSubs(void *param)
{
    int count;
    int ccode;
    unsigned long lcode;
    unsigned char *ptr;
    unsigned char *name;
    unsigned char path[XPL_MAX_PATH + 1];
    struct stat sb;
    FILE *fh;
    NMAPClient *client = (NMAPClient *)param;
    NMAPClient *sclient;

    if (client->states & NMAP_CLIENT_USER) {
        ptr = client->buffer + 4;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if (client->states & NMAP_CLIENT_MBOX) {
        MailboxOOBNotifications(client, &sb);
    }

    /* SUBS <mailbox> */
    if ((*ptr++ == ' ') 
            && (!isspace(*ptr)) 
            && (*ptr)
            && ((strchr(ptr, '.') == NULL) || ((strstr(ptr, "../") == NULL) && (strstr(ptr, "..\\") == NULL)))) {
        sclient = NMAPClientAlloc();
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (sclient != NULL) {
        strcpy(sclient->user, client->user);
        sclient->userHash = client->userHash; 
        sclient->store = client->store;
        sclient->mailbox.newline = TRUE;

        strcpy(sclient->mailbox.name, ptr);
    } else {
        return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
    }

    sprintf(sclient->path, "%s/%s/%s.box", sclient->store, sclient->user, sclient->mailbox.name);
    if (access(sclient->path, 0) == 0) {
        sclient->mailbox.flags = 0;

        sprintf(sclient->path, "%s/%s/%s.idx", sclient->store, sclient->user, sclient->mailbox.name);
        lcode = ReadNLockAquire(&sclient->mailbox.lock, &sclient->userHash, sclient->user, sclient->mailbox.name, 3000);
        if (lcode == NLOCK_AQUIRED) {
            fh = fopen(sclient->path, "rb");
            if (fh != NULL) {
                fseek(fh, 0, SEEK_SET);
                if (fgets(sclient->line, CONN_BUFSIZE, fh)) {
                    if ((strncmp(sclient->line, NMAP_IDX_VERSION, 7) == 0) && (sclient->line[7] == '\r')) {
                        fgets(sclient->line, CONN_BUFSIZE, fh);
                        sclient->mailbox.flags = atol(sclient->line);
                        fclose(fh);
                    } else {
                        fclose(fh);
                        ParseMaildrop(sclient);
                    }
                } else {
                    fclose(fh);
                    ParseMaildrop(sclient);
                }
            }

            if (sclient->mailbox.flags & RESOURCE_FLAG_UNSUBSCRIBED) {
                lcode = WriteNLockAquire(sclient->mailbox.lock, 2000);
                if (lcode == NLOCK_AQUIRED) {
                    fh = fopen(sclient->path, "r+b");
                    if (fh != NULL) {
                        sclient->mailbox.flags &= ~RESOURCE_FLAG_UNSUBSCRIBED;

                        count = sprintf(sclient->line, "%010lu\r\n", sclient->mailbox.flags);

                        fseek(fh, 9, SEEK_SET);
                        fwrite(sclient->line, sizeof(unsigned char), count, fh);
                        fclose(fh);
                    }

                    WriteNLockRelease(sclient->mailbox.lock);
                }
            }

            ReadNLockRelease(sclient->mailbox.lock);
            sclient->mailbox.lock = NULL;

            if (sclient->mailbox.message.info) {
                MemFree(sclient->mailbox.message.info);
                sclient->mailbox.message.info = NULL;
            }
        }

        if (lcode == NLOCK_AQUIRED) {
            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
        } else {
            ccode = ConnWrite(client->conn, MSG4229CANTWRITEMBOX, sizeof(MSG4229CANTWRITEMBOX) - 1);
        }
    } else if (IsAvailableShare(sclient->user, sclient->mailbox.name, NMAP_SHARE_MAILBOX, NULL, NULL)) {
        name = strrchr(sclient->mailbox.name, '/');
        if (name != NULL) {
            ptr = strchr(sclient->mailbox.name, '/');

            while (ptr) {
                *ptr = '\0';

                sprintf(sclient->path, "%s/%s/%s", sclient->store, sclient->user, sclient->mailbox.name);
                if (stat(sclient->path, &sb) != 0) {
                    XplMakeDir(sclient->path);
                    SetLongName(sclient->path, path);
                }

                *ptr = '/';
                ptr = strchr(ptr + 1, '/');
            }

            *name = '/';
        }

        if (SetShareSubscription(sclient->store, sclient->user, sclient->userHash, sclient->mailbox.name, NMAP_SHARE_MAILBOX, TRUE)) {
            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
        } else {
            ccode = ConnWrite(client->conn,MSG4221MBOXLOCKED, sizeof(MSG4221MBOXLOCKED) - 1);
        }
    } else {
        ccode = ConnWrite(client->conn, MSG4224NOMBOX, sizeof(MSG4224NOMBOX) - 1);
    }

    NMAPClientFree(sclient);

    return(ccode);
}

int 
NmapCommandUnsubs(void *param)
{
    int count;
    int ccode;
    unsigned long lcode;
    unsigned char *ptr;
    struct stat sb;
    FILE *fh;
    NMAPClient *client = (NMAPClient *)param;
    NMAPClient *sclient;

    if (client->states & NMAP_CLIENT_USER) {
        ptr = client->buffer + 6;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if (client->states & NMAP_CLIENT_MBOX) {
        MailboxOOBNotifications(client, &sb);
    }

    /* UNSUBS <mailbox> */
    if ((*ptr++ == ' ') 
            && (!isspace(*ptr)) 
            && (*ptr)
            && ((strchr(ptr, '.') == NULL) || ((strstr(ptr, "../") == NULL) && (strstr(ptr, "..\\") == NULL)))) {
        if (XplStrCaseCmp(client->mailbox.name, ptr) != 0) {
            if (XplStrCaseCmp("INBOX", ptr) != 0) {
                sclient = NMAPClientAlloc();
            } else {
                return(ConnWriteF(client->conn, MSG3014REQNOTALLOWED, "  INBOX cannot be unsubscribed"));
            }
        } else {
            return(ConnWrite(client->conn, MSG4227TGTMBOXSELECTED, sizeof(MSG4227TGTMBOXSELECTED) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (sclient != NULL) {
        strcpy(sclient->user, client->user);
        sclient->userHash = client->userHash; 
        sclient->store = client->store;
        sclient->mailbox.newline = TRUE;

        strcpy(sclient->mailbox.name, ptr);
    } else {
        return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
    }

    sprintf(sclient->path, "%s/%s/%s.box", sclient->store, sclient->user, sclient->mailbox.name);
    if (access(sclient->path, 0) == 0) {
        sclient->mailbox.flags = 0;

        sprintf(sclient->path, "%s/%s/%s.idx", sclient->store, sclient->user, sclient->mailbox.name);
        lcode = ReadNLockAquire(&sclient->mailbox.lock, &sclient->userHash, sclient->user, sclient->mailbox.name, 0);
        if (lcode == NLOCK_AQUIRED) {
            fh = fopen(sclient->path, "rb");
            if (fh != NULL) {
                fseek(fh, 0, SEEK_SET);
                if (fgets(sclient->line, CONN_BUFSIZE, fh)) {
                    if ((strncmp(sclient->line, NMAP_IDX_VERSION, 7) == 0) && (sclient->line[7] == '\r')) {
                        fgets(sclient->line, CONN_BUFSIZE, fh);
                        sclient->mailbox.flags = atol(sclient->line);
                        fclose(fh);
                    } else {
                        fclose(fh);
                        ParseMaildrop(sclient);
                    }
                } else {
                    fclose(fh);
                    ParseMaildrop(sclient);
                }
            }

            if (!(sclient->mailbox.flags & RESOURCE_FLAG_UNSUBSCRIBED)) {
                lcode = WriteNLockAquire(sclient->mailbox.lock, 3000);
                if (lcode == NLOCK_AQUIRED) {
                    fh = fopen(sclient->path, "r+b");
                    if (fh != NULL) {
                        sclient->mailbox.flags |= RESOURCE_FLAG_UNSUBSCRIBED;
                        count = sprintf(sclient->line, "%010lu\r\n", sclient->mailbox.flags);

                        fseek(fh, 9, SEEK_SET);    /*    Skip    NMAP_IDX_VERSION "NIMSv04/r/n"    */
                        fwrite(sclient->line, sizeof(unsigned char), count, fh);
                        fclose(fh);
                    }

                    WriteNLockRelease(sclient->mailbox.lock);
                }
            }

            if (sclient->mailbox.message.info != NULL) {
                MemFree(sclient->mailbox.message.info);
                sclient->mailbox.message.info = NULL;
            }

            ReadNLockRelease(sclient->mailbox.lock);
            sclient->mailbox.lock = NULL;
        }

        if (lcode == NLOCK_AQUIRED) {
            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
        } else {
            ccode = ConnWrite(client->conn,MSG4221MBOXLOCKED, sizeof(MSG4221MBOXLOCKED) - 1);
        }
    } else if (IsAvailableShare(sclient->user, sclient->mailbox.name, NMAP_SHARE_MAILBOX, NULL, NULL)) {
        if (SetShareSubscription(sclient->store, sclient->user, sclient->userHash, sclient->mailbox.name, NMAP_SHARE_MAILBOX, FALSE)) {
            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
        } else {
            ccode = ConnWrite(client->conn,MSG4221MBOXLOCKED, sizeof(MSG4221MBOXLOCKED) - 1);
        }
    } else {
        ccode = ConnWrite(client->conn, MSG4224NOMBOX, sizeof(MSG4224NOMBOX) - 1);
    }

    NMAPClientFree(sclient);

    return(ccode);
}
