/****************************************************************************
 *
 * Copyright (c) 1997-2004 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#ifndef _HULA_CONNIO_H
#define _HULA_CONNIO_H

#include <stdarg.h>
#include <openssl/md5.h>
#include <openssl/ssl.h>
#include <openssl/rand.h>
#include <openssl/err.h>

#define CONN_BUFSIZE                    1023
#define CONN_TCP_MTU                    (1536 * 3)
#define CONN_TCP_THRESHOLD              256

#define SSL_USE_CLIENT_CERT             (1 << 0)
#define SSL_ALLOW_SSL2                  (1 << 1)
#define SSL_ALLOW_SSL3                  (1 << 2)
#define SSL_ALLOW_CHAIN                 (1 << 3)
#define SSL_DISABLE_EMPTY_FRAGMENTS     (1 << 4)
#define SSL_DONT_INSERT_EMPTY_FRAGMENTS (1 << 5)

#define CHOP_NEWLINE(s) \
        {   unsigned char *p; p = strchr((s), 0x0A); \
            if (p) { \
                *p = '\0'; \
            } \
            p = strrchr((s), 0x0D); \
            if (p) { \
                *p = '\0'; \
            } \
        }

#define SET_POINTER_TO_VALUE(p,s) \
        {   (p) = (s); \
            while(isspace(*(p))) { \
                (p)++; \
            } \
            if ((*(p) == '=') || (*(p) == ':')) { \
                (p)++; \
            } \
            while(isspace(*(p))) { \
                (p)++; \
            } \
        }


#if defined(LINUX) || defined(S390RH) || defined(SOLARIS)

#include <unistd.h>
#include <sys/poll.h>

#define IPSOCKET int

#define ConnSockOpt(c, o, v, p, l) setsockopt((c)->socket, (o), (v), (p), (l))

#define IPInit() XplGetInterfaceList()                                                                                 
#define IPCleanup() XplDestroyInterfaceList()                                                                             
#define IPsocket(d, t, p) socket((d), (t), (p))
#define IPaccept(s, a, l) accept((s), (a), (l))
#define IPlisten(s, b) listen((s), (b))
#define IPbind(s, a, l) bind((s), (a), (l))
#define IPconnect(s, a, l) connect((s), (a), (l))
#define IPrecv(s, b, l, f) recv((s), (b), (l), (f))
#define IPsend(s, b, l, f) send((s), (b), (l), (f))
#define IPacceptTO(s, a, l, t) accept((s), (a), (l))
#define IPconnectTO(s, a, l, t) connect((s), (a), (l))
#define IPrecvTO(s, b, l, f, t) recv((s), (b), (l), (f))
#define IPsendTO(s, b, l, f, t) send((s), (b), (l), (f))
#define IPclose(s) close((s))
#define IPshutdown(s, h) shutdown((s), (h))
#define IPgetsockname(s, a, l) getsockname((s), (a), (l))
#define IPgetpeername(s, a, l) getpeername((s), (a), (l))

#define CONN_TCP_READ(c, b, l, r) \
        { \
            struct pollfd pfd; \
            pfd.fd = (int)(c)->socket; \
            pfd.events = POLLIN; \
            (r) = poll(&pfd, 1, (c)->receive.timeOut * 1000); \
            if ((r) > 0) { \
                if (!(pfd.revents & (POLLERR | POLLHUP | POLLNVAL))) { \
                    do { \
                        (r) = IPrecv((c)->socket, b, l, 0); \
                        if ((r) >= 0) { \
                            break; \
                        } else if (errno == EINTR) { \
                            continue; \
                        } \
                        (r) = -1; \
                        break; \
                    } while (TRUE); \
                } else { \
                    (r) = -1; \
                } \
            } else { \
                (r) = -1; \
            } \
        }

#define CONN_TCP_WRITE(c, b, l, r) \
        { \
            do { \
                (r) = IPsend((c)->socket, b, l, 0); \
                if ((r) >= 0) { \
                    break; \
                } else if (errno == EINTR) { \
                    continue; \
                } \
                (r) = -1; \
                break; \
            } while (TRUE); \
        }

#elif defined(LIBC) || defined(NETWARE) || defined(WIN32)

#ifdef WIN32
typedef int socklen_t;
typedef unsigned int in_addr_t;
#define	ETIMEDOUT			WSAETIMEDOUT
#define	ECONNREFUSED		WSAECONNREFUSED
#define	ENETUNREACH			WSAENETUNREACH
#endif

#define ConnSockOpt(c, o, v, p, l) setsockopt((c)->socket, (o), (v), (p), (l))

#define	IPSOCKET												SOCKET
#define	IPInit()												{WSADATA data; WSAStartup(MAKEWORD(1,1), &data);}
#define	IPCleanup()											WSACleanup();
#define	IPsocket(domain, type, protocol)				socket(domain, type, protocol)
#define	IPaccept(s, addr, addrlen)						accept(s, addr, addrlen)
#define	IPlisten(sock, backlog)							listen(sock, backlog)
#define	IPbind(sock, addr, addrlen)					bind(sock, addr, addrlen)
#define	IPconnect(sock, addr, addrlen)				connect(sock, addr, addrlen)
#define	IPrecv(sock, buf, len, flags)					recv(sock, buf, len, flags)
#define	IPsend(sock, buf, len, flags)					send(sock, buf, len, flags)
#define	IPclose(sock)										closesocket(sock)
#define	IPshutdown(s, how)								shutdown(s, how)
#define	IPgetsockname(s, addr, addrlen)				getsockname(s, addr, addrlen)
#define	IPgetpeername(s, addr, addrlen)				getpeername(s, addr, addrlen)
#define	IPselect(nfds, rfds, wfds, efds, t)			select(nfds, rfds, wfds, efds, t)

#define CONN_TCP_READ(c, b, l, r) \
       { \
            fd_set rfds; \
            struct timeval timeout; \
            FD_ZERO(&rfds); \
            FD_SET((c)->socket, &rfds); \
            timeout.tv_usec = 0; \
            timeout.tv_sec = (c)->receive.timeOut; \
            (r) = select(FD_SETSIZE, &rfds, NULL, NULL, &timeout); \
            if ((r) > 0) { \
                (r) = IPrecv((c)->socket, (b), (l), 0); \
            } else { \
                (r) = -1; \
            } \
        }

#define CONN_TCP_WRITE(c, b, l, r)      (r) = IPsend((c)->socket, (b), (l), 0);

#else

#error Connection management library not implemented on this platform.

#endif

#define CONN_TCP_SEND(c, b, l, r) \
        { \
            if (!(c)->ssl.enable) { \
                CONN_TCP_WRITE((c), (b), (l), (r)); \
            } else { \
                (r) = CONN_SSL_WRITE((c), (b), (l)); \
            } \
        }

#define CONN_TCP_RECEIVE(c, b, l, r) \
        { \
            if (!(c)->ssl.enable) { \
                CONN_TCP_READ((c), (b), (l), (r)); \
            } else { \
                (r) = CONN_SSL_READ((c), (b), (l)); \
            } \
        }

#define CONN_TCP_FLUSH(c, b, e, r) \
        { \
            if ((b) < (e)) { \
                register char *curPTR = (char *)(b); \
                if (!(c)->ssl.enable) { \
                    while (curPTR < (e)) { \
                        CONN_TCP_WRITE((c), curPTR, (e) - curPTR, (r)); \
                        if ((r) > 0) { \
                            curPTR += (r); \
                            continue; \
                        } \
                        break; \
                    } \
                } else { \
                    while (curPTR < (e)) { \
                        (r) = CONN_SSL_WRITE((c), curPTR, (e) - curPTR); \
                        if ((r) > 0) { \
                            curPTR += (r); \
                            continue; \
                        } \
                        break; \
                    } \
                } \
                if (curPTR == (e)) { \
                    (r) = (e) - (b); \
                } else { \
                    (r) = -1; \
                } \
            } else { \
                (r) = 0; \
            } \
        }

#define CONN_TCP_CLOSE(c) \
        { \
            if ((c)->receive.buffer) { \
                (c)->receive.remaining = CONN_TCP_MTU; \
            } else { \
                (c)->receive.remaining = 0; \
            } \
            (c)->receive.read = (c)->receive.write = (c)->receive.buffer; \
            if ((c)->send.buffer) { \
                ConnFlush(c); \
                (c)->send.remaining = CONN_TCP_MTU; \
            } else { \
                (c)->send.remaining = 0; \
            } \
            (c)->send.read = (c)->send.write = (c)->send.buffer; \
            if ((c)->ssl.enable) { \
                SSL_shutdown((c)->ssl.context); \
                SSL_free((c)->ssl.context); \
                (c)->ssl.context = NULL; \
                (c)->ssl.enable = FALSE; \
            } \
            if ((c)->socket != -1) { \
                IPclose((c)->socket); \
                (c)->socket = -1; \
            } \
        }

#define CONN_SSL_NEW(c, s) \
        { \
            (c)->ssl.context = SSL_new(s); \
            if ((c)->ssl.context \
                    && (SSL_set_bsdfd((c)->ssl.context, (c)->socket) == 1)) \
                (c)->ssl.enable = TRUE; \
            } else { \
                if ((c)->ssl.context) { \
                    SSL_free((c)->ssl.context); \
                    (c)->ssl.context = NULL; \
                } \
                (c)->ssl.enable = FALSE; \
            } \
        }

#define CONN_SSL_FREE(c) \
        { \
            if ((c)->ssl.enable) { \
                SSL_free((c)->ssl.context); \
                (c)->ssl.context = NULL; \
                (c)->ssl.enable = FALSE; \
            } \
        }

#define CONN_SSL_ACCEPT(c, s) \
        { \
            (c)->ssl.context = SSL_new(s); \
            if ((c)->ssl.context \
                    && (SSL_set_bsdfd((c)->ssl.context, (c)->socket) == 1) \
                    && (SSL_accept((c)->ssl.context) == 1)) { \
                (c)->ssl.enable = TRUE; \
            } else { \
                if ((c)->ssl.context) { \
                    SSL_free((c)->ssl.context); \
                    (c)->ssl.context = NULL; \
                } \
                (c)->ssl.enable = FALSE; \
            } \
        }

#define CONN_SSL_CONNECT(c, s, r) \
        { \
            (c)->ssl.context = SSL_new(s); \
            if ((c)->ssl.context \
                    && (SSL_set_bsdfd((c)->ssl.context, (c)->socket) == 1) \
                    && (SSL_connect((c)->ssl.context) == 1)) { \
                (c)->ssl.enable = TRUE; \
            } else { \
                if ((c)->ssl.context) { \
                    SSL_free((c)->ssl.context); \
                    (c)->ssl.context = NULL; \
                } \
                (c)->ssl.enable = FALSE; \
            } \
        }

#define CONN_SSL_READ(c, b, l) SSL_read((c)->ssl.context, (void *)(b), (l));

#define CONN_SSL_WRITE(c, b, l) SSL_write((c)->ssl.context, (void *)(b), (l));

typedef struct _ConnectionBuffer {
    char *read;
    char *write;

    char *buffer;

    size_t remaining;

    int timeOut;
} ConnectionBuffer;

typedef SSL_METHOD *(* SSLMethod)(void);

typedef struct _ConnSSLConfiguration {
    unsigned long options;

    SSLMethod method;

    long mode;

    const char *cipherList;

    struct {
        long type;
        const unsigned char *file;
    } certificate;

    struct {
        long type;
        const unsigned char *file;
    } key;
} ConnSSLConfiguration;

typedef struct _Connection {
    IPSOCKET socket;

    struct {
        BOOL enable;

        SSL *context;
    } ssl;

    struct sockaddr_in socketAddress;

    ConnectionBuffer receive;
    ConnectionBuffer send;

    struct {
        struct _Connection *next;
        struct _Connection *previous;
    } queue;

    struct {
        struct _Connection *next;
        struct _Connection *previous;
    } allocated;
} Connection;

BOOL ConnStartup(unsigned long TimeOut, BOOL EnableSSL);
void ConnShutdown(void);

void ConnSSLContextFree(SSL_CTX *Context);
SSL_CTX *ConnSSLContextAlloc(ConnSSLConfiguration *ConfigSSL);

Connection *ConnAlloc(BOOL Buffers);
IPSOCKET ConnSocket(Connection *conn);
IPSOCKET ConnServerSocket(Connection *conn, int backlog);

IPSOCKET ConnConnect(Connection *conn, struct sockaddr *saddr, socklen_t slen, SSL_CTX *context);

int ConnEncrypt(Connection *conn, SSL_CTX *context);
BOOL ConnNegotiate(Connection *conn, SSL_CTX *Context);

int ConnClose(Connection *Conn, unsigned long Flags);
void ConnCloseAll(unsigned long Flags);

void ConnFree(Connection *Conn);

int ConnAccept(Connection *Server, Connection **Client);

int ConnSend(Connection *Conn, char *Buffer, unsigned int Length);
int ConnReceive(Connection *Conn, char *Buffer, unsigned int Length);

int ConnRead(Connection *Conn, char *Dest, int Length);
int ConnReadCount(Connection *Conn, char *Dest, int Count);
int ConnReadLine(Connection *Conn, char *Line, int Length);
int ConnReadAnswer(Connection *Conn, char *Line, int Length);
int ConnReadToFile(Connection *Conn, FILE *Dest, int Count);
int ConnReadToFileUntilEOS(Connection *Src, FILE *Dest);
int ConnReadToConn(Connection *Src, Connection *Dest, int Count);
int ConnReadToConnUntilEOS(Connection *Src, Connection *Dest);

int ConnWrite(Connection *Conn, const char *Source, int Count);
int ConnWriteF(Connection *Conn, const char *Format, ...);
int ConnWriteFile(Connection *Conn, FILE *Source);
int ConnWriteFromFile(Connection *Conn, FILE *Source, int Count);

int ConnFlush(Connection *Conn);

BIO_METHOD *BIO_s_bsdsocket(void);
BIO *BIO_new_bsdsocket(int fd, int close_flag);
int SSL_set_bsdfd(SSL *s,int fd);

/* fixme - to be deprecated */
#define XPLNETDB_DEFINE_CONTEXT

typedef int (*GenericReadFunc)(void *sktCtx, unsigned char *Buf, int Len, int readTimeout);
typedef int (*GenericWriteFunc)(void *sktCtx, unsigned char *Buf, int Len);

unsigned long XplGetHostIPAddress(void);
int XplPrintIPAddress(char *buffer, int bufLen, unsigned long address);
#define XplPrintHostIPAddress(buffer, bufLen) XplPrintIPAddress(buffer, bufLen, XplGetHostIPAddress());

BOOL XplIsLocalIPAddress(unsigned long Address);

int XplGetInterfaceList(void);
int XplDestroyInterfaceList(void);

int XplIPRead(void *sktCtx, unsigned char *Buf, int Len, int socketTimeout);
int XplIPReadSSL(void *sktCtx, unsigned char *Buf, int Len, int socketTimeout);
int XplIPWrite(void *sktCtx, unsigned char *Buf, int Len);
int XplIPWriteSSL(void *sktCtx, unsigned char *Buf, int Len);

#endif /* _HULA_CONNIO_H */
