#!/usr/bin/env python
# Copyright (C) 2005-2006 Sivan Greenberg. <sivan@ubuntu.com>
# 
# This software is distributed under the terms and conditions of the GNU General
# Public License. See http://www.gnu.org/copyleft/gpl.html for details.

import sys

sys.path.append("../") # add search path for common stuff and external modules needed

import os
from fsInfo import *
from ConffileParser import *
from HUBackup.common import HUB
import fsMisc
import glob
import time
import re
import signal
from pexpect import *

DEBUG_MODE = True
DEBUG_MODE_PROGRESS = True
DEBUG_PRINT = True


COPY_CMD_LINE = '-Q -y -v -m 256 -s %s -D -R %s -c %s -Z "*.gz" -Z "*.bz2" -Z "*.zip" ' 
COPY_DIFF_CMD_LINE = COPY_CMD_LINE + " -A %s "
ISOLATE_CMD_LINE = '-Q -v -s %s -A %s -C %s '

# will be used to exclude media files from operations at first stage, later could be used to extend file exclusion
# to custom specified by user.
# those two consts should be iterated with the respected arguments, to form the comprehensive command line to dar.
FILE_EXCL_ARG_LINE = ' -X ' # plain exclude
PATH_EXCL_ARG_LINE = ' -P ' # --prune from dar

def handleDarExceptions(rc):
    if rc:
        if rc  < 0: raise ChildTerminatedBySignalError
        if rc == 2: raise HardwareOrMemoryError
        if rc == 4: raise UserOrQuestionAbortedError
        if rc == 5: raise FileReadWriteError
        if rc == 6: raise UserCommandError
        if rc == 7: raise DarInternalError
        if rc == 8: raise IntegerOverFlowError
        if rc == 9: raise DarUnknownError
        if rc == 10: raise NonExistenFeatureError
    else:
        pass

class NoCatalogError(Exception):
    pass
class NoArchiveError(Exception):
    pass
class ChildTerminatedBySignalError(Exception):
    pass
class HardwareOrMemoryError(Exception):
    pass
class UserOrQuestionAbortedError(Exception):
    pass
class FileReadWriteError(Exception):
    pass
class UserCommandError(Exception):
    pass
class DarInternalError(Exception):
    pass
class IntegerOverFlowError(Exception):
    pass
class DarUnknownError(Exception):
    pass
class NonExistenFeatureError(Exception):
    pass
    
 

class Backup:
    def __init__(self, sliceSizeMB="700M",
                 backupName=None,
                 sourcePath=None,
                 target=None,
                 referencePath=None,
                 mediaInclude=True):
        # local const initializations
        # These will be used to store command line to instruct files and path exclusion from the backup archive.
        EX_PATH_CMDLINE = ''
        EX_MASK_CMDLINE = ''
        
        # regular members
        self.no_media = not mediaInclude
        self.proc = None
        self.changeSlice = False
        self.changeSliceMsg = ""
        self.changeReason = ""
        self.lastSliceNotFoundIn = ""
        self.neededSliceNumber = 0
        self.curOutputLine ="unset"
        self.curErrLine ="unset"
        self.oper_log = []
        self.err_log = []
        self.diffPath = referencePath
        self.restoreFromArchivePATH = referencePath # used also for specifying where is the archive to test integiry of.
        myuser = UserInfo()
        self.myhomedir = myuser.homedir()
        if not backupName: backupName = myuser.username()
        if not target: target  = myuser.homedir() + HUB.TEMP_DIR 
        if not sourcePath: sourcePath = myuser.homedir()
        self.catalogName = backupName + "-master-catalog"
        self.backupName =  backupName + "-master-archive"
        self.diffBackupName = backupName + "-diff-archive"
        self.diffCatalogName = backupName + "-diff-catalog"
        fsMisc.ensure_path(True,target)
        

        for myExclPath in HUB.EXCLUDE_PATHS:
            if myExclPath == HUB.TEMP_DIR:
                myExclPath = myExclPath[1:]
            EX_PATH_CMDLINE += PATH_EXCL_ARG_LINE +  myExclPath

        if DEBUG_MODE: print "Excluding: %s" % EX_PATH_CMDLINE
        self.sliceSizeMB = sliceSizeMB
        self.sourcePath = sourcePath
        self.target = target

        # flag the indicates what was the last backup operation , for cleaning up purpose.
        self.last_backup = "master"

        # set dummy slice size for testing purposes
        # COMMENT_ME
        # sliceSizeMB = "28M"
        
        # each backup phase is suggested as composed of an archiving phase and isloation phase.
        self.BACKUP_CMD_LINE = COPY_CMD_LINE % (sliceSizeMB, sourcePath, target + self.backupName)

        # checking if we need to exclude media files and act accordingly
        if self.no_media:
            for myExMask in HUB.EXCLUDE_MASKS:
                EX_MASK_CMDLINE += FILE_EXCL_ARG_LINE + myExMask
            if DEBUG_MODE: print "Excluding: %s" % EX_MASK_CMDLINE
        else:
            EX_MASK_CMDLINE = ''

        # prepare isolation command line
        self.ISOLATE_CMD_LINE = ISOLATE_CMD_LINE % (sliceSizeMB, target + self.backupName, target + self.catalogName )

        # differential backup handling
        if self.diffPath:
            self.last_backup = "diff" 
            self.BACKUP_CMD_LINE = COPY_DIFF_CMD_LINE % (sliceSizeMB,
                                                         sourcePath,
                                                         target + self.diffBackupName ,
                                                         self.diffPath + self.catalogName)
            self.ISOLATE_CMD_LINE = ISOLATE_CMD_LINE % (sliceSizeMB,
                                                        target + self.diffBackupName,
                                                        target + self.diffCatalogName)

        self.BACKUP_CMD_LINE += EX_PATH_CMDLINE + ' ' + EX_MASK_CMDLINE                        
            
        if DEBUG_MODE:
            print "Backup arguments to dar:"
            print "  ",self.BACKUP_CMD_LINE
            print "Isolation arguments to dar:" 
            print "  ",self.ISOLATE_CMD_LINE

    def cleanBackupData(self):
        print "* Erasing previous backup data."
        for p in glob.glob(self.target + self.backupName + "*"):
            if os.path.exists(p): os.unlink(p)

    def cleanLastDiff(self):
        print "* Removing last differential archive (%s)." % self.oldCatalogCtime
        for p in glob.glob(self.target + self.diffBackupName + "*"):
            if os.path.exists(p): os.unlink(p)
        print "* Removing last differential catalog (%s)." % self.oldCatalogCtime
        for p in glob.glob(self.target + self.diffCatalogName + "*"):
            if os.path.exists(p): os.unlink(p)

    def cleanLastCatalog(self):
        print "* Removing last catalog  (%s, %s)." % (self.catalogName, self.oldCatalogCtime)
        for p in glob.glob(self.target + self.catalogName + "*"):
            if os.path.exists(p): os.unlink(p)
        
    def getSliceNum(self, fileName):
        regularexp = r'.*/*\.([\d]*)\.dar\b'
        rexp = re.compile(regularexp,re.IGNORECASE)
        match = rexp.match(fileName) # see if we need to change to next slice
        if match:
            return match.group(1)
        else:
            return None

    def getExistantCatalogSlices(self):
        regularexp = r'.*/*\-master-catalog\.([\d]*)\.dar\b'
        mytempdir = self.myhomedir
        mytempdir = mytempdir + HUB.TEMP_DIR
        fList = ListFiles(regularexp,mytempdir)
        return fList

    def getExistantArchiveSlices(self):
        regularexp = r'.*/*\-master-archive\.([\d]*)\.dar\b'
        mytempdir = self.myhomedir
        mytempdir = mytempdir + HUB.TEMP_DIR
        fList = ListFiles(regularexp,mytempdir)
        return fList

    def getExistantDiffCatalogSlices(self):
        regularexp = r'.*/*\-diff-catalog\.([\d]*)\.dar\b'
        mytempdir = self.myhomedir
        mytempdir = mytempdir + HUB.TEMP_DIR
        fList = ListFiles(regularexp,mytempdir)
        return fList

    def getExistantDiffArchiveSlices(self):
        regularexp = r'.*/*\-diff-archive\.([\d]*)\.dar\b'
        mytempdir = self.myhomedir
        mytempdir = mytempdir + HUB.TEMP_DIR
        fList = ListFiles(regularexp,mytempdir)
        return fList

    def pre_build_kick(self):
        if self.diffPath:
            print "* Looking for %s." % (self.diffPath + self.catalogName + ".1.dar")
            if not os.path.exists(self.diffPath + self.catalogName +".1.dar"):
                        raise NoCatalogError
            self.oldCatalogCtime = time.ctime(os.path.getctime(self.diffPath + self.catalogName +".1.dar"))
            self.cleanLastDiff()
        self.proc = spawn('dar',self.BACKUP_CMD_LINE.split())
        return self.proc # return the spanwed process object for further manipulation

    def build(self):
        outputLine = ''
        proc_isatty = True
        while proc_isatty:
            try:
                outputLine = self.proc.readline()
            except TIMEOUT:
                yield True
            self.curOutputLine = outputLine
            if DEBUG_MODE_PROGRESS:
                print self.curOutputLine
            self.oper_log.append(self.curOutputLine)
            proc_isatty = self.proc.isalive()
            yield True
        handleDarExceptions(self.proc.exitstatus)
        yield False
        

    def pre_isolate_kick(self):
        if os.path.exists(self.target + self.catalogName +".1.dar"):
            self.oldCatalogCtime = time.ctime(os.path.getctime(self.target + self.catalogName +".1.dar"))
            self.cleanLastCatalog()
        self.proc = spawn('dar',self.ISOLATE_CMD_LINE.split())
        return self.proc

    def isolate(self):
        proc_eof = False
        err_eof = False
        out_eof = False
        outputLine = ''
        proc_isatty = True
        while proc_isatty:
            outputLine = self.proc.readline()
            self.curOutputLine = outputLine
            self.oper_log.append(self.curOutputLine)
            proc_isatty = self.proc.isalive()
            yield True            
        handleDarExceptions(self.proc.exitstatus)
        yield False



    def pre_restore_kick(self, restoreTarget=None, restoreArchiveType="master",restoreWhat="lost"):
        """
        restoreWhat = lost | lost-changed | all-files
        restoreArchiveType = "master" --> restore from the master archive
        restoreArchiveType = "diff" --> restore from the differential archive
        """
        if DEBUG_MODE: print "* Kicking restoration process."
        self.uid = os.getuid()
        self.RESTORE_CMD_LINE = "-v -k --no-warn -x %s -R %s"
        if restoreTarget==None:
            if DEBUG_MODE: print "* Restore path was not specified. Restoring to %s" % self.myhomedir
            restoreTarget = self.myhomedir

        # check if we are running as root
        if self.uid != 0:
            self.RESTORE_CMD_LINE += " -O " # neccessary when running as an unpriviliged user

        # prepare cmd args 
        if restoreWhat=="lost":
            # restore only lost files, -n does not permit overwrite of existing (changed) files.
            self.RESTORE_CMD_LINE += " -n"
        elif restoreWhat=="lost-changed":
            # restore changed and lost files, as overwrite is allowed by default when not specifying -n
            self.RESTORE_CMD_LINE += " -r"
        else:
            # this means restoreWhat=="all-files", meaning that we don't prevent overwriting, and then
            # we don't -r to restore only changed files. so the default is this.
            pass
            
        if restoreArchiveType=="master":
            self.RESTORE_CMD_LINE = self.RESTORE_CMD_LINE % (self.restoreFromArchivePATH + self.backupName, restoreTarget)
        else:
            self.RESTORE_CMD_LINE = self.RESTORE_CMD_LINE % (self.restoreFromArchivePATH + self.diffBackupName, restoreTarget)
        if DEBUG_MODE:
            print "restore CMD LINE:"
            print "  %s " % self.RESTORE_CMD_LINE
        if self.diffPath:
            if restoreArchiveType!="master":
                check_path = self.restoreFromArchivePATH + self.diffBackupName + ".1.dar"
            else:
                check_path = self.restoreFromArchivePATH + self.backupName + ".1.dar"
            print "* Looking for %s." % check_path
            if not os.path.exists(check_path):
                        raise NoArchiveError
        if DEBUG_MODE: print "* Finished kicking."

        self.proc = spawn('dar',self.RESTORE_CMD_LINE.split())
        return self.proc # return the spanwed process object for further manipulation

    def restore(self):
        regexNextSlice = re.compile(r'.*/*\.([\d]*)\.dar\b\ is required for further operation',re.IGNORECASE)
        regexLastSlice = re.compile(r'The last file of the set is not present in\ (\/[^ ,]*)',re.IGNORECASE)
        outputLine = ''
        proc_isatty=True
        matchNext = False
        matchLast = False
        while proc_isatty:
            if not matchNext and not matchLast:
                outputLine = self.proc.readline()
            matchNext = regexNextSlice.match(outputLine) # see if we need to change to next slice
            matchLast = regexLastSlice.match(outputLine) # see if we need the last slice for restore init
            if matchNext:
                self.neededSliceNumber = matchNext.group(1)
                if DEBUG_MODE: print "* Waiting for slice #%s" % self.neededSliceNumber                
                self.changeSliceMsg = self.curOutputLine
                self.changeReason = "next"                
                self.changeSlice = True
                yield True
                if not self.changeSlice:
                    self.proc.sendline("\n")
                    matchNext = False
            elif matchLast:
                self.lastSliceNotFoundIn = matchLast.group(1)
                if DEBUG_MODE: print "* Last slice of the set is expected at %s ." % self.lastSliceNotFoundIn
                self.changeSliceMsg = self.curOutputLine
                self.changeSlice = True
                self.changeReason = "last"
                yield True
                if not self.changeSlice:
                    self.proc.sendline("\n")
                    matchLast = False
            else:
                self.curOutputLine = outputLine
                self.oper_log.append(self.curOutputLine)
                yield True
            proc_isatty = self.proc.isalive()
            yield True
        handleDarExceptions(self.proc.exitstatus)
        yield False




    def pre_test_kick(self, testArchiveType="master"):
        """
        testArchiveType = "master" --> test integrity of a master archive
        testArchiveType = "diff" --> test integrity of a differential archive
        """
        if DEBUG_MODE: print "* Kicking testing/verification process."
        self.TEST_CMD_LINE = "-v -t %s" # setting the test command line arguments to dar.
            
        if testArchiveType=="master":
            self.TEST_CMD_LINE = self.TEST_CMD_LINE % (self.restoreFromArchivePATH + self.backupName)
        else:
            self.TEST_CMD_LINE = self.TEST_CMD_LINE % (self.restoreFromArchivePATH + self.diffBackupName)
        if DEBUG_MODE:
            print "test archive CMD_LINE:"
            print "  %s " % self.TEST_CMD_LINE
        if self.diffPath:
            if testArchiveType!="master":
                check_path = self.restoreFromArchivePATH + self.diffBackupName + ".1.dar"
            else:
                check_path = self.restoreFromArchivePATH + self.backupName + ".1.dar"
            print "* Looking for %s." % check_path
            if not os.path.exists(check_path):
                        raise NoArchiveError
        if DEBUG_MODE: print "* Finished kicking."

        self.proc = spawn('dar',self.TEST_CMD_LINE.split())
        return self.proc # return the spanwed process object for further manipulation


    def test(self):
        regexNextSlice = re.compile(r'.*/*\.([\d]*)\.dar\b\ is required for further operation',re.IGNORECASE)
        regexLastSlice = re.compile(r'The last file of the set is not present in\ (\/[^ ,]*)',re.IGNORECASE)
        outputLine = ''
        proc_isatty=True
        matchNext = False
        matchLast = False
        while proc_isatty:
            if DEBUG_MODE:
                print "* curOutputLine = %s" % self.curOutputLine
            if not matchNext and not matchLast:
                outputLine = self.proc.readline()
            matchNext = regexNextSlice.match(outputLine) # see if we need to change to next slice
            matchLast = regexLastSlice.match(outputLine) # see if we need the last slice for restore init
            if matchNext:
                self.neededSliceNumber = matchNext.group(1)
                if DEBUG_MODE: print "* Waiting for slice #%s" % self.neededSliceNumber                
                self.changeSliceMsg = self.curOutputLine
                self.changeReason = "next"                
                self.changeSlice = True
                yield True
                if not self.changeSlice:
                    self.proc.sendline("\n")
                    matchNext = False
            elif matchLast:
                self.lastSliceNotFoundIn = matchLast.group(1)
                if DEBUG_MODE: print "* Last slice of the set is expected at %s ." % self.lastSliceNotFoundIn
                self.changeSliceMsg = self.curOutputLine
                self.changeSlice = True
                self.changeReason = "last"
                yield True
                if not self.changeSlice:
                    self.proc.sendline("\n")
                    matchLast = False
            else:
                self.curOutputLine = outputLine
                self.oper_log.append(self.curOutputLine)
                yield True
            proc_isatty = self.proc.isalive()
            yield True
        handleDarExceptions(self.proc.exitstatus)
        yield False


    def abort(self):
        print "! Backup abort request. Killing backend process !"
        if self.proc:
            self.proc.kill(signal.SIGTERM)
        if self.last_backup=="master":
            self.cleanBackupData()
        
    def sliceChanged(self):
        self.changeSlice = False

    def waitingForSliceChange(self):
        return self.changeSlice

    def reportNeededSliceNumber(self):
        return self.neededSliceNumber

    def changeSliceReason(self):
        return self.changeReason

    def reportLastSliceNotFoundIn(self):
        return self.lastSliceNotFoundIn

    def search(self, fileSpec=None):
        """
        will be implemented in next version, where there will be probably an "advanced" mode
        for the next release, HUBackup will show differences between current state and backup,
        and offer users to restore those files to alternative location for cherry-picking
        pass
        """
        pass

    def reportLine(self):
        return self.curOutputLine

    def reportErrLine(self):
        return self.curErrLine




def doUserProgress(line):
        indi = ["[*--]", "[-*-]", "[--*]"]
        i = -1
        if line!="unset":
                   line = line.rstrip()
                   str = line[0:100]
                   if len(str) < 100:
                        str = str + (" " * (100 - len(str)))
                   sys.stdout.write("%s...\r" % str)
                   sys.stdout.flush()

if __name__ == "__main__":
    if len(sys.argv) < 2:
        print "usage: BackupEngine.py <slice-size> [<master-archive>] , or"
        print "       BackupEngine.py restore [<master-archive>] [<restoreToPath>]"
        sys.exit(1)
    if len(sys.argv) == 4:
        print "* Performing differential backup..."
        a = Backup(sys.argv[1],referencePath=sys.argv[2],sourcePath=sys.argv[3])
    else:
        a = Backup(sys.argv[1],mediaInclude=False)
    try:
        a.pre_build_kick()
        g = a.build()
        while g.next():
            line = a.reportLine()
            doUserProgress(line)
        sys.stdout.write("* Backup: Done. \r\n")
        a.pre_isolate_kick()
        g = a.isolate()
        while g.next():
            line = a.reportLine()
            doUserProgress(line)
        sys.stdout.write("* Cataloging: Done.\r\n")
    except UserOrQuestionAbortedError:
            print "ABORT: User interaction required but not provided."
    if DEBUG_MODE:
            print "=== log: ==="
            print a.oper_log
            print "=== error log ==="
            print a.err_log
            print "\n"

