/* 
 * hotwayd.c -- is a POP3-HTTPMail gateway. It will allow you to
 * check your HOTMAIL account from your unix box.
 * Created: Espeleta Tomas <espeleta@libero.it>, 12-Apr-2001
 *
 * My code uses libxml2 & a modified version of libghttp-1.0.9
 * POP3 code is based on nupop-0.4, by Kevin Stone <kstone@nuvox.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <pwd.h>
#include <string.h>
#include "httpmail.h"
#include "hotwayd.h"
#include "inet.h"
#include <sys/time.h>
#include "libghttp-1.0.9-mod/ghttp.h"

#undef AUTHORIZATION
#define AUTHORIZATION 10
#define TRANSACTION 20
#define UPDATE 40
#define TERMINATE 80

#define AUTHORIZATION_TIMEOUT 15
#define TRANSACTION_TIMEOUT   30

#define KG_STATE_UNCACHE -1
#define KG_STATE_CACHE 0
#define KG_STATE_READ 1
#define KG_STATE_DUMP 2

#define LAZY_BUFSIZE ((8 * 1024) - 1)

/* Global variables */
int  state;
char input[N_BUFLEN];
char username[N_BUFLEN];
struct passwd *userpw;
int timeout, timed_out = 0;

int cur_message = 0;
int top_lines = -1;
int pastheader = 0;
int log_level = 1; /* log level of 0-2 depending how much detail you want */
char *prog; /* program name */

/* extern FILE *stdin; */

extern int lazy_set(handler_t);
extern FOLDER_STRUCT *inbox_props;
extern int validate_username(char *);
extern cmdtable *cached_command(cmdtable *command, int argc, char **argv);
extern char *proxy;
extern char *proxy_username;
extern char *proxy_password;
char *folder=NULL; /* which folder of our mailbox to work on */
const char *default_folder="inbox";
char buffer[N_BUFFER];

void usage(void);
int args_sanity_check(void);
static str2val(register const char *str, register const char *what,
	       register int mi, register int ma);

/****** MISC FUNCTIONS ******/
/* get the next command from the client 
 * and put it in the global input buffer
 */
void get_input(void)
{
  alarm(timeout);
  memset(input, 0, N_BUFLEN);
  if (fgets(input, N_BUFLEN, stdin) == NULL) {
    if (errno != EINTR) {
      timed_out = errno;
      alarm(0);
      strlcpy(input, "timeout\n", N_BUFLEN);
    } else { 
      if (timed_out) {
	strlcpy(input, "timeout\n", N_BUFLEN);
      }
    }
  }

  alarm(0);
}

static void sig_alarm(int sig)
{
  timed_out = -1;
}	

/***** COMMAND HANDLERS ******/

void not_spoken_cmd( cmdtable *c, int argc, char *argv[])
{
  PSOUT("-ERR ");
  PSOUT(argv[0]);
  PSOUT(" not spoken here.");
  PCRLF;
  /*  register_command(NULL, NULL, 0, NULL, NULL); */
}

int stat_cmd_no= 0;

void pass_cmd( cmdtable *c, int argc, char *argv[])
{ extern cmdtable commands[];
 int ret; 
 
 register_command(NULL, NULL, 0, NULL, NULL);
  if (username[0] == '\0') {
    PSOUT("-ERR Please log in with USER first");
    PCRLF;
    return;
  }
	
  if (argc < 2) {
    PSOUT("-ERR Missing password argument");
    PCRLF;
    return;
  }
#ifdef DEBUG
  fprintf(stderr,"attempting to login with %s\n",username);
#endif
  switch (ret=httpmail_authenticate_user(username, argv[1])) {
  case INVALIDUSER:
    if (log_level > 0)
      LOG("AUTH failed, hotmail said username %s invalid or server too busy, host %s\n", username, inet_client_name);
    PSOUT("-ERR Remote server too busy (or possibly userid invalid)");
    PCRLF;
    return;
  case PASSWORDFAILED:
    if (log_level > 0)
      LOG("AUTH failed, hotmail said password for user %s invalid, host=%s\n", username, inet_client_name);
    PSOUT("-ERR Remote server said password was invalid");
    PCRLF;
    return;
  case ERROR:
    if (log_level > 0)
      LOG("AUTH failed, unexpected response from hotmail");
    PSOUT("-ERR Remote server returned unexpected status");
    PCRLF;
    return;
  case 0:
    if (log_level > 0)
      LOG("AUTH failed, couldn't find folder %s", folder);
    snprintf(buffer, N_BUFFER, "-ERR Unable to find folder %s on remote server", folder);
    PSOUT(buffer);
    PCRLF;
    return;
  case 1:
	break; /* we logged in ok */
  default:
    if (log_level > 0)
      LOG("AUTH failed, unexpected response in pass_cmd");
    PSOUT("-ERR Unexpected response in pass_cmd");
    PCRLF;
    return;
  }
	
  /* mailbox is open-- we go to transaction state
   * According to RFC 1939, a POP server should be able to handle
   * the RETR command immideately after entering the TRANSACTION
   * state. For this to work, we need to do an implicit STAT on the
   * mail store, otherwise RETR will fail. The STAT will also produce
   * the needed +OK response. To add insult to injury, we need to do
   * the 'main' STAT action (getting the list of messages from the server)
   * TWICE to make sure we get all the cookies we need to retrieve
   * messages. So, that httpmail_init() call is nothing more than a
   * 'bare bones', no output version of STAT.
   */

  httpmail_stat( &commands[stat_cmd_no] );
  state = TRANSACTION;

  /* log it */
  if (log_level > 1)
    LOG("  IN user=%s host=%s messages=%ld\n", username, inet_client_name, httpmail_nemails());
}

void user_cmd( cmdtable *c, int argc, char *argv[])
{
  username[0] = '\0'; /* reset username */
  folder = (char *) realloc(folder, strlen(default_folder)+1);
  strlcpy(folder, default_folder, strlen(default_folder)+1); /* reset folder */
  if (argc < 2) {
    PSOUT("-ERR Missing username argument");
    PCRLF;
    return;
  }
  switch (validate_username(argv[1])) {
  case E_LOGIN_NO_DOMAIN:
	PFSOUT("-ERR username invalid, specify your full address, e.g. %s@hotmail.com\n", argv[1]);
	break;
  case E_LOGIN_UNRECOGNISED_DOMAIN:
	PFSOUT( "-ERR domain not supported (must be a hotmail/msn/lycos/spray domain) in %s\n", argv[1]);
	break;
  case E_LOGIN_OK:
	PSOUT("+OK Username validated, Password required");
	PCRLF;
	{
	  char *domain_begin, *folder_pos;
 	  strlcpy(username,argv[1], N_BUFLEN); 
	  domain_begin = index( username, '@');
	  folder_pos = index(username,'/');
	  if (folder_pos) {
		/* if the folder name has spaces in it it will come up as extra
		 * argvs */
		int i;
		/* use folder_pos+1 so we get one past the slash */
		int foldername_len = strlen(folder_pos+1);
		for(i=2;i<argc;i++) {
		  foldername_len++; /* make room for space */
		  foldername_len += strlen(argv[i]); /* add next part of folder name */
		}
		/* use folder_pos+1 so we get one past the slash */
		folder = (char *) realloc(folder, foldername_len+1);
		strlcpy(folder, folder_pos+1, foldername_len+1);
		for(i=2;i<argc;i++) {
		  strlcat(folder, " ", foldername_len+1); /* put a space char */
		  strlcat(folder, argv[i], foldername_len+1);
		}
		*folder_pos='\0'; /* now kill the folder part of the username */
	  }
	  /* this needs to change, if it is a @hotmail.com address then remove the
		 domain from username, if it is @msn.com then keep the domain */
	  if ((strcasecmp("@hotmail.com",domain_begin) == 0)) {
		/* remove the domain from username for @hotmail.com addresses */
		*domain_begin = '\0'; 
	  }
	  register_command(NULL, NULL, 0, NULL, NULL);
	}
	break;
  default:
	PSOUT("-ERR Unrecognised error in function user_cmd");
	PCRLF;
	break;
  }
  return;
}

void list_cmd( cmdtable *c, int argc, char *argv[])
{
  if (argc == 1) {
    httpmail_list(c, NULL);
    return;
  }
  if (argc == 2) {
    httpmail_list(c, argv[1]);
  }
}		

void stat_cmd( cmdtable *c, int argc, char *argv[])
{
  httpmail_stat(c);
}

void rset_cmd( cmdtable *c, int argc, char *argv[])
{
  httpmail_rset(c);
}

void uidl_cmd( cmdtable *c, int argc, char *argv[])
{
  if (argc == 1) {
    httpmail_uidl(c,NULL);
    return;
  }
  if (argc == 2) {
    httpmail_uidl(c,argv[1]);
  }
}		

void top_cmd( cmdtable *c, int argc, char *argv[])
{
  if (argc < 3) {
    PSOUT("-ERR Missing arguments");
    PCRLF;
    return;
  }
  httpmail_top(c, argv[1], argv[2]);
}

void retr_cmd( cmdtable *c, int argc, char *argv[])
{
  if (argc < 2) {
    PSOUT("-ERR Missing message number argument");
    PCRLF;
    return;
  }
  httpmail_retrieve(c, argv[1]);
}

void dele_cmd( cmdtable *c, int argc, char *argv[])
{
  if (argc < 2) {
    PSOUT("-ERR Missing message number argument");
    PCRLF;
    return;
  }
  httpmail_delete(c, argv[1]);
}

void noop_cmd( cmdtable *c, int argc, char *argv[])
{
  PSOUT("+OK let us wait.");
  PCRLF;
    /* should a noop clear the cache?! */
/*   register_command( NULL, NULL, 0, NULL, NULL );	*/
}

void quit_cmd( cmdtable *c, int argc, char *argv[])
{
  PSOUT("+OK see you later!");
  PCRLF;
  if (state == TRANSACTION) {
    state = UPDATE;
  } else {
    state = TERMINATE;
  }
  register_command( NULL, NULL, 0, NULL, NULL );
}

void exit_cmd( cmdtable *c, int argc, char *argv[])
{
  PSOUT("-ERR Client timeout.  Exiting without update.");
  PCRLF;
  state = TERMINATE;
  register_command( NULL, NULL, 0, NULL, NULL );
}

/* 
 * Global jumptable for commands
 */
cmdtable commands[] = 	{ 	
  { "rpop", AUTHORIZATION, &not_spoken_cmd },
  { "apop", AUTHORIZATION, &not_spoken_cmd },
  { "auth", AUTHORIZATION, &not_spoken_cmd },
  { "user", AUTHORIZATION, &user_cmd },
  { "pass", AUTHORIZATION, &pass_cmd },
  { "list", TRANSACTION, &list_cmd },
  { "stat", TRANSACTION, &stat_cmd },
  { "dele", TRANSACTION, &dele_cmd },
  { "rset", TRANSACTION, &rset_cmd },
  { "noop", TRANSACTION, &noop_cmd },
  { "uidl", TRANSACTION, &uidl_cmd },
  { "top",  TRANSACTION, &top_cmd },
  { "retr", TRANSACTION, &retr_cmd },
  { "quit", (AUTHORIZATION | TRANSACTION), &quit_cmd },
  { "timeout", (AUTHORIZATION | TRANSACTION), &exit_cmd },
  { "", 0, NULL }
};


/******* MAIN LOOP ******/
void main_loop(void)
{
  int   argc;
  char *argv[MAXARGC],
    *x;
  int   cmd_implemented;
  cmdtable *c;
  extern char version[]; /* version number of hotway, set in VERSION */

  state = AUTHORIZATION; 
  username[0] = '\0';

  PSOUT("+OK POP3 ");
  if (inet_server_domain[0] != '\0') {
    PFSOUT("%s ", inet_server_domain);
  }
  PFSOUT("hotwayd v%s -> The POP3-HTTPMail Gateway.", version);
  PFSOUT(" Server on %s active.\015\012", inet_server_name);
  PFLUSH;
		
  while(state < UPDATE) {	
    /* make sure we've flushed all output before waiting on client */
    PFLUSH;

    /* set timeout according to our state */
    switch (state) {
    case AUTHORIZATION:
      timeout = AUTHORIZATION_TIMEOUT;
      break;

    case TRANSACTION:
      timeout = TRANSACTION_TIMEOUT;
      break;
    }

    /* get the input from the client */	
    get_input();

#ifdef DEBUG
    if (log_level > 1)
      LOG("Command: %s",input);
#endif
	
    /* init before continuing */
    cmd_implemented = 0;

    /* init argc, argv */
    for (argc = MAXARGC; argc; argv[--argc] = NULL);

    /* strip command from args --
     *  	assumes the cmd points to a properly null terminated string 
     *  	we'll strip off the crlf as well!! 
     */
    for(x = input; *x != '\0'; x++) {
      if (*x == ' ' || *x == '\n' || *x == '\r') {
	*x = '\0';
	if (argv[argc] == NULL) { continue; }
	else {
	  argc++;
	  if (argc >= MAXARGC) {
	    break;
	  }
	}
      } else {
	if (argv[argc] == NULL)
	  argv[argc] = x;
      }
    }
			
    if (!argc) {
      PSOUT("-ERR Null command");
      PCRLF;
      continue;
    }	

    /* cycle through jumptable */
    for (c = commands; c->handler != NULL; c++) {
	    if (!strcasecmp(c->name, argv[0])) {
		    cmd_implemented++;
		    if (c->states & state) {
			    cmdtable *prev_command;
			
			    if((prev_command = cached_command( c, argc, argv ))) {
#ifdef DEBUG
				    fprintf( stderr, "Command \"%s\" found in cache with timestamp %ld and %d arg(s)\n", prev_command->name, prev_command->timestamp, prev_command->argc 
			  );
#endif
			  PSOUT( prev_command->response );
			  if( prev_command->response[ strlen(prev_command->response)-1 ] != 0x10 ){
				  PCRLF;
			  }
		  }
		  else{
			  (c->handler)( c, argc, argv);
		  }
		} else {
		  PSOUT("-ERR That command is not available right now.");
		  PCRLF;
		  register_command( NULL, NULL, 0, NULL, NULL );
		}		
      }
    }

    if (!cmd_implemented) {
      PSOUT("-ERR Command not implemented");
      PCRLF;
    }
  }
  PFLUSH;

  snprintf(input, N_BUFLEN, "user=%s host=%s", username, inet_client_name);

  if (timed_out) {
    if ((timed_out == 1) || (timed_out == 9)) {
      if (log_level > 0)
	LOG("TIMEOUT %s (%d) client hangup?\n", input, timed_out);
    } else {
      if (log_level > 0)
	LOG("TIMEOUT %s (%d)\n", input, timed_out);
    }
    return;
  }

  if (state == TERMINATE) {
    if (log_level > 1)
      LOG("EXIT host=%s\n", inet_client_name);
    return;
  }

  if (state == UPDATE) {
    if (log_level > 1)
      LOG(" OUT\n");
    state = TERMINATE;
  }
}

char *grow_and_copy(char *dest, int curpos, char *src, int i)
{
  if(!(dest = realloc(dest, curpos+i+1)))
    return NULL;

  memcpy(dest+curpos, src, i);
  dest[curpos+i] = '\0';

  return dest;
}

/*
 * Function lazy_dump
 *
 * This function is used by lazy_handler to print out the buffer collected by
 * lazy_handler line by line. The reason we do it like this is two fold. First
 * so that we can implement TOP and second so that if we are downloading a big
 * message it means the mail reader won't time out waiting for hotwayd to spew
 * the message out in a big chunk. It also makes the problem more complex so
 * why not :-)
 */

int lazy_dump(char *pos, int len, int *curpos_p, int *curline_p, int maxlines) 
 {
  char foo[LAZY_BUFSIZE+1];

  char *cureol = NULL;

  int curpos = *curpos_p;
  int curline = *curline_p;

  int i;

  int just_sent_line = 1;
  int will_send_line = 0;

  foo[LAZY_BUFSIZE] = '\0';

  if(len == -1)
    len = strlen(pos);

  /* hack hack hack */
  if(maxlines == 0 && curline == 0)
    curline = -1;

  /* flush from pos */
  while(((maxlines == -1) || (curline < maxlines)) && (curpos < len)) {
	char *next_n = NULL;
	char *next_r = NULL;

	next_n=strstr(pos+curpos, "\n");
	next_r=strstr(pos+curpos, "\r");

	if (next_r && (*(next_r+1) == '\n')) next_r++; /* handle \r\n */

	/* check which is the next to come, a \r, a \n or a \r\n */
	if (next_r && next_n && (next_r < next_n)) {
	  cureol = next_r;
	} else if (next_n) {
	  cureol = next_n;
	} else {
	  cureol = NULL;
	}
	  
	if(cureol) 
      i = MIN(cureol-(pos+curpos)+1, LAZY_BUFSIZE);

    if(cureol && (i == (cureol-(pos+curpos)+1))) {

      if(maxlines && pastheader) {
		curline++; /* we are past the header, start counting lines */
	  }

      if( !pastheader && /* still in header */
	  just_sent_line &&  /* just flushed to a \r\n */
	  ( (pos[curpos] == '\r' && pos[curpos+1] == '\n') ||
	    (pos[curpos] == '\n') ) ) { /* and find ourselves at another \r\n */ 
		pastheader++; /* right, now we can start counting lines.. */
	if(maxlines == 0 && curline == -1)
	   curline = 0;
      }
      
      will_send_line = 1;
    } else {
      i = MIN(len-curpos, LAZY_BUFSIZE);
      just_sent_line = 0;
    }

    memcpy(foo, pos+curpos, i);
    foo[i] = '\0';

    if(just_sent_line && foo[0] == '.')
	printf(".");

    printf("%s", foo);

    just_sent_line = will_send_line;

    will_send_line = 0;

    curpos += i;
  }

  *curline_p = curline;
  *curpos_p = curpos;

  if(curpos >= len) {
	return 1;
  }
  if(curline >= maxlines) {
    return 2;
  }
  return 0;
}

/*
 * Function lazy_handler
 *
 * We use this function to progressively read a message from the mail
 * server. It works by filling an internal buffer (of LAZY_BUFSIZE) and
 * sending chunks to lazy_dump to be output. 
 * 
 * what i said about "it was a mess when i got here" being no excuse..
 * i was lying through my libghttping[1] teeth
 *  -jbm
 * [1]: yes, it's now officially profanity.
 */
   

int lazy_handler(char *pos, int len, int totallen, int fd)
{
  char foo[LAZY_BUFSIZE + 1];
  char *dest = NULL;
  int curpos = 0;
  int i;
  int kg = KG_STATE_READ;
  int curline = 0;
  int curlen = 0;
  foo[LAZY_BUFSIZE] = '\0';

  PSOUT("+OK Going to Lazy-download that one\r\n");

  if(!(dest = grow_and_copy(dest, 0, pos, len))) {
    printf("-ERR OR damn, i dunno: %s\r\n", strerror(errno));
    return -1;
  }
  curlen += len;

  while(kg > KG_STATE_CACHE) {
    if(kg == KG_STATE_READ) {
      /* better not to call read at end of buffer so we'll use totallen which
       * was discovered from Content-Length of headers to see if we have
       * finished reading all of the message. It is also possible to use
       * select here, but I'd rather leave it as the default timeout incase
       * someone has a very slow connection */
      if (curlen < totallen)
	i = read(fd, foo, LAZY_BUFSIZE);
      else
	i = 0;
      if(i < 0) {
	kg = KG_STATE_UNCACHE;
	continue;
      }

      if(i == 0)
	kg = KG_STATE_DUMP;

      foo[i] = '\0';

      if(!(dest = grow_and_copy(dest, curlen, foo, i))) { 
	printf("-ERR OR damn, i dunno: %s\r\n", strerror(errno));
	return 0;
      }
      curlen += i;
    }

    if(kg == KG_STATE_DUMP || kg == KG_STATE_READ) {
      i = lazy_dump(dest, curlen, &curpos, &curline, top_lines);

      if(kg == KG_STATE_DUMP && i == 1)
	kg = KG_STATE_CACHE;
      if(i == 2) /* to handle TOP as well as RETR -- top probably won't hit DUMP*/
	kg = KG_STATE_UNCACHE;

    }
  }

  PSOUT(".\r\n"); /* required by POP3 standard at end of message */

  free(inbox_props->msg_props[cur_message]->message);

  if(kg == KG_STATE_CACHE)
    inbox_props->msg_props[cur_message]->message = dest;
  else
    free(dest);

  return 1;
}

/****** MAIN *****/
int main(int argc, char **argv)
{
  cmdtable *c= commands;
  int		opt;

  prog=strdup(argv[0]);
  
  while ((opt = getopt(argc, argv, "hp:u:q:l:")) != -1) {

	switch (opt) {

	case 'h': /* display the usage page */
	  usage();
	  return 0;

	case 'l': /* setup proxy usage */
	  log_level = str2val(optarg, "log level", 0, 3);
	  break;

	case 'p': /* setup proxy usage */
	  proxy = (char *) malloc (strlen(optarg)+1);
	  strlcpy(proxy, optarg, strlen(optarg)+1);
	  break;

	case 'u':
	  proxy_username = (char *) malloc (strlen(optarg)+1);
	  strlcpy(proxy_username, optarg, strlen(optarg)+1);
	  break;

	case 'q':
	  proxy_password = (char *) malloc (strlen(optarg)+1);
	  strlcpy(proxy_password, optarg, strlen(optarg)+1);
	  break;

	default:
	  usage();
	  return -1;
	}
  }

  /* check validity of command line arguments */
  if (!args_sanity_check()) {
	usage();
	return -1;
  }

  OPENLOG;
  inet_init();
  signal(SIGALRM, sig_alarm);

  lazy_set(&lazy_handler);

  while( c && c->handler ){
	  if( strcmp( c->name, "stat")== 0 ){
		  stat_cmd_no= (int) (c- commands);
	  }
	  c++;
  }

  main_loop();
	
  httpmail_destroy();
  CLOSELOG;
  DPRINTF("exiting.\n");
  return(0);
}


void usage(void) {
  printf("Usage: hotwayd [options and arguments]\n"
	     "  -h print usage page\n"
	     "  -l <loglevel> specify the logging level (0-2)\n"
	     "  -p [proxy:port] specify proxy\n"
	     "  -u [username] specify proxy username\n"
	     "  -q [password] specify proxy password\n"
	     "Example: hotwayd -p http://proxy:8080 -u dave -q proxypass\n"
	     "Note: proxy can be specified without a username or password\n");
}


int args_sanity_check(void) {
  if (proxy_username && !proxy_password) {
	free(proxy_username);
	printf("Invalid args: Proxy user name supplied, but no password!\n");
	return 0;
  } else if (!proxy_username && proxy_password) {
	free(proxy_password);
	printf("Invalid args: Proxy password supplied but no user name!\n");
	return 0;	  
  } else if ((proxy_username || proxy_password) && !proxy) {
	if (proxy_username) free(proxy_username);
	if (proxy_password) free(proxy_password);
	printf("Invalid args: Proxy user name or password supplied but no proxy given\n");
	return 0;
  }	
  return 1; /* arguments are sane */
}


static str2val(register const char *str, register const char *what,
    register int mi, register int ma)
{
        register const char *cp;
        register int val;
        char *ep;

        if (str[0] == '0' && (str[1] == 'x' || str[1] == 'X')) {
                cp = str + 2;
                val = (int)strtol(cp, &ep, 16);
        } else
                val = (int)strtol(str, &ep, 10);
        if (*ep != '\0') {
                fprintf(stderr, "%s: \"%s\" bad value for %s \n",
			prog, str, what);
                exit(1);
        }
        if (val < mi && mi >= 0) {
                if (mi == 0)
                        fprintf(stderr, "%s: %s must be >= %d\n",
                            prog, what, mi);
                else
                        fprintf(stderr, "%s: %s must be > %d\n",
                            prog, what, mi - 1);
                exit(1);
        }
        if (val > ma && ma >= 0) {
                fprintf(stderr, "%s: %s must be <= %d\n", prog, what, ma);
                exit(1);
        }
        return (val);
}
