/* $Id: search.h 1154 2006-02-22 09:50:05Z holger $
 *
 * HoiChess/search.h
 *
 * Copyright (C) 2004, 2005 Holger Ruckdeschel <holger@hoicher.de>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 */
#ifndef SEARCH_H
#define SEARCH_H

#include "common.h"
#include "board.h"
#include "book.h"
#include "clock.h"
#include "eval.h"
#include "game.h"
#include "hash.h"
#include "historytable.h"
#include "move.h"
#include "movelist.h"
#include "shell.h"
#include "thread.h"
#include "tree.h"

/* forward declaration */
class Shell;

class Search
{
      public:
	enum search_modes { MOVE, ANALYZE, PONDER };
	enum features_e {
		FEAT_TIME_FL = 0x01,
		FEAT_TIME_IC = 0x02,
		FEAT_TIME_NB = 0x04,
		FEAT_TIME_FH = 0x08
	};
		
      private:
	Shell * shell;
	
	Evaluator * evaluator;
	HashTable * hashtable;
	HistoryTable * histtable[2];
	Book * book;
	
	const Game * game;
	Clock * clock;

	Tree tree;
	int rootdepth;
	int maxdepth;
	
	int mode;
	unsigned long features;
	
	Mutex start_mutex;
	Mutex main_mutex;
	Thread * thread;
	bool stop;
		
	unsigned long stat_nodes;
	unsigned long stat_nodes_quiesce;
	unsigned long stat_cut;
	unsigned long stat_nullcut;
	unsigned long stat_moves_sum;
	unsigned long stat_moves_cnt;
	unsigned long stat_moves_sum_quiesce;
	unsigned long stat_moves_cnt_quiesce;

	/* Overall statistics */
	unsigned long ostat_knodes;
	unsigned long ostat_csecs;
	unsigned int  ostat_depth_sum;
	unsigned int  ostat_depth_cnt;
	
	unsigned long next_timecheck;
	unsigned long next_update;

	bool showthinking;
	
      public:
	Search(Shell * shell);
	~Search();
	
      public:
	void start(Game game, int mode);
	void start(const Board & board, const Clock & clock, int mode);
	void start_thread(Game game, int mode);
	void stop_thread();
	static void * thread_main(void * arg);

	void interrupt();
	Move get_best();
	void set_book(Book * book);
	void set_depthlimit(unsigned int depth);
	void set_hashtable(HashTable * hashtable);
	void set_pawnhashtable(PawnHashTable * pawnhashtable);
	void set_evalcache(EvaluationCache * evalcache);
	void set_showthinking(bool x);
	void set_features(unsigned long f);
	unsigned long get_features() const;
	
      private:
	void main();
	void iterate();
	int search_root(unsigned int ply, int depth, int alpha,
			int beta);
	int search(unsigned int ply, int depth, int alpha, int beta);
	int quiescence_search(unsigned int ply, int alpha, int beta);

	bool is_draw();

      private:
	void check_time();
	void reset_statistics();
	void print_statistics();
	void update_overall_statistics();
	void print_overall_statistics();
	void print_header();
	void print_thinking(unsigned int depth);
	void print_thinking_terminal(unsigned int depth);
	void print_thinking_xboard(unsigned int depth);
	void print_result(unsigned int depth, int score, char c);
	void print_result_terminal(unsigned int depth, int score, char c);
	void print_result_xboard(unsigned int depth, int score, char c);
	std::string get_best_line(unsigned int depth, char c) const;
	void clear_line();
};

#endif // SEARCH_H
