/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: rpparser.cpp,v 1.3.24.1 2004/07/09 01:54:51 hubbe Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */

// include
#include "hxtypes.h"
#include "hxwintyp.h"
#include "hxcom.h"
#include "hxresult.h"
#include "hxcomm.h"
#include "ihxpckts.h"
#include "hxxml.h"
#include "hxxres.h"
#include "hxxrsmg.h"
#include "hxerror.h"

// hxmisc
#include "baseobj.h"
#include "unkimp.h"
#include "hxxmlprs.h"
#include "hxparse.h"

// hxcont
#include "hxbuffer.h"
#include "hxslist.h"
#include "hxmap.h"

// coreres
#include "pixres.h"

// pxcomlib
#include "pxrect.h"
#include "pxcolor.h"
#include "pxeffect.h"
#include "gstring.h"
#include "parseurl.h"
#include "rpfile.h"
#include "pxerror.h"
#include "rpparser.h"

// hxdebug
#include "hxheap.h"
#ifdef _DEBUG
#undef HX_THIS_FILE		
static char HX_THIS_FILE[] = __FILE__;
#endif

#define BASE_VERSION      HX_ENCODE_PROD_VERSION(0, 0, 0, 0)
#define U2_VERSION        HX_ENCODE_PROD_VERSION(1, 1, 0, 0)
#define REDSTONE_VERSION  HX_ENCODE_PROD_VERSION(1, 1, 0, 0) // XXXMEH - for now, we keep it the same
#define OPACITY_VERSION   HX_ENCODE_PROD_VERSION(1, 4, 0, 0)

const UINT32 PXRealPixParser::m_ulHighestSupportedContentVersion = OPACITY_VERSION;

const PXRealPixParser::PXStringTable PXRealPixParser::m_pTagTable[] = 
{
    { kTagRoot,             "imfl"             },
    { kTagHead,             "head"             },
    { kTagImage,            "image"            },
    { kTagFill,             "fill"             },
    { kTagFadein,           "fadein"           },
    { kTagCrossfade,        "crossfade"        },
    { kTagFadeout,          "fadeout"          },
    { kTagWipe,             "wipe"             },
    { kTagAnimate,          "animate"          },
    { kTagViewchange,       "viewchange"       },
    { kTagEffect,           "effect"           },
    { kNumTags,             NULL               }
};

const PXRealPixParser::PXStringTable PXRealPixParser::m_pAttrTable[] =
{
    { kAttrWidth,             "width"            },
    { kAttrHeight,            "height"           },
    { kAttrBitrate,           "bitrate"          },
    { kAttrTimeformat,        "timeformat"       },
    { kAttrVersion,           "version"          },
    { kAttrTitle,             "title"            },
    { kAttrAuthor,            "author"           },
    { kAttrCopyright,         "copyright"        },
    { kAttrBackgroundcolor,   "background-color" },
    { kAttrBackgroundOpacity, "backgroundOpacity"},
    { kAttrStart,             "start"            },
    { kAttrPreroll,           "preroll"          },
    { kAttrAspect,            "aspect"           },
    { kAttrUrl,               "url"              },
    { kAttrMaxfps,            "maxfps"           },
    { kAttrDuration,          "duration"         },
    { kAttrHandle,            "handle"           },
    { kAttrName,              "name"             },
    { kAttrTarget,            "target"           },
    { kAttrSrcx,              "srcx"             },
    { kAttrSrcy,              "srcy"             },
    { kAttrSrcw,              "srcw"             },
    { kAttrSrch,              "srch"             },
    { kAttrDstx,              "dstx"             },
    { kAttrDsty,              "dsty"             },
    { kAttrDstw,              "dstw"             },
    { kAttrDsth,              "dsth"             },
    { kAttrColor,             "color"            },
    { kAttrDirection,         "direction"        },
    { kAttrType,              "type"             },
    { kAttrPackage,           "package"          },
    { kAttrData,              "data"             },
    { kAttrFile,              "file"             },
    { kAttrStartsrcx,         "startsrcx"        },
    { kAttrStartsrcy,         "startsrcy"        },
    { kAttrStartsrcw,         "startsrcw"        },
    { kAttrStartsrch,         "startsrch"        },
    { kAttrStartdstx,         "startdstx"        },
    { kAttrStartdsty,         "startdsty"        },
    { kAttrStartdstw,         "startdstw"        },
    { kAttrStartdsth,         "startdsth"        },
    { kAttrCenter,            "center"           },
    { kAttrSize,              "size"             },
    { kAttrMime,              "mime"             },
    { kNumAttr,               NULL               }
};

// This list will be used to initialize a data structure for
// legal attribute lookup. The pattern is:
//
// tag-id1, id1-attr1, id1-attr2, ..., kNumAttr
// tag-id2, id2-attr1, id2-attr2, ..., kNumAttr
// ...
// kNumTags
// 
//
// kNumAttr is a marker for the end of the list of attribute ids.
//
const BYTE PXRealPixParser::m_pAttrList[] =
{
    kTagRoot,       kNumAttr,
    kTagHead,       kAttrWidth, kAttrHeight, kAttrBitrate, kAttrTimeformat,
                    kAttrVersion, kAttrTitle, kAttrAuthor, kAttrCopyright,
                    kAttrBackgroundcolor, kAttrStart, kAttrPreroll, kAttrAspect,
                    kAttrUrl, kAttrMaxfps, kAttrDuration, kAttrCenter, kNumAttr,
    kTagImage,      kAttrHandle, kAttrName, kAttrSize, kAttrMime, kNumAttr,
    kTagFill,       kAttrStart, kAttrDstx, kAttrDsty, kAttrDstw, kAttrDsth,
                    kAttrUrl, kAttrMaxfps, kAttrColor, kNumAttr,
    kTagFadein,     kAttrStart, kAttrDuration, kAttrTarget, kAttrSrcx,
                    kAttrSrcy, kAttrSrcw, kAttrSrch, kAttrDstx, kAttrDsty,
                    kAttrDstw, kAttrDsth, kAttrUrl, kAttrMaxfps,
                    kAttrAspect, kAttrCenter, kAttrBackgroundcolor, kNumAttr,
    kTagCrossfade,  kAttrStart, kAttrDuration, kAttrTarget, kAttrSrcx,
                    kAttrSrcy, kAttrSrcw, kAttrSrch, kAttrDstx, kAttrDsty,
                    kAttrDstw, kAttrDsth, kAttrUrl, kAttrMaxfps,
                    kAttrAspect, kAttrCenter, kAttrBackgroundcolor, kNumAttr,
    kTagFadeout,    kAttrStart, kAttrDuration, kAttrDstx, kAttrDsty,
                    kAttrDstw, kAttrDsth, kAttrUrl, kAttrMaxfps,
                    kAttrColor, kNumAttr,
    kTagWipe,       kAttrStart, kAttrDuration, kAttrTarget, kAttrSrcx,
                    kAttrSrcy, kAttrSrcw, kAttrSrch, kAttrDstx, kAttrDsty,
                    kAttrDstw, kAttrDsth, kAttrUrl, kAttrMaxfps, kAttrAspect,
                    kAttrDirection, kAttrType, kAttrCenter, kAttrBackgroundcolor,
                    kNumAttr,
    kTagAnimate,    kAttrStart, kAttrDuration, kAttrTarget, kAttrSrcx,
                    kAttrSrcy, kAttrSrcw, kAttrSrch, kAttrDstx, kAttrDsty,
                    kAttrDstw, kAttrDsth, kAttrUrl, kAttrMaxfps,
                    kAttrAspect, kAttrCenter, kAttrBackgroundcolor, kNumAttr,
    kTagViewchange, kAttrStart, kAttrDuration, kAttrSrcx, kAttrSrcy, kAttrSrcw,
                    kAttrSrch, kAttrDstx, kAttrDsty, kAttrDstw, kAttrDsth,
                    kAttrUrl, kAttrMaxfps, kAttrStartsrcx, kAttrStartsrcy,
                    kAttrStartsrcw, kAttrStartsrch, kAttrStartdstx, kAttrStartdsty,
                    kAttrStartdstw, kAttrStartdsth, kAttrTarget, kAttrAspect, kNumAttr,
    kTagEffect,     kAttrStart, kAttrDuration, kAttrTarget, kAttrSrcx,
                    kAttrSrcy, kAttrSrcw, kAttrSrch, kAttrDstx, kAttrDsty,
                    kAttrDstw, kAttrDsth, kAttrUrl, kAttrMaxfps, kAttrAspect,
                    kAttrPackage, kAttrName, kAttrData,
                    kAttrFile, kNumAttr,
    kNumTags
};

// This list will be used to initialize a data structure for
// required attribute lookup. The pattern is:
//
// tag-id1, id1-attr1, id1-attr2, ..., kNumAttr
// tag-id2, id2-attr1, id2-attr2, ..., kNumAttr
// ...
// kNumTags
// 
//
// kNumAttr is a marker for the end of the list of attribute ids.
//
const BYTE PXRealPixParser::m_pRequiredAttrList[] =
{
    kTagRoot,       kNumAttr,
    kTagHead,       kAttrWidth, kAttrHeight, kAttrBitrate, kNumAttr,
    kTagImage,      kAttrHandle, kAttrName, kNumAttr,
    kTagFill,       kAttrStart, kAttrColor, kNumAttr,
    kTagFadein,     kAttrStart, kAttrDuration, kAttrTarget, kNumAttr,
    kTagCrossfade,  kAttrStart, kAttrDuration, kAttrTarget, kNumAttr,
    kTagFadeout,    kAttrStart, kAttrDuration, kNumAttr,
    kTagWipe,       kAttrStart, kAttrDuration, kAttrTarget, kNumAttr,
    kTagAnimate,    kAttrStart, kAttrDuration, kAttrTarget, kNumAttr,
    kTagViewchange, kAttrStart, kAttrDuration, kNumAttr,
    kTagEffect,     kAttrStart, kAttrDuration, kAttrTarget, kAttrPackage, kAttrName, kNumAttr,
    kNumTags
};

BEGIN_INTERFACE_LIST(PXRealPixParser)
    INTERFACE_LIST_ENTRY(IID_IHXXMLParserResponse, IHXXMLParserResponse)
END_INTERFACE_LIST

PXRealPixParser::PXRealPixParser()
{
    m_pContext            = NULL;
    m_pCommonClassFactory = NULL;
    m_pParser             = NULL;
    m_pRealPixFile        = NULL;
    m_pErrorText          = NULL;
    m_pLegalAttrLUT       = NULL;
    m_pTagToIDMap         = NULL;
    m_pAttrToIDMap        = NULL;
    m_bRealPixError       = FALSE;
    m_pByteListCursor     = NULL;
    m_ulState             = kStateConstructed;
    m_pLastEffect         = NULL;
    m_bVersionSpecified   = FALSE;
    m_ulStrictnessLevel   = REALPIX_STRICTNESS_LOW;
}

PXRealPixParser::~PXRealPixParser()
{
    HX_RELEASE(m_pContext);
    HX_RELEASE(m_pCommonClassFactory);
    HX_RELEASE(m_pParser);
    HX_RELEASE(m_pRealPixFile);
    HX_RELEASE(m_pErrorText);
    HX_RELEASE(m_pLegalAttrLUT);
    HX_DELETE(m_pTagToIDMap);
    HX_DELETE(m_pAttrToIDMap);
    HX_RELEASE(m_pLastEffect);
}

STDMETHODIMP PXRealPixParser::Init(IUnknown*      pContext,
                                   PXRealPixFile* pRealPixFile,
                                   UINT32         ulStrictnessLevel)
{
    HX_RESULT retVal = HXR_OK;

    if (pContext && pRealPixFile)
    {
        // Init members
        HX_RELEASE(m_pContext);
        m_pContext = pContext;
        m_pContext->AddRef();
        HX_RELEASE(m_pRealPixFile);
        m_pRealPixFile = pRealPixFile;
        m_pRealPixFile->AddRef();
        if (ulStrictnessLevel == REALPIX_STRICTNESS_LOW    ||
            ulStrictnessLevel == REALPIX_STRICTNESS_MEDIUM ||
            ulStrictnessLevel == REALPIX_STRICTNESS_HIGH)
        {
            m_ulStrictnessLevel = ulStrictnessLevel;
        }
        else
        {
            m_ulStrictnessLevel = REALPIX_STRICTNESS_LOW;
        }

        // Get an IHXXMLParser interface
        HX_RELEASE(m_pCommonClassFactory);
        retVal = m_pContext->QueryInterface(IID_IHXCommonClassFactory,
                                            (void**) &m_pCommonClassFactory);
        if (SUCCEEDED(retVal))
        {
            BOOL bAllowBadComments = (m_ulStrictnessLevel >= REALPIX_STRICTNESS_MEDIUM ? FALSE : TRUE);
            HXXMLParser* pParser = NULL;
            pParser = new HXXMLParser(bAllowBadComments);
            if (pParser)
            {
                pParser->AddRef();
                HX_RELEASE(m_pParser);
                retVal = pParser->QueryInterface(IID_IHXXMLParser, (void**) &m_pParser);
                if (SUCCEEDED(retVal))
                {
                    BOOL bXMLStrictness = (m_ulStrictnessLevel >= REALPIX_STRICTNESS_MEDIUM ? TRUE : FALSE);
                    // Init the parser interface
                    retVal = m_pParser->Init(this,  // this class holds the response interface
                                             NULL,  // no special encoding
                                             bXMLStrictness); // strict XML parsing or not
                    if (SUCCEEDED(retVal))
                    {
                        retVal = SetupLegalAttrLUT();
                        if (SUCCEEDED(retVal))
                        {
                            retVal = SetupIDMaps();
                            if (SUCCEEDED(retVal))
                            {
                                // Clear the RealPix error flag
                                m_bRealPixError = FALSE;
                                // Set the state
                                m_ulState       = kStateInitialized;
                            }
                        }
                    }
                }
            }
            else
            {
                retVal = HXR_OUTOFMEMORY;
            }
            HX_RELEASE(pParser);
        }
    }
    else
    {
        retVal = HXR_INVALID_PARAMETER;
    }

    return retVal;
}

STDMETHODIMP PXRealPixParser::Parse(IHXBuffer* pFileBuffer, BOOL bIsFinal, REF(IHXBuffer*) rpErrorText)
{
    HX_RESULT retVal = HXR_OK;

    if (pFileBuffer)
    {
        if (m_pParser && m_pTagToIDMap &&
            m_pAttrToIDMap && m_pLegalAttrLUT)
        {
            // Parse the buffer
            retVal = m_pParser->Parse(pFileBuffer, bIsFinal);
            if (FAILED(retVal))
            {
                // If the m_bRealPixError flag is set, then this error
                // was a RealPix error as opposed to an XML error. If
                // it was a RealPix error, then m_pErrorText should already
                // contain an error string. If it was an XML error, then we
                // need to get the error string from the XML parser.
                HX_RELEASE(rpErrorText);
                if (m_bRealPixError && m_pErrorText)
                {
                    // We just need to assign m_pErrorText to the output buffer
                    rpErrorText = m_pErrorText;
                    rpErrorText->AddRef();
                }
                else
                {
                    // This was an error detected by the XML parser,
                    // so we need to get the error string from the parser
                    m_pParser->GetCurrentErrorText(rpErrorText);
                }
            }
            else
            {
                // Check to make sure we have some effects.
                if (m_pRealPixFile->GetNumEffects())
                {
                    // We parsed ok, so now we need to do a lot of post-parse initialization.
                    retVal = m_pRealPixFile->PostParseInit();
                }
                else
                {
                    HX_RELEASE(rpErrorText);
                    SetError(IDS_ERR_PIX_NOEFFECTS, 0, 0, NULL, NULL, rpErrorText);
                    retVal = HXR_FAIL;
                }
            }
//            if (bIsFinal)
//            {
//                m_pParser->Close();
//            }
        }
        else
        {
            retVal = HXR_UNEXPECTED;
        }
    }
    else
    {
        retVal = HXR_INVALID_PARAMETER;
    }

    return retVal;
}

STDMETHODIMP PXRealPixParser::HandleStartElement(const char* pName,
                                                 IHXValues* pAttributes,
                                                 UINT32      ulLineNumber,
                                                 UINT32      ulColumnNumber)
{
    HX_RESULT retVal = HXR_OK;

    // Check the input
    if (pName && pAttributes)
    {
        if (m_ulState >= kStateInitialized)
        {
            // First look up the tag ID
            UINT32 ulTagID = 0;
            retVal         = GetTagIDFromName(pName, ulTagID);
            if (SUCCEEDED(retVal))
            {
                // Loop through the attributes, checking to make sure each one
                // is a legal attribute of this tag
                const char* pszAttr = NULL;
                IHXBuffer* pValue  = NULL;
                HX_RESULT   rv      = pAttributes->GetFirstPropertyCString(pszAttr, pValue);
                while (SUCCEEDED(rv))
                {
                    if (!IsLegalAttr(ulTagID, pszAttr))
                    {
                        if (m_ulStrictnessLevel == REALPIX_STRICTNESS_HIGH)
                        {
                            retVal = HXR_FAIL;
                            break;
                        }
                        else if (m_ulStrictnessLevel == REALPIX_STRICTNESS_MEDIUM)
                        {
                            IHXBuffer* pErr = NULL;
                            SetError(IDS_ERR_PIX_ILLEGALATTR, ulLineNumber, ulColumnNumber, pszAttr, pName, pErr);
                            ReportError(HXLOG_WARNING, HXR_OK, pErr);
                            HX_RELEASE(pErr);
                        }
                    }
                    HX_RELEASE(pValue);
                    rv = pAttributes->GetNextPropertyCString(pszAttr, pValue);
                }
                HX_RELEASE(pValue);

                if (SUCCEEDED(retVal))
                {
                    // Now check to see if all the REQUIRED attributes for this
                    // tag are present.
                    const char* pszReqAttr = NULL;
                    rv                     = GetFirstRequiredAttribute(ulTagID, pszReqAttr);
                    while (SUCCEEDED(rv))
                    {
                        // Check if this attribute is present
                        IHXBuffer* pTmp = NULL;
                        retVal           = pAttributes->GetPropertyCString(pszReqAttr, pTmp);
                        HX_RELEASE(pTmp);
                        if (FAILED(retVal))
                        {
                            break;
                        }

                        // Get the next required attribute ID
                        rv = GetNextRequiredAttribute(ulTagID, pszReqAttr);
                    }

                    if (SUCCEEDED(retVal))
                    {
                        // All of the attributes were legal, and all the required
                        // attributes were present, so process the tag
                        retVal = ParseTag(ulTagID, pAttributes, ulLineNumber, ulColumnNumber, m_pErrorText);
                        if (FAILED(retVal))
                        {
                            m_bRealPixError = TRUE;
                        }
                    }
                    else
                    {
                        // We are missing a required attribute
                        m_bRealPixError = TRUE;
                        HX_RELEASE(m_pErrorText);
                        SetError(IDS_ERR_PIX_MISSREQATTR, ulLineNumber, ulColumnNumber,
                                 pszReqAttr, pName, m_pErrorText);
                    }
                }
                else
                {
                    // One of the attributes of this tag wasn't legal
                    m_bRealPixError = TRUE;
                    HX_RELEASE(m_pErrorText);
                    SetError(IDS_ERR_PIX_ILLEGALATTR, ulLineNumber, ulColumnNumber,
                             pszAttr, pName, m_pErrorText);
                }
            }
            else
            {
                // This is not a legal tag name
                m_bRealPixError = TRUE;
                HX_RELEASE(m_pErrorText);
                SetError(IDS_ERR_PIX_UNKNOWNTAG, ulLineNumber, ulColumnNumber,
                         pName, NULL, m_pErrorText);
            }
        }
        else
        {
            retVal = HXR_UNEXPECTED;
        }
    }
    else
    {
        retVal = HXR_INVALID_PARAMETER;
    }

    return retVal;
}

STDMETHODIMP PXRealPixParser::HandleEndElement(const char* pName,
                                               UINT32      ulLineNumber,
                                               UINT32      ulColumnNumber)
{
    HX_RESULT retVal = HXR_OK;

    if (pName)
    {
        // Get the tag ID
        UINT32 ulTagID = 0;
        retVal         = GetTagIDFromName(pName, ulTagID);
        if (SUCCEEDED(retVal))
        {
            if (ulTagID == kTagRoot)
            {
                if (m_ulState == kStateSawHead)
                {
                    m_ulState = kStateSawRootEnd;
                }
                else
                {
                    m_bRealPixError = TRUE;
                    retVal          = HXR_FAIL;
                    HX_RELEASE(m_pErrorText);
                    SetError(IDS_ERR_PIX_INVALIDHEAD, ulLineNumber, ulColumnNumber, NULL, NULL, m_pErrorText);
                }
            }
        }
        else
        {
            // This not a known tag
            m_bRealPixError = TRUE;
            HX_RELEASE(m_pErrorText);
            SetError(IDS_ERR_PIX_UNKNOWNTAG, ulLineNumber, ulColumnNumber,
                     pName, NULL, m_pErrorText);
        }
    }
    else
    {
        retVal = HXR_INVALID_PARAMETER;
    }

    return retVal;
}

STDMETHODIMP PXRealPixParser::HandleCharacterData(IHXBuffer* pBuffer,
                                                  UINT32      ulLineNumber,
                                                  UINT32      ulColumnNumber)
{
    HX_RESULT retVal = HXR_OK;

    return retVal;
}

STDMETHODIMP PXRealPixParser::HandleProcessingInstruction(const char* pTarget,
                                                          IHXValues* pAttributes,
                                                          UINT32      ulLineNumber,
                                                          UINT32      ulColumnNumber)
{
    HX_RESULT retVal = HXR_OK;

    return retVal;
}

STDMETHODIMP PXRealPixParser::HandleUnparsedEntityDecl(const char* pEntityName,
                                                       const char* pSystemID,
                                                       const char* pPublicID,
                                                       const char* pNotationName,
                                                       UINT32      ulLineNumber,
                                                       UINT32      ulColumnNumber)
{
    HX_RESULT retVal = HXR_OK;

    return retVal;
}

STDMETHODIMP PXRealPixParser::HandleNotationDecl(const char* pNotationName,
                                                 const char* pSystemID,
                                                 const char* pPublicID,
                                                 UINT32      ulLineNumber,
                                                 UINT32      ulColumNumber)
{
    HX_RESULT retVal = HXR_OK;

    return retVal;
}

STDMETHODIMP PXRealPixParser::HandleComment(const char* pComment,
                                            UINT32      ulLineNumber,
                                            UINT32      ulColumnNumber)
{
    HX_RESULT retVal = HXR_OK;

    return retVal;
}

STDMETHODIMP PXRealPixParser::HandleUnparsedDoctypeDecl(const char* pDoctype,
                                                        const char* pSystemID,
                                                        const char* pPublicID,
                                                        UINT32      ulLineNumber,
                                                        UINT32      ulColumnNumber)
{
    HX_RESULT retVal = HXR_OK;

    return retVal;
}

STDMETHODIMP PXRealPixParser::HandleDefault(IHXBuffer* pBuffer,
                                            UINT32      ulLineNumber,
                                            UINT32      ulColumnNumber)
{
    HX_RESULT retVal = HXR_OK;

    return retVal;
}

HX_RESULT PXRealPixParser::SetError(UINT32 ulErrorID, UINT32 ulLine, UINT32 ulCol,
                                    const char* pszArg1, const char* pszArg2,
                                    REF(IHXBuffer*) rpErrStr)
{
    HX_RESULT retVal = HXR_OK;

    const char* pszFileName = NULL;
    if (m_pRealPixFile)
    {
        pszFileName = m_pRealPixFile->GetFileName();
    }

    PXError cErr(m_pContext);
    if (pszFileName)
    {
        retVal = cErr.SetError(pszFileName, ulErrorID, ulLine, ulCol, pszArg1, pszArg2, rpErrStr);
    }
    else
    {
        retVal = cErr.SetError(ulErrorID, ulLine, ulCol, pszArg1, pszArg2, rpErrStr);
    }

    return retVal;
}

HX_RESULT PXRealPixParser::ParseTag(UINT32 ulTagID, IHXValues* pAttr, UINT32 ulLine,
                                    UINT32 ulCol, REF(IHXBuffer*) rpError)
{
    HX_RESULT retVal = HXR_OK;

    // Clear out any present error string
    HX_RELEASE(rpError);

    switch(ulTagID)
    {
        case kTagRoot:
            {
                if (m_ulState == kStateInitialized)
                {
                    m_ulState = kStateSawRootStart;
                }
                else
                {
                    SetError(IDS_ERR_PIX_ROOTNOTFIRST, ulLine, ulCol, NULL, NULL, rpError);
                    retVal = HXR_FAIL;
                }
            }
            break;
        case kTagHead:
            {
                if (m_ulState == kStateInitialized)
                {
                    SetError(IDS_ERR_PIX_ROOTNOTFIRST, ulLine, ulCol, NULL, NULL, rpError);
                    retVal = HXR_FAIL;
                }
                else if (m_ulState == kStateSawRootStart)
                {
                    // Process head here
                    retVal = ParseHeadTag(pAttr, ulLine, ulCol, rpError);
                    if (SUCCEEDED(retVal))
                    {
                        // Set the next state
                        m_ulState = kStateSawHead;
                    }
                }
                else
                {
                    SetError(IDS_ERR_PIX_HEADNOTFIRST, ulLine, ulCol, NULL, NULL, rpError);
                    retVal = HXR_FAIL;
                }
            }
            break;
        case kTagImage:
            {
                if (m_ulState == kStateInitialized)
                {
                    SetError(IDS_ERR_PIX_ROOTNOTFIRST, ulLine, ulCol, NULL, NULL, rpError);
                    retVal = HXR_FAIL;
                }
                else if (m_ulState == kStateSawRootStart)
                {
                    SetError(IDS_ERR_PIX_HEADNOTFIRST, ulLine, ulCol, NULL, NULL, rpError);
                    retVal = HXR_FAIL;
                }
                else if (m_ulState == kStateSawHead)
                {
                    retVal = ParseImageTag(pAttr, ulLine, ulCol, rpError);
                }
            }
            break;
        case kTagFill:
        case kTagFadein:
        case kTagCrossfade:
        case kTagFadeout:
        case kTagWipe:
        case kTagAnimate:
        case kTagViewchange:
        case kTagEffect:
            {
                if (m_ulState == kStateInitialized)
                {
                    SetError(IDS_ERR_PIX_ROOTNOTFIRST, ulLine, ulCol, NULL, NULL, rpError);
                    retVal = HXR_FAIL;
                }
                else if (m_ulState == kStateSawRootStart)
                {
                    SetError(IDS_ERR_PIX_HEADNOTFIRST, ulLine, ulCol, NULL, NULL, rpError);
                    retVal = HXR_FAIL;
                }
                else if (m_ulState == kStateSawHead)
                {
                    retVal = ParseEffectTag(ulTagID, pAttr, ulLine, ulCol, rpError);
                }
            }
            break;
    }

    return retVal;
}

HX_RESULT PXRealPixParser::ParseHeadTag(IHXValues* pAttr, UINT32 ulLine, UINT32 ulCol,
                                        REF(IHXBuffer*) rpError)
{
    HX_RESULT retVal = HXR_OK;

    if (pAttr)
    {
        // First we check the content version attribute
        const char* pszAttr = NULL;
        IHXBuffer* pValue  = NULL;
        GetAttributeNameFromID(kAttrVersion, pszAttr);
        HX_RESULT rv        = pAttr->GetPropertyCString(pszAttr, pValue);
        if (SUCCEEDED(rv))
        {
            UINT32 ulTmp = 0;
            retVal       = ConvertVersionValue(pValue, ulTmp);
            if (SUCCEEDED(retVal))
            {
                if (ulTmp <= m_ulHighestSupportedContentVersion)
                {
                    m_pRealPixFile->SetContentVersion(ulTmp);
                    m_bVersionSpecified = TRUE;
                }
                else
                {
                    SetError(IDS_ERR_PIX_FUTUREVERSION, ulLine, ulCol, (const char*) pValue->GetBuffer(), NULL, rpError);
                    retVal = HXR_FAIL;
                }
            }
            else
            {
                SetError(IDS_ERR_PIX_NULLVERSION, ulLine, ulCol, NULL, NULL, rpError);
            }
        }
        else
        {
            // Set the default to 0.0.0.0
            m_pRealPixFile->SetContentVersion(0);
            m_bVersionSpecified = FALSE;
        }
        HX_RELEASE(pValue);

        // Next we get the timeformat property in order to be able to
        // correctly interpret time values
        if (SUCCEEDED(retVal))
        {
            GetAttributeNameFromID(kAttrTimeformat, pszAttr);
            rv = pAttr->GetPropertyCString(pszAttr, pValue);
            if (SUCCEEDED(rv))
            {
                if (!strcmp((const char*) pValue->GetBuffer(), "dd:hh:mm:ss.xyz"))
                {
                    // Set timeformat to days/hours/minutes/seconds
                    m_pRealPixFile->SetTimeFormat(PXRealPixFile::kTimeFormatDHMS);
                }
                else if (!strcmp((const char*) pValue->GetBuffer(), "milliseconds"))
                {
                    // default is milliseconds
                    m_pRealPixFile->SetTimeFormat(PXRealPixFile::kTimeFormatMilliseconds);
                }
                else
                {
                    // Unknown time format
                    SetError(IDS_ERR_PIX_BADTIMEFORMAT, ulLine, ulCol, (const char*) pValue->GetBuffer(), NULL, rpError);
                    retVal = HXR_FAIL;
                }
            }
            else
            {
                // No timeformat specified, so set to default of milliseconds
                m_pRealPixFile->SetTimeFormat(PXRealPixFile::kTimeFormatMilliseconds);
            }
            HX_RELEASE(pValue);

            // Now loop through the attributes of the head tag
            const char* pszName = NULL;
            rv                  = pAttr->GetFirstPropertyCString(pszName, pValue);
            while (SUCCEEDED(rv) && SUCCEEDED(retVal))
            {
                // Get the attribute ID
                UINT32 ulAttrID = kNumAttr;
                GetAttributeIDFromName(pszName, ulAttrID);

                UINT32 ulMinVer = BASE_VERSION;
                switch (ulAttrID)
                {
                    case kAttrTitle:
                        {
                            retVal = CheckStringContents(pValue);
                            if (SUCCEEDED(retVal))
                            {
                                retVal = m_pRealPixFile->SetTitle(pValue);
                            }

                            if (FAILED(retVal))
                            {
                                SetError(IDS_ERR_PIX_NULLTITLE, ulLine, ulCol, NULL, NULL, rpError);
                            }
                        }
                        break;
                    case kAttrAuthor:
                        {
                            retVal = CheckStringContents(pValue);
                            if (SUCCEEDED(retVal))
                            {
                                retVal = m_pRealPixFile->SetAuthor(pValue);
                            }

                            if (FAILED(retVal))
                            {
                                SetError(IDS_ERR_PIX_NULLAUTHOR, ulLine, ulCol, NULL, NULL, rpError);
                            }
                        }
                        break;
                    case kAttrCopyright:
                        {
                            retVal = CheckStringContents(pValue);
                            if (SUCCEEDED(retVal))
                            {
                                retVal = m_pRealPixFile->SetCopyright(pValue);
                            }

                            if (FAILED(retVal))
                            {
                                SetError(IDS_ERR_PIX_NULLCOPYRIGHT, ulLine, ulCol, NULL, NULL, rpError);
                            }
                        }
                        break;
                    case kAttrStart:
                        {
                            UINT32 ulTmp = 0;
                            retVal       = ConvertTimeValue(pValue, m_pRealPixFile->GetTimeFormat(), ulTmp);
                            if (SUCCEEDED(retVal))
                            {
                                m_pRealPixFile->SetStart(ulTmp);
                            }
                            else
                            {
                                SetError(IDS_ERR_PIX_BADSTARTTIME, ulLine, ulCol, NULL, NULL, rpError);
                            }
                        }
                        break;
                    case kAttrDuration:
                        {
                            UINT32 ulTmp = 0;
                            retVal       = ConvertTimeValue(pValue, m_pRealPixFile->GetTimeFormat(), ulTmp);
                            if (SUCCEEDED(retVal))
                            {
                                m_pRealPixFile->SetDuration(ulTmp);
                            }
                            else
                            {
                                SetError(IDS_ERR_PIX_BADDURATION, ulLine, ulCol, NULL, NULL, rpError);
                            }
                        }
                        break;
                    case kAttrPreroll:
                        {
                            UINT32 ulTmp = 0;
                            retVal       = ConvertTimeValue(pValue, m_pRealPixFile->GetTimeFormat(), ulTmp);
                            if (SUCCEEDED(retVal))
                            {
                                m_pRealPixFile->SetPreroll(ulTmp);
                            }
                            else
                            {
                                SetError(IDS_ERR_PIX_BADPREROLL, ulLine, ulCol, NULL, NULL, rpError);
                            }
                        }
                        break;
                    case kAttrBitrate:
                        {
                            UINT32 ulTmp = strtoul((const char*) pValue->GetBuffer(), NULL, 10);
                            m_pRealPixFile->SetBitrate(ulTmp);
                        }
                        break;
                    case kAttrWidth:
                        {
                            UINT32 ulTmp = strtoul((const char*) pValue->GetBuffer(), NULL, 10);
                            m_pRealPixFile->SetDisplayWidth(ulTmp);
                        }
                        break;
                    case kAttrHeight:
                        {
                            UINT32 ulTmp = strtoul((const char*) pValue->GetBuffer(), NULL, 10);
                            m_pRealPixFile->SetDisplayHeight(ulTmp);
                        }
                        break;
                    case kAttrMaxfps:
                        {
                            UINT32 ulTmp = strtoul((const char*) pValue->GetBuffer(), NULL, 10);
                            m_pRealPixFile->SetDefaultMaxFps(ulTmp);
                        }
                        break;
                    case kAttrTimeformat:
                        break;
                    case kAttrAspect:
                        {
                            BOOL bTmp = FALSE;
                            retVal    = ConvertBoolValue(pValue, bTmp);
                            if (SUCCEEDED(retVal))
                            {
                                m_pRealPixFile->SetDefaultAspectFlag(bTmp);
                            }
                            else
                            {
                                SetError(IDS_ERR_PIX_BADASPECTFLAG, ulLine, ulCol, NULL, NULL, rpError);
                            }
                        }
                        break;
                    case kAttrCenter:
                        {
                            ulMinVer = REDSTONE_VERSION;
                            BOOL bTmp = FALSE;
                            retVal    = ConvertBoolValue(pValue, bTmp);
                            if (SUCCEEDED(retVal))
                            {
                                m_pRealPixFile->SetDefaultCenterFlag(bTmp);
                            }
                            else
                            {
                                SetError(IDS_ERR_PIX_BADCENTERFLAG, ulLine, ulCol, NULL, NULL, rpError);
                            }
                        }
                        break;
                    case kAttrUrl:
                        {
                            retVal = CheckStringContents(pValue);
                            if (SUCCEEDED(retVal))
                            {
                                retVal = m_pRealPixFile->SetDefaultURL(pValue);
                            }

                            if (FAILED(retVal))
                            {
                                SetError(IDS_ERR_PIX_NULLURL, ulLine, ulCol, NULL, NULL, rpError);
                            }
                        }
                        break;
                    case kAttrVersion:
                        break;
                    case kAttrBackgroundcolor:
                        {
                            ulMinVer     = U2_VERSION;
                            BYTE ucRed   = 0;
                            BYTE ucGreen = 0;
                            BYTE ucBlue  = 0;
                            retVal       = ConvertColorValue(pValue, ucRed, ucGreen, ucBlue);
                            if (SUCCEEDED(retVal))
                            {
                                m_pRealPixFile->SetBackgroundColor(ucRed, ucGreen, ucBlue);
                            }
                            else
                            {
                                SetError(IDS_ERR_PIX_BADBGCOLOR, ulLine, ulCol, NULL, NULL, rpError);
                            }
                        }
                        break;
                    case kAttrBackgroundOpacity:
                        {
                            // Set the minimum version
                            ulMinVer = OPACITY_VERSION;
                            // Parse the opacity
                            UINT32 ulOpacity = 255;
                            HXParseOpacity((const char*) pValue->GetBuffer(), ulOpacity);
                            m_pRealPixFile->SetBackgroundOpacity(ulOpacity);
                        }
                        break;
                }

                if (SUCCEEDED(retVal))
                {
                    retVal = CheckVersion(ulMinVer);
                    if (SUCCEEDED(retVal))
                    {
                        HX_RELEASE(pValue);
                        rv = pAttr->GetNextPropertyCString(pszName, pValue);
                    }
                    else
                    {
                        SetError(IDS_ERR_PIX_INCOMPATVERSION, ulLine, ulCol, (const char*) pValue->GetBuffer(), NULL, rpError);
                    }
                }
            }
            HX_RELEASE(pValue);
        }

        if (SUCCEEDED(retVal))
        {
            // Here we check for logical (parsed OK, but conflicting values)
            // kind of errors
        }
    }
    else
    {
        retVal = HXR_INVALID_PARAMETER;
    }

    return retVal;
}

HX_RESULT PXRealPixParser::ParseImageTag(IHXValues* pAttr, UINT32 ulLine, UINT32 ulCol,
                                         REF(IHXBuffer*) rpError)
{
    HX_RESULT retVal = HXR_OK;

    if (pAttr)
    {
        // Get the handle attribute
        const char* pszAttrStr = NULL;
        IHXBuffer* pValue     = NULL;
        GetAttributeNameFromID(kAttrHandle, pszAttrStr);
        retVal = pAttr->GetPropertyCString(pszAttrStr, pValue);
        if (SUCCEEDED(retVal))
        {
            UINT32 ulHandle = strtoul((const char*) pValue->GetBuffer(), NULL, 10);
            if (ulHandle)
            {
                // Get the name attribute
                GetAttributeNameFromID(kAttrName, pszAttrStr);
                HX_RELEASE(pValue);
                retVal = pAttr->GetPropertyCString(pszAttrStr, pValue);
                if (SUCCEEDED(retVal))
                {
                    retVal = CheckStringContents(pValue);
                    if (SUCCEEDED(retVal))
                    {
                        if (!m_pRealPixFile->IsImagePresent(ulHandle))
                        {
                            retVal = m_pRealPixFile->AddImage(ulHandle, pValue);
                            if (SUCCEEDED(retVal))
                            {
                                // Get the size attribute string
                                GetAttributeNameFromID(kAttrSize, pszAttrStr);
                                // See if we have a size attribute
                                HX_RELEASE(pValue);
                                HX_RESULT rv = pAttr->GetPropertyCString(pszAttrStr, pValue);
                                if (SUCCEEDED(rv))
                                {
                                    retVal = CheckVersion(REDSTONE_VERSION);
                                    if (SUCCEEDED(retVal))
                                    {
                                        UINT32 ulSize = strtoul((const char*) pValue->GetBuffer(), NULL, 10);
                                        if (ulSize)
                                        {
                                            retVal = m_pRealPixFile->SetImageSize(ulHandle, ulSize);
                                        }
                                        else
                                        {
                                            retVal = HXR_FAIL;
                                            SetError(IDS_ERR_PIX_ZEROSIZE, ulLine, ulCol, NULL, NULL, rpError);
                                        }
                                    }
                                    else
                                    {
                                        SetError(IDS_ERR_PIX_INCOMPATVERSION, ulLine, ulCol,
                                                 (const char*) pValue->GetBuffer(), NULL, rpError);
                                    }
                                }
                                // Get the mime type attribute string
                                GetAttributeNameFromID(kAttrMime, pszAttrStr);
                                // See if we have a mime type attribute
                                HX_RELEASE(pValue);
                                rv = pAttr->GetPropertyCString(pszAttrStr, pValue);
                                if (SUCCEEDED(rv))
                                {
                                    retVal = CheckVersion(REDSTONE_VERSION);
                                    if (SUCCEEDED(retVal))
                                    {
                                        retVal = m_pRealPixFile->SetImageFileMimeType(ulHandle, pValue);
                                    }
                                    else
                                    {
                                        SetError(IDS_ERR_PIX_INCOMPATVERSION, ulLine, ulCol,
                                                 (const char*) pValue->GetBuffer(), NULL, rpError);
                                    }
                                }
                            }
                        }
                        else
                        {
                            retVal = HXR_FAIL;
                            SetError(IDS_ERR_PIX_DUPHANDLE, ulLine, ulCol, NULL, NULL, rpError);
                        }
                    }
                    else
                    {
                        SetError(IDS_ERR_PIX_NULLNAME, ulLine, ulCol, NULL, NULL, rpError);
                    }
                }
                else
                {
                    SetError(IDS_ERR_PIX_NONAME, ulLine, ulCol, NULL, NULL, rpError);
                }
            }
            else
            {
                SetError(IDS_ERR_PIX_BADHANDLE, ulLine, ulCol, NULL, NULL, rpError);
                retVal = HXR_FAIL;
            }
        }
        else
        {
            SetError(IDS_ERR_PIX_NOHANDLE, ulLine, ulCol, NULL, NULL, rpError);
        }
        HX_RELEASE(pValue);
    }
    else
    {
        retVal = HXR_INVALID_PARAMETER;
    }

    return retVal;
}

HX_RESULT PXRealPixParser::ParseEffectTag(UINT32 ulTagID, IHXValues* pAttr, UINT32 ulLine,
                                          UINT32 ulCol, REF(IHXBuffer*) rpError)
{
    HX_RESULT retVal = HXR_OK;

    if (ulTagID < kNumTags && pAttr)
    {
        // Get the tag name
        const char* pszTagName = NULL;
        GetTagNameFromID(ulTagID, pszTagName);

        // Create a PXEffect object
        PXEffect* pEffect = NULL;
        retVal            = PXEffect::CreateObject(&pEffect);
        if (SUCCEEDED(retVal))
        {
            // AddRef the object
            pEffect->AddRef();

            // Set effect type
            BYTE ucEffectType = 0;
            switch (ulTagID)
            {
                case kTagFill:
                    ucEffectType = PXEffect::kEffectTypeFill;
                    break;
                case kTagFadein:
                    ucEffectType = PXEffect::kEffectTypeFadeIn;
                    break;
                case kTagFadeout:
                    ucEffectType = PXEffect::kEffectTypeFadeOut;
                    break;
                case kTagCrossfade:
                    ucEffectType = PXEffect::kEffectTypeCrossFade;
                    break;
                case kTagWipe:
                    ucEffectType = PXEffect::kEffectTypeWipe;
                    break;
                case kTagViewchange:
                    ucEffectType = PXEffect::kEffectTypeViewChange;
                    break;
                case kTagEffect:
                    ucEffectType = PXEffect::kEffectTypeExternal;
                    break;
                case kTagAnimate:
                    ucEffectType = PXEffect::kEffectTypeAnimate;
                    break;
            }
            pEffect->SetEffectType(ucEffectType);

            // Set defaults
            pEffect->SetAspectFlag(m_pRealPixFile->GetDefaultAspectFlag());
            pEffect->SetMaxFps(m_pRealPixFile->GetDefaultMaxFps());
            pEffect->SetCenterFlag(m_pRealPixFile->GetDefaultCenterFlag());

            // Loop through the attributes
            BOOL   bThrowError      = TRUE;
            BOOL   bNeedLastTarget  = TRUE;
            BOOL   bNeedLastRects   = TRUE;
            const char* pszAttrName = NULL;
            IHXBuffer* pAttrValue  = NULL;
            HX_RESULT   rv          = pAttr->GetFirstPropertyCString(pszAttrName, pAttrValue);
            while (SUCCEEDED(rv) && SUCCEEDED(retVal))
            {
                // Get the attribute ID. Don't need to check return value
                // since we have already check validity of each attribute
                UINT32 ulAttrID = 0;
                GetAttributeIDFromName(pszAttrName, ulAttrID);

                // Choose different action based on attribute ID
                UINT32 ulMinVer = BASE_VERSION;
                switch (ulAttrID)
                {
                    case kAttrStart:
                        {
                            UINT32 ulTmp = 0;
                            retVal       = ConvertTimeValue(pAttrValue, m_pRealPixFile->GetTimeFormat(), ulTmp);
                            if (SUCCEEDED(retVal))
                            {
                                pEffect->SetStart(ulTmp);
                            }
                        }
                        break;
                    case kAttrDuration:
                        {
                            UINT32 ulTmp = 0;
                            retVal       = ConvertTimeValue(pAttrValue, m_pRealPixFile->GetTimeFormat(), ulTmp);
                            if (SUCCEEDED(retVal))
                            {
                                pEffect->SetDuration(ulTmp);
                            }
                        }
                        break;
                    case kAttrTarget:
                        {
                            bNeedLastTarget = FALSE;
                            UINT32 ulTmp    = strtoul((const char*) pAttrValue->GetBuffer(), NULL, 10);
                            pEffect->SetTarget(ulTmp);
                        }
                        break;
                    case kAttrSrcx:
                        {
                            UINT32 ulTmp = strtoul((const char*) pAttrValue->GetBuffer(), NULL, 10);
                            pEffect->SetSrcX(ulTmp);
                        }
                        break;
                    case kAttrSrcy:
                        {
                            UINT32 ulTmp = strtoul((const char*) pAttrValue->GetBuffer(), NULL, 10);
                            pEffect->SetSrcY(ulTmp);
                        }
                        break;
                    case kAttrSrcw:
                        {
                            UINT32 ulTmp = strtoul((const char*) pAttrValue->GetBuffer(), NULL, 10);
                            pEffect->SetSrcWidth(ulTmp);
                        }
                        break;
                    case kAttrSrch:
                        {
                            UINT32 ulTmp = strtoul((const char*) pAttrValue->GetBuffer(), NULL, 10);
                            pEffect->SetSrcHeight(ulTmp);
                        }
                        break;
                    case kAttrStartsrcx:
                        {
                            UINT32 ulTmp = strtoul((const char*) pAttrValue->GetBuffer(), NULL, 10);
                            pEffect->SetStartSrcX(ulTmp);
                            bNeedLastRects = FALSE;
                            ulMinVer       = REDSTONE_VERSION;
                        }
                        break;
                    case kAttrStartsrcy:
                        {
                            UINT32 ulTmp = strtoul((const char*) pAttrValue->GetBuffer(), NULL, 10);
                            pEffect->SetStartSrcY(ulTmp);
                            bNeedLastRects = FALSE;
                            ulMinVer       = REDSTONE_VERSION;
                        }
                        break;
                    case kAttrStartsrcw:
                        {
                            UINT32 ulTmp = strtoul((const char*) pAttrValue->GetBuffer(), NULL, 10);
                            pEffect->SetStartSrcWidth(ulTmp);
                            bNeedLastRects = FALSE;
                            ulMinVer       = REDSTONE_VERSION;
                        }
                        break;
                    case kAttrStartsrch:
                        {
                            UINT32 ulTmp = strtoul((const char*) pAttrValue->GetBuffer(), NULL, 10);
                            pEffect->SetStartSrcHeight(ulTmp);
                            bNeedLastRects = FALSE;
                            ulMinVer       = REDSTONE_VERSION;
                        }
                        break;
                    case kAttrDstx:
                        {
                            UINT32 ulTmp = strtoul((const char*) pAttrValue->GetBuffer(), NULL, 10);
                            pEffect->SetDstX(ulTmp);
                        }
                        break;
                    case kAttrDsty:
                        {
                            UINT32 ulTmp = strtoul((const char*) pAttrValue->GetBuffer(), NULL, 10);
                            pEffect->SetDstY(ulTmp);
                        }
                        break;
                    case kAttrDstw:
                        {
                            UINT32 ulTmp = strtoul((const char*) pAttrValue->GetBuffer(), NULL, 10);
                            pEffect->SetDstWidth(ulTmp);
                        }
                        break;
                    case kAttrDsth:
                        {
                            UINT32 ulTmp = strtoul((const char*) pAttrValue->GetBuffer(), NULL, 10);
                            pEffect->SetDstHeight(ulTmp);
                        }
                        break;
                    case kAttrStartdstx:
                        {
                            UINT32 ulTmp = strtoul((const char*) pAttrValue->GetBuffer(), NULL, 10);
                            pEffect->SetStartDstX(ulTmp);
                            bNeedLastRects = FALSE;
                            ulMinVer       = REDSTONE_VERSION;
                        }
                        break;
                    case kAttrStartdsty:
                        {
                            UINT32 ulTmp = strtoul((const char*) pAttrValue->GetBuffer(), NULL, 10);
                            pEffect->SetStartDstY(ulTmp);
                            bNeedLastRects = FALSE;
                            ulMinVer       = REDSTONE_VERSION;
                        }
                        break;
                    case kAttrStartdstw:
                        {
                            UINT32 ulTmp = strtoul((const char*) pAttrValue->GetBuffer(), NULL, 10);
                            pEffect->SetStartDstWidth(ulTmp);
                            bNeedLastRects = FALSE;
                            ulMinVer       = REDSTONE_VERSION;
                        }
                        break;
                    case kAttrStartdsth:
                        {
                            UINT32 ulTmp = strtoul((const char*) pAttrValue->GetBuffer(), NULL, 10);
                            pEffect->SetDstHeight(ulTmp);
                            bNeedLastRects = FALSE;
                            ulMinVer       = REDSTONE_VERSION;
                        }
                        break;
                    case kAttrUrl:
                        {
                            retVal = CheckStringContents(pAttrValue);
                            if (SUCCEEDED(retVal))
                            {
                                pEffect->SetURL((const char*) pAttrValue->GetBuffer());
                            }
                            else
                            {
                                if (m_ulStrictnessLevel == REALPIX_STRICTNESS_LOW ||
                                    m_ulStrictnessLevel == REALPIX_STRICTNESS_MEDIUM)
                                {
                                    bThrowError = FALSE;
                                }
                            }
                        }
                        break;
                    case kAttrMaxfps:
                        {
                            UINT32 ulTmp = strtoul((const char*) pAttrValue->GetBuffer(), NULL, 10);
                            pEffect->SetMaxFps(ulTmp);
                        }
                        break;
                    case kAttrAspect:
                        {
                            BOOL bTmp = FALSE;
                            retVal    = ConvertBoolValue(pAttrValue, bTmp);
                            if (SUCCEEDED(retVal))
                            {
                                pEffect->SetAspectFlag(bTmp);
                            }
                        }
                        break;
                    case kAttrCenter:
                        {
                            BOOL bTmp = FALSE;
                            retVal    = ConvertBoolValue(pAttrValue, bTmp);
                            if (SUCCEEDED(retVal))
                            {
                                pEffect->SetCenterFlag(bTmp);
                                ulMinVer = REDSTONE_VERSION;
                            }
                        }
                        break;
                    case kAttrBackgroundcolor:
                        ulMinVer = REDSTONE_VERSION;
                    case kAttrColor:
                        {
                            BYTE ucRed   = 0;
                            BYTE ucGreen = 0;
                            BYTE ucBlue  = 0;
                            retVal       = ConvertColorValue(pAttrValue, ucRed, ucGreen, ucBlue);
                            if (SUCCEEDED(retVal))
                            {
                                pEffect->SetColor(ucRed, ucGreen, ucBlue);
                            }
                        }
                        break;
                    case kAttrDirection:
                        {
                            BYTE ucTmp = 0;
                            retVal     = ConvertWipeDirectionValue(pAttrValue, ucTmp);
                            if (SUCCEEDED(retVal))
                            {
                                pEffect->SetWipeDirection(ucTmp);
                            }
                        }
                        break;
                    case kAttrType:
                        {
                            BYTE ucTmp = 0;
                            retVal     = ConvertWipeTypeValue(pAttrValue, ucTmp);
                            if (SUCCEEDED(retVal))
                            {
                                pEffect->SetWipeType(ucTmp);
                            }
                        }
                        break;
                    case kAttrPackage:
                        {
                            retVal = CheckStringContents(pAttrValue);
                            if (SUCCEEDED(retVal))
                            {
                                pEffect->SetExFxPackage((const char*) pAttrValue->GetBuffer());
                            }
                        }
                        break;
                    case kAttrName:
                        {
                            retVal = CheckStringContents(pAttrValue);
                            if (SUCCEEDED(retVal))
                            {
                                pEffect->SetExFxName((const char*) pAttrValue->GetBuffer());
                            }
                        }
                        break;
                    case kAttrData:
                        {
                            retVal = CheckStringContents(pAttrValue);
                            if (SUCCEEDED(retVal))
                            {
                                pEffect->SetExFxData((const char*) pAttrValue->GetBuffer());
                            }
                        }
                        break;
                    case kAttrFile:
                        {
                            retVal = CheckStringContents(pAttrValue);
                            if (SUCCEEDED(retVal))
                            {
                                pEffect->SetExFxFile((const char*) pAttrValue->GetBuffer());
                            }
                        }
                        break;
                }

                if (SUCCEEDED(retVal) ||
                    (FAILED(retVal) && !bThrowError))
                {
                    if (FAILED(retVal))
                    {
                        if (m_ulStrictnessLevel >= REALPIX_STRICTNESS_MEDIUM)
                        {
                            // We detected bad syntax but were told not to throw an error,
                            // so we will log a warning instead.
                            IHXBuffer* pErr = NULL;
                            SetError(IDS_ERR_PIX_BADATTRVALUE, ulLine, ulCol, pszAttrName, pszTagName, pErr);
                            ReportError(HXLOG_WARNING, HXR_OK, pErr);
                            HX_RELEASE(pErr);
                        }
                    }

                    retVal = CheckVersion(ulMinVer);
                    if (SUCCEEDED(retVal))
                    {
                        // Get the next attribute
                        HX_RELEASE(pAttrValue);
                        if (SUCCEEDED(retVal))
                        {
                            rv = pAttr->GetNextPropertyCString(pszAttrName, pAttrValue);
                        }
                    }
                    else
                    {
                        SetError(IDS_ERR_PIX_INCOMPATVERSION, ulLine, ulCol, (const char*) pAttrValue->GetBuffer(), NULL, rpError);
                    }
                }
                else
                {
                    SetError(IDS_ERR_PIX_BADATTRVALUE, ulLine, ulCol, pszAttrName, pszTagName, rpError);
                }
            }
            HX_RELEASE(pAttrValue);

            if (SUCCEEDED(retVal))
            {
                // Here we do the hack to make sure that viewchange effects get the proper target
                if (pEffect->GetEffectType() == PXEffect::kEffectTypeViewChange &&
                    pEffect->GetTarget()     == 0                               &&
                    m_pLastEffect                                               &&
                    bNeedLastTarget)
                {
                    if (m_pLastEffect->HasTarget())
                    {
                        pEffect->SetTarget(m_pLastEffect->GetTarget());
                    }
                }
                // Here we do the hack to make sure that viewchange effects get the proper start rects
                if (pEffect->GetEffectType() == PXEffect::kEffectTypeViewChange &&
                    m_pLastEffect                                               &&
                    bNeedLastRects)
                {
                    if (m_pLastEffect->HasTarget())
                    {
                        pEffect->SetStartSrcRect(m_pLastEffect->GetSrcRect());
                    }
                    pEffect->SetStartDstRect(m_pLastEffect->GetDstRect());
                }
                // Part of the hack requires we save the last effect
                HX_RELEASE(m_pLastEffect);
                m_pLastEffect = pEffect;
                m_pLastEffect->AddRef();

                // Set the dst width and height if they defaulted to zero.
                if (!pEffect->GetDstWidth())
                {
                    pEffect->SetDstWidth(m_pRealPixFile->GetDisplayWidth());
                }
                if (!pEffect->GetDstHeight())
                {
                    pEffect->SetDstHeight(m_pRealPixFile->GetDisplayHeight());
                }
                if (!pEffect->GetStartDstWidth())
                {
                    pEffect->SetStartDstWidth(m_pRealPixFile->GetDisplayWidth());
                }
                if (!pEffect->GetStartDstHeight())
                {
                    pEffect->SetStartDstHeight(m_pRealPixFile->GetDisplayHeight());
                }

                // Do some error checking on dst rect
                if (pEffect->GetDstX()      + pEffect->GetDstWidth()       > m_pRealPixFile->GetDisplayWidth()  ||
                    pEffect->GetDstY()      + pEffect->GetDstHeight()      > m_pRealPixFile->GetDisplayHeight() ||
                    pEffect->GetStartDstX() + pEffect->GetStartDstWidth()  > m_pRealPixFile->GetDisplayWidth()  ||
                    pEffect->GetStartDstY() + pEffect->GetStartDstHeight() > m_pRealPixFile->GetDisplayHeight())
                {
                    if (m_ulStrictnessLevel == REALPIX_STRICTNESS_HIGH)
                    {
                        SetError(IDS_ERR_PIX_BADDSTRECT, ulLine, ulCol, NULL, NULL, rpError);
                        retVal = HXR_FAIL;
                    }
                    else if (m_ulStrictnessLevel == REALPIX_STRICTNESS_MEDIUM)
                    {
                        IHXBuffer* pErr = NULL;
                        SetError(IDS_ERR_PIX_BADDSTRECT, ulLine, ulCol, NULL, NULL, pErr);
                        ReportError(HXLOG_WARNING, HXR_OK, pErr);
                        HX_RELEASE(pErr);
                    }
                }

                if (SUCCEEDED(retVal))
                {
                    // Add the effect to the file
                    retVal = m_pRealPixFile->AddEffect(pEffect);
                }
            }
        }
        HX_RELEASE(pEffect);
    }
    else
    {
        retVal = HXR_INVALID_PARAMETER;
    }

    return retVal;
}

HX_RESULT PXRealPixParser::SetupLegalAttrLUT()
{
    HX_RESULT retVal = HXR_OK;

    // Create the LUT IHXBuffer
    HX_RELEASE(m_pLegalAttrLUT);
    retVal = m_pCommonClassFactory->CreateInstance(CLSID_IHXBuffer,
                                                   (void**) &m_pLegalAttrLUT);
    if (SUCCEEDED(retVal))
    {
        // Set the size
        retVal = m_pLegalAttrLUT->SetSize(kNumTags * kNumAttr);
        if (SUCCEEDED(retVal))
        {
            // Zero it out
            BYTE* pLUT = m_pLegalAttrLUT->GetBuffer();
            memset(pLUT, 0, m_pLegalAttrLUT->GetSize());

            // Now fill it in - we advance through the list
            // until we see kNumTags, which marks the end
            BYTE* pAttrList = (BYTE*) m_pAttrList;
            while (*pAttrList < kNumTags)
            {
                // Get the tag and advance to attribute list
                UINT32 ulTag = *pAttrList++;

                // Advance through the attribute list until
                // we see a kNumAttr, which marks the end
                // of the list
                while (*pAttrList < kNumAttr)
                {
                    // Get the attribute
                    UINT32 ulAttr = *pAttrList++;
                    // Set the value in the LUT to 1
                    pLUT[ulTag * kNumAttr + ulAttr] = 1;
                }

                // Advance to next tag
                pAttrList++;
            }
        }
    }

    return retVal;
}

HX_RESULT PXRealPixParser::SetupIDMaps()
{
    HX_RESULT retVal = HXR_OK;

    // Create the tag map object
    HX_DELETE(m_pTagToIDMap);
    m_pTagToIDMap = new CHXMapStringToOb();
    if (m_pTagToIDMap)
    {
        // Go through the tag string list
        PXStringTable* pStr = (PXStringTable*) m_pTagTable;
        while (pStr->m_ulStringID < kNumTags)
        {
            m_pTagToIDMap->SetAt(pStr->m_pszString, (void*) pStr->m_ulStringID);
            pStr++;
        }

        // Create the attribute map object
        HX_DELETE(m_pAttrToIDMap);
        m_pAttrToIDMap = new CHXMapStringToOb();
        if (m_pAttrToIDMap)
        {
            // Go through the attribute string list
            pStr = (PXStringTable*) m_pAttrTable;
            while (pStr->m_ulStringID < kNumAttr)
            {
                m_pAttrToIDMap->SetAt(pStr->m_pszString, (void*) pStr->m_ulStringID);
                pStr++;
            }
        }
        else
        {
            retVal = HXR_OUTOFMEMORY;
        }
    }
    else
    {
        retVal = HXR_OUTOFMEMORY;
    }

    return retVal;
}

BOOL PXRealPixParser::IsLegalAttr(UINT32 ulTagID, const char* pszAttr)
{
    BOOL bLegal = FALSE;

    if (m_pAttrToIDMap && m_pLegalAttrLUT &&
        pszAttr && ulTagID < kNumTags)
    {
        // Lookup attribute ID from string map. If it's
        // not in the string map, then it can't be legal
        void* pVoid    = NULL;
        BOOL  bPresent = m_pAttrToIDMap->Lookup(pszAttr, pVoid);
        if (bPresent)
        {
            // Get the ID
            UINT32 ulAttrID = (UINT32) pVoid;
            // See if there's a non-zero value in the legal attr LUT
            BYTE *pLUT = m_pLegalAttrLUT->GetBuffer();
            if (pLUT[ulTagID * kNumAttr + ulAttrID])
            {
                bLegal = TRUE;
            }
        }
    }

    return bLegal;
}

HX_RESULT PXRealPixParser::GetTagNameFromID(UINT32 ulTagID, REF(const char*) rpszTag)
{
    HX_RESULT retVal = HXR_OK;

    PXStringTable* pTable = (PXStringTable*) m_pTagTable;
    BOOL           bFound = FALSE;
    while (pTable->m_ulStringID < kNumTags)
    {
        if (pTable->m_ulStringID == ulTagID)
        {
            bFound = TRUE;
            break;
        }
        pTable++;
    }

    if (bFound)
    {
        rpszTag = pTable->m_pszString;
    }
    else
    {
        retVal = HXR_FAIL;
    }

    return retVal;
}

HX_RESULT PXRealPixParser::GetTagIDFromName(const char* pszTag, REF(UINT32) rulTagID)
{
    HX_RESULT retVal = HXR_FAIL;

    if (m_pTagToIDMap)
    {
        void* pVoid    = NULL;
        BOOL  bPresent = m_pTagToIDMap->Lookup(pszTag, pVoid);
        if (bPresent)
        {
            retVal   = HXR_OK;
            rulTagID = (UINT32) pVoid;
        }
    }

    return retVal;
}

HX_RESULT PXRealPixParser::GetAttributeNameFromID(UINT32 ulAttrID, REF(const char*) rpszAttr)
{
    HX_RESULT retVal = HXR_OK;

    PXStringTable* pTable = (PXStringTable*) m_pAttrTable;
    BOOL           bFound = FALSE;
    while (pTable->m_ulStringID < kNumAttr)
    {
        if (pTable->m_ulStringID == ulAttrID)
        {
            bFound = TRUE;
            break;
        }
        pTable++;
    }

    if (bFound)
    {
        rpszAttr = pTable->m_pszString;
    }
    else
    {
        retVal = HXR_FAIL;
    }

    return retVal;
}

HX_RESULT PXRealPixParser::GetAttributeIDFromName(const char* pszAttr, REF(UINT32) rulAttrID)
{
    HX_RESULT retVal = HXR_FAIL;

    if (m_pAttrToIDMap)
    {
        void* pVoid    = NULL;
        BOOL  bPresent = m_pAttrToIDMap->Lookup(pszAttr, pVoid);
        if (bPresent)
        {
            retVal    = HXR_OK;
            rulAttrID = (UINT32) pVoid;
        }
    }

    return retVal;
}

HX_RESULT PXRealPixParser::GetFirstRequiredAttribute(UINT32 ulTagID, REF(const char*) rpszAttr)
{
    HX_RESULT retVal = HXR_OK;

    // Make sure this is a valid ID
    if (ulTagID < kNumTags)
    {
        // First, find the tag ID in the list
        BYTE* pList = (BYTE*) m_pRequiredAttrList;
        while ((UINT32) *pList != ulTagID)
        {
            // Advance into the attribute list
            pList++;
            // Search for end of attribute list
            while (*pList < kNumAttr)
            {
                pList++;
            }
            // Advance to next tag
            pList++;
        }
        // Advance to first required attribute
        pList++;
        // If we are already at the end of the attribute
        // list, then there are no required attributes for
        // this tag, so we fail.
        if (*pList < kNumAttr)
        {
            retVal = GetAttributeNameFromID((UINT32) *pList, rpszAttr);
            if (SUCCEEDED(retVal))
            {
                // Set the state variable
                m_pByteListCursor = pList;
            }
        }
        else
        {
            retVal = HXR_FAIL;
        }
    }
    else
    {
        retVal = HXR_FAIL;
    }

    return retVal;
}

HX_RESULT PXRealPixParser::GetNextRequiredAttribute(UINT32 ulTagID, REF(const char*) rpszAttr)
{
    HX_RESULT retVal = HXR_OK;

    if (ulTagID < kNumTags && m_pByteListCursor)
    {
        // Advance to the next required attribute
        m_pByteListCursor++;

        // Check to see if we're at the end of the list
        if (*m_pByteListCursor < kNumAttr)
        {
            retVal = GetAttributeNameFromID((UINT32) *m_pByteListCursor, rpszAttr);
        }
        else
        {
            m_pByteListCursor = NULL;
            retVal            = HXR_FAIL;
        }
    }
    else
    {
        retVal = HXR_FAIL;
    }

    return retVal;
}

HX_RESULT PXRealPixParser::CheckStringContents(IHXBuffer* pString)
{
    HX_RESULT retVal = HXR_FAIL;

    if (pString)
    {
        const char* pszStr = (const char*) pString->GetBuffer();
        if (pszStr)
        {
            UINT32 ulLen = (UINT32) strlen(pszStr);
            if (ulLen > 0 && strspn(pszStr, " \r\n\t") < ulLen)
            {
                retVal = HXR_OK;
            }
        }
    }

    return retVal;
}

HX_RESULT PXRealPixParser::ConvertTimeValue(IHXBuffer* pValue, UINT32 ulTimeFormat, REF(UINT32) ulTime)
{
    HX_RESULT retVal = HXR_OK;

    if (ulTimeFormat == PXRealPixFile::kTimeFormatDHMS)
    {
        BOOL bRet = ConvertTimeStringToULONG32((char*) pValue->GetBuffer(),
                                               strlen((const char*) pValue->GetBuffer()),
                                               ulTime);
        if (!bRet)
        {
            retVal = HXR_FAIL;
        }
    }
    else
    {
        ulTime = strtoul((const char*) pValue->GetBuffer(), NULL, 10);
    }

    return retVal;
}

HX_RESULT PXRealPixParser::ConvertBoolValue(IHXBuffer* pValue, REF(BOOL) rbValue)
{
    HX_RESULT retVal = HXR_FAIL;

    if (pValue)
    {
        const char* pszValue = (const char*) pValue->GetBuffer();
        if (!strcmp(pszValue, "true"))
        {
            rbValue = TRUE;
            retVal  = HXR_OK;
        }
        else if (!strcmp(pszValue, "false"))
        {
            rbValue = FALSE;
            retVal  = HXR_OK;
        }
    }

    return retVal;
}

HX_RESULT PXRealPixParser::ConvertVersionValue(IHXBuffer* pValue, REF(UINT32) rulVersion)
{
    HX_RESULT retVal = HXR_FAIL;

    if (pValue)
    {
        const char* pszStr = (const char*) pValue->GetBuffer();
        if (pszStr)
        {
            char*  pszToken = strtok((char *) pszStr, ".");
            UINT32 ulDotNum = 0;
            UINT32 ulVer[4] = {0, 0, 0, 0};
            while(pszToken && ulDotNum < 4)
            {
                ulVer[ulDotNum++] = (UINT32) atol(pszToken);
                pszToken          = strtok(NULL, ".");
            }
            rulVersion = HX_ENCODE_PROD_VERSION(ulVer[0], ulVer[1], ulVer[2], ulVer[3]);
            retVal     = HXR_OK;
        }
    }

    return retVal;
}

HX_RESULT PXRealPixParser::ConvertColorValue(IHXBuffer* pValue, REF(BYTE) rucRed,
                                             REF(BYTE) rucGreen, REF(BYTE) rucBlue)
{
    HX_RESULT retVal = HXR_OK;

    if (pValue)
    {
        PXColor cColor;
        retVal = cColor.InitFromString((const char*) pValue->GetBuffer());
        if (SUCCEEDED(retVal))
        {
            rucRed   = cColor.GetRed();
            rucGreen = cColor.GetGreen();
            rucBlue  = cColor.GetBlue();
        }
    }
    else
    {
        retVal = HXR_FAIL;
    }

    return retVal;
}

HX_RESULT PXRealPixParser::ConvertWipeDirectionValue(IHXBuffer* pValue, REF(BYTE) rucDir)
{
    HX_RESULT retVal = HXR_FAIL;

    if (pValue)
    {
        const char* pszValue = (const char*) pValue->GetBuffer();
        if (pszValue)
        {
            if (!strcmp(pszValue, "up"))
            {
                rucDir = PXEffect::kWipeDirectionUp;
                retVal = HXR_OK;
            }
            else if (!strcmp(pszValue, "down"))
            {
                rucDir = PXEffect::kWipeDirectionDown;
                retVal = HXR_OK;
            }
            else if (!strcmp(pszValue, "left"))
            {
                rucDir = PXEffect::kWipeDirectionLeft;
                retVal = HXR_OK;
            }
            else if (!strcmp(pszValue, "right"))
            {
                rucDir = PXEffect::kWipeDirectionRight;
                retVal = HXR_OK;
            }
        }
    }

    return retVal;
}

HX_RESULT PXRealPixParser::ConvertWipeTypeValue(IHXBuffer* pValue, REF(BYTE) rucType)
{
    HX_RESULT retVal = HXR_FAIL;

    if (pValue)
    {
        const char* pszValue = (const char*) pValue->GetBuffer();
        if (pszValue)
        {
            if (!strcmp(pszValue, "normal"))
            {
                rucType = PXEffect::kWipeTypeNormal;
                retVal  = HXR_OK;
            }
            else if (!strcmp(pszValue, "push"))
            {
                rucType = PXEffect::kWipeTypePush;
                retVal  = HXR_OK;
            }
        }
    }

    return retVal;
}

HX_RESULT PXRealPixParser::CheckVersion(UINT32 ulMinVersion)
{
    HX_RESULT retVal = HXR_OK;

    // Was the version specified in the <head>?
    if (m_bVersionSpecified)
    {
        // The version WAS specified in the <head> tag, so 
        // the <head> version must be at least as big as ulMinVersion.
        if (m_pRealPixFile->GetContentVersion() < ulMinVersion)
        {
            retVal = HXR_FAIL;
        }
    }
    else
    {
        // The version was NOT specified in the <head> tag, so we
        // need to set the content version to at LEAST ulMinVersion.
        m_pRealPixFile->SetContentVersionToAtLeast(ulMinVersion);
    }

    return retVal;
}

void PXRealPixParser::ReportError(const UINT8 ucSeverity,
                                  HX_RESULT   lHXCode,
                                  IHXBuffer* pErrorStr)
{
    if (pErrorStr)
    {
        if (m_pContext)
        {
            IHXErrorMessages* pErrorMessages = NULL;
            m_pContext->QueryInterface(IID_IHXErrorMessages, (void**) &pErrorMessages);
            if (pErrorMessages)
            {
                pErrorMessages->Report(ucSeverity,                           // severity
                                       lHXCode,                             // RMA result (i.e. - HXR_FAIL, HXR_OUTOFMEMORY, etc.)
                                       0,                                    // user code (not used here)
                                       (const char*) pErrorStr->GetBuffer(), // string to be displayed
                                       NULL);                                // more info string (not used here)
            }
            HX_RELEASE(pErrorMessages);
        }
    }
}
