/* Implementation of the PreferencesWindow class.
 *
 * Copyright (c) 2003 Ole Laursen.
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA.
 */

#include <config.h>

#include <sigc++/bind.h>

#include <gtkmm/button.h>
#include <gconfmm/client.h>

#include "preferences-window.hpp"
#include "choose-monitor-window.hpp"
#include "gui-helpers.hpp"
#include "applet.hpp"
#include "monitor.hpp"
#include "i18n.hpp"



PreferencesWindow::PreferencesWindow(Applet &applet_)
  : applet(applet_)
{
  ui = get_glade_xml("preferences_window");

  ui->get_widget("preferences_window", window);
  window->set_type_hint(Gdk::WINDOW_TYPE_HINT_DIALOG);
  window->set_icon(applet.get_icon());
  

  // connect the General tab widgets
  ui->get_widget("update_interval_spinbutton", update_interval_spinbutton);
  update_interval_spinbutton->signal_value_changed()
    .connect(SigC::slot(*this, &PreferencesWindow::
			on_update_interval_spinbutton_changed));

  ui->get_widget("background_colorpicker", background_colorpicker);
  background_colorpicker->signal_color_set()
    .connect(SigC::slot(*this, &PreferencesWindow::
			on_background_colorpicker_set));

  ui->get_widget("panel_background_radiobutton", panel_background_radiobutton);
  ui->get_widget("background_color_radiobutton", background_color_radiobutton);
  background_color_radiobutton->signal_toggled()
    .connect(SigC::slot(*this, &PreferencesWindow::
			on_background_color_radiobutton_toggled));
  
  
  // connect the Viewer tab radiobuttons
  ui->get_widget("curve_radiobutton", curve_radiobutton);
  curve_radiobutton->signal_toggled()
    .connect(SigC::slot(*this, &PreferencesWindow::
			on_curve_radiobutton_toggled));
  
  ui->get_widget("bar_radiobutton", bar_radiobutton);
  bar_radiobutton->signal_toggled()
    .connect(SigC::slot(*this, &PreferencesWindow::
			on_bar_radiobutton_toggled));
  
  ui->get_widget("column_radiobutton", column_radiobutton);
  column_radiobutton->signal_toggled()
    .connect(SigC::slot(*this, &PreferencesWindow::
			on_column_radiobutton_toggled));
  
  ui->get_widget("text_radiobutton", text_radiobutton);
  text_radiobutton->signal_toggled()
    .connect(SigC::slot(*this, &PreferencesWindow::
			on_text_radiobutton_toggled));

  ui->get_widget("flame_radiobutton", flame_radiobutton);
  flame_radiobutton->signal_toggled()
    .connect(SigC::slot(*this, &PreferencesWindow::
			on_flame_radiobutton_toggled));
  
  ui->get_widget("curve_options", curve_options);

  ui->get_widget("curve_size_spinbutton", curve_size_spinbutton);
  curve_size_spinbutton->signal_value_changed()
    .connect(SigC::bind(SigC::slot(*this, &PreferencesWindow::
				   on_size_spinbutton_changed),
			curve_size_spinbutton));

  ui->get_widget("curve_samples_spinbutton", curve_samples_spinbutton);
  curve_samples_spinbutton->signal_value_changed()
    .connect(SigC::bind(SigC::slot(*this, &PreferencesWindow::
				   on_samples_spinbutton_changed),
			curve_samples_spinbutton));


  ui->get_widget("bar_options", bar_options);

  ui->get_widget("bar_size_spinbutton", bar_size_spinbutton);
  bar_size_spinbutton->signal_value_changed()
    .connect(SigC::bind(SigC::slot(*this, &PreferencesWindow::
				   on_size_spinbutton_changed),
			bar_size_spinbutton));


  ui->get_widget("column_options", column_options);

  ui->get_widget("column_size_spinbutton", column_size_spinbutton);
  column_size_spinbutton->signal_value_changed()
    .connect(SigC::bind(SigC::slot(*this, &PreferencesWindow::
				   on_size_spinbutton_changed),
			column_size_spinbutton));

  ui->get_widget("column_samples_spinbutton", column_samples_spinbutton);
  column_samples_spinbutton->signal_value_changed()
    .connect(SigC::bind(SigC::slot(*this, &PreferencesWindow::
				   on_samples_spinbutton_changed),
			column_samples_spinbutton));


  ui->get_widget("text_options", text_options);

  ui->get_widget("font_checkbutton", font_checkbutton);
  font_checkbutton->signal_toggled()
    .connect(SigC::slot(*this, &PreferencesWindow::
			on_font_checkbutton_toggled));

  ui->get_widget("fontpicker", fontpicker);
  fontpicker->signal_font_set()
    .connect(SigC::slot(*this, &PreferencesWindow::on_fontpicker_set));


  ui->get_widget("flame_options", flame_options);

  ui->get_widget("flame_size_spinbutton", flame_size_spinbutton);
  flame_size_spinbutton->signal_value_changed()
    .connect(SigC::bind(SigC::slot(*this, &PreferencesWindow::
				   on_size_spinbutton_changed),
			flame_size_spinbutton));

  
  // connect the Monitor tab widgets
  Gtk::Button *add_button;
  ui->get_widget("add_button", add_button);
  add_button->signal_clicked()
    .connect(SigC::slot(*this, &PreferencesWindow::on_add_button_clicked));

  ui->get_widget("remove_button", remove_button);
  remove_button->signal_clicked()
    .connect(SigC::slot(*this, &PreferencesWindow::on_remove_button_clicked));

  ui->get_widget("change_button", change_button);
  change_button->signal_clicked()
    .connect(SigC::slot(*this, &PreferencesWindow::on_change_button_clicked));

  ui->get_widget("monitor_treeview", monitor_treeview);
  monitor_treeview->get_selection()->signal_changed()
    .connect(SigC::slot(*this, &PreferencesWindow::on_selection_changed));

  ui->get_widget("monitor_options", monitor_options);

  
  static MonitorColumns mc;
  monitor_store = Gtk::ListStore::create(mc);
  monitor_treeview->set_model(monitor_store);
  monitor_treeview->append_column(_("Device"), mc.name);
  
  ui->get_widget("monitor_curve_options", monitor_curve_options);
  ui->get_widget("line_width_spinbutton", line_width_spinbutton);
  line_width_spinbutton->signal_value_changed()
    .connect(SigC::slot(*this, &PreferencesWindow::
			on_line_width_spinbutton_changed));
  ui->get_widget("line_colorpicker", line_colorpicker);
  line_colorpicker->signal_color_set()
    .connect(SigC::slot(*this, &PreferencesWindow::on_monitor_colorpicker_set));

  ui->get_widget("monitor_bar_options", monitor_bar_options);
  ui->get_widget("bar_colorpicker", bar_colorpicker);
  bar_colorpicker->signal_color_set()
    .connect(SigC::slot(*this, &PreferencesWindow::on_monitor_colorpicker_set));
  
  ui->get_widget("monitor_column_options", monitor_column_options);
  ui->get_widget("column_colorpicker", column_colorpicker);
  column_colorpicker->signal_color_set()
    .connect(SigC::slot(*this, &PreferencesWindow::on_monitor_colorpicker_set));

  ui->get_widget("monitor_flame_options", monitor_flame_options);
  ui->get_widget("flame_colorpicker", flame_colorpicker);
  flame_colorpicker->signal_color_set()
    .connect(SigC::slot(*this, &PreferencesWindow::on_monitor_colorpicker_set));
  
  
  // connect GConf
  Glib::RefPtr<Gnome::Conf::Client> &client = applet.get_gconf_client();
  Glib::ustring dir = applet.get_gconf_dir();
  
  client->notify_add(dir + "/viewer_type",
		     SigC::slot(*this, &PreferencesWindow::
				viewer_type_listener));

  client->notify_add(dir + "/update_interval",
		     SigC::slot(*this, &PreferencesWindow::
				update_interval_listener));

  client->notify_add(dir + "/background_interval",
		     SigC::slot(*this, &PreferencesWindow::
				background_color_listener));

  client->notify_add(dir + "/viewer/size",
		     SigC::slot(*this, &PreferencesWindow::size_listener));

  client->notify_add(dir + "/viewer/samples",
		     SigC::slot(*this, &PreferencesWindow::samples_listener));

  client->notify_add(dir + "/viewer/font",
		     SigC::slot(*this, &PreferencesWindow::font_listener));

  client->notify_add(dir + "/monitors",
		     SigC::slot(*this, &PreferencesWindow::monitors_listener));
  
  
  // fill in values
  viewer_type_listener(0, client->get_entry(dir + "/viewer_type"));
  update_interval_listener(0, client->get_entry(dir + "/update_interval"));
  background_color_listener(0, client->get_entry(dir + "/background_color"));
  use_background_color_listener(0, client
				->get_entry(dir + "/use_background_color"));
  size_listener(0, client->get_entry(dir + "/viewer/size"));
  samples_listener(0, client->get_entry(dir + "/viewer/samples"));
  font_listener(0, client->get_entry(dir + "/viewer/font"));

  Glib::SListHandle<Glib::ustring> list
    = applet.get_gconf_client()->all_dirs(applet.get_gconf_dir() + "/monitors");
  for (Glib::SListHandle<Glib::ustring>::const_iterator i = list.begin(),
	 end = list.end(); i != end; ++i)
    monitors_listener(0, client->get_entry(*i + "/type"));

  // this allows the user to discover the relationship between the grayed-out
  // buttons and the treeview
  monitor_treeview->get_selection()->unselect_all();

  // make sure background colorpicker is grayed out
  background_color_radiobutton->toggled();
  
  // connect close operations
  Gtk::Button *close_button;
  ui->get_widget("close_button", close_button);

  close_button->signal_clicked()
    .connect(SigC::slot(*this, &PreferencesWindow::on_close_button_clicked));

  window->signal_delete_event()
    .connect(SigC::slot(*this, &PreferencesWindow::on_closed));
}

PreferencesWindow::~PreferencesWindow()
{
  window->hide();
  stop_monitor_listeners();
}

void PreferencesWindow::show()
{
  window->show();
  window->raise();
}


// GConf callbacks

void PreferencesWindow::viewer_type_listener(unsigned int,
					     Gnome::Conf::Entry entry)
{
  if (entry.get_value().get_type() != Gnome::Conf::VALUE_STRING)
    return;

  Glib::ustring s = entry.get_value().get_string();
  
  if (s == "curve") {
    if (!curve_radiobutton->get_active())
      curve_radiobutton->property_active() = true;
    curve_options->property_visible() = true;
    monitor_curve_options->property_visible() = true;
  }
  else if (s == "bar") {
    if (!bar_radiobutton->get_active())
      bar_radiobutton->property_active() = true;
    bar_options->property_visible() = true;
    monitor_bar_options->property_visible() = true;
  }
  else if (s == "column") {
    if (!column_radiobutton->get_active())
      column_radiobutton->property_active() = true;
    column_options->property_visible() = true;
    monitor_column_options->property_visible() = true;
  }
  else if (s == "text") {
    if (!text_radiobutton->get_active())
      text_radiobutton->property_active() = true;
    text_options->property_visible() = true;
  }
  else if (s == "flame") {
    if (!flame_radiobutton->get_active())
      flame_radiobutton->property_active() = true;
    flame_options->property_visible() = true;
    monitor_flame_options->property_visible() = true;
  }
}

void PreferencesWindow::update_interval_listener(unsigned int,
						 Gnome::Conf::Entry entry)
{
  if (entry.get_value().get_type() != Gnome::Conf::VALUE_FLOAT)
    return;

  double d = entry.get_value().get_float();
  
  if (update_interval_spinbutton->get_value() != d)
    update_interval_spinbutton->set_value(d);
}

void PreferencesWindow::background_color_listener(unsigned int,
						  Gnome::Conf::Entry entry)
{
  if (entry.get_value().get_type() != Gnome::Conf::VALUE_INT)
    return;

  unsigned int i = entry.get_value().get_int();
  
  unsigned char r = i >> 24, g = i >> 16, b = i >> 8, a = i;
  unsigned char pr, pg, pb, pa;

  background_colorpicker->get(pr, pg, pb, pa);
  if (pr != r || pg != g || pb != b || pa != a)
    background_colorpicker->set(r, g, b, a);
}

void PreferencesWindow::use_background_color_listener(unsigned int,
						      Gnome::Conf::Entry entry)
{
  if (entry.get_value().get_type() != Gnome::Conf::VALUE_BOOL)
    return;

  bool b = entry.get_value().get_bool();

  if (b)
    background_color_radiobutton->set_active();
  else
    panel_background_radiobutton->set_active();
}

void PreferencesWindow::size_listener(unsigned int,
				      Gnome::Conf::Entry entry)
{
  if (entry.get_value().get_type() != Gnome::Conf::VALUE_INT)
    return;

  int i = entry.get_value().get_int();
  
  if (curve_size_spinbutton->get_value() != i)
    curve_size_spinbutton->set_value(i);

  if (bar_size_spinbutton->get_value() != i)
    bar_size_spinbutton->set_value(i);

  if (column_size_spinbutton->get_value() != i)
    column_size_spinbutton->set_value(i);

  if (flame_size_spinbutton->get_value() != i)
    flame_size_spinbutton->set_value(i);
}

void PreferencesWindow::samples_listener(unsigned int,
					 Gnome::Conf::Entry entry)
{
  if (entry.get_value().get_type() != Gnome::Conf::VALUE_INT)
    return;

  int i = entry.get_value().get_int();
  
  if (curve_samples_spinbutton->get_value() != i)
    curve_samples_spinbutton->set_value(i);

  if (column_samples_spinbutton->get_value() != i)
    column_samples_spinbutton->set_value(i);
}

void PreferencesWindow::font_listener(unsigned int, Gnome::Conf::Entry entry)
{
  if (entry.get_value().get_type() != Gnome::Conf::VALUE_STRING)
    return;

  Glib::ustring font = entry.get_value().get_string();

  if (font.empty())
    font_checkbutton->set_active(false);
  else {
    font_checkbutton->set_active(true);
    if (fontpicker->get_font_name() != font)
      fontpicker->set_font_name(font);
  }
}

void PreferencesWindow::monitors_listener(unsigned int,
					  Gnome::Conf::Entry entry)
{
  static MonitorColumns mc;

  Glib::ustring key = entry.get_key();
  if (key.substr(key.size() - 5) != "/type")
    return;

  Glib::ustring mdir = key.substr(0, key.size() - 5);

  store_iter
    pos = monitor_store->children().begin(),
    end = monitor_store->children().end();
  while (pos != end && (*pos)[mc.dir] != mdir)
    ++pos;

  bool exists = pos != monitor_store->children().end();

  // value was added
  if (entry.get_value().get_type() == Gnome::Conf::VALUE_STRING) {
    if (!exists) {
      // add value
      store_iter i = monitor_store->append();
      (*i)[mc.name] = applet.look_up_monitor(mdir)->get_name();
      (*i)[mc.dir] = mdir;
      
      monitor_treeview->get_selection()->select(i);
    }
  }
  else				// value was removed
    if (exists)
      monitor_store->erase(pos);
}

void PreferencesWindow::line_width_listener(unsigned int,
					    Gnome::Conf::Entry entry)
{
  if (entry.get_value().get_type() != Gnome::Conf::VALUE_FLOAT)
    return;

  double d = entry.get_value().get_float();
  
  if (line_width_spinbutton->get_value() != d)
    line_width_spinbutton->set_value(d);
}

void PreferencesWindow::monitor_color_listener(unsigned int,
					       Gnome::Conf::Entry entry)
{
  if (entry.get_value().get_type() != Gnome::Conf::VALUE_INT)
    return;

  unsigned int i = entry.get_value().get_int();

  unsigned char r = i >> 24, g = i >> 16, b = i >> 8, a = i;
  unsigned char pr, pg, pb, pa;

  line_colorpicker->get(pr, pg, pb, pa);
  if (pr != r || pg != g || pb != b || pa != a)
    line_colorpicker->set(r, g, b, a);
  
  bar_colorpicker->get(pr, pg, pb, pa);
  if (pr != r || pg != g || pb != b || pa != a)
    bar_colorpicker->set(r, g, b, a);

  column_colorpicker->get(pr, pg, pb, pa);
  if (pr != r || pg != g || pb != b || pa != a)
    column_colorpicker->set(r, g, b, a);
  
  flame_colorpicker->get(pr, pg, pb, pa);
  if (pr != r || pg != g || pb != b || pa != a)
    flame_colorpicker->set(r, g, b, a);
}




// UI callbacks

void PreferencesWindow::on_update_interval_spinbutton_changed()
{
  Glib::RefPtr<Gnome::Conf::Client> &client = applet.get_gconf_client();
  Glib::ustring dir = applet.get_gconf_dir();
  
  double d = update_interval_spinbutton->get_value();
  client->set(dir + "/update_interval", d);
}

unsigned int to_int(unsigned int r, unsigned int g, unsigned int b,
		    unsigned int a)
{
  return ((r & 255) << 24) | ((g & 255) << 16) | ((b & 255) << 8) | (a & 255);
}

void PreferencesWindow::on_background_colorpicker_set(unsigned int r,
						      unsigned int g,
						      unsigned int b,
						      unsigned int a)
{
  Glib::RefPtr<Gnome::Conf::Client> &client = applet.get_gconf_client();
  Glib::ustring dir = applet.get_gconf_dir();

  client->set(dir + "/background_color", int(to_int(r / 256, g / 256,
						    b / 256, a / 256)));
}

void PreferencesWindow::on_background_color_radiobutton_toggled()
{
  Glib::RefPtr<Gnome::Conf::Client> &client = applet.get_gconf_client();
  Glib::ustring dir = applet.get_gconf_dir();

  bool on = background_color_radiobutton->get_active();
  
  background_colorpicker->set_sensitive(on);
  
  client->set(dir + "/use_background_color", on);
}

void PreferencesWindow::on_curve_radiobutton_toggled()
{
  Glib::RefPtr<Gnome::Conf::Client> &client = applet.get_gconf_client();
  Glib::ustring dir = applet.get_gconf_dir();

  bool active = curve_radiobutton->get_active();
  
  if (active)
    client->set(dir + "/viewer_type", Glib::ustring("curve"));

  curve_options->property_visible() = active;
  monitor_curve_options->property_visible() = active;
}

void PreferencesWindow::on_bar_radiobutton_toggled()
{
  Glib::RefPtr<Gnome::Conf::Client> &client = applet.get_gconf_client();
  Glib::ustring dir = applet.get_gconf_dir();

  bool active = bar_radiobutton->get_active();
  
  if (active)
    client->set(dir + "/viewer_type", Glib::ustring("bar"));

  bar_options->property_visible() = active;
  monitor_bar_options->property_visible() = active;
}

void PreferencesWindow::on_column_radiobutton_toggled()
{
  Glib::RefPtr<Gnome::Conf::Client> &client = applet.get_gconf_client();
  Glib::ustring dir = applet.get_gconf_dir();

  bool active = column_radiobutton->get_active();
  
  if (active)
    client->set(dir + "/viewer_type", Glib::ustring("column"));

  column_options->property_visible() = active;
  monitor_column_options->property_visible() = active;
}

void PreferencesWindow::on_text_radiobutton_toggled()
{
  Glib::RefPtr<Gnome::Conf::Client> &client = applet.get_gconf_client();
  Glib::ustring dir = applet.get_gconf_dir();

  bool active = text_radiobutton->get_active();
  
  if (active)
    client->set(dir + "/viewer_type", Glib::ustring("text"));

  text_options->property_visible() = active;
}

void PreferencesWindow::on_flame_radiobutton_toggled()
{
  Glib::RefPtr<Gnome::Conf::Client> &client = applet.get_gconf_client();
  Glib::ustring dir = applet.get_gconf_dir();

  bool active = flame_radiobutton->get_active();
  
  if (active)
    client->set(dir + "/viewer_type", Glib::ustring("flame"));

  flame_options->property_visible() = active;
  monitor_flame_options->property_visible() = active;
}

void PreferencesWindow::on_size_spinbutton_changed(Gtk::SpinButton *size_sb)
{
  Glib::RefPtr<Gnome::Conf::Client> &client = applet.get_gconf_client();
  Glib::ustring dir = applet.get_gconf_dir();

  int i = int(size_sb->get_value());
  client->set(dir + "/viewer/size", i);
}

void PreferencesWindow::on_samples_spinbutton_changed(Gtk::SpinButton *samples_sb)
{
  Glib::RefPtr<Gnome::Conf::Client> &client = applet.get_gconf_client();
  Glib::ustring dir = applet.get_gconf_dir();

  int i = int(samples_sb->get_value());
  client->set(dir + "/viewer/samples", i);
}

void PreferencesWindow::on_font_checkbutton_toggled()
{
  Glib::RefPtr<Gnome::Conf::Client> &client = applet.get_gconf_client();
  Glib::ustring dir = applet.get_gconf_dir();

  bool active = font_checkbutton->get_active();
  
  fontpicker->set_sensitive(active);

  if (active)
    client->set(dir + "/viewer/font", fontpicker->get_font_name());
  else
    client->set(dir + "/viewer/font", Glib::ustring(""));
}

void PreferencesWindow::on_fontpicker_set(const Glib::ustring &font)
{
  Glib::RefPtr<Gnome::Conf::Client> &client = applet.get_gconf_client();
  Glib::ustring dir = applet.get_gconf_dir();
  
  client->set(dir + "/viewer/font", font);
}

void PreferencesWindow::on_add_button_clicked()
{
  ChooseMonitorWindow window;
  window.set_icon(applet.get_icon());

  Monitor *monitor = window.run(applet.get_gconf_client(), Glib::ustring());

  if (monitor)
    applet.add_monitor(monitor);
}

void PreferencesWindow::on_remove_button_clicked()
{
  static MonitorColumns mc;
  
  store_iter i = monitor_treeview->get_selection()->get_selected();
  
  if (i)
    applet.remove_monitor((*i)[mc.dir]);
}

void PreferencesWindow::on_change_button_clicked()
{
  static MonitorColumns mc;
  
  store_iter i = monitor_treeview->get_selection()->get_selected();
  
  if (i) {
    Glib::ustring mon_dir = (*i)[mc.dir];
    
    ChooseMonitorWindow window;
    window.set_icon(applet.get_icon());

    Monitor *monitor = window.run(applet.get_gconf_client(), mon_dir);

    if (monitor) {
      applet.replace_monitor(mon_dir, monitor);

      (*i)[mc.name] = monitor->get_name();
    }
  }
}

void PreferencesWindow::stop_monitor_listeners()
{
  Glib::RefPtr<Gnome::Conf::Client> &client = applet.get_gconf_client();
  
  for (std::vector<unsigned int>::iterator i = monitor_listeners.begin(),
	 end = monitor_listeners.end(); i != end; ++i)
    client->notify_remove(*i);

  monitor_listeners.clear();
}

void PreferencesWindow::on_selection_changed()
{
  Glib::RefPtr<Gnome::Conf::Client> &client = applet.get_gconf_client();

  static MonitorColumns mc;
  store_iter i = monitor_treeview->get_selection()->get_selected();

  bool sel = i;

  stop_monitor_listeners();

  if (sel) {
    Glib::ustring key;
    unsigned int con;

    key = (*i)[mc.dir] + "/line_width";
    con = client->notify_add(key, SigC::slot(*this, &PreferencesWindow::
					     line_width_listener));

    line_width_listener(0, client->get_entry(key));

    monitor_listeners.push_back(con);

    key = (*i)[mc.dir] + "/color";
    con = client->notify_add(key, SigC::slot(*this, &PreferencesWindow::
					     monitor_color_listener));

    monitor_color_listener(0, client->get_entry(key));

    monitor_listeners.push_back(con);
  }
  
  remove_button->set_sensitive(sel);
  change_button->set_sensitive(sel);
  monitor_options->set_sensitive(sel);
}

void PreferencesWindow::on_line_width_spinbutton_changed()
{
  static MonitorColumns mc;
  
  Glib::RefPtr<Gnome::Conf::Client> &client = applet.get_gconf_client();

  store_iter i = monitor_treeview->get_selection()->get_selected();
  
  if (i) {
    Glib::ustring mon_dir = (*i)[mc.dir];

    double d = line_width_spinbutton->get_value();
    client->set(mon_dir + "/line_width", d);
  }
}

void PreferencesWindow::on_monitor_colorpicker_set(unsigned int r,
						   unsigned int g,
						   unsigned int b,
						   unsigned int a)
{
  static MonitorColumns mc;
  
  Glib::RefPtr<Gnome::Conf::Client> &client = applet.get_gconf_client();
  
  store_iter i = monitor_treeview->get_selection()->get_selected();
  
  if (i) {
    Glib::ustring mon_dir = (*i)[mc.dir];
    client->set(mon_dir + "/color", int(to_int(r / 256, g / 256,
					       b / 256, a / 256)));
  }
}

void PreferencesWindow::on_close_button_clicked()
{
  window->hide();
}

bool PreferencesWindow::on_closed(GdkEventAny *)
{
  window->hide();
  return false;
}
