/* 
 * File: gkrellxmms2.c
 *
 * GKrellM 2 Plugin to control XMMS2
 * Copyright (c) 2005-2006 Johannes Heimansberg
 *
 * requires GKrellM 2 and XMMS2
 *
 * Version: 0.7.0
 *
 * Released under the GNU General Public License v2
 */

#define VERSION_NUMBER "0.7.0"

#if !defined(WIN32)
#include <gkrellm2/gkrellm.h>
#else
#include <src/gkrellm.h>
#include <src/win32-plugin.h>
#endif

#include <xmmsclient/xmmsclient-glib.h>
#include "xmms2ctrl/xmms2ctrl.h"
#include "gtrackinfo/trackinfo.h"
#include "gtrackinfo/gtrackinfo.h"
#include "gmedialib/gmedialib.h"
#include "buttons.h"
#include "playlist.xpm"

#define PLUGIN_NAME "gkrellxmms2"
#define SCROLL_TEXT_LENGTH 1200

static GkrellmMonitor *monitor;
static GkrellmPanel   *panel;
static GkrellmDecal   *text1_decal;
static GkrellmKrell   *krell;
static ControlButtons  cb;
static GtkTooltips    *tooltip = NULL;
static gchar           scroll_text[SCROLL_TEXT_LENGTH] = "XMMS2 Control  ";
static gint            style_id;
static gboolean        reset_scroller = TRUE;
static gboolean        seek_in_progress = FALSE;
static GtkWidget      *gml = NULL;
xmmsc_connection_t    *connection;
static trackinfo       track;
static gboolean        local_xmms2d = FALSE;

#if defined(GKRELLM_HAVE_DECAL_SCROLL_TEXT)
static gboolean       scroll_loop_mode;
#endif

static int xmms2_connect(void);

static struct Configuration {
	gint     scroll_speed;
	gint     medialib_window_width,            medialib_window_height;
	gint     medialib_window_pos_x,            medialib_window_pos_y;
	gboolean medialib_search_column_id,        medialib_search_column_artist;
	gboolean medialib_search_column_track,     medialib_search_column_album;
	gboolean medialib_search_column_rating;
	gboolean medialib_playlist_column_id,      medialib_playlist_column_artist;
	gboolean medialib_playlist_column_track,   medialib_playlist_column_album;
	gboolean medialib_playlist_column_rating,  medialib_playlist_column_pos;
	gboolean medialib_playlist_new_confirm;
	gboolean auto_reconnect;
	gchar    ipc_path[256];
} config;

static void update_krell(void)
{
	gchar scroll_text_old[SCROLL_TEXT_LENGTH] = "";

	if (trackinfo_has_playtime(&track) && trackinfo_get_playtime(&track) > 0) {
		gkrellm_set_krell_full_scale(krell, 
		                             trackinfo_get_playtime(&track) / 1000, 1);
	} else {
		gkrellm_set_krell_full_scale(krell, 1, 1);
	}

	g_strlcpy(scroll_text_old, scroll_text, sizeof(scroll_text_old));
	trackinfo_get_full_title(scroll_text, SCROLL_TEXT_LENGTH, &track, TRUE);
	if (strcmp(scroll_text, scroll_text_old) != 0) reset_scroller = TRUE;

	gtk_tooltips_set_tip(tooltip, panel->drawing_area, scroll_text, "empty");
}

static void cb_menu_trackinfo(GtkWidget *widget, gpointer pt)
{
	GtkWidget *dialog = gtrackinfo_new();
	gtrackinfo_set_info(GTRACKINFO(dialog), &track);
	gtk_widget_show(dialog);
}

static void cb_menu_playlist(GtkWidget *widget, gpointer pt)
{
	if (!GTK_IS_WIDGET(gml)) {
		gml = gmedialib_new(TRUE, TRUE, local_xmms2d, TRUE, TRUE);
		gml_pl_set_new_playlist_confirm(GMEDIALIB(gml), config.medialib_playlist_new_confirm);
		gtk_window_resize(GTK_WINDOW(gml), config.medialib_window_width, config.medialib_window_height);
		if (gml_notebook_is_mlib_search_visible(GMEDIALIB(gml))) {
			gml_search_set_column_id_visible(GMEDIALIB(gml),      config.medialib_search_column_id);
			gml_search_set_column_artist_visible(GMEDIALIB(gml),  config.medialib_search_column_artist);
			gml_search_set_column_track_visible(GMEDIALIB(gml),   config.medialib_search_column_track);
			gml_search_set_column_album_visible(GMEDIALIB(gml),   config.medialib_search_column_album);
			gml_search_set_column_rating_visible(GMEDIALIB(gml),  config.medialib_search_column_rating);
		}
		if (gml_notebook_is_playlist_visible(GMEDIALIB(gml))) {
			gml_pl_set_column_id_visible(GMEDIALIB(gml),          config.medialib_playlist_column_id);
			gml_pl_set_column_artist_visible(GMEDIALIB(gml),      config.medialib_playlist_column_artist);
			gml_pl_set_column_track_visible(GMEDIALIB(gml),       config.medialib_playlist_column_track);
			gml_pl_set_column_album_visible(GMEDIALIB(gml),       config.medialib_playlist_column_album);
			gml_pl_set_column_rating_visible(GMEDIALIB(gml),      config.medialib_playlist_column_rating);
			gml_pl_set_column_pos_visible(GMEDIALIB(gml),         config.medialib_playlist_column_pos);
		}
	}
	gtk_window_present(GTK_WINDOW(gml));
}

static void open_menu()
{
	GtkWidget *menu, *menuitem, *image;

	menu = gtk_menu_new();

	menuitem = gtk_image_menu_item_new_from_stock(GTK_STOCK_DIALOG_INFO, NULL);

	g_signal_connect(menuitem, "activate",
	                 G_CALLBACK(cb_menu_trackinfo), NULL);

	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	menuitem = gtk_separator_menu_item_new();
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	menuitem = gtk_image_menu_item_new_with_mnemonic("_Open Playlist Editor");
	image = gtk_image_new_from_pixbuf(gdk_pixbuf_new_from_xpm_data((const char **)playlist_xpm));
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem), image);
	g_signal_connect(menuitem, "activate",
	                 G_CALLBACK(cb_menu_playlist), NULL);

	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	gtk_widget_show_all(menu);

	gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, 0, 0);
}

static void cb_panel_click_event(GtkWidget *widget, GdkEventButton *event,
                                 gpointer user_data)
{
	guint yoffset = *(guint *)user_data;

	if (event->button == 3)
		gkrellm_open_config_window(monitor);
	if (event->button == 1 && event->y < yoffset) {
		seek_in_progress = TRUE;
		gkrellm_update_krell(panel, krell, 
		                     event->x * (trackinfo_get_playtime(&track) / 1000)
		                     / gkrellm_chart_width());
	}
}

static void cb_panel_release_event(GtkWidget *widget, GdkEventButton *event,
                                   gpointer user_data)
{
	guint yoffset = *(guint *)user_data;

	if (event->button == 1) {
		xmmsc_result_t *res;
		guint           seek_ms;

		seek_ms = (guint)(event->x * 
		                  trackinfo_get_playtime(&track)/gkrellm_chart_width());

		if (xmms2ctrl_get_error() != ERROR_CONNECTION &&
		    trackinfo_get_playtime(&track) > seek_ms && event->y < yoffset &&
		    seek_in_progress) {
			res = xmmsc_playback_seek_ms(connection, seek_ms);
			xmmsc_result_unref(res);
		}
		seek_in_progress = FALSE;
	}
}

static gboolean cb_motion_notify_event(GtkWidget *widget, GdkEventMotion *event,
                                       gpointer user_data)
{
	int             x, y;
	GdkModifierType state;
	guint           yoffset = *(guint *)user_data;

	if (event->is_hint) {
		gdk_window_get_pointer(event->window, &x, &y, &state);
	} else {
		x = event->x;
		y = event->y;
		state = event->state;
	}

	if (state & GDK_BUTTON1_MASK && y < yoffset && seek_in_progress) {
		int pos = x*(trackinfo_get_playtime(&track)/1000)/gkrellm_chart_width();
		gkrellm_update_krell(panel, krell, pos);
	} else if (seek_in_progress) {
		seek_in_progress = FALSE;
	}
	return TRUE;
}

static gboolean reconnect(void)
{
	gboolean result = TRUE;
	printf("gkrellxmms2: reconnecting to xmms2d...");
	/* try to reconnect */
	if (xmms2_connect() != ERROR_NONE) {
		strncpy(scroll_text, xmmsc_get_last_error(connection), SCROLL_TEXT_LENGTH);
		result = FALSE;
	} else if (GTK_IS_WIDGET(gml)) {
		gml_pl_setup_xmms_callbacks((GMedialib *)gml);
		gml_setup_xmms_callbacks((GMedialib *)gml);
	}
	printf("%s\n", result ? "ok" : "failed");
	return result;
}

static void cb_button(GkrellmDecalbutton *button, gpointer data)
{
	gint button_pressed = GPOINTER_TO_INT(data);
	if (xmms2ctrl_get_error() == ERROR_CONNECTION) {
		if (!reconnect()) button_pressed = -1;
	}

	switch(button_pressed) {
		case PREV_BUTTON:
			xmms2ctrl_do_reljump(connection, -1);
			break;
		case PLAY_BUTTON:
			xmms2ctrl_toggle_pause(connection, TRUE);
			break;
		case NEXT_BUTTON:
			xmms2ctrl_do_reljump(connection, 1);
			break;
		case STOP_BUTTON:
			xmms2ctrl_stop(connection);
			break;
		case MENU_BUTTON:
			open_menu();
			break;
	}
}

static gint panel_expose_event(GtkWidget *widget, GdkEventExpose *ev)
{
	gdk_draw_pixmap(widget->window,
			widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
			panel->pixmap, ev->area.x, ev->area.y, ev->area.x, ev->area.y,
			ev->area.width, ev->area.height);
	return FALSE;
}

static void sig_handle_playtime(xmmsc_result_t *res, void *userdata)
{
	xmmsc_result_t *res2;
	guint          ptime;
	static guint   cnt = 0;

	if (xmms2ctrl_get_error() != ERROR_CONNECTION) {
		if (xmmsc_result_iserror(res))
			return;
		if (!xmmsc_result_get_uint(res, &ptime))
			return;

		res2 = xmmsc_result_restart(res);
		xmmsc_result_unref(res);
		xmmsc_result_unref(res2);
	}
	if (cnt > 10) {
		cnt = 0;
		if (trackinfo_get_playtime(&track) <= 0)
			ptime = 0;
		if (!seek_in_progress)
			gkrellm_update_krell(panel, krell, ptime / 1000);
	}
	cnt++;
}

static void update_plugin()
{
	gint	 	   w_scroll, w_decal;
	static gint	   x_scroll;
	/* static guint   current_track_time = 0; */

#if defined(GKRELLM_HAVE_DECAL_SCROLL_TEXT)
	/* Gkrellm version 2.2.0 adds a scrolling text mode to a text decal so
	|  that inherently slow Pango text draws can be avoided at each scroll
	|  step as long as the sroll text string is unchanged.
	*/
	gkrellm_decal_scroll_text_set_text(panel, text1_decal, scroll_text);
	gkrellm_decal_scroll_text_get_size(text1_decal, &w_scroll, NULL);
	gkrellm_decal_get_size(text1_decal, &w_decal, NULL);
	if (reset_scroller) {
		x_scroll = w_decal;
		reset_scroller = FALSE;
	}

	x_scroll -= config.scroll_speed;
	if (x_scroll <= -w_scroll)
		x_scroll = scroll_loop_mode ? 0 : w_decal;
	gkrellm_decal_text_set_offset(text1_decal, x_scroll, 0);
#else
	w_decal = text1_decal->w;
	w_scroll = gdk_string_width(text1_decal->text_style.font, scroll_text);
	x_scroll -= config.scroll_speed;
	if (reset_scroller) {
		x_scroll = w_decal;
		reset_scroller = FALSE;
	}
	if (x_scroll <= -w_scroll)
		x_scroll = w_decal;
	text1_decal->x_off = x_scroll;
	gkrellm_draw_decal_text(panel, text1_decal, scroll_text, x_scroll);
#endif

	gkrellm_draw_panel_layers(panel);
}


static void n_media_lib_get_info(xmmsc_result_t *res, void *arg)
{
	trackinfo_update(res, &track);
	update_krell();
	update_plugin();
	xmmsc_result_unref(res);
}

static guint current_id;

static void bc_handle_medialib_entry_changed(xmmsc_result_t *res, 
                                             void           *userdata)
{
	guint id;

	if (xmmsc_result_get_uint(res, &id) && 
	    xmms2ctrl_get_error() != ERROR_CONNECTION) {
	    if (current_id == id) {
			xmmsc_result_t *res2;

			res2 = xmmsc_medialib_get_info(connection, current_id);
			xmmsc_result_notifier_set(res2, n_media_lib_get_info, &current_id);
			xmmsc_result_unref(res2);
		}
	}
}

static void bc_handle_current_id(xmmsc_result_t *res, void *userdata)
{
	if (xmmsc_result_get_uint(res, &current_id) && 
	    xmms2ctrl_get_error() != ERROR_CONNECTION) {
	    xmmsc_result_t *res2;

	    res2 = xmmsc_medialib_get_info(connection, current_id);
	    xmmsc_result_notifier_set(res2, n_media_lib_get_info, &current_id);
	    xmmsc_result_unref(res2);
	}
}

static void sig_handle_current_id(xmmsc_result_t *res, void *userdata)
{
	if (xmmsc_result_get_uint(res, &current_id) && 
	    xmms2ctrl_get_error() != ERROR_CONNECTION) {
	    xmmsc_result_t *res2;

	    res2 = xmmsc_medialib_get_info(connection, current_id);
	    xmmsc_result_notifier_set(res2, n_media_lib_get_info, &current_id);
	    xmmsc_result_unref(res2);
	}
	xmmsc_result_unref(res);
}

static void bc_handle_playback_status_change(xmmsc_result_t *res, void *userdata)
{
	guint pb_status = 0;

	if (xmmsc_result_iserror(res))
		return;
	if (!xmmsc_result_get_uint(res, &pb_status))
		return;

	switch (pb_status) {
		case XMMS_PLAYBACK_STATUS_PLAY:
			control_buttons_set_play_pause_button(&cb, PAUSE_BUTTON);
			break;
		case XMMS_PLAYBACK_STATUS_STOP:
			gkrellm_update_krell(panel, krell, 0);
		case XMMS_PLAYBACK_STATUS_PAUSE:
			control_buttons_set_play_pause_button(&cb, PLAY_BUTTON);
			break;
	}
}

static void n_handle_playback_status(xmmsc_result_t *res, void *userdata)
{
	guint pb_status = 0;

	if (xmmsc_result_iserror(res))
		return;
	if (!xmmsc_result_get_uint(res, &pb_status))
		return;

	if (pb_status == XMMS_PLAYBACK_STATUS_PLAY) {
		control_buttons_set_play_pause_button(&cb, PAUSE_BUTTON);
	} else {
		control_buttons_set_play_pause_button(&cb, PLAY_BUTTON);
	}
	xmmsc_result_unref(res);
}

static gboolean cb_timer_connection_lost(gpointer userdata)
{
	gboolean result = TRUE;
	if (xmms2ctrl_get_error() == ERROR_CONNECTION) {
		if (reconnect()) result = FALSE;   /* do not restart timer on success */
	} else {
		result = FALSE; /* do not restart timer if there is no connection prob*/
	}
	return result;
}

static void connection_lost(void *data)
{
	strcpy(scroll_text, "ERROR: Connection to xmms2d lost.");
	xmms2ctrl_set_error(ERROR_CONNECTION);
	xmmsc_unref(connection);
	if (config.auto_reconnect)
		g_timeout_add(6000, &cb_timer_connection_lost, NULL);
}

static int xmms2_connect()
{
	gchar *path = NULL;

	path = getenv("XMMS_PATH");
	if (path == NULL) path = config.ipc_path;
	xmms2ctrl_set_error(ERROR_NONE);

	if (!path || g_ascii_strncasecmp(path, "unix://", 7) == 0)
		local_xmms2d = TRUE;

	/* Initialize connection to XMMS2 daemon */
	if (!xmmsc_connect(connection, path)) {
		strncpy(scroll_text, xmmsc_get_last_error(connection), 
		        SCROLL_TEXT_LENGTH);
		xmms2ctrl_set_error(ERROR_CONNECTION);
	} else {
		xmmsc_mainloop_gmain_init(connection);
		XMMS_CALLBACK_SET(connection, xmmsc_playback_current_id,
	 	                  sig_handle_current_id, connection);
		XMMS_CALLBACK_SET(connection, xmmsc_broadcast_playback_current_id,
		                  bc_handle_current_id, connection);
		XMMS_CALLBACK_SET(connection, xmmsc_broadcast_medialib_entry_changed,
		                  bc_handle_medialib_entry_changed, connection);
		XMMS_CALLBACK_SET(connection, xmmsc_signal_playback_playtime, 
		                  sig_handle_playtime, NULL);
		XMMS_CALLBACK_SET(connection, xmmsc_broadcast_playback_status,
		                  bc_handle_playback_status_change, NULL);
		xmmsc_disconnect_callback_set(connection, connection_lost, NULL);
	}
	seek_in_progress = FALSE;
	return xmms2ctrl_get_error();
}

static void create_plugin(GtkWidget *vbox, gint first_create)
{
	GkrellmPiximage  *krell_image = NULL;

	GkrellmStyle     *style;
	GkrellmTextstyle *ts, *ts_alt;
	static gint      y;
	xmmsc_result_t   *res;

	/* Initialize connection to XMMS2 daemon */
	connection = xmmsc_init("gkrellxmms2");
	xmms2_connect();

	if (first_create)
		panel = gkrellm_panel_new0();

	style = gkrellm_meter_style(style_id);

	krell_image = gkrellm_krell_meter_piximage(style_id);
	krell = gkrellm_create_krell(panel, krell_image, style);
	gkrellm_set_krell_full_scale(krell, 230, 1);
	gkrellm_monotonic_krell_values(krell, FALSE);

	ts = gkrellm_meter_textstyle(style_id);
	ts_alt = gkrellm_meter_alt_textstyle(style_id);

	/* text decal for scroll text */
	text1_decal = gkrellm_create_decal_text(panel, "Ay", ts, style,
				-1,     /* x = -1 places at left margin */
				-1,     /* y = -1 places at top margin	*/
				-1);    /* w = -1 makes decal the panel width minus margins */
	y = text1_decal->y + text1_decal->h + 2;

	control_buttons_create(&cb, panel, y, &cb_button);

	gkrellm_panel_configure(panel, NULL, style);
	gkrellm_panel_create(vbox, monitor, panel);
	
	tooltip = gtk_tooltips_new();
	gtk_tooltips_set_tip(tooltip, panel->drawing_area,
	                     "XMMS2 Plugin", "empty");
	gtk_tooltips_set_delay(tooltip, 750);
	gtk_tooltips_enable(tooltip);

	if (first_create) {
		g_signal_connect(G_OBJECT(panel->drawing_area), "expose_event",
		                 G_CALLBACK(panel_expose_event), NULL);
		g_signal_connect(G_OBJECT(panel->drawing_area), "button_press_event",
		                 G_CALLBACK(cb_panel_click_event), &y);
		g_signal_connect(G_OBJECT(panel->drawing_area), "button_release_event",
		                 G_CALLBACK(cb_panel_release_event), &y);
		g_signal_connect(G_OBJECT(panel->drawing_area), "motion_notify_event",
		                 G_CALLBACK(cb_motion_notify_event), &y);
	}
	
	if (xmms2ctrl_get_error() == ERROR_NONE) {
		res = xmmsc_playback_status(connection);
		xmmsc_result_notifier_set(res, n_handle_playback_status, NULL);
		xmmsc_result_unref(res);
	}
}

static gchar *plugin_info_text[] =
{
"<b>GkrellM XMMS2 Control (gkrellxmms2)\n",
"Version: ", VERSION_NUMBER, "\n",
"gkrellxmms2 Copyright (c) 2005-2007 Johannes Heimansberg\n",
"http://wejp.k.vu/\n\n",
"<b>Usage:\n",
"gkrellxmms2 has five buttons to control the XMMS2 audio player.\n",
"From left to right those buttons are:\n",
"- Previous track button: Jumps back to the previously played track.\n",
"- Play/Pause button: Toggles play/pause.\n",
"- Stop button: Stops playback.\n",
"- Next track button: Jumps to the next track in the playlist.\n",
"- Menu button: Opens a menu, where you can either open a track info dialog\n",
"  or the medialib/playlist editor.\n",
};

static void create_plugin_tab(GtkWidget *tab_vbox)
{
	GtkWidget	*tabs, *text, *label;
	GtkWidget	*vbox;
	gint		i;
	gchar       *about_text;

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

	/* Info tab */
	vbox = gkrellm_gtk_framed_notebook_page(tabs, "Info");
	text = gkrellm_gtk_scrolled_text_view(vbox, NULL,
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	for (i = 0; i < sizeof(plugin_info_text)/sizeof(gchar *); i++)
		gkrellm_gtk_text_view_append(text, plugin_info_text[i]);

	/* About tab */
	about_text = g_strdup_printf(
		"gkrellxmms2 %s\n"
		"GKrellM XMMS2 Control Plugin\n\n"
		"Copyright (c) 2005-2007 Johannes Heimansberg\n"
		"http://wejp.k.vu/\n\n"
		"Released under the GNU General Public License v2.0", 
		VERSION_NUMBER);
	text = gtk_label_new(about_text);
	label = gtk_label_new("About");
	gtk_notebook_append_page(GTK_NOTEBOOK(tabs), text, label);
	g_free(about_text);
}

static void save_config(FILE *f)
{
	/* printf("gkrellxmms2: saving config...\n"); */

	if (gml != NULL) {
		config.medialib_window_width            = gml_window_get_width(GMEDIALIB(gml));
		config.medialib_window_height           = gml_window_get_height(GMEDIALIB(gml));
		config.medialib_window_pos_x            = gml_window_get_pos_x(GMEDIALIB(gml));
		config.medialib_window_pos_y            = gml_window_get_pos_y(GMEDIALIB(gml));
		config.medialib_playlist_new_confirm    = gml_pl_get_new_playlist_confirm(GMEDIALIB(gml));
		if (gml_notebook_is_mlib_search_visible(GMEDIALIB(gml))) {
			config.medialib_search_column_id        = gml_search_get_column_id_visible(GMEDIALIB(gml));
			config.medialib_search_column_artist    = gml_search_get_column_artist_visible(GMEDIALIB(gml));
			config.medialib_search_column_track     = gml_search_get_column_track_visible(GMEDIALIB(gml));
			config.medialib_search_column_album     = gml_search_get_column_album_visible(GMEDIALIB(gml));
			config.medialib_search_column_rating    = gml_search_get_column_rating_visible(GMEDIALIB(gml));
		}
		if (gml_notebook_is_playlist_visible(GMEDIALIB(gml))) {
			config.medialib_playlist_column_id      = gml_pl_get_column_id_visible(GMEDIALIB(gml));
			config.medialib_playlist_column_artist  = gml_pl_get_column_artist_visible(GMEDIALIB(gml));
			config.medialib_playlist_column_track   = gml_pl_get_column_track_visible(GMEDIALIB(gml));
			config.medialib_playlist_column_album   = gml_pl_get_column_album_visible(GMEDIALIB(gml));
			config.medialib_playlist_column_rating  = gml_pl_get_column_rating_visible(GMEDIALIB(gml));
			config.medialib_playlist_column_pos     = gml_pl_get_column_pos_visible(GMEDIALIB(gml));
		}
	}
	fprintf(f, "%s scroll_speed %d\n",                     PLUGIN_NAME,
	        config.scroll_speed);
	fprintf(f, "%s medialib_window_width %d\n",            PLUGIN_NAME,
	        config.medialib_window_width);
	fprintf(f, "%s medialib_window_height %d\n",           PLUGIN_NAME,
	        config.medialib_window_height);
	fprintf(f, "%s medialib_window_pos_x %d\n",            PLUGIN_NAME,
	        config.medialib_window_pos_x);
	fprintf(f, "%s medialib_window_pos_y %d\n",            PLUGIN_NAME,
	        config.medialib_window_pos_y);
	fprintf(f, "%s medialib_search_column_id %d\n",        PLUGIN_NAME,
	        config.medialib_search_column_id);
	fprintf(f, "%s medialib_search_column_artist %d\n",    PLUGIN_NAME,
	        config.medialib_search_column_artist);
	fprintf(f, "%s medialib_search_column_track %d\n",     PLUGIN_NAME,
	        config.medialib_search_column_track);
	fprintf(f, "%s medialib_search_column_album %d\n",     PLUGIN_NAME,
	        config.medialib_search_column_album);
	fprintf(f, "%s medialib_search_column_rating %d\n",    PLUGIN_NAME,
	        config.medialib_search_column_rating);
	fprintf(f, "%s medialib_playlist_column_id %d\n",      PLUGIN_NAME,
	        config.medialib_playlist_column_id);
	fprintf(f, "%s medialib_playlist_column_artist %d\n",  PLUGIN_NAME,
	        config.medialib_playlist_column_artist);
	fprintf(f, "%s medialib_playlist_column_track %d\n",   PLUGIN_NAME,
	        config.medialib_playlist_column_track);
	fprintf(f, "%s medialib_playlist_column_album %d\n",   PLUGIN_NAME,
	        config.medialib_playlist_column_album);
	fprintf(f, "%s medialib_playlist_column_rating %d\n",  PLUGIN_NAME,
	        config.medialib_playlist_column_rating);
	fprintf(f, "%s medialib_playlist_column_pos %d\n",     PLUGIN_NAME,
	        config.medialib_playlist_column_pos);
	fprintf(f, "%s medialib_playlist_new_confirm %d\n",    PLUGIN_NAME,
	        config.medialib_playlist_new_confirm);
	fprintf(f, "%s ipc_path %s\n",                         PLUGIN_NAME,
	        config.ipc_path);
	fprintf(f, "%s auto_reconnect %d\n",                   PLUGIN_NAME,
	        config.auto_reconnect);
}

static void load_config(gchar *arg)
{
	gchar cfg[64], item[1024];
	gint  n;

	/* printf("gkrellxmms2: loading config: "); */
	n = sscanf(arg, "%s %[^\n]", cfg, item);
	if (n == 2) {
		/* printf("%s = %s (%d)\n", cfg, item, atoi(item)); */
		if (strcmp(cfg, "scroll_speed") == 0)
			config.scroll_speed = atoi(item);
		if (strcmp(cfg, "medialib_window_width") == 0)
			config.medialib_window_width = (atoi(item) < 200 ? 200 : atoi(item));
		if (strcmp(cfg, "medialib_window_height") == 0)
			config.medialib_window_height = (atoi(item) < 200 ? 200 : atoi(item));
		if (strcmp(cfg, "medialib_window_pos_x") == 0)
			config.medialib_window_pos_x = (atoi(item) < 0 ? 0 : atoi(item));
		if (strcmp(cfg, "medialib_window_pos_y") == 0)
			config.medialib_window_pos_y = (atoi(item) < 0 ? 0 : atoi(item));
		if (strcmp(cfg, "medialib_search_column_id") == 0)
			config.medialib_search_column_id = atoi(item);
		if (strcmp(cfg, "medialib_search_column_artist") == 0)
			config.medialib_search_column_artist = atoi(item);
		if (strcmp(cfg, "medialib_search_column_track") == 0)
			config.medialib_search_column_track = atoi(item);
		if (strcmp(cfg, "medialib_search_column_album") == 0)
			config.medialib_search_column_album = atoi(item);
		if (strcmp(cfg, "medialib_search_column_rating") == 0)
			config.medialib_search_column_rating = atoi(item);
		if (strcmp(cfg, "medialib_playlist_column_id") == 0)
			config.medialib_playlist_column_id = atoi(item);
		if (strcmp(cfg, "medialib_playlist_column_artist") == 0)
			config.medialib_playlist_column_artist = atoi(item);
		if (strcmp(cfg, "medialib_playlist_column_track") == 0)
			config.medialib_playlist_column_track = atoi(item);
		if (strcmp(cfg, "medialib_playlist_column_album") == 0)
			config.medialib_playlist_column_album = atoi(item);
		if (strcmp(cfg, "medialib_playlist_column_rating") == 0)
			config.medialib_playlist_column_rating = atoi(item);
		if (strcmp(cfg, "medialib_playlist_column_pos") == 0)
			config.medialib_playlist_column_pos = atoi(item);
		if (strcmp(cfg, "medialib_playlist_new_confirm") == 0)
			config.medialib_playlist_new_confirm = atoi(item);
		if (strcmp(cfg, "ipc_path") == 0)
			g_strlcpy(config.ipc_path, item, sizeof(config.ipc_path));
		if (strcmp(cfg, "auto_reconnect") == 0)
			config.auto_reconnect = atoi(item);
	}
}

static GkrellmMonitor	plugin_mon	=
{
	PLUGIN_NAME,           /* Name, for config tab.    */
	0,                     /* Id,  0 if a plugin       */
	create_plugin,         /* The create function      */
	update_plugin,         /* The update function      */
	create_plugin_tab,     /* The config tab create function */
	NULL,                  /* Apply the config function      */

	save_config,           /* Save user config   */
	load_config,           /* Load user config   */
	PLUGIN_NAME,           /* config keyword     */

	NULL,                  /* Undefined 2  */
	NULL,                  /* Undefined 1  */
	NULL,                  /* private      */

	MON_MAIL,              /* Insert plugin before this monitor */

	NULL,                  /* Handle if a plugin, filled in by GKrellM */
	NULL                   /* path if a plugin, filled in by GKrellM   */
};


#if defined(WIN32)
__declspec(dllexport) GkrellmMonitor *
gkrellm_init_plugin(win32_plugin_callbacks* calls)
#else
GkrellmMonitor * gkrellm_init_plugin(void)
#endif
{
	gchar       *ipc_path_default;
	const gchar *username;
	gint         str_length = 0;

	/* set default config values */
	config.scroll_speed = 1;
	config.medialib_window_width  = 460;
	config.medialib_window_height = 320;
	config.medialib_window_pos_x  = 20;
	config.medialib_window_pos_y  = 20;
	config.medialib_search_column_id        = TRUE;
	config.medialib_search_column_artist    = TRUE;
	config.medialib_search_column_track     = TRUE;
	config.medialib_search_column_album     = TRUE;
	config.medialib_search_column_rating    = TRUE;
	config.medialib_playlist_column_id      = TRUE;
	config.medialib_playlist_column_artist  = TRUE;
	config.medialib_playlist_column_track   = TRUE;
	config.medialib_playlist_column_album   = TRUE;
	config.medialib_playlist_column_rating  = TRUE;
	config.medialib_playlist_column_pos     = TRUE;
	config.medialib_playlist_new_confirm    = TRUE;
	config.auto_reconnect                   = FALSE;

	username = g_get_user_name();
	str_length = strlen(username) + 128;
	ipc_path_default = g_malloc(str_length * sizeof(gchar));
	snprintf(ipc_path_default, str_length, "unix:///tmp/xmms-ipc-%s", username);
	g_strlcpy(config.ipc_path, ipc_path_default, sizeof(config.ipc_path));
	g_free(ipc_path_default);

	/* If this call is made, the background and krell images for this plugin
	|  can be custom themed by putting bg_meter.png or krell.png in the
	|  subdirectory PLUGIN_NAME of the theme directory.  Text colors (and
	|  other things) can also be specified for the plugin with gkrellmrc
	|  lines like:  StyleMeter PLUGIN_NAME.textcolor orange black shadow
	|  If no custom themeing has been done, then all above calls using
	|  style_id will be equivalent to style_id = DEFAULT_STYLE_ID.
	*/
	style_id = gkrellm_add_meter_style(&plugin_mon, PLUGIN_NAME);
	monitor = &plugin_mon;
	return &plugin_mon;
}
