"""

Gwibber Client Interface Library
SegPhault (Ryan Paul) - 05/26/2007

"""

import webkit, gtk, urllib2, hashlib, os, gobject, dbus
from . import gintegration, resources, config, configui

from mako.template import Template
from mako.lookup import TemplateLookup
from gettext import lgettext as _


IMG_CACHE_DIR = os.path.join(resources.CACHE_BASE_DIR, "gwibber", "images")

try:
  import gtkspell
except:
  gtkspell = None

class Color:
  def __init__(self, hex):
    self.hex = hex
    self.dec = int(hex.replace("#", ""), 16)
    self.r = (self.dec >> 16) & 0xff
    self.g = (self.dec >> 8) & 0xff
    self.b = self.dec & 0xff
    self.rgb = "%s, %s, %s" % (self.r, self.g, self.b)

  def darker(self, pct):
    return Color("#%02x%02x%02x" % (self.r * pct, self.g * pct, self.b * pct))

  @classmethod
  def from_gtk_color(self, c):
    if isinstance(c, gtk.gdk.Color): c = str(c)
    return self("#" + "".join([c[1:3], c[5:7], c[9:11]]))

def get_theme_colors(w):
  d = {}
  
  for i in ["base", "text", "fg", "bg"]:
    d[i] = Color.from_gtk_color(
      getattr(w.get_style(), i)[gtk.STATE_NORMAL].to_string())

    d["%s_selected" % i] = Color.from_gtk_color(
      getattr(w.get_style(), i)[gtk.STATE_SELECTED].to_string())

  return d

class GwibberSearch(gtk.Entry):
  def __init__(self, accounts, microblog):
    gtk.Entry.__init__(self)
    try:
      self.set_property("primary-icon-stock", gtk.STOCK_FIND)
      self.set_property("secondary-icon-stock", gtk.STOCK_CLEAR)
      self.connect("icon-press", self.on_icon_press)
    except: pass
    
    """
    self.accounts = accounts
    self.microblog = microblog
    self.features = self.microblog.features()
    self.protocols = self.microblog.protocols()
    """

  def clear(self):
    self.set_text("")

  def on_icon_press(self, w, pos, e):
    if pos == 1: return self.clear()
    
    """
    menu = gtk.Menu()
    for acct in self.accounts:
      if "search" in self.protocols[acct["protocol"]]["features"]:
        mi = gtk.ImageMenuItem(self.protocols[acct["protocol"]]["name"])
        img = resources.get_ui_asset("icons/%s.png" % acct["protocol"])
        if img: mi.set_image(gtk.image_new_from_file(img))
        if hasattr(mi, "set_always_show_image"): mi.set_always_show_image(True)
        mi.connect("activate", self.on_change_search_target, acct)
        menu.append(mi)

    menu.show_all()
    menu.popup(None, None, None, 3, 0)
    """

  def on_change_search_target(self, mi, acct):
    icon = resources.get_ui_asset("icons/%s.png" % acct["protocol"])
    pb = gtk.gdk.pixbuf_new_from_file(icon)
    self.set_property("primary-icon-pixbuf", pb)

class AccountTreeStore(gtk.TreeStore):
  def __init__(self, accounts, microblog):
    gtk.TreeStore.__init__(self,
        str, # Account name
        str, # Message path
        str, # Account ID
        str, # Feature
        gtk.gdk.Pixbuf, # Icon
        bool, # Transient
        str) # Op ID
    self.accounts = accounts
    self.microblog = microblog
    self.features = self.microblog.features()
    self.protocols = self.microblog.protocols()

  def add_account(self, acct):
    pbfile = gtk.gdk.pixbuf_new_from_file
    aname = (acct["username"] or "None").capitalize()
    apath = "/gwibber/%s/%s" % (acct["protocol"], acct["id"])
    aicon = resources.get_ui_asset("icons/%s.png" % acct["protocol"])

    ti = self.append(None, [aname, apath, acct["id"], None, pbfile(aicon), None, None])

    tree_features = [self.features[f] for f in
      self.protocols[acct["protocol"]]["features"]
        if self.features[f]["account_tree"]]

    if len(tree_features) > 1:
      for f in tree_features:
        apath = "/gwibber/%s/%s/%s" % (acct["protocol"], acct["id"], f["stream"])
        aname = f["stream"].capitalize()
        aicon = resources.icon(f["icon"], use_theme=False) or resources.get_ui_asset("icons/%s.png" % f["icon"])
        self.append(ti, [aname, apath, acct["id"], f["name"], pbfile(aicon), None, None])

    return ti

  def remove_account(self, acctid):
    for i in self:
      if i[2] == acctid:
        return self.remove(i.iter)

  def update_account(self, acctid):
    acct = self.accounts.get_account(acctid)
    for i in self:
      if i[2] == acctid:
        self.move_after(self.add_account(acct), i.iter)
        return self.remove(i.iter)

    for i in self:
      if i[1] == "search":
        return self.move_before(self.add_account(acct), i.iter)

  def populate_tree(self):
    toplevel = {
      "home": "go-home",
      "messages": "messages",
      "replies": "mail-reply-all",
    }

    pbfile = gtk.gdk.pixbuf_new_from_file

    for n, i in toplevel.items():
      icf = resources.icon(i, use_theme=False) or resources.get_ui_asset("icons/%s.png" % i)
      icon = pbfile(icf)
      self.append(None, [n.capitalize(), n, None, None, icon, None, None])

    for acct in self.accounts: self.add_account(acct)
      
    icon = pbfile(resources.icon("gtk-find", use_theme=False))
    self.search_item = self.append(None, ["Search", "search", None, None, icon, None, None])

  def add_errors(self):
    if not hasattr(self, "error_item"):
      icon = gtk.gdk.pixbuf_new_from_file(resources.icon("gtk-dialog-warning", use_theme=False))
      self.error_item = self.append(None, ["Errors", "errors", None, None, icon, None, None])

  def add_search(self, query, feature="search"):
    pbfile = gtk.gdk.pixbuf_new_from_file
    icon = pbfile(resources.icon("gtk-find", use_theme=False))
    path = "/gwibber/%s/%s" % (feature, query)
    opid = "search-%s" % query
    ti = self.append(self.search_item, [query, path, None, feature, icon, True, opid])

    for acct in self.accounts:
      if feature in self.protocols[acct["protocol"]]["features"]:
        aname = (acct["username"] or "None").capitalize()
        apath = "/gwibber/%s/%s/%s/%s" % (feature, query, acct["protocol"], acct["id"])
        aicon = resources.get_ui_asset("icons/%s.png" % acct["protocol"])
        opid = "search-%s-%s" % (query, acct["id"])
        self.append(ti, [aname, apath, acct["id"], None, pbfile(aicon), True, opid])

    return ti

class AccountTreeView(gtk.TreeView):
  def __init__(self, store):
    gtk.TreeView.__init__(self, store)
    self.store = store
    self.get_selection().set_mode(gtk.SELECTION_MULTIPLE)

    celltext = gtk.CellRendererText()
    cellimg = gtk.CellRendererPixbuf()
    cellclose = gtk.CellRendererPixbuf()
    cellclose.set_property("stock-id", gtk.STOCK_CLOSE)
    cellclose.set_property("stock-size", gtk.ICON_SIZE_MENU)
    self.cellclose = cellclose
    
    col = gtk.TreeViewColumn("Streams")
    col.pack_start(cellimg, False)
    col.pack_start(celltext, True)
    col.pack_start(cellclose, False)
    col.add_attribute(celltext, "text", 0)
    col.add_attribute(cellimg, "pixbuf", 4)
    col.add_attribute(cellclose, "visible", 5)
    self.append_column(col)
    self.column = col

  def get_selected_accounts(self):
    return set([row[2] for row in self.get_multiselected_data() if row[2]])

  def get_multiselected_data(self):
    model, rows = self.get_selection().get_selected_rows()
    return [model[model.get_iter(row)] for row in rows]
  
  def get_selected_iter(self):
    model, rows = self.get_selection().get_selected_rows()
    return model.get_iter(rows[-1])
    
  def get_selected_data(self):
    return self.store[self.get_selected_iter()]

  def select_iter(self, ti, unselect_others=True):
    if unselect_others: self.get_selection().unselect_all()
    path = self.get_model().get_path(ti)
    self.expand_to_path(path[:-1])
    self.get_selection().select_path(path)

class AccountComboBoxView(gtk.ComboBox):
  def __init__(self, store):
    gtk.ComboBox.__init__(self, store)
    self.store = store

    celltext = gtk.CellRendererText()
    cellimg = gtk.CellRendererPixbuf()

    self.pack_start(cellimg, False)
    self.pack_start(celltext, True)
    self.add_attribute(celltext, "text", 0)
    self.add_attribute(cellimg, "pixbuf", 4)
    
  def get_selected_iter(self):
    return self.get_active_iter()

  def get_selected_data(self):
    return self.store[self.get_selected_iter()]

class MessageStreamView(webkit.WebView):
  def __init__(self, theme, client):
    webkit.WebView.__init__(self)
    self.connect("navigation-requested", self.on_click_link)

    self.settings = webkit.WebSettings()
    self.set_settings(self.settings)

    self.client = client
    self.load_theme(theme)
    self.load_fonts(resources.get_font())

  def load_theme(self, theme):
    self.theme = theme

  def load_fonts(self, fstring):
    if fstring:
      fname, fsize = fstring.rsplit(None, 1)
      self.settings.set_property("default-font-family", fname)
      self.settings.set_property("sans-serif-font-family", fname)
      self.settings.set_property("serif-font-family", fname)
      self.settings.set_property("default-font-size", (eval(fsize) + 2))

  def display_page(self, template_name, messages=None):
    theme_path = resources.get_theme_path(self.theme)
    template_path = os.path.join(theme_path, template_name)
    template_lookup_paths = list(resources.get_template_dirs()) + [theme_path]

    template = Template(
      open(template_path).read(),
      lookup=TemplateLookup(directories=template_lookup_paths))
    
    content = template.render(
        message_store=messages,
        theme=get_theme_colors(self.client),
        resources=resources,
        preferences=self.client.preferences,
        protocols=self.client.protocols,
        accounts=self.client.accounts,
        _=_)

    #def on_finish_load(v, f, vscroll_pos):
    #  self.scroll.get_vadjustment().set_value(vscroll_pos)
    #self.connect("load-finished", on_finish_load, self.scroll.get_vadjustment().get_value())
    
    self.load_html_string(content, "file://%s/" % resources.get_theme_path(self.theme))

  def load_messages(self, messages, count=None, total=None):
    for n, m in enumerate(messages):
      m.message_index = n

    theme_path = resources.get_theme_path(self.theme)
    template_path = os.path.join(theme_path, "template.mako")
    template_lookup_paths = list(resources.get_template_dirs()) + [theme_path]

    template = Template(
      open(template_path).read(),
      lookup=TemplateLookup(directories=template_lookup_paths))
    
    content = template.render(
        message_store=messages,
        theme=get_theme_colors(self.client),
        resources=resources,
        preferences=self.client.preferences,
        protocols=self.client.protocols,
        accounts=self.client.accounts,
        count=count, total=total, _=_)

    #def on_finish_load(v, f, vscroll_pos):
    #  self.scroll.get_vadjustment().set_value(vscroll_pos)
    #self.connect("load-finished", on_finish_load, self.scroll.get_vadjustment().get_value())
    
    self.load_html_string(content, "file://%s/" % resources.get_theme_path(self.theme))

  def on_click_link(self, view, frame, req):
    uri = req.get_uri()
    if uri.startswith("file:///"): return False
    
    if not self.link_handler(uri, self):
      gintegration.load_url(uri)
    
    return True

  def link_handler(self, uri, view):
    pass

class TargetBar(gtk.Frame):
  __gsignals__ = {
      "canceled": (gobject.SIGNAL_RUN_LAST | gobject.SIGNAL_ACTION, None, ())
  }
  def __init__(self, account_store):
    gtk.Frame.__init__(self)
    self.set_shadow_type(gtk.SHADOW_NONE)
    
    # Target ComboBox
    self.combo_store = account_store.filter_new()
    self.combo_store.set_visible_func(self.filter_combo)
    self.target_combo = gtk.ComboBox(self.combo_store)

    celltext = gtk.CellRendererText()
    cellimg = gtk.CellRendererPixbuf()

    self.target_combo.pack_start(cellimg, False)
    self.target_combo.pack_start(celltext, True)
    self.target_combo.add_attribute(celltext, "text", 0)
    self.target_combo.add_attribute(cellimg, "pixbuf", 4)

    # Close button
    close_button = gtk.EventBox()
    close_button.add(gtk.image_new_from_stock(gtk.STOCK_CLOSE, gtk.ICON_SIZE_MENU))
    close_button.connect("button-release-event", self.on_close_target)

    # Layout
    self.layout1 = gtk.HBox(spacing=5)
    self.layout1.pack_start(gtk.Label("Sending message from:"), False)
    self.layout1.pack_start(self.target_combo)
    self.layout1.pack_start(close_button, False, False)
    self.add(self.layout1)

  def set_account(self, acctid):
    for row in self.combo_store:
      if row[2] == acctid:
        self.target_combo.set_active_iter(row.iter)

  def get_account(self):
    return self.combo_store[self.target_combo.get_active()][2]

  def on_close_target(self, w, e):
    if e.button == 1:
      self.emit("canceled")
      self.hide()

  def filter_combo(self, model, iter):
    account = model.accounts[model[iter][2]]
    return model[iter][3] == None and model[iter][2] != None and \
      "send" in model.protocols[account["protocol"]]["features"]

class Input(gtk.Frame):
  __gsignals__ = {
    "submit": (gobject.SIGNAL_RUN_LAST | gobject.SIGNAL_ACTION, None, (str, int)),
    "changed": (gobject.SIGNAL_RUN_LAST | gobject.SIGNAL_ACTION, None, (str, int))
  }

  def __init__(self, client):
    gtk.Frame.__init__(self)

    self.textview = InputTextView(client)
    self.textview.connect("submit", self.do_submit_event)
    self.textview.get_buffer().connect("changed", self.do_changed_event)

    scroll = gtk.ScrolledWindow()
    scroll.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    scroll.add(self.textview)
    self.add(scroll)

  def get_text(self):
    return self.textview.get_text()

  def set_text(self, t):
    self.textview.get_buffer().set_text(t)

  def clear(self):
    self.set_text("")

  def do_changed_event(self, tb):
    text = self.textview.get_text()
    chars = self.textview.get_char_count()
    self.emit("changed", text, chars)
  
  def do_submit_event(self, tv):
    text = tv.get_text()
    chars = tv.get_char_count()
    self.emit("submit", text, chars)

class InputTextView(gtk.TextView):
  __gsignals__ = {
    "submit": (gobject.SIGNAL_RUN_LAST | gobject.SIGNAL_ACTION, None, ())
  }

  def __init__(self, client):
    gtk.TextView.__init__(self)
    self.client = client

    bus = dbus.SessionBus()
    obj = bus.get_object("com.Gwibber", "/com/gwibber/URLShorten")
    self.shortener = dbus.Interface(obj, "com.Gwibber")
    
    self.connect("populate-popup", self.on_input_context_menu)
    self.get_buffer().connect("insert-text", self.on_add_text)
    self.get_buffer().connect("changed", self.on_text_changed)

    # set properties
    self.set_border_width(0)
    self.set_accepts_tab(True)
    self.set_editable(True)
    self.set_cursor_visible(True)
    self.set_wrap_mode(gtk.WRAP_WORD_CHAR)
    self.set_left_margin(2)
    self.set_right_margin(2)
    self.set_pixels_above_lines(2)
    self.set_pixels_below_lines(2)
    
    self.base_color = self.get_style().base[gtk.STATE_NORMAL]
    self.error_color = gtk.gdk.color_parse("indianred")

    if gtkspell:
      self.spell = gtkspell.Spell(self, None)

  def get_text(self):
    buf = self.get_buffer()
    return buf.get_text(*buf.get_bounds()).strip()

  def get_char_count(self):
    return len(unicode(self.get_text(), "utf-8"))

  def on_add_text(self, buf, iter, text, tlen):
    if self.client.preferences["shorten_urls"]:
      if text and text.startswith("http") and not " " in text \
          and len(text) > 30:

        buf = self.get_buffer()
        buf.stop_emission("insert-text")
        service = self.client.preferences["urlshorter"] or "is.gd"
        short = self.shortener.shorten(text, service)
        buf.insert(iter, short)

  def on_text_changed(self, w):
    chars = self.get_char_count()
    color = self.error_color if chars > 140 else self.base_color
    self.modify_base(gtk.STATE_NORMAL, color)

  def on_input_context_menu(self, obj, menu):
    menu.append(gtk.SeparatorMenuItem())
    for acct in self.client.accounts:
      if "send" in self.client.protocols[acct["protocol"]]["features"]:
        mi = gtk.CheckMenuItem("%s (%s)" % (acct["username"],
          self.client.protocols[acct["protocol"]]["name"]))
        acct.bind(mi, "send_enabled")
        menu.append(mi)
    menu.show_all()

gtk.binding_entry_add_signal(InputTextView, gtk.keysyms.Return, 0, "submit")
gtk.binding_entry_add_signal(InputTextView, gtk.keysyms.KP_Enter, 0, "submit")
