config_domain = ("apps", "gwibber")

import logging

try:
    import gconf
except:
    from gnome import gconf

try:
    # disable this for now, it is causing problems
    raise ImportError
    from desktopcouch.records.server import CouchDatabase2
    from desktopcouch.records.record import Record as CouchRecord2

    RECORD_TYPE = "https://wiki.ubuntu.com/Gwibber/DesktopCouch"
    couch_database = CouchDatabase("-".join(config_domain), create=True)

    def new_couchdb_record_stub():
        """Create mutable containers, but not replacable values."""
        return dict(
            record_type=RECORD_TYPE,
            record_type_version="1",
            #protocol=str(),
            #username=str(),
            #password=str(),
            contacts=list(),  ## unused
            application_annotations=dict(
                gwibber=dict(
                    #receive_enabled=bool(),
                    #search_enabled=bool()
                    #message_color=str()
                ),
            ),
        )

    generic_keys = set(["protocol", "username", "password"])

    def couchdb_create_new_record():
        """Creates record in Db and returns doc-id that referrers need."""
        data = new_couchdb_record_stub()
        record = CouchRecord(data, record_type=RECORD_TYPE)
        return couch_database.put_record(record)

    def couchdb_set_config(document_id, key, value):
        results = couch_database.get_records(create_view=True)
        for row in results[RECORD_TYPE]:
            if row.id == document_id and _couchdb_record_is_not_deleted(row.value):
                data = row.value
                if key in generic_keys:
                    data[str(key)] = value
                else:
                    data["application_annotations"]["gwibber"][str(key)] = value
                couch_database.update_fields(data["_id"], data)
                break

    def couchdb_get_config(document_id, key):
        document_id = str(document_id)
        results = couch_database.get_records(create_view=True)
        for row in results[RECORD_TYPE]:
            if row.id == document_id and _couchdb_record_is_not_deleted(row.value):
                if key in generic_keys:
                    v = row.value[str(key)]
                else:
                    v = row.value["application_annotations"]["gwibber"][str(key)]
                if v is None:
                    raise KeyError("%r not found in %r" % (key, RECORD_TYPE))
                return v
        raise KeyError("%r not found in %r" % (key, RECORD_TYPE))

    def couchdb_purge_document(document_id):
        results = couch_database.get_records(create_view=True)
        for row in results[RECORD_TYPE]:
            if row.id == document_id:
                couch_database.delete_record(row.id)
                #del couch_database[row.id]
                return 1
        return 0

    def couchdb_list_document_ids():
        results = couch_database.get_records(create_view=True)
        for r in results[RECORD_TYPE]:
            if _couchdb_record_is_not_deleted(r.value):
                yield r.id

    def _couchdb_record_is_not_deleted(ob):
        for part in (
                "application_annotations",
                "Ubuntu One",
                "private_application_annotations",
                "deleted"):
            try:
                ob = ob[part]
            except KeyError, e:
                # a step to deleted flag is missing, so not deleted.
                return True
                break  # so, the else block not run.
        else:
            if object:
                return False
            else:
                return True # flag exists, but is false.

    logging.warn("using desktopcouch as primary storage")
except ImportError, e:
    logging.warn("desktopcouch is not available. %s.  falling back to gconf", e)
    def couchdb_create_new_record(): return None
    def couchdb_set_config(document_id, key, value): return False
    def couchdb_get_config(document_id, key): raise KeyError()
    def couchdb_purge_document(document_id): raise KeyError()
    def couchdb_list_document_ids(): return []


from . import gwp
import microblog
try:
  gnomekeyring = None
  #import gnomekeyring
except:
  gnomekeyring = None

GCONF_DIR = "/" + "/".join(config_domain)
GCONF_PREFERENCES_DIR = GCONF_DIR + "/preferences"
GCONF_ACCOUNTS_DIR = GCONF_DIR + "/accounts"
GCONF = gconf.client_get_default()

class Wrapper:
  def __init__(self, path):
    self.path = path

  def __getitem__(self, key):
    value = GCONF.get("%s/%s" % (self.path, key))

    if value:
      return {
        "string": value.get_string,
        "int": value.get_int,
        "float": value.get_float,
        "bool": value.get_bool,
        "list": value.get_list}[value.type.value_nick]()
    else:
      return None

  def __setitem__(self, key, value):
    if type(value).__name__ == "list":
      GCONF.set_list("%s/%s" % (self.path, key), gconf.VALUE_INT, value)
    else:
      { "str": GCONF.set_string,
        "String": GCONF.set_string,
        "int": GCONF.set_int,
        "float": GCONF.set_float,
        "bool": GCONF.set_bool}[type(value).__name__](
          "%s/%s" % (self.path, key), value)

  def bind(self, widget, key, **args):
    gwp.create_persistency_link(widget, "%s/%s" % (self.path, key), **args)
    return widget

  def notify(self, key, method):
    GCONF.notify_add("%s/%s" % (self.path, key), method)

class Account(Wrapper):
  def __init__(self, id, path = GCONF_ACCOUNTS_DIR, gconf_only=False):
    Wrapper.__init__(self, path)
    GCONF.add_dir("%s/%s" % (path, id), gconf.CLIENT_PRELOAD_NONE)
    self.id = id
    if gconf_only:
      # The id we have is GConf.  We create a new Account object and copy all
      # of this into it, and update it when we update this.
      self.copy_dest_id = self.port_item_to_couchdb()
    else:
      self.copy_dest_id = None

  def __getitem__(self, key):
    if key == "id": return self.id
    if gnomekeyring and key.startswith("private:"):
      try:
        key = key.replace("private:", "")
        return gnomekeyring.find_items_sync(
          gnomekeyring.ITEM_GENERIC_SECRET,
          {"id": "%s/%s/%s" % (self.path, self.id, key)})[0].secret
      except gnomekeyring.NoMatchError:
        print "Couldn't retrieve GConf value for key: %s" % key
        return Wrapper.__getitem__(self, "%s/%s" % (self.id, key))
    else:
      if key.startswith("private:"):
        key = key.replace("private:", "")
      try:
        return couchdb_get_config(self.id, key)
      except KeyError:
        pass
      v = Wrapper.__getitem__(self, "%s/%s" % (self.id, key))
      if v:
        couchdb_set_config(self.copy_dest_id, key, v)  # migrate to couchdb
      return v

  def __setitem__(self, key, value):
    if gnomekeyring and key.startswith("private:"):
      key = key.replace("private:", "")

      try:
        token = gnomekeyring.item_create_sync(
          gnomekeyring.get_default_keyring_sync(),
          gnomekeyring.ITEM_GENERIC_SECRET, "Gwibber preference: %s/%s" % (self.id, self.key),
          {"id": "%s/%s/%s" % (self.path, self.id, key)}, value, True)
        Wrapper.__setitem__(self, "%s/%s" % (self.id, key), ":KEYRING:%s" % token)
      except:
        Wrapper.__setitem__(self, "%s/%s" % (self.id, key), value)
    else:
      key = key.replace("private:", "")
      Wrapper.__setitem__(self, "%s/%s" % (self.id, key), value)

  def port_item_to_couchdb(self):
    c_record_id = couchdb_create_new_record()
    if c_record_id is None:
        return None
    l = len(self.path) + 1 + len(self.id) + 1
    for entry in GCONF.all_entries("%s/%s" % (self.path, self.id)):
      value = {
        "string": entry.value.get_string,
        "int": entry.value.get_int,
        "float": entry.value.get_float,
        "bool": entry.value.get_bool}[entry.value.type.value_nick]()
      key = entry.key[l:]
      couchdb_set_config(c_record_id, key, value)
    return c_record_id

  def clear_values(self):
    try:
      couchdb_purge_document(self.id)
      if self.copy_dest_id:
        couchdb_purge_document(self.copy_dest_id)
    except KeyError:
      for entry in GCONF.all_entries("%s/%s" % (self.path, self.id)):
        GCONF.unset(entry.key)

  def bind(self, widget, key, **args):
    return Wrapper.bind(self, widget, "%s/%s" % (self.id, key), **args)

  def notify(self, key, method):
    Wrapper.notify(self, "%s/%s" % (self.id, key), method)

class Accounts:
  def __init__(self, path = GCONF_ACCOUNTS_DIR):
    self.path = path

  def get_account(self, id):
    return Account(id, self.path)

  def new_account(self):
    id = couchdb_create_new_record()
    if not id:
      id = gconf.unique_key()
      index = GCONF.get_list("%s/index" % self.path, gconf.VALUE_STRING)
      index.append(id)
      GCONF.set_list("%s/index" % self.path, gconf.VALUE_STRING, index)

    return Account(id, self.path)

  def delete_account(self, arg):
    index = GCONF.get_list("%s/index" % self.path, gconf.VALUE_STRING)
    try:
      index.remove(isinstance(arg, Account) and arg.id or arg)
    except ValueError:
      pass  # It may exist only in Desktopcouch
    GCONF.set_list("%s/index" % self.path, gconf.VALUE_STRING, index)
    
    if isinstance(arg, Account): arg.clear_values()
    else: Account(id, self.path).clear_values()

  def __getitem__(self, key):
    return self.get_account(key)

  def __iter__(self):
    found_couchdb_values = False
    for document_id in couchdb_list_document_ids():
        found_couchdb_values = True
        a = Account(document_id)
        if a["protocol"] in microblog.PROTOCOLS:
          yield a
    if found_couchdb_values:
        logging.warn("Gconf ignored because we found items in Desktopcouch.")
        return
    for i in GCONF.get_list("%s/index" % self.path, gconf.VALUE_STRING):
      a = Account(i, gconf_only=True)
      if a["protocol"] in microblog.PROTOCOLS:
        yield Account(i)

class Preferences(Wrapper):
  def __init__(self, path = GCONF_PREFERENCES_DIR):
    Wrapper.__init__(self, path)
