/*****************************************************************************
*   Gnome Wave Cleaner Version 0.19
*   Copyright (C) 2001 Jeffrey J. Welty
*   
*   This program is free software; you can redistribute it and/or
*   modify it under the terms of the GNU General Public License
*   as published by the Free Software Foundation; either version 2
*   of the License, or (at your option) any later version.
*   
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*   
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the Free Software
*   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*******************************************************************************/

/* denoise.c */

#include <stdlib.h>
#include <gnome.h>
#include "gtkledbar.h"
#include "gwc.h"
#include <math.h>
#include <float.h>

#include <unistd.h>
#include <sys/time.h>

//struct timeb start_time, middle_time, end_time ;
struct timeval start_time, middle_time, end_time ;
struct timezone tzp;

void start_timer(void)
{
        gettimeofday(&start_time,&tzp);
}

void stop_timer(char *message)
{
    gettimeofday(&end_time,&tzp) ;
    {
	double fstart = start_time.tv_sec + (double)start_time.tv_usec/1000000.0 ;
	double fend = end_time.tv_sec + (double)end_time.tv_usec/1000000.0 ;
	fprintf(stderr, "%s in %7.3lf real seconds\n", message, fend-fstart) ;
    }
}

#ifdef OLD_TIMER
struct timeb start_time, middle_time, end_time ;

void start_timer(void)
{
        ftime(&start_time) ;
}

void stop_timer(char *message)
{
    ftime(&end_time) ;

    {
	double fstart = start_time.time + (double)start_time.millitm/1000.0 ;
	double fend = end_time.time + (double)end_time.millitm/1000.0 ;
	fprintf(stderr, "%s in %7.3lf real seconds\n", message, fend-fstart) ;
    }
}
#endif



double window_coef[DENOISE_MAX_FFT] ;

double gain_weiner(double Yk2, double Dk2)
{
    double gain ;
    double Xk2 = Yk2 - Dk2 ;

    if(Yk2 > Dk2)
	gain = (Xk2) / (Xk2+Dk2) ;
    else
	gain = 0.0 ;

    return gain ;
}

double gain_power_subtraction(double Yk2, double Dk2)
{
    double level = MAX(Yk2-Dk2, 0.0) ;

    if(Yk2 > DBL_MIN)
	return level/Yk2 ;
    else
	return 0.0 ;
}

double gain_lorber(double snr)
{
    double snr_db = log10(snr) ;

    double alpha = MAX((3.0 - 0.10*snr_db), 1.0) ;

    double gain = sqrt(1.0 - alpha/(snr+1)) ;

    if(gain > DBL_MIN)
	return gain ;
    else
	return 0.0 ;
}

#define SLOW_EM
#ifdef SLOW_EM
double hypergeom(double theta)
{
    double i0(double), i1(double) ;
    if(theta < 7.389056)
	return exp(-theta/2.0)*(1.0+theta*i0(theta/2.0)+theta*i1(theta/2.0)) ;
    else
	return exp(0.09379 + 0.50447*log(theta)) ;
}

double gain_em(double Rprio, double Rpost, double alpha)
{
    /* Ephraim-Malah classic noise suppression, from 1984 paper */

    double gain = 0.886226925*sqrt(1.0/(1.0+Rpost)*(Rprio/(1.0+Rprio))) ;

    gain *= hypergeom((1.0+Rpost)*(Rprio/(1.0+Rprio))) ;

    return gain ;
}
#else

double gain_em(double Rprio, double Rpost, double alpha)
{
    /* Ephraim-Malah noise suppression, from Godsill and Wolfe 2001 paper */
    double r = MAX(Rprio/(1.0+Rprio),DBL_MIN) ;
    double V = (1.0+Rpost)/r  ;

    return sqrt( (1.0+V)/Rpost * r  ) ;
}
#endif

double blackman(int k, int N)
{
    double p = ((double)(k))/(double)(N-1) ;
    return 0.42-0.5*cos(2.0*M_PI*p) + 0.08*cos(4.0*M_PI*p) ;
}

double hanning(int k, int N)
{
    double p = ((double)(k))/(double)(N-1) ;
    return 0.5 - 0.5 * cos(2.0*M_PI*p) ;
}

double blackman_hybrid(int k, int n_flat, int N)
{
    if(k >= (N-n_flat)/2 && k <= n_flat+(N-n_flat)/2-1) {
	return 1.0 ;
    } else {
	double p ;
	if(k >= n_flat+(N-n_flat)/2-1) k -= n_flat ;
	p = (double)(k)/(double)(N-n_flat-1) ;
	return 0.42-0.5*cos(2.0*M_PI*p) + 0.08*cos(4.0*M_PI*p) ;
    }
}

double welty_alpha(double w, double x)
{
    double alpha = ( log(acos(-2.0*w+1)) - log(M_PI) ) / log(1.0 - x) ;
/*      d_print("Welty alpha=%g\n", alpha) ;  */
    return alpha ;
}

/*  double welty(int k, int N, double alpha)  */
/*  {  */
/*      double n2 = (double)N/2.0 ;  */
/*      double x = fabs(((double)k - n2) / (n2)) ;  */
/*      double tx = pow(1.0-x, alpha)*M_PI ;  */
/*      double w =  -( cos(tx)-1.0 )/2.0 ;  */
/*      d_print("Welty x=%g, w=%g, k=%d N=%d\n", x, w, k, N) ;  */
/*  }  */

double fft_window(int k, int N, int window_type)
{
    if(window_type == DENOISE_WINDOW_BLACKMAN) {
	return blackman(k, N) ;
    } else if(window_type == DENOISE_WINDOW_BLACKMAN_HYBRID) {
	return blackman_hybrid(k, N-N/4, N) ;
    } else if(window_type == DENOISE_WINDOW_HANNING_OVERLAP_ADD) {
	return hanning(k, N) ;
    }

    return 0.0 ;
}

int prev_sample[2] ;

static fftw_real windowed[DENOISE_MAX_FFT] ;
static fftw_real out[DENOISE_MAX_FFT] ;

#ifdef HAVE_FFTW3
static void fft_remove_noise(fftw_real sample[], fftw_real noise_min2[], fftw_real noise_max2[], fftw_real noise_avg2[],
                      FFTW(plan) *pFor, FFTW(plan) *pBak,
#else /* HAVE_FFTW3 */
void fft_remove_noise(fftw_real sample[], fftw_real noise_min2[], fftw_real noise_max2[], fftw_real noise_avg2[],
                      rfftw_plan *pFor, rfftw_plan *pBak,
#endif /* HAVE_FFTW3 */
		      struct denoise_prefs *pPrefs, int ch)
{
    int k ;
    fftw_real noise2[DENOISE_MAX_FFT] ;
    fftw_real Y2[DENOISE_MAX_FFT/2+1] ;
    fftw_real maskedY2[DENOISE_MAX_FFT/2+1] ;
    fftw_real gain_k[DENOISE_MAX_FFT] ;
    static fftw_real bsig_prev[2][DENOISE_MAX_FFT],bY2_prev[2][DENOISE_MAX_FFT/2+1],bgain_prev[2][DENOISE_MAX_FFT/2+1] ;
    fftw_real *sig_prev,*Y2_prev,*gain_prev ;

    sig_prev = bsig_prev[ch] ;
    Y2_prev = bY2_prev[ch] ;
    gain_prev = bgain_prev[ch] ;


    for(k = 0 ; k < pPrefs->FFT_SIZE ; k++) {
	windowed[k] = sample[k]*window_coef[k] ;
	noise2[k] = noise_max2[k] ;
	noise2[k] = noise_min2[k] + 0.5*(noise_max2[k] - noise_min2[k]) ;
	noise2[k] = noise_avg2[k] ;
    }

#ifdef HAVE_FFTW3
    FFTW(execute)(*pFor);
#else /* HAVE_FFTW3 */
    rfftw_one(*pFor, windowed, out);
#endif /* HAVE_FFTW3 */

    for (k = 1; k <= pPrefs->FFT_SIZE/2 ; ++k) {
	Y2[k] = k < pPrefs->FFT_SIZE/2 ?  out[k]*out[k] + out[pPrefs->FFT_SIZE-k]*out[pPrefs->FFT_SIZE-k] : out[k]*out[k] ;
    }

    if(pPrefs->noise_suppression_method == DENOISE_LORBER) {
	for (k = 1; k <= pPrefs->FFT_SIZE/2 ; ++k) {
	    double sum = 0 ;
	    double sum_wgts = 0 ;
	    int j ;

	    int j1 = MAX(k-10,1) ;
	    int j2 = MIN(k+10,pPrefs->FFT_SIZE/2) ;

	    for(j = j1 ; j <= j2 ; j++) {
		double d = ABS(j-k)+1.0 ;
		double wgt = 1./d ;
		sum += Y2[k]*wgt ;
		sum_wgts += wgt ;
	    }

	    maskedY2[k] = sum / (sum_wgts+1.e-300) ;
	}
    }


	

#ifdef TEST
    if(pPrefs->noise_suppression_method == DENOISE_AUDACITY) {

	   for(i=0; i<=len/2; i++)
	      plog[i] = log(power[i]);
	    
	   int half = len/2;
	   for(i=0; i<=half; i++) {
	      float smooth;
	      
	      if (plog[i] < noiseGate[i] + (level/2.0))
		 smooth = 0.0;
	      else
		 smooth = 1.0;
	      
	      smoothing[i] = smooth * 0.5 + smoothing[i] * 0.5;
	   }

	   /* try to eliminate tinkle bells */
	   for(i=2; i<=half-2; i++) {
	      if (smoothing[i]>=0.5 &&
		  smoothing[i]<=0.55 &&
		  smoothing[i-1]<0.1 &&
		  smoothing[i-2]<0.1 &&
		  smoothing[i+1]<0.1 &&
		  smoothing[i+2]<0.1)
		 smoothing[i] = 0.0;
	   }

	   outr[0] *= smoothing[0];
	   outi[0] *= smoothing[0];
	   outr[half] *= smoothing[half];
	   outi[half] *= smoothing[half];
	   for(i=1; i<half; i++) {
	      int j = len - i;
	      float smooth = smoothing[i];

	      outr[i] *= smooth;
	      outi[i] *= smooth;
	      outr[j] *= smooth;
	      outi[j] *= smooth;
	   }
#else
    if(0) {
#endif
    } else {
	for (k = 1; k <= pPrefs->FFT_SIZE/2 ; ++k) {
	    if(noise2[k] > DBL_MIN) {
		double gain, Fk, Gk ;

		if(pPrefs->noise_suppression_method == DENOISE_EM) {
		    double Rpost = MAX(Y2[k]/noise2[k]-1.0, 0.0) ;
		    double alpha = pPrefs->dn_gamma ;
		    double Rprio ;

		    if(prev_sample[ch] == 1)
			Rprio = (1.0-alpha)*MAX(Rpost, 0.0)+alpha*gain_prev[k]*gain_prev[k]*Y2_prev[k]/noise2[k] ;
		    else
			Rprio = MAX(Rpost, 0.0) ;

		    gain = gain_em(Rprio, Rpost, alpha) ;
    /*  		g_print("Rpost:%lg Rprio:%lg gain:%lg gain_prev:%lg y2_prev:%lg\n", Rpost, Rprio, gain, gain_prev[k], Y2_prev[k]) ;  */
		    gain_prev[k] = gain ;
		    Y2_prev[k] = Y2[k] ;
		} else if(pPrefs->noise_suppression_method == DENOISE_LORBER) {
		    double SNRlocal = maskedY2[k]/noise2[k]-1.0 ;
		    double SNRfilt, SNRprio ;

		    if(prev_sample[ch] == 1) {
			SNRfilt = (1.-pPrefs->dn_gamma)*SNRlocal + pPrefs->dn_gamma*(sig_prev[k]/noise2[k]) ;
			SNRprio = SNRfilt ; /* note, could use another parameter, like pPrefs->dn_gamma, here to compute SNRprio */
		    }else {
			SNRfilt = SNRlocal ;
			SNRprio = SNRfilt ;
		    }
		    gain = gain_lorber(SNRprio) ;
		    sig_prev[k] = MAX(Y2[k]*gain,0.0) ;
		} else if(pPrefs->noise_suppression_method == DENOISE_WEINER)
		    gain = gain_weiner(Y2[k], noise2[k]) ;
		else
		    gain = gain_power_subtraction(Y2[k], noise2[k]) ;

		Fk = pPrefs->amount*(1.0-gain) ;

		if(Fk < 0.0) Fk = 0.0 ;
		if(Fk > 1.0) Fk = 1.0 ;

		Gk =  1.0 - Fk ;

		out[k] *= Gk ;
		if(k < pPrefs->FFT_SIZE/2) out[pPrefs->FFT_SIZE-k] *= Gk ;

		gain_k[k] = Gk ;
	    }
	}
    }

    /* the inverse fft */
#ifdef HAVE_FFTW3
    FFTW(execute)(*pBak);
#else /* HAVE_FFTW3 */
    rfftw_one(*pBak, out, windowed);
#endif /* !HAVE_FFTW3 */

    for(k = 0 ; k < pPrefs->FFT_SIZE ; k++)
	windowed[k] /= (double)(pPrefs->FFT_SIZE) ;

    if(pPrefs->window_type != DENOISE_WINDOW_HANNING_OVERLAP_ADD) {
	/* merge results back into sample data based on window function */
	for(k = 0 ; k < pPrefs->FFT_SIZE ; k++) {
	    double w = window_coef[k] ;
	    sample[k] = (1.0-w) * sample[k] + windowed[k] ;
	}
    } else {
	/* make sure the tails of the sample approach zero magnitude */
	double offset = windowed[0] ;
	double hs1 = pPrefs->FFT_SIZE/2 - 1 ;

	for(k = 0 ; k < pPrefs->FFT_SIZE/2 ; k++) {
	    double p = (hs1-(double)k)/hs1 ;
	    sample[k] = windowed[k] - offset*p ;
	}

	offset = windowed[pPrefs->FFT_SIZE-1] ;
	for(k = pPrefs->FFT_SIZE/2 ; k < pPrefs->FFT_SIZE ; k++) {
	    double p = ((double)k-hs1)/hs1 ;
	    sample[k] = windowed[k] - offset*p ;
	}
    }

    prev_sample[ch] = 1 ;
}

int denoise_normalize = 0 ;

#ifdef DEBUG
void print_denoise(char *header, struct denoise_prefs *pDnprefs)
{
    g_print("******** %s ************\n", header) ;
    g_print("FFT_SIZE:%d\n", pDnprefs->FFT_SIZE) ;
    g_print("n_noise_samples:%d\n", pDnprefs->n_noise_samples) ;
    g_print("amount:%lf\n", pDnprefs->amount) ;
    g_print("smoothness:%d\n", pDnprefs->smoothness) ;
    g_print("window_type:") ;
    switch(pDnprefs->window_type) {
	case DENOISE_WINDOW_BLACKMAN : g_print("Blackman\n") ; break ;
	case DENOISE_WINDOW_BLACKMAN_HYBRID : g_print("Blackman-hybrid\n") ; break ;
	case DENOISE_WINDOW_HANNING_OVERLAP_ADD : g_print("Hanning-overlap-add\n") ; break ;
	default : g_print("!!!!!!!!! UNKNOWN !!!!!!!!!!!!\n") ; break ;
    }
    g_print("Suppression method:") ;
    switch(pDnprefs->noise_suppression_method) {
	case DENOISE_LORBER : g_print("Lorber-Hoeldrich\n") ; break ;
	case DENOISE_WEINER : g_print("Weiner\n") ; break ;
	case DENOISE_EM : g_print("Ephram\n") ; break ;
	default : g_print("Spectral Subtraction\n") ; break ;
    }
}
#else
void print_denoise(char *header, struct denoise_prefs *pDnprefs) {}
#endif

void get_noise_sample(struct sound_prefs *pPrefs, struct denoise_prefs *pDnprefs,
		    long noise_start, long noise_end,
		    fftw_real *left_noise_min, fftw_real *left_noise_max, fftw_real *left_noise_avg,
		    fftw_real *right_noise_min, fftw_real *right_noise_max, fftw_real *right_noise_avg) ;

int denoise(struct sound_prefs *pPrefs, struct denoise_prefs *pDnprefs, long noise_start, long noise_end,
            long first_sample, long last_sample, int channel_mask) {
    long current ;
    int k ;
    fftw_real left[DENOISE_MAX_FFT], right[DENOISE_MAX_FFT] ;
    fftw_real left_noise_max[DENOISE_MAX_FFT], right_noise_max[DENOISE_MAX_FFT], left_noise_avg[DENOISE_MAX_FFT] ;
    fftw_real left_noise_min[DENOISE_MAX_FFT], right_noise_min[DENOISE_MAX_FFT], right_noise_avg[DENOISE_MAX_FFT] ;
    fftw_real tmp[DENOISE_MAX_FFT] ;
    fftw_real left_prev_frame[DENOISE_MAX_FFT] ;
    fftw_real right_prev_frame[DENOISE_MAX_FFT] ;
#ifdef HAVE_FFTW3
    FFTW(plan) pForLeft, pForRight ;
    FFTW(plan) pFor, pBak ;
#else /* HAVE_FFTW3 */
    rfftw_plan pFor, pBak ;
#endif /* HAVE_FFTW3 */
    int framenum = 0 ;
    double alpha ;
    double s_amount ; /* amount, reduced to account for oversampling
                         due to smoothness */
                         

    start_timer() ;
    current = first_sample ;

    push_status_text("Denoising audio") ;
    update_status_bar(0.0,STATUS_UPDATE_INTERVAL,TRUE) ;

#ifdef HAVE_FFTW3
    pFor =
        FFTW(plan_r2r_1d)(pDnprefs->FFT_SIZE, windowed, out, FFTW_R2HC, FFTW_ESTIMATE);
    pBak =
        FFTW(plan_r2r_1d)(pDnprefs->FFT_SIZE, out, windowed, FFTW_HC2R, FFTW_ESTIMATE);
#endif /* HAVE_FFTW3 */


#ifdef HAVE_FFTW3
    pForLeft =
        FFTW(plan_r2r_1d)(pDnprefs->FFT_SIZE, left, tmp, FFTW_R2HC, FFTW_ESTIMATE);
    pForRight =
        FFTW(plan_r2r_1d)(pDnprefs->FFT_SIZE, right, tmp, FFTW_R2HC, FFTW_ESTIMATE);
#else /* HAVE_FFTW3 */
    pFor = rfftw_create_plan(pDnprefs->FFT_SIZE, FFTW_REAL_TO_COMPLEX, FFTW_ESTIMATE);
    pBak = rfftw_create_plan(pDnprefs->FFT_SIZE, FFTW_COMPLEX_TO_REAL, FFTW_ESTIMATE);
#endif /* HAVE_FFTW3 */

    alpha = welty_alpha(0.5, 1.0/(double)pDnprefs->smoothness) ;
    alpha = 1.0 ;

    for(k = 0 ; k < pDnprefs->FFT_SIZE ; k++) {
	window_coef[k] = fft_window(k,pDnprefs->FFT_SIZE, pDnprefs->window_type) ;
	d_print("k:%d wc:%lg\n", k, window_coef[k]) ;
	left_prev_frame[k] = 0.0 ;
	left[k] = 0.0 ;
	right_prev_frame[k] = 0.0 ;
	right[k] = 0.0 ;
    }


    audio_normalize(denoise_normalize) ;

    get_noise_sample(pPrefs, pDnprefs, noise_start, noise_end,
		    left_noise_min, left_noise_max, left_noise_avg,
		    right_noise_min, right_noise_max, right_noise_avg) ;

    audio_normalize(denoise_normalize) ;


/*      if(smoothness <= 4 || window_type == DENOISE_WINDOW_BLACKMAN_HYBRID)  */
	s_amount = pDnprefs->amount ;
/*      else  */
/*  	s_amount = amount/(double)(smoothness-3) ;  */

    prev_sample[0] = 0 ;
    prev_sample[1] = 0 ;

    if(pDnprefs->amount > DBL_MIN) {

	while(current <= last_sample-pDnprefs->FFT_SIZE) {
	    long n = pDnprefs->FFT_SIZE ;
	    long tmplast = current + n - 1 ;
	    gfloat p = (gfloat)(current-first_sample)/(last_sample-first_sample) ;

	    n = read_fft_real_wavefile_data(left,  right, current, current+pDnprefs->FFT_SIZE-1) ;
	    if(n < pDnprefs->FFT_SIZE) break ; /*  hit the end of all data? */

#ifdef MAC_OS_X
	    // This usleep fixes a segfault on OS X.  Rob Frohne
	    usleep(2);
#endif
	    update_status_bar(p,STATUS_UPDATE_INTERVAL,FALSE) ;

	    /*
	    */
	    if(framenum == 0) {
		for(k = 0 ; k < pDnprefs->FFT_SIZE ; k++) {
		    if(k < pDnprefs->FFT_SIZE/2) {
			d_print("OA sum %d:%lg\n", k, window_coef[k]+window_coef[k+pDnprefs->FFT_SIZE/2]) ;
			window_coef[k] = 1.0 ;
		    }
		}
		framenum = 1 ;
	    } else if(framenum == 1) {
		for(k = 0 ; k < pDnprefs->FFT_SIZE ; k++) {
		    window_coef[k] = fft_window(k,pDnprefs->FFT_SIZE, pDnprefs->window_type) ;
		}
		framenum = 2 ;
	    }

	    if(channel_mask & 0x01)
		fft_remove_noise(left, left_noise_min, left_noise_max, left_noise_avg,
		                  &pFor, &pBak, pDnprefs,0) ;
	    if(channel_mask & 0x02)
		fft_remove_noise(right, right_noise_min, right_noise_max, right_noise_avg,
		                  &pFor, &pBak, pDnprefs,1) ;

	    if(pDnprefs->window_type == DENOISE_WINDOW_HANNING_OVERLAP_ADD) {
		for(k = 0 ; k < pDnprefs->FFT_SIZE/2 ; k++) {
		    if(channel_mask & 0x01) {
			/* add in the last half of the previous output frame */
			left[k] += left_prev_frame[k+pDnprefs->FFT_SIZE/2] ;

			/* save the last half of this output frame */
			left_prev_frame[k+pDnprefs->FFT_SIZE/2] = left[k+pDnprefs->FFT_SIZE/2] ;
		    }

		    if(channel_mask & 0x02) {
			/* add in the last half of the previous output frame */
			right[k] += right_prev_frame[k+pDnprefs->FFT_SIZE/2] ;

			/* save the last half of this output frame */
			right_prev_frame[k+pDnprefs->FFT_SIZE/2] = right[k+pDnprefs->FFT_SIZE/2] ;
		    }
		}
/*  		write_fft_real_wavefile_data(left, right, current, MIN(tmplast, last_sample)) ;  */
		write_fft_real_wavefile_data(left, right, current, current+pDnprefs->FFT_SIZE/2-1) ;

		current += pDnprefs->FFT_SIZE/2 ;
	    } else {
		write_fft_real_wavefile_data(left, right, current, MIN(tmplast, last_sample)) ;

		if(pDnprefs->window_type == DENOISE_WINDOW_BLACKMAN)
		    current += pDnprefs->FFT_SIZE/pDnprefs->smoothness ;
		else
		    current += 3*pDnprefs->FFT_SIZE/4 ;
	    }


	}
    }

#ifdef HAVE_FFTW3
    FFTW(destroy_plan)(pForLeft);
    FFTW(destroy_plan)(pForRight);
    FFTW(destroy_plan)(pBak);
    FFTW(destroy_plan)(pFor);
#else /* HAVE_FFTW3 */
    rfftw_destroy_plan(pFor);
    rfftw_destroy_plan(pBak);
#endif /* HAVE_FFTW3 */

    audio_normalize(1) ;

    update_status_bar(0.0,STATUS_UPDATE_INTERVAL,TRUE) ;

    pop_status_text() ;

    main_redraw(FALSE, TRUE) ;

    stop_timer("DENOISE") ;

    return 0 ;
}

int print_noise_sample(struct sound_prefs *pPrefs, struct denoise_prefs *pDnprefs, long noise_start, long noise_end)
{
    int k ;
    FILE *fp ;

    fftw_real left_noise_max[DENOISE_MAX_FFT], right_noise_max[DENOISE_MAX_FFT], left_noise_avg[DENOISE_MAX_FFT] ;
    fftw_real left_noise_min[DENOISE_MAX_FFT], right_noise_min[DENOISE_MAX_FFT], right_noise_avg[DENOISE_MAX_FFT] ;

    get_noise_sample(pPrefs, pDnprefs, noise_start, noise_end,
		    left_noise_min, left_noise_max, left_noise_avg,
		    right_noise_min, right_noise_max, right_noise_avg) ;

    fp = fopen("noise.dat", "w") ;

    fprintf(stderr, "FFT_SIZE in print_noise_sample is %d\n", pDnprefs->FFT_SIZE) ;

    for(k = 1 ; k <= pDnprefs->FFT_SIZE/2 ; k++) {
	double freq = (double)pPrefs->rate / 2.0 /(double)(pDnprefs->FFT_SIZE/2)*(double)k ;
	fprintf(fp, "%10lg %lg %lg\n", freq, left_noise_avg[k], right_noise_avg[k]) ;
    }

    fclose(fp) ;

    return 0 ;
}

void get_noise_sample(struct sound_prefs *pPrefs, struct denoise_prefs *pDnprefs, long noise_start, long noise_end,
		    fftw_real *left_noise_min, fftw_real *left_noise_max, fftw_real *left_noise_avg,
		    fftw_real *right_noise_min, fftw_real *right_noise_max, fftw_real *right_noise_avg)
{
    int i, k ;
#ifdef HAVE_FFTW3
    FFTW(plan) pForLeft, pForRight ;
#else /* HAVE_FFTW3 */
    rfftw_plan pFor, pBak ;
#endif /* HAVE_FFTW3 */

    fftw_real left[DENOISE_MAX_FFT], right[DENOISE_MAX_FFT] ;
    fftw_real tmp[DENOISE_MAX_FFT] ;

#ifdef HAVE_FFTW3
    pForLeft =
        FFTW(plan_r2r_1d)(pDnprefs->FFT_SIZE, left, tmp, FFTW_R2HC, FFTW_ESTIMATE);
    pForRight =
        FFTW(plan_r2r_1d)(pDnprefs->FFT_SIZE, right, tmp, FFTW_R2HC, FFTW_ESTIMATE);
#else /* HAVE_FFTW3 */
    pFor = rfftw_create_plan(pDnprefs->FFT_SIZE, FFTW_REAL_TO_COMPLEX, FFTW_ESTIMATE);
    pBak = rfftw_create_plan(pDnprefs->FFT_SIZE, FFTW_COMPLEX_TO_REAL, FFTW_ESTIMATE);
#endif /* HAVE_FFTW3 */

    for(k = 0 ; k < pDnprefs->FFT_SIZE ; k++) {
	window_coef[k] = fft_window(k,pDnprefs->FFT_SIZE, pDnprefs->window_type) ;
	left_noise_max[k] = 0.0 ;
	right_noise_max[k] = 0.0 ;
	left_noise_avg[k] = 0.0 ;
	right_noise_avg[k] = 0.0 ;
	left_noise_min[k] = DBL_MAX ;
	right_noise_min[k] = DBL_MAX ;
    }

    audio_normalize(denoise_normalize) ;

    for(i = 0 ; i < pDnprefs->n_noise_samples ; i++) {
	long first = noise_start + i*(noise_end-noise_start-pDnprefs->FFT_SIZE)/pDnprefs->n_noise_samples ;

	read_fft_real_wavefile_data(left,  right, first, first+pDnprefs->FFT_SIZE-1) ;

	for(k = 0 ; k < pDnprefs->FFT_SIZE ; k++) {
	    left[k] *= window_coef[k] ;
	    right[k] *= window_coef[k] ;
	}

#ifdef HAVE_FFTW3
	FFTW(execute)(pForLeft);
#else /* HAVE_FFTW3 */
	rfftw_one(pFor, left, tmp);
#endif /* HAVE_FFTW3 */

	/* convert noise sample to power spectrum */
	for(k = 1 ; k <= pDnprefs->FFT_SIZE/2 ; k++) {
	    if(k < pDnprefs->FFT_SIZE/2) {
		left_noise_min[k] = MIN(left_noise_min[k], tmp[k] * tmp[k] + tmp[pDnprefs->FFT_SIZE-k]*tmp[pDnprefs->FFT_SIZE-k]) ;
		left_noise_max[k] = MAX(left_noise_max[k], tmp[k] * tmp[k] + tmp[pDnprefs->FFT_SIZE-k]*tmp[pDnprefs->FFT_SIZE-k]) ;
		left_noise_avg[k] += tmp[k] * tmp[k] + tmp[pDnprefs->FFT_SIZE-k]*tmp[pDnprefs->FFT_SIZE-k] ;
	    } else {
		/* Nyquist Frequency */
		left_noise_min[k] = MIN(left_noise_min[k], tmp[k] * tmp[k]) ;
		left_noise_max[k] = MAX(left_noise_max[k], tmp[k] * tmp[k]) ;
		left_noise_avg[k] += tmp[k] * tmp[k] ;
	    }
	}

#ifdef HAVE_FFTW3
	FFTW(execute)(pForRight);
#else /* HAVE_FFTW3 */
	rfftw_one(pFor, right, tmp);
#endif /* HAVE_FFTW3 */

	/* convert noise sample to power spectrum */
	for(k = 1 ; k <= pDnprefs->FFT_SIZE/2 ; k++) {
	    if(k < pDnprefs->FFT_SIZE/2) {
		right_noise_min[k] = MIN(right_noise_min[k], tmp[k] * tmp[k] + tmp[pDnprefs->FFT_SIZE-k]*tmp[pDnprefs->FFT_SIZE-k]) ;
		right_noise_max[k] = MAX(right_noise_max[k], tmp[k] * tmp[k] + tmp[pDnprefs->FFT_SIZE-k]*tmp[pDnprefs->FFT_SIZE-k]) ;
		right_noise_avg[k] += tmp[k] * tmp[k] + tmp[pDnprefs->FFT_SIZE-k]*tmp[pDnprefs->FFT_SIZE-k] ;
	    } else {
		/* Nyquist Frequency */
		right_noise_min[k] = MIN(right_noise_min[k], tmp[k] * tmp[k]) ;
		right_noise_max[k] = MAX(right_noise_max[k], tmp[k] * tmp[k]) ;
		right_noise_avg[k] += tmp[k] * tmp[k] ;
	    }
	}

    }


    /* average out the power spectrum samples */
    for(k = 1 ; k <= pDnprefs->FFT_SIZE/2 ; k++) {
	left_noise_avg[k] /= (double)pDnprefs->n_noise_samples ;
	right_noise_avg[k] /= (double)pDnprefs->n_noise_samples ;
    }

    if(pDnprefs->freq_filter) {
	if(pDnprefs->estimate_power_floor) {
	    double half_freq_w = (pDnprefs->max_sample_freq - pDnprefs->min_sample_freq)/2.0 ;
	    double sum_left = 0.0 ;
	    double n_left = 0.0 ;
	    double sum_right = 0.0 ;
	    double n_right = 0.0 ;
	    for(k = 1 ; k <= pDnprefs->FFT_SIZE/2 ; k++) {
		double freq = (double)pPrefs->rate / 2.0 /(double)(pDnprefs->FFT_SIZE/2)*(double)k ;
		if(freq < pDnprefs->min_sample_freq && freq > pDnprefs->min_sample_freq-half_freq_w) {
		    sum_left += left_noise_avg[k] ;
		    n_left += 1.0 ;
		    sum_right += right_noise_avg[k] ;
		    n_right += 1.0 ;
		}
		if(freq > pDnprefs->max_sample_freq && freq < pDnprefs->max_sample_freq+half_freq_w) {
		    sum_left += left_noise_avg[k] ;
		    n_left += 1.0 ;
		    sum_right += right_noise_avg[k] ;
		    n_right += 1.0 ;
		}
	    }
	    if(n_left > 1.e-30) sum_left /= n_left ;
	    if(n_right > 1.e-30) sum_right /= n_left ;
	    for(k = 1 ; k <= pDnprefs->FFT_SIZE/2 ; k++) {
		double freq = (double)pPrefs->rate / 2.0 /(double)(pDnprefs->FFT_SIZE/2)*(double)k ;
		if(freq < pDnprefs->min_sample_freq || freq > pDnprefs->max_sample_freq) {
		    left_noise_avg[k] -= sum_left ;
		    right_noise_avg[k] -= sum_right ;
		}
	    }

	}
	for(k = 1 ; k <= pDnprefs->FFT_SIZE/2 ; k++) {
	    double freq = (double)pPrefs->rate / 2.0 /(double)(pDnprefs->FFT_SIZE/2)*(double)k ;
	    if(freq < pDnprefs->min_sample_freq || freq > pDnprefs->max_sample_freq) {
		left_noise_avg[k] = 0.0 ;
		right_noise_avg[k] = 0.0 ;
	    }
	}
    }

#ifdef HAVE_FFTW3
    FFTW(destroy_plan)(pForLeft);
    FFTW(destroy_plan)(pForRight);
#else /* HAVE_FFTW3 */
    rfftw_destroy_plan(pFor);
    rfftw_destroy_plan(pBak);
#endif /* HAVE_FFTW3 */

    audio_normalize(1) ;
}
