/*
 * /etc/network/interfaces parser
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#define APPNAME PACKAGE
#else
#warning No config.h found: using fallback values
#define APPNAME __FILE__
#define VERSION "unknown"
#endif

#include "IfaceParser.h"
#include "stringf.h"
#include "Regexp.h"
#include "Environment.h"

#include <map>

using namespace std;
using namespace stringf;

class Tokenizer
{
protected:
	std::string str;
	std::string::size_type s;
public:
	Tokenizer(const std::string& str) throw ()
		: str(str), s(0) {}

	std::string next()
	{
		// Skip leading spaces
		while (s < str.size() && isspace(str[s]))
			s++;

		if (s == str.size()) return string();
		
		string::size_type start = s;

		while (s < str.size() && !isspace(str[s]))
			s++;

		return str.substr(start, s - start);
	}
};

/* Parse the input from `input'
 * To make it simple, use regexps on input lines instead of implementing a real
 * parser.
 */
void IfaceParser::parse(FILE* input, ScanConsumer& sc)
	throw (Exception)
{
#define ATLINESTART "^[[:blank:]]*(guessnet[0-9]*[[:blank:]]+)?test[0-9]*(-| )"
#define MACPATTERN "[0-9A-Fa-f]{2}:[0-9A-Fa-f]{2}:[0-9A-Fa-f]{2}:[0-9A-Fa-f]{2}:[0-9A-Fa-f]{2}:[0-9A-Fa-f]{2}"
	ExtendedRegexp null_line("^[[:blank:]]*(#.*)?$");
	ExtendedRegexp iface_line(
		"^[[:blank:]]*iface[[:blank:]]+"
		"([^[:blank:]]+)[[:blank:]]+"
		"([^[:blank:]]+)[[:blank:]]+"
		"([^[:blank:]]+)[[:blank:]]*$", 4);
	ExtendedRegexp peer_line(
		ATLINESTART "peer[[:blank:]]+(.+)$", 4);
	ExtendedRegexp cable_line(
		ATLINESTART "missing-cable[[:blank:]]*([[:blank:]]+.+)?$");
	ExtendedRegexp script_line(
		ATLINESTART "(script|command)[[:blank:]]+(.+)$", 5);
	ExtendedRegexp dhcp_line(
		ATLINESTART "dhcp[[:blank:]]*([[:blank:]]+.+)?$");
	ExtendedRegexp pppoe_line(
		ATLINESTART "pppoe[[:blank:]]*([[:blank:]]+.+)?$");
	ExtendedRegexp wireless_mac_essid_line(
		ATLINESTART "wireless[[:blank:]]+mac[[:blank:]]+([^[:blank:]]+)[[:blank:]]+essid[[:blank:]](.+)$", 5);
	ExtendedRegexp wireless_mac_line(
		ATLINESTART "wireless[[:blank:]]+mac[[:blank:]]+([^[:blank:]]+)[[:blank:]]*$", 4);
	ExtendedRegexp wireless_essid_line(
		ATLINESTART "wireless[[:blank:]]+essid[[:blank:]](.+)$", 4);
	ExtendedRegexp old_default_line(
		"^[[:blank:]]*guessnet[[:blank:]]+"
		"default[[:blank:]]*$");
	ExtendedRegexp parm_line(
		"^[[:blank:]]*([^[:blank:]]+)[[:blank:]]+(.+)$", 2);
	string profileName;

	string line;
	int linenum = 1;
	int found = 0;
	int c;
	while ((c = fgetc(input)) != EOF)
	{
		if (c != '\n')
			line += c;
		else
		{
			if (null_line.match(line))
			{
				//fprintf(stderr, "EMPTY\n");
			}
			else if (iface_line.match(line))
			{
				//string name(line, parts[1].rm_so, parts[1].rm_eo - parts[1].rm_so);
				//string net(line, parts[2].rm_so, parts[2].rm_eo - parts[2].rm_so);
				//string type(line, parts[3].rm_so, parts[3].rm_eo - parts[3].rm_so);

				//fprintf(stderr, "IFACE: %.*s/%.*s/%.*s\n", PFSTR(name), PFSTR(net), PFSTR(type));
				profileName = iface_line[1];
			}
			else if (peer_line.match(line))
			{
				//string ip(line, parts[1].rm_so, parts[1].rm_eo - parts[1].rm_so);
				//string mac(line, parts[2].rm_so, parts[2].rm_eo - parts[2].rm_so);
				//fprintf(stderr, "PEER: %.*s/%.*s\n", PFSTR(ip), PFSTR(mac));
				if (profileName.size())
				{
					string argstr = peer_line[3];

					// split in a map of key->val
					map<string, string> args;
					string key;
					string val;
					enum { SKEY, KEY, SVAL, VAL } state = KEY;
					for (string::const_iterator s = argstr.begin();
							s != argstr.end(); s++)
					{
						//debug("Read `%c', state: %d\n", *s, (int)state);
						if (isspace(*s))
							switch (state)
							{
								case SKEY: break;
								case KEY: state = SVAL; break;
								case SVAL: break;
								case VAL:
									state = SKEY;
									//debug("Found args: %.*s: %.*s\n", PFSTR(key), PFSTR(val));
									args.insert(make_pair(key, val));
									key = string();
									val = string();
									break;
							}
						else
							switch (state)
							{
								case SKEY: key += *s; state = KEY; break;
								case KEY: key += *s; break;
								case SVAL: val += *s; state = VAL; break;
								case VAL: val += *s; break;
							}
					}
					if (key.size() > 0 && val.size() > 0)
						args.insert(make_pair(key, val));

					map<string, string>::const_iterator ip = args.find("address");
					map<string, string>::const_iterator mac = args.find("mac");
					map<string, string>::const_iterator src = args.find("source");

					if (ip != args.end())
					{
						IPAddress ipAddr(ip->second);

						struct ether_addr macAddr;
						if (mac != args.end())
							parse_mac(&macAddr, mac->second);
						else {
							debug("Missing mac at line %d: only check for the IP\n", linenum);
							bzero(&macAddr, sizeof(struct ether_addr));
						}

						if (src != args.end())
							sc.handleScan(new PeerScan(profileName, macAddr, ipAddr, IPAddress(src->second)));
						else
							sc.handleScan(new PeerScan(profileName, macAddr, ipAddr));
						found++;
					} else {
						warning("Missing address at line %d: skipping line\n", linenum);
					}
				}
			}
			else if (cable_line.match(line))
			{
				if (profileName.size())
				{
					sc.handleScan(new LinkBeatScan(profileName));
					found++;
				}
			}
			else if (script_line.match(line))
			{
				if (profileName.size())
				{
					sc.handleScan(new ScriptScan(profileName, script_line[4]));
					found++;
				}
			}
			else if (dhcp_line.match(line))
			{
				if (profileName.size())
				{
					sc.handleScan(new DHCPScan(profileName));
					found++;
				}
			}
#ifdef HAVE_PPPOE
			else if (pppoe_line.match(line))
			{
				if (profileName.size())
				{
					sc.handleScan(new ScriptScan(profileName, string(PPPOE " -I ") + Environment::get().iface() + " -A >/dev/null 2>&1"));
					found++;
				}
			}
#endif
			else if (wireless_mac_essid_line.match(line))
			{
				if (profileName.size())
				{
					sc.handleScan(new ScriptScan(profileName, string(SCRIPTDIR "/test-wireless ") + Environment::get().iface() + " mac " + wireless_mac_essid_line[3] + " essid \"" + wireless_mac_essid_line[4] + "\"" ));
					found++;
				}
			}
			else if (wireless_mac_line.match(line))
			{
				if (profileName.size())
				{
					sc.handleScan(new ScriptScan(profileName, string(SCRIPTDIR "/test-wireless ") + Environment::get().iface() + " mac " + wireless_mac_line[3] ));
					found++;
				}
			}
			else if (wireless_essid_line.match(line))
			{
				if (profileName.size())
				{
					sc.handleScan(new ScriptScan(profileName, string(SCRIPTDIR "/test-wireless ") + Environment::get().iface() + " essid \"" + wireless_essid_line[3] + "\"" ));
					found++;
				}
			}
			else if (old_default_line.match(line))
			{
				warning("line %d: Use of \"guessnet default\" lines is obsolete and will be discontinued in the future.  Use \"map default: profile\" in the \"mapping\" section instead.\n", linenum);
				//fprintf(stderr, "DEFAULT\n");
				if (profileName.size())
				{
					//debug("Will use tag %.*s as default\n", PFSTR(profileName));

					Environment::get().defprof(profileName);
				}
			}
			else if (parm_line.match(line))
			{
				//string name(line, parts[1].rm_so, parts[1].rm_eo - parts[1].rm_so);
				//string parms(line, parts[2].rm_so, parts[2].rm_eo - parts[2].rm_so);
				//fprintf(stderr, "PARM: %.*s/%.*s\n", PFSTR(name), PFSTR(parms));
			}
			else
			{
				warning("Parse error at line %d: line ignored\n", linenum);
			}
			line = string();
			linenum++;
		}
	}
	debug("%d candidates found in input\n", found);
}

// vim:set ts=4 sw=4:
