/*
 * Thread to capture packets from a network
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include "NetWatcher.h"

#include "Thread.h"
#include "Mutex.h"

extern "C" {
#include <pcap.h>
#include <libnet.h>
}

#include <list>
#include <queue>

#include <netinet/in.h>	// ntohs, htons, ...

#include "Environment.h"


using namespace std;

class NetWatcherImpl : public Thread
{
protected:
	int _ref;
	string iface;
	pcap_t *pcap_interface;
	bool _canceled;

	Mutex listenersMutex;

	list<PacketListener*> listeners_ethernet;
	list<PacketListener*> listeners_arp;

	struct libnet_link_int* link_interface;

	virtual void* main() throw ();

public:
	NetWatcherImpl(const string& iface) throw (SystemException, pcapException)
		: _ref(0), iface(iface), _canceled(false)
	{
		char errbuf[PCAP_ERRBUF_SIZE];

		// TODO: Try to use 1 from "promisc": maybe there won't be a need to setup an
		// address on the interface
		const int captureSize = LIBNET_ETH_H + (LIBNET_ARP_H >? LIBNET_IPV4_H);
		if (!(pcap_interface = pcap_open_live (
						(char*)iface.c_str(), captureSize, 1, 500, errbuf)))
			throw pcapException (errbuf, "initializing pcap packet capture library");
	}
	~NetWatcherImpl() throw ()
	{
		pcap_close(pcap_interface);
	}

	bool canceled() const throw () { return _canceled; }
	bool canceled(bool val) throw () { return _canceled = val; }

	/// Increment the reference count for this object
	void ref() throw () { ++_ref; }

	/// Decrement the reference count for this object, returning true when it
	/// reaches 0
	bool unref() throw () { return --_ref == 0; }

	void addARPListener(PacketListener* pl) throw ()
	{
		MutexLock lock(listenersMutex);
		listeners_arp.push_back(pl);
	}

	void addEthernetListener(PacketListener* pl) throw ()
	{
		MutexLock lock(listenersMutex);
		listeners_ethernet.push_back(pl);
	}
};

void* NetWatcherImpl::main() throw ()
{
	struct pcap_pkthdr pcap_header;

	// Let the signals be caught by some other process
	sigset_t sigs, oldsigs;
	sigfillset(&sigs);
	sigdelset(&sigs, SIGFPE);
	sigdelset(&sigs, SIGILL);
	sigdelset(&sigs, SIGSEGV);
	sigdelset(&sigs, SIGBUS);
	sigdelset(&sigs, SIGABRT);
	sigdelset(&sigs, SIGIOT);
	sigdelset(&sigs, SIGTRAP);
	sigdelset(&sigs, SIGSYS);
	pthread_sigmask(SIG_SETMASK, &sigs, &oldsigs);

	try {
		while (true)
		{
			unsigned char* packet = (u_char *)pcap_next (pcap_interface, &pcap_header);
			if (packet == 0)
				throw pcapException (pcap_geterr(pcap_interface), "getting a new packet from the network");

			struct libnet_ethernet_hdr* packet_header = (struct libnet_ethernet_hdr *)packet;
			MutexLock lock(listenersMutex);
			if (!listeners_ethernet.empty())
				for (list<PacketListener*>::iterator i = listeners_ethernet.begin();
						i != listeners_ethernet.end(); i++)
					(*i)->handleEthernet(packet_header);
			if (ntohs (packet_header->ether_type) == ETHERTYPE_ARP)
			{
				if (!listeners_arp.empty())
				{
					struct libnet_arp_hdr* arp_header = (struct libnet_arp_hdr *)(packet + LIBNET_ETH_H);
					for (list<PacketListener*>::iterator i = listeners_arp.begin();
							i != listeners_arp.end(); i++)
						(*i)->handleARP(arp_header);
				}
			}
		}
	} catch (Exception& e) {
		error("%s: %.*s.  Quitting NetWatcher thread.\n", e.type(), PFSTR(e.desc()));
	}
}



NetWatcher::NetWatcher(const string& iface) throw (SystemException, pcapException)
{
	impl = new NetWatcherImpl(iface);
	impl->ref();
	impl->start();
}

NetWatcher::NetWatcher(const NetWatcher& ns) throw ()
{
	if (ns.impl)
		ns.impl->ref();
	impl = ns.impl;
}

NetWatcher::~NetWatcher() throw (SystemException)
{
	if (impl && impl->unref())
	{
		shutdown();
		delete impl;
	}
}

NetWatcher& NetWatcher::operator=(const NetWatcher& ns) throw (SystemException)
{
	if (ns.impl)
		ns.impl->ref();  // Do it early to correctly handle the case of x = x;
	if (impl && impl->unref())
	{
		shutdown();
		delete impl;
	}
	impl = ns.impl;
	return *this;
}

void NetWatcher::addARPListener(PacketListener* pl) throw ()
{
	return impl->addARPListener(pl);
}

void NetWatcher::addEthernetListener(PacketListener* pl) throw ()
{
	return impl->addEthernetListener(pl);
}

void NetWatcher::shutdown() throw ()
{
	if (!impl->canceled())
		try {
			impl->cancel();
			impl->join();
			impl->canceled(true);
		} catch (SystemException& e) {
			warning("%s: %.*s when shutting down NetWatcher\n", e.type(), PFSTR(e.desc()));
		}
}

// vim:set ts=4 sw=4:
