/* ----------------------------------------------------------------------------
 * gtkpbinit.c
 * funtions that prepare program workflow and initialization
 *
 * Copyright 2002 Matthias Grimm (joker@cymes.de).
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * ----------------------------------------------------------------------------*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <syslog.h>
#include <signal.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <getopt.h>
#include <sys/stat.h>

#include <pbb.h>

#include "gtkpbinit.h"
#include "gtkinterface.h"
#include "audio.h"
#include "themes.h"
#include "gettext_macros.h"

extern volatile int prgexit;

/* This function does the first initialization of the init-structure. Some values
   are used very early to decide if some cleanup is nessecary or not. */
void init_init(struct init *md)
{
	md->themename = DEFAULT_THEME;
	md->themedata = NULL;
	md->timeout = 0;

	md->dsp.active = 0;
	md->dsp.audiodev = DEFAULT_AUDIO;
	md->dsp.sample = NULL;   /* no sound to play */
}

int prg_init(struct init *md)
{
	int err;

	if ((md->themedata = theme_init(md->themename)) == NULL)
		return 1;

	if ((md->window = popup_create_window(md->themedata)) == NULL)
		return 1;

	if ((err = ipc_init (LIBMODE_CLIENT, 1)) != 0) {        /* setup IPC subsystem for clients */
		if (err == E_NOSERVER)
			print_error (_("ERROR: Server message port not found. Server isn't running.\n"));
		else if (err == E_REGISTER)
			print_error (_("ERROR: Client list full. Server doesn't accept more clients.\n"));
		else if (err == E_MSGPORT)
			print_error (_("ERROR: Can't create message port for client.\n"));
		return err;
	}

#ifdef HAVE_SOUND
	if (md->themedata->havesound) {
		if ((init_sound_thread(&md->dsp)) == -1)
			print_error(_("WARNING: Problems initializing sound subsystem. Beep disabled.\n"));
		else
			md->dsp.active = 1;
	}
#endif
	return 0;
}

/* --- Signal Handler --- */

void
signalhandler (int signum)
{
	prgexit = 1;            /* cleanup and exit the program */
}

int
install_sighandler ()
{
	struct sigaction sa = { {signalhandler}, {{0}}, SA_RESTART, 0 };

	/* install the signal handlers */
	if (sigaction (SIGINT, &sa, NULL)) {
		print_error(_("ERROR: Can't install SIGINT handler\n"));
		return -1;
	}
	if (sigaction (SIGTERM, &sa, NULL)) {
		print_error(_("ERROR: Can't install SIGTERM handler\n"));
		return -1;
	}
	return 0;
}

int evaluate_args(struct init *md, int argc, char *argv[])
{
	struct option const long_options[] = {
		  {"help", no_argument, 0, ARG_HELP},
		  {"version", no_argument, 0, ARG_VERSION},
		  {"audio", required_argument, 0, ARG_AUDIODEV},
		  {"detach", optional_argument, 0, ARG_DETACH},
		  {"theme", required_argument, 0, ARG_THEME},
		  {NULL, 0, NULL, 0}
	};
	int c, err;
	char *optarg2, *prgname;

	if((prgname = strrchr(argv[0],'/')) == NULL)
		prgname = argv[0];
	else prgname++;		/* ignore first slash*/
	argv[0] = prgname;

	while ((c = getopt_long (argc, argv, ARG_ALL, long_options, (int *) 0)) != EOF) {
		switch (c) {
			case ARG_VERSION:
				printf(_("%s, version %s"), PACKAGE, VERSION);
				printf(", (c) 2002/2003 Matthias Grimm\n");
				return E_INFO;
			case ARG_DETACH:
				optarg2 = (optarg) ? optarg : argv[optind];
				if (optarg2 == NULL || optarg2[0] == '-')
					optarg2 = DEFAULT_PIDFILE;
				prepare_daemon (prgname, optarg2, PBBDF_FORCE);
				break;
			case ARG_AUDIODEV:
				if((err = check_path (md->dsp.audiodev = optarg, TYPE_CHARDEV)))
					return err;
				break;
			case ARG_THEME:
				if ((err = is_theme(optarg)) != 0)
					printf(_("ERROR: %s is not a theme for %s. Using default.\n"), optarg, PACKAGE);
				else
					md->themename = optarg;
				break;
			case ARG_HELP:
			default:
				printf(_("%s - visualization client for pbbuttonsd, support for special laptop functions.\n"), prgname);
				printf(_("Usage: %s [OPTION]... \n"), prgname);
				printf (_("Options:\n"
					"   -%c, --help             display this help and exit\n"
					"   -%c, --version          display version information and exit\n"
					"   -%c, --detach[=PIDFILE] start %s as background process and\n"
					"                          optional use an alternative pid-file\n"
					"                          (default: %s)\n"
					"   -%c, --audio=DSP        use alternative audio device\n"
					"                          (default: %s)\n"
					"   -%c, --theme=THEME      set theme, that should be used.\n"
					"                          (default %s)\n"),
					ARG_HELP, ARG_VERSION, ARG_DETACH, prgname, DEFAULT_PIDFILE,
					ARG_AUDIODEV, md->dsp.audiodev, ARG_THEME, DEFAULT_THEME);
				return E_INFO;
		}
	}
	return 0;
}

int
check_path (char *source, int type)
{
	int rc = 0;

	rc = check_devorfile (source, type);
	switch (rc) {
	case E_NOEXIST:
		print_error(_("ERROR: Can't access %s: %s.\n"), source, strerror(errno));
		break;
	case E_NOFILE:
		print_error(_("ERROR: %s is not a file.\n"), source);
		break;
	case E_NOCHAR:
		print_error(_("ERROR: %s is not a character device.\n"), source);
		break;
	case E_NOBLK:
		print_error(_("ERROR: %s is not a block device.\n"), source);
		break;
	}
	return rc;
}

