/*
 *   Copyright (C) 2007-2008 Tristan Heaven <tristanheaven@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License along
 *   with this program; if not, write to the Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#endif

#include <stdlib.h>
#include <stdbool.h>
#include <gtk/gtk.h>

#include "gui.h"
#include "main.h"
#include "hash.h"
#include "prefs.h"
#include "list.h"

void on_window_delete_event(GtkWidget *widget, GdkEvent *event, void *data)
{
	exit(EXIT_SUCCESS);
}

void on_window_destroy(GtkObject *object, void *data)
{
	exit(EXIT_SUCCESS);
}

void on_window_size_request(GtkWidget *widget, GtkRequisition *requisition,
	void *data)
{
	if (gui_window_maximised())
		return;

	if (gui_view() == VIEW_FILE_LIST) {
		int width, height;
		gtk_window_get_size(GTK_WINDOW(widget), &width, &height);
		prefs.width = width;
		prefs.height = height;
	}
}

void on_menuitem_file_activate(GtkAction *action, void *data)
{
	if (gui.busy) {
		gtk_action_set_sensitive(gui_get_action("menuitem_save_as"), false);
		return;
	}

	switch (gui_view()) {
		case VIEW_FILE:
		case VIEW_TEXT:
			for (int i = 0; i < HASH_N; i++) {
				if (gtk_toggle_button_get_active(hash[i].button) &&
					*gtk_entry_get_text(hash[i].entry))
				{
					gtk_action_set_sensitive(gui_get_action("menuitem_save_as"), true);
					return;
				}
			}
			break;
		case VIEW_FILE_LIST:
			for (int i = 0; i < HASH_N; i++) {
				if (gtk_toggle_button_get_active(hash[i].button)) {
					char *digest = list_get_digest(0, i);
					if (digest != NULL && *digest) {
						gtk_action_set_sensitive(gui_get_action("menuitem_save_as"), true);
						g_free(digest);
						return;
					}
				}
			}
			break;
		default:
			g_assert_not_reached();
	}

	gtk_action_set_sensitive(gui_get_action("menuitem_save_as"), false);
}

void on_menuitem_save_as_activate(GtkAction *action, void *data)
{
	// Create save dialog
	GtkFileChooser *chooser = GTK_FILE_CHOOSER(
		gtk_file_chooser_dialog_new(_("Save Digests To File"), NULL,
		GTK_FILE_CHOOSER_ACTION_SAVE,
		GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
		GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
		NULL));
	gtk_file_chooser_set_do_overwrite_confirmation(chooser, true);

	if (gtk_dialog_run(GTK_DIALOG(chooser)) == GTK_RESPONSE_ACCEPT) {
		// Open dialog
		char *filename = gtk_file_chooser_get_filename(chooser);
		GString *string = g_string_sized_new(0);
		char *data;

		for (int i = 0; i < HASH_N; i++) {
			if (gtk_toggle_button_get_active(hash[i].button)) {
				g_string_append_printf(string, "# %s\n", hash[i].name);
				switch (gui_view()) {
					case VIEW_FILE: {
						// Save file digest
						char *path = gtk_file_chooser_get_filename(
							GTK_FILE_CHOOSER(gui_get_widget("filechooserbutton")));
						char *base = g_path_get_basename(path);
						g_free(path);
						g_string_append_printf(string, "%s  %s\n",
							gtk_entry_get_text(hash[i].entry), base);
						g_free(base);
						break; }
					case VIEW_TEXT:
						// Save text digest
						g_string_append_printf(string, "%s  \"%s\"\n",
							gtk_entry_get_text(hash[i].entry),
							gtk_entry_get_text(GTK_ENTRY(gui_get_widget("entry"))));
						break;
					case VIEW_FILE_LIST:
						// Save list digests
						for (unsigned int row = 0; row < list_count_rows(); row++) {
							char *digest = list_get_digest(row, i);
							char *file = list_get_basename(row);
							g_string_append_printf(string, "%s  %s\n",
								digest, file);
							g_free(digest);
							g_free(file);
						}
						break;
					default:
						g_assert_not_reached();
				}
			}
		}

		data = g_string_free(string, false);
		g_file_set_contents(filename, data, -1, NULL);

		g_free(data);
		g_free(filename);
	}

	gtk_widget_destroy(GTK_WIDGET(chooser));
}

void on_menuitem_quit_activate(GtkAction *action, void *data)
{
	exit(EXIT_SUCCESS);
}

void on_menuitem_edit_activate(GtkAction *action, void *data)
{
	GtkWidget *widget = gtk_window_get_focus(GTK_WINDOW(gui_get_widget("window")));
	bool selection, editable, clipboard;

	if (GTK_IS_ENTRY(widget)) {
		selection = gtk_editable_get_selection_bounds(
			GTK_EDITABLE(widget), NULL, NULL);
		editable = gtk_editable_get_editable(GTK_EDITABLE(widget));
		clipboard = gtk_clipboard_wait_is_text_available(
			gtk_clipboard_get(GDK_NONE));

		gtk_action_set_sensitive(gui_get_action("menuitem_cut"), selection && editable);
		gtk_action_set_sensitive(gui_get_action("menuitem_copy"), selection);
		gtk_action_set_sensitive(gui_get_action("menuitem_paste"), editable && clipboard);
		gtk_action_set_sensitive(gui_get_action("menuitem_delete"), selection && editable);
		gtk_action_set_sensitive(gui_get_action("menuitem_select_all"), true);
	} else {
		gtk_action_set_sensitive(gui_get_action("menuitem_cut"), false);
		gtk_action_set_sensitive(gui_get_action("menuitem_copy"), false);
		gtk_action_set_sensitive(gui_get_action("menuitem_paste"), false);
		gtk_action_set_sensitive(gui_get_action("menuitem_delete"), false);
		gtk_action_set_sensitive(gui_get_action("menuitem_select_all"), false);
	}
}

void on_menuitem_cut_activate(GtkAction *action, void *data)
{
	GtkEditable *widget = GTK_EDITABLE(gtk_window_get_focus(
		GTK_WINDOW(gui_get_widget("window"))));

	gtk_editable_cut_clipboard(widget);
}

void on_menuitem_copy_activate(GtkAction *action, void *data)
{
	GtkEditable *widget = GTK_EDITABLE(gtk_window_get_focus(
		GTK_WINDOW(gui_get_widget("window"))));

	gtk_editable_copy_clipboard(widget);
}

void on_menuitem_paste_activate(GtkAction *action, void *data)
{
	GtkEditable *widget = GTK_EDITABLE(gtk_window_get_focus(
		GTK_WINDOW(gui_get_widget("window"))));

	gtk_editable_paste_clipboard(widget);
}

void on_menuitem_delete_activate(GtkAction *action, void *data)
{
	GtkEditable *widget = GTK_EDITABLE(gtk_window_get_focus(
		GTK_WINDOW(gui_get_widget("window"))));

	gtk_editable_delete_selection(widget);
}

void on_menuitem_select_all_activate(GtkAction *action, void *data)
{
	GtkEditable *widget = GTK_EDITABLE(gtk_window_get_focus(
		GTK_WINDOW(gui_get_widget("window"))));

	gtk_editable_set_position(widget, -1);
	gtk_editable_select_region(widget, 0, -1);
}

void on_menuitem_prefs_activate(GtkAction *action, void *data)
{
	gtk_widget_show(gui_get_widget("dialog"));
}

void on_radiomenuitem_toggled(GtkToggleAction *toggleaction, void *data)
{
	switch (gui_view()) {
		case VIEW_FILE:
			gtk_widget_hide(gui_get_widget("toolbar"));
			gtk_widget_hide(gui_get_widget("label_text"));
			gtk_widget_hide(gui_get_widget("entry"));
			gtk_widget_hide(gui_get_widget("vbox_list"));
			gtk_widget_show(gui_get_widget("label_file"));
			gtk_widget_show(gui_get_widget("filechooserbutton"));
			gtk_widget_show(gui_get_widget("vbox_single"));

			char *filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(
				gui_get_widget("filechooserbutton")));
			gtk_widget_set_sensitive(gui_get_widget("button_hash"),
				g_file_test(filename, G_FILE_TEST_EXISTS));
			g_free(filename);

			gui_window_resizable(false);
			break;
		case VIEW_TEXT:
			gtk_widget_hide(gui_get_widget("toolbar"));
			gtk_widget_hide(gui_get_widget("label_file"));
			gtk_widget_hide(gui_get_widget("filechooserbutton"));
			gtk_widget_hide(gui_get_widget("vbox_list"));
			gtk_widget_show(gui_get_widget("label_text"));
			gtk_widget_show(gui_get_widget("entry"));
			gtk_widget_show(gui_get_widget("vbox_single"));

			gtk_widget_set_sensitive(gui_get_widget("button_hash"), true);

			gui_window_resizable(false);
			break;
		case VIEW_FILE_LIST:
			gtk_widget_hide(gui_get_widget("vbox_single"));
			gtk_widget_show(gui_get_widget("toolbar"));
			gtk_widget_show(gui_get_widget("vbox_list"));

			gtk_widget_set_sensitive(gui_get_widget("button_hash"),
				list_count_rows());

			gui_window_resizable(true);
			break;
		default:
			g_assert_not_reached();
	}

	gui_clear();
}

void on_menuitem_about_activate(GtkAction *action, void *data)
{
	const char *license = {
		"This program is free software; you can redistribute it and/or modify\n"
		"it under the terms of the GNU General Public License as published by\n"
		"the Free Software Foundation; either version 2 of the License, or\n"
		"(at your option) any later version.\n\n"
		"This program is distributed in the hope that it will be useful,\n"
		"but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
		"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
		"GNU General Public License for more details.\n\n"
		"You should have received a copy of the GNU General Public License along\n"
		"with this program; if not, write to the Free Software Foundation, Inc.,\n"
		"51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.\n"
	};

	const char *authors[] = {
		"Tristan Heaven <tristanheaven@gmail.com>",
		NULL
	};

	gtk_show_about_dialog(
			GTK_WINDOW(gui_get_widget("window")),
			"authors", authors,
			"comments", _("A GTK+ utility for computing digests using the mhash library."),
			"license", license,
			"program-name", PACKAGE_NAME,
#if ENABLE_NLS
			"translator-credits", _("translator-credits"),
#endif
			"version", VERSION,
			"website", "http://gtkhash.sourceforge.net/",
			NULL);
}

void on_filechooserbutton_selection_changed(GtkFileChooser *chooser, void *data)
{
	char *filename = gtk_file_chooser_get_filename(chooser);
	gtk_widget_set_sensitive(gui_get_widget("button_hash"),
		g_file_test(filename, G_FILE_TEST_EXISTS));
	g_free(filename);

	gui_clear();
}

void on_entry_changed(GtkEditable *editable, void *data)
{
	gui_clear();
}

void on_toolbutton_add_clicked(GtkToolButton *toolbutton, void *data)
{
	GtkFileChooser *chooser = GTK_FILE_CHOOSER(
		gtk_file_chooser_dialog_new(_("Select Files"), NULL,
		GTK_FILE_CHOOSER_ACTION_OPEN,
		GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
		GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
		NULL));
	gtk_file_chooser_set_select_multiple(chooser, true);

	if (gtk_dialog_run(GTK_DIALOG(chooser)) == GTK_RESPONSE_ACCEPT) {
		GSList *filenames = gtk_file_chooser_get_filenames(chooser);
		bool has_selected = false;

		for (unsigned int i = 0; i < g_slist_length(filenames); i++) {
			char *path = g_slist_nth(filenames, i)->data;
			list_append_row(path);
			g_free(path);
			has_selected = true;
		}

		gtk_widget_set_sensitive(gui_get_widget("button_hash"), has_selected);
		g_slist_free(filenames);
	}

	gtk_widget_destroy(GTK_WIDGET(chooser));
}

void on_toolbutton_remove_clicked(GtkToolButton *toolbutton, void *data)
{
	list_remove_selection();
}

void on_toolbutton_clear_clicked(GtkToolButton *toolbutton, void *data)
{
	list_clear();
}

void on_button_hash_clicked(GtkButton *button, void *data)
{
	bool success = true;

	gui_busy(true);
	gui_clear();

	switch (gui_view()) {
		case VIEW_FILE: {
			char *filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(
				gui_get_widget("filechooserbutton")));
			if (file_exists(filename))
				success = hash_file(filename);
			g_free(filename);
			break; }
		case VIEW_TEXT:
			for (int i = 0; i < HASH_N; i++) {
				if (gtk_toggle_button_get_active(hash[i].button)) {
					char *digest = hash_string(gtk_entry_get_text(GTK_ENTRY(
						gui_get_widget("entry"))), i);
					gtk_entry_set_text(hash[i].entry, digest);
					g_free(digest);
				}
			}
			break;
		case VIEW_FILE_LIST:
			for (unsigned int i = 0; i < list_count_rows(); i++) {
				char *filename = list_get_path(i);
				if (file_exists(filename))
					success = hash_file(filename);
				g_free(filename);
				if (!success)
					break;
			}
			break;
		default:
			g_assert_not_reached();
	}

	if (success)
		gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(
			gui_get_widget("progressbar")), 1.0);
	else
		gui_clear();

	gui_busy(false);
}

void on_button_stop_clicked(GtkButton *button, void *data)
{
	hash_abort = true;
}

void on_button_close_clicked(GtkButton *button, void *data)
{
	exit(EXIT_SUCCESS);
}

void on_dialog_delete_event(GtkWidget *widget, GdkEvent *event, void *data)
{
	gui_dialog_hide();
}

void on_dialog_button_close_clicked(GtkButton *button, void *data)
{
	gui_dialog_hide();
}
