/*
 *     gtkatlantic - the gtk+ monopd client, enjoy network monopoly games
 *
 *
 *  Copyright (C) 2002-2003 Rochet Sylvain
 *
 *  gtkatlantic is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <glib.h>

/* ENG_MAIN: main functions for engine
 *
 *
 * all int functions return 0 if error occur
 * return >0 when sucess
 *
 * all functions kill process if error occurs when allocate memory
 */


/* ---- info about: x, y, z
 *
 *           o-------- x
 *          /|
 *         / |
 *        /  |
 *       /   |
 *      z    |
 *           y
 *
 * z = ground, 0 is back plane
 *
 */


/* ---- info about: trans, mask, alphamask, alpha, bgcolor
 *
 * transp: set a transparency color
 *
 * mask: set a mask img (this mask was created with:  pic_create()
 *
 * alphamask: set a alphamask img (this mask was created with:  pic_create()
 *
 * alpha: set an uniform alpha
 *
 * colorbg: set a background color
 *
 *
 * -- you can have images :
 *
 * - image without features
 * - image with  alpha
 * - image with  alpha + bgcolor
 * - image with  alphamask
 * - image with  alphamask + bgcolor
 * - image with  mask
 * - image with  mask + bgcolor
 * - image with  transparency
 * - image with  transparency + bgcolor
 * - image with  mask + alpha
 * - image with  mask + alpha + bgcolor
 * - image with  transparency + alpha
 * - image with  transparency + alpha + bgcolor
 * - image with  alphamask + alpha (NEW)
 * - image with  alphamask + alpha + bgcolor (NEW)
 *
 */


/* ---- info about minimizing memory usage
 *
 * ==> normal mode
 *
 *   In normal mode, engine store all pictures when invoking `pic_buff();'
 * with this mode you can destroy in your own code images data :
 *
 * Example for normal mode :
 *
 *    ---------8<---------8<---------8<---------8<---------
 *    char *buff;
 *
 *    buff = malloc(size);
 *    read_image(buff, "namefile);  // just for understand
 *    pic_buff(frame, pic, buff);
 *    free(buff);
 *    ---------8<---------8<---------8<---------8<---------
 * 
 *   The problem is when you want more than one picture with same image,
 * it will allocated in engine each time new picture is filled.
 *   If this image is tiny this is not really a problem, but for big
 * images engine take quickly a lot of memory !
 * 
 * 
 * ==> minimize memory mode
 *
 * The solution is to call:
 *
 *    `engine_minimize_memory_usage();'
 *
 *   Now engine doesn't take control about memory allocation, all errors
 * you made get in trouble engine (segfault). But you can use more than one
 * time your images buffers.
 *
 *
 *   ++++ IMPORTANT (1) ++++
 *
 *     Engine need to modify mask for turn it more readable by a computer
 *   (each mask pixel on one byte, see eng_conv.h for details)
 *   
 *     For normal mode: engine make it internaly, for minimize memory mode:
 *   you do make an engine mask compatible.
 *     Just call `engine_maskcompat();' with the good arguments. Call this
 *   function only one time for each mask buffer, even if two pics use
 *   the same. Don't forget to call `pic_set_depth();' before.
 *     When calling `engine_maskcompat();' engine make modifications directly
 *   to your original buffer, please take care you don't need the original
 *   mask.
 *   
 *   ++++ IMPORTANT (2) ++++
 *
 *     Much set flag minimizing memory usage just after `engine_init();' ,
 *   you can understand why :)...
 *
 */


/* ---- info about: picture standard coding
 *
 * R = red
 * G = green
 * B = blue
 * 0 = unused
 * A = alpha
 *
 * 1  bit > black&white (often for mask)
 *        >   8 pixel per byte
 *
 * 8  bit > each color on 8bits, often indexed
 *
 * 15 bit > each pixel on 16 bit (5-5-5-1)
 *        >   RRRRRGGG GGBBBBB0
 *
 * 16 bit > each pixel on 16 bit (5-6-5)
 *        >   RRRRRGGG GGGBBBBB
 *
 * 24 bit > each pixel on 24 bits (8-8-8)
 *        >   RRRRRRRR GGGGGGGG BBBBBBBB
 *
 * 32 bit > each pixel on 24 bits (8-8-8), 8 bits unused
 *        >   RRRRRRRR GGGGGGGG BBBBBBBB 00000000
 *
 * 32 bit > each pixel on 24 bits (8-8-8), 8 bits for alpha channel
 *        >   RRRRRRRR GGGGGGGG BBBBBBBB AAAAAAAA
 */


#define MAX_FRAME 32
#define MAX_PIC_PER_FRAME 512
#define MAX_HEIGHT 0xffff
#define MAX_WIDTH 0xffff

#define ENG_DEBUG FALSE


/* -- prototypes for eng_main.c -- */
void engine_init();
void eng_init_alpha_table();
void eng_free_alpha_table();
void engine_close();
void set_lastframe();

void engine_minimize_memory_usage();
void engine_maskcompat(guint32 size, guint16 depth, guchar  * buff);

guint16 frame_create();
void frame_destroy        (guint16 frame_id);
void frame_destroy_all();
void frame_reset          (guint16 frame_id);
void frame_set_compute    (guint16 frame_id);
void frame_unset_compute  (guint16 frame_id);
void frame_clean          (guint16 frame_id);
void frame_set_height     (guint16 frame_id, guint16 height);
void frame_set_width      (guint16 frame_id, guint16 width);
void frame_set_bgcolor    (guint16 frame_id, guint32 bgcolor);
void frame_set_lastpic    (guint16 frame_id);
void frame_set_lastground (guint16 frame_id);
guint32 frame_size_frame  (guint16 frame_id);
void frame_showarg        (guint16 frame_id);
gboolean frame_test       (guint16 frame_id);

guint16 pic_create      (guint16 frame_id);
void pic_destroy        (guint16 frame_id, guint16 pic_id);
void pic_reset          (guint16 frame_id, guint16 pic_id);
void pic_show           (guint16 frame_id, guint16 pic_id);
void pic_unshow         (guint16 frame_id, guint16 pic_id);
void pic_is_mask        (guint16 frame_id, guint16 pic_id);
void pic_isnot_mask     (guint16 frame_id, guint16 pic_id);
void pic_is_alphamask   (guint16 frame_id, guint16 pic_id);
void pic_isnot_alphamask(guint16 frame_id, guint16 pic_id);
void pic_set_x          (guint16 frame_id, guint16 pic_id, guint16 x);
void pic_set_y          (guint16 frame_id, guint16 pic_id, guint16 y);
void pic_set_z          (guint16 frame_id, guint16 pic_id, guint16 z);
void pic_set_height     (guint16 frame_id, guint16 pic_id, guint16 height);
void pic_set_width      (guint16 frame_id, guint16 pic_id, guint16 width);
void pic_set_depth      (guint16 frame_id, guint16 pic_id, guint16 depth);
void pic_set_transp     (guint16 frame_id, guint16 pic_id, guint32 transp);
void pic_unset_transp   (guint16 frame_id, guint16 pic_id);
void pic_set_alpha      (guint16 frame_id, guint16 pic_id, guint8 alpha);
void pic_unset_alpha    (guint16 frame_id, guint16 pic_id);
void pic_set_mask       (guint16 frame_id, guint16 pic_id, guint16 mask_id);
void pic_unset_mask     (guint16 frame_id, guint16 pic_id);
void pic_set_alphamask  (guint16 frame_id, guint16 pic_id, guint16 alpha_id);
void pic_unset_alphamask(guint16 frame_id, guint16 pic_id);
void pic_set_bgcolor    (guint16 frame_id, guint16 pic_id, guint32 bgcolor);
void pic_unset_bgcolor  (guint16 frame_id, guint16 pic_id);
void pic_buff           (guint16 frame_id, guint16 pic_id, guchar  * buff);
void pic_fill_buff      (guint16 frame_id, guint16 pic_id, guchar  * buff);
void pic_set_buff       (guint16 frame_id, guint16 pic_id, guchar  * buff);
void pic_showarg        (guint16 frame_id, guint16 pic_id);
gboolean pic_mask_test_depth(guint16 frame_id, guint16 pic_id);
gboolean pic_test       (guint16 frame_id, guint16 pic_id);

guint32  max_col_val(guint16 depth);
guint32  max_col_val_24();
guint32  calc_buf_size(guint32 width, guint32 height, guint16 depth);
guint32  calc_buf_size_24(guint32 width, guint32 height);
/* -- end of prototypes -- */



/* ---- struct hierarchy
 *
 * engine           > global engine variables
 *   `-> alpha      > alpha tables
 * frame            > property of frames (height, width, ...)
 *   |-> zone_upd   > display zone to update (x1, y1, x2, y1)
 *   `-> obj        > property of pictures (x, y, height, width, ...)
 */



typedef struct {

	guint16 x1;
	guint16 y1;
	guint16 x2;
	guint16 y2;

} _zone_upd;



typedef struct {

	gboolean show;         // show/hide the pic
	gboolean is_mask;      // pic is a mask 0/1
	gboolean is_alphamask; // pic is a mask alpha 0/1

	guint16 x;       // x position
	guint16 y;       // y position
	guint16 z;       // z position (ground)

	guint16 width;   // width  of img
	guint16 height;  // height of img

	guint16 depth;   // color depth (bits per pixel)

	gboolean have_transp;    // have a  transparency
	gboolean have_alpha;     // have an alpha
	gboolean have_mask;      // have a  mask
	gboolean have_alphamask; // have an alphamask
	gboolean have_bgcolor;   // have a  backgroundcolor

	guint32 transp;    // transparency color
	guint8  alpha;     // alpha value
	guint16 mask;      // mask img
	guint16 alphamask; // alpha mask img
	guint8  bgcolor[3]; // background color

	guchar * buf;    // buf contain data pixel color

// PRIVATE
	guint32  memalloc;    // how much bytes are allocate for buf
	gboolean buf_fill;    // buffer was fill (0 / 1)
	guint16  depth_buf;   // color depth of buf (8, 24)
	gboolean change;      // if pic arg has changed

	guint16 x_old;        // x position old
	guint16 y_old;        // y position old

	guint16 width_old;    // width  of img old
	guint16 height_old;   // height of img old

} _obj;



typedef struct {

	gboolean active_pic[MAX_PIC_PER_FRAME +1]; // object enable(1)/disable(!1)
	gboolean compute;     // compute/or not the frame

	guint32 width;        // set the width  of out buffer
	guint32 height;       // set the height of out buffer

	guint8  bgcolor[3];   // background color

	guchar * bufout;      // bufout contain image would you want to draw
	guchar * part;        // contain part of bufout

// PRIVATE
	guint16 lastpic;     // last pic_id initialised
	guint16 lastground;  // last ground(z) set
	guint32 memalloc;    // how much mem are allocate for buf
	guint32 partalloc;   // if part is allocate or not
	gboolean gray_bgcolor; // if bgcolor is gray (R=G=B)

	guint16 nb_zone;     // number of zone to update
	guint16 x_min;       // min/max of zone to refresh
	guint16 y_min;
	guint16 x_max;
	guint16 y_max;
	_zone_upd *zone_upd[MAX_PIC_PER_FRAME +1];
	_obj *obj[MAX_PIC_PER_FRAME +1];

} _frame;

_frame *frame[MAX_FRAME +1];



typedef struct {

	guint8  fg[256][256]; // foreground [alpha][fg]
	guint8  bg[256][256]; // background [alpha][bg]

} _alpha;



typedef struct {

	gboolean active_frame[MAX_FRAME +1]; // frame enable(1)/disable(!1)

	guint16 lastframe;  // last frame_id initialise

	gboolean table_alpha_is_init; // table initialized 0/1
	_alpha *alpha;             // alpha tables

	gboolean minimize_memory; // minimize memory mode 0/1

} _engine;

_engine *engine;
