/*
 * Copyright 2005 Fluendo S.L.
 */
 /*********************************************************************
 * Adapted from dist10 reference code and used under the license therein:
 * ISO MPEG Audio Subgroup Software Simulation Group (1996)
 * ISO 13818-3 MPEG-2 Audio Decoder - Lower Sampling Frequency Extension
 **********************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "common.h"
#include "huffman.h"
#include "stdlib.h"
#include "table-huffdec.h"

#define HUFFBITS guint32 
#define HTSIZE  34
#define MXOFF   250

const HUFFBITS dmask = 1 << (sizeof (HUFFBITS) * 8 - 1);
const guint hs = sizeof (HUFFBITS) * 8;

/* do the huffman-decoding 						*/
/* note! for counta,countb -the 4 bit value is returned in y, discard x */
gboolean
huffman_decoder (huffdec_bitbuf *bb, 
    gint tnum, int *x, int *y, int *v, int *w)
{
  HUFFBITS level;
  guint point = 0;
  gboolean error = TRUE;
  const struct huffcodetab *h;

  g_return_val_if_fail (tnum >= 0 && tnum <= HTSIZE, FALSE);

  /* Grab a ptr to the huffman table to use */
  h = huff_tables + tnum;

  level = dmask;

  /* table 0 needs no bits */
  if (h->treelen == 0) {
    *x = *y = *v = *w = 0;
    return TRUE;
  }

  /* Lookup in Huffman table. */
  do {
    if (h->val[point][0] == 0) {        /*end of tree */      
      *x = h->val[point][1] >> 4;
      *y = h->val[point][1] & 0xf;

      error = FALSE;
      break;
    }
    if (h_get1bit (bb)) {
      while (h->val[point][1] >= MXOFF)
        point += h->val[point][1];
      point += h->val[point][1];
    } else {
      while (h->val[point][0] >= MXOFF)
        point += h->val[point][0];
      point += h->val[point][0];
    }
    level >>= 1;
  } while (level || (point < h->treelen));

  /* Check for error. */
  if (error) {
    /* set x and y to a medium value as a simple concealment */
    g_warning ("Illegal Huffman code in data.\n");
    *x = (h->xlen - 1) << 1;
    *y = (h->ylen - 1) << 1;
  }

  /* Process sign encodings for quadruples tables. */
  if (h->quad_table) {
    *v = (*y >> 3) & 1;
    *w = (*y >> 2) & 1;
    *x = (*y >> 1) & 1;
    *y = *y & 1;

    if (*v && (h_get1bit (bb) == 1))
        *v = -*v;
    if (*w && (h_get1bit (bb) == 1))
        *w = -*w;
    if (*x && (h_get1bit (bb) == 1))
        *x = -*x;
    if (*y && (h_get1bit (bb) == 1))
        *y = -*y;
  }
  /* Process sign and escape encodings for dual tables. */
  else {
    /* x and y are reversed in the test bitstream.
       Reverse x and y here to make test bitstream work. */

    if (h->linbits && ((h->xlen - 1) == *x))
        *x += h_getbits (bb, h->linbits);
    if (*x && (h_get1bit (bb) == 1))
        *x = -*x;

    if (h->linbits && ((h->ylen - 1) == *y))
        *y += h_getbits (bb, h->linbits);
    if (*y && (h_get1bit (bb) == 1))
        *y = -*y;
  }

  return !error;
}
