//*****************************************************************************
//                               PnlNgSpiceAC.cpp                             *
//                              ------------------                            *
//  Started     : 04/08/2004                                                  *
//  Last Update : 29/02/2008                                                  *
//  Copyright   : (C) 2004 by MSWaters                                        *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "ngspice/panels/PnlNgSpiceAC.hpp"

//*****************************************************************************
// Implement an event table.

BEGIN_EVENT_TABLE( PnlNgSpiceAC, PnlAnaBase )

  EVT_RADIOBOX( ID_RBX_STEPSCALE,  PnlNgSpiceAC::OnStepScale  )
  EVT_CHOICE  ( ID_CHO_SIGSRCCPNT, PnlNgSpiceAC::OnSigSrcCpnt )

END_EVENT_TABLE( )

//*****************************************************************************
// Constructor.

PnlNgSpiceAC::PnlNgSpiceAC( wxWindow * poWin ) : PnlAnaBase( poWin )
{
  bSetSimrType( eSIMR_NGSPICE );
  bSetAnaType ( eANA_AC );

  Create( );  // Create the analysis panel
  bClear( );  // Clear all object attributes
}

//*****************************************************************************
// Destructor.

PnlNgSpiceAC::~PnlNgSpiceAC( )
{
}

//*****************************************************************************
// Create the display objects.

void  PnlNgSpiceAC::Create( void )
{
  int  ix, iy;

  // Disable the checkboxes for the parameters NG-Spice cannot calculate
  m_oCbxCurrent.Disable( );
  m_oCbxPower  .Disable( );
  m_oCbxResist .Disable( );

  // Set the frequency sweep parameter labels
  m_oSbxSwpPars.GetSize( &ix, &iy );
  m_oSbxSwpPars.SetSize(  ix, 167 );
  m_oSbxSwpPars.SetLabel( wxT(" AC Sweep ") );
  m_oPnlStart  .bSetName( wxT("Start Frequency") );
  m_oPnlStop   .bSetName( wxT("Stop Frequency") );

  // Set sweep parameter spin control parameters and units
  m_oPnlStart.bSetUnitsType( ChoUnits::eUNITS_FREQ );
  m_oPnlStop .bSetUnitsType( ChoUnits::eUNITS_FREQ );
  m_oPnlStep .bSetVarType( SpinCtrl::eVAR_INT );
  m_oPnlStep .bShowUnits( FALSE );

  // Create and add the scale radio buttons
  wxString  osScale[ 4 ] = { wxT("Lin  "), wxT("Log "), wxT("Dec "), wxT("Oct ") };
  m_oRbxStepScale.Create( this, ID_RBX_STEPSCALE, wxT(" Step Scale "),
                          wxPoint( 13, 123 ), wxDefaultSize, 4, osScale, 4 );
  m_oRbxStepScale.Enable( 1, FALSE ); // Disable scale option NG-Spice doesn't support
  bSetStepScale( eSCALE_LIN );

  CreateCpxPrt( );  // Create the simulation parameter complex part check boxes
  CreateSigSrc( );  // Create input source controls
  CreateTemp( );    // Create the analysis temperature controls
}

//*****************************************************************************
// Initialize the step scale.

void  PnlNgSpiceAC::InitStepScale( void )
{
  switch( m_oRbxStepScale.GetSelection( ) )
  {
    case eSCALE_LIN :
      m_oPnlStep.bSetName( wxT("Steps in Total") );
      m_oPnlStep.bSetParms( 10, 1, 100000, 1, 100 );
      break;
    case eSCALE_DEC :
      m_oPnlStep.bSetName( wxT("Steps / Decade") );
      m_oPnlStep.bSetParms( 10, 1, 10000, 1, 100 );
      break;
    case eSCALE_OCT :
      m_oPnlStep.bSetName( wxT("Steps / Octave") );
      m_oPnlStep.bSetParms( 10, 1, 10000, 1, 100 );
      break;
    default:
      break;
  }
}

//*****************************************************************************
// Set the state of the step scale radio box.
//
// Argument List:
//   eScale - The enumerated scale specifier
//
// Return Values:
//   Success - TRUE
//   Failure - FALSE

bool  PnlNgSpiceAC::bSetStepScale( eScaleType eScale )
{
  if( eScale<eSCALE_FST || eScale>eSCALE_LST )          return( FALSE );
#if wxCHECK_VERSION( 2,8,0 )
  if( m_oRbxStepScale.GetCount( ) < (uint) eScale + 1 ) return( FALSE );
#else
  if( m_oRbxStepScale.GetCount( ) < (int)  eScale + 1 ) return( FALSE );
#endif

  m_oRbxStepScale.SetSelection( (int) eScale );

  InitStepScale( );

  return( TRUE );
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PnlNgSpiceAC::bClear( void )
{
  bool  bRtnValue=TRUE;

  // Clear the base class
  if( ! PnlAnaBase::bClear( ) )       bRtnValue = FALSE;

  // Set the sweep parameters to their defaults
  m_oPnlStart.bSetValue( (float) 1.0 );
  m_oPnlStep .bSetValue( (float) 10.0 );
  m_oPnlStart.bSetUnits( wxT("kHz") );
  m_oPnlStop .bSetUnits( wxT("kHz") );

  // Set default scale value
  if( ! bSetStepScale( eSCALE_DEC ) ) bRtnValue = FALSE;

  // Set the complex part check box default values
  m_oCbxMag  .SetValue( TRUE );
  m_oCbxPhase.SetValue( FALSE );
  m_oCbxReal .SetValue( FALSE );
  m_oCbxImag .SetValue( FALSE );
  m_oCbxMagDb.SetValue( TRUE );

  return( bRtnValue );
}

//*****************************************************************************
// Load information from a Simulation object.
//
// Argument List:
//   roSim - The Simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PnlNgSpiceAC::bLoad( Simulation & roSim )
{
  bool  bRtnValue=TRUE;

  // Load the components into the choice box
  LoadSigSrcCpnts( roSim, wxT("VI") );

  // Don't go any further if the analysis type doesn't match
  if( roSim.eGetAnaType( ) != eGetAnaType( ) ) return( TRUE );

  // Set the source component
  if( ! bSetSigSrc( roSim ) )                  bRtnValue = FALSE;

  // Set the step scale
  m_oRbxStepScale.SetSelection( roSim.iGetSwpScale( ) );

  // Set the complex parts to derive
  m_oCbxMag  .SetValue( roSim.bGetOutCpx( eCPX_MAG   ) );
  m_oCbxPhase.SetValue( roSim.bGetOutCpx( eCPX_PHASE ) );
  m_oCbxReal .SetValue( roSim.bGetOutCpx( eCPX_REAL  ) );
  m_oCbxImag .SetValue( roSim.bGetOutCpx( eCPX_IMAG  ) );
  m_oCbxMagDb.SetValue( roSim.bGetOutCpx( eCPX_MAGDB ) );
  if( !m_oCbxPhase.GetValue() && !m_oCbxReal.GetValue() && !m_oCbxImag.GetValue() )
  {
    m_oCbxMag  .SetValue( TRUE );
    m_oCbxMagDb.SetValue( TRUE );
  }

  // Perform any base class load tasks
  if( ! PnlAnaBase::bLoad( roSim ) )           bRtnValue = FALSE;

  return( bRtnValue );
}

//*****************************************************************************
// Save information to a Simulation object.
//
// Argument List:
//   roSim - The Simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PnlNgSpiceAC::bSave( Simulation & roSim )
{
  wxString  os1;
  size_t    sz1;
  bool      b1;

  // Execute base class save tasks
  PnlAnaBase::bSave( roSim );

  // Check the start frequency (can't set a frequency of zero so set one as
  // small as possible, 1.4e-45 is as small as I can go as a float but it will
  // be converted to a string in engineering unit so use 1.0e-15 = 1.0f)
  if( roSim.fGetSwpStart( ) == 0.0 ) roSim.bSetSwpStart( 1.0e-15 );

  // Modify the component to be used as the sweep source
  if( m_oPnlSigSrcLvl.dfGetValue( ) != 0.0 )
  {
    os1 = roSim.rosGetSigSrc( );
    sz1 = os1.Find( wxT(' '), TRUE );
    os1.insert( sz1, wxT(" AC") );
    if( ! roSim.bSetSigSrc( os1 ) )
      SetErrMsg( wxT("Signal source component couldn't be set.") );
  }
  else SetErrMsg( wxT("Signal source component value of zero is not permitted.") );

  // Set the complex parts of the parameters to derive
  for( sz1=eCPX_FST; sz1<=eCPX_LST; sz1++ )
    roSim.bSetOutCpx( (eCpxType) sz1, FALSE );
  b1 = FALSE;
  if( m_oCbxMag.GetValue( ) )
    { roSim.bSetOutCpx( eCPX_MAG,   TRUE ); b1 = TRUE; }
  if( m_oCbxPhase.GetValue( ) )
    { roSim.bSetOutCpx( eCPX_PHASE, TRUE ); b1 = TRUE; }
  if( m_oCbxReal.GetValue( ) )
    { roSim.bSetOutCpx( eCPX_REAL,  TRUE ); b1 = TRUE; }
  if( m_oCbxImag.GetValue( ) )
    { roSim.bSetOutCpx( eCPX_IMAG,  TRUE ); b1 = TRUE; }
  if( ! b1 ) SetErrMsg( wxT("No complex parts have been selected.") );
  if( m_oCbxMagDb.GetValue( ) )
      roSim.bSetOutCpx( eCPX_MAGDB, TRUE );

  return( bIsOk( ) );
}

//*****************************************************************************
//                                                                            *
//                             Event Handlers                                 *
//                                                                            *
//*****************************************************************************
// Step scale radio box event handler.
//
// Argument List:
//   roEvtCmd - An object holding information about the event

void  PnlNgSpiceAC::OnStepScale( wxCommandEvent & roEvtCmd )
{
  InitStepScale( );
}

//*****************************************************************************
// Source component choice box event handler.
//
// Argument List:
//   roEvtCmd - An object holding information about the event

void  PnlNgSpiceAC::OnSigSrcCpnt( wxCommandEvent & roEvtCmd )
{
  bSetSigSrcUnits( );

  if( m_oChoSigSrcCpnt.GetStringSelection( ) == wxT("None") )
    m_oPnlSigSrcLvl.bSetValue( (double) 0.0 );
  else if( m_oPnlSigSrcLvl.dfGetValue( ) == 0.0 )
    m_oPnlSigSrcLvl.bSetValue( (double) 1.0 );
}

//*****************************************************************************
