/**
* Copyright 2005-2007 ECMWF
* 
* Licensed under the GNU Lesser General Public License which
* incorporates the terms and conditions of version 3 of the GNU
* General Public License.
* See LICENSE and gpl-3.0.txt for details.
*/

#include "grib_api_internal.h"

/*
   This is used by make_class.pl

   START_CLASS_DEF
   CLASS      = accessor
   SUPER      = grib_accessor_class_data_g1simple_packing
   IMPLEMENTS = init
   IMPLEMENTS = unpack_double 
   IMPLEMENTS = value_count 
   IMPLEMENTS = pack_double
   MEMBERS=const char*  missing_value
   MEMBERS=const char*  numberOfPoints
   END_CLASS_DEF

 */

/* START_CLASS_IMP */

/*

Don't edit anything between START_CLASS_IMP and END_CLASS_IMP
Instead edit values between START_CLASS_DEF and END_CLASS_DEF
or edit "accessor.class" and rerun ./make_class.pl

*/

static int pack_double(grib_accessor*, const double* val,size_t *len);
static int unpack_double(grib_accessor*, double* val,size_t *len);
static long value_count(grib_accessor*);
static void init(grib_accessor*,const long, grib_arguments* );
static void init_class(grib_accessor_class*);

typedef struct grib_accessor_data_dummy_field {
    grib_accessor          att;
/* Members defined in gen */
/* Members defined in values */
	int  carg;
	const char* seclen;
	const char* offsetdata;
	const char* offsetsection;
	int dirty;
/* Members defined in data_simple_packing */
	const char*  number_of_values;
	const char*  bits_per_value;
	const char*  reference_value;
	const char*  binary_scale_factor;
	const char*  decimal_scale_factor;
/* Members defined in data_g1simple_packing */
	const char*  half_byte;
	const char*  packingType;
	const char*  ieee_packing;
	const char*  precision;
/* Members defined in data_dummy_field */
	const char*  missing_value;
	const char*  numberOfPoints;
} grib_accessor_data_dummy_field;

extern grib_accessor_class* grib_accessor_class_data_g1simple_packing;

static grib_accessor_class _grib_accessor_class_data_dummy_field = {
    &grib_accessor_class_data_g1simple_packing,                      /* super                     */
    "data_dummy_field",                      /* name                      */
    sizeof(grib_accessor_data_dummy_field),  /* size                      */
    0,                           /* inited */
    &init_class,                 /* init_class */
    &init,                       /* init                      */
    0,                  /* post_init                      */
    0,                    /* free mem                       */
    0,                       /* describes himself         */
    0,                /* get length of section     */
    &value_count,                /* get number of values      */
    0,                 /* get number of bytes      */
    0,                /* get offset to bytes           */
    0,            /* get native type               */
    0,                /* get sub_section                */
    0,               /* grib_pack procedures long      */
    0,               /* grib_pack procedures long      */
    0,                  /* grib_pack procedures long      */
    0,                /* grib_unpack procedures long    */
    &pack_double,                /* grib_pack procedures double    */
    &unpack_double,              /* grib_unpack procedures double  */
    0,                /* grib_pack procedures string    */
    0,              /* grib_unpack procedures string  */
    0,                 /* grib_pack procedures bytes     */
    0,               /* grib_unpack procedures bytes   */
    0,            /* pack_expression */
    0,              /* notify_change   */
    0,                /* update_size   */
    0,            /* preferred_size   */
    0,                    /* resize   */
    0,      /* nearest_smaller_value */
    0,                       /* next accessor    */
    0,                    /* compare vs. another accessor   */
    0,             /* unpack only ith value          */
};


grib_accessor_class* grib_accessor_class_data_dummy_field = &_grib_accessor_class_data_dummy_field;


static void init_class(grib_accessor_class* c)
{
	c->dump	=	(*(c->super))->dump;
	c->next_offset	=	(*(c->super))->next_offset;
	c->byte_count	=	(*(c->super))->byte_count;
	c->byte_offset	=	(*(c->super))->byte_offset;
	c->get_native_type	=	(*(c->super))->get_native_type;
	c->sub_section	=	(*(c->super))->sub_section;
	c->pack_missing	=	(*(c->super))->pack_missing;
	c->is_missing	=	(*(c->super))->is_missing;
	c->pack_long	=	(*(c->super))->pack_long;
	c->unpack_long	=	(*(c->super))->unpack_long;
	c->pack_string	=	(*(c->super))->pack_string;
	c->unpack_string	=	(*(c->super))->unpack_string;
	c->pack_bytes	=	(*(c->super))->pack_bytes;
	c->unpack_bytes	=	(*(c->super))->unpack_bytes;
	c->pack_expression	=	(*(c->super))->pack_expression;
	c->notify_change	=	(*(c->super))->notify_change;
	c->update_size	=	(*(c->super))->update_size;
	c->preferred_size	=	(*(c->super))->preferred_size;
	c->resize	=	(*(c->super))->resize;
	c->nearest_smaller_value	=	(*(c->super))->nearest_smaller_value;
	c->next	=	(*(c->super))->next;
	c->compare	=	(*(c->super))->compare;
	c->unpack_double_element	=	(*(c->super))->unpack_double_element;
}
/* END_CLASS_IMP */

static void init(grib_accessor* a,const long v, grib_arguments* args)
{
	grib_accessor_data_dummy_field* self =  (grib_accessor_data_dummy_field*)a;
	self->missing_value = grib_arguments_get_name(a->parent->h,args,self->carg++);
  self->numberOfPoints = grib_arguments_get_name(a->parent->h,args,self->carg++);
}

static int  unpack_double(grib_accessor* a, double* val, size_t *len)
{
	grib_accessor_data_dummy_field* self =  (grib_accessor_data_dummy_field*)a;
	size_t i = 0;
	size_t n_vals = 0;
  long numberOfPoints;
	double missing_value = 0;
	int err = 0;

  if((err = grib_get_long_internal(a->parent->h,self->numberOfPoints, &numberOfPoints))
      != GRIB_SUCCESS)  return err;
  n_vals =numberOfPoints;
  
	if(*len < n_vals){
		*len = n_vals;
		return GRIB_ARRAY_TOO_SMALL;
	}

	if((err = grib_get_double_internal(a->parent->h,self->missing_value, &missing_value))
       != GRIB_SUCCESS)		return err;

	for(i=0;i < n_vals;i++)
		val[i] = missing_value;

	*len = (long) n_vals;
	return err;
}

static int pack_double(grib_accessor* a, const double* val, size_t *len)
{
	grib_accessor_data_dummy_field* self =  (grib_accessor_data_dummy_field*)a;


	size_t n_vals = *len;
	int err = 0;

	long   bits_per_value = 0;

	long   half_byte = 0;

	size_t buflen = 0;
	unsigned char*  buf = NULL;

  if (*len ==0) return GRIB_NO_VALUES;

	if((err = grib_get_long_internal(a->parent->h,self->bits_per_value,&bits_per_value)) != GRIB_SUCCESS)
		return err;

	buflen = (1+((bits_per_value*n_vals)/8))*sizeof(unsigned char);

	buf = grib_context_malloc_clear(a->parent->h->context,buflen);
	if (!buf) return GRIB_OUT_OF_MEMORY;

	half_byte = (buflen*8)-((*len)*bits_per_value);

	if((err = grib_set_long_internal(a->parent->h,self->half_byte, half_byte)) != GRIB_SUCCESS){
		grib_context_free(a->parent->h->context,buf);
		return err;
	}
	grib_buffer_replace(a, buf, buflen,1,1);

	grib_context_free(a->parent->h->context,buf);

	return GRIB_SUCCESS;

}

static long value_count(grib_accessor* a)
{
  grib_accessor_data_dummy_field* self =  (grib_accessor_data_dummy_field*)a;
  int err=0;
  long numberOfPoints = 0;

  if((err = grib_get_long_internal(a->parent->h,self->numberOfPoints, &numberOfPoints))
      != GRIB_SUCCESS) {
    grib_context_log(a->parent->h->context,GRIB_LOG_ERROR,"unable to get count of %s (%s)",a->name,grib_get_error_message(err));
    return -1;
  }
  return numberOfPoints;
}

