// -*- c++ -*-
//------------------------------------------------------------------------------
//                              Deck.cpp
//------------------------------------------------------------------------------
// $Id: Deck.cpp,v 1.41 2005/11/13 02:50:48 vlg Exp $
//
//  Copyright (c) 2004-2005 by Vladislav Grinchenko
//
//  This program is free software; you can redistribute it and/or 
//  modify it under the terms of the GNU General Public License   
//  as published by the Free Software Foundation; either version  
//  2 of the License, or (at your option) any later version.      
//
// Date: Jan 8, 2004
//------------------------------------------------------------------------------

#include <fstream>
using std::ofstream;
using std::ifstream;

#include <assa/CommonUtils.h>
#include <assa/Fork.h>
#include <assa/TimeVal.h>

#include "Granule.h"
#include "Granule-main.h"
#include "DeckManager.h"
#include "Deck.h"
#include "Card.h"
#include "MainWindow.h"
#include "Config.h"
#include "CSVImportDialog.h"
#include "Intern.h"

/** Compare real cards by their IDs
 */
class Compare_Cards 
{
public:
	int operator () (VCard* const& vc1_, VCard* const& vc2_)
	{
		Card* cr1 = dynamic_cast<Card*> (vc1_);
		Card* cr2 = dynamic_cast<Card*> (vc2_);
		Assure_exit (cr1 && cr2);
		return (cr1->get_id () < cr2->get_id ());
	}
};	

/** C'tor for the new deck
 */
Deck::
Deck (const string& fname_, DeckEditStatus status_) 
	:
	m_fname  (fname_), 
	m_desc   ("Created on "),
	m_status (status_),
	m_needs_renaming ((status_ == NEW_DECK))
{
	trace_with_mask("Deck::Deck(untitled)",GUITRACE|DECK);

	m_author   = Glib::locale_to_utf8   (CONFIG->get_user_name ());
	m_snd_path = Glib::filename_to_utf8 (CONFIG->get_snd_archive_path ());
	{
		ASSA::TimeVal today = ASSA::TimeVal::gettimeofday ();
		m_desc += Glib::locale_to_utf8 (today.fmtString ("%c"));
	}
	dump_deck_status ();
}

int
Deck::
load (const string& fname_, DeckAction action_)
{
	trace_with_mask("Deck::load",GUITRACE|DECK);
	DL((DECK|APP,"Loading \"%s\" ...\n", fname_.c_str ()));

	string error_msg;
	xmlParserCtxtPtr context;
	xmlDocPtr parser;

	context = xmlCreateFileParserCtxt (fname_.c_str ()); // new

	if (!context) {
		DL((GRAPP,"Failed to parse \"%s\"\n", fname_.c_str ()));
		DECKMGR->on_deck_loaded (this, LOAD_FAILED);
		xmlFreeParserCtxt (context);
		return -1;
	}
	// Initialize context
	context->linenumbers = 1;
	context->validate = 1;
	
	//Tell the validity context about the callbacks:
	//(These are only called if validation is on - see above)
	//
	// context->vctxt.error = &callback_validity_error;
	// context->vctxt.warning = &callback_validity_warning;
	
	context->replaceEntities = 1;

	// Parse the document
	xmlParseDocument (context);

	if(!context->wellFormed) {
		xmlFreeParserCtxt (context); 
		return -1;
	}
	if (context->errNo != 0) {
		xmlFreeParserCtxt (context); 
		return -1;
	}
	parser = context->myDoc;
	xmlFreeParserCtxt (context); // not needed anymore?
	
	if (action_ == INSTALL_DECK) {
		m_fname = fname_;
	}

	DL((GRAPP,"Parsing the deck ...\n"));

	if (parse_xml_tree (parser, error_msg) < 0) {
		DECKMGR->on_deck_loaded (this, LOAD_FAILED);
		Gtk::MessageDialog emsg (error_msg, MESSAGE_ERROR);
		emsg.run ();
		xmlFreeDoc (parser);
		return -1;
	}

	if (action_ == INSTALL_DECK) {
		DECKMGR->on_deck_loaded (this, LOAD_OK);
	}
	m_status = CLEAN_DECK;
	dump_deck_status ();

	xmlFreeDoc (parser);
	return 0;
}

int
Deck::
parse_xml_tree (xmlDocPtr parser_, std::string& error_msg_)
{
	trace_with_mask("Deck::parse_xml_tree",GUITRACE|DECK);

	char xpath [64];
	xmlChar* result;
	Card* card;
	std::ostringstream os;

	result = Granule::get_node_by_xpath (parser_, "/deck/author");
	if (result) {
		m_author = (const char*) result;
		xmlFree (result);
	}

	result = Granule::get_node_by_xpath (parser_, "/deck/description");
	if (result) {
		m_desc = (const char*) result;
		xmlFree (result);
	}

	result = Granule::get_node_by_xpath (parser_, "/deck/sound_path");
	if (result) {
		m_snd_path = Glib::locale_from_utf8 ((const char*) result);
		xmlFree (result);
	}
		
	struct stat st;
	if (::stat (m_snd_path.c_str (), &st) != 0 || !S_ISDIR (st.st_mode)) {
		DL((APP,"Running with sound disabled\n"));
		DL((APP,"Path \"%s\" is wrong or non-existent\n",m_snd_path.c_str ())); 
		m_with_snd = false;
	}
	dump ();

	/** Load the cards
	 */
	for (int cards_counter = 1; ; cards_counter++) {
		sprintf (xpath, "/deck/card[%d]", cards_counter);

		result = Granule::get_node_by_xpath (parser_, xpath);
		if (result == NULL) {
			break;
		}
		/** Detected new card
		 */
		card = new Card;

		sprintf (xpath, "/deck/card[%d]/@id", cards_counter);
		result = Granule::get_node_by_xpath (parser_, xpath);
		if (result == NULL) {
			os << "id is missing in card # " << cards_counter
			   << "\nof the deck " << m_fname;
			error_msg_ = os.str ();
			return -1;
		}
		card->set_id (::atol ((const char*) result+1));
		xmlFree (result);

		sprintf (xpath, "/deck/card[%d]/front", cards_counter);
		result = Granule::get_node_by_xpath (parser_, xpath);
		if (result == NULL) {
			os << "<front> tag is missing in card # " << cards_counter
			   << "\nof the deck " << m_fname;
			error_msg_ = os.str ();
			return -1;
		}
		card->set_question ((const char*) result);
		xmlFree (result);

		sprintf (xpath, "/deck/card[%d]/back", cards_counter);
		result = Granule::get_node_by_xpath (parser_, xpath);
		if (result == NULL) {
			os << "<back> tag is missing in card # " << cards_counter
			   << "\nof the deck " << m_fname;
			error_msg_ = os.str ();
			return -1;
		}
		card->set_answer ((const char*) result);
		xmlFree (result);

		/** back_example node might be empty - thus non-critical.
		 */
		sprintf (xpath, "/deck/card[%d]/back_example", cards_counter);
		result = Granule::get_node_by_xpath (parser_, xpath);
		if (result) {
			card->set_example ((const char*) result);
			xmlFree (result);
		}
		card->mark_clean ();
		m_cards.push_back (card);
		card->dump ();
	}

	if (m_cards.size () == 0) {
		error_msg_ = "No cards found";
		return -1;
	}

	DL((GRAPP,"Processed %d cards\n", m_cards.size ()));
	return 0;
}

void 
Deck::
insert (const string& from_fname_)
{
	trace_with_mask("Deck::insert",GUITRACE|DECK);

	Card* cr_p = NULL;
	Card* cr_q = NULL;

	load (from_fname_, ADD_TO_DECK); 
	m_status = MODIFIED_DECK;

	/** Wack duplicates
	 */
	std::sort (m_cards.begin (), m_cards.end (), Compare_Cards ());
	cardlist_iterator p = m_cards.begin ();
	cardlist_iterator q = p + 1;

	while (q != m_cards.end ()) {
		cr_p = dynamic_cast<Card*> (*p);
		cr_q = dynamic_cast<Card*> (*q);
		Assure_exit (cr_p && cr_q);
		if (cr_p->get_id () == cr_q->get_id ()) {
			delete (*q);
			m_cards.erase (q);
			q = p;
		}
		else {
			p = q;
		}
		q++;
	}
}

/**
 *  Variations:
 *
 *    1: [ ]
 *    2: [ ];[ ]
 *    3: [ ];[ ];[_________]
 *    4: [ ];[ ];[ ];[ ]; ... ;[______]
 */
bool
Deck::
import_from_csv (const string& file_)
{
	trace_with_mask("Deck::import_from_csv",GUITRACE|DECK);

    const int size = 1024;
    static char inbuf [size];
	Card* card;
	std::list<Card*> card_list;
	std::list<Card*>::iterator clist_iter;

	TimeVal t (TimeVal::gettimeofday ());
	long card_id = t.sec ();

	CSVImportDialog impd;
	if (impd.run () != Gtk::RESPONSE_APPLY) {
		return false;
	}
	impd.hide ();

	ifstream source (file_.c_str (), std::ios::in);
	if (!source.good ()) {
		Gtk::MessageDialog emsg (
			_("Failed to open import file.\nCheck file permissions."),
			MESSAGE_ERROR);
        emsg.run ();
		return false;
    }
	DL ((GRAPP,"separator = %s\n", impd.get_separator ()));
	DL ((GRAPP,"doubld_separator = %s\n", impd.get_double_separator ()));

	while (source) {
		DL ((GRAPP,"-----------------------------------------\n"));
        source.getline (inbuf, size, '\n');
        DL((GRAPP,"Input: \"%s\"\n", inbuf));

		if (::strlen (inbuf) == 0) {
			continue;
		}

		if (impd.is_windoz_linebreak () && inbuf [strlen (inbuf)-1] == 0x0D) {
			inbuf [strlen (inbuf)-1] = '\0';
		}

		/** Remove extra adjacent separators.
		 */
		Glib::ustring line (inbuf);
		Glib::ustring lhs;
		Glib::ustring rhs;
		Glib::ustring::size_type idx;

		if (impd.consume_adjacent_separators ()) {
		   while ((idx = line.find (impd.get_double_separator ())) 
				  != Glib::ustring::npos) 
		   {
			   line.replace (idx, 2, impd.get_separator ());
		   }
		}
		card = new Card;
		card->set_id (card_id++);
		if ((idx = line.find (impd.get_separator ())) == Glib::ustring::npos) {
			DL ((GRAPP,"question = \"%s\"\n", line.c_str ()));
			card->set_question (line);
			m_cards.push_back (card);
			continue;
		}
		DL ((GRAPP,"question = \"%s\"\n", line.substr (0,idx).c_str ()));
		card->set_question (line.substr (0, idx));
		line.replace (0, idx+1, "");
		DL ((GRAPP,"remains = \"%s\"\n", line.c_str ()));

		if ((idx = line.find (impd.get_separator ())) == Glib::ustring::npos) {
			DL ((GRAPP,"answer = \"%s\"\n", line.c_str ()));
			card->set_answer (line);
			m_cards.push_back (card);
			continue;
		}
		DL ((GRAPP,"answer = \"%s\"\n", line.substr (0,idx).c_str ()));
		card->set_answer (line.substr (0, idx));
		line.replace (0, idx+1, "");
		DL ((GRAPP,"remains = \"%s\"\n", line.c_str ()));

		if (line.size ()) {
			DL ((GRAPP,"example = \"%s\"\n", line.c_str ()));
			card->set_example (line);
		}
		m_cards.push_back (card);
	}
	return true;
}

bool
Deck::
erase (VCard* vcard_)
{
	trace_with_mask("Deck::erase",GUITRACE|DECK);

	cardlist_iterator pos;
	Assure_exit (vcard_);

	pos = find (m_cards.begin (), m_cards.end (), vcard_);
	if (pos != m_cards.end ()) {
		m_cards.erase (pos);
		m_status = MODIFIED_DECK;
		dump_deck_status ();
		return true;
	}
	DL((GRAPP, "Card (%d) cannot be deleted (not found)\n", vcard_->get_id ()));
	return false;
}

int
Deck::
save_as (const string& fname_)
{
	trace_with_mask("Deck::save_as",GUITRACE|DECK);
	string fname_ext = fname_;

	/** If extension is missing, add it.
	 */
	string ext (DECK_FILE_EXT);
    if (fname_ext.find (ext) == string::npos) {
		fname_ext += ext;
    }
	if (Glib::file_test (fname_ext, Glib::FILE_TEST_IS_REGULAR)) {
		string msg = "File \n" + fname_ext
			+ "\nalready exists.\nWould you like to delete it first?";
		Gtk::MessageDialog qm (*MAINWIN, msg, false,
							   Gtk::MESSAGE_QUESTION, Gtk::BUTTONS_NONE,
							   true);
		qm.add_button ("Yes", Gtk::RESPONSE_YES);
		qm.add_button ("No",  Gtk::RESPONSE_NO);
		if (qm.run () == Gtk::RESPONSE_NO) {
			return -1;
		}
		::unlink (fname_ext.c_str ());
	}

	if (needs_renaming ()) {
		m_fname = fname_ext;
		save ();
		CONFIG->set_flipside_history (m_fname, FRONT);
	}
	else {
		string tmp = m_fname;
		m_fname = fname_ext;
		save ();
		m_fname = tmp;
	}
	return 0;
}

int
Deck::
save ()
{
	trace_with_mask("Deck::save",GUITRACE|DECK);

	static const char dtd_url [] = "http://granule.sourceforge.net/granule.dtd";
	int ret = 0;
	xmlTextWriterPtr writer;

	::unlink (m_fname.c_str ());

	writer = xmlNewTextWriterFilename (m_fname.c_str (), 0);
	if (writer == NULL) {
		return -1;
	}

	ret = xmlTextWriterStartDocument (writer, NULL, "UTF-8", NULL);
	if (ret < 0) {
		xmlFreeTextWriter (writer);
		return -1;
	}

	/** We have to disable indent prior to StartDTD call and 
		enable it back again before EndDTD call to generate expected
		indentation of XML file. Oddly enough, xmlTextWriterWriteDTD()
		doesn't do the right thing itself.
	*/
	xmlTextWriterSetIndent (writer, 0);
	xmlTextWriterStartDTD  (writer, BAD_CAST "deck", NULL, BAD_CAST dtd_url);
	xmlTextWriterSetIndent (writer, 1);
	xmlTextWriterEndDTD    (writer);

	xmlTextWriterSetIndentString (writer, (xmlChar*) "  ");

	ret = xmlTextWriterStartElement (writer, BAD_CAST "deck");
	ret = xmlTextWriterWriteElement (writer, BAD_CAST "author", 
									 BAD_CAST m_author.c_str ());
	ret = xmlTextWriterWriteElement (writer, BAD_CAST "description", 
									 BAD_CAST m_desc.c_str ());
	ret = xmlTextWriterWriteElement (writer, BAD_CAST "sound_path", 
						 BAD_CAST Glib::locale_to_utf8 (m_snd_path).c_str ());

	cardlist_const_iterator citer = m_cards.begin ();
	while (citer != m_cards.end ()) {
		ret = xmlTextWriterStartElement (writer, BAD_CAST "card");
		ret = xmlTextWriterWriteAttribute (writer, BAD_CAST "id",
							   BAD_CAST (*citer)->get_id_str ().c_str ());
		ret = xmlTextWriterWriteElement (writer, BAD_CAST "front",
								 BAD_CAST (*citer)->get_question ().c_str ());
		ret = xmlTextWriterWriteElement (writer, BAD_CAST "back",
								 BAD_CAST (*citer)->get_answer ().c_str ());
		ret = xmlTextWriterWriteElement (writer, BAD_CAST "back_example",
								 BAD_CAST (*citer)->get_example ().c_str ());
		ret = xmlTextWriterEndElement (writer);	// @end card
		citer++;
	}
	/** Close <deck> and write out the file
	 */
	xmlTextWriterEndElement (writer); // @end deck
	xmlTextWriterEndDocument (writer);
	xmlTextWriterFlush (writer);
	xmlFreeTextWriter (writer);	      // clenup

	m_status = CLEAN_DECK;
	m_needs_renaming = false;

	/** clear up all cards
	 */
	cardlist_iterator iter = m_cards.begin ();
	while (iter != m_cards.end ()) {
		(*iter)->mark_clean ();
		iter++;
	}

	dump_deck_status ();
	return (ret);
}

void
Deck::
get_progress (float& pct_done_, std::string& msg_, int idx_)
{
	trace_with_mask("Deck::get_progress",GUITRACE|DECK);

	DL ((APP,"idx_ = %d\n", idx_));

	std::ostringstream os;
	os << (idx_ + 1) << G_DIR_SEPARATOR_S << m_cards.size ();
	msg_ = os.str ();
	if (m_cards.size () <= 1) {
		pct_done_ = 0;
	}
	else {
		pct_done_ = idx_ * 1.0 / (m_cards.size () - 1);
	}
}

VCard* 
Deck::
find_card_by_id (long id_)
{
	trace_with_mask("Deck::find_card_by_id",GUITRACE|DECK);

	cardlist_iterator iter = m_cards.begin ();
	while (iter != m_cards.end ()) {
		DL ((GRAPP,"Trying id=%ld\n", (*iter)->get_id ()));
		if ((*iter)->get_id () == id_) {
			DL ((GRAPP,"... that's a match\n"));
			return (*iter);
		}
		iter++;
	}
	return (NULL);
}

/** Rules for playable name are as follows:

    1) A word can be preceeded by "to " to indicate it is a verb.
	2) All multiword expressions are ignored.
	3) The WAV file is assumed to be stored in the directory
       with the name of its first letter, i.e. "granule" will be
       stored in the directory "/path/to/dictionary/g/", and thus 
	   the full name would be "/path/to/dictionary/g/granule.wav".

	RETURN: playable name on success; an empty string on error.
*/
void
Deck::
play_card (Deck::cardlist_iterator& iter_)
{
	trace_with_mask("Deck::play_card",GUITRACE|DECK);
	static char execmdbuf [1024];

	if (! m_with_snd) {
		return;
	}
	std::vector<string> v;
	string w = Glib::locale_from_utf8 ((*iter_)->get_question ());
	ASSA::Utils::split (w.c_str (), v);
	if (v.size () > 1) {
		if (v[0] != "to") {
			return;
		}
		w = v[1];
	}
	string sfn (m_snd_path + G_DIR_SEPARATOR_S);
	sfn += w[0];
	sfn += G_DIR_SEPARATOR_S + w + ".wav";

	struct stat st;
	if (::stat (sfn.c_str (), &st) != 0 || !S_ISREG (st.st_mode)) {
		DL((APP,"Can't stat sound file \"%s\"\n", sfn.c_str ()));
		return;
	}
	snprintf (execmdbuf, 512, CONFIG->get_snd_player_args (), sfn.c_str ());
	DL((APP,"Playing \"%s %s\"\n", CONFIG->get_snd_player_cmd (), execmdbuf));

	/**	
	 * Better yet to route WAV playing through esd (if it is running):
	 * esdplay <fname>.wav.
	 *
	 * NOTE: New Gtk 2.4 FileChooserDialog changes disposition mask for SIGCHLD.
	 *       We have to preserve it - otherwise, we would crash and burn.
	 *       COLLECT_STATUS option does that for us here.
	*/
	ASSA::Fork::fork_exec (CONFIG->get_snd_player_cmd (), 
						   execmdbuf, ASSA::Fork::COLLECT_STATUS);
}

/** 
	Deck information consists of two parts: the header (More... dialog)
    and a list of cards. Either could have been modified by the user.
	In the first case, the status would be MODIFIED_DECK; in the second,
	however, the status would be CLEAN_DECK. We double check here to 
	see if any of the cards have been edited (if new card were added 
	or deleted, the status would be MODIFIED_DECK
*/
DeckEditStatus
Deck::
get_status ()
{	
	if (m_status == CLEAN_DECK) {
		dump_deck_status ();
		cardlist_const_iterator citer = m_cards.begin ();
		while (citer != m_cards.end ()) {
			if ((*citer)->is_dirty ()) {
				m_status = MODIFIED_DECK;
				break;
			}
			citer++;
		}
	}
	dump_deck_status ();
	return m_status;
}

void
Deck::
dump () const 
{
	DL((DECK, "fname = \"%d\"\n", m_fname.c_str ()));
	DL((DECK, "author = \"%s\"\n", m_author.c_str ()));
	DL((DECK, "description = \"%s\"\n", m_desc.c_str ()));
}

void
Deck::
dump_deck_status () const
{
	static const char* names [] = { "NEW_DECK", "INVALID_DECK", 
								   "MODIFIED_DECK", "CLEAN_DECK" };

	DL((APP, "Deck \"%s\"\n", get_name ().c_str ()));
	DL((APP, "Status = <%s>\n", names [m_status]));
}

void
Deck::
swap_with_next (VCard* vcard_)
{
	cardlist_iterator p;
	cardlist_iterator n;
	p = m_cards.begin ();
	while (p != m_cards.end ()) {
		if (vcard_ == *p) {
			n = p+1;
			VCard* next_card = *n;
			m_cards.erase (n);
			m_cards.insert (p, next_card);
			m_status = MODIFIED_DECK;
			break;
		}
		p++;
	}
}

void
Deck::
swap_with_prev (VCard* vcard_)
{
	cardlist_iterator p;
	cardlist_iterator n;
	p = m_cards.begin ();
	while (p != m_cards.end ()) {
		if (vcard_ == *p) {
			n = p-1;
			VCard* prev_card = *p;
			m_cards.erase (p);
			m_cards.insert (n, prev_card);
			m_status = MODIFIED_DECK;
			break;
		}
		p++;
	}
}

//------------------------------------------------------------------------------
// LEFTOVERS FOR HEAVY DEBUGGING
//------------------------------------------------------------------------------
#if 0
void
Deck::
traverse_xml_tree (const xmlpp::Node* node_)
{
	trace_with_mask("Deck::traverse_xml_tree",GUITRACE);

	string nodename;
	if (node_) {
		nodename = node_->get_name ();
	}
	const xmlpp::TextNode* node_text;
	node_text = dynamic_cast<const xmlpp::TextNode*> (node_);

	/** Ignore indentation if XML file is indented.
		I wonder how it would work with UTF-8.
	 */
	if (node_text && node_text->is_white_space ()) {
		return;
	}

	DL((DECK,"%sPath = \"%s\"\n", 
		m_indent.c_str(), node_->get_path ().c_str ()));
	
	/** We cast node pointer to all possible types and then 
	 *  determine which one we are dealing with by elimination.
	 */
	const xmlpp::ContentNode* node_content;
	node_content = dynamic_cast<const xmlpp::ContentNode*> (node_);

	const xmlpp::CommentNode* node_comment;
	node_comment = dynamic_cast<const xmlpp::CommentNode*> (node_);

	const xmlpp::Element* node_element;
	node_element = dynamic_cast<const xmlpp::Element*> (node_);

	/** If either Content or Element, indent and print out its name.
	 */
	if (!node_text && !node_comment && !nodename.empty ()) {
		DL((DECK,"%sNode name = %s\n", 
			m_indent.c_str(), nodename.c_str ()));
	}

	/** Find the right type
	 */
	if (node_text) {
		DL((DECK,"%sType = TEXT_NODE\n", m_indent.c_str()));
		DL((DECK,"%stext = \"%s\"\n", m_indent.c_str(), 
			node_text->get_content ().c_str ()));

		if (m_element_node_name == "front_encoding") {
			m_front_encoding = node_text->get_content ();
		}
		else if (m_element_node_name == "back_encoding") {
			m_back_encoding = node_text->get_content ();
		}
		else if (m_element_node_name == "card") {
			m_current_card = new Card;
			m_cards.push_back (m_current_card);
		}
		else if (m_element_node_name == "front") {
			m_current_card->set_question (node_text->get_content ());
		}
		else if (m_element_node_name == "back") {
			m_current_card->set_answer (node_text->get_content ());
		}
		else if (m_element_node_name == "back_example") {
			m_current_card->set_example (node_text->get_content ());
		}
		else if (m_element_node_name == "") {
			= node_text->get_content ();
		}

	}
	else if (node_comment) {
		DL((DECK,"%sType = COMMENT_NODE\n", m_indent.c_str()));
		DL((DECK,"%scomment = \"%s\"\n", m_indent.c_str(), 
			node_comment->get_content ().c_str ()));
	}
	else if (node_content) {
		DL((DECK,"%sType = CONTENT_NODE\n", m_indent.c_str()));
		DL((DECK,"%scontent = \"%s\"\n", m_indent.c_str(), 
			node_content->get_content ().c_str ()));
	}
	else if (node_element) {
		DL((DECK,"%sType = ELEMENT_NODE\n", m_indent.c_str()));
		DL((DECK,"%sline = \"%d\"\n", m_indent.c_str(), 
			node_element->get_line ()));     // line() is for Elements

		/** Print attributes of the element
		 */
		const xmlpp::Element::AttributeList& 
			attributes = node_element->get_attributes ();
		const xmlpp::Attribute* attr;
		xmlpp::Element::AttributeList::const_iterator iter = attributes.begin();
		while (iter != attributes.end ()) {
			attr = *iter;
			DL((DECK,"%sType = ATTRIBUTE_NODE\n", m_indent.c_str()));
			DL((DECK,"%sAttribute: %s = \"%s\"\n", m_indent.c_str(),
				attr->get_name ().c_str (), attr->get_value ().c_str ()));
			iter++;
		}
		attr = node_element->get_attribute ("id");
		if (attr) {
			DL((DECK,"%sID found = %s\n", m_indent.c_str(), 
				attr->get_value().c_str()));
			if (m_element_node_name == "card") {
				std::string value = attr->get_value ();
				m_current_card->set_id (atol (value.substr (1, value.size ())));
				m_element_node_name = "":
			}

		}
	}

	/** If not Content, recurse through the children
	 */
	m_element_node_name = "";
	if (!node_content) {
		xmlpp::Node::NodeList list = node_->get_children ();
		xmlpp::Node::NodeList::iterator iter = list.begin ();
		while (iter != list.end ()) {
			m_indent += "  ";
			traverse_xml_tree (*iter);
			iter++;
		}
	}
	m_indent = "";
}
#endif
