# -*- python -*-

### Copyright (C) 2005 Peter Williams <pwil3058@bigpond.net.au>

### This program is free software; you can redistribute it and/or modify
### it under the terms of the GNU General Public License as published by
### the Free Software Foundation; version 2 of the License only.

### This program is distributed in the hope that it will be useful,
### but WITHOUT ANY WARRANTY; without even the implied warranty of
### MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
### GNU General Public License for more details.

### You should have received a copy of the GNU General Public License
### along with this program; if not, write to the Free Software
### Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# This file provides access to "quilt" functionality required by "gquilt"

import gquilt_utils, os.path, os, gquilt_tool, gquilt_pfuns, re

class quilt_commands(gquilt_utils.cmd_list):
    def __init__(self):
        gquilt_utils.cmd_list.__init__(self)
    def _get_commands(self):
        res, sop, eop = gquilt_utils.run_cmd("quilt")
        if sop == "":
            return None
        lines = sop.splitlines()
        index = 2
        cmds = []
        while True:
            lcmds = lines[index].split()
            if len(lcmds) > 0:
                cmds += lcmds
            else:
                cmds.sort()
                return cmds
            index += 1

def find_patchfns():
    binloc = gquilt_utils.which("quilt")
    if binloc is None:
        return None
    bi = binloc.find("bin" + os.path.sep + "quilt")
    if bi == -1:
        return None
    res = os.path.join(binloc[0:bi], "share", "quilt", "scripts", "patchfns")
    if not os.path.exists(res) or not os.path.isfile(res):
        return None
    return res

patchfns = find_patchfns()
if patchfns is None:
    qbsfe = None
else:
    qbsfe = 'bash -c ". ' + os.environ['GQUILT_LIB_DIR'] + '/qbsfe.sh" gquilt ' + patchfns

def internal(cmd, input=None):
    if qbsfe is None:
        raise "quilt_specific_error", "Couldn't find patchfns"
    return gquilt_utils.run_cmd(" ".join([qbsfe, cmd]), input)

# run a command and log the result to the provided console
def _exec_console_cmd(console, cmd, error=gquilt_tool.ERROR):
    res, so, se = gquilt_utils.run_cmd(cmd)
    if console is not None:
        console.log_cmd(cmd + "\n", so, se)
    if res != 0:
        return (error, so, se)
    else:
        return (gquilt_tool.OK, so, se)

# Some useful private quilt functions (that shouldn't fail)
def _patch_file_name(patch):
    res, so, se = internal(" ".join(["patch_file_name", patch]))
    if res != 0:
        raise "quilt_specific_error", se
    return so[:-1]

def _patch_backup_dir(patch):
    res, so, se = internal(" ".join(["backup_file_name", patch, "x"]))
    if res != 0:
        raise "quilt_specific_error", se
    return so[:-3]

def _patch_backup_file(patch, filename):
    res, so, se = internal(" ".join(["backup_file_name", patch, filename]))
    if res != 0:
        raise "quilt_specific_error", se
    return so[:-1]

def _patch_files_may_have_changed(patch):
    res, so, se = internal(" ".join(["files_may_have_changed", patch]))
    if res != 0:
        raise "quilt_specific_error", se
    return res == 0

# Now implement the tool interface for quilt
class interface(gquilt_tool.interface):
    def __init__(self):
        gquilt_tool.interface.__init__(self, "quilt")
    def is_patch_applied(self, patch):
        res, so, se = internal("is_applied " + patch)
        return res == 0
    def top_patch(self):
        res, so, se =gquilt_utils.run_cmd("quilt top")
        if res == 0 or (se == "" and so == ""):
            return so.strip()
        else:
            raise "quilt_specific_error", se
    def last_patch_in_series(self):
        res, op, err = gquilt_utils.run_cmd("quilt series")
        if res != 0:
            raise "quilt_specific_error", se
        return op.splitlines()[-1]
    def display_file_diff_in_viewer(self, viewer, file, patch=None):
        difffld = "--diff=" + viewer
        if patch is None:
            patchfld = ""
        else:
            patchfld = "-P " + patch
        pid = os.spawnlp(os.P_NOWAIT, "quilt", "quilt", "diff", difffld, patchfld, file)
    def get_patch_description(self, patch):
        pfn = _patch_file_name(patch)
        if os.path.exists(pfn):
            res, lines = gquilt_pfuns.get_patch_descr_lines(pfn)
            if res:
                return (gquilt_tool.OK, "\n".join(lines) + "\n", "")
            else:
                return (gquilt_tool.ERROR, "", "Error reading patch description\n")
        else:
            return (gquilt_tool.OK, "", "")
    def get_patch_status(self, patch):
        if not self.is_patch_applied(patch):
            return (gquilt_tool.OK, gquilt_tool.NOT_APPLIED, "")
        patch_fn = _patch_file_name(patch)
        patch_dirn = _patch_backup_dir(patch)
        if (not os.path.exists(patch_fn)) or os.path.exists(patch_dirn + "~refresh"):
            needs_refresh = True
        else:
            timestamp_fn = patch_dirn + "/.timestamp"
            if not os.path.exists(timestamp_fn):
                needs_refresh = True
            else:
                timestamp = os.stat(timestamp_fn).st_mtime
                if timestamp > os.stat(patch_fn).st_mtime:
                    needs_refresh = True
                else:
                    needs_refresh = False
                    res, files, err = gquilt_utils.run_cmd("quilt files " + patch)
                    for f in files.splitlines():
                        if timestamp < os.stat(f).st_mtime:
                            needs_refresh = True
                            break
        res, tp, se = self.get_top_patch()
        if tp == patch:
            if needs_refresh:
                return (gquilt_tool.OK, gquilt_tool.TOP_PATCH_NEEDS_REFRESH, "")
            else:
                return (gquilt_tool.OK, gquilt_tool.TOP_PATCH, "")
        else:
            if needs_refresh:
                return (gquilt_tool.OK, gquilt_tool.APPLIED_NEEDS_REFRESH, "")
            else:
                return (gquilt_tool.OK, gquilt_tool.APPLIED, "")
    def get_series(self):
        res, op, err = gquilt_utils.run_cmd("quilt series -v")
        if res != 0:
            return (gquilt_tool.ERROR, (None, None), err)
        series = []
        for line in op.splitlines():
            if line[0] == " ":
                series.append((line[2:], gquilt_tool.NOT_APPLIED))
            else:
                pn = line[2:]
                #res, status, err = self.get_patch_status(pn)
                #series.append((pn, status))
                series.append((pn, gquilt_tool.APPLIED))
        try:
            patch_dir = os.environ['QUILT_PATCHES']
        except:
            patch_dir = ""
        return (gquilt_tool.OK, (patch_dir, series), "")
    def get_diff(self, filelist=[], patch=None):
        if patch is None:
            cmd = "quilt diff"
        else:
            cmd = "quilt diff -P " + patch
        res, so, se = gquilt_utils.run_cmd(" ".join([cmd, " ".join(filelist)]))
        if res != 0:
            return (gquilt_tool.ERROR, so, se)
        return (res, so, se)
    def get_combined_diff(self, start_patch=None, end_patch=None):
        cmd = "quilt diff --sort --combine "
        if start_patch is None:
            cmd += "-"
        else:
            cmd += start_patch
        if end_patch is not None:
            cms += " -P " + end_patch
        res, so, se = gquilt_utils.run_cmd(cmd)
        if res != 0:
            return (gquilt_tool.ERROR, so, se)
        return (res, so, se)
    def get_patch_files(self, patch=None, withstatus=True):
        if self.top_patch() == "":
            return (gquilt_tool.OK, "", "")
        cmd = "quilt files"
        if withstatus:
            cmd += " -v"
        if patch is not None:
            cmd += " " + patch
        res, so, se = gquilt_utils.run_cmd(cmd)
        if res != 0:
            return (gquilt_tool.ERROR, so, se)
        if withstatus:
            filelist = []
            for line in so.splitlines():
                if line[0] == "+":
                    filelist.append((line[2:], gquilt_tool.ADDED))
                elif line[0] == "-":
                    filelist.append((line[2:], gquilt_tool.DELETED))
                else:
                    filelist.append((line[2:], gquilt_tool.EXTANT))
        else:
            filelist = so.splitlines()
        return (res, filelist, se)
    def do_set_patch_description(self, console, patch, description):
        pfn = _patch_file_name(patch)
        res = gquilt_pfuns.set_patch_descr_lines(pfn, description.splitlines())
        if res:
            res = gquilt_tool.OK
            se = ""
        else:
            res = gquilt_tool.ERROR
            se = "Error reading patch description\n"
        if console is not None:
            console.log_cmd('set description for "' + patch + '"', "\n", se)
        return (res, "", se)
    def do_rename_patch(self, console, patch, newname):
        if quilt_commands().has_cmd("rename"):
            cmd = "quilt rename "
            if patch is not None:
                cmd += "-p " + patch + " "
            cmd += newname
            res, so, se = gquilt_utils.run_cmd(cmd)
            if console is not None:
                console.log_cmd(cmd + "\n", so, se)
            if res != 0:
                return (gquilt_tool.ERROR, so, se)
            else:
                return (gquilt_tool.OK, so, se)
        else:
            newname_pfn = _patch_file_name(newname)
            newname_backup_dir =  _patch_backup_dir(newname)
            if os.path.exists(newname_pfn) or os.path.exists(newname_backup_dir):
                return (gquilt_tool.ERROR, "", 'Patch rename failed. "' + newname + '" is already being used')
            if patch is None:
                res, patch, se = self.get_top_patch()
                if res != gquilt_tool.OK:
                    return (res, patch, se)
            patchname_pfn = _patch_file_name(patch)
            if self.is_patch_applied(patch):
                res, so, se = internal(" ".join(["rename_in_db",  patch, newname]))
                if res != 0:
                    return (gquilt_tool.ERROR, so, se)
                os.rename(_patch_backup_dir(patch), newname_backup_dir)
                if os.path.exists(patchname_pfn):
                    os.rename(patchname_pfn, newname_pfn)
            else:
                if os.path.exists(patchname_pfn):
                    os.rename(patchname_pfn, newname_pfn)
            res, so, se = internal(" ".join(["rename_in_series", patch, newname]))
            if res != 0:
                return (gquilt_tool.ERROR, so, se)
            else:
            	return (gquilt_tool.OK, so, se)
    def do_pop_patch(self, console, force=False):
        cmd = "quilt pop"
        if force:
            cmd += " -f"
        res, so, se = _exec_console_cmd(console, cmd)
        if res != gquilt_tool.OK:
            if re.search("needs to be refreshed first", so + se):
                res = gquilt_tool.ERROR_REFRESH
            elif re.search("\(refresh it or enforce with -f\)", so + se):
                res = gquilt_tool.ERROR_FORCE_OR_REFRESH
            else:
                res = gquilt_tool.ERROR
        return (res, so, se)
    def do_push_patch(self, console, force=False):
        cmd = "quilt push"
        if force:
            cmd += " -f"
        res, so, se = _exec_console_cmd(console, cmd)
        if res != gquilt_tool.OK:
            if re.search("\(forced; needs refresh\)", so + se):
                res = gquilt_tool.OK
            elif re.search("needs to be refreshed first", so + se):
                res = gquilt_tool.ERROR_REFRESH
            elif re.search("\(enforce with -f\)", so + se):
                res = gquilt_tool.ERROR_FORCE
            else:
                res = gquilt_tool.ERROR
        return (res, so, se)
    def do_refresh_patch(self, console, patch=None, force=False):
        cmd = "quilt refresh"
        if force:
            cmd += " -f"
        if patch is not None:
            cmd = " ".join([cmd, patch])
        res, so, se = _exec_console_cmd(console, cmd)
        if res != gquilt_tool.OK:
            if re.search("Enforce refresh with -f", so + se):
                res = gquilt_tool.ERROR_FORCE
            else:
                res = gquilt_tool.ERROR
        return (res, so, se)
    def do_create_new_patch(self, console, name):
        return _exec_console_cmd(console, " ".join(["quilt", "new", name]))
    def do_import_patch(self, console, filename):
        return _exec_console_cmd(console, " ".join(["quilt", "import", filename]))
    def do_merge_patch(self, console, filename):
        return _exec_console_cmd(console, " < ".join(["quilt fold", filename]))
    def do_delete_patch(self, console, patch):
        return _exec_console_cmd(console, " ".join(["quilt", "delete", patch]))
    def do_add_files_to_patch(self, console, filelist, patch=None):
        if patch == None:
            cmd = "quilt add"
        else:
            cmd = "quilt add -p " + patch
        # quilt will screw up semi silently if you try to add directories to a patch
        for f in filelist:
            if os.path.isdir(f):
                return (gquilt_tool.ERROR, "", f + ' is a directory.\n "quilt" does not handle adding directories to patches\n')
        return _exec_console_cmd(console, " ".join([cmd, " ".join(filelist)]))
    def do_remove_files_from_patch(self, console, filelist, patch=None):
        if patch == None:
            cmd = "quilt remove"
        else:
            cmd = "quilt remove -p " + patch
        # quilt will screw up semi silently if you try to remove directories to a patch
        for f in filelist:
            if os.path.isdir(f):
                return (gquilt_tool.ERROR, "", f + ' is a directory.\n "quilt" does not handle removing directories from patches\n')
        return _exec_console_cmd(console, " ".join([cmd, " ".join(filelist)]))
    def do_exec_tool_cmd(self, console, cmd):
        return _exec_console_cmd(console, " ".join(["quilt", cmd]))
