/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*

  util.c: Utility functions.

  Copyright (C)  2001-2005  Alexandru Csete.

  Authors:  Alexandru Csete <csete@users.sourceforge.net>

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/

#include <glib.h>
#include <math.h>
#include <ctype.h>
#include <time.h>
#include <sys/time.h>
#include "defaults.h"
#include "util.h"

extern gboolean clock_show_local;  /* show local time instead of UTC */

gdouble FixAngle(gdouble x)
{
	/* This function reduces angles greater than
	   two pi by subtracting two pi from the angle.
	   Stolen from Predict.
	*/

	while (x>TP)
		x-=TP;

	return x;
}

gulong DayNum(gint m, gint d, gint y)
{
	/* This function calculates the day number from m/d/y.
	   Stolen from Predict.
	 */

	glong dn;
	gdouble mm, yy;

	if (m<3)
	{ 
		y--; 
		m+=12; 
	}

	/* Correct for Y2K... */

	if (y<=50)
		y+=100;

	yy=(gdouble)y;
	mm=(gdouble)m;
	dn=(glong)(floor(365.25*(yy-80.0))-floor(19.0+yy/100.0)+floor(4.75+yy/400.0)-16.0);
	dn+=d+30*m+(long)floor(0.6*mm-0.3);
	return dn;
}

gdouble CurrentDaynum ()
{
	/* Read the system clock and return the number
	   of days since 31Dec79 00:00:00 UTC (daynum 0)
	   Stolen from Predict.
	*/
	struct timeval tval;
	struct timezone tzone;
	gint x;

	x = gettimeofday (&tval, &tzone);

	return ((((gdouble)tval.tv_sec+0.000001*(gdouble)tval.tv_usec)/86400.0)-3651.0);
}



gchar *dnum2fstr(gdouble daynum, const gchar *format)
{
	/* This function takes the given epoch as a fractional number of
	   days since 31Dec79 00:00:00 UTC and returns the corresponding
	   date as a string of the form given by the format string .
	   Stolen from Predict.
	*/
	guint size;
	time_t t;
	gchar *buff;

	buff = g_malloc(50);
	/* Convert daynum to Unix time (seconds since 01-Jan-70) */
	t=(time_t)rint(86400.0*(daynum+3651.0));

	if(clock_show_local)
		size = strftime( buff, 50, format, localtime(&t));
	else
		size = strftime( buff, 50, format, gmtime(&t));

	if (size<50)
		buff[size]='\0';
	else
		buff[49]='\0';

	return buff;
}




gchar timestr[20];

gchar *dnum2str(gdouble daynum)
{
	/* This function takes the given epoch as a fractional number of
	   days since 31Dec79 00:00:00 UTC and returns the corresponding
	   date as a string of the form "2001/03/25 12:45:32".
	   Stolen from Predict.
	*/

	time_t t;

	/* Convert daynum to Unix time (seconds since 01-Jan-70) */
	t=(time_t)rint(86400.0*(daynum+3651.0));
	if(clock_show_local)
		strftime(timestr,25,"%Y/%m/%d %H:%M:%S",localtime(&t));
	else
		strftime(timestr,25,"%Y/%m/%d %H:%M:%S",gmtime(&t));
	timestr[20]='\0';

	return timestr;
}

gchar ltimestr[35];

gchar *dnum2lstr(gdouble daynum)
{
	/* This function takes the given epoch as a fractional number of
	   days since 31Dec79 00:00:00 UTC and returns the corresponding
	   date as a string of the form "Tue Oct 03 2001, 13:45:32 UTC".
	   Stolen from Predict.
	*/

	time_t t;

	/* Convert daynum to Unix time (seconds since 01-Jan-70) */
	t=(time_t)rint(86400.0*(daynum+3651.0));

	if(clock_show_local)
		strftime(ltimestr,35,"%a %b %d %Y, %H:%M:%S %Z",localtime(&t));
	else
		strftime(ltimestr,35,"%a %b %d %Y, %H:%M:%S %Z",gmtime(&t));

	ltimestr[34]='\0';


	return ltimestr;
}


extern guchar val[]; /* engine.c */

guint KepCheck( gchar *line1, gchar *line2 )
{
	/* This function scans line 1 and line 2 of a NASA 2-Line element
	   set and returns a 1 if the element set appears to be valid or
	   a 0 if it does not.  If the data survives this torture test,
	   it's a pretty safe bet we're looking at a valid 2-line
	   element set and not just some random garbage that might pass
	   as orbital data based on a simple checksum calculation alone. */

	gint x;
	guint sum1, sum2;

	/* Compute checksum for each line */

	for (x=0, sum1=0, sum2=0; x<=67; sum1+=val[(gint)line1[x]], sum2+=val[(gint)line2[x]], x++);

	/* Perform a "torture test" on the data */

	x=(val[(gint)line1[68]]^(sum1%10)) | (val[(gint)line2[68]]^(sum2%10)) |
	  (line1[0]^'1')  | (line1[1]^' ')  | (line1[7]^'U')  |
	  (line1[8]^' ')  | (line1[17]^' ') | (line1[23]^'.') |
	  (line1[32]^' ') | (line1[34]^'.') | (line1[43]^' ') |
	  (line1[52]^' ') | (line1[61]^' ') | (line1[63]^' ') |
	  (line2[0]^'2')  | (line2[1]^' ')  | (line2[7]^' ')  |
	  (line2[11]^'.') | (line2[16]^' ') | (line2[20]^'.') |
	  (line2[25]^' ') | (line2[33]^' ') | (line2[37]^'.') |
	  (line2[42]^' ') | (line2[46]^'.') | (line2[51]^' ') |
	  (line2[54]^'.') | (line1[2]^line2[2]) | (line1[3]^line2[3]) |
	  (line1[4]^line2[4]) | (line1[5]^line2[5]) | (line1[6]^line2[6]) |
	  (isdigit(line1[68]) ? 0 : 1) | (isdigit(line2[68]) ? 0 : 1) |
	  (isdigit(line1[18]) ? 0 : 1) | (isdigit(line1[19]) ? 0 : 1) |
	  (isdigit(line2[31]) ? 0 : 1) | (isdigit(line2[32]) ? 0 : 1);

	return (x ? 0 : 1);
}

gchar temp[80];

gchar *SubString(gchar *string, gchar start, gchar end)
{
	/* This function returns a substring based on the starting
	   and ending positions provided.  It is used heavily in
	   the AutoUpdate function when parsing 2-line element data.
	   Stolen from Predict.
	*/

	unsigned x, y;

	if ( end >= start )
	{
		for ( x = start, y = 0; x <= end && string[x] != 0; x++ )
			if ( string[x] != ' ' )
			{
				temp[y] = string[x];
				y++;
			}

		temp[y] = 0;
		return temp;
	}
	else
		return NULL;
}


void free_vector_notify_cb (gpointer data)
{
	/* This function can be used as a
	   GtkDestroyNotify callback
	   function to free a gchar**
	*/
	g_strfreev ((gchar **) data);
}
