/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
  satmap.c: Graphical visualization module for Gnome Predict showing
            selected satellites on a map.


  Copyright (C)  2001-2005  Alexandru Csete.

  Authors:   Alexandru Csete <csete@users.sourceforge.net>

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/


#include <gnome.h>
#include <libgnomeui/gnome-window-icon.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gdk-pixbuf/gnome-canvas-pixbuf.h>
#include <gconf/gconf-client.h>
#include <math.h>
#include <string.h>
#include <time.h>
#include "satlog.h"
#include "satdata.h"
#include "sataos.h"
#include "fileio.h"
#include "util.h"
#include "qth.h"
#include "satmodules.h"
#include "iconbar.h"
#include "sattabs.h"
#include "extra-widgets.h"
#include "satmap-druid.h"
#include "satmap-print.h"
#include "satmap-info.h"
#include "satmap-single-aos.h"
#include "satmap-multi-aos.h"
#include "satmap.h"

#ifdef HAVE_CONFIG_H
#  include "../config.h"
#endif

#ifndef PI
#  define PI 3.141592653589793
#endif

#ifndef TWOPI
#  define TWOPI 2.0*PI
#endif

#ifndef HALFPI
#  define HALFPI PI/2.0
#endif

#ifndef R0
#  define R0 6378.135
#endif

extern gboolean clock_show_local;

enum satmap_mode_e {
	SATMAP_MODE_NONE = 0,  /* actually not a valid mode... */
	SATMAP_MODE_TAB,       /* widget is in the notebook */
	SATMAP_MODE_WIN,       /* widget is in a separate window */
	SATMAP_MODE_FULLSCR    /* widget is blowing out screen! */
};


/* Private function prototypes */
static void    satmap_switch_mode_cb     (GtkWidget *, GtkWidget *);
static void    satmap_save_cb            (GtkWidget *, GtkWidget *);
static void    satmap_store_filename     (GtkWidget *, GtkWidget *);
static void    satmap_filesel_destroy_cb (GtkWidget *, GtkWidget *);
static void    satmap_close_cb           (GtkWidget *, GtkWidget *);
static void    satmap_close_cleanup      (GtkWidget *);
static gint    satmap_timer_cb           (gpointer);
static gint    satmap_timer2_cb          (gpointer);
static gdouble arccos                    (gdouble, gdouble);
static void    satmap_prefs_changed      (GConfClient *, guint, GConfEntry *, gpointer);
static gdouble satmap_cx                 (gint mapwidth, gdouble lon);
static gdouble satmap_cy                 (gint mapheight, gdouble lat);
static gint    satmap_map_event_cb       (GtkWidget *, GdkEvent *, gpointer);

gint satwin_delete_cb (GtkWidget *, GdkEvent *, GtkWidget *);
void satwin_close_cb  (GtkWidget *, GtkWidget *);


extern qth_struc qth;       /* qth.c  */
extern GtkWidget *app;      /* main.c */
extern GConfClient *client; /* main.c */


void satmap_open_cb (GtkWidget *widget, gpointer data)
{
	/* This is just a gtk-callback function that calls
	   satmap_open.
	*/
	satmap_open (NULL, TRUE);
}


gint satmap_open (gchar *name, gboolean addtolist)
{
	/* This function creates the widget that shows selected
	   satellites on a map. The configuration file is expected
	   to be located at $HOME/.gpredict/name.satmap. If name is
	   NULL the druid will be invoked.
	   If addtolist is TRUE, the module will be added to the 
	   list of open modules (this should not be the case when
	   we start the program and open the modules from the
	   existing list).
	*/

	GtkWidget *hbox,*vbox,*topbox;
	GtkWidget *close;          /* Close button */
	GtkWidget *save;           /* Save button */
	GtkWidget *print;          /* Print Button */
	GtkWidget *tabwin;         /* Tab/window button */
	GtkWidget *info;           /* Info button */
	GtkWidget *maos;           /* Multi AOS button */
	GtkWidget *saos;           /* Single AOS button */
	GtkWidget *satmap;         /* The map canvas */
	GnomeCanvasItem *mapitem;  /* The background map */
	GnomeCanvasItem *qthmark;  /* The QTH dot */
	GnomeCanvasItem *qthlabel; /* The QTH label */
	GnomeCanvasItem *mapgroup; /* Canvas group */
	GnomeCanvasItem *bgd;      /* Background rectangle */
	GnomeCanvasItem *header1;  /* Left, info about QTH */
	GnomeCanvasItem *header2;  /* Right, clock */
	GnomeCanvasItem *footer1;  /* Left, info about selected satellite */
	GnomeCanvasItem *footer2;  /* Right, upcoming pass */
	GdkPixbuf *map;

	/* sat related */
	sat_t *sat;             /* pointer to a single sat struct */
	satmap_sat_t *smsat = NULL;    /* pointer to a single satmap sat struct */
	GList *slist = NULL;    /* list of satmap sat structs */
	/* range circle */
	gdouble ssplat, ssplong, beta, azimuth, rangelat, rangelong;
	gdouble num, dem, sx, sy, sx2;
	gdouble deg2rad=1.74532925199e-2;
	guint azi;
	/* other vars */
	gchar *configfile, *menulabel;
	gchar *tmp,**argv = NULL;
	gint argc, i;
	gint timer, timer2, delay;
	guint gconf_id;
	gint satindex;
	gchar *logmsg;
	guint mapw, maph;  /* map width and height */


	/**** FIXME: If there is an open module with the specified name, the new
	      module will not be created, but the contents of the config file
	      will be overwritten! See bug #521816 in bug tracker.
	*/
	if (!name)
		name = g_strdup (satmap_druid_run (NULL));

	/* Try again */
	if (!name)
		return -1;

	/* Test whether module exists or not */
	if (!satmodules_test_module (name, SATMOD_TYPE_MAP)) {
		/* delete from list, just in case ... */
		satmodules_remove_module (name, SATMOD_TYPE_MAP);
		return -1;
	}

	if (addtolist) {
		/* is there a module with the specified name already open? */
		if (satmodules_test_open_modules (name, SATMOD_TYPE_MAP)) {
			gnome_app_error (GNOME_APP (app),
					 _("There is already a MAP with the selected name open.\n"\
					   "Please close that MAP or specify another name."));
			return -1;
		}
		iconbar_remove_module (name, SATMOD_TYPE_MAP);
		satmodules_add_module (name, SATMOD_TYPE_MAP);
	}
	/* Create config file path and get sats */
	configfile = g_strconcat ("=", g_get_home_dir (), "/.gpredict/",
				  name, ".satmap=", SATMAP_SAT_LIST_PATH, NULL);
	gnome_config_get_vector (configfile, &argc, &argv);
	g_free (configfile);

	/* Empty list? */
	if (!g_strcasecmp (argv[0], "")) {
		/**********  FIXME: fix up error dialog */
		satmodules_remove_module (name, SATMOD_TYPE_MAP);
		return -1;
	}

	/* get some default values */
	/**** FIXME: Add more */
	delay = gconf_client_get_int (client, SATMAP_TIMER_PATH, NULL);
	if (!delay)
		delay = SATMAP_DEF_TIMER;

	/****  buttons  ****/
	/* close button */
	close = gpredict_mini_button ("icons", "module-close.png", _("Close this map"));
	gtk_button_set_relief (GTK_BUTTON (close), GTK_RELIEF_NONE);

	/* detach */
	tabwin = gpredict_mini_button ("icons", "module-detach.png", _("Detach map from notebook"));
	gtk_button_set_relief (GTK_BUTTON (tabwin), GTK_RELIEF_NONE);

	/* save */
	save = gpredict_pixmap_button (_("Save"), "icons", "save_as_20.png", _("Save map to file"));

	/* print */
	print = gpredict_pixmap_button (_("Print"), "icons", "print_20.png", _("Print map"));

	/* info */
	info = gpredict_stock_pixmap_button (_("Info"), GNOME_STOCK_PIXMAP_HELP,
					     _("Show Keplerian elements for the selected satellite"));

	/* AOS/LOS */
	maos = gpredict_stock_pixmap_button (_("AOS/LOS"), GNOME_STOCK_PIXMAP_TIMER,
					     _("Show AOS and LOS times for upcoming passes"));

	/* Next AOS */
	saos = gpredict_stock_pixmap_button (_("Next AOS"), GNOME_STOCK_PIXMAP_PROPERTIES,
					     _("Get detailed info about the upcoming pass"));

	hbox = gtk_hbutton_box_new ();
	gtk_button_box_set_spacing (GTK_BUTTON_BOX (hbox), 0);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (hbox), GTK_BUTTONBOX_START);
	gtk_box_pack_start (GTK_BOX (hbox), save, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), print, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), GTK_WIDGET (info), TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), GTK_WIDGET (maos), TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), GTK_WIDGET (saos), TRUE, TRUE, 0);


	/* upper hbox to contain detach and close buttons */
	topbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_end (GTK_BOX (topbox), close, FALSE, FALSE, 0);
	gtk_box_pack_end (GTK_BOX (topbox), tabwin, FALSE, FALSE, 0);


	/* Load Earth map */
	configfile = g_strconcat ("=", g_get_home_dir (), "/.gpredict/",
				  name, ".satmap=", SATMAP_SAT_MAP_PATH, NULL);
	tmp = gnome_config_get_string (configfile);
	g_free (configfile);

	if (!tmp) {
		/* try default map */
		tmp = gconf_client_get_string (client, SATMAP_DEFMAP_PATH, NULL);
		if (!tmp) {
			/* try hardcoded version */
			tmp = g_strconcat (PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S, "maps",
					   G_DIR_SEPARATOR_S, "earth_800.jpg", NULL);
		}
	}
	map = gdk_pixbuf_new_from_file (tmp);
	g_free (tmp);

	mapw = gdk_pixbuf_get_width (map);
	maph = gdk_pixbuf_get_height (map);

	/* Gnome Canvas */
	gtk_widget_push_visual (gdk_rgb_get_visual ());
	gtk_widget_push_colormap (gdk_rgb_get_cmap ());
	satmap = gnome_canvas_new ();
	gtk_widget_pop_visual ();
	gtk_widget_pop_colormap ();
	gtk_widget_set_usize (satmap, mapw + SATMAP_CANVAS_EXTRA_W, maph + SATMAP_CANVAS_EXTRA_H);
	gnome_canvas_set_pixels_per_unit (GNOME_CANVAS (satmap), 1);
	gnome_canvas_set_scroll_region (GNOME_CANVAS (satmap),
					0.0, 0.0,
					mapw + SATMAP_CANVAS_EXTRA_W,
					maph + SATMAP_CANVAS_EXTRA_H);

	/* create black border */
	bgd = gnome_canvas_item_new (gnome_canvas_root (GNOME_CANVAS (satmap)),
				     gnome_canvas_rect_get_type (),
				     "x1", (gdouble) 0.0,
				     "y1", (gdouble) 0.0,
				     "x2", (gdouble) (mapw + SATMAP_CANVAS_EXTRA_W),
				     "y2", (gdouble) (maph + SATMAP_CANVAS_EXTRA_H),
				     "fill_color", "black",
				     NULL);

	/* Create a main group which corresponds to the position and
	   size of the map. This is to allow a black border around the
	   world map. Thus all canvas items that should be displayed
	   on the map should use this group instead of the canvas group
	   (gnome_canvas_root).
	*/
	mapgroup = gnome_canvas_item_new (gnome_canvas_root (GNOME_CANVAS (satmap)),
					  gnome_canvas_group_get_type (),
					  "x", SATMAP_CANVAS_EXTRA_W/2.0,
					  "y", SATMAP_CANVAS_EXTRA_H/2.0,
					  NULL);

	/* create a new map canvas item */
	mapitem = gnome_canvas_item_new (GNOME_CANVAS_GROUP (mapgroup),
					 gnome_canvas_pixbuf_get_type (),
					 "pixbuf", map,
					 NULL);
	gdk_pixbuf_unref (map);

	
	/* QTH dot and label on the map */
	qthmark = gnome_canvas_item_new (GNOME_CANVAS_GROUP (mapgroup),
					 gnome_canvas_rect_get_type (),
					 "x1", satmap_cx (mapw, qth.lon)-1.0,
					 "y1", satmap_cy (maph, qth.lat)-1.0,
					 "x2", satmap_cx (mapw, qth.lon)+1.0,
					 "y2", satmap_cy (maph, qth.lat)+1.0,
					 "fill_color", "cyan",
					 NULL);
	qthlabel = gnome_canvas_item_new (GNOME_CANVAS_GROUP (mapgroup),
					  gnome_canvas_text_get_type (),
					  "text", qth.name,
					  "x", satmap_cx (mapw, qth.lon)+3.0,
					  "y", satmap_cy (maph, qth.lat),
					  "anchor", GTK_ANCHOR_W,
					  "font", "-*-clean-medium-r-*-*-12-*-*-*-*-*-*-*",
					  "fill_color", "cyan",
					  NULL);

	/* HEADER 1: QTH info on the upper left corner */
	/* Create the QTH text depending on whether we have QRA locator or not */
	if (qth.qra != NULL) { 
		/* Use the QRA locator */
		tmp = g_strdup_printf ("%s - %s - %s - %.2f%c%c %.2f%c%c",
				       qth.name, qth.loc, qth.qra,
				       fabs (qth.lon), SATMAP_DEGREE, (qth.lon < 0.0000) ? 'E' : 'W',
				       fabs (qth.lat), SATMAP_DEGREE, (qth.lat < 0.0000) ? 'S' : 'N');
	}
	else {
		/* Without the locator square */
		tmp = g_strdup_printf ("%s - %s - %.2f%c%c %.2f%c%c",
				       qth.name, qth.loc,
				       fabs (qth.lon), SATMAP_DEGREE, (qth.lon < 0.0000) ? 'E' : 'W',
				       fabs (qth.lat), SATMAP_DEGREE, (qth.lat < 0.0000) ? 'S' : 'N');
	}
	/* Create canvas item */
	header1 = gnome_canvas_item_new (gnome_canvas_root (GNOME_CANVAS (satmap)),
					 gnome_canvas_text_get_type (),
					 "text", tmp,
					 "x", (gfloat) SATMAP_CANVAS_EXTRA_W,
					 "y", 2.0,
					 "anchor", GTK_ANCHOR_NW,
					 "font", "-*-clean-medium-r-*-*-12-*-*-*-*-*-*-*",
					 "fill_color", "goldenrod1",
					 NULL);
	g_free (tmp);

	/* HEADER 2: Time */
	header2 = gnome_canvas_item_new (gnome_canvas_root (GNOME_CANVAS (satmap)),
					 gnome_canvas_text_get_type (),
					 "text", "",
					 "x", (gfloat) mapw,
					 "y", 2.0,
					 "anchor", GTK_ANCHOR_NE,
					 "font", "-*-clean-medium-r-*-*-12-*-*-*-*-*-*-*",
					 "fill_color", "goldenrod1",
					 NULL);

	/* FOOTER 1: Cursor coordinates */
	footer1 = gnome_canvas_item_new (gnome_canvas_root (GNOME_CANVAS (satmap)),
					 gnome_canvas_text_get_type (),
					 "text", "",
					 "x", (gfloat) SATMAP_CANVAS_EXTRA_W,
					 "y", (gfloat) maph + SATMAP_CANVAS_EXTRA_H,
					 "anchor", GTK_ANCHOR_SW,
					 "font", "-*-clean-medium-r-*-*-12-*-*-*-*-*-*-*",
					 "fill_color", "goldenrod1",
					 NULL);
	/* the info footer has to contain the index of the satellite to which
	   the mouse currently points at. -1 means no satellite.
	*/
	gtk_object_set_data (GTK_OBJECT (footer1), "index", GINT_TO_POINTER (-1));

	/* FOOTER 2: Upcoming pass */
	footer2 = gnome_canvas_item_new (gnome_canvas_root (GNOME_CANVAS (satmap)),
					 gnome_canvas_text_get_type (),
					 "text", "",
					 "x", (gfloat) mapw,
					 "y", (gfloat) maph + SATMAP_CANVAS_EXTRA_H,
					 "anchor", GTK_ANCHOR_SE,
					 "font", "-*-clean-medium-r-*-*-12-*-*-*-*-*-*-*",
					 "fill_color", "goldenrod1",
					 NULL);

	/* create list of satellites */
	for (i=0; i<argc; i++) {
		/* check whether we really have this satellite in the database */
		satindex = satdata_get_sat_index (0, argv[i]);
		if (satindex!= -1) {
			smsat = SATMAP_SAT (g_malloc (sizeof (satmap_sat_t)));
			smsat->name = g_strdup (argv[i]);
			smsat->index = satindex;
			sat = satdata_get_sat (smsat->index);
			/* canvas group for this satellite (reference coords) */
			smsat->group = gnome_canvas_item_new (GNOME_CANVAS_GROUP (mapgroup),
							      gnome_canvas_group_get_type (),
							      "x", satmap_cx (mapw, sat->lon),
							      "y", satmap_cy (maph, sat->lat),
							      NULL);
			smsat->mark = gnome_canvas_item_new (GNOME_CANVAS_GROUP (smsat->group),
							     gnome_canvas_rect_get_type (),
							     "x1", -1.0, "y1", -1.0,
							     "x2", 1.0, "y2", 1.0,
							     "fill_color", "yellow",
							     NULL);
			smsat->text = gnome_canvas_item_new (GNOME_CANVAS_GROUP (smsat->group),
							     gnome_canvas_text_get_type (),
							     "text", smsat->name,
							     "x", 0.0, "y", 1.0,
							     "anchor", GTK_ANCHOR_N,
							     "font", "-*-clean-medium-r-*-*-12-*-*-*-*-*-*-*",
							     "fill_color", "yellow",
							     NULL);
			
			/* Range Circle Calculations
			   Borrowed from gsat 0.9.0 by Xavier Crehueras, EB3CZS
			   who borrowed from John Magliacane, KD2BD.
			*/
			ssplat = sat->lat*deg2rad;
			ssplong = sat->lon*deg2rad;
			beta = (0.5*sat->fp)/R0;
			
			for (azi=0; azi<SATMAP_RANGE_CIRCLE_POINTS; azi++)
			{
				azimuth = deg2rad*(double)azi;
				rangelat = asin(sin(ssplat)*cos(beta)+cos(azimuth)*sin(beta)*cos(ssplat));
				num = cos(beta)-(sin(ssplat)*sin(rangelat));
				dem = cos(ssplat)*cos(rangelat);
				
				if (azi==0 && (beta > HALFPI-ssplat))
					rangelong = ssplong+PI;
				
				else if (azi==180 && (beta > HALFPI+ssplat))
					rangelong = ssplong+PI;
				
				else if (fabs(num/dem) > 1.0)
					rangelong = ssplong;
				
				else {
					if ((180-azi) >= 0)
						rangelong = ssplong-arccos(num,dem);
					else
						rangelong = ssplong+arccos(num,dem);
				}
				
				while (rangelong < 0.0)
					rangelong += TWOPI;
				
				while (rangelong > (2.0*PI))
					rangelong -= TWOPI;
				
				rangelat = rangelat/deg2rad;
				rangelong = rangelong/deg2rad;
				
				/* Convert range circle data to map-based
				   coordinates and draw on map */
				
				if (rangelong > 180.0)
				{
					rangelong = rangelong-180.0;
					sx = (gint) (mapw-(rangelong*mapw/360.0));
				}
				else
					sx = (gint)((mapw/2.0)-(rangelong*mapw/360.0));
				
				sy = (gint)((maph/2.0)-(rangelat*maph/180.0));
				
				/* store position for Plot Range Circle*/
				/**** FIXME: VERY SLOW!!!!! */
				smsat->range[azi] = gnome_canvas_item_new (GNOME_CANVAS_GROUP (mapgroup),
									   gnome_canvas_rect_get_type (),
									   "x1", sx, "x2", sx,
									   "y1", sy, "y2", sy,
									   "fill_color", "yellow",
									   "outline_color", "yellow",
									   "width_pixels", 1,
									   NULL);
				/* fixes bug #537372 */
				if (sx < satmap_cx (mapw, sat->lon))
					sx += mapw;
				/* Mirror the half circle to the left */
				sx2 = sx-2*(sx - satmap_cx (mapw, sat->lon));
				if (sx2 < 0.0) /* outside the map */
					sx2 = mapw + sx2; 
				smsat->range[azi+180] = gnome_canvas_item_new (GNOME_CANVAS_GROUP (mapgroup),
									       gnome_canvas_rect_get_type (),
									       "x1", sx2, "x2", sx2,
									       "y1", sy, "y2", sy,
									       "fill_color", "yellow",
									       "outline_color", "yellow",
									       "width_pixels", 1,
									       NULL);
				
			}
			/* attach satellite item to list */
			slist = g_list_append (slist, smsat);

			/* attach pointer to footer label so that we can change
			   the text from the callback handlers.
			*/
			gtk_object_set_data (GTK_OBJECT (smsat->mark), "sati", footer1);
			gtk_object_set_data (GTK_OBJECT (smsat->text), "sati", footer1);

			/* connect event handler */
			gtk_signal_connect (GTK_OBJECT (smsat->mark), "event",
					    GTK_SIGNAL_FUNC (satmap_map_event_cb),
					    smsat);
			gtk_signal_connect (GTK_OBJECT (smsat->text), "event",
					    GTK_SIGNAL_FUNC (satmap_map_event_cb),
					    smsat);

		}
		else {
			logmsg = g_strdup_printf (_("%s: Satellite %s not in database!"),
						  __FUNCTION__, argv[i]);
			satlog_log (SAT_LOG_WARNING, logmsg);
			g_free (logmsg);
		}
	}
	g_strfreev (argv);

	/* create detach and close buttons */


	/* Create main vertical box */
	vbox = gtk_vbox_new (FALSE, 0);

	/* start timers */
	timer  = gtk_timeout_add (delay, satmap_timer_cb, vbox);
	timer2 = gtk_timeout_add (1007, satmap_timer2_cb, vbox);

	/* attach some data to the main container widget */
	gtk_object_set_data (GTK_OBJECT (vbox), "timer",
			     GINT_TO_POINTER (timer));              /* main timer */
	gtk_object_set_data (GTK_OBJECT (vbox), "timer2",
			     GINT_TO_POINTER (timer2));             /* clock and info timer */
	gtk_object_set_data (GTK_OBJECT (vbox), "name", name);      /* module name */
	gtk_object_set_data (GTK_OBJECT (vbox), "list", slist);     /* satellite list */
	gtk_object_set_data (GTK_OBJECT (vbox), "mode",
			     GUINT_TO_POINTER (SATMAP_MODE_TAB));   /* display mode */
	gtk_object_set_data (GTK_OBJECT (vbox), "qthm", qthmark);   /* QTH dot */
	gtk_object_set_data (GTK_OBJECT (vbox), "qthl", qthlabel);  /* QTH label */
	gtk_object_set_data (GTK_OBJECT (vbox), "qthi", header1);   /* QTH info */
	gtk_object_set_data (GTK_OBJECT (vbox), "clock", header2);  /* Clock */
	gtk_object_set_data (GTK_OBJECT (vbox), "sati", footer1);   /* pop-up sat info */
	gtk_object_set_data (GTK_OBJECT (vbox), "next", footer2);   /* next AOS info */
	gtk_object_set_data (GTK_OBJECT (vbox), "mapw",
			     GUINT_TO_POINTER (mapw));              /* Map width */
	gtk_object_set_data (GTK_OBJECT (vbox), "maph",
			     GUINT_TO_POINTER (maph));              /* Map height */



	gtk_box_pack_start (GTK_BOX (vbox), topbox, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), satmap, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

	/* add gconf notifier (module reloads prefs autiomatically) */
	gconf_id = gconf_client_notify_add (client, SATMAP_ROOT_PATH,
					    satmap_prefs_changed,
					    vbox, NULL, NULL);
	gtk_object_set_data (GTK_OBJECT (vbox), "gconfid", GUINT_TO_POINTER (gconf_id));

	/* Connect button signals or disable */
	gtk_signal_connect (GTK_OBJECT (save), "clicked",
			    GTK_SIGNAL_FUNC (satmap_save_cb), satmap);
	gtk_signal_connect (GTK_OBJECT (print), "clicked",
			    GTK_SIGNAL_FUNC (satmap_print_cb), satmap);
	gtk_signal_connect (GTK_OBJECT (tabwin), "clicked",
			    GTK_SIGNAL_FUNC (satmap_switch_mode_cb), vbox);
	gtk_signal_connect (GTK_OBJECT (close), "clicked",
			    GTK_SIGNAL_FUNC (satmap_close_cb), vbox);
	gtk_signal_connect (GTK_OBJECT (info), "clicked",
			    GTK_SIGNAL_FUNC (satmap_info_cb), vbox);
	gtk_signal_connect (GTK_OBJECT (maos), "clicked",
			    GTK_SIGNAL_FUNC (satmap_multi_aos_cb), vbox);
	gtk_signal_connect (GTK_OBJECT (saos), "clicked",
			    GTK_SIGNAL_FUNC (satmap_single_aos_cb), vbox);

	gtk_widget_show_all (vbox);
	menulabel = g_strconcat (name, _(" (map)"), NULL);
	sattabs_add_tab_menu (vbox, name, menulabel);
	g_free (menulabel);

// you can open in window using this...
//	satmap_switch_mode_cb (tabwin, vbox);

	return 0;
}


static void
satmap_switch_mode_cb (GtkWidget *button, GtkWidget *vbox)
{
	/* This function is called when the user clicks on the
	   Tab/Window button.
	*/
	guint        mode;
	gchar       *name,*menulabel,*buff;
	gint         page;
	GtkWidget   *satwin;
	GtkWidget   *pixmap;
	gchar       *pixmappath;
	GtkTooltips *tips;


	mode = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (vbox), "mode"));
	name = gtk_object_get_data (GTK_OBJECT (vbox), "name");
	switch (mode) {
	case SATMAP_MODE_TAB:
		page = sattabs_lookup_tab (vbox);
		if (page != -1) {

			/* delete map from notebook */
			gtk_widget_ref (vbox);
			sattabs_remove_tab (page);

			/* update state */
			gtk_object_set_data (GTK_OBJECT (vbox), "mode",
					     GUINT_TO_POINTER (SATMAP_MODE_WIN));

			/* create window */
			satwin = gtk_window_new (GTK_WINDOW_TOPLEVEL);
			gtk_object_set_data (GTK_OBJECT (vbox), "window", satwin);
			gtk_window_set_title (GTK_WINDOW (satwin), name);

			/* window icon */
			buff = g_strconcat (PACKAGE_PIXMAPS_DIR, "/icons/map.png", NULL);
			gnome_window_icon_set_from_file (GTK_WINDOW (satwin), buff);
			g_free (buff);

			/* connect window signals */
			gtk_signal_connect (GTK_OBJECT (satwin), "delete-event",
					    GTK_SIGNAL_FUNC (satwin_delete_cb), vbox);
			gtk_signal_connect (GTK_OBJECT (satwin), "destroy",
					    GTK_SIGNAL_FUNC (satwin_close_cb), vbox);
			gtk_container_add (GTK_CONTAINER (satwin), vbox);

			/* change icon to attach */
			pixmap = GTK_WIDGET (gtk_object_get_data (GTK_OBJECT (button), "pixmap"));
			gtk_container_remove (GTK_CONTAINER (button), pixmap);
			gtk_object_remove_data (GTK_OBJECT (button), "pixmap");
			pixmappath = g_strconcat (PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S, "icons",
						  G_DIR_SEPARATOR_S, "module-attach.png", NULL);
			pixmap = gnome_pixmap_new_from_file (pixmappath);
			gtk_container_add (GTK_CONTAINER (button), pixmap);
			gtk_object_set_data (GTK_OBJECT (button), "pixmap", pixmap);
			g_free (pixmappath);

			tips = gtk_tooltips_new ();
			gtk_tooltips_set_tip (tips, button,
					      _("Attach map to notebook"), NULL);

			gtk_widget_show_all (satwin);
		}
		break;

	case SATMAP_MODE_WIN:
		/* get and destroy the window */
		satwin = gtk_object_get_data (GTK_OBJECT (vbox), "window");
		gtk_object_remove_data (GTK_OBJECT (vbox), "window");
		gtk_container_remove (GTK_CONTAINER (satwin), vbox);
		gtk_widget_destroy (satwin);
		/* reconfigure module */
		gtk_object_set_data (GTK_OBJECT (vbox), "mode",
				     GUINT_TO_POINTER (SATMAP_MODE_TAB));
		menulabel = g_strconcat (name, _(" (map)"), NULL);
		sattabs_add_tab_menu (vbox, name, menulabel);
		g_free (menulabel);

		/* change icon to detach */
		pixmap = GTK_WIDGET (gtk_object_get_data (GTK_OBJECT (button), "pixmap"));
		gtk_container_remove (GTK_CONTAINER (button), pixmap);
		gtk_object_remove_data (GTK_OBJECT (button), "pixmap");
		pixmappath = g_strconcat (PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S, "icons",
					  G_DIR_SEPARATOR_S, "module-detach.png", NULL);
		pixmap = gnome_pixmap_new_from_file (pixmappath);
		gtk_container_add (GTK_CONTAINER (button), pixmap);
		gtk_object_set_data (GTK_OBJECT (button), "pixmap", pixmap);
		g_free (pixmappath);

		tips = gtk_tooltips_new ();
		gtk_tooltips_set_tip (tips, button,
				      _("Detach map from notebook"), NULL);

		gtk_widget_show_all (button);

		break;

	case SATMAP_MODE_FULLSCR:
		break;
	default:
		satlog_log (SAT_LOG_CRITICAL, _("SATMAP: Invalid mode (BUG!)"));
	}

}

static void
satmap_save_cb  (GtkWidget *widget, GtkWidget *canvas)
{
	/* This function is called when the user clicks on the
	   Save button. It pops up a file selector dialog.
	*/
	GdkImlibImage *tmp = 0;
	GtkWidget *filesel,*filter,*hbox,*label;
	gdouble x1,x2,y1,y2;

	gnome_canvas_get_scroll_region (GNOME_CANVAS (canvas),
					&x1, &y1, &x2, &y2);

	/* get image */
	tmp = gdk_imlib_create_image_from_drawable (canvas->window, NULL,
						    (gint) x1, (gint) y1,
						    (gint) x2, (gint) y2);

	/* create fileselector widget and attach image */
	filesel = gtk_file_selection_new (_("Save Map As..."));
	gtk_object_set_data (GTK_OBJECT (filesel), "image", (gpointer) tmp);
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION(filesel)->ok_button),
			    "clicked", GTK_SIGNAL_FUNC (satmap_store_filename),
			    (gpointer) filesel);
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION(filesel)->ok_button),
			    "clicked", GTK_SIGNAL_FUNC (satmap_filesel_destroy_cb),
			    (gpointer) filesel);
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION(filesel)->cancel_button),
			    "clicked", GTK_SIGNAL_FUNC (satmap_filesel_destroy_cb),
			    (gpointer) filesel);

	/* Create filter widget */
	label = gtk_label_new (_("File type:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	filter = filesel_graphic_filter ();
	gtk_object_set_data (GTK_OBJECT (filesel), "filter", filter);

	hbox = gtk_hbox_new (TRUE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 5);
	gtk_box_pack_start (GTK_BOX (hbox), filter, TRUE, TRUE, 5);
	gtk_container_add (GTK_CONTAINER (GTK_FILE_SELECTION (filesel)->main_vbox),
			   hbox);

	gtk_widget_show_all (filesel);
}


/* in extra-widgets.c */
extern const gchar *gfx_filter_desc[];
extern const gchar *gfx_filter_ext[];


static void
satmap_store_filename (GtkWidget *button, GtkWidget *filesel)
{
	/* This function is called when the user clicks on the
	   OK-button of the file selector dialog.
	*/
	gchar *fname=NULL,*tmp=NULL;
	gint i=0;
	guint filti;
	GtkWidget *menu,*item;

	menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (gtk_object_get_data (GTK_OBJECT (filesel), "filter")));
	item = gtk_menu_get_active (GTK_MENU (menu));
	filti = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (item), "filter"));
	if (filti >= GFX_FILTER_NUM) {
		satlog_log (SAT_LOG_CRITICAL, _("Selected file type out of range (bug)"));
		gnome_app_error (GNOME_APP (app), _("Selected file type out of range (bug)"));
		return;
	}
	if (filti) {
		/* User selected type */
		tmp = gtk_file_selection_get_filename (GTK_FILE_SELECTION (filesel));
		if (tmp) {
			fname = g_strconcat (tmp, ".", gfx_filter_ext[filti], NULL);
			i = gdk_imlib_save_image ((GdkImlibImage *) gtk_object_get_data (GTK_OBJECT (filesel), "image"),
						  fname, 0);
		}
	}
	else {
		/* Automatic file type from extension */
		fname = gtk_file_selection_get_filename (GTK_FILE_SELECTION (filesel));
		if (fname) {
			i = gdk_imlib_save_image((GdkImlibImage *) gtk_object_get_data (GTK_OBJECT (filesel),
											"image"),
						 fname, 0);
		}
	}

	if (i)
		satlog_log (SAT_LOG_INFO, _("SATMAP: gdk_imlib_save() returned non-zero value."));
}


static void
satmap_filesel_destroy_cb (GtkWidget *button, GtkWidget *filesel)
{
	gdk_imlib_destroy_image ((GdkImlibImage *) gtk_object_get_data (GTK_OBJECT (filesel), "image"));
	gtk_widget_destroy (GTK_WIDGET (gtk_object_get_data (GTK_OBJECT (filesel), "filter")));
	gtk_widget_destroy (filesel);
}



static void
satmap_close_cb (GtkWidget *button, GtkWidget *vbox)
{
	/* This function is called when the user clicks on
	   the "Close" button. It checks whether the Satmap
	   module is in a notebook tab or in a window, and
	   removes the widget.
	*/
	gint tabpos;
	GtkWidget *window;
	gchar *module;

	/* stop timers */
	gtk_timeout_remove (GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (vbox), "timer")));
	gtk_timeout_remove (GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (vbox), "timer2")));

	gtk_object_remove_data (GTK_OBJECT (vbox), "timer");
	gtk_object_remove_data (GTK_OBJECT (vbox), "timer2");

	/* remove GConf notifier */
	gconf_client_notify_remove (client, GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (vbox), "gconfid")));

	module = g_strdup (gtk_object_get_data (GTK_OBJECT (vbox), "name"));
	gtk_object_remove_data (GTK_OBJECT (vbox), "name");
	gtk_object_remove_data (GTK_OBJECT (vbox), "mode");
	gtk_object_remove_data (GTK_OBJECT (vbox), "qthm");
	gtk_object_remove_data (GTK_OBJECT (vbox), "qthl");
	gtk_object_remove_data (GTK_OBJECT (vbox), "mapw");
	gtk_object_remove_data (GTK_OBJECT (vbox), "maph");

	/* re-add module to iconbar */
	iconbar_add_module (module, SATMOD_TYPE_MAP);
	/* remove module from list */
	satmodules_remove_module (module, SATMOD_TYPE_MAP );
	g_free (module);

	tabpos = sattabs_lookup_tab (vbox);
	if (tabpos != -1) {
		sattabs_remove_tab (tabpos);
		satmap_close_cleanup (vbox);
		gtk_widget_destroy (vbox);
	}
	else {
		window = gtk_object_get_data (GTK_OBJECT (vbox), "window");
		gtk_object_remove_data (GTK_OBJECT (vbox), "window");
		satmap_close_cleanup (vbox);
		gtk_widget_destroy (vbox);
		if (window)
			gtk_widget_destroy (window);
	}
}


static void
satmap_close_cleanup (GtkWidget *vbox)
{
	/* This function cleans up some memory that has been
	   allocated to a satmap. The data is attached to the
	   "vbox".
	*/
	GList *slist;

        slist = (GList *) gtk_object_get_data (GTK_OBJECT (vbox), "list");
	gtk_object_remove_data (GTK_OBJECT (vbox), "list");
	while (slist) {
		g_free (SATMAP_SAT (slist->data)->name);
		g_free (slist->data);
		slist = g_list_next (slist);
	}
	slist = g_list_first (slist);
	g_list_free (slist);
}


static gint
satmap_timer_cb (gpointer data)
{
	/* This function does the update on the satmap.
	   It is called automatically at regular intervals.
	*/
	GList *list;
	satmap_sat_t *ssat;
	sat_t *sat;
	/* range circle */
	gdouble ssplat,ssplong,beta,azimuth,rangelat,rangelong,num,dem,sx,sy,sx2;
	gdouble deg2rad=1.74532925199e-2;
	gchar *logmsg,*tmp;
	guint azi;
	GnomeCanvasItem *item;
	guint mapw,maph;
	gdouble aosbuffer = -1.0;    /* used to store the next AOS */
	gint    aosindex  = 0;       /* index pointing to the satellite which has AOS at aosbuffer */


	/* getm map width and height */
	mapw = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (data), "mapw"));
	maph = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (data), "maph"));

	/* update QTH mark */
	item = GNOME_CANVAS_ITEM (gtk_object_get_data (GTK_OBJECT (data), "qthm"));
	gnome_canvas_item_set (item,
			       "x1", satmap_cx (mapw, qth.lon) - 1.0,
			       "y1", satmap_cy (maph, qth.lat) - 1.0,
			       "x2", satmap_cx (mapw, qth.lon) + 1.0,
			       "y2", satmap_cy (maph, qth.lat) + 1.0,
			       NULL);

	/* update QTH label */
	item = GNOME_CANVAS_ITEM (gtk_object_get_data (GTK_OBJECT (data), "qthl"));
	gnome_canvas_item_set (item, "text", qth.name,
			       "x", satmap_cx (mapw, qth.lon) + 3.0,
			       "y", satmap_cy (maph, qth.lat),
			       NULL);

	/* update QTH info header (HEADER 1) */
	item = GNOME_CANVAS_ITEM (gtk_object_get_data (GTK_OBJECT (data), "qthi"));
	if (qth.qra != NULL) { 
		/* Use the QRA locator */
		tmp = g_strdup_printf ("%s - %s - %s - %.2f%c%c %.2f%c%c",
				       qth.name, qth.loc, qth.qra,
				       fabs (qth.lon), SATMAP_DEGREE, (qth.lon < 0.0000) ? 'E' : 'W',
				       fabs (qth.lat), SATMAP_DEGREE, (qth.lat < 0.0000) ? 'S' : 'N');
	}
	else {
		/* Without the locator square */
		tmp = g_strdup_printf ("%s - %s - %.2f%c%c %.2f%c%c",
				       qth.name, qth.loc,
				       fabs (qth.lon), SATMAP_DEGREE, (qth.lon < 0.0000) ? 'E' : 'W',
				       fabs (qth.lat), SATMAP_DEGREE, (qth.lat < 0.0000) ? 'S' : 'N');
	}
	gnome_canvas_item_set (item, "text", tmp, NULL);
	g_free (tmp);

	/* get the list of satellites */
	list = (GList *) gtk_object_get_data (GTK_OBJECT (data), "list");

	while (list) {
		/* get next item */
		ssat = (satmap_sat_t *)list->data;

		/* get pointer to current satellite data */
		sat = satdata_get_sat (ssat->index);

		if (sat) {

			/* update NEXT AOS (FOOTER 2) */
			/* Check whether this satellite will have AOS earlier
			   than the time which is in the AOS buffer.
			   First check whether this satellite is applicable.
			*/
			if (sat->hasaos  && !sat->geostat && !sat->decayed) {
				/* If aosbuffer is negative, then it is the first
				   applicable satellite, since time can't be negative.
				*/ 
				if ((aosbuffer < 0.0) || (sat->aos < aosbuffer)) {
					aosbuffer = sat->aos;
					aosindex  = ssat->index;
			    }
			}

			gnome_canvas_item_set (ssat->group,
					       "x", satmap_cx (mapw, sat->lon),
					       "y", satmap_cy (maph, sat->lat),
					       NULL);
			/* we need to do this, otherwise canvas won't get updated */
			gnome_canvas_item_set (ssat->mark,
					       "x1", -1.0,
					       NULL);
			gnome_canvas_item_set (ssat->text,
					       "x", 0.0, "y", 1.0,
					       NULL);
			/* Range Circle Calculations
			   Borrowed from gsat 0.9.0 by Xavier Crehueras, EB3CZS
			   who borrowed from John Magliacane, KD2BD.
			*/
			/* as of CVS rev. 1.36 some optimizations have been done by alexc */
			ssplat = sat->lat*deg2rad;
			ssplong = sat->lon*deg2rad;
			beta = (0.5*sat->fp)/R0;
			for (azi=0; azi<SATMAP_RANGE_CIRCLE_POINTS; azi++)
			{
				azimuth = deg2rad*(double)azi;
				rangelat = asin(sin(ssplat)*cos(beta)+cos(azimuth)*sin(beta)*cos(ssplat));
				num = cos(beta)-(sin(ssplat)*sin(rangelat));
				dem = cos(ssplat)*cos(rangelat);
				
				if (azi==0 && (beta > HALFPI-ssplat))
					rangelong = ssplong+PI;
				
				else if (azi==180 && (beta > HALFPI+ssplat))
					rangelong = ssplong+PI;
				
				else if (fabs(num/dem) > 1.0)
					rangelong = ssplong;
				
				else {
					if ((180) >= azi)
						rangelong = ssplong-arccos(num,dem);
					else
						rangelong = ssplong+arccos(num,dem);
				}
				
				/* OPTIMIZATION:
				   Ususally, the first while is executed once (at most)
				*/
				while (rangelong < 0.0)
					rangelong += TWOPI;
				
				while (rangelong > (TWOPI))
					rangelong -= TWOPI;

				
				rangelat = rangelat/deg2rad;
				rangelong = rangelong/deg2rad;
				
				/* Convert range circle data to map-based
				   coordinates and draw on map */
				if (rangelong > 180.0)
				{
					rangelong = rangelong-180.0;
					sx = (gint) (mapw - (rangelong * mapw/360.0));
				}
				else
					sx = (gint) ((mapw/2.0) - (rangelong*mapw/360.0));
				
				sy = (gint) ((maph/2.0)-(rangelat * maph/180.0));
				
				/* store position for Plot Range Circle*/
				/**** FIXME: VERY SLOW!!!!! */
/*  			gnome_canvas_item_set (ssat->range[azi], "x1", sx, "x2", sx, */
/*  					       "y1", sy, "y2", sy, NULL); */
				/* Let's try to access rect->re coordinates directly ... uuuhh... */
				GNOME_CANVAS_RECT (ssat->range[azi])->re.x1 = sx;
				GNOME_CANVAS_RECT (ssat->range[azi])->re.x2 = sx;
				GNOME_CANVAS_RECT (ssat->range[azi])->re.y1 = sy;
				GNOME_CANVAS_RECT (ssat->range[azi])->re.y2 = sy;
				gnome_canvas_item_request_update (ssat->range[azi]);
				/* fixes bug #537372 */
				if (sx < satmap_cx (mapw, sat->lon))
					sx += mapw;
				/* Mirror the half circle to the left */
				sx2 = sx-2*(sx-satmap_cx (mapw, sat->lon));
				if (sx2 < 0.0) /* outside the map */
					sx2 = mapw+sx2;
				/* TO SLOOOOW... */
/*  			gnome_canvas_item_set (ssat->range[azi+180], "x1", sx2, "x2", sx2, */
/*  					       "y1", sy, "y2", sy, NULL); */
				GNOME_CANVAS_RECT (ssat->range[azi+180])->re.x1 = sx2;
				GNOME_CANVAS_RECT (ssat->range[azi+180])->re.x2 = sx2;
				GNOME_CANVAS_RECT (ssat->range[azi+180])->re.y1 = sy;
				GNOME_CANVAS_RECT (ssat->range[azi+180])->re.y2 = sy;
				gnome_canvas_item_request_update (ssat->range[azi+180]);
			}
		}
		else {
			logmsg = g_strdup_printf (_("%s: Missing satellite detected. This satellite is now a zombie."),
						    __FUNCTION__);
			satlog_log (SAT_LOG_WARNING, logmsg);
			g_free (logmsg);
		}
		list = g_list_next (list);
	}

	/* update NEXT AOS (FOOTER 2) */
	if (aosbuffer > 0.0) {
		/* we have a meaningful AOS time */

		/* at this point aosindex points to the satellite which has the next AOS at time aosindex */
		sat = satdata_get_sat (aosindex);
 
		/* create text */
		tmp = g_strdup_printf ("NEXT: %s on %s", sat->name, dnum2lstr(sat->aos)); 

		/* get item and update text */
		item = GNOME_CANVAS_ITEM (gtk_object_get_data (GTK_OBJECT (data), "next"));
		gnome_canvas_item_set (item, "text", tmp, NULL);
		g_free (tmp);
	}

	return TRUE;
}


static gint
satmap_timer2_cb (gpointer data)
{
	/* This is the 1000 msec timer. It updates the clock on the map
	   and the info footer if applicable.
	*/
	GnomeCanvasItem *item;
	gchar *buff;
	struct tm *utc;
	time_t tim;
	sat_t *sat;
	gint index;

	/* Update clock header */
	tim = time (NULL);
 	if (clock_show_local) 
 		utc = localtime (&tim); 
 	else 
		utc = gmtime (&tim);
	
	buff = g_malloc (15);	
	strftime (buff, 15, "%H:%M:%S %Z", utc);

	/* get item and update text */
	item = GNOME_CANVAS_ITEM (gtk_object_get_data (GTK_OBJECT (data), "clock"));
	gnome_canvas_item_set (item, "text", buff, NULL);
	g_free (buff);


	/* Check whether the mouse is over a satellite. If so, update the info footer.
	   If the user points the mouse to a satellite, the info footer contains the 
	   index of the satellite.
	*/
	item = GNOME_CANVAS_ITEM (gtk_object_get_data (GTK_OBJECT (data), "sati"));

	/* get pointer to satellite structure */
	index = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (item), "index"));
	if (index >= 0) {
		sat = satdata_get_sat (index);

		/* Check whether satellite ever comes within range. If so, show the
		   AOS or LOS time, otherwise just the position.
		*/
		if (sat->hasaos) {

			/* If satellite is out of range we show next AOS, if satellite is
			   within range we show next LOS and the pointing coordinates.
			*/
			if (sat->el > 0.0) {
				buff = g_strdup_printf ("%s - %.2f%c%c %.2f%c%c - AZ:%.2f%c EL:%.2f%c - LOS: %s",
							sat->name,
							fabs (sat->lon), SATMAP_DEGREE, (sat->lon < 0.0000) ? 'E' : 'W',
							fabs (sat->lat), SATMAP_DEGREE, (sat->lat < 0.0000) ? 'S' : 'N',
							sat->az, SATMAP_DEGREE, sat->el, SATMAP_DEGREE, dnum2lstr (sat->los));
			}
			else {
				buff = g_strdup_printf ("%s - %.2f%c%c %.2f%c%c - AOS: %s",
							sat->name,
							fabs (sat->lon), SATMAP_DEGREE, (sat->lon < 0.0000) ? 'E' : 'W',
							fabs (sat->lat), SATMAP_DEGREE, (sat->lat < 0.0000) ? 'S' : 'N',
							dnum2lstr (sat->aos));
			}
		}
		else {
			if (sat->el > 0.0) {
				buff = g_strdup_printf ("%s - %.2f%c%c %.2f%c%c - AZ:%.2f%c EL:%.2f%c",
							sat->name,
							fabs (sat->lon), SATMAP_DEGREE, (sat->lon < 0.0000) ? 'E' : 'W',
							fabs (sat->lat), SATMAP_DEGREE, (sat->lat < 0.0000) ? 'S' : 'N',
							sat->az, SATMAP_DEGREE, sat->el, SATMAP_DEGREE);
			}
			else {
				buff = g_strdup_printf ("%s - %.2f%c%c %.2f%c%c - NO AOS",
							sat->name,
							fabs (sat->lon), SATMAP_DEGREE, (sat->lon < 0.0000) ? 'E' : 'W',
							fabs (sat->lat), SATMAP_DEGREE, (sat->lat < 0.0000) ? 'S' : 'N');
			}
		}
		gnome_canvas_item_set (item, "text", buff, NULL);
		g_free (buff);
	}

	return TRUE;
}

gint satwin_delete_cb (GtkWidget *widget, GdkEvent *event, GtkWidget *vbox)
{
	/* This function is connected to the "delete" event signal
	   of the satmap window. It returns false, so that the window
	   will be destroyed, ie. the next function will be called.
	*/
	satmap_close_cb (NULL, vbox);
	return FALSE;
}


void satwin_close_cb (GtkWidget *widget, GtkWidget *vbox)
{
	/****** FIXME: should clean up some memory */

	/* This function is connected to the "destroy" signal
	   of the satmap window. It calls the satmap_close
	   function, which in turn will clean up the memory
	   allocated for one module.
	*/



}



static gdouble arccos (gdouble x, gdouble y)
{
	/* This function implements the arccosine function,
	   returning a value between 0 and two pi.
	   Borrowed from gsat 0.9 by Xavier Crehueras, EB3CZS
	*/

	/* somewhat optimized version by alexc (as of CVS rev. 1.36 */

	if (x && y) {
		if (y > 0.0)
			return acos (x/y);
		else if (y < 0.0)
			return PI + acos (x/y);
	}

	return 0.0;
}


static void
satmap_prefs_changed (GConfClient *client, guint cid, GConfEntry *entry, gpointer vbox)
{
	/* This function reloads the preferences for the satellite map.
	   It is called automatically by GConf when a value under
	   /apps/gpredict/satmap is changed.
	*/

	gint timer, delay;

	/* stop timers */
	gtk_timeout_remove (GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (vbox), "timer")));
/*	gtk_timeout_remove (GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (vbox), "timer2"))); */

	/* get delay */
	delay = gconf_client_get_int (client, SATMAP_TIMER_PATH, NULL);
	if (!delay)
		delay = SATMAP_DEF_TIMER;

	/* restart timers and store timer IDs */
	timer = gtk_timeout_add (delay, satmap_timer_cb, vbox);
	gtk_object_set_data (GTK_OBJECT (vbox), "timer", GINT_TO_POINTER (timer));
} 


/* convert lat/lon to canvas coordinates */
static gdouble
satmap_cx (gint mapwidth, gdouble lon)
{
	/* This function replaces CX(lon) so that it
	   can be used with variable map sizes.
	*/

	return ((180.0-lon)*mapwidth/360.0);
}

static gdouble
satmap_cy (gint mapheight, gdouble lat)
{
	/* This function replaces CY(lat) so that it
	   can be used with variable map sizes.
	*/

	return ((90.0-lat)*mapheight/180.0);
}




static gint
satmap_map_event_cb   (GtkWidget *item,
		       GdkEvent  *event,
		       gpointer   data)
{
	/* handle map events */

	satmap_sat_t *smsat;
	GnomeCanvasItem *footer;
	sat_t *sat;
	gchar *buff;
	gint selection;   /* active satellite */
	gint selected;    /* whether satellite is selected by mouse click or just pointing at */
	
	/* Check the event type and source. If no match is found return FALSE
	   indicating that the event has not been treated and the parent object
	   should try to do it */
	switch (event->type) {
	case GDK_BUTTON_PRESS:
		switch (event->button.button) {

			/* the best is probably to use the left klick only! Then, the
			   sat is selected, and the buttons (info, aos, aos2, target, ...)
			   - actually the SAME butons for all modules - can be used!
			*/


		case 1: /* LEFT button */
			/* Select the satellite by highlighting it and showing static info about it.
			 */

			/* get footer item on map */
			footer  = GNOME_CANVAS_ITEM (gtk_object_get_data (GTK_OBJECT (item), "sati"));

			/* get satellite item (event source) */
			smsat = SATMAP_SAT (data);
				
			/* get pointer to satellite structure */
			sat = satdata_get_sat (smsat->index);
			
			/* Check whether the satellite ever comes within range. If yes, show
			   the position and the AOS time, otherwise just the position.
			*/
			if (sat->hasaos) {
				
				/* If satellite is within range we show LOS and pointing coordinates.
				   If satellite is out of range we show AOS.
				*/
				if (sat->el > 0.0) {
					buff = g_strdup_printf ("%s - %.2f%c%c %.2f%c%c - AZ:%.2f%c EL:%.2f%c - LOS: %s",
								sat->name,
								fabs (sat->lon), SATMAP_DEGREE, (sat->lon < 0.0000) ? 'E' : 'W',
								fabs (sat->lat), SATMAP_DEGREE, (sat->lat < 0.0000) ? 'S' : 'N',
								sat->az, SATMAP_DEGREE, sat->el, SATMAP_DEGREE,
								dnum2lstr(sat->los));
				}
				else {
					buff = g_strdup_printf ("%s - %.2f%c%c %.2f%c%c - AOS: %s",
								sat->name,
								fabs (sat->lon), SATMAP_DEGREE, (sat->lon < 0.0000) ? 'E' : 'W',
								fabs (sat->lat), SATMAP_DEGREE, (sat->lat < 0.0000) ? 'S' : 'N',
								dnum2lstr(sat->aos));
				}
			}
			else {
				if (sat->el > 0.0) {
					buff = g_strdup_printf ("%s - %.2f%c%c %.2f%c%c - AZ:%.2f%c EL:%.2f%c",
								sat->name,
								fabs (sat->lon), SATMAP_DEGREE, (sat->lon < 0.0000) ? 'E' : 'W',
								fabs (sat->lat), SATMAP_DEGREE, (sat->lat < 0.0000) ? 'S' : 'N',
								sat->az, SATMAP_DEGREE, sat->el, SATMAP_DEGREE);
				}
				else {
					buff = g_strdup_printf ("%s - %.2f%c%c %.2f%c%c - NO AOS",
								sat->name,
								fabs (sat->lon), SATMAP_DEGREE, (sat->lon < 0.0000) ? 'E' : 'W',
								fabs (sat->lat), SATMAP_DEGREE, (sat->lat < 0.0000) ? 'S' : 'N');
				}
			}
			gnome_canvas_item_set (footer, "text", buff, NULL);
			g_free (buff);
			
			/* the info footer has to contain the index of the satellite to which
			   the mouse currently points at.
			*/
			gtk_object_set_data (GTK_OBJECT (footer), "index", GINT_TO_POINTER (smsat->index));
			gtk_object_set_data (GTK_OBJECT (footer), "select", GINT_TO_POINTER (1));

			break;

		case 2: /* MIDDLE mouse button */
			break;

		case 3: /* RIGHT button */
			/* deselect satellite if any */

			/* get satellite item (event source) */
			smsat = SATMAP_SAT (data);

			/* get footer item on map */
			footer  = GNOME_CANVAS_ITEM (gtk_object_get_data (GTK_OBJECT (item), "sati"));
			
			/* get selection */
			selection = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (footer), "index"));
			selected  = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (footer), "select"));

			/* check we have a permanent selection, if yes, clear it */
			if (selected && (selection == smsat->index)) { 

				/* clear footer */
				gnome_canvas_item_set (footer, "text", "", NULL);

				/* clear selection */
				gtk_object_set_data (GTK_OBJECT (footer), "index", GINT_TO_POINTER (-1));
				gtk_object_set_data (GTK_OBJECT (footer), "select", GINT_TO_POINTER (0));
			}

			break;

//		case 4: /* wheel up */
//		case 5: /* wheel down */

		default:
			break;
		}

		return TRUE;  /* should never happen though */
		break;

	case GDK_ENTER_NOTIFY: /* display info */

		/* get footer item on map */
		footer  = GNOME_CANVAS_ITEM (gtk_object_get_data (GTK_OBJECT (item), "sati"));

		/* get stored index */
		selection = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (footer), "index"));
		selected  = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (footer), "select"));

		/* if selection is -1 no sat is selected */
		if ((selection == -1) && (selected == 0)) {

			/* get satellite item (event source) */
			smsat = SATMAP_SAT (data);

			/* get pointer to satellite structure */
			sat = satdata_get_sat (smsat->index);

			/* Check whether the satellite ever comes within range. If yes, show
			   the position and the AOS time, otherwise just the position.
			*/
			if (sat->hasaos) {

				/* If satellite is within range we show LOS and pointing coordinates.
				   If satellite is out of range we show AOS.
				*/
				if (sat->el > 0.0) {
					buff = g_strdup_printf ("%s - %.2f%c%c %.2f%c%c - AZ:%.2f%c EL:%.2f%c - LOS: %s",
								sat->name,
								fabs (sat->lon), SATMAP_DEGREE, (sat->lon < 0.0000) ? 'E' : 'W',
								fabs (sat->lat), SATMAP_DEGREE, (sat->lat < 0.0000) ? 'S' : 'N',
								sat->az, SATMAP_DEGREE, sat->el, SATMAP_DEGREE,
								dnum2lstr(sat->los));
				}
				else {
					buff = g_strdup_printf ("%s - %.2f%c%c %.2f%c%c - AOS: %s",
								sat->name,
								fabs (sat->lon), SATMAP_DEGREE, (sat->lon < 0.0000) ? 'E' : 'W',
								fabs (sat->lat), SATMAP_DEGREE, (sat->lat < 0.0000) ? 'S' : 'N',
								dnum2lstr(sat->aos));
				}
			}
			else {
				if (sat->el > 0.0) {
					buff = g_strdup_printf ("%s - %.2f%c%c %.2f%c%c - AZ:%.2f%c EL:%.2f%c",
								sat->name,
								fabs (sat->lon), SATMAP_DEGREE, (sat->lon < 0.0000) ? 'E' : 'W',
								fabs (sat->lat), SATMAP_DEGREE, (sat->lat < 0.0000) ? 'S' : 'N',
								sat->az, SATMAP_DEGREE, sat->el, SATMAP_DEGREE);
				}
				else {
					buff = g_strdup_printf ("%s - %.2f%c%c %.2f%c%c - NO AOS",
								sat->name,
								fabs (sat->lon), SATMAP_DEGREE, (sat->lon < 0.0000) ? 'E' : 'W',
								fabs (sat->lat), SATMAP_DEGREE, (sat->lat < 0.0000) ? 'S' : 'N');
				}
			}
			gnome_canvas_item_set (footer, "text", buff, NULL);
			g_free (buff);

			/* the info footer has to contain the index of the satellite to which
			   the mouse currently points at.
			*/
			gtk_object_set_data (GTK_OBJECT (footer), "index", GINT_TO_POINTER (smsat->index));
		}

		return TRUE;
		break;
	case GDK_LEAVE_NOTIFY: /* clear label */
		/* get footer item on map */
		footer  = GNOME_CANVAS_ITEM (gtk_object_get_data (GTK_OBJECT (item), "sati"));

		/* get stored index */
		selection = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (footer), "index"));
		selected  = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (footer), "select"));

		/* if selection is -1 no sat is selected */
		if ((selection != -1) && (selected == 0)) {

			gnome_canvas_item_set (footer, "text", "", NULL);

			/* the info footer has to contain the index of the satellite to which
			   the mouse currently points at. -1 means no satellite.
			*/
			gtk_object_set_data (GTK_OBJECT (footer), "index", GINT_TO_POINTER (-1));
		}

		return TRUE;
		break;
	default:
		return TRUE;
	}

	return FALSE;  /* Try parent, although it should never happen (default...) */
}
