/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
  satloc-tree.c:  Functions to read a specified locations file and
                  insert the locations into a GtkCTree.

  Copyright (C)  2001-2005  Alexandru Csete.

  Authors: Alexandru Csete <csete@users.sourceforge.net>

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/
#include <gnome.h>
#include "satlog.h"
#include "satloc-tree.h"

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif


static gchar *satloc_tree_titles[SATLOC_TREE_COL_NUM] = {
	N_("Location"),
	N_("Latitude"),
	N_("Longitude"),
	N_("Altitude"),
	N_("Station")
};


/* static gchar *satloc_conv_lat (const gchar *lat); */
/* static gchar *satloc_conv_lon (const gchar *lon); */


GtkWidget *
satloc_tree_create (void)
{
	/* create a new GtkCTree containing the
	   locations.
	*/
	GtkWidget *tree;
	GtkCTreeNode *reg = NULL, *state = NULL;
	FILE *locfile;
	gchar *msg,**vbuff,**vvbf,*buff,line[100];
	gint i=0;
	gchar *curreg=NULL,*cursta=NULL;  /* these are used to keep track of the current reg. and country */
	gchar *locfnam;

	/* Create a new tree */
	tree = gtk_ctree_new_with_titles (SATLOC_TREE_COL_NUM,
					  SATLOC_TREE_COL_LOC,
					  satloc_tree_titles);
	g_return_val_if_fail (tree, NULL);

	/* configure the tree */
	for (i=0; i<SATLOC_TREE_COL_NUM; i++)
		gtk_clist_set_column_auto_resize (GTK_CLIST (tree), i, TRUE);
	gtk_clist_column_titles_passive (GTK_CLIST (tree));
	gtk_clist_set_sort_type (GTK_CLIST (tree), GTK_SORT_ASCENDING);
	gtk_clist_set_sort_column (GTK_CLIST (tree), 0);
	gtk_clist_set_auto_sort (GTK_CLIST (tree), TRUE);

	/* some dummy stuff */
	curreg = g_strdup ("DUMMY REGION");
	cursta = g_strdup ("DUMMY STATE");

	/* open data file */ 
	locfnam = g_strconcat (PACKAGE_DATA_DIR, G_DIR_SEPARATOR_S, "locations.dat", NULL);
	locfile = fopen (locfnam, "r");
	g_free (locfnam);
	if (locfile) {
		msg = g_strconcat (__FUNCTION__, ": ", _("opened data file."), NULL);
		satlog_log (SAT_LOG_INFO, msg);
		g_free (msg);

		/* avoid reading the last line twice ... */
		fgets (line, 100, locfile);
		while (!feof (locfile)) {
			g_strdelimit (line, "\n", '\0');
			vbuff = g_strsplit (line, ";", 0);

			/* 1. REGION */

			/* Create a new Region node if necessary */
			if (g_strcasecmp (curreg, vbuff[0])) {

				g_free (curreg);
				curreg = g_strdup (vbuff [0]);

				/* add region / create tree node */
				buff = g_strconcat (vbuff[0], "<!SEP> <!SEP> <!SEP> <!SEP> ", NULL);
				vvbf = g_strsplit (buff, "<!SEP>", 0);
				reg = gtk_ctree_insert_node (GTK_CTREE (tree),
							     NULL, NULL,       /* parent/sibling */
							     vvbf,             /* text */
							     0,                /* spacing */
							     NULL, NULL,       /* pixmaps */
							     NULL, NULL,       /* pixmaps */
							     FALSE, FALSE);    /* isleaf/expanded */

				/* This row should not be selectable */
				if (reg) {
					gtk_ctree_node_set_selectable (GTK_CTREE (tree),
								       GTK_CTREE_NODE (reg),
								       FALSE);
				}
				g_strfreev (vvbf);
				g_free (buff);

				/* log message */
				msg = g_strconcat (__FUNCTION__, ": ", _("Creating region node"),
						   ": ", vbuff[0], NULL);
				satlog_log (SAT_LOG_DEBUG, msg);
				g_free (msg);
			}

			/* At this point "reg" is a pointer to a GtkCTreeNode and shall be
			   used as parent for the country node.

			   2. Country or State
			*/
			
			if (g_strcasecmp (cursta, vbuff[1])) {
				g_free (cursta);
				cursta = g_strdup (vbuff [1]);

				/* add region / create tree node */
				buff = g_strconcat (vbuff[1], "<!SEP> <!SEP> <!SEP> <!SEP> ", NULL);
				vvbf = g_strsplit (buff, "<!SEP>", 0);
				state = gtk_ctree_insert_node (GTK_CTREE (tree),
							       reg, NULL,        /* parent/sibling */
							       vvbf,             /* text */
							       0,                /* spacing */
							       NULL, NULL,       /* pixmaps */
							       NULL, NULL,       /* pixmaps */
							       FALSE, FALSE);    /* isleaf/expanded */
				/* This row should not be selectable */
				if (state) {
					gtk_ctree_node_set_selectable (GTK_CTREE (tree),
								       GTK_CTREE_NODE (state),
								       FALSE);
				}
				g_strfreev (vvbf);
				g_free (buff);

			}

			/* At this point "state" is a pointer to a GtkCTreeNode and shall be
			   used as parent for the location node.

			   3. City or Locations
			*/

			/* add region / create tree node */
			buff = g_strconcat (vbuff[2], "<!SEP>",
					    vbuff[4], "<!SEP>",
					    vbuff[5], "<!SEP>",
					    vbuff[6], "<!SEP>",
					    vbuff[3], NULL);
			vvbf = g_strsplit (buff, "<!SEP>", 0);
			gtk_ctree_insert_node (GTK_CTREE (tree),
					       state, NULL,        /* parent/sibling */
					       vvbf,             /* text */
					       0,                /* spacing */
					       NULL, NULL,       /* pixmaps */
					       NULL, NULL,       /* pixmaps */
					       TRUE, FALSE);    /* isleaf/expanded */
			g_strfreev (vvbf);
			g_free (buff);
			g_strfreev (vbuff);
			fgets (line, 100, locfile);
			i++;
		}
		fclose (locfile);
		msg = g_strdup_printf (_("%s: Read %d locations."), __FUNCTION__, i);
		satlog_log (SAT_LOG_INFO, msg);
		g_free (msg);
	}
	else {
		msg = g_strconcat (__FUNCTION__, ": ", _("failed to open data file."), NULL);
		satlog_log (SAT_LOG_CRITICAL, msg);
		g_free (msg);
	}

	g_free (cursta);
	g_free (curreg);

	return tree;
}



/* static gchar *satloc_conv_lat (const gchar *lat) */
/* { */
/* 	gchar *tmp,**buff; */
/* 	gfloat deg,min,sec; */
/* 	gint sign; */
	
/* 	tmp = g_strdup (lat); */

/* 	/\* get sign *\/ */
/* 	switch (lat[strlen(lat) - 1]) { */
/* 	case 'N': */
/* 		sign = 1; */
/* 		break; */
/* 	case 'S': */
/* 		sign = -1; */
/* 		break; */
/* 	default: */
/* 		sign = 1; */
/* 		g_print ("**** UNKNOWN SIGN!!!! (%s) ****\n", lat); */
/* 	} */

/* 	/\* Remove N/S *\/ */
/* 	g_strdelimit (tmp, "NS", '\0'); */
/* 	buff = g_strsplit (tmp, ":", 0); */

/* 	deg = g_strtod (buff[0], NULL); */
/* 	min = g_strtod (buff[1], NULL); */
/* 	if (buff[2]) */
/* 		sec = g_strtod (buff[2], NULL); */
/* 	else */
/* 		sec = 0.00; */

/* 	g_free (tmp); */
/* 	g_strfreev (buff); */

/* 	return (g_strdup_printf ("%.4f", sign*(deg+min/60))); */

/* } */


/* static gchar *satloc_conv_lon (const gchar *lon) */
/* { */
/* 	gchar *tmp,**buff; */
/* 	gfloat deg,min,sec; */
/* 	gint sign; */
	
/* 	tmp = g_strdup (lon); */

/* 	/\* get sign *\/ */
/* 	switch (lon[strlen(lon) - 1]) { */
/* 	case 'W': */
/* 		sign = 1; */
/* 		break; */
/* 	case 'E': */
/* 		sign = -1; */
/* 		break; */
/* 	default: */
/* 		sign = 1; */
/* 		g_print ("**** UNKNOWN SIGN!!!! (%s) ****\n", lon); */
/* 	} */

/* 	/\* Remove N/S *\/ */
/* 	g_strdelimit (tmp, "EW", '\0'); */
/* 	buff = g_strsplit (tmp, ":", 0); */

/* 	deg = g_strtod (buff[0], NULL); */
/* 	min = g_strtod (buff[1], NULL); */
/* 	if (buff[2]) */
/* 		sec = g_strtod (buff[2], NULL); */
/* 	else */
/* 		sec = 0.00; */

/* 	g_free (tmp); */
/* 	g_strfreev (buff); */

/* 	return (g_strdup_printf ("%.4f", sign*(deg+min/60+sec/3600))); */

/* } */
