/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*

  satconfig.c: Configuration wizard for Gnome Predict. Available as of
               version 0.4 and obsoletes the newuser.c and newuser.h
	       files.
	       The idea is, that every separate module, which needs some
	       initial configuration, is located in separate files and
	       these files contain functions to create the required
	       widgets, their callbacks and finally functions to store
	       the configuration values.


  Copyright (C)  2002-2005  Alexandru Csete.

  Authors: Alexandru Csete <csete@users.sourceforge.net>

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <libgnomevfs/gnome-vfs.h>
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#include "fileio.h"
#include "qth.h"
#include "satconfig.h"
#include "satconfig-locations.h"
#include "satconfig-rig.h"
#include "satconfig-rot.h"
#include "satlog.h"



GdkColor mainbg,logobg;
gboolean CONFIGOK = FALSE;
gboolean CONFIGERROR = FALSE;
GtkWidget *druidwin,*druid;

extern GConfClient *client;  /* in main.c */


/* Provate function prototypes */
static GtkWidget *satconfig_page_start       (void);
static GtkWidget *satconfig_page_finish      (void);
static void       satconfig_cancel_cb        (GnomeDruid *, gpointer);
static void       satconfig_finish_cb        (GnomeDruid *, gpointer);


/* list of distributed TLE files */
static gchar *tlelist[] = {
	"amateur.tle",
	"iridium.tle",
	"military.tle",
	"radar.tle",
	"science.tle",
	"weather.tle",
	NULL
};


void satconfig_open ()
{
	/* This function should be called to create and open a new
	   configuration druid. The druid will recycle existing
	   configuration values, so it is safe to call it as a
	   reconfiguration thingy (although the main purpose is
	   to configure Gnome Predict when it is run or the first
	   time).
	*/

	GtkWidget *endpage;


	/* initialize colors */
	mainbg.red   = (gushort)0xFFFF;
	mainbg.green = (gushort)0xFFFF;
	mainbg.blue  = (gushort)0xFFFF;
	logobg.red   = (gushort)6425;
	logobg.green = (gushort)6425;
	logobg.blue  = (gushort)28784;

	/* Druid window */
	druidwin = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title (GTK_WINDOW (druidwin), _("Gnome Predict Configuration Druid"));

	/* create the druid and the pages */
	druid = gnome_druid_new ();
	/* set cancel callback with window as user data */
	gtk_signal_connect (GTK_OBJECT (druid), "cancel",
			    GTK_SIGNAL_FUNC (satconfig_cancel_cb),
			    NULL);

	/* start page */
	gnome_druid_append_page (GNOME_DRUID (druid),
				 GNOME_DRUID_PAGE (satconfig_page_start ()));

	/* config pages */
	gnome_druid_append_page (GNOME_DRUID (druid),
				 GNOME_DRUID_PAGE (satconfig_locations_open ()));

#if defined (HAVE_HAMLIB)
	gnome_druid_append_page (GNOME_DRUID (druid),
				 GNOME_DRUID_PAGE (satconfig_rig_create ()));
	gnome_druid_append_page (GNOME_DRUID (druid),
				 GNOME_DRUID_PAGE (satconfig_rot_create ()));
#endif

	/* end page */
	endpage = satconfig_page_finish ();
	gtk_signal_connect (GTK_OBJECT (endpage), "finish",
			    GTK_SIGNAL_FUNC (satconfig_finish_cb),
			    NULL);
	gnome_druid_append_page (GNOME_DRUID (druid),
				 GNOME_DRUID_PAGE (endpage));

	gtk_container_add (GTK_CONTAINER (druidwin), druid);
	gtk_widget_show_all (druidwin);

	gtk_main ();

}




const gchar *welcome = {
	N_("On the following pages you will be guided through the initial\n"\
	   "setup of Gnome Predict. Please note, that these steps are\n"
	   "necessary to run the program.\n\n"\
	   "Don't be afraid of doing anything wrong. You can always make\n"
	   "changes later, while running the program.")
};


static GtkWidget *
satconfig_page_start ()
{
	/* This function creates the start page in the
	   configuration druid.
	*/
	GtkWidget *page;
	GdkImlibImage *logo;
	gchar *fname;

	/* create logo */
	fname = g_strconcat (PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S,
			     "icons", G_DIR_SEPARATOR_S, "gpredict-icon.png",
			     NULL);
	logo = gdk_imlib_load_image (fname);
	g_free (fname);
	/* create the page */
	page = gnome_druid_page_start_new_with_vals (_("Welcome to Gnome Predict !"),
						     welcome, logo, NULL);
	gnome_druid_page_start_set_bg_color (GNOME_DRUID_PAGE_START (page), &logobg);
	gnome_druid_page_start_set_logo_bg_color (GNOME_DRUID_PAGE_START (page), &logobg);
	gdk_imlib_destroy_image (logo);
	
	return page;
}



const gchar *end = {
	N_("We have now gathered all the required information.\n"\
	   "Press \"Finish\" to save the data and start Gnome\n"\
	   "Predict.\n\n"\
	   "Happy Tracking!")
};


static GtkWidget *
satconfig_page_finish ()
{
	/* This function creates the last page in the
	   configuration druid.
	*/
	GtkWidget *page;
	GdkImlibImage *logo;
	gchar *fname;

	/* create logo */
	fname = g_strconcat (PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S,
			     "icons", G_DIR_SEPARATOR_S, "gpredict-icon.png",
			     NULL);
	logo = gdk_imlib_load_image (fname);
	g_free (fname);

	/* create the page */
	page = gnome_druid_page_finish_new_with_vals (_("Configuration Complete !"),
						      end, logo, NULL);
	gnome_druid_page_finish_set_bg_color (GNOME_DRUID_PAGE_FINISH (page), &logobg);
	gnome_druid_page_finish_set_logo_bg_color (GNOME_DRUID_PAGE_FINISH (page), &logobg);
	gdk_imlib_destroy_image (logo);


	return page;

}


static void
satconfig_cancel_cb (GnomeDruid *druid, gpointer data)
{
	/* This function is called if the user clicks on the
	   CANCEL button. This will quit the program.
	*/
	CONFIGOK = FALSE;

	/* cancel all modules */
	satconfig_locations_cancel ();

#if defined (HAVE_HAMLIB)
	satconfig_rig_cancel ();
	satconfig_rot_cancel ();
#endif

	gtk_widget_destroy (GTK_WIDGET (druidwin));
	gtk_main_quit ();
}


/** \brief Finish new user setup.
 *  \param druid The running configuration druid.
 *  \param data User data (NULL).
 *
 * This function is called when the user clicks on the "Finish" button
 * on the final page of the new user druid. First it checks whether the
 * directory containing the TLE files exists and creates it if necessary.
 * When done it checks for the existence of the stock TLE files and in case
 * any of them is missing it copies it to the TLE directory. Lastly the
 * "finish" function of the various modules are called to save the configuration
 * data.
 *
 * \bug Some error messages are sent to g_error() while others to satlog_log().
 */
static void
satconfig_finish_cb (GnomeDruid *druid, gpointer data)
{
	GnomeVFSDirectoryHandle *dir;
	GnomeVFSResult result;
	gchar *path,*outf,*err;
	gint i;

	CONFIGOK = TRUE;


	/* THIS COPY TLE STUFF REALLY NEEDS TO BE REWRITTEN... */


	/* Copy TLE files into $(HOME)/.gpredict/tle/ */

	/* check wether we need to create .gpredict subdir */
	path = g_strconcat (g_get_home_dir (), G_DIR_SEPARATOR_S, ".gpredict", NULL);	
	result = gnome_vfs_directory_open (&dir, path,
					   GNOME_VFS_FILE_INFO_DEFAULT,
					   NULL);
	if (result == GNOME_VFS_ERROR_NOT_FOUND) {
		result = gnome_vfs_make_directory (path, 0755);
		if (result != GNOME_VFS_OK) {
			err = g_strconcat (__FUNCTION__, ": Failed to create ", path, NULL);
			g_error (err);
			g_free (err);
		}
	}
	else
		gnome_vfs_directory_close (dir);

	g_free (path);

	/* check wether we need to create tle subdir */
	path = g_strconcat (g_get_home_dir (), G_DIR_SEPARATOR_S, ".gpredict",
			    G_DIR_SEPARATOR_S, "tle", NULL);
	result = gnome_vfs_directory_open (&dir, path,
					   GNOME_VFS_FILE_INFO_DEFAULT,
					   NULL);
	if (result == GNOME_VFS_ERROR_NOT_FOUND) {
		result = gnome_vfs_make_directory (path, 0755);
		if (result != GNOME_VFS_OK) {
			err = g_strconcat (__FUNCTION__, ": Failed to create ", path, NULL);
			g_error (err);
			g_free (err);
		}
	}
	else
		gnome_vfs_directory_close (dir);

	g_free (path);
	
	/* Directories created, copy files but only thos which do not already exist */
	for (i = 0; tlelist[i] != NULL; i++) {
		path = g_strconcat (PACKAGE_DATA_DIR, G_DIR_SEPARATOR_S,
				    tlelist[i], NULL);
		outf = g_strconcat (g_get_home_dir (), G_DIR_SEPARATOR_S,
				    ".gpredict", G_DIR_SEPARATOR_S, "tle",
				    G_DIR_SEPARATOR_S, tlelist[i], NULL);
		
		/* only copy if file it does not already exist */
		if (g_file_test (outf, G_FILE_TEST_ISFILE)) {
			err = g_strdup_printf (_("Skipping %s (file exists)"), outf);
		}
		else {
			err = g_strconcat (path, " ==> ", outf, "  ...",
					   fileio_file_copy (path, outf) ? _("ERROR") : _("OK"),
					   NULL);
		}

		satlog_log (SAT_LOG_INFO, err);
		g_free (path);
		g_free (outf);
		g_free (err);
	}


	/* Finish all modules */
	satconfig_locations_finish ();

#if defined (HAVE_HAMLIB)
	satconfig_rig_finish ();
	satconfig_rot_finish ();
#endif

	gconf_client_suggest_sync (client, NULL);

	gtk_widget_destroy (GTK_WIDGET (druidwin));
	gtk_main_quit ();
}


