# JobInterval scheduler
# (c) Wijnand 'tehmaze' Modderman - http://tehmaze.com
# BSD License

from gozerbot.generic import calledfrom, lockdec, rlog, strtotime
import gozerbot.thr as thr
import datetime
import sys
import time
import thread
import types

plock    = thread.allocate_lock()
locked   = lockdec(plock)
pidcount = 0

class JobError(Exception):

    """ job error exception """

    pass

class Job(object):

    """ job to be scheduled """

    group = ''
    pid   = -1

    def __init__(self):
        global pidcount
        pidcount += 1
        self.pid = pidcount

    def id(self):
        """ return job id """
        return self.pid

    def member(self, group):
        """ check for group membership """
        return self.group == group

class JobAt(Job):

    """ job to run at a specific time/interval/repeat """

    def __init__(self, start, interval, repeat, func, *args, **kw):
        Job.__init__(self)
        self.func = func
        self.args = args
        self.kw = kw
        self.repeat = repeat
        self.counts = 0
        if type(start) in [types.IntType, types.FloatType]:
            self.next = datetime.datetime(*time.localtime(float(start))[:7])
        elif type(start) in [types.StringType, types.UnicodeType]:
            self.next = \
datetime.datetime(*time.localtime(float(strtotime(start)))[:7])
        else:
            raise JobError, "invalid start time '%s' (%s)'" % (start, \
str(type(start)))
        if type(interval) in [types.IntType]:
            self.delta = datetime.timedelta(days=interval)
        else:
            self.delta = interval
        # boundary check
        while self.next < datetime.datetime.now():
            self.next += self.delta

    def __repr__(self):
        return '<JobAt instance next=%s, interval=%s, repeat=%d, function=%s>' % (str(self.next),
            str(self.delta), self.repeat, str(self.func))

    def check(self):
        """ run check to see if job needs to be scheduled """
        if self.next <= datetime.datetime.now():
            rlog(5, 'periodical', 'running %s' % str(self.func))
            self.func(*self.args, **self.kw)
            self.next += self.delta
            self.counts += 1
            if self.repeat > 0 and self.counts >= self.repeat:
                return False # remove this job
        return True

class JobInterval(Job):

    """ job to be scheduled at certain interval """

    def __init__(self, interval, repeat, func, *args, **kw):
        Job.__init__(self)
        self.func = func
        self.args = args
        self.kw = kw
        self.repeat = repeat
        self.counts = 0
        self.interval = interval
        self.next = time.time() + self.interval
        rlog(5, 'periodical', 'scheduled next run of %s in %d seconds' % \
(str(self.func), self.interval))

    def __repr__(self):
        return '<JobInterval instance next=%s, interval=%s, repeat=%d, \
function=%s>' % (str(self.next), str(self.interval), self.repeat, \
str(self.func))

    def check(self):
        """ run check to see if job needs to be scheduled """
        if self.next <= time.time():
            rlog(5, 'periodical', 'running %s' % (str(self.func)))
            self.func(*self.args, **self.kw)
            self.next = time.time() + self.interval
            self.counts += 1
            if self.repeat > 0 and self.counts >= self.repeat:
                return False # remove this job
        return True

class Periodical(object):

    """ periodical scheduler """

    SLEEPTIME = 5 # smallest interval possible

    def __init__(self):
        self.jobs = []
        self.run = True
        thr.start_new_thread(self.checkloop, ())

    def checkloop(self):
        """ main loop """
        while self.run:
            if self.jobs:
                rlog(-1, 'periodical', 'checking %d jobs' % len(self.jobs))
                for job in self.jobs:
                    thr.start_new_thread(self.runjob, (job,))
                    time.sleep(0.05)
            time.sleep(self.SLEEPTIME)

    def runjob(self, job):
        """ kill job is not to be runned """
        if not job.check():
            self.killjob(job.id())

    def kill(self):
        ''' kill all jobs invoked by another module '''
        group = calledfrom(sys._getframe())
        self.killgroup(group)

    def killgroup(self, group):
        ''' kill all jobs with the same group '''
        @locked
        def shoot():
            """ knock down all jobs belonging to group """
            deljobs = [job for job in self.jobs if job.member(group)]
            for job in deljobs:
                self.jobs.remove(job)
            rlog(10, 'periodical', 'killed %d jobs for %s' % (len(deljobs), \
group))
            del deljobs
        shoot() # *pow* you're dead ;)

    def killjob(self, jobId):
        ''' kill one job by its id '''
        @locked
        def shoot():
            deljobs = [x for x in self.jobs if x.id() == jobId]
            numjobs = len(deljobs)
            for job in deljobs:
                self.jobs.remove(job)
            del deljobs
            return numjobs
        return shoot() # *pow* you're dead ;)

periodical = Periodical()

def interval(sleeptime, repeat=0):
    """ interval decorator """
    def decorator(function):
        decorator.func_dict = function.func_dict
        def wrapper(*args, **kw):
            job = JobInterval(sleeptime, repeat, function, *args, **kw)
            job.group = calledfrom(sys._getframe())
            periodical.jobs.append(job)
            rlog(5, 'periodical', 'new interval job %d with sleeptime %d' % \
(job.id(), sleeptime))
        return wrapper
    return decorator

def at(start, interval=1, repeat=1):
    """ at decorator """
    def decorator(function):
        decorator.func_dict = function.func_dict
        def wrapper(*args, **kw):
            job = JobAt(start, interval, repeat, function, *args, **kw)
            job.group = calledfrom(sys._getframe())
            periodical.jobs.append(job)
        wrapper.func_dict = function.func_dict
        return wrapper
    return decorator

def minutely(function):
    """ minute decorator """
    def wrapper(*args, **kw):
        job = JobInterval(60, 0, function, *args, **kw)
        job.group = calledfrom(sys._getframe())
        periodical.jobs.append(job)
        rlog(5, 'periodical', 'new interval job %d running minutely' % \
job.id())
    return wrapper

def hourly(function):
    """ hour decorator """
    rlog(0, 'periodical', '@hourly(%s)' % str(function))
    hourly.func_dict = function.func_dict
    def wrapper(*args, **kw):
        job = JobInterval(3600, 0, function, *args, **kw)
        job.group = calledfrom(sys._getframe())
        rlog(5, 'periodical', 'new interval job %d running hourly' % job.id())
        periodical.jobs.append(job)
    return wrapper

def daily(function):
    """ day decorator """
    rlog(0, 'periodical', '@daily(%s)' % str(function))
    daily.func_dict = function.func_dict
    def wrapper(*args, **kw):
        job = JobInterval(86400, 0, function, *args, **kw)
        job.group = calledfrom(sys._getframe())
        periodical.jobs.append(job)
        rlog(5, 'periodical', 'new interval job %d running daily' % job.id())
    return wrapper
