;;; utils.scm: common utility procedures
;;; author: M.P.Anand Babu <ab@gnu.org.in>

;;; This program is free software; you can redistribute it and/or
;;; modify it under the terms of the GNU General Public License as
;;; published by the Free Software Foundation; either version 2, or (at
;;; your option) any later version.
;;; 
;;; This program is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; General Public License for more details.
;;; 
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA
;;; 
;;; utils.scm should be automatically loaded thru init.scm

(define (symbolrnumber->symbol item)
  "converts a number into a symbol"
  (if (number? item)
      (string->symbol (number->string item))
      item))

(define (list->asv li any) 
  "convert list to any separated vector"
  (if (=(length li) 1)
      (car li)
      (string-append (car li) any (list->asv (cdr li) any))))

(define (list->csv li) 
  "convert list to comma separated vector"
  (list->asv li ", "))

(define (list->symlist li)
  "converts a list of numbers/symbols to list of symbols"
  (if (= (length li) 1)
      (list (symbolrnumber->symbol (car li)))
      (append (list (symbolrnumber->symbol (car li))) (list->symlist (cdr li)))))

(define (send-message-to-group group message)
  "send message to group"
  (for-each (lambda (buddy)
	      (begin
		(gy-send-message-no-hook buddy message)))
	    ;; since yahoo dont allow message flooding, solve this
	    ;; you need to sleep for every n number of user
	    ;; (usleep 99999)))
	    group))

(define (local-date-time)
  (string-append
                 (strftime "%a" (localtime (current-time)))
		 " "
                 (strftime "%b" (localtime (current-time)))
		 " "
                 (strftime "%e" (localtime (current-time)))
		 " "
                 (strftime "%T" (localtime (current-time)))
		 " "
                 (strftime "%Z" (localtime (current-time)))
		 " "
                 (strftime "%Y" (localtime (current-time)))))


(define (skip-comment port)
  (let ((char (peek-char port)))
    (if (or (eof-object? char)
            (char=? #\newline char))
        char
        (begin
          (read-char port)
          (skip-comment port)))))

(define (skip-whitespace port)
  (let ((char (peek-char port)))
    (cond ((or (eof-object? char)
	       (char=? #\newline char))
	   char)
	  ((char-whitespace? char)
	   (read-char port)
	   (skip-whitespace port))
;;  	  ((char=? #\# char)
;;  	   (read-char port)
;;  	   (skip-comment port))
	  (else char))))


(define (read-token port delim)
  (letrec
      ((loop
	(lambda (chars)
	  (let ((char (peek-char port)))
	    (cond ((eof-object? char)
		   (do-eof char chars))
		  ((char=? #\newline char)
		   (do-eot chars))
		  ((char-whitespace? char)
		   (do-eot chars))
;;  		   (let ((terminator (skip-comment port)))
;;  		     (if (eof-object? char)
;;  			 (do-eof char chars)
;;  			 (do-eot chars))))
		  (else
		   (read-char port)
		   (loop (cons char chars)))))))
       (do-eof
	(lambda (eof chars)
	  (if (null? chars)
	      eof
	      (do-eot chars))))
       (do-eot
	(lambda (chars)
	  (if (null? chars)
	      #f
	      (list->string (reverse! chars))))))
    (skip-whitespace port)
    (loop '())))

(define (sentence->words sentence)
  "convert sentence to list of words"
  (with-input-from-string sentence
    (lambda ()
      (letrec
	  ((next-token
	    (lambda ()
	      (read-token (current-input-port) #\#)))
	   (append-word
	    (lambda (word-list)
	      (let
		  ((word (next-token)))
		(if (eof-object? word)
		    word-list
		    (begin
		      (append-word 
		       (append word-list (list word)))))))))
	(append-word '())))))


(define (blank-line? line)
  "return true if line is blank"
  (null? (sentence->words line)))
