
// created 06.2003 by Stefan Kleine Stegemann
// 
// licensed under GPL

#include <PDFKit/PDFImageRep.h>
#include <PDFKit/PDFFontManager.h>

#include <Foundation/NSString.h>
#include <Foundation/NSException.h>
#include <AppKit/NSBezierPath.h>
#include <AppKit/NSColor.h>
#include <AppKit/NSGraphicsContext.h>
#include <AppKit/NSFont.h>
#include <AppKit/NSBitmapImageRep.h>
#include <AppKit/NSGraphics.h>
#include <AppKit/DPSOperators.h>

#include <stdlib.h>

#include "XPDFBridge.h"


const double PDFBaseResolution = 72.0;

/*
 * Non-Public methods.
 */
@interface PDFImageRep(Private)
- (void) _updatePage;
@end


/*
 */
@implementation PDFImageRep

+ (void) initialize
{
   [super initialize];

   // we have to make sure that font mappings are initialized
   [PDFFontManager sharedManager];
}


- (id) initWithDocument: (PDFDocument*)aDocument
{
   if ((self = [super init]))
   {
      pdfDoc  = RETAIN(aDocument);
      outputDevice = PDFRender_CreateOutputDevice([pdfDoc xpdfobject]);
      [self setResolution: PDFBaseResolution];
      [self setPageNum: 1];
      pageNeedsUpdate = YES;
      page = nil;
   }

   return self;
}


- (void) dealloc
{
   NSLog(@"dealloc PDFImageRep");
   
   PDFRender_DestroyOutputDevice(outputDevice);
   RELEASE(pdfDoc);
   RELEASE(page);

   [super dealloc];
}


- (void) setPageNum: (int)aPage
{
   pageNum         = aPage;
   pageNeedsUpdate = YES;
}


- (int) pageNum
{
   return pageNum;
}


- (NSSize) size
{
   NSSize size;

   size = [pdfDoc pageSize: [self pageNum] considerRotation: YES];

   // consider resolution
   size.height = ([self resolution] / PDFBaseResolution) * size.height;
   size.width  = ([self resolution] / PDFBaseResolution) * size.width;

   return size;
}


- (void) setResolution: (double)aResolution
{
   resolution      = aResolution;
   pageNeedsUpdate = YES;
}


- (double) resolution
{
   return resolution;
}


+ (BOOL) canInitWithData: (NSData*)data
{
   return NO;
}


+ (BOOL) canInitWithPasteboard: (NSPasteboard*)pasteboard
{
   return NO;
}


+ (NSArray*) imageFileTypes
{
   return [self imageUnfilteredFileTypes];
}


+ (NSArray*) imageUnfilteredFileTypes
{
   return [NSArray array];
}


+ (NSArray*) imagePasteboardTypes
{
   return [self imageUnfilteredPasteboardTypes];
}


+ (NSArray*) imageUnfilteredPasteboardTypes
{
   return [NSArray array];
}


- (BOOL) drawInRect: (NSRect)rect
{
   NSGraphicsContext* gc = GSCurrentContext();
   
   // fill background with white color
   [[NSColor whiteColor] set];
   DPSrectfill(gc, rect.origin.x, rect.origin.y, rect.size.width, rect.size.height);

   if (pageNeedsUpdate)
   {
      [self _updatePage];
   }

   [page drawInRect: rect];

   // display page content
   //[outputDevice renderPage: [self pageNum]
   //              ofDocument: pdfDoc
   //                 withDPI: [self resolution]];

   return YES;
}

@end



@implementation PDFImageRep(Private)

/* Render the current page with the current resolution into
 * a bitmap.  */
- (void) _updatePage
{
   XPDFObject     bitmap;
   int            width, height;
   unsigned char* repData;

   AUTORELEASE(page);

   NSLog(@"Render page %d with resolution %f", [self pageNum], [self resolution]);
   bitmap = PDFRender_RenderPage([pdfDoc xpdfobject],
                                 outputDevice,
                                 [self pageNum],
                                 [self resolution],
                                 0);
   NSAssert(bitmap, @"could not render page");

   PDFRender_GetBitmapSize(bitmap, &width, &height);

   page = [[NSBitmapImageRep alloc] initWithBitmapDataPlanes: NULL
                                                  pixelsWide: width
                                                  pixelsHigh: height
                                               bitsPerSample: 8
                                             samplesPerPixel: 3
                                                    hasAlpha: NO
                                                    isPlanar: NO
                                              colorSpaceName: NSCalibratedRGBColorSpace
                                                 bytesPerRow: 3 * width
                                                bitsPerPixel: 8 * 3];

   repData = [page bitmapData];
   PDFRender_GetRGB(bitmap, &repData);
   [page setSize: [self size]];
}

@end
