/*
     This file is part of GNUnet.
     (C) 2001, 2002, 2003, 2004 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file applications/afs/tools/gnunet-insert.c 
 * @brief Tool to insert or index files into GNUnet's AFS.
 * @author Christian Grothoff
 * @author Krista Bennett
 * @author James Blackwell
 * @author Igor Wronsky
 **/

#include "gnunet_afs_esed2.h"
#include "platform.h"
#if USE_LIBEXTRACTOR
#include <extractor.h>
#endif

/* hmm. Man says time.h, but that doesn't yield the
   prototype.  Strange... */
extern char *strptime(const char *s, const char *format, struct tm *tm);

/**
 * Print progess message.
 **/
static void printstatus(ProgressStats * stats,
			void * verboselevel) {
  if (*(int*)verboselevel == YES)
    printf("%8u of %8u bytes inserted\r",
	   (unsigned int) stats->progress,
	   (unsigned int) stats->filesize);  
}

/**
 * Prints the usage information for this command if the user errs.
 * Aborts the program.
 **/
static void printhelp() {
  static Help help[] = {
    { 'b', "builddir", NULL,
      "build a directory listing all processed files" },
    HELP_CONFIG,
    { 'D', "desc", "DESCRIPTION",
      "set description for all files" },
    { 'e', "sprev", "FILENAME",
      "filename of the SBlock of a previous version of the content"
      " (for namespace insertions only)" },
    { 'E', "extract", NULL,
      "print list of extracted keywords that would be used, but do not perform insertion or indexing" },
    { 'f', "name", "NAME",
      "publish NAME as the name of the file or directory" },
    HELP_HELP,
    HELP_HOSTNAME,
    { 'i', "interval", "SECONDS",
      "set interval for availability of updates to SECONDS"
      " (for namespace insertions only)" },
    { 'k', "key", "KEYWORD",
      "add an additional keyword for the top-level file or directory"
      " (this option can be specified multiple times)" },
    { 'K', "global-key", "KEYWORD",
      "add an additional keyword for all files and directories"
      " (this option can be specified multiple times)" },   
    { 'l', "link", NULL,
      "if gnunetd is running on the local machine, create a link instead of making a copy in the GNUnet share directory" },
    HELP_LOGLEVEL,
    { 'm', "mime", "MIMETYPE",
      "set the mimetype for the file to be MIMETYPE" },
    { 'n', "noindex", NULL,
      "do not index, perform full insertion (stores entire "
      "file in encrypted form in GNUnet database)" },
    { 'N', "next", "ID",
      "specify ID of an updated version to be published in the future"
      " (for namespace insertions only)" },
    { 'o', "sout", "FILENAME",
      "write the created SBlock in plaintext to FILENAME" 
      " (for namespace insertions only)" },
    { 'p', "prio", "PRIORITY",
      "specify the priority of the content" },
    { 'P', "pass", "PASSWORD",
      "use PASSWORD to decrypt the secret key of the pseudonym" 
      " (for namespace insertions only)" },
    { 'R', "recursive", NULL,
      "process directories recursively" },
    { 's', "pseudonym", "NAME",
      "publish the files under the pseudonym NAME (place file into namespace)" },
    { 'S', "sporadic", NULL,
      "specifies this as an aperiodic but updated publication"
      " (for namespace insertions only)" },
    { 't', "this", "ID",
      "set the ID of this version of the publication"
      " (for namespace insertions only)" },
    { 'T', "time", "TIME",
      "specify creation time for SBlock (see man-page for format)" },
    { 'u', "url", NULL,
      "print the GNUnet URL of the inserted file(s)" },
    HELP_VERSION,
    HELP_VERBOSE,
    { 'x', "noextraction", NULL,
      "disable automatic metadata extraction" },
    { 'X', "nodirectindex", NULL,
      "disable generation of RBlocks for keywords extracted from each file" },
    HELP_END,
  };
  formatHelp("gnunet-insert [OPTIONS] FILENAME*",
	     "Make files available to GNUnet for sharing.",
	     help);
}

/**
 * Insert a single file.
 *
 * @param filename the name of the file to insert
 * @param fid resulting file identifier for the node
 * @returns OK on success, SYSERR on error
 **/
static int doFile(GNUNET_TCP_SOCKET * sock,
		  char * filename,
		  FileIdentifier * fid,
		  int * verbose) {
  Block * top;
  cron_t startTime;

  cronTime(&startTime);
  if (YES == *verbose)
    printf("Working on file %s...\n",
	   filename); 
  top = insertFile(sock,
		   filename, 
		   &printstatus,
		   verbose);
  if (top == NULL) {
    printf("Error inserting file %s.\n"
	   "You may want to check whether or not you are out of space.\n"
	   "Run gnunet-stats | grep \"AFS storage left\" to check.\n",
	   filename);
    return SYSERR;
  } else {
    memcpy(&fid->chk, 
	   &top->chk, 
	   sizeof(CHK_Hashes));
    fid->crc = htonl(crc32N(top->data, top->len));
    fid->file_length = htonl(top->filesize);
    if (testConfigurationString("GNUNET-INSERT",
				"PRINTURL",
				"YES")) {
      char * fstring;
      fstring = fileIdentifierToString(fid);	
      printf("%s\n",
	     fstring);
      FREE(fstring);
    }
    if (*verbose == YES) {
      char * fstring;

      fstring = fileIdentifierToString(fid);	    
      printf("File %s successfully indexed -- %s\n",
	     filename,
	     fstring);
      printf("Speed was %8.3f kilobyte per second.\n",
	     (top->filesize/1024.0) / 
	     (((double)(cronTime(NULL)-startTime)) / (double)cronSECONDS) );
      FREE(fstring);
    }
    top->vtbl->done(top, NULL);
    return OK;
  }
}

static char ** topKeywords = NULL;
int topKeywordCnt = 0;
static char ** gloKeywords = NULL;
int gloKeywordCnt = 0;

static int parseOptions(int argc,
			char ** argv) {
  int c;
  int printAndReturn = NO;

  FREENONNULL(setConfigurationString("GNUNET-INSERT",
	  		 	     "INDEX-CONTENT",
			             "YES"));
  while (1) {
    int option_index=0;
    static struct GNoption long_options[] = {
      LONG_DEFAULT_OPTIONS,
      { "builddir",      0, 0, 'b' },
      { "sprev",         1, 0, 'e' },
      { "desc",          1, 0, 'D' },
      { "sporadic",      0, 0, 'S' },
      { "name",          1, 0, 'f' },
      { "interval",      1, 0, 'i' },
      { "extract",       0, 0, 'E' },
      { "link",          0, 0, 'l' },
      { "global-key",    1, 0, 'K' },
      { "key",           1, 0, 'k' },
      { "mime",          1, 0, 'm' },
      { "noindex",       0, 0, 'n' },
      { "next",          1, 0, 'N' },
      { "sout",          1, 0, 'o' },
      { "prio",          1, 0, 'p' },
      { "pass",          1, 0, 'P' },
      { "recursive",     0, 0, 'R' },
      { "pseudonym",     1, 0, 's' },
      { "this",          1, 0, 't' },
      { "time",          1, 0, 'T' },
      { "url",           0, 0, 'u' },
      { "verbose",       0, 0, 'V' },
      { "noextraction",  1, 0, 'x' },
      { "nodirectindex", 1, 0, 'X' },     
      { 0,0,0,0 }
    };    
    c = GNgetopt_long(argc,
		      argv, 
		      "vhdc:Ee:L:H:bD:Sf:i:lK:k:m:nN:o:p:P:Rs:t:T:uVxX", 
		      long_options, 
		      &option_index);    
    if (c == -1) 
      break;  /* No more flags to process */
    if (YES == parseDefaultOptions(c, GNoptarg))
      continue;
    switch(c) {
    case 'b':
      FREENONNULL(setConfigurationString("GNUNET-INSERT",
					 "BUILDDIR",
					 "YES"));
      break;
    case 'e':
      FREENONNULL(setConfigurationString("GNUNET-INSERT",
      					 "PREVIOUS_SBLOCK",
					 GNoptarg));
      break;
    case 'D':
      FREENONNULL(setConfigurationString("GNUNET-INSERT",
					 "DESCRIPTION",
					 GNoptarg));
      break;
    case 'E': 
      printAndReturn = YES;
      break;
    case 'f': {
      char * root;
      FREENONNULL(setConfigurationString("GNUNET-INSERT",
					 "FILENAME",
					 GNoptarg));
      root = getConfigurationString("GNUNET-INSERT",
				    "FILENAMEROOT");
      if (root == NULL) {
	/* if filename is '/home/user/foo', use 'foo' as the filenameRoot */
	unsigned int i;
	root = GNoptarg;
	for (i=0;i<strlen(GNoptarg);i++)
	  if (GNoptarg[i] == DIR_SEPARATOR)
	    root = &GNoptarg[i+1];
	root = STRDUP(root);
      }
      FREENONNULL(setConfigurationString("GNUNET-INSERT",
					 "FILENAMEROOT",
					 root));
      FREE(root);
      break;
    }
    case 'h': 
      printhelp(); 
      return SYSERR;
    case 'i': {
      unsigned int interval;
      if (1 != sscanf(GNoptarg, "%ud", &interval)) {
        LOG(LOG_FAILURE,
	    "FAILURE: You must pass a positive number to the -i option.\n");
	return -1;
      } else
	setConfigurationInt("GNUNET-INSERT",
			    "INTERVAL",
			    interval);
      break;    
    }      
    case 'k':
      GROW(topKeywords,
	   topKeywordCnt,
	   topKeywordCnt+1);
      topKeywords[topKeywordCnt-1] = STRDUP(GNoptarg);
      break;
    case 'K':
      GROW(gloKeywords,
	   gloKeywordCnt,
	   gloKeywordCnt+1);
      gloKeywords[gloKeywordCnt-1] = STRDUP(GNoptarg);
      break;
    case 'l':
      FREENONNULL(setConfigurationString("GNUNET-INSERT",
					 "LINK",
					 "YES"));
      break;
    case 'm':
      FREENONNULL(setConfigurationString("GNUNET-INSERT",
					 "MIMETYPE",
					 GNoptarg));
      break;
    case 'N': {
      HexName hex;
      HashCode160 nextId;
      
      if (tryhex2hash(GNoptarg,
		      &nextId) == SYSERR) 
	hash(GNoptarg,
	     strlen(GNoptarg),
	     &nextId);
      hash2hex(&nextId, &hex);
      FREENONNULL(setConfigurationString("GNUNET-INSERT",
					 "NEXTHASH",
					 (char*)&hex));
      break;
    }
    case 'o':
      FREENONNULL(setConfigurationString("GNUNET-INSERT",
      					 "OUTPUT_SBLOCK",
					 GNoptarg));
      break;
    case 'p': {
      unsigned int contentPriority;
      
      if (1 != sscanf(GNoptarg, "%ud", &contentPriority)) {
	LOG(LOG_FAILURE,
	    "FAILURE: You must pass a number to the -p option.\n");
	return SYSERR;
      }
      setConfigurationInt("GNUNET-INSERT",
			  "CONTENT-PRIORITY",
			  contentPriority);
      break;
    }
    case 'P':
      FREENONNULL(setConfigurationString("GNUNET-INSERT",
					 "PASSWORD",
					 GNoptarg));
      break;
    case 'R':
      FREENONNULL(setConfigurationString("GNUNET-INSERT",
					 "RECURSIVE",
					 "YES"));
      break;
    case 's':
      FREENONNULL(setConfigurationString("GNUNET-INSERT",
					 "PSEUDONYM",
					 GNoptarg));
      break;
    case 'S':
      FREENONNULL(setConfigurationString("GNUNET-INSERT",
					 "SPORADIC",
					 "YES"));
      break;
    case 't': {
      HexName hex;
      HashCode160 thisId;

      if (tryhex2hash(GNoptarg,
		      &thisId) == SYSERR) 
	hash(GNoptarg,
	     strlen(GNoptarg),
	     &thisId);
      hash2hex(&thisId, &hex);
      FREENONNULL(setConfigurationString("GNUNET-INSERT",
					 "THISHASH",
					 (char*)&hex));
      break;
    }
    case 'T':
      FREENONNULL(setConfigurationString("GNUNET-INSERT",
					 "INSERTTIME",
					 GNoptarg));
      break;
    case 'u':
      FREENONNULL(setConfigurationString("GNUNET-INSERT",
					 "PRINTURL",
					 "YES"));
      break;
    case 'V':
      FREENONNULL(setConfigurationString("GNUNET-INSERT",
					 "VERBOSE",
					 "YES"));
      break;
    case 'v': 
      printf("GNUnet v%s, gnunet-insert v%s\n",
	     VERSION, 
	     AFS_VERSION);
      return SYSERR;
    case 'n':
      FREENONNULL(setConfigurationString("GNUNET-INSERT",
					 "INDEX-CONTENT",
					 "NO"));
      break;
    case 'x':
#if USE_LIBEXTRACTOR
      FREENONNULL(setConfigurationString("GNUNET-INSERT",
					 "EXTRACT-KEYWORDS",
					 "NO"));
#else
      LOG(LOG_INFO,
      	  "INFO: compiled without libextractor, -x automatic\n");
#endif
      break;
    case 'X':
#if USE_LIBEXTRACTOR
      FREENONNULL(setConfigurationString("GNUNET-INSERT",
					 "ADDITIONAL-RBLOCKS",
					 "NO"));
#else
      LOG(LOG_INFO,
      	  "INFO: compiled without libextractor, -X automatic\n");
#endif
      break;
    default: 
      LOG(LOG_FAILURE,
	  "FAILURE: Unknown option %c. Aborting.\n"\
	  "Use --help to get a list of options.\n",
	  c);
      return SYSERR;
    } /* end of parsing commandline */
  } /* while (1) */
  if (argc == GNoptind) {
    printf("ERROR: you must specify a list of files to insert.\n");
    return SYSERR;
  }
  if (printAndReturn) {
#if USE_LIBEXTRACTOR
    EXTRACTOR_ExtractorList * l;
    l = getExtractors();
    for (c=GNoptind;c<argc;c++) {
      printf("Keywords for file %s:\n",
	     argv[c]);
      EXTRACTOR_KeywordList * list 
	= EXTRACTOR_getKeywords(l, argv[c]);
      EXTRACTOR_printKeywords(stdout,
			      list);
      EXTRACTOR_freeKeywords(list);
    }
    EXTRACTOR_removeAll(l);
#else
    printf("libextractor not used, no keywords will be extracted.\n");
#endif
    return SYSERR;
  }
  setConfigurationStringList(&argv[GNoptind],
			     argc - GNoptind);
  return OK;
}

/**
 * Insert the given RBlock into GNUnet.
 * @param rb the root node
 * @param keyword the keyword to use
 **/
static void insertRBlock(GNUNET_TCP_SOCKET * sock,
			 RootNode * rb,
			 char * keyword) {
  if (OK != insertRootWithKeyword(sock,
				  rb,
				  keyword,
				  getConfigurationInt("GNUNET-INSERT",
						      "CONTENT-PRIORITY")))
    printf("ERROR inserting RootNode. "
	   "Is gnunetd running and space available?\n");
}



/**
 * The main function to insert files into GNUnet.
 *
 * @param argc number of arguments from the command line
 * @param argv command line arguments
 * @return return 0 for ok, -1 on error
 **/   
int main(int argc, char ** argv) {
  RootNode * roots;
  int i;
  char * pname;
  Hostkey pseudonym;
  HashCode160 hc;
  HexName hex1;
  HexName hex2;
  char ** fileNames;
  char * fileName;
  int fileNameCount;
  FileIdentifier fid;
  RootNode * r;
  char * description;
  char * mimetype;
  char * shortFN;
#if USE_LIBEXTRACTOR
  EXTRACTOR_ExtractorList * extractors;
  char ** keywords;
  int num_keywords;
#endif
  int interval;
  int skip;
  GNUNET_TCP_SOCKET * sock;
  int verbose;
  
  if (SYSERR == initUtil(argc, argv, &parseOptions)) 
    return 0;

  verbose = testConfigurationString("GNUNET-INSERT",
				    "VERBOSE",
				    "YES");
 
  /* check arguments */
  pname = getConfigurationString("GNUNET-INSERT",
				 "PSEUDONYM");
  if (pname != NULL) {
    char * password 
      = getConfigurationString("GNUNET-INSERT",
			       "PASSWORD");
    pseudonym = readPseudonym(pname,
			      password);
    if (pseudonym == NULL) {
      printf("ERROR: could not read pseudonym %s (does not exist or password invalid).\n",
	     pname);
      FREE(pname);
      FREENONNULL(password);
      doneUtil();
      return -1;
    }
    FREENONNULL(password);
    FREE(pname);
  } else
    pseudonym = NULL;
  /* fixme: other sanity checks here? */

  fileNameCount = getConfigurationStringList(&fileNames);

  if ( ( testConfigurationString("GNUNET-INSERT",
				"BUILDDIR",
				"YES") ||
	 testConfigurationString("GNUNET-INSERT",
				 "RECURSIVE",
				 "YES") ||
	 (fileNameCount > 1) )
       && (NULL != getConfigurationString("GNUNET-INSERT",
					  "FILENAMEROOT") ) )
    errexit("FATAL: the options -b, -r or multiple file"
	    " arguments can not be used together with option -f.\n");

  if (pseudonym == NULL) {
    if (NULL != getConfigurationString("GNUNET-INSERT",
				       "NEXTHASH"))
      errexit("FATAL: Option -N makes no sense without -P\n");
    if (NULL != getConfigurationString("GNUNET-INSERT",
				       "THISHASH"))
      errexit("FATAL: Option -t makes no sense without -P\n");
    if (NULL != getConfigurationString("GNUNET-INSERT",
				       "PASSWORD"))
      errexit("FATAL: Option -P makes no sense without -P\n");
    if (0 != getConfigurationInt("GNUNET-INSERT",
				 "INTERVAL"))
      errexit("FATAL: Option -i makes no sense without -P\n");
    if (testConfigurationString("GNUNET-INSERT",
				"SPORADIC",
				"YES"))
      errexit("FATAL: Option -S makes no sense without -P\n");
  }
#if USE_LIBEXTRACTOR
    if (testConfigurationString("GNUNET-INSERT",
				"EXTRACT-KEYWORDS",
				"NO") &&
	testConfigurationString("GNUNET-INSERT",
				"ADDITIONAL-RBLOCKS",
				"NO") )
      printf("WARNING: -X is implied by -x.\n");
#endif
  
  
  /* fundamental init */
  sock = getClientSocket();
  if (sock == NULL)
    errexit("FATAL: could not connect to gnunetd.\n");
#if USE_LIBEXTRACTOR
  extractors = getExtractors();
#endif
  mimetype = getConfigurationString("GNUNET-INSERT",
				    "MIMETYPE");

  roots = MALLOC(sizeof(RootNode) * fileNameCount);
  skip = 0;
  fileName = NULL;
  for (i=0;i<fileNameCount;i++) {
    FREENONNULL(fileName);
    fileName = expandFileName(fileNames[i]);
    r = insertRecursively(sock,
			  fileName,
			  &fid,
			  gloKeywords,
			  gloKeywordCnt,
#if USE_LIBEXTRACTOR
			  extractors,
#else
			  NULL,
#endif
			  &printstatus,
			  &verbose,
			  (InsertWrapper)&doFile,
			  &verbose);    
    if (r != NULL) {
      memcpy(&roots[i], r, sizeof(RootNode));
      FREE(r);
    } else
      skip++;
  }
  if ( (fileNameCount == 1) &&
       isDirectory(fileName) ) {
    FREENONNULL(mimetype);
    mimetype = STRDUP(GNUNET_DIRECTORY_MIME);
  }


  shortFN = getConfigurationString("GNUNET-INSERT",
				   "FILENAME");
  if ( (shortFN == NULL) && (fileName != NULL)) {
    shortFN = &fileName[strlen(fileName)-1];
    while (shortFN[-1] != DIR_SEPARATOR)
      shortFN--;
    shortFN = STRDUP(shortFN);
  }
  fileNameCount -= skip;
  /* if build directory and > 1 file, build directory
     and reduce to 1 file */
  if ( (fileNameCount > 1) &&
       testConfigurationString("GNUNET-INSERT",
			       "BUILDDIR",
			       "YES") ) {
    FREENONNULL(fileName);
    fileName = getConfigurationString("GNUNET-INSERT",
				      "FILENAMEROOT");
    if (fileName == NULL)
      fileName = STRDUP("no name specified");
    i = insertDirectory(sock,
			fileNameCount,
			roots,
			fileName,
			&fid,
			&printstatus,
			&verbose);
    if (i == SYSERR) {
      skip += fileNameCount;
      fileNameCount = 0;
    } else {      
      if (testConfigurationString("GNUNET-INSERT",
				  "PRINTURL",
				  "YES")) {
	char * fstring;
	fstring = fileIdentifierToString(&fid);	
	printf("%s\n",
	       fstring);
	FREE(fstring);
      }
      if (verbose == YES) {
	char * fstring;
	fstring = fileIdentifierToString(&fid);	
	printf("Directory %s successfully indexed -- %s\n",
	       fileName,
	       fstring);
	FREE(fstring);
      }

      description = getConfigurationString("GNUNET-INSERT",
					   "DESCRIPTION");
      if (description == NULL)
        description = STRDUP("no description supplied");
      r = buildDirectoryRBlock(sock,
			       &fid,
			       fileName,
			       description,
			       gloKeywords,
			       gloKeywordCnt);
      FREE(description);
      FREE(r);
      mimetype = STRDUP(GNUNET_DIRECTORY_MIME);
      skip = fileNameCount - 1;
      fileNameCount = 1;
    }
  }
  FREE(roots);

#if USE_LIBEXTRACTOR
  num_keywords = 0;
  keywords = NULL;
#endif

  /* if fileNameCount == 1 and !isDirectory(fileName): run libextractor! */
  description = getConfigurationString("GNUNET-INSERT",
				       "DESCRIPTION");
  if ( (fileNameCount == 1) &&
       (!isDirectory(fileName)) ) {
#if USE_LIBEXTRACTOR
    if (!testConfigurationString("GNUNET-INSERT",
				 "EXTRACT-KEYWORDS",
				 "NO")) {
      extractKeywordsMulti(fileName,
			   &description,
			   &mimetype,
			   &keywords,
			   &num_keywords,
			   extractors);
    }
#endif
  }
  if (mimetype == NULL)
    mimetype = STRDUP("unknown");
  if (description == NULL)
    description = STRDUP("no description supplied");
  
  /* if a directory, add mimetype as key unless forbidden */
  if (strcmp(mimetype,GNUNET_DIRECTORY_MIME)==0 &&
      !testConfigurationString("GNUNET-INSERT",
                               "ADDITIONAL-RBLOCKS",
                               "NO")) {
      GROW(topKeywords,
           topKeywordCnt,
           topKeywordCnt+1);
      topKeywords[topKeywordCnt-1] = STRDUP(GNUNET_DIRECTORY_MIME);
  }

  /* if fileNameCount == 1, add RBlocks for all keywords */
  if ( fileNameCount == 1) {
    r = createRootNode(&fid,
		       description,
		       shortFN,
		       mimetype);
    for (i=0;i<gloKeywordCnt;i++) {
      insertRBlock(sock, 
		   r, 
		   gloKeywords[i]);    
      printf("Inserting %s (%s, %s) under keyword %s.\n",
	     shortFN, description, mimetype, gloKeywords[i]);
    }
    for (i=0;i<topKeywordCnt;i++) {
      insertRBlock(sock,
		   r,
		   topKeywords[i]);    
      printf("Inserting %s (%s, %s) under keyword %s.\n",
	     shortFN, description, mimetype, topKeywords[i]);
    }
#if USE_LIBEXTRACTOR
    if (! testConfigurationString("GNUNET-INSERT",
				  "ADDITIONAL-RBLOCKS",
				  "NO") )
      for (i=0;i<num_keywords;i++) {
	insertRBlock(sock,
		     r,
		     keywords[i]);    
	printf("Inserting %s (%s, %s) under keyword %s.\n",
	       shortFN, description, mimetype, keywords[i]);
      }
#endif
    FREE(r);
  }

  /* if SBlock and == 1 file, create SBlock */
  if ( (pseudonym != NULL) &&
       (fileNameCount == 1) ) {
    HashCode160 thisId;
    HashCode160 nextId;
    char * hx;
    char * prevname;
    SBlock * sb;
    TIME_T creationTime;
    TIME_T now;
    char * timestr;
   
    timestr = getConfigurationString("GNUNET-INSERT",
                    		     "INSERTTIME");
    if(timestr != NULL) {
      struct tm t;

      if((NULL == strptime(timestr, "%j-%m-%Y %R", &t)))
        errexit("FATAL: parsing time failed. Use \"DAY-MONTHNUMBER-YEAR HOUR:MINUTE\" format\n");
      now = mktime(&t);
      FREE(timestr);
      /* On my system, the printed time is in advance +1h 
	 to what was specified? -- It is in UTC! */
      timestr = GN_CTIME(&now);
      LOG(LOG_DEBUG, 
          "DEBUG: read time %s\n", 
	  timestr);
      FREE(timestr);
    } else {
      /* use current time */
      TIME(&now);
    }

    prevname = getConfigurationString("GNUNET-INSERT",
    				      "PREVIOUS_SBLOCK");
    if(prevname != NULL) {
      /* options from the previous sblock override */
      SBlock pb; 	
      PublicKey pkey;
      
      if (sizeof(SBlock) != readFile(prevname,
      				     sizeof(SBlock),
				     &pb) ) {
        errexit("FATAL: SBlock in %s either doesn't exist or is malformed\n",
		prevname);
      }
      /* check that it matches the selected pseudonym */
      getPublicKey(pseudonym,
      		   &pkey);
      if (0 != memcmp(&pkey,
      	  	      &pb.subspace,
		      sizeof(PublicKey)))
        errexit("FATAL: Given SBlock doesn't match the selected pseudonym");
					   
      if (SYSERR == verifySBlock(&pb)) {
        errexit("FATAL: Verification of SBlock in %s failed\n", 
		prevname);
      }
      interval = ntohl(pb.updateInterval);

      /* now, compute CURRENT ID and next ID */
      computeIdAtTime(&pb,
      	              now,
		      &thisId); 
      if ( (interval != SBLOCK_UPDATE_NONE) &&
           (interval != SBLOCK_UPDATE_SPORADIC) ) {  
        int delta;
	
        /* periodic update */
        delta = now - ntohl(pb.creationTime);
        delta = delta / ntohl(pb.updateInterval);
        if (delta <= 0)
          delta = 1; /* force to be in the future from the updated block! */
        creationTime = ntohl(pb.creationTime) + delta * ntohl(pb.updateInterval);

        /* periodic update, compute _next_ ID as increment! */
        addHashCodes(&thisId,
		     &pb.identifierIncrement,
		     &nextId); /* n = k + inc */
      } else {
        creationTime = now;
        if (interval == SBLOCK_UPDATE_SPORADIC) {
          LOG(LOG_DEBUG,
              "DEBUG: sporadic update in sblock...\n");
          hx = getConfigurationString("GNUNET-INSERT",
                                      "NEXTHASH");
          if (hx == NULL) {
            makeRandomId(&nextId);
          } else {
            tryhex2hashOrHashString(hx, &nextId);
            FREE(hx);
          }
        } else {
          errexit("FATAL: trying to update nonupdatable sblock\n");
        }
      }
    } else {
      /* no previous sblock specified */
      creationTime = now;
      interval = getConfigurationInt("GNUNET-INSERT",
	  	 	  	     "INTERVAL");
      hx = getConfigurationString("GNUNET-INSERT",
 	 	 	  	  "THISHASH");
      tryhex2hashOrHashString(hx, &thisId);
      FREENONNULL(hx);
      hx = getConfigurationString("GNUNET-INSERT",
 	  	 	  	  "NEXTHASH");
      if (hx == NULL) {
        if (interval == SBLOCK_UPDATE_NONE) {
	  /* no next id and no interval specified, to be    */
	  /* consistent with gnunet-gtk, nextId == thisId   */
	  memcpy(&nextId,
	  	 &thisId,
		 sizeof(HashCode160));
	} else {
          makeRandomId(&nextId);
	}
      } else {
        tryhex2hashOrHashString(hx, &nextId);
        if (interval == SBLOCK_UPDATE_NONE) {
	  /* if next was specified, aperiodic is default */
  	  interval = SBLOCK_UPDATE_SPORADIC; 
	}
        FREE(hx); 
      }
      if (testConfigurationString("GNUNET-INSERT",
  	 	 	  	  "SPORADIC",
				  "YES"))
        interval = SBLOCK_UPDATE_SPORADIC;
    }
  
    sb = buildSBlock(pseudonym,
		     &fid,
		     description,
		     shortFN,
		     mimetype,
		     creationTime,
		     interval,
		     &thisId,
		     &nextId);
    freeHostkey(pseudonym);
    hash2hex(&thisId,
	     &hex1);
    hash(&sb->subspace,
	 sizeof(PublicKey),
	 &hc);
    hash2hex(&hc,
	     &hex2);
    if (OK == insertSBlock(sock,
			   sb)) {
      char * outname;
      
      outname = getConfigurationString("GNUNET-INSERT",
      				       "OUTPUT_SBLOCK");
      if (outname != NULL) {
        SBlock plainSBlock;

	decryptSBlock(&thisId,
		      sb,
                      &plainSBlock);
        writeFile(outname, 
	          &plainSBlock,
		  sizeof(SBlock),
		  "600");
	FREE(outname);
      }     
      /* FIXME: until URI is decided for sblocks, stick to SPACE KEY -format */
      printf("File %s (%s, %s) successfully inserted into namespace under\n"
	     "  %s %s\n",
	     shortFN, description, mimetype,
	     (char*)&hex2,
	     (char*)&hex1);
    } else
      printf("Insertion of file into namespace failed.\n");
    FREE(sb);
  }

#if USE_LIBEXTRACTOR
  for (i=0;i<num_keywords;i++) 
    FREE(keywords[i]);
  GROW(keywords, num_keywords, 0);
  EXTRACTOR_removeAll(extractors);
#endif
  for (i=0;i<fileNameCount+skip;i++)
    FREE(fileNames[i]);
  FREE(fileNames);
  FREENONNULL(fileName);
  for (i=0;i<topKeywordCnt;i++) 
    FREE(topKeywords[i]);
  GROW(topKeywords, topKeywordCnt, 0);
  for (i=0;i<gloKeywordCnt;i++) 
    FREE(gloKeywords[i]);
  GROW(gloKeywords, gloKeywordCnt, 0);
  FREE(mimetype);
  FREE(description);
  FREE(shortFN);
  releaseClientSocket(sock); 
  doneUtil();
  return 0;
}  

/* end of gnunet-insert.c */
