/*
     This file is part of gnunet-qt.
     (C) 2007 Nils Durner (and other contributing authors)

     gnunet-qt is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     gnunet-qt is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with gnunet-qt; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/plugins/fs/uploadController.cc
 * @brief Controller for *all* uploads
 * @author Nils Durner
 */

#include <math.h>
#include <QDir>
#include <QFileInfo>
#include <QStringList>
#include <QHeaderView>
#include <extractor.h>
#include <GNUnet/gnunet_ecrs_lib.h>
#include "uploadController.h"

static int getKeyword(const char *data, void *list)
{
  ((QStringList *) list)->append(QString::fromUtf8(data));
  
  return OK;
}

static int getMetaData(EXTRACTOR_KeywordType type, const char *data, void *list)
{
  ((GFSMetaData *) list)->insert(type, data);
  
  return OK;
}

GFSUploadController::GFSUploadController(GFSPlugin *fs)
{
  this->fs = fs;
  extractors = NULL;
  
  uploadModel.setColumnCount(3);
  uploadModel.setHeaderData(0, Qt::Horizontal, tr("Filename"), Qt::DisplayRole);
  uploadModel.setHeaderData(1, Qt::Horizontal, tr("Progress"), Qt::DisplayRole);
  uploadModel.setHeaderData(2, Qt::Horizontal, tr("Status"), Qt::DisplayRole);
  
  fs->uploadView()->setModel(&uploadModel);
  fs->uploadView()->setItemDelegate(&delegate);
}

GFSUploadController::~GFSUploadController()
{
  if (extractors)
    EXTRACTOR_removeAll(extractors);
}

void GFSUploadController::setProgress(QPersistentModelIndex *idx,
  unsigned long long completed, unsigned long long total)
{
  QModelIndex item, parent;
  double progress;
  
  // compute progress for this entry
  progress = ((double) completed) / total * 100;
  if (isnan(progress))
    progress = 0;

  parent = idx->parent();
  item = uploadModel.index(idx->row(), 1, parent);
  uploadModel.setData(item, QVariant(progress), Qt::DisplayRole);
  
  // compute overall progress for parent items
  while(parent.isValid())
  {
    int children = 0;
    double overallProgress = 0;
    QModelIndex progressIdx;
    
    item = parent.child(children, 1);
    while (item.isValid())
    {
      children++;
      overallProgress += uploadModel.data(item).toDouble();
      item = parent.child(children, 1);
    }
    
    progressIdx = uploadModel.index(parent.row(), 1, parent.parent());
    uploadModel.setData(progressIdx, QVariant(overallProgress / children),
      Qt::DisplayRole);
    
    parent = parent.parent();
  }
}

bool GFSUploadController::start(QWidget *parent, const QString &strPath,
  bool index, int prio, int anon)
{
  GFSMetaData metaData;
  QStringList keywords;
  struct ECRS_MetaData *meta;
  QFileInfo fileInfo;
  struct ECRS_URI *keywordUri;
  int thumbSize;
  unsigned char *thumb;

  parent->setCursor(Qt::WaitCursor);

  // load libextractor
  if (! extractors)
  {
    char *config;

    extractors = EXTRACTOR_loadDefaultLibraries();
    if (GC_get_configuration_value_string(fs->config(), "FS", "EXTRACTORS",
      NULL, &config) == 0 && config)
    {
      extractors = EXTRACTOR_loadConfigLibraries(extractors, config);
      FREE(config);
    }
  }

  // extract meta data
  meta = ECRS_createMetaData();
  if (!meta)
  {
      GE_LOG(fs->errorContext(), (GE_KIND) (GE_ERROR | GE_USER | GE_IMMEDIATE),
        qPrintable(tr("Internal error: failed to create meta data for publication.")));
      return false;
  }
  
  ECRS_extractMetaData(fs->errorContext(), meta, strPath.toLocal8Bit().data(), extractors);
  ECRS_getMetaData(meta, &getMetaData, &metaData);
  thumbSize = ECRS_getThumbnailFromMetaData(meta, &thumb);
  if (thumbSize)
    metaData.insert(EXTRACTOR_THUMBNAIL_DATA, QByteArray((const char *) thumb,
      thumbSize));
  
  // Keywords
  keywordUri = ECRS_metaDataToUri(meta);
  ECRS_freeMetaData(meta);
  if (keywordUri)
  {
    ECRS_getKeywordsFromUri(keywordUri, getKeyword, &keywords);
    ECRS_freeUri(keywordUri);
  }

  fileInfo.setFile(strPath);
  if (fileInfo.isDir())
  {
    QDir dir(strPath);
    QString name = dir.dirName();

    if (!keywords.contains(name))    
      keywords.append(name);
  }  
 
  parent->setCursor(Qt::ArrowCursor);
 
  GFSUploadDialog uploadDlg(&metaData, &keywords, fs);
  
  if (uploadDlg.exec() == QDialog::Accepted)
  {
    struct FSUI_UploadList *handle;
    GFSMetaData::iterator it;
    struct ECRS_URI *globalURI, *keywordURI;
    char **keys;
    long idx;
    
    // prepare meta data
    meta = ECRS_createMetaData();
    for(it = metaData.begin(); it != metaData.end(); it++)
      ECRS_addToMetaData(meta, it.key(), it->data());

    // prepare global URI
    globalURI = ECRS_stringToUri(fs->errorContext(), ECRS_URI_PREFIX
      ECRS_SEARCH_INFIX);
      
    // prepare keywords URI
    idx = keywords.count();
    keys = new char *[idx + 1];
    keys[idx] = NULL;
    idx--;
    while (idx >= 0)
    {
      keys[idx] = strdup(keywords[idx].toUtf8());
    
      idx--;
    }
    
    keywordURI = ECRS_keywordsToUri((const char **) keys);
    
    // upload
    handle = FSUI_startUpload(fs->context(), strPath.toLocal8Bit(),
      (DirectoryScanCallback) &disk_directory_scan, fs->errorContext(),
      anon, prio, index, uploadDlg.extract(), uploadDlg.useKeywords(),
      get_time() + 2 * cronYEARS, meta, globalURI, keywordURI);
    
    // cleanup
    idx = 0;
    while(keys[idx])
    {
      free(keys[idx]);
      idx++;
    }
    delete [] keys;
    
    ECRS_freeMetaData(meta);
    ECRS_freeUri(globalURI);
    ECRS_freeUri(keywordURI);
    
    return true;
  }
  
  return false;
}

void GFSUploadController::state(QPersistentModelIndex *idx, FSUI_EventType event)
{
  QModelIndex index = uploadModel.index(idx->row(), 2, idx->parent());
  
  uploadModel.setData(index, QVariant(fs->fsuiState(event)), Qt::DisplayRole);
  uploadModel.setData(index, QVariant(event == FSUI_upload_completed),
    Qt::UserRole);
}

QAbstractItemModel *GFSUploadController::model()
{
  return &uploadModel;
}

QPersistentModelIndex *GFSUploadController::started(QPersistentModelIndex *parent,
    QString path, unsigned long long total, unsigned long long completed)
{
  QPersistentModelIndex *idx;
  QStandardItem *item, *parentItem;
  QString displayPath;
  
  displayPath = QFileInfo(path).fileName();
  if (displayPath == "")
    displayPath = QDir(path).dirName();
  
  if (parent)
    parentItem = uploadModel.itemFromIndex(uploadModel.index(parent->row(), parent->column(), parent->parent()));
  else
    parentItem = uploadModel.invisibleRootItem();
  
  item = new QStandardItem(displayPath);
  item->setColumnCount(3);
  parentItem->appendRow(item);
  
  idx = new QPersistentModelIndex(item->index());
  setProgress(idx, completed, total);

  state(idx, (total != completed || total == 0) ? FSUI_upload_started : FSUI_upload_completed);
  
  return idx;
}

void GFSUploadController::progress(QPersistentModelIndex *idx,
  unsigned long long completed, unsigned long long total)
{
  setProgress(idx, completed, total);
  state(idx, FSUI_upload_progress);
}

void GFSUploadController::clear()
{
  int row = 0;
  QModelIndex idx = uploadModel.index(0, 2);
  
  while(idx.isValid())
  {
    if (uploadModel.data(idx, Qt::UserRole).toInt() == 1)
      uploadModel.removeRow(row);
    else
      idx = uploadModel.index(++row, 2);
  }
}

/* end of uploadController.cc */
