/*
     This file is part of gnunet-qt.
     (C) 2007 Nils Durner (and other contributing authors)

     gnunet-qt is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     gnunet-qt is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with gnunet-qt; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/plugins/fs/downloadController.cc
 * @brief Controller for *all* downloads
 * @author Nils Durner
 */

#include <math.h>
#include <QFileInfo>
#include <QDir>
#include <QStandardItem>
#include "gnunet_qt_common.h"
#include "downloadController.h"

GFSDownloadController::GFSDownloadController(GFSPlugin *fs)
{
  QTreeView *view;
  
  this->fs = fs;
  view = fs->downloadView();
  
  downloadModel.setColumnCount(COL_COUNT);
  downloadModel.setHeaderData(COL_FILENAME, Qt::Horizontal, tr("Filename"),
    Qt::DisplayRole);
  downloadModel.setHeaderData(COL_SIZE, Qt::Horizontal, tr("Size"),
    Qt::DisplayRole);
  downloadModel.setHeaderData(COL_PROGRESS, Qt::Horizontal, tr("Progress"),
    Qt::DisplayRole);
  downloadModel.setHeaderData(COL_STATUS, Qt::Horizontal, tr("Status"),
    Qt::DisplayRole);
  downloadModel.setHeaderData(COL_ETA, Qt::Horizontal, tr("ETA"),
    Qt::DisplayRole);
  downloadModel.setHeaderData(COL_DST_PATH, Qt::Horizontal,
    tr("Destination path"), Qt::DisplayRole);
  
  view->setModel(&downloadModel);
  view->setItemDelegate(&delegate);
  view->hideColumn(COL_ETA);
  view->hideColumn(COL_DST_PATH);
}

GFSDownloadController::~GFSDownloadController()
{
}

void GFSDownloadController::setProgress(QPersistentModelIndex *idx,
  unsigned long long completed, unsigned long long total, cron_t eta)
{
  QModelIndex item, parent;
  double progress;
  
  // compute progress for this entry
  progress = ((double) completed) / total * 100;
  if (isnan(progress))
    progress = 0;

  parent = idx->parent();
  item = downloadModel.index(idx->row(), COL_PROGRESS, parent);
  downloadModel.setData(item, QVariant(progress), Qt::DisplayRole);
  item = downloadModel.index(idx->row(), COL_ETA, parent);
  downloadModel.setData(item, QVariant(eta), Qt::DisplayRole);
  
  // compute overall progress for parent items
  while(parent.isValid())
  {
    int children = 0;
    double overallProgress = 0;
    cron_t overallETA = 0;
    QModelIndex progressIdx;
    
    item = parent.child(children, COL_PROGRESS);
    while (item.isValid())
    {
      children++;
      overallProgress += downloadModel.data(item).toDouble();
      item = parent.child(children, COL_ETA);
      overallETA += downloadModel.data(item).toLongLong();
      item = parent.child(children, COL_PROGRESS);
    }
    
    progressIdx = downloadModel.index(parent.row(), COL_PROGRESS, parent.parent());
    downloadModel.setData(progressIdx, QVariant(overallProgress / children),
      Qt::DisplayRole);
    progressIdx = downloadModel.index(parent.row(), COL_ETA, parent.parent());
    downloadModel.setData(progressIdx, QVariant(overallETA), Qt::DisplayRole);
    
    parent = parent.parent();
  }
}

/**
 * @brief Initiates a download
 * @param searchIdx index of the search result item
 * @param parentSearch pointer to search
 * @param uri ECRS URI to download
 * @param meta meta data
 * @param gnPath path inside a GNUnet directory
 * @param name name of the file
 * @param destPath top-level destination path
 * @param anonymity desired receiver anonymity
 * @param recursive true for a recursive directory download
 */
void GFSDownloadController::start(QPersistentModelIndex &searchIdx,
  struct FSUI_SearchList *parentSearch, GFSEcrsUri &uri, GFSEcrsMetaData &meta,
  QString gnPath, QString name, QString destPath, int anonymity, bool recursive)
{
  GString path;
  QString gnDir;

  // FIXME: check if download is pending
  
  // cleanup filename
  if (gnPath.endsWith("/") || gnPath.endsWith("\\"))
    gnPath = gnPath.left(gnPath.length() - 1);
  
  path = gnPath.replace("//", "/");
  while (path != gnPath)
  {
    gnPath = path;
    path = gnPath.replace("//", "/");
  }

  path = gnPath.replace("\\\\", "\\");
  while (path != gnPath)
  {
    gnPath = path;
    path = gnPath.replace("\\\\", "\\");
  }

  path = gnPath.replace("..", ".");
  while (path != gnPath)
  {
    gnPath = path;
    path = gnPath.replace("..", ".");
  }
  
  // generate name if needed
  if (name == "")
    name = uri.toString().left(16);
  GNUNETQT_ASSERT(name != "");
  
  if (destPath == "")
  { 
    char *dir;
    
    GC_get_configuration_value_filename(fs->config(), "FS", "INCOMINGDIR",
      "$HOME/gnunet-downloads/", &dir);
    destPath = dir;
  }
  
  path = destPath + QDir::separator() + gnPath +  QDir::separator() + name;
  
  FSUI_startDownload(fs->context(), anonymity, recursive, uri.uri(), meta.meta(),
    path.toCString(), parentSearch, NULL /* FIXME */);
    
  downloadList.insert(uri, searchIdx);
}

QPersistentModelIndex *GFSDownloadController::started(struct FSUI_DownloadList *handle,
  QPersistentModelIndex *parent, const ECRS_FileInfo *fi, QString name, unsigned long long total,
  unsigned long long completed)
{
  QModelIndex tmpIdx;
  QPersistentModelIndex *idx;
  QStandardItem *item, *parentItem;
  QString displayPath;
  unsigned long long size;
  
  displayPath = QFileInfo(name).fileName();
  if (displayPath == "")
    displayPath = QDir(name).dirName();
  
  if (parent)
    parentItem = downloadModel.itemFromIndex(downloadModel.index(parent->row(),
      parent->column(), parent->parent()));
  else
    parentItem = downloadModel.invisibleRootItem();
  
  item = new QStandardItem(displayPath);
  item->setColumnCount(COL_COUNT);
  item->setData(QVariant::fromValue((void *) handle), Qt::UserRole);
  parentItem->appendRow(item);
  
  tmpIdx = downloadModel.index(item->index().row(), COL_DST_PATH);
  downloadModel.setData(tmpIdx, QVariant(name), Qt::DisplayRole);
  
  size = ECRS_fileSize(fi->uri);
  tmpIdx = downloadModel.index(item->index().row(), COL_SIZE);
  downloadModel.setData(tmpIdx, QVariant(GString::fromByteSize(size)),
    Qt::DisplayRole);
  
  idx = new QPersistentModelIndex(item->index());
  setProgress(idx, completed, total, (cron_t) -1);
  
  state(idx, (total != completed) ? FSUI_download_started : FSUI_download_completed);
  
  return idx;
}

void GFSDownloadController::progress(QPersistentModelIndex *idx, unsigned long long completed,
  unsigned long long total, cron_t eta)
{
  setProgress(idx, completed, total, eta);
  state(idx, FSUI_download_progress);
}

void GFSDownloadController::completed(QPersistentModelIndex *idx, GFSEcrsUri uri, QString file)
{
  GFSDownloadList::iterator it;

  it = downloadList.find(uri);
  if (it != downloadList.end())
    fs->searchController()->downloadCompleted(*it, file);
  state(idx, FSUI_download_completed);
}

void GFSDownloadController::state(QPersistentModelIndex *idx, FSUI_EventType type)
{
  QModelIndex index = downloadModel.index(idx->row(), COL_STATUS, idx->parent());
  
  downloadModel.setData(index, QVariant(fs->fsuiState(type)), Qt::DisplayRole);
  if (type == FSUI_download_stopped)
    downloadModel.removeRow(idx->row());
  else
    downloadModel.setData(index,
      QVariant(type == FSUI_download_completed || type == FSUI_download_aborted),
      Qt::UserRole);
}

void GFSDownloadController::clear()
{
  int row = 0;
  QModelIndex idx = downloadModel.index(0, COL_STATUS);
  
  while(idx.isValid())
  {
    if (downloadModel.data(idx, Qt::UserRole).toInt() == 1)
    {
      QStandardItem *item;
      
      item = downloadModel.item(idx.row());
      FSUI_stopDownload(fs->context(), (struct FSUI_DownloadList *)
        item->data(Qt::UserRole).value<void *>());      
    }
    else
      idx = downloadModel.index(++row, COL_STATUS);
  }
}

void GFSDownloadController::cancel(struct FSUI_DownloadList *handle)
{
  FSUI_abortDownload(fs->context(), handle);
  FSUI_stopDownload(fs->context(), handle);
}

QAbstractItemModel *GFSDownloadController::model()
{
  return &downloadModel;
}

/* end of downloadController.cc */
