/*
     This file is part of GNUnet
     (C) 2012 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/setup/gnunet-setup-gns-edit.h
 * @author Christian Grothoff
 * @brief editing dialogs for GNS records
 */
#ifndef GNUNET_SETUP_GNS_EDIT_H
#define GNUNET_SETUP_GNS_EDIT_H

#include "gnunet_gtk.h"
#include "gnunet-setup-gns.h"
#include <gnunet/gnunet_namestore_service.h>
#include <gnunet/gnunet_dnsparser_lib.h>


/**
 * Context for edit operations.
 */
struct EditDialogContext;


/**
 * Signature of the continuation function invoked once the
 * edit dialog is done.
 *
 * @param edc dialog context
 * @param ret return code from the dialog
 */
typedef void (*GNS_EditContinuation)(struct EditDialogContext *edc,
				     GtkResponseType ret);


/**
 * Signature of a dialog-specific validation function to validate the
 * value.  Should either call 'edit_dialog_disable_save' if the value
 * is invalid, or 'edit_dialog_check_save' if the value is valid.
 */
typedef void (*GNS_EditValidator)(struct EditDialogContext *edc);


/**
 * Context for edit operations.
 */
struct EditDialogContext
{

  /**
   * If a record move operation is desired, we keep these in a DLL.
   */
  struct EditDialogContext *next;

  /**
   * If a record move operation is desired, we keep these in a DLL.
   */
  struct EditDialogContext *prev;

  /**
   * Where in the tree view are we editing?
   */
  GtkTreeIter it;

  /**
   * Expiration time value (absolute or relative).
   */
  guint64 n_exp_time;

  /**
   * Function to call at the end.
   */
  GNS_EditContinuation cont;

  /**
   * Function to call to validate the state of the dialog and
   * ultimately update the 'save' button sensitivity.
   */
  GNS_EditValidator validator;

  /**
   * Builder for the dialog (internal to gnunet-setup-gns-edit.c).
   */
  GtkBuilder *builder;

  /**
   * Main dialog window.
   */
  GtkDialog *dialog;

  /**
   * Context for loading/generating the zone key for the target zone
   * (used if the edit operation causes the record to be moved).
   */
  struct GNUNET_CRYPTO_RsaKeyGenerationContext *rkgc;

  /**
   * Associated namestore operation.
   * (used if the edit operation causes the record to be moved).
   */
  struct GNUNET_NAMESTORE_QueueEntry *qe;

  /**
   * Old name of the record (for deletion).
   */
  gchar *n_name;

  /**
   * New name.
   */
  gchar *n_new_name;

  /**
   * Value of the record in string format.
   */
  gchar *n_value;

  /**
   * Zone into which the record should be placed.
   */
  gchar *new_zone_option;

  /**
   * Flag indicating if the old record was in the namestore.
   */
  int old_record_in_namestore;

  /**
   * Type of the record.
   */
  uint32_t record_type;

  /**
   * Is this record 'public'?
   */
  gboolean n_public;

  /**
   * Is the expiration time relative?
   */
  gboolean n_is_relative;

  /**
   * Is this record a shadow record?
   */
  gboolean n_is_shadow;

};



/**
 * Run an GNS Edit dialog for an 'A' Record.
 *
 * @param edc editing context to use
 */
void
GNS_edit_dialog_a (struct EditDialogContext *edc);


/**
 * Run an GNS Edit dialog for an 'AAAA' Record.
 *
 * @param cont continuation to call when done
 * @param edc editing context to use
 */
void
GNS_edit_dialog_aaaa (struct EditDialogContext *edc);


/**
 * Run an GNS Edit dialog for an 'CNAME' Record.
 *
 * @param cont continuation to call when done
 * @param edc editing context to use
 */
void
GNS_edit_dialog_cname (struct EditDialogContext *edc);


/**
 * Run an GNS Edit dialog for an 'LEHO' Record.
 *
 * @param cont continuation to call when done
 * @param edc editing context to use
 */
void
GNS_edit_dialog_leho (struct EditDialogContext *edc);


/**
 * Run an GNS Edit dialog for an 'MX' Record.
 *
 * @param cont continuation to call when done
 * @param edc editing context to use
 */
void
GNS_edit_dialog_mx (struct EditDialogContext *edc);


/**
 * Run an GNS Edit dialog for an 'NS' Record.
 *
 * @param cont continuation to call when done
 * @param edc editing context to use
 */
void
GNS_edit_dialog_ns (struct EditDialogContext *edc);


/**
 * Run an GNS Edit dialog for an 'PKEY' Record.
 *
 * @param cont continuation to call when done
 * @param edc editing context to use
 */
void
GNS_edit_dialog_pkey (struct EditDialogContext *edc);


/**
 * Run an GNS Edit dialog for an 'PTR' Record.
 *
 * @param cont continuation to call when done
 * @param edc editing context to use
 */
void
GNS_edit_dialog_ptr (struct EditDialogContext *edc);


/**
 * Run an GNS Edit dialog for an 'SOA' Record.
 *
 * @param cont continuation to call when done
 * @param edc editing context to use
 */
void
GNS_edit_dialog_soa (struct EditDialogContext *edc);


/**
 * Run an GNS Edit dialog for an 'SRV' Record.
 *
 * @param cont continuation to call when done
 * @param edc editing context to use
 */
void
GNS_edit_dialog_srv (struct EditDialogContext *edc);


/**
 * Run an GNS Edit dialog for an 'TXT' Record.
 *
 * @param cont continuation to call when done
 * @param edc editing context to use
 */
void
GNS_edit_dialog_txt (struct EditDialogContext *edc);


/**
 * Run an GNS Edit dialog for an 'VPN' Record.
 *
 * @param cont continuation to call when done
 * @param edc editing context to use
 */
void
GNS_edit_dialog_vpn (struct EditDialogContext *edc);


#endif
