/*
     This file is part of GNUnet
     (C) 2012 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/
/**
 * @file src/setup/gnunet-setup-gns-edit.c
 * @author Christian Grothoff
 * @brief editing dialogs for GNS records
 */
#include "gnunet-setup-gns-edit.h"
#include <gnunet/gnunet_gns_service.h>


/**
 * Columns in the 'zone_liststore'.
 */
enum ZoneListColumn
{
  /**
   * A gchararray with the name of the zone for users.
   */
  ZONELIST_COL_NAME = 0,
  
  /**
   * A gchararray with the name of the configuration file option with 
   * the zone key.
   */
  ZONELIST_COL_OPTION = 1
};


/**
 * Disable 'save' button, dialog state is not acceptable.
 *
 * @param edc dialog to modify
 */
static void
edit_dialog_disable_save (struct EditDialogContext *edc)
{  
  gtk_widget_set_sensitive (GTK_WIDGET (gtk_builder_get_object (edc->builder,
								"edit_dialog_save_button")),
			    FALSE);
}


/**
 * Enable 'save' button, dialog state is acceptable.
 *
 * @param edc dialog to modify
 */
static void
edit_dialog_enable_save (struct EditDialogContext *edc)
{  
  gtk_widget_set_sensitive (GTK_WIDGET (gtk_builder_get_object (edc->builder,
								"edit_dialog_save_button")),
			    TRUE);
}


/**
 * Check that the common elements of the edit dialog are valid;
 * if so, call 'edit_dialog_enable_save', otherwise 'edit_dialog_disable_save'.
 *
 * @param edc edit dialog to check
 */
static void
edit_dialog_check_save (struct EditDialogContext *edc)
{
  GtkEditable *entry;
  const gchar *name;

  /* check name */
  entry = GTK_EDITABLE (gtk_builder_get_object (edc->builder,
						"edit_dialog_name_entry"));
  name = gtk_editable_get_chars (entry, 0, -1);
  if ( (GNUNET_SYSERR == GNUNET_DNSPARSER_check_label (name)) &&
       (0 != strcmp (name, GNUNET_GNS_MASTERZONE_STR)) )
  {
    edit_dialog_disable_save (edc);
    return;
  }
  /* any other checks should go here */
  edit_dialog_enable_save (edc);
}


/**
 * The user has edited the A record value.  Enable/disable 'save'
 * button depending on the validity of the value.
 *
 * @param entry editing widget
 * @param preedit new value
 * @param user_data the 'struct EditDialogContext' of the dialog
 */
void
GNS_edit_dialog_name_entry_changed_cb (GtkEditable *entry,
				       gpointer user_data)
{
  struct EditDialogContext *edc = user_data;

  edc->validator (edc);
}


/**
 * The 'relative' expiration time radiobutton was toggled (on or off).
 *
 * @param button the button
 * @param user_data the 'struct EditDialogContext' of the dialog
 */
void
GNS_edit_dialog_expiration_relative_radiobutton_toggled_cb (GtkToggleButton *button,
							    gpointer user_data)
{
  struct EditDialogContext *edc = user_data;

  if (gtk_toggle_button_get_active (button))  
    gtk_widget_show (GTK_WIDGET (gtk_builder_get_object (edc->builder,
							 "edit_dialog_expiration_relative_combobox")));  
  else
    gtk_widget_hide (GTK_WIDGET (gtk_builder_get_object (edc->builder,
							 "edit_dialog_expiration_relative_combobox")));
}


/**
 * The 'forever' expiration time radiobutton was toggled (on or off).
 *
 * @param button the button
 * @param user_data the 'struct EditDialogContext' of the dialog
 */
void
GNS_edit_dialog_expiration_forever_radiobutton_toggled_cb (GtkToggleButton *button,
							    gpointer user_data)
{
  /* nothing to do */
}


/**
 * The 'absolute' expiration time radiobutton was toggled (on or off).
 *
 * @param button the button
 * @param user_data the 'struct EditDialogContext' of the dialog
 */
void
GNS_edit_dialog_expiration_absolute_radiobutton_toggled_cb (GtkToggleButton *button,
							    gpointer user_data)
{
  struct EditDialogContext *edc = user_data;

  if (gtk_toggle_button_get_active (button))  
  {
    gtk_widget_show (GTK_WIDGET (gtk_builder_get_object (edc->builder,
							 "edit_dialog_expiration_absolute_calendar")));
    gtk_widget_show (GTK_WIDGET (gtk_builder_get_object (edc->builder,
							 "edit_dialog_expiration_absolute_hbox")));
  }
  else
  {
    gtk_widget_hide (GTK_WIDGET (gtk_builder_get_object (edc->builder,
							 "edit_dialog_expiration_absolute_calendar")));
    gtk_widget_hide (GTK_WIDGET (gtk_builder_get_object (edc->builder,
							 "edit_dialog_expiration_absolute_hbox")));
  }
}


/**
 * Initialize widgets of the edit dialog that are the same regardless of
 * the type of the record.
 *
 * @param edc dialog context
 */
static void
edit_dialog_setup_common_elements (struct EditDialogContext *edc)
{
  GtkComboBox *cb;
  GtkTreeModel *tm;
  GtkListStore *ls;
  GtkTreeIter iter;
  gchar *opt;
  struct GNUNET_TIME_Absolute at;
  struct GNUNET_TIME_Relative rt;
  time_t tp;
  struct tm *ymd;
  GtkCalendar *cal;

  if (GNUNET_YES !=
      edc->old_record_in_namestore)
  {
    gtk_widget_hide (GTK_WIDGET (gtk_builder_get_object (edc->builder,
							 "edit_dialog_delete_button")));
    edit_dialog_disable_save (edc);
  }
  gtk_entry_set_text (GTK_ENTRY (gtk_builder_get_object (edc->builder,
							 "edit_dialog_name_entry")),
		      edc->n_new_name);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (gtk_builder_get_object (edc->builder,
									   "edit_dialog_options_public_checkbutton")),
				edc->n_public);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (gtk_builder_get_object (edc->builder,
									   "edit_dialog_options_shadow_checkbutton")),
				edc->n_is_shadow);
  if (GNUNET_TIME_UNIT_FOREVER_REL.rel_value == edc->n_exp_time)
  {
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (gtk_builder_get_object (edc->builder,
									     "edit_dialog_expiration_never_radiobutton")),
				  TRUE);
    gtk_widget_hide (GTK_WIDGET (gtk_builder_get_object (edc->builder,
							 "edit_dialog_expiration_absolute_calendar")));
    gtk_widget_hide (GTK_WIDGET (gtk_builder_get_object (edc->builder,
							 "edit_dialog_expiration_absolute_hbox")));
    gtk_widget_hide (GTK_WIDGET (gtk_builder_get_object (edc->builder,
							 "edit_dialog_expiration_relative_combobox")));
  }
  if ( (edc->n_is_relative) &&
       (GNUNET_TIME_UNIT_FOREVER_REL.rel_value != edc->n_exp_time) )
  {
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (gtk_builder_get_object (edc->builder,
									     "edit_dialog_expiration_relative_radiobutton")),
				  TRUE);
    gtk_widget_hide (GTK_WIDGET (gtk_builder_get_object (edc->builder,
							 "edit_dialog_expiration_absolute_calendar")));
    gtk_widget_hide (GTK_WIDGET (gtk_builder_get_object (edc->builder,
							 "edit_dialog_expiration_absolute_hbox")));
    rt.rel_value = edc->n_exp_time;
  }
  else
  {
    /* select a sane default */
    rt = GNUNET_TIME_UNIT_DAYS;
  }
  cb = GTK_COMBO_BOX (gtk_builder_get_object (edc->builder,
					      "edit_dialog_expiration_relative_combobox"));
  ls = GTK_LIST_STORE (gtk_combo_box_get_model (cb));
  gtk_list_store_insert_with_values (ls, &iter,
				     -1 /* position: append */,
				     0, GNUNET_STRINGS_relative_time_to_string (rt, GNUNET_NO),
				     -1);
  gtk_combo_box_set_active_iter (cb, &iter);    
  if ( (! edc->n_is_relative) &&
       (GNUNET_TIME_UNIT_FOREVER_ABS.abs_value != edc->n_exp_time) )
  {      
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (gtk_builder_get_object (edc->builder,
									     "edit_dialog_expiration_absolute_radiobutton")),
				  TRUE);
    
    gtk_widget_hide (GTK_WIDGET (gtk_builder_get_object (edc->builder,
							 "edit_dialog_expiration_relative_combobox")));
    at.abs_value = edc->n_exp_time;
  }
  else
  {
    /* select a sane default: right now */
    at = GNUNET_TIME_absolute_get ();
  }
  tp = (time_t) (at.abs_value / 1000LL); /* convert to seconds */
  ymd = gmtime (&tp);
  cal = GTK_CALENDAR (gtk_builder_get_object (edc->builder,
					      "edit_dialog_expiration_absolute_calendar"));
  gtk_calendar_select_month (cal,
			     ymd->tm_mon,
			     ymd->tm_year + 1900); 
  gtk_calendar_mark_day (cal,
			 ymd->tm_mday);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (gtk_builder_get_object (edc->builder,
								      "edit_dialog_expiration_absolute_hours_spinbutton")),
			     (double) ymd->tm_hour);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (gtk_builder_get_object (edc->builder,
								      "edit_dialog_expiration_absolute_minutes_spinbutton")),
			     (double) ymd->tm_min);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (gtk_builder_get_object (edc->builder,
								      "edit_dialog_expiration_absolute_seconds_spinbutton")),
			     (double) ymd->tm_sec);

  /* set source zone! */
  cb = GTK_COMBO_BOX (gtk_builder_get_object (edc->builder,
					      "edit_dialog_zone_combobox"));
  tm = gtk_combo_box_get_model (cb);
  opt = NULL;
  if (gtk_tree_model_get_iter_first (tm, &iter))
  {
    do
    {
      gtk_tree_model_get (tm, &iter,
			  ZONELIST_COL_OPTION, &opt,
			  -1);
      if ( (NULL != opt) &&
	   (0 == strcasecmp (opt,
			     edc->new_zone_option)) )
	break;
      g_free (opt);
      opt = NULL;
    } while (gtk_tree_model_iter_next (tm, &iter));    
  }
  if (NULL != opt)
  {
    gtk_combo_box_set_active_iter (cb, &iter);
    g_free (opt);
  }
}


/**
 * Perform the reverse of the 'edit_dialog_setup_common_elements' function,
 * that is, extract the values from the (common) widgets and store the
 * values in 'edc'. 
 *
 * @param edc edit dialog to extract data from
 */
static void
edit_dialog_putes_common_elements (struct EditDialogContext *edc)
{
  GtkEntry *entry;
  const char *rt_s;
  struct GNUNET_TIME_Relative rt;
  GtkComboBox *cb;
  GtkTreeModel *tm;
  GtkTreeIter iter; 
  gchar *opt;

  /* record name */
  entry = GTK_ENTRY (gtk_builder_get_object (edc->builder,
					     "edit_dialog_name_entry"));
  g_free (edc->n_new_name);
  edc->n_new_name = g_strdup (gtk_entry_get_text (entry));

  /* is public flag */
  edc->n_public = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gtk_builder_get_object (edc->builder,
											   "edit_dialog_options_public_checkbutton")));
  /* is shadow flag */
  edc->n_is_shadow = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gtk_builder_get_object (edc->builder,
											      "edit_dialog_options_shadow_checkbutton")));
  
  /* 'forever' expiration time */
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gtk_builder_get_object (edc->builder,
									       "edit_dialog_expiration_never_radiobutton"))))
  {
    edc->n_exp_time = GNUNET_TIME_UNIT_FOREVER_REL.rel_value;
    edc->n_is_relative = TRUE; /* doesn't matter, but make sure it is well-defined anyway */
  }
  
  /* 'relative' expiration time */
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gtk_builder_get_object (edc->builder,
									       "edit_dialog_expiration_relative_radiobutton"))))
  {
    cb = GTK_COMBO_BOX (gtk_builder_get_object (edc->builder,
						"edit_dialog_expiration_relative_combobox"));
    tm = gtk_combo_box_get_model (cb);
    if (! gtk_combo_box_get_active_iter (cb, &iter))
    {
      GNUNET_break (0);
      return;
    }
    gtk_tree_model_get (tm, &iter, 
			0, &rt_s,
			-1);
    GNUNET_break (GNUNET_YES ==
		  GNUNET_STRINGS_fancy_time_to_relative (rt_s,
							 &rt));
    edc->n_exp_time = rt.rel_value;
    edc->n_is_relative = TRUE;
  }

  /* 'absolute' expiration time */
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gtk_builder_get_object (edc->builder,
									       "edit_dialog_expiration_absolute_radiobutton"))))
  {
    guint year;
    guint month;
    guint day;
    guint hour;
    guint minute;
    guint second;
    char fancydate[128];
    struct GNUNET_TIME_Absolute atime;
    
    gtk_calendar_get_date (GTK_CALENDAR (gtk_builder_get_object (edc->builder,
								 "edit_dialog_expiration_absolute_calendar")),
			   &year, &month, &day);
    hour = gtk_spin_button_get_value (GTK_SPIN_BUTTON (gtk_builder_get_object (edc->builder,
									       "edit_dialog_expiration_absolute_hours_spinbutton")));
    minute = gtk_spin_button_get_value (GTK_SPIN_BUTTON (gtk_builder_get_object (edc->builder,
									  "edit_dialog_expiration_absolute_minutes_spinbutton")));
    second = gtk_spin_button_get_value (GTK_SPIN_BUTTON (gtk_builder_get_object (edc->builder,
										 "edit_dialog_expiration_absolute_seconds_spinbutton")));
    GNUNET_snprintf (fancydate,
		     sizeof (fancydate),
		     "%u-%u-%u %u:%u:%u",
		     (unsigned int) year, 
		     (unsigned int) month + 1, 
		     (unsigned int) day,
		     (unsigned int) hour,
		     (unsigned int) minute,
		     (unsigned int) second);
    GNUNET_break (GNUNET_OK ==
		  GNUNET_STRINGS_fancy_time_to_absolute (fancydate,
							 &atime));
    edc->n_exp_time = atime.abs_value;
    edc->n_is_relative = FALSE;
  }

  /* extract target zone! */
  cb = GTK_COMBO_BOX (gtk_builder_get_object (edc->builder,
					      "edit_dialog_zone_combobox"));
  tm = gtk_combo_box_get_model (cb);
  if (! gtk_combo_box_get_active_iter (cb, &iter))
  {
    GNUNET_break (0);
  }
  else
  {
    gtk_tree_model_get (tm, &iter,
			ZONELIST_COL_OPTION, &opt,
			-1);
    if (NULL == opt)
      GNUNET_break (0);
    else
    {
      g_free (edc->new_zone_option);
      edc->new_zone_option = g_strdup (opt);
      g_free (opt);
    }
  }
}


/**
 * Run the edit dialog.  Performs all of the common initialization
 * steps to run an edit dialog for records.
 *
 * @param edc editing context
 */
static void
run_edit_dialog (struct EditDialogContext *edc)
{
  edit_dialog_setup_common_elements (edc);
  gtk_dialog_set_default_response (edc->dialog,
				   GTK_RESPONSE_OK);
  gtk_window_present (GTK_WINDOW (edc->dialog));
}


/* ************************ A records *********************** */

/**
 * Check validity of the value in the edit dialog for A-records.
 * Then call the shared validity check if the result is OK.
 *
 * @param edc edit dialog context
 */
static void
edit_dialog_a_validity_check (struct EditDialogContext *edc)
{
  GtkEditable *entry;
  const gchar *preedit;
  struct in_addr v4;

  entry = GTK_EDITABLE (gtk_builder_get_object (edc->builder,
						"edit_dialog_a_entry")),
  preedit = gtk_editable_get_chars (entry, 0, -1);
  if ( (NULL == preedit) ||
       (1 != inet_pton (AF_INET, preedit, &v4)) )
  {
    edit_dialog_disable_save (edc);
    return;
  }
  edit_dialog_check_save (edc);
}


/**
 * Editing dialog was closed, get the data and call the
 * continuation.
 *
 * @param dialog editing dialog
 * @param user_data the 'struct EditDialogContext'
 */
void
GNS_edit_a_dialog_response_cb (GtkDialog *dialog,
			       gint response_id,
			       gpointer user_data)
{
  struct EditDialogContext *edc = user_data;
  GtkEntry *entry;
  const gchar *value;
    
  if (GTK_RESPONSE_OK == response_id)
  {
    edit_dialog_putes_common_elements (edc);
    entry = GTK_ENTRY (gtk_builder_get_object (edc->builder,
					       "edit_dialog_a_entry"));
    value = gtk_entry_get_text (entry);
    g_free (edc->n_value);
    edc->n_value = g_strdup (value);
  }
  gtk_widget_destroy (GTK_WIDGET (edc->dialog));
  g_object_unref (edc->builder);
  edc->builder = NULL;
  edc->cont (edc, response_id);
}


/**
 * The user has edited the A record value.  Enable/disable 'save'
 * button depending on the validity of the value.
 *
 * @param entry editing widget
 * @param preedit new value
 * @param user_data the 'struct EditDialogContext' of the dialog
 */
void
GNS_edit_dialog_a_entry_changed_cb (GtkEditable *entry,
				    gpointer user_data)
{
  struct EditDialogContext *edc = user_data;

  edit_dialog_a_validity_check (edc);
}


/**
 * Run an GNS Edit dialog for an 'A' Record.
 *
 * @param cont continuation to call when done
 * @param edc editing context to use
 */
void
GNS_edit_dialog_a (struct EditDialogContext *edc)
{
  edc->builder = GNUNET_GTK_get_new_builder ("gnunet_setup_gns_edit_a.glade",
					     edc);
  if (NULL == edc->builder)
  {
    GNUNET_break (0);
    edc->cont (edc, GTK_RESPONSE_CANCEL);  /* treat as 'cancel' */
    return;    
  }
  if (GNUNET_YES ==
      edc->old_record_in_namestore)
  {
    /* set A record */
    gtk_entry_set_text (GTK_ENTRY (gtk_builder_get_object (edc->builder,
							   "edit_dialog_a_entry")),
			edc->n_value);
  }
  edc->validator = &edit_dialog_a_validity_check;
  edc->dialog = GTK_DIALOG (gtk_builder_get_object (edc->builder,
						    "edit_a_dialog"));
  run_edit_dialog (edc);
}


/* ************************ AAAA records *********************** */

/**
 * Check validity of the value in the edit dialog for AAAA-records.
 * Then call the shared validity check if the result is OK.
 *
 * @param edc edit dialog context
 */
static void
edit_dialog_aaaa_validity_check (struct EditDialogContext *edc)
{
  GtkEditable *entry;
  const gchar *preedit;
  struct in6_addr v6;

  entry = GTK_EDITABLE (gtk_builder_get_object (edc->builder,
						"edit_dialog_aaaa_entry")),
  preedit = gtk_editable_get_chars (entry, 0, -1);
  if ( (NULL == preedit) ||
       (1 != inet_pton (AF_INET6, preedit, &v6)) )
  {
    edit_dialog_disable_save (edc);
    return;
  }
  edit_dialog_check_save (edc);
}


/**
 * Editing dialog was closed, get the data and call the
 * continuation.
 *
 * @param dialog editing dialog
 * @param user_data the 'struct EditDialogContext'
 */
void
GNS_edit_aaaa_dialog_response_cb (GtkDialog *dialog,
				  gint response_id,
				  gpointer user_data)
{
  struct EditDialogContext *edc = user_data;
  GtkEntry *entry;
  const gchar *value;
    
  if (GTK_RESPONSE_OK == response_id)
  {
    edit_dialog_putes_common_elements (edc);
    entry = GTK_ENTRY (gtk_builder_get_object (edc->builder,
					       "edit_dialog_aaaa_entry"));
    value = gtk_entry_get_text (entry);
    g_free (edc->n_value);
    edc->n_value = g_strdup (value);
  }
  gtk_widget_destroy (GTK_WIDGET (edc->dialog));
  g_object_unref (edc->builder);
  edc->builder = NULL;
  edc->cont (edc, response_id);
}


/**
 * The user has edited the AAAA record value.  Enable/disable 'save'
 * button depending on the validity of the value.
 *
 * @param entry editing widget
 * @param preedit new value
 * @param user_data the 'struct EditDialogContext' of the dialog
 */
void
GNS_edit_dialog_aaaa_entry_changed_cb (GtkEditable *entry,
				       gpointer user_data)
{
  struct EditDialogContext *edc = user_data;

  edit_dialog_aaaa_validity_check (edc);
}


/**
 * Run an GNS Edit dialog for an 'AAAA' Record.
 *
 * @param cont continuation to call when done
 * @param edc editing context to use
 */
void
GNS_edit_dialog_aaaa (struct EditDialogContext *edc)
{
  edc->builder = GNUNET_GTK_get_new_builder ("gnunet_setup_gns_edit_aaaa.glade",
					     edc);
  if (NULL == edc->builder)
  {
    GNUNET_break (0);
    edc->cont (edc, GTK_RESPONSE_CANCEL);  /* treat as 'cancel' */
    return;    
  }
  if (GNUNET_YES ==
      edc->old_record_in_namestore)
  {
    /* set AAAA record */
    gtk_entry_set_text (GTK_ENTRY (gtk_builder_get_object (edc->builder,
							   "edit_dialog_aaaa_entry")),
			edc->n_value);
  }
  edc->validator = &edit_dialog_aaaa_validity_check;
  edc->dialog = GTK_DIALOG (gtk_builder_get_object (edc->builder,
						    "edit_aaaa_dialog"));
  run_edit_dialog (edc);
}


/* ************************ CNAME records *********************** */

/**
 * Check validity of the value in the edit dialog for AAAA-records.
 * Then call the shared validity check if the result is OK.
 *
 * @param edc edit dialog context
 */
static void
edit_dialog_cname_validity_check (struct EditDialogContext *edc)
{
  GtkEditable *entry;
  const gchar *preedit;

  entry = GTK_EDITABLE (gtk_builder_get_object (edc->builder,
						"edit_dialog_cname_entry")),
  preedit = gtk_editable_get_chars (entry, 0, -1);
  if ( (NULL == preedit) ||
       (GNUNET_OK != GNUNET_DNSPARSER_check_name (preedit)) )
  {
    edit_dialog_disable_save (edc);
    return;
  }
  edit_dialog_check_save (edc);
}


/**
 * Editing dialog was closed, get the data and call the
 * continuation.
 *
 * @param dialog editing dialog
 * @param user_data the 'struct EditDialogContext'
 */
void
GNS_edit_cname_dialog_response_cb (GtkDialog *dialog,
				   gint response_id,
				   gpointer user_data)
{
  struct EditDialogContext *edc = user_data;
  GtkEntry *entry;
  const gchar *value;
    
  if (GTK_RESPONSE_OK == response_id)
  {
    edit_dialog_putes_common_elements (edc);
    entry = GTK_ENTRY (gtk_builder_get_object (edc->builder,
					       "edit_dialog_cname_entry"));
    value = gtk_entry_get_text (entry);
    g_free (edc->n_value);
    edc->n_value = g_strdup (value);
  }
  gtk_widget_destroy (GTK_WIDGET (edc->dialog));
  g_object_unref (edc->builder);
  edc->builder = NULL;
  edc->cont (edc, response_id);
}


/**
 * The user has edited the CNAME record value.  Enable/disable 'save'
 * button depending on the validity of the value.
 *
 * @param entry editing widget
 * @param preedit new value
 * @param user_data the 'struct EditDialogContext' of the dialog
 */
void
GNS_edit_dialog_cname_entry_changed_cb (GtkEditable *entry,
				       gpointer user_data)
{
  struct EditDialogContext *edc = user_data;

  edit_dialog_cname_validity_check (edc);
}


/**
 * Run an GNS Edit dialog for an 'CNAME' Record.
 *
 * @param cont continuation to call when done
 * @param edc editing context to use
 */
void
GNS_edit_dialog_cname (struct EditDialogContext *edc)
{
  edc->builder = GNUNET_GTK_get_new_builder ("gnunet_setup_gns_edit_cname.glade",
					     edc);
  if (NULL == edc->builder)
  {
    GNUNET_break (0);
    edc->cont (edc, GTK_RESPONSE_CANCEL);  /* treat as 'cancel' */
    return;    
  }
  if (GNUNET_YES ==
      edc->old_record_in_namestore)
  {
    /* set CNAME record */
    gtk_entry_set_text (GTK_ENTRY (gtk_builder_get_object (edc->builder,
							   "edit_dialog_cname_entry")),
			edc->n_value);
  }
  edc->validator = &edit_dialog_cname_validity_check;
  edc->dialog = GTK_DIALOG (gtk_builder_get_object (edc->builder,
						    "edit_cname_dialog"));
  run_edit_dialog (edc);
}


/* ************************ LEHO records *********************** */

/**
 * Check validity of the value in the edit dialog for LEHO-records.
 * Then call the shared validity check if the result is OK.
 *
 * @param edc edit dialog context
 */
static void
edit_dialog_leho_validity_check (struct EditDialogContext *edc)
{
  GtkEditable *entry;
  const gchar *preedit;

  entry = GTK_EDITABLE (gtk_builder_get_object (edc->builder,
						"edit_dialog_leho_entry")),
  preedit = gtk_editable_get_chars (entry, 0, -1);
  if ( (NULL == preedit) ||
       (GNUNET_OK != GNUNET_DNSPARSER_check_name (preedit)) )
  {
    edit_dialog_disable_save (edc);
    return;
  }
  edit_dialog_check_save (edc);
}


/**
 * Editing dialog was closed, get the data and call the
 * continuation.
 *
 * @param dialog editing dialog
 * @param user_data the 'struct EditDialogContext'
 */
void
GNS_edit_leho_dialog_response_cb (GtkDialog *dialog,
				   gint response_id,
				   gpointer user_data)
{
  struct EditDialogContext *edc = user_data;
  GtkEntry *entry;
  const gchar *value;
    
  if (GTK_RESPONSE_OK == response_id)
  {
    edit_dialog_putes_common_elements (edc);
    entry = GTK_ENTRY (gtk_builder_get_object (edc->builder,
					       "edit_dialog_leho_entry"));
    value = gtk_entry_get_text (entry);
    g_free (edc->n_value);
    edc->n_value = g_strdup (value);
  }
  gtk_widget_destroy (GTK_WIDGET (edc->dialog));
  g_object_unref (edc->builder);
  edc->builder = NULL;
  edc->cont (edc, response_id);
}


/**
 * The user has edited the LEHO record value.  Enable/disable 'save'
 * button depending on the validity of the value.
 *
 * @param entry editing widget
 * @param preedit new value
 * @param user_data the 'struct EditDialogContext' of the dialog
 */
void
GNS_edit_dialog_leho_entry_changed_cb (GtkEditable *entry,
				       gpointer user_data)
{
  struct EditDialogContext *edc = user_data;

  edit_dialog_leho_validity_check (edc);
}


/**
 * Run an GNS Edit dialog for an 'LEHO' Record.
 *
 * @param cont continuation to call when done
 * @param edc editing context to use
 */
void
GNS_edit_dialog_leho (struct EditDialogContext *edc)
{
  edc->builder = GNUNET_GTK_get_new_builder ("gnunet_setup_gns_edit_leho.glade",
					     edc);
  if (NULL == edc->builder)
  {
    GNUNET_break (0);
    edc->cont (edc, GTK_RESPONSE_CANCEL);  /* treat as 'cancel' */
    return;    
  }
  if (GNUNET_YES ==
      edc->old_record_in_namestore)
  {
    /* set LEHO record */
    gtk_entry_set_text (GTK_ENTRY (gtk_builder_get_object (edc->builder,
							   "edit_dialog_leho_entry")),
			edc->n_value);
  }
  edc->validator = &edit_dialog_leho_validity_check;
  edc->dialog = GTK_DIALOG (gtk_builder_get_object (edc->builder,
						    "edit_leho_dialog"));
  run_edit_dialog (edc);
}


/* ************************ MX records *********************** */

/**
 * Check validity of the value in the edit dialog for MX-records.
 * Then call the shared validity check if the result is OK.
 *
 * @param edc edit dialog context
 */
static void
edit_dialog_mx_validity_check (struct EditDialogContext *edc)
{
  GtkEditable *entry;
  const gchar *preedit;

  entry = GTK_EDITABLE (gtk_builder_get_object (edc->builder,
						"edit_dialog_mx_entry")),
  preedit = gtk_editable_get_chars (entry, 0, -1);
  if ( (NULL == preedit) ||
       (GNUNET_OK != GNUNET_DNSPARSER_check_name (preedit)) )
  {
    edit_dialog_disable_save (edc);
    return;
  }
  edit_dialog_check_save (edc);
}


/**
 * Editing dialog was closed, get the data and call the
 * continuation.
 *
 * @param dialog editing dialog
 * @param user_data the 'struct EditDialogContext'
 */
void
GNS_edit_mx_dialog_response_cb (GtkDialog *dialog,
				   gint response_id,
				   gpointer user_data)
{
  struct EditDialogContext *edc = user_data;
  GtkEntry *entry;
  const gchar *value;
  char *result;
  unsigned int distance;
    
  if (GTK_RESPONSE_OK == response_id)
  {
    edit_dialog_putes_common_elements (edc);
    entry = GTK_ENTRY (gtk_builder_get_object (edc->builder,
					       "edit_dialog_mx_entry"));
    value = gtk_entry_get_text (entry);
    distance = gtk_spin_button_get_value (GTK_SPIN_BUTTON (gtk_builder_get_object (edc->builder,
										   "edit_dialog_mx_distance_spinbutton")));
    g_free (edc->n_value);
    GNUNET_asprintf (&result,
		     "%hu,%s", 
		     distance, 
		     value);
    edc->n_value = g_strdup (result);
    GNUNET_free (result);
  }
  gtk_widget_destroy (GTK_WIDGET (edc->dialog));
  g_object_unref (edc->builder);
  edc->builder = NULL;
  edc->cont (edc, response_id);
}


/**
 * The user has edited the MX record value.  Enable/disable 'save'
 * button depending on the validity of the value.
 *
 * @param entry editing widget
 * @param preedit new value
 * @param user_data the 'struct EditDialogContext' of the dialog
 */
void
GNS_edit_dialog_mx_entry_changed_cb (GtkEditable *entry,
				       gpointer user_data)
{
  struct EditDialogContext *edc = user_data;

  edit_dialog_mx_validity_check (edc);
}


/**
 * Run an GNS Edit dialog for an 'MX' Record.
 *
 * @param cont continuation to call when done
 * @param edc editing context to use
 */
void
GNS_edit_dialog_mx (struct EditDialogContext *edc)
{
  uint16_t mx_pref;
  char result[253 + 1];

  edc->builder = GNUNET_GTK_get_new_builder ("gnunet_setup_gns_edit_mx.glade",
					     edc);
  if (NULL == edc->builder)
  {
    GNUNET_break (0);
    edc->cont (edc, GTK_RESPONSE_CANCEL);  /* treat as 'cancel' */
    return;    
  }
  if (GNUNET_YES ==
      edc->old_record_in_namestore)
  {
    if (2 != SSCANF(edc->n_value, 
		    "%hu,%253s", &mx_pref, result))
    {
      GNUNET_break (0);
      GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                  _("Unable to parse MX record `%s'\n"),
		  edc->n_value);
    }
    else
    {
      gtk_entry_set_text (GTK_ENTRY (gtk_builder_get_object (edc->builder,
							     "edit_dialog_mx_entry")),
			  result);
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (gtk_builder_get_object (edc->builder,
									  "edit_dialog_mx_distance_spinbutton")),
				 mx_pref);
    }
  }
  edc->validator = &edit_dialog_mx_validity_check;
  edc->dialog = GTK_DIALOG (gtk_builder_get_object (edc->builder,
						    "edit_mx_dialog"));
  run_edit_dialog (edc);
}


/* ************************ NS records *********************** */

/**
 * Check validity of the value in the edit dialog for NS-records.
 * Then call the shared validity check if the result is OK.
 *
 * @param edc edit dialog context
 */
static void
edit_dialog_ns_validity_check (struct EditDialogContext *edc)
{
  GtkEditable *entry;
  const gchar *preedit;

  entry = GTK_EDITABLE (gtk_builder_get_object (edc->builder,
						"edit_dialog_ns_entry")),
  preedit = gtk_editable_get_chars (entry, 0, -1);
  if ( (NULL == preedit) ||
       (GNUNET_OK != GNUNET_DNSPARSER_check_name (preedit)) )
  {
    edit_dialog_disable_save (edc);
    return;
  }
  edit_dialog_check_save (edc);
}


/**
 * Editing dialog was closed, get the data and call the
 * continuation.
 *
 * @param dialog editing dialog
 * @param user_data the 'struct EditDialogContext'
 */
void
GNS_edit_ns_dialog_response_cb (GtkDialog *dialog,
				   gint response_id,
				   gpointer user_data)
{
  struct EditDialogContext *edc = user_data;
  GtkEntry *entry;
  const gchar *value;
    
  if (GTK_RESPONSE_OK == response_id)
  {
    edit_dialog_putes_common_elements (edc);
    entry = GTK_ENTRY (gtk_builder_get_object (edc->builder,
					       "edit_dialog_ns_entry"));
    value = gtk_entry_get_text (entry);
    g_free (edc->n_value);
    edc->n_value = g_strdup (value);
  }
  gtk_widget_destroy (GTK_WIDGET (edc->dialog));
  g_object_unref (edc->builder);
  edc->builder = NULL;
  edc->cont (edc, response_id);
}


/**
 * The user has edited the NS record value.  Enable/disable 'save'
 * button depending on the validity of the value.
 *
 * @param entry editing widget
 * @param preedit new value
 * @param user_data the 'struct EditDialogContext' of the dialog
 */
void
GNS_edit_dialog_ns_entry_changed_cb (GtkEditable *entry,
				       gpointer user_data)
{
  struct EditDialogContext *edc = user_data;

  edit_dialog_ns_validity_check (edc);
}


/**
 * Run an GNS Edit dialog for an 'NS' Record.
 *
 * @param cont continuation to call when done
 * @param edc editing context to use
 */
void
GNS_edit_dialog_ns (struct EditDialogContext *edc)
{
  edc->builder = GNUNET_GTK_get_new_builder ("gnunet_setup_gns_edit_ns.glade",
					     edc);
  if (NULL == edc->builder)
  {
    GNUNET_break (0);
    edc->cont (edc, GTK_RESPONSE_CANCEL);  /* treat as 'cancel' */
    return;    
  }
  if (GNUNET_YES ==
      edc->old_record_in_namestore)
  {
    /* set NS record */
    gtk_entry_set_text (GTK_ENTRY (gtk_builder_get_object (edc->builder,
							   "edit_dialog_ns_entry")),
			edc->n_value);
  }
  edc->validator = &edit_dialog_ns_validity_check;
  edc->dialog = GTK_DIALOG (gtk_builder_get_object (edc->builder,
						    "edit_ns_dialog"));
  run_edit_dialog (edc);
}


/* ************************ PKEY records *********************** */

/**
 * Check validity of the value in the edit dialog for PKEY-records.
 * Then call the shared validity check if the result is OK.
 *
 * @param edc edit dialog context
 */
static void
edit_dialog_pkey_validity_check (struct EditDialogContext *edc)
{
  GtkEditable *entry;
  const gchar *preedit;
  struct GNUNET_HashCode hc;

  entry = GTK_EDITABLE (gtk_builder_get_object (edc->builder,
						"edit_dialog_pkey_entry")),
  preedit = gtk_editable_get_chars (entry, 0, -1);
  if ( (NULL == preedit) ||
       (GNUNET_OK != 
	GNUNET_CRYPTO_hash_from_string (preedit, &hc)) )
  {
    edit_dialog_disable_save (edc);
    return;
  }
  edit_dialog_check_save (edc);
}


/**
 * Editing dialog was closed, get the data and call the
 * continuation.
 *
 * @param dialog editing dialog
 * @param user_data the 'struct EditDialogContext'
 */
void
GNS_edit_pkey_dialog_response_cb (GtkDialog *dialog,
				   gint response_id,
				   gpointer user_data)
{
  struct EditDialogContext *edc = user_data;
  GtkEntry *entry;
  const gchar *value;
    
  if (GTK_RESPONSE_OK == response_id)
  {
    edit_dialog_putes_common_elements (edc);
    entry = GTK_ENTRY (gtk_builder_get_object (edc->builder,
					       "edit_dialog_pkey_entry"));
    value = gtk_entry_get_text (entry);
    g_free (edc->n_value);
    edc->n_value = g_strdup (value);
  }
  gtk_widget_destroy (GTK_WIDGET (edc->dialog));
  g_object_unref (edc->builder);
  edc->builder = NULL;
  edc->cont (edc, response_id);
}


/**
 * The user has edited the PKEY record value.  Enable/disable 'save'
 * button depending on the validity of the value.
 *
 * @param entry editing widget
 * @param preedit new value
 * @param user_data the 'struct EditDialogContext' of the dialog
 */
void
GNS_edit_dialog_pkey_entry_changed_cb (GtkEditable *entry,
				       gpointer user_data)
{
  struct EditDialogContext *edc = user_data;

  edit_dialog_pkey_validity_check (edc);
}


/**
 * Run an GNS Edit dialog for an 'PKEY' Record.
 *
 * @param cont continuation to call when done
 * @param edc editing context to use
 */
void
GNS_edit_dialog_pkey (struct EditDialogContext *edc)
{
  edc->builder = GNUNET_GTK_get_new_builder ("gnunet_setup_gns_edit_pkey.glade",
					     edc);
  if (NULL == edc->builder)
  {
    GNUNET_break (0);
    edc->cont (edc, GTK_RESPONSE_CANCEL);  /* treat as 'cancel' */
    return;    
  }
  if (GNUNET_YES ==
      edc->old_record_in_namestore)
  {
    /* set PKEY record */
    gtk_entry_set_text (GTK_ENTRY (gtk_builder_get_object (edc->builder,
							   "edit_dialog_pkey_entry")),
			edc->n_value);
  }
  edc->validator = &edit_dialog_pkey_validity_check;
  edc->dialog = GTK_DIALOG (gtk_builder_get_object (edc->builder,
						    "edit_pkey_dialog"));
  run_edit_dialog (edc);
}


/* ************************ PTR records *********************** */

/**
 * Check validity of the value in the edit dialog for PTR-records.
 * Then call the shared validity check if the result is OK.
 *
 * @param edc edit dialog context
 */
static void
edit_dialog_ptr_validity_check (struct EditDialogContext *edc)
{
  GtkEditable *entry;
  const gchar *preedit;

  entry = GTK_EDITABLE (gtk_builder_get_object (edc->builder,
						"edit_dialog_ptr_entry")),
  preedit = gtk_editable_get_chars (entry, 0, -1);
  if ( (NULL == preedit) ||
       (GNUNET_OK != GNUNET_DNSPARSER_check_name (preedit)) )
  {
    edit_dialog_disable_save (edc);
    return;
  }
  edit_dialog_check_save (edc);
}


/**
 * Editing dialog was closed, get the data and call the
 * continuation.
 *
 * @param dialog editing dialog
 * @param user_data the 'struct EditDialogContext'
 */
void
GNS_edit_ptr_dialog_response_cb (GtkDialog *dialog,
				   gint response_id,
				   gpointer user_data)
{
  struct EditDialogContext *edc = user_data;
  GtkEntry *entry;
  const gchar *value;
    
  if (GTK_RESPONSE_OK == response_id)
  {
    edit_dialog_putes_common_elements (edc);
    entry = GTK_ENTRY (gtk_builder_get_object (edc->builder,
					       "edit_dialog_ptr_entry"));
    value = gtk_entry_get_text (entry);
    g_free (edc->n_value);
    edc->n_value = g_strdup (value);
  }
  gtk_widget_destroy (GTK_WIDGET (edc->dialog));
  g_object_unref (edc->builder);
  edc->builder = NULL;
  edc->cont (edc, response_id);
}


/**
 * The user has edited the PTR record value.  Enable/disable 'save'
 * button depending on the validity of the value.
 *
 * @param entry editing widget
 * @param preedit new value
 * @param user_data the 'struct EditDialogContext' of the dialog
 */
void
GNS_edit_dialog_ptr_entry_changed_cb (GtkEditable *entry,
				       gpointer user_data)
{
  struct EditDialogContext *edc = user_data;

  edit_dialog_ptr_validity_check (edc);
}


/**
 * Run an GNS Edit dialog for an 'PTR' Record.
 *
 * @param cont continuation to call when done
 * @param edc editing context to use
 */
void
GNS_edit_dialog_ptr (struct EditDialogContext *edc)
{
  edc->builder = GNUNET_GTK_get_new_builder ("gnunet_setup_gns_edit_ptr.glade",
					     edc);
  if (NULL == edc->builder)
  {
    GNUNET_break (0);
    edc->cont (edc, GTK_RESPONSE_CANCEL);  /* treat as 'cancel' */
    return;    
  }
  if (GNUNET_YES ==
      edc->old_record_in_namestore)
  {
    /* set PTR record */
    gtk_entry_set_text (GTK_ENTRY (gtk_builder_get_object (edc->builder,
							   "edit_dialog_ptr_entry")),
			edc->n_value);
  }
  edc->validator = &edit_dialog_ptr_validity_check;
  edc->dialog = GTK_DIALOG (gtk_builder_get_object (edc->builder,
						    "edit_ptr_dialog"));
  run_edit_dialog (edc);
}


/* ************************ SOA records *********************** */

/**
 * Check validity of the value in the edit dialog for SOA-records.
 * Then call the shared validity check if the result is OK.
 *
 * @param edc edit dialog context
 */
static void
edit_dialog_soa_validity_check (struct EditDialogContext *edc)
{
  GtkEditable *entry;
  const gchar *preedit;

  entry = GTK_EDITABLE (gtk_builder_get_object (edc->builder,
						"edit_dialog_soa_source_host_entry")),
  preedit = gtk_editable_get_chars (entry, 0, -1);
  if ( (NULL == preedit) ||
       (GNUNET_OK != GNUNET_DNSPARSER_check_name (preedit)) )
  {
    edit_dialog_disable_save (edc);
    return;
  }
  /* check for '@' in the e-mail --- required format uses "." instead! */
  entry = GTK_EDITABLE (gtk_builder_get_object (edc->builder,
						"edit_dialog_soa_contact_email_entry")),
  preedit = gtk_editable_get_chars (entry, 0, -1);
  if ( (NULL == preedit) ||
       (NULL != strstr (preedit, "@")) ||
       (GNUNET_OK != GNUNET_DNSPARSER_check_name (preedit)) )
  {
    /* E-mail is specified in the RFC also as a 'domain-name', hence
       we check above that it follows those conventions as well; the '@'
       is a common mistake, and while it should be illegal despite IDN,
       it feels better to check explicitly. */
    edit_dialog_disable_save (edc);
    return;
  }
  edit_dialog_check_save (edc);
}


/**
 * Editing dialog was closed, get the data and call the
 * continuation.
 *
 * @param dialog editing dialog
 * @param user_data the 'struct EditDialogContext'
 */
void
GNS_edit_soa_dialog_response_cb (GtkDialog *dialog,
				   gint response_id,
				   gpointer user_data)
{
  struct EditDialogContext *edc = user_data;
  GtkEntry *entry;
  const gchar *source_host;
  const gchar *contact_email;
  unsigned int soa_serial;
  unsigned int soa_refresh;
  unsigned int soa_retry;
  unsigned int soa_expire;
  unsigned int soa_min;
  char *result;
    
  if (GTK_RESPONSE_OK == response_id)
  {
    edit_dialog_putes_common_elements (edc);
    entry = GTK_ENTRY (gtk_builder_get_object (edc->builder,
					       "edit_dialog_soa_source_host_entry"));
    source_host = gtk_entry_get_text (entry);
    entry = GTK_ENTRY (gtk_builder_get_object (edc->builder,
					       "edit_dialog_soa_contact_email_entry"));
    contact_email = gtk_entry_get_text (entry);
    soa_serial = gtk_spin_button_get_value (GTK_SPIN_BUTTON (gtk_builder_get_object (edc->builder,
										     "edit_dialog_soa_serial_number_spinbutton")));
    soa_refresh = gtk_spin_button_get_value (GTK_SPIN_BUTTON (gtk_builder_get_object (edc->builder,
										      "edit_dialog_soa_refresh_time_spinbutton")));
    soa_retry = gtk_spin_button_get_value (GTK_SPIN_BUTTON (gtk_builder_get_object (edc->builder,
										    "edit_dialog_soa_retry_time_spinbutton")));
    soa_expire = gtk_spin_button_get_value (GTK_SPIN_BUTTON (gtk_builder_get_object (edc->builder,
										     "edit_dialog_soa_expire_time_spinbutton")));
    soa_min = gtk_spin_button_get_value (GTK_SPIN_BUTTON (gtk_builder_get_object (edc->builder,
										  "edit_dialog_soa_minimum_ttl_spinbutton")));
    GNUNET_asprintf (&result, 
		     "rname=%s mname=%s %lu,%lu,%lu,%lu,%lu",
		     source_host, contact_email,
		     soa_serial, 
		     soa_refresh,
		     soa_retry, 
		     soa_expire,
		     soa_min);
    edc->n_value = g_strdup (result);
    GNUNET_free (result);
  }
  gtk_widget_destroy (GTK_WIDGET (edc->dialog));
  g_object_unref (edc->builder);
  edc->builder = NULL;
  edc->cont (edc, response_id);
}


/**
 * The user has edited the SOA record value.  Enable/disable 'save'
 * button depending on the validity of the value.
 *
 * @param entry editing widget
 * @param preedit new value
 * @param user_data the 'struct EditDialogContext' of the dialog
 */
void
GNS_edit_dialog_soa_contact_email_entry_changed_cb (GtkEditable *entry,
						    gpointer user_data)
{
  struct EditDialogContext *edc = user_data;

  edit_dialog_soa_validity_check (edc);
}


/**
 * The user has edited the SOA record value.  Enable/disable 'save'
 * button depending on the validity of the value.
 *
 * @param entry editing widget
 * @param preedit new value
 * @param user_data the 'struct EditDialogContext' of the dialog
 */
void
GNS_edit_dialog_soa_source_host_entry_changed_cb (GtkEditable *entry,
						  gpointer user_data)
{
  struct EditDialogContext *edc = user_data;

  edit_dialog_soa_validity_check (edc);
}


/**
 * Run an GNS Edit dialog for an 'SOA' Record.
 *
 * @param cont continuation to call when done
 * @param edc editing context to use
 */
void
GNS_edit_dialog_soa (struct EditDialogContext *edc)
{
  char soa_rname[253 + 1];
  char soa_mname[253 + 1];
  unsigned int soa_serial;
  unsigned int soa_refresh;
  unsigned int soa_retry;
  unsigned int soa_expire;
  unsigned int soa_min;

  edc->builder = GNUNET_GTK_get_new_builder ("gnunet_setup_gns_edit_soa.glade",
					     edc);
  if (NULL == edc->builder)
  {
    GNUNET_break (0);
    edc->cont (edc, GTK_RESPONSE_CANCEL);  /* treat as 'cancel' */
    return;    
  }
  if (GNUNET_YES ==
      edc->old_record_in_namestore)
  {
    if (7 != SSCANF (edc->n_value, 
		     "rname=%253s mname=%253s %u,%u,%u,%u,%u",
		     soa_rname, soa_mname,
		     &soa_serial, &soa_refresh, &soa_retry, &soa_expire, &soa_min))
    {
      GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                  _("Unable to parse SOA record `%s'\n"),
		  edc->n_value);
    }
    else
    {
      /* set SOA record */
      gtk_entry_set_text (GTK_ENTRY (gtk_builder_get_object (edc->builder,
							     "edit_dialog_soa_source_host_entry")),
			  soa_rname);
      gtk_entry_set_text (GTK_ENTRY (gtk_builder_get_object (edc->builder,
							     "edit_dialog_soa_contact_email_entry")),
			  soa_mname);
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (gtk_builder_get_object (edc->builder,
									  "edit_dialog_soa_serial_number_spinbutton")),
				 soa_serial);
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (gtk_builder_get_object (edc->builder,
									  "edit_dialog_soa_refresh_time_spinbutton")),
				 soa_refresh);
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (gtk_builder_get_object (edc->builder,
									  "edit_dialog_soa_retry_time_spinbutton")),
				 soa_retry);
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (gtk_builder_get_object (edc->builder,
									  "edit_dialog_soa_expire_time_spinbutton")),
				 soa_expire);
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (gtk_builder_get_object (edc->builder,
									  "edit_dialog_soa_minimum_ttl_spinbutton")),
				 soa_min);
    }
  }
  edc->validator = &edit_dialog_soa_validity_check;
  edc->dialog = GTK_DIALOG (gtk_builder_get_object (edc->builder,
						    "edit_soa_dialog"));
  run_edit_dialog (edc);
}


/* ************************ SRV records *********************** */
/* *** Not implemented: namestore_common.c as well as here! *** */
/* ************************ SRV records *********************** */

/**
 * Check validity of the value in the edit dialog for SRV-records.
 * Then call the shared validity check if the result is OK.
 *
 * @param edc edit dialog context
 */
static void
edit_dialog_srv_validity_check (struct EditDialogContext *edc)
{
  GtkEditable *entry;
  const gchar *preedit;

  entry = GTK_EDITABLE (gtk_builder_get_object (edc->builder,
						"edit_dialog_srv_target_entry")),
  preedit = gtk_editable_get_chars (entry, 0, -1);
  if ( (NULL == preedit) ||
       (GNUNET_OK != GNUNET_DNSPARSER_check_name (preedit)) )
  {
    edit_dialog_disable_save (edc);
    return;
  }
  entry = GTK_EDITABLE (gtk_builder_get_object (edc->builder,
						"edit_dialog_name_entry")),
  preedit = gtk_editable_get_chars (entry, 0, -1);
#if 0
  /* FIXME: check service name format! */
#endif
  edit_dialog_check_save (edc);
}


/**
 * Editing dialog was closed, get the data and call the
 * continuation.
 *
 * @param dialog editing dialog
 * @param user_data the 'struct EditDialogContext'
 */
void
GNS_edit_srv_dialog_response_cb (GtkDialog *dialog,
				   gint response_id,
				   gpointer user_data)
{
  struct EditDialogContext *edc = user_data;
    
  if (GTK_RESPONSE_OK == response_id)
  {
    edit_dialog_putes_common_elements (edc);
#if 0
    GtkEntry *entry;
    const gchar *value;

    entry = GTK_ENTRY (gtk_builder_get_object (edc->builder,
					       "edit_dialog_srv_target_entry"));
    /* FIXME: build srv record */
    value = gtk_entry_get_text (entry);
    g_free (edc->n_value);
    edc->n_value = g_strdup (value);
#endif

  }
  gtk_widget_destroy (GTK_WIDGET (edc->dialog));
  g_object_unref (edc->builder);
  edc->builder = NULL;
  edc->cont (edc, response_id);
}


/**
 * The user has edited the SRV record value.  Enable/disable 'save'
 * button depending on the validity of the value.
 *
 * @param entry editing widget
 * @param preedit new value
 * @param user_data the 'struct EditDialogContext' of the dialog
 */
void
GNS_edit_dialog_srv_target_entry_changed_cb (GtkEditable *entry,
					     gpointer user_data)
{
  struct EditDialogContext *edc = user_data;

  edit_dialog_srv_validity_check (edc);
}


/**
 * Run an GNS Edit dialog for an 'SRV' Record.
 *
 * @param cont continuation to call when done
 * @param edc editing context to use
 */
void
GNS_edit_dialog_srv (struct EditDialogContext *edc)
{
  edc->builder = GNUNET_GTK_get_new_builder ("gnunet_setup_gns_edit_srv.glade",
					     edc);
  if (NULL == edc->builder)
  {
    GNUNET_break (0);
    edc->cont (edc, GTK_RESPONSE_CANCEL);  /* treat as 'cancel' */
    return;    
  }
  if (GNUNET_YES ==
      edc->old_record_in_namestore)
  {
#if 0
    /* FIXME */
    /* set SRV record */
    gtk_entry_set_text (GTK_ENTRY (gtk_builder_get_object (edc->builder,
							   "edit_dialog_srv_target_entry")),
			edc->n_value);
#endif
  }
  edc->validator = &edit_dialog_srv_validity_check;
  edc->dialog = GTK_DIALOG (gtk_builder_get_object (edc->builder,
						    "edit_srv_dialog"));
  run_edit_dialog (edc);
}


/* ************************ TXT records *********************** */

/**
 * Check validity of the value in the edit dialog for TXT-records.
 * Then call the shared validity check if the result is OK.
 *
 * @param edc edit dialog context
 */
static void
edit_dialog_txt_validity_check (struct EditDialogContext *edc)
{
  edit_dialog_check_save (edc);
}


/**
 * Editing dialog was closed, get the data and call the
 * continuation.
 *
 * @param dialog editing dialog
 * @param user_data the 'struct EditDialogContext'
 */
void
GNS_edit_txt_dialog_response_cb (GtkDialog *dialog,
				   gint response_id,
				   gpointer user_data)
{
  struct EditDialogContext *edc = user_data;
  GtkEntry *entry;
  const gchar *value;
    
  if (GTK_RESPONSE_OK == response_id)
  {
    edit_dialog_putes_common_elements (edc);
    entry = GTK_ENTRY (gtk_builder_get_object (edc->builder,
					       "edit_dialog_txt_entry"));
    value = gtk_entry_get_text (entry);
    g_free (edc->n_value);
    edc->n_value = g_strdup (value);
  }
  gtk_widget_destroy (GTK_WIDGET (edc->dialog));
  g_object_unref (edc->builder);
  edc->builder = NULL;
  edc->cont (edc, response_id);
}


/**
 * The user has edited the TXT record value.  Enable/disable 'save'
 * button depending on the validity of the value.
 *
 * @param entry editing widget
 * @param preedit new value
 * @param user_data the 'struct EditDialogContext' of the dialog
 */
void
GNS_edit_dialog_txt_entry_changed_cb (GtkEditable *entry,
				       gpointer user_data)
{
  struct EditDialogContext *edc = user_data;

  edit_dialog_txt_validity_check (edc);
}


/**
 * Run an GNS Edit dialog for an 'TXT' Record.
 *
 * @param cont continuation to call when done
 * @param edc editing context to use
 */
void
GNS_edit_dialog_txt (struct EditDialogContext *edc)
{
  edc->builder = GNUNET_GTK_get_new_builder ("gnunet_setup_gns_edit_txt.glade",
					     edc);
  if (NULL == edc->builder)
  {
    GNUNET_break (0);
    edc->cont (edc, GTK_RESPONSE_CANCEL);  /* treat as 'cancel' */
    return;    
  }
  if (GNUNET_YES ==
      edc->old_record_in_namestore)
  {
    /* set TXT record */
    gtk_entry_set_text (GTK_ENTRY (gtk_builder_get_object (edc->builder,
							   "edit_dialog_txt_entry")),
			edc->n_value);
  }
  edc->validator = &edit_dialog_txt_validity_check;
  edc->dialog = GTK_DIALOG (gtk_builder_get_object (edc->builder,
						    "edit_txt_dialog"));
  run_edit_dialog (edc);
}


/* ************************ VPN records *********************** */

/**
 * Check validity of the value in the edit dialog for VPN-records.
 * Then call the shared validity check if the result is OK.
 *
 * @param edc edit dialog context
 */
static void
edit_dialog_vpn_validity_check (struct EditDialogContext *edc)
{
  GtkEditable *entry;
  const gchar *preedit;
  struct GNUNET_HashCode hc;
  
  entry = GTK_EDITABLE (gtk_builder_get_object (edc->builder,
						"edit_dialog_vpn_peer_entry")),
  preedit = gtk_editable_get_chars (entry, 0, -1);
  if ( (NULL == preedit) ||
       (GNUNET_OK != 
	GNUNET_CRYPTO_hash_from_string (preedit, &hc)) )
  {
    edit_dialog_disable_save (edc);
    return;
  }
  edit_dialog_check_save (edc);
}


/**
 * Editing dialog was closed, get the data and call the
 * continuation.
 *
 * @param dialog editing dialog
 * @param user_data the 'struct EditDialogContext'
 */
void
GNS_edit_vpn_dialog_response_cb (GtkDialog *dialog,
				   gint response_id,
				   gpointer user_data)
{
  struct EditDialogContext *edc = user_data;
  GtkEntry *entry;
  const gchar *identifier;
  const gchar *peer;
  unsigned int proto;
  char *result;
    
  if (GTK_RESPONSE_OK == response_id)
  {
    edit_dialog_putes_common_elements (edc);
    entry = GTK_ENTRY (gtk_builder_get_object (edc->builder,
					       "edit_dialog_vpn_identifier_entry"));
    identifier = gtk_entry_get_text (entry);
    entry = GTK_ENTRY (gtk_builder_get_object (edc->builder,
					       "edit_dialog_vpn_peer_entry"));
    peer = gtk_entry_get_text (entry);
    proto = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gtk_builder_get_object (edc->builder,
										     "edit_dialog_vpn_protocol_tcp_radiobutton"))) ? IPPROTO_TCP : IPPROTO_UDP;
    GNUNET_asprintf (&result, 
		     "%u %s %s",
		     proto,
		     (const char *) peer,
		     (const char *) identifier);
    g_free (edc->n_value);
    edc->n_value = g_strdup (result);
    GNUNET_free (result);
  }
  gtk_widget_destroy (GTK_WIDGET (edc->dialog));
  g_object_unref (edc->builder);
  edc->builder = NULL;
  edc->cont (edc, response_id);
}


/**
 * The user has edited the VPN record value.  Enable/disable 'save'
 * button depending on the validity of the value.
 *
 * @param entry editing widget
 * @param preedit new value
 * @param user_data the 'struct EditDialogContext' of the dialog
 */
void
GNS_edit_dialog_vpn_peer_entry_changed_cb (GtkEditable *entry,
					   gpointer user_data)
{
  struct EditDialogContext *edc = user_data;

  edit_dialog_vpn_validity_check (edc);
}


/**
 * Run an GNS Edit dialog for an 'VPN' Record.
 *
 * @param cont continuation to call when done
 * @param edc editing context to use
 */
void
GNS_edit_dialog_vpn (struct EditDialogContext *edc)
{
  char s_peer[103 + 1];
  char s_serv[253 + 1];
  unsigned int proto;
 
  edc->builder = GNUNET_GTK_get_new_builder ("gnunet_setup_gns_edit_vpn.glade",
					     edc);
  if (NULL == edc->builder)
  {
    GNUNET_break (0);
    edc->cont (edc, GTK_RESPONSE_CANCEL);  /* treat as 'cancel' */
    return;    
  }
  if (GNUNET_YES ==
      edc->old_record_in_namestore)
  {
    /* set VPN record */
    if (3 != SSCANF (edc->n_value,"%u %103s %253s",
		     &proto, s_peer, s_serv))
    {
      GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                  _("Unable to parse VPN record string `%s'\n"),
		  edc->n_value);
    }
    else
    {
      gtk_entry_set_text (GTK_ENTRY (gtk_builder_get_object (edc->builder,
							     "edit_dialog_vpn_identifier_entry")),
			  s_serv);
      gtk_entry_set_text (GTK_ENTRY (gtk_builder_get_object (edc->builder,
							     "edit_dialog_vpn_peer_entry")),
			  s_peer);
      if (IPPROTO_UDP == proto)      
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (gtk_builder_get_object (edc->builder,
										 "edit_dialog_vpn_protocol_udp_radiobutton")), TRUE);
      else
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (gtk_builder_get_object (edc->builder,
										 "edit_dialog_vpn_protocol_tcp_radiobutton")), TRUE);
    }
  }
  edc->validator = &edit_dialog_vpn_validity_check;
  edc->dialog = GTK_DIALOG (gtk_builder_get_object (edc->builder,
						    "edit_vpn_dialog"));
  run_edit_dialog (edc);
}


/* end of gnunet-setup-gns-edit.c */
