/*
     This file is part of GNUnet.
     (C) 2005, 2006, 2007, 2008 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/plugins/daemon/setup.c
 * @brief code for gnunet-setup interaction
 * @author Christian Grothoff
 */

#include "platform.h"
#include "gnunetgtk_common.h"
#include <GNUnet/gnunet_directories.h>
#include <GNUnet/gnunet_util.h>
#include <glib/gerror.h>
#ifdef WITH_LIBGKSU2
#include <libgksu/libgksu.h>
#endif
#include "daemon.h"

/**
 * Can we write to the given file (with our current
 * permissions)?
 */
static int
can_write(const char * f)
{
  struct stat buf;
  char * fn;
  int ret;
  int sret;

  fn = GNUNET_expand_file_name(NULL, f);
  sret = STAT (fn, &buf);
  ret =
    (0 < strlen(fn)) &&
    ( ( (0 == ACCESS (fn, W_OK)) &&
	(! S_ISDIR(buf.st_mode)) ) ||
      ( (0 != sret) &&
	(GNUNET_OK == GNUNET_disk_directory_create_for_file(NULL,
							    fn))) );
  GNUNET_free(fn);
  return ret;
}

static char *
get_gnunetdconfigfile (void)
{
  GtkWidget * entry;

  entry
    = glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                            "configfilenameEntry");
  return
    GNUNET_strdup(gtk_entry_get_text (GTK_ENTRY(entry)));
}

/**
 * Change the configuration filename.
 */
void
on_change_config_name_clicked_daemon (GtkWidget * editFNbutton, 
				      GtkWidget * ent)
{
  GtkWidget *dialog;
  char * fn;
  GtkWidget * entry;

  dialog = gtk_file_chooser_dialog_new ("Select Configuration File",
					NULL,
					GTK_FILE_CHOOSER_ACTION_SAVE,
					GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
					NULL);
  gtk_file_chooser_set_do_overwrite_confirmation (GTK_FILE_CHOOSER (dialog), FALSE);
  fn = get_gnunetdconfigfile ();
  gtk_file_chooser_set_filename (GTK_FILE_CHOOSER (dialog), fn);
  GNUNET_free(fn);
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
    {
      fn = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
      entry
	= glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
				"configfilenameEntry");
      gtk_entry_set_text (GTK_ENTRY(entry), fn);
      GNUNET_free(fn);
    }
  gtk_widget_destroy (dialog);
}

/**
 * Update sensitivity of daemon buttons
 */
void
on_gnunetdconfigfile_changed_daemon (GtkWidget * wid, gpointer data)
{
  char *conffile;
  int ret;

  conffile =
    get_gnunetdconfigfile();
  GNUNET_GC_set_configuration_value_string (daemon_get_cfg_handle(),
					    NULL,
					    "DAEMON",
					    "CONFIGFILE",
					    conffile);
  GNUNET_GC_write_configuration(daemon_get_cfg_handle(),
				GNUNET_GTK_get_client_config_filename());

#ifndef WITH_LIBGKSU2
  GtkWidget *startDaemonConfWizard;
  GtkWidget *startDaemonConfTool;

  startDaemonConfWizard
    = glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                            "startDaemonConfWizard");
  startDaemonConfTool
    = glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                            "startDaemonConfTool");
  if (can_write(conffile))
    {
      gtk_widget_set_sensitive (startDaemonConfWizard, TRUE);
      gtk_widget_set_sensitive (startDaemonConfTool, TRUE);
    }
  else
    {
      gtk_widget_set_sensitive (startDaemonConfWizard, FALSE);
      gtk_widget_set_sensitive (startDaemonConfTool, FALSE);
    }
  ret = 0;
#endif
  daemon_doUpdateMenus(&ret);
  GNUNET_free_non_null (conffile);
  return;
}
/**
 * Start gnunet-setup, asking for a password if needed
 *
 * @param conffile path to the gnunetd configuration file
 * @param run_wizard TRUE to run the wizard, FALSE to run the complete tool
 * @returns FALSE if gnunet-setup was run, TRUE else
 */
static gboolean
GNUNET_GTK_run_gnunet_setup (gchar * conffile, gboolean run_wizard)
{
  GtkWidget *mainWindow;
  GtkWidget *messageDialog;
  GError *gerror = NULL;
  char *error_message = NULL;

  if (can_write(conffile))
    {
      if (run_wizard)
        {
          char *argv[]
            = { "gnunet-setup", "-d", "wizard-gtk", "-c", conffile, NULL };
          g_spawn_async (NULL, argv, NULL,
                         G_SPAWN_SEARCH_PATH | G_SPAWN_CHILD_INHERITS_STDIN,
                         NULL, NULL, NULL, &gerror);
        }
      else
        {
          char *argv[]
            = { "gnunet-setup", "-d", "gconfig", "-c", conffile, NULL };
          g_spawn_async (NULL, argv, NULL,
                         G_SPAWN_SEARCH_PATH | G_SPAWN_CHILD_INHERITS_STDIN,
                         NULL, NULL, NULL, &gerror);
        }
      if (gerror)
        {
          error_message = GNUNET_strdup (gerror->message);
          g_error_free (gerror);
        }
    }
  else
#ifndef WITH_LIBGKSU2
    {
      error_message =
        GNUNET_strdup (_
                       ("You don't have rights to write to the provided configuration file."));
    }
#else
    {
      char *commandline;

      if (run_wizard)
        {
          commandline =
            g_strconcat ("gnunet-setup -d wizard-gtk -c ", conffile, NULL);
          gksu_run (commandline, &gerror);
        }
      else
        {
          commandline =
            g_strconcat ("gnunet-setup -d gconfig -c ", conffile, NULL);
          gksu_run (commandline, &gerror);
        }
      GNUNET_free (commandline);
      if (gerror)
        {
          error_message = GNUNET_strdup (gerror->message);
          g_error_free (gerror);
        }
    }
#endif
  if (error_message)
    {
      mainWindow =
        glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (), "mainWindow");
      messageDialog =
        gtk_message_dialog_new (GTK_WINDOW (mainWindow),
                                GTK_DIALOG_MODAL |
                                GTK_DIALOG_DESTROY_WITH_PARENT,
                                GTK_MESSAGE_WARNING, GTK_BUTTONS_CLOSE,
                                _
                                ("Failed to run the configuration tool (gnunet-setup): %s"),
                                error_message);
      gtk_dialog_run (GTK_DIALOG (messageDialog));
      gtk_widget_destroy (messageDialog);
      GNUNET_free (error_message);
      return FALSE;
    }
  return TRUE;
}

/**
 * Launch 'gnunet-setup -d wizard-gtk' with needed rights
 */
void
on_startDaemonConfWizard_clicked_daemon (GtkWidget * widget, gpointer data)
{
  char *conffile;

  conffile =
    get_gnunetdconfigfile();
  GNUNET_GTK_run_gnunet_setup (conffile, TRUE);
  GNUNET_free (conffile);
}

/**
 * Launch 'gnunet-setup -d gconfig' with needed rights
 */
void
on_startDaemonConfTool_clicked_daemon (GtkWidget * widget, gpointer data)
{
  char *conffile;

  conffile =
    get_gnunetdconfigfile();
  GNUNET_GTK_run_gnunet_setup (conffile, FALSE);
  GNUNET_free (conffile);
}

/* end of setup.c */
