/*
     This file is part of GNUnet
     (C) 2006 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/
/**
 * @file src/common/logging.c
 * @brief This file contains GUI functions related to logging
 * @author Igor Wronsky
 * @author Christian Grothoff
 */

#include "platform.h"
#include "gnunetgtk_common.h"
#include <GNUnet/gnunet_util_crypto.h>
#include <glib.h>
#include <gmodule.h>

static GtkWidget *infoWindow;

static GtkWidget *infoWindowTextView;

static GladeXML *statusXML;

static void
init ()
{
  GtkWidget *button;

  if (statusXML != NULL)
    return;
  statusXML
    = glade_xml_new (getGladeFileName (), "statusWindow", PACKAGE_NAME);
  infoWindow = glade_xml_get_widget (statusXML, "statusWindow");
  infoWindowTextView
    = glade_xml_get_widget (statusXML, "messageWindowTextView");
  button = glade_xml_get_widget (statusXML, "messageWindowCloseButton");
  connectGladeWithPlugins (statusXML);

  g_signal_connect_data (infoWindow,
                         "delete-event",
                         G_CALLBACK (&gtk_widget_hide_on_delete),
                         NULL, NULL, 0);
  g_signal_connect_data (button,
                         "clicked",
                         G_CALLBACK (&gtk_widget_hide_on_delete),
                         infoWindow, NULL, G_CONNECT_SWAPPED);
}

/**
 * Closure for doInfoMessage.
 */
typedef struct
{
  int doPopup;
  char *note;
} InfoMessage;

/**
 * Do not track more than MAX messages
 */
#define MAX_LINES 1000

/**
 * Callback for infoMessage()
 */
static void *
doInfoMessage (void *args)
{
  const InfoMessage *info = args;
  GtkTextIter iter;
  GtkTextBuffer *buffer;
  GtkTextIter end;

  init ();
  if (info->doPopup == YES)
    gtk_widget_show (infoWindow);
  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (infoWindowTextView));
  gtk_text_buffer_get_iter_at_offset (buffer, &iter, -1);
  gtk_text_buffer_insert (buffer, &iter, info->note, -1);
  if (gtk_text_buffer_get_line_count (buffer) >= MAX_LINES)
    {
      gtk_text_buffer_get_iter_at_line (buffer, &iter, MAX_LINES - 1);
      gtk_text_buffer_get_iter_at_line (buffer, &end, MAX_LINES);
      gtk_text_buffer_delete (buffer, &iter, &end);
    }
  return NULL;
}

/**
 * Appends a message to the info window
 *
 * @param doPopup do we open the window, YES or NO
 */
void
infoMessage (int doPopup, const char *format, ...)
{
  va_list args;
  InfoMessage info;

  va_start (args, format);
  info.note = g_strdup_vprintf (format, args);
  va_end (args);
  info.doPopup = doPopup;
  gtkSaveCall (&doInfoMessage, &info);
  g_free (info.note);
}

static void *
saveAddLogEntry (void *args)
{
  static GtkWidget *s = NULL;
  static int once = 1;
  static guint id;
  char * val;

  init ();
  if (once)
    {
      once = 0;
      s = glade_xml_get_widget (getMainXML (), "statusbar");
      id = gtk_statusbar_get_context_id (GTK_STATUSBAR (s), "LOG");
    }
  else
    gtk_statusbar_pop (GTK_STATUSBAR (s), id);
  val = STRDUP((const char*) args);
  if (strstr(val, "\n") != NULL)
    strstr(val, "\n")[0] = '\0';
  gtk_statusbar_push (GTK_STATUSBAR (s), id, val);
  FREE(val);
  return NULL;
}

/**
 * Appends a log entry to the info window
 *
 * @param txt the log entry
 *
 */
void
addLogEntry (const char *txt, ...)
{
  va_list args;
  gchar *note;

  va_start (args, txt);
  note = g_strdup_vprintf (txt, args);
  va_end (args);
  infoMessage (NO, note);
  gtkSaveCall (&saveAddLogEntry, (void *) note);
  g_free (note);
}

static void
ge_gtk_log_handler (void *ctx,
                    GE_KIND kind, const char *date, const char *msg)
{
  int popUp;

  popUp = NO;
  if ((kind & (GE_FATAL | GE_ERROR | GE_WARNING)) > 0)
    popUp = YES;
  infoMessage (popUp, "%s: %s", date, msg);
}

struct GE_Context *
createGtkLogger (GE_KIND mask)
{
  struct GE_Context *myLog;

  myLog = GE_create_context_callback (mask,
                                      &ge_gtk_log_handler, NULL, NULL, NULL);
  return myLog;
}

void __attribute__ ((destructor)) gnunet_gtk_common_ltdl_fini ()
{
  if (statusXML != NULL)
    {
      gtk_widget_destroy (infoWindow);
      infoWindow = NULL;
      UNREF (statusXML);
      statusXML = NULL;
    }
}
