/*
     This file is part of GNUnet.
     (C) 2005, 2006 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/plugins/fs/fs.c
 * @brief main file-sharing code of gnunet-gtk
 * @author Christian Grothoff
 */

#include "platform.h"
#include <GNUnet/gnunet_collection_lib.h>
#include "fs.h"
#include "download.h"
#include "search.h"
#include "upload.h"
#include "collection.h"
#include "namespace.h"

struct FSUI_Context * ctx;

struct GE_Context * ectx;

struct GC_Configuration * cfg;

SearchList * search_head;

DownloadList * download_head;

UploadList * upload_head;

GtkListStore * search_summary;

GtkTreeStore * download_summary;

GtkTreeStore * upload_summary;

/**
 * Last right-click event coordinates in summary.
 */
static unsigned int last_x;

/**
 * Last right-click event coordinates in summary.
 */
static unsigned int last_y;


static void *
saveEventProcessor(void * cls) {
  const FSUI_Event * event = cls;
  void * ret;

  ret = NULL;
  switch (event->type) {
    /* search events */
  case FSUI_search_started:
    ret = fs_search_started(event->data.SearchStarted.sc.pos,
			    event->data.SearchStarted.searchURI,
			    event->data.SearchStarted.anonymityLevel,
			    0,
			    NULL,
			    FSUI_ACTIVE);
    break;
  case FSUI_search_result:
    fs_search_result_received(event->data.SearchResult.sc.cctx,
			      &event->data.SearchResult.fi,
			      event->data.SearchResult.searchURI);
    break;
  case FSUI_search_error:
    fs_search_aborted(event->data.SearchError.sc.cctx);
    break;
  case FSUI_search_aborted:
    fs_search_aborted(event->data.SearchAborted.sc.cctx);
    break;
  case FSUI_search_completed:
    // FIXME...
    fs_search_aborted(event->data.SearchCompleted.sc.cctx);
    break;
  case FSUI_search_suspended:
    fs_search_aborted(event->data.SearchSuspended.sc.cctx);
    break;
  case FSUI_search_resumed:
    ret = fs_search_started(event->data.SearchResumed.sc.pos,
			    event->data.SearchResumed.searchURI,
			    event->data.SearchResumed.anonymityLevel,
			    event->data.SearchResumed.fisSize,
			    event->data.SearchResumed.fis,
			    event->data.SearchResumed.state);
    break;
  case FSUI_search_stopped:
    fs_search_stopped(event->data.SearchStopped.sc.cctx);
    break;

    /* download events */
  case FSUI_download_aborted:
    fs_download_aborted(event->data.DownloadAborted.dc.cctx);
    break;
  case FSUI_download_error:
    fs_download_aborted(event->data.DownloadError.dc.cctx);
    break;
  case FSUI_download_suspended:
    fs_download_stopped(event->data.DownloadSuspended.dc.cctx);
    break;
  case FSUI_download_progress:
    fs_download_update(event->data.DownloadProgress.dc.cctx,
		       event->data.DownloadProgress.completed,
		       event->data.DownloadProgress.last_block,
		       event->data.DownloadProgress.last_size);
    break;
  case FSUI_download_completed:
    fs_download_completed(event->data.DownloadCompleted.dc.cctx);
    break;
  case FSUI_download_stopped:
    fs_download_stopped(event->data.DownloadStopped.dc.cctx);
    break;
  case FSUI_download_started:
    ret = fs_download_started(event->data.DownloadStarted.dc.pos,
			      event->data.DownloadStarted.dc.pcctx,
			      event->data.DownloadStarted.dc.sctx,
			      event->data.DownloadStarted.total,
			      event->data.DownloadStarted.anonymityLevel,
			      &event->data.DownloadStarted.fi,
			      event->data.DownloadStarted.filename,
			      0,
			      get_time(),
			      FSUI_ACTIVE);
    break;
  case FSUI_download_resumed:
    ret = fs_download_started(event->data.DownloadResumed.dc.pos,
			      event->data.DownloadResumed.dc.pcctx,
			      event->data.DownloadResumed.dc.sctx,
			      event->data.DownloadResumed.total,
			      event->data.DownloadResumed.anonymityLevel,
			      &event->data.DownloadResumed.fi,
			      event->data.DownloadResumed.filename,
			      event->data.DownloadResumed.completed,
			      event->data.DownloadResumed.eta,
			      event->data.DownloadResumed.state);
    break;

    /* upload events */
  case FSUI_upload_progress:
    fs_upload_update(event->data.UploadProgress.uc.cctx,
		     event->data.UploadProgress.completed);
    break;
  case FSUI_upload_completed:
    fs_upload_complete(event->data.UploadCompleted.uc.cctx,
		       event->data.UploadCompleted.uri);
    break;
  case FSUI_upload_error:
    fs_upload_error(event->data.UploadError.uc.cctx);
    break;
  case FSUI_upload_aborted:
    fs_upload_error(event->data.UploadAborted.uc.cctx);
    break;
  case FSUI_upload_stopped:
    fs_upload_stopped(event->data.UploadStopped.uc.cctx);
    break;
  case FSUI_upload_suspended:
    fs_upload_stopped(event->data.UploadSuspended.uc.cctx);
    break;
  case FSUI_upload_started:
    ret = fs_upload_started(event->data.UploadStarted.uc.pos,
			    event->data.UploadStarted.uc.pcctx,
			    event->data.UploadStarted.filename,
			    NULL,
			    event->data.UploadStarted.total,
			    0,
			    FSUI_ACTIVE);
    break;
  case FSUI_upload_resumed:
    ret = fs_upload_started(event->data.UploadResumed.uc.pos,
			    event->data.UploadResumed.uc.pcctx,
			    event->data.UploadResumed.filename,
			    event->data.UploadResumed.uri,
			    event->data.UploadResumed.total,
			    event->data.UploadResumed.completed,
			    event->data.UploadResumed.state);
    break;
    /* TODO: unindex events */
  default:
    GE_BREAK(ectx, 0);
    GE_LOG(ectx,
	   GE_ERROR,
	   _("Unhandled (unknown) FSUI event: %u.\n"),
	   event->type);
    break;
  }
  return ret;
}

static void *
eventProcessor(void * unused,
	       const FSUI_Event * event) {
  return gtkSaveCall(&saveEventProcessor,
		     (void*) event);
}

/**
 * The selection of the upload summary changed.
 * Update button status.
 */
static void on_upload_summary_selection_changed(gpointer signal,
						gpointer cls) {
  GtkTreeSelection * selection;
  GtkWidget * button;

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(glade_xml_get_widget(getMainXML(),
									     "activeUploadsList")));
  button
    = glade_xml_get_widget(getMainXML(),
			   "cancelUploadButton");
  gtk_widget_set_sensitive(button,
			   gtk_tree_selection_count_selected_rows(selection) > 0);
  button
    = glade_xml_get_widget(getMainXML(),
			   "stopUploadButton");
  gtk_widget_set_sensitive(button,
			   gtk_tree_selection_count_selected_rows(selection) > 0);
}

/**
 * The selection of the download summary changed.
 * Update button status.
 */
static void on_download_summary_selection_changed(gpointer signal,
						  gpointer cls) {
  GtkTreeSelection * selection;
  GtkWidget * button;

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(glade_xml_get_widget(getMainXML(),
									     "activeDownloadsList")));
  button
    = glade_xml_get_widget(getMainXML(),
			   "abortDownloadButton");
  gtk_widget_set_sensitive(button,
			   gtk_tree_selection_count_selected_rows(selection) > 0);
  button
    = glade_xml_get_widget(getMainXML(),
			   "stopDownloadButton");
  gtk_widget_set_sensitive(button,
			   gtk_tree_selection_count_selected_rows(selection) > 0);
}

/**
 * The selection of the download summary changed.
 * Update button status.
 */
static void on_search_summary_selection_changed(gpointer signal,
						gpointer cls) {
  GtkTreeSelection * selection;
  GtkWidget * button;

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(glade_xml_get_widget(getMainXML(),
									     "activeSearchesSummary")));
  button
    = glade_xml_get_widget(getMainXML(),
			   "cancelSearchButton");
  gtk_widget_set_sensitive(button,
			   gtk_tree_selection_count_selected_rows(selection) > 0);
  button
    = glade_xml_get_widget(getMainXML(),
			   "stopSearchButton");
  gtk_widget_set_sensitive(button,
			   gtk_tree_selection_count_selected_rows(selection) > 0);
}

static int 
on_upload_copy_uri_activate(void * cls,
			    GtkWidget * searchEntry) {
  GtkTreeView * uploadList = cls;
  GtkTreePath *path;
  GtkTreeIter iter;
  struct ECRS_URI * uri;
  char * str;
  GtkClipboard * clip;

  path = NULL;
  if (FALSE == gtk_tree_view_get_path_at_pos(uploadList,
					     last_x,
                                             last_y,
                                             &path,
                                             NULL,
                                             NULL,
                                             NULL)) {
    GE_BREAK(NULL, 0);
    return FALSE;
  }
  if (FALSE == gtk_tree_model_get_iter(GTK_TREE_MODEL(upload_summary),
				       &iter,
				       path)) {
    GE_BREAK(NULL, 0);
    gtk_tree_path_free(path);
    return FALSE;
  }
  gtk_tree_path_free(path);
  uri = NULL;
  gtk_tree_model_get(GTK_TREE_MODEL(upload_summary),
		     &iter,
		     UPLOAD_URISTRING, &str,
		     -1);
  clip = gtk_clipboard_get(GDK_SELECTION_CLIPBOARD);
  gtk_clipboard_set_text(clip,
			 str,
			 strlen(str));
  FREE(str);
  return FALSE;
} 

static gint
upload_click_handler(void * cls, 
		     GdkEvent *event) {
  GtkTreeView * uploadList = cls;
  GtkMenu * menu;
  GtkWidget * entry;
  GdkEventButton * event_button;

  g_return_val_if_fail (event != NULL, FALSE);
  if (event->type != GDK_BUTTON_PRESS) 
    return FALSE;
  event_button = (GdkEventButton *) event;
  if (event_button->button != 3) 
    return FALSE;  
  last_x = event_button->x;
  last_y = event_button->y;
  entry = gtk_menu_item_new_with_label(_("_Copy URI to Clipboard"));
  g_signal_connect_swapped (entry,
			    "activate",
			    G_CALLBACK(on_upload_copy_uri_activate), 
			    uploadList);
  gtk_label_set_use_underline(GTK_LABEL(gtk_bin_get_child(GTK_BIN(entry))),
			      TRUE);
  gtk_widget_show(entry);
  menu = GTK_MENU(gtk_menu_new());
  gtk_menu_shell_append(GTK_MENU_SHELL(menu),
		        entry);
  gtk_menu_popup (menu,
		  NULL,
		  NULL,
		  NULL,
		  NULL, 
		  event_button->button,
		  event_button->time);
  return TRUE;
}

/**
 * Setup the summary views (in particular the models
 * and the renderers).
 */
static void fs_summary_start() {
  GtkComboBoxEntry * searchCB;
  GtkWidget * uploadEntry;
  GtkTreeView * searchList;
  GtkTreeView * downloadList;
  GtkTreeView * uploadList;
  GtkListStore * model;
  GtkCellRenderer * renderer;
  GtkTreeViewColumn * column;
  int col;

  /* search namespace selection setup */
  searchCB
    = GTK_COMBO_BOX_ENTRY(glade_xml_get_widget(getMainXML(),
					       "fssearchKeywordComboBoxEntry"));

  model = gtk_list_store_new(NS_SEARCH_NUM,
			     G_TYPE_STRING, /* what we show */
			     G_TYPE_STRING, /* EncName of namespace */
			     G_TYPE_POINTER, /* ECRS MetaData */
			     G_TYPE_INT);  /* Meta-data about namespace */
  gtk_combo_box_set_model(GTK_COMBO_BOX(searchCB),
			  GTK_TREE_MODEL(model));
  gtk_combo_box_entry_set_text_column(searchCB,
				      NS_SEARCH_DESCRIPTION);

  /* search summary setup */
  searchList = GTK_TREE_VIEW(glade_xml_get_widget(getMainXML(),
						  "activeSearchesSummary"));
  search_summary =
    gtk_list_store_new(SEARCH_SUMMARY_NUM,
		       G_TYPE_STRING, /* name */
		       G_TYPE_INT,    /* # results */
		       G_TYPE_POINTER);  /* internal: search list */
  gtk_tree_view_set_model(searchList,
			  GTK_TREE_MODEL(search_summary));
  gtk_tree_selection_set_mode(gtk_tree_view_get_selection(searchList),
			      GTK_SELECTION_MULTIPLE);
  g_signal_connect_data(gtk_tree_view_get_selection(searchList),
			"changed",
			G_CALLBACK(&on_search_summary_selection_changed),
			NULL,
			NULL,
			0);

  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes(searchList,
						    -1,
						    _("Query"),
						    renderer,
						    "text", SEARCH_SUMMARY_NAME,
					      NULL);
  column = gtk_tree_view_get_column(searchList,
				    col - 1);
  gtk_tree_view_column_set_resizable(column, TRUE);
  gtk_tree_view_column_set_clickable(column, TRUE);
  gtk_tree_view_column_set_reorderable(column, TRUE);
  gtk_tree_view_column_set_sort_column_id(column, SEARCH_SUMMARY_NAME);
  gtk_tree_view_column_set_resizable(column, TRUE);
  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes(searchList,
						    -1,
						    _("Results"),
						    renderer,
						    "text", SEARCH_SUMMARY_RESULT_COUNT,
						    NULL);
  column = gtk_tree_view_get_column(searchList,
				    col - 1);
  gtk_tree_view_column_set_resizable(column, TRUE);
  gtk_tree_view_column_set_clickable(column, TRUE);
  gtk_tree_view_column_set_reorderable(column, TRUE);
  gtk_tree_view_column_set_sort_column_id(column, SEARCH_SUMMARY_RESULT_COUNT);
  gtk_tree_view_column_set_resizable(column, TRUE);

  /* download summary setup */
  downloadList = GTK_TREE_VIEW(glade_xml_get_widget(getMainXML(),
						    "activeDownloadsList"));
  download_summary =
    gtk_tree_store_new(DOWNLOAD_NUM,
                       G_TYPE_STRING, /* name (full-path file name) */
                       G_TYPE_STRING, /* name (user-friendly name) */
                       G_TYPE_UINT64,  /* size */
		       G_TYPE_STRING, /* human readable size */
                       G_TYPE_INT,  /* progress */
                       G_TYPE_STRING, /* uri as string */
		       G_TYPE_POINTER); /* internal download list ptr */
  gtk_tree_view_set_model(downloadList,
                          GTK_TREE_MODEL(download_summary));
  gtk_tree_selection_set_mode(gtk_tree_view_get_selection(downloadList),
			      GTK_SELECTION_MULTIPLE);
  g_signal_connect_data(gtk_tree_view_get_selection(downloadList),
			"changed",
			G_CALLBACK(&on_download_summary_selection_changed),
			NULL,
			NULL,
			0);

  renderer = gtk_cell_renderer_text_new();
  g_object_set (renderer, "xalign", 1.00, NULL);
  col = gtk_tree_view_insert_column_with_attributes(downloadList,
						    -1,
						    _("Size"),
						    renderer,
						    "text", DOWNLOAD_HSIZE,
						    NULL);

  column = gtk_tree_view_get_column(downloadList,
				    col - 1);
  gtk_tree_view_column_set_resizable(column, TRUE);
  gtk_tree_view_column_set_clickable(column, TRUE);
  gtk_tree_view_column_set_reorderable(column, TRUE);
  gtk_tree_view_column_set_sort_column_id(column, DOWNLOAD_SIZE);
  gtk_tree_view_column_set_resizable(column, TRUE);

  renderer = gtk_cell_renderer_progress_new();
  col = gtk_tree_view_insert_column_with_attributes(downloadList,
						    -1,
						    _("Name"),
						    renderer,
						    "value", DOWNLOAD_PROGRESS,
						    "text", DOWNLOAD_SHORTNAME,
						    NULL);
  g_object_set(G_OBJECT(renderer),
	       "width", 400,
	       NULL);
  column = gtk_tree_view_get_column(downloadList,
				    col - 1);
  gtk_tree_view_column_set_resizable(column, TRUE);
  gtk_tree_view_column_set_clickable(column, TRUE);
  gtk_tree_view_column_set_reorderable(column, TRUE);
  gtk_tree_view_column_set_sort_column_id(column, DOWNLOAD_PROGRESS);
  gtk_tree_view_column_set_resizable(column, TRUE);

  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes(downloadList,
						    -1,
						    _("URI"),
						    renderer,
						    "text", DOWNLOAD_URISTRING,
						    NULL);
  g_object_set(G_OBJECT(renderer),
	       "wrap-width", 30,
	       "width-chars", 30,
	       "ellipsize", PANGO_ELLIPSIZE_END,
	       NULL);
  column = gtk_tree_view_get_column(downloadList,
				    col - 1);
  gtk_tree_view_column_set_resizable(column, TRUE);
  gtk_tree_view_column_set_reorderable(column, TRUE);
  gtk_tree_view_column_set_resizable(column, TRUE);

  /* upload summary setup */
  uploadList = GTK_TREE_VIEW(glade_xml_get_widget(getMainXML(),
						  "activeUploadsList"));
  g_signal_connect_swapped (uploadList,
			    "button-press-event",
			    G_CALLBACK(upload_click_handler), 
			    uploadList);
  upload_summary =
    gtk_tree_store_new(UPLOAD_NUM,
		       G_TYPE_STRING, /* filename */
		       G_TYPE_INT,    /* progress */
		       G_TYPE_STRING, /* URI as string */
		       G_TYPE_POINTER);  /* UploadList */
  gtk_tree_view_set_model(uploadList,
			  GTK_TREE_MODEL(upload_summary));
  gtk_tree_selection_set_mode(gtk_tree_view_get_selection(uploadList),
			      GTK_SELECTION_MULTIPLE);
  g_signal_connect_data(gtk_tree_view_get_selection(uploadList),
			"changed",
			G_CALLBACK(&on_upload_summary_selection_changed),
			NULL,
			NULL,
			0);

  renderer = gtk_cell_renderer_progress_new();
  col = gtk_tree_view_insert_column_with_attributes(uploadList,
						    -1,
						    _("Filename"),
						    renderer,
						    "text", UPLOAD_FILENAME,
						    "value", UPLOAD_PROGRESS,
						    NULL);
  gtk_tree_view_column_set_resizable(gtk_tree_view_get_column(uploadList,
							      col - 1),
				     TRUE);

  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes(uploadList,
						    -1,
						    _("URI"),
						    renderer,
						    "text", UPLOAD_URISTRING,
						    NULL);
  gtk_tree_view_column_set_resizable(gtk_tree_view_get_column(uploadList,
							      col - 1),
				     TRUE);
  /* upload entry setup */
  uploadEntry
    = glade_xml_get_widget(getMainXML(),
			   "uploadFilenameComboBoxEntry");

  model = gtk_list_store_new(1, G_TYPE_STRING);
  gtk_combo_box_set_model(GTK_COMBO_BOX(uploadEntry),
			  GTK_TREE_MODEL(model));
  gtk_combo_box_entry_set_text_column(GTK_COMBO_BOX_ENTRY(uploadEntry),
				      0);
}

/**
 * Shutdown the summary dialogs.
 */
static void fs_summary_stop() {
  struct ECRS_MetaData * meta;
  GtkComboBox * searchCB;
  GtkTreeModel * model;
  GtkTreeIter iter;

  searchCB
    = GTK_COMBO_BOX(glade_xml_get_widget(getMainXML(),
					 "fssearchKeywordComboBoxEntry"));
  model = gtk_combo_box_get_model(searchCB);
  if (gtk_tree_model_get_iter_first(model,
				    &iter)) {
    do {	
      gtk_tree_model_get(model,
			 &iter,
			 NS_SEARCH_METADATA, &meta,
			 -1);
      gtk_list_store_set(GTK_LIST_STORE(model),
			 &iter,
			 NS_SEARCH_METADATA, NULL,
			 -1);
      if (meta != NULL)
	ECRS_freeMetaData(meta);
    } while (gtk_list_store_remove(GTK_LIST_STORE(model),
				   &iter));
  }

}

void init_fs(struct GE_Context * e,
	     struct GC_Configuration * c) {
  GtkWidget * tab;
  GtkWidget * book;
  gint num;

  ectx = e;
  cfg = c;
  CO_init(ectx, cfg);
  tab
    = glade_xml_get_widget(getMainXML(),
			   "fsnotebook");
  gtk_widget_show(tab);
  book
    = glade_xml_get_widget(getMainXML(), "mainnotebook");
  num = gtk_notebook_get_current_page(GTK_NOTEBOOK(book));
  gtk_notebook_set_current_page(GTK_NOTEBOOK(book), 1);
  gtk_notebook_set_current_page(GTK_NOTEBOOK(book), num);
  fs_summary_start();
  fs_collection_start();
  fs_namespace_start();
  ctx = FSUI_start(ectx,
		   cfg,
		   "gnunet-gtk",
		   128, /* FIXME: allow user to configure download parallelism */
		   YES,
		   &eventProcessor,
		   NULL);
}

void done_fs() {
  GtkWidget * tab;

  tab
    = glade_xml_get_widget(getMainXML(),
			   "fsnotebook");
  gtk_widget_hide(tab);
  FSUI_stop(ctx);
  fs_summary_stop();
  fs_namespace_stop();
  CO_done();
}

/* end of fs.c */
