/*
     This file is part of GNUnet
     (C) 2003, 2004, 2005, 2006 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/
/**
 * @file src/common/helper.c
 * @brief This file contains some GUI helper functions
 * @author Igor Wronsky
 * @author Christian Grothoff
 */

#include "platform.h"
#include "gnunetgtk_common.h"
#include <GNUnet/gnunet_util_crypto.h>
#include <glib.h>
#include <gmodule.h>

#define HELPER_DEBUG NO

#ifdef WITH_LIBNOTIFY
#include <libnotify/notify.h>
#endif

#ifdef WITH_LIBGKSU2
/*#include <libgksu/libgksu.h>
* commented because support is broken in Debian,
* and these headers are not really useful for us */
#endif

typedef struct {
  struct SEMAPHORE * sem;
  void * args;
  PThreadMain func;
  int destroyed;
  void * rval;
} SaveCall;

typedef struct Plugin {
  struct Plugin * next;
  char * name;
  void * library;
} Plugin;

static GladeXML * mainXML;

static char * gladeFile;

static GdkWindowState main_window_state;

/**
 * the main thread
 */
static struct PTHREAD * mainThread;

static SaveCall ** psc;

static unsigned int pscCount;

static struct MUTEX * sclock;

static int saveCallsUp;

static Plugin * plugin;

static void * shutdown_function;

static struct GE_Context * ectx;

static struct GC_Configuration * cfg;

static gboolean saveCallWrapper(gpointer data) {
  SaveCall * call = data;
  int i;

  /* clearly, we are no longer pending,
     so remove from psc */
  if (call->sem != NULL) {
    MUTEX_LOCK(sclock);
    for (i=0;i<pscCount;i++) {
      if (psc[i] == call) {
        psc[i] = psc[pscCount-1];
        break;
      }
    }
    GE_ASSERT(NULL, i != pscCount);
    GROW(psc,
         pscCount,
         pscCount-1);
    MUTEX_UNLOCK(sclock);
  }
  call->rval = call->func(call->args);
  if (call->sem != NULL)
    SEMAPHORE_UP(call->sem);
  return FALSE;
}

/**
 * Call a callback function from the mainloop/main thread ("SaveCall").
 * Since GTK doesn't work with multi-threaded applications under Windows,
 * all GTK operations have to be done in the main thread
 */
void * gtkSaveCall(PThreadMain func,
		   void * args) {
  SaveCall call;

  MUTEX_LOCK(sclock);
  if ( (saveCallsUp == NO) ||
       (! PTHREAD_TEST_SELF(mainThread)) ) {
    call.args = args;
    call.func = func;
    call.sem  = SEMAPHORE_CREATE(0);
    call.destroyed = 0;
    call.rval = NULL;
    GROW(psc,
	 pscCount,
	 pscCount+1);
    psc[pscCount-1] = &call;
    gtk_idle_add(&saveCallWrapper,
		 &call);
    MUTEX_UNLOCK(sclock);
    PTHREAD_STOP_SLEEP(mainThread);
    SEMAPHORE_DOWN(call.sem, YES);
    SEMAPHORE_DESTROY(call.sem);
    return call.rval;
  } else {
    MUTEX_UNLOCK(sclock);
    return func(args);
  }
}

/**
 * Simple accessor method.
 */
const char * getGladeFileName() {
  return gladeFile;
}

/**
 * Simple accessor method.
 */
GladeXML * getMainXML() {
  return mainXML;
}

static void connector(const gchar *handler_name,
		      GObject *object,
		      const gchar *signal_name,
		      const gchar *signal_data,
		      GObject *connect_object,
		      gboolean after,
		      gpointer user_data) {
  GModule * module;
  GladeXML * xml = user_data;
  Plugin * plug;
  void * method;

  plug = plugin;
  method = NULL;
  while (plug != NULL) {
    method = os_plugin_resolve_function(plug->library,
					handler_name,
					NO);
    if (method != NULL)
      break;
    plug = plug->next;
  }
  if (0 == strcmp(handler_name,
		  "gnunet_gtk_main_quit"))
    method = shutdown_function;
  if (method == NULL) {
    module = g_module_open(NULL, 0);
    if ( (module == NULL) ||
	 (TRUE != g_module_symbol(module,
				handler_name,
				  &method)) ) {
      GE_LOG(ectx,
	     GE_WARNING | GE_DEVELOPER | GE_IMMEDIATE,
	     _("Failed to find handler for `%s'\n"),
	     handler_name);
      g_module_close(module);
      return;
    }
    g_module_close(module);
  }
  glade_xml_signal_connect(xml,
			   handler_name,
			   (GCallback) method);
}

void connectGladeWithPlugins(GladeXML * xml) {
  glade_xml_signal_autoconnect_full(xml,
				    &connector,
				    xml);
}

typedef void (*InitCall)(struct GE_Context * ectx,
			 struct GC_Configuration * cfg);

typedef void (*PlainCall)(void);

static void loadPlugin(const char * name) {
  Plugin * p;
  struct PluginHandle * lib;
  InitCall init;

  lib = os_plugin_load(ectx,
		       "libgnunetgtkmodule_",
		       name);
  if (lib == NULL)
    return;
  p = MALLOC(sizeof(Plugin));
  p->name = STRDUP(name);
  p->next = plugin;
  p->library = lib;
  plugin = p;
  init = os_plugin_resolve_function(lib,
				    "init_",
				    NO);
  if (init != NULL)
    init(ectx, cfg);
}

static void loadPlugins(const char * names) {
  char * dup;
  char * next;
  const char * pos;

  if (names == NULL)
    return;

  dup = STRDUP(names);
  next = dup;
  do {
    while (*next == ' ')
      next++;
    pos = next;
    while ( (*next != '\0') &&
	    (*next != ' ') )
      next++;
    if (*next == '\0') {
      next = NULL; /* terminate! */
    } else {
      *next = '\0'; /* add 0-termination for pos */
      next++;
    }
    if (strlen(pos) > 0)
      loadPlugin(pos);
  } while (next != NULL);
  FREE(dup);
}

static void *unloadPlugin(void *p) {
  PlainCall done;
  Plugin * plug = (Plugin *) p;

  done = os_plugin_resolve_function(plug->library,
				    "done_",
				    NO);
  if (done != NULL)
    done();
  os_plugin_unload(plug->library);
  FREE(plug->name);
  FREE(plug);
  
  return NULL;
}

void initGNUnetGTKCommon(struct GE_Context * e,
			 struct GC_Configuration * c,
			 void * callback) {
  char * load;
  char * path;
  char * filename;

  ectx = e;
  cfg = c;
  shutdown_function = callback;
  sclock = MUTEX_CREATE(YES);
  mainThread = PTHREAD_GET_SELF();
  saveCallsUp = YES;

  /* load the interface */
  path = os_get_installation_path(IPK_DATADIR);
  filename = MALLOC(strlen(path) + strlen("/../gnunet-gtk/gnunet-gtk.glade") + 2);
  strcpy(filename, path);
  FREE(path);
  strcat(filename, "/../gnunet-gtk/gnunet-gtk.glade");
#if MINGW
  gladeFile = MALLOC(_MAX_PATH + 1);
  plibc_conv_to_win_path(filename,
			 gladeFile);
#else
  gladeFile = STRDUP(filename);
#endif
  FREE(filename);

  mainXML = glade_xml_new(gladeFile,
			  "mainWindow",
			  PACKAGE_NAME);
  if (mainXML == NULL)
    GE_DIE_STRERROR_FILE(ectx,
			 GE_FATAL | GE_USER | GE_IMMEDIATE,
			 "glade_xml_new",
			 gladeFile);
  /* load the plugins */
  GC_get_configuration_value_string(cfg,
				    "GNUNET-GTK",
				    "PLUGINS",
				    "about daemon fs stats",
				    &load);
  loadPlugins(load);
  FREE(load);
  connectGladeWithPlugins(mainXML);
}

void shutdownPlugins() {
  int i;

  /* unload the plugins */
  while (plugin != NULL) {
    Plugin * next;

    next = plugin->next;
    gtkSaveCall(&unloadPlugin, (void *) plugin);
    plugin = next;
  }
  UNREF(mainXML);
  mainXML = NULL;
  FREE(gladeFile);
  gladeFile = NULL;

  saveCallsUp = NO;
  MUTEX_LOCK(sclock);
  for (i=0;i<pscCount;i++)
    psc[i]->func(psc[i]);
  i = pscCount;
  MUTEX_UNLOCK(sclock);
  /* wait until all PSC-jobs have left
     the gtkSaveCall method before destroying
     the mutex! */
  while (i != 0) {
    PTHREAD_SLEEP(50 * cronMILLIS);
    MUTEX_LOCK(sclock);
    i = pscCount;
    MUTEX_UNLOCK(sclock);
  }
}

void doneGNUnetGTKCommon() {
  PTHREAD_REL_SELF(mainThread);
  MUTEX_DESTROY(sclock);
}

struct rwsc_closure {
  struct SEMAPHORE * sig;
  PThreadMain realMain;
  void * arg;
};

static void * shutdownCode(void * arg) {
  struct rwsc_closure * cls = arg;
  void * ret;

  ret = cls->realMain(cls->arg);
  SEMAPHORE_UP(cls->sig);
  PTHREAD_STOP_SLEEP(mainThread);
  return ret;
}

void * run_with_save_calls(PThreadMain cb,
			   void * arg) {
  struct PTHREAD * doneThread;
  void * retval;
  struct rwsc_closure cls;
  int i;

  cls.sig = SEMAPHORE_CREATE(0);
  cls.realMain = cb;
  cls.arg = arg;
  doneThread = PTHREAD_CREATE(&shutdownCode,
			      &cls,
			      64*1024);
  if (doneThread == NULL)
    GE_DIE_STRERROR(ectx,
		    GE_FATAL | GE_ADMIN | GE_BULK,
		    "pthread_create");
  if (! PTHREAD_TEST_SELF(mainThread)) {
    /* another thread will run the save calls */
    SEMAPHORE_DOWN(cls.sig, YES);
  } else {
    while (SYSERR == SEMAPHORE_DOWN(cls.sig, NO)) {
      MUTEX_LOCK(sclock);
      if (pscCount > 0) {
	i = weak_randomi(pscCount);
	if (TRUE == g_idle_remove_by_data(psc[i]))
	  saveCallWrapper(psc[i]);
      } else {
	i = -1;
      }
      MUTEX_UNLOCK(sclock);
      if ( (i == -1) &&
	   (OK != SEMAPHORE_DOWN(cls.sig, NO)) ) {
	PTHREAD_SLEEP(50 * cronMILLIS);
      }
    }
  }
  PTHREAD_JOIN(doneThread,
	       &retval);
  SEMAPHORE_DESTROY(cls.sig);
  return retval;
}

/**
 * Simple glue to libnotify, and others?
 *
 */
void gnunetgtk_notify(int type,
		      const char *message,
		      ...) {
#ifdef WITH_LIBNOTIFY
  static int once;
  char * msg;
  size_t size;
  va_list arg;
  GtkWidget * root;
  NotifyNotification *libnotify;
  NotifyUrgency libnotify_urgency = NOTIFY_URGENCY_NORMAL;
  long libnotify_expire_timeout = NOTIFY_EXPIRES_DEFAULT;

  if (! notify_is_initted()){
    if (once == 1)
      return;
    if (! notify_init ("gnunet-gtk")) {
      once = 1;
      GE_LOG(ectx,
	     GE_WARNING | GE_BULK | GE_USER | GE_ADMIN,
	     _("Could not initialize libnotify\n"));
      return;
    }
  }

  root = glade_xml_get_widget(getMainXML(),"mainWindow");
  if (gtk_window_is_active(GTK_WINDOW(root)) == FALSE) {
    if (type == NOTIFY_LOW)
      libnotify_urgency = NOTIFY_URGENCY_LOW;
     else if( type == NOTIFY_NORMAL)
       libnotify_urgency = NOTIFY_URGENCY_NORMAL;
     else
       libnotify_urgency = NOTIFY_URGENCY_CRITICAL;
    va_start(arg, message);
    size = vsnprintf(NULL, 0, message, arg);
    va_end(arg);
    msg = MALLOC(size+1);
    va_start(arg, message);
    vsnprintf(msg, size, message, arg);
    va_end(arg);
    libnotify = notify_notification_new("gnunet-gtk",
                                        msg,
                                        PACKAGE_DATA "/gnunet-gtk/gnunet-gtk-notify.png",
					NULL);
    FREE(msg);
    notify_notification_set_timeout(libnotify, libnotify_expire_timeout);
    notify_notification_set_urgency(libnotify, libnotify_urgency);
    if (! notify_notification_show (libnotify, NULL)) {
      once = 1;
      GE_LOG(ectx,
	     GE_WARNING | GE_USER | GE_ADMIN | GE_BULK,
	     _("Could not send notification via libnotify\n"));
    }
    g_object_unref(G_OBJECT(libnotify));
    notify_uninit();
  }
#endif
}

/**
 * Validate that a string is a Utf-8 string.
 * If validation fails, msg is freed and a valid
 * Utf-8 string is returned.
 */
char * validate_utf8(char * msg) {
  const gchar * end;
  char * ret;
  gsize send;

  end = NULL;
  if (TRUE == g_utf8_validate(msg,
			      -1,
			      &end))
    return msg;
  /* hope that it is ISO8859-1 */
  ret = g_convert_with_fallback(msg,
				-1,
				"UTF-8",
				"ISO8859-1",
				".",
				NULL,
				&send,
				NULL);
  FREE(msg);
  msg = STRDUP(ret);
  g_free(ret);
  GE_BREAK(NULL,
	   TRUE == g_utf8_validate(msg,
				   -1,
				   &end));
  return msg;
}

/**
* Gtk callback to save the main window state (tray icon use)
*/
void saveMainWindowState(GtkWidget *main_window,
                         GdkEventWindowState *event,
                         gpointer user_data) {
  main_window_state = (*event).new_window_state;
  return;
}

/**
* Get the last main window state when restoring (tray icon use)
*/
GdkWindowState getMainWindowState() {
  return main_window_state;
}

/**
 * Start gnunet-setup, asking for a password if needed
 */
gboolean startGNUnetSetup (gboolean run_wizard) {
  GtkWidget * mainWindow;
  GtkWidget * messageDialog;
  int code;
  char *error_message;
#ifdef WITH_LIBGKSU2
  GError *gerror = NULL;
  if(run_wizard) {
    code = gksu_run("gnunet-setup -d wizard-gtk",
		    &gerror);
  } else {
    code = gksu_run("gnunet-setup -d",
		    &gerror);
  }
  if (code && !gerror) {
    error_message = STRDUP(_("GKSu encountered an unknown error running the configuration tool (gnunet-setup)."));
  } else if (code && gerror) {
    error_message = g_strdup_printf(_("GKSu returned:\n%s"),
				    gerror->message);
    g_error_free(gerror);
  } else {
    error_message = NULL;
  }
#elif defined(WINDOWS)
/* FIXME: run gnunet-setup, assuming we can get the needed rights */
  error_message = STRDUP(_("Not implemented yet!"));
  code = TRUE;
#else
  error_message = STRDUP(_("GKSu support is not enabled, impossible to get the needed rights. You should build gnunet-gtk with the --enable-libgksu2 option, or get the right binary package. Note you can still start the configuration tool (gnunet-setup) manually."));
  code = TRUE;
#endif
  mainWindow = glade_xml_get_widget(getMainXML(),
				    "mainWindow");
  messageDialog = gtk_message_dialog_new(GTK_WINDOW(mainWindow),
					 GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
					 GTK_MESSAGE_WARNING,
					 GTK_BUTTONS_CLOSE,
					 _("Failed to run the configuration tool (gnunet-setup): %s"),
					 error_message);
  gtk_dialog_run(GTK_DIALOG(messageDialog));
  gtk_widget_destroy(messageDialog);
  FREE(error_message);
  return code;
}

/* end of helper.c */
