// Copyright (C) 2001 Nils Bokermann <Nils.Bokermann@mediaWays.net>
//
// PURPOSE OF THIS FILE: Implement the ldap_bind and ldap_init functions
//
// - Automatic Version Information via RCS:
//   $Id: bind.cxx,v 1.1 2001/12/17 16:36:17 nilsb Exp $
//   $Source: /cvsroot/openh323gk/openh323gk/ldap/src/bind.cxx,v $
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "ldapapi.h"
#include <ldap-int.h>
#include "LDAP_SBindRequest_authentication.h"
#include <ptlib/sockets.h>
#include <string.h>

#ifndef lint
// mark object with version info
static const char vcid[] = "@(#) $Id: bind.cxx,v 1.1 2001/12/17 16:36:17 nilsb Exp $";
static const char vcHid[] = __LDAP_OPENH323_H;
#endif

LDAP * ldap_init(const char *hosts, int port) {
  LDAP *returnvalue=ldap_new();
  if(NULL==returnvalue)
    return NULL;
  returnvalue->socket=new PTCPSocket();
  if(NULL==returnvalue->socket){
    ldap_free(returnvalue);
    return NULL;
  }
  returnvalue->std_port=port;
  
  if(NULL!=hosts){
    char *s;
    char *last=(char *)hosts;
    for(s=(char *)hosts; s<hosts+strlen(hosts)+1; s++){ // Copy the entries 
      int current_port;
      char *hostname;
      if((' '==*s)||('\0'==*s)){
	hostname=strndup(last,s-last);
	char *first_colon;
	if(NULL!=(first_colon=strchr(hostname,':'))){
	  char *tmphost=strndup(hostname,first_colon-hostname);
	  if(NULL==tmphost){
	    ldap_free(returnvalue);
	    return NULL;
	  }
	  current_port=atoi(first_colon+1);
	  ldap_memfree(hostname);
	  hostname=tmphost;
	}
	if((current_port<=0)||(current_port>65535))
	  current_port=(port==0 ? LDAP_PORT : port);
	LDAP_Hostentry *host=new LDAP_Hostentry();
	host->hostname=PString(hostname);
	host->port=current_port;
	returnvalue->hostlist.push_back(host);
	ldap_memfree(hostname);
      }
    }
  }else{ // No host provided 
    LDAP_Hostentry *host=new LDAP_Hostentry();
    host->hostname="localhost";
    host->port=port==0 ? LDAP_PORT : port;
    returnvalue->hostlist.push_back(host);
  }
  ldap_int_initialize(returnvalue->ld_options);
  return returnvalue;
}


LDAP *ldap_open(const char *host, int port){
  LDAP *ld=ldap_init(host,port);
  for(vector<LDAP_Hostentry*>::iterator iter=ld->hostlist.begin();
      iter!=ld->hostlist.end();++iter){
    ld->socket->SetPort((*iter)->port);
    if(ld->socket->Connect((*iter)->hostname))
      break;
  }
  if(!ld->socket->IsOpen()){
    ldap_free(ld);
    return NULL;
  }
  return ld;
}

int ldap_sasl_bind(LDAP *ld, LDAP_CONST char *dn, const char *mechanism, 
		   LDAP_CONST struct berval *cred, LDAPControl **serverctrls,
		   LDAPControl **clientctrls, int *msgidp) {
  LDAP_LDAPMessage *msg=new LDAP_LDAPMessage();
  
  *msgidp=++(ld->msg_id);
  msg->m_messageID=*msgidp;
  msg->m_protocolOp=LDAP_LDAPMessage_protocolOp(LDAP_LDAPMessage_protocolOp::e_bindRequest);
  //  msg->m_protocolOp.CreateObject();

  LDAP_BindRequest & bind = msg->m_protocolOp;
  bind.m_version=LDAP_VERSION2;
  if(NULL==dn) {
    bind.m_name=PString("");
  } else {
    bind.m_name=dn;
  }
 

  LDAP_SBindRequest_authentication auth;
  if(mechanism==LDAP_SASL_SIMPLE){
    auth.SetCredentials(cred->bv_val, cred->bv_len);
    bind.m_authentication=auth;
  } else if( ld->ld_version < LDAP_VERSION3 ) {
    ld->ld_errno = LDAP_NOT_SUPPORTED;
    return ld->ld_errno;
  }

  if(!ld->socket->IsOpen()) {
    for(vector<LDAP_Hostentry*>::iterator iter=ld->hostlist.begin();
	iter!=ld->hostlist.end();++iter){
      ld->socket->SetPort((*iter)->port);
      if(ld->socket->Connect((*iter)->hostname))
	break;
    }
    if(!ld->socket->IsOpen()){
      delete msg;
      ld->ld_errno=LDAP_SERVER_DOWN;
      return ld->ld_errno;
    }
  }
#ifdef DEBUG
  cerr << *msg << endl;
#endif 
  
  PBER_Stream encoding_stream;
  msg->Encode(encoding_stream);

#ifdef BER_DEBUG
  cerr << encoding_stream << endl;
#endif
  if(encoding_stream.Write(*(ld->socket))) {
    delete msg;
    return LDAP_SUCCESS;
  }
  delete msg;
  ld->ld_errno=LDAP_UNAVAILABLE;
  return ld->ld_errno;
}

int ldap_sasl_bind_s(LDAP *ld, LDAP_CONST char *dn, const char *mechanism, 
		     LDAP_CONST struct berval *cred, LDAPControl **serverctrls,
		     LDAPControl **clientctrls, struct berval **servercredp) {
  int msgid;
  int rv;
  LDAPMessage *result;
  struct berval	*scredp = NULL;

  if( servercredp != NULL ) {
    if (ld->ld_version < LDAP_VERSION3) {
      ld->ld_errno = LDAP_NOT_SUPPORTED;
      return ld->ld_errno;
    }
    *servercredp = NULL;
  }

  rv=ldap_sasl_bind(ld, dn, mechanism, cred, serverctrls, clientctrls, &msgid);
  if(LDAP_SUCCESS!=rv)
    return rv;
  
  if ( ldap_result( ld, msgid, 1, NULL, &result ) == -1 ) {
    return( ld->ld_errno );	/* ldap_result sets ld_errno */
  }

  /* parse the results */
  scredp = NULL;
  if( servercredp != NULL ) {
    rv = ldap_parse_sasl_bind_result( ld, result, &scredp, 0 );
  }
  
  if (rv != LDAP_SUCCESS && rv != LDAP_SASL_BIND_IN_PROGRESS ) {
    ldap_msgfree( result );
    return( rv );
  }

  rv = ldap_result2error(ld, result, 1);

  if (rv == LDAP_SUCCESS || rv == LDAP_SASL_BIND_IN_PROGRESS ) {
    if( servercredp != NULL ) {
      *servercredp = scredp;
      scredp = NULL;
    }
  }
  
  if ( scredp != NULL ) {
    ber_bvfree(scredp); 
  }
  
  return rv;
}
  
int ldap_simple_bind (LDAP *ld, LDAP_CONST char *dn, LDAP_CONST char *passwd){
  int retval;
  struct berval cred;

  if(NULL==passwd){
    cred.bv_val=strndup("",0);
    cred.bv_len=0;
  } else {
    cred.bv_len=strlen(passwd);
    cred.bv_val=strndup(passwd,cred.bv_len);
  }

  if(LDAP_SUCCESS!=ldap_sasl_bind(ld, dn, LDAP_SASL_SIMPLE, &cred, NULL, NULL, &retval)) {
    ldap_memfree(cred.bv_val);
    return (-1);
  }
  ldap_memfree(cred.bv_val);
  return retval;
}
  

int ldap_simple_bind_s (LDAP *ld, LDAP_CONST char *dn, LDAP_CONST char *passwd) {
  int msgid;
  LDAPMessage *result;

  msgid=ldap_simple_bind(ld,dn,passwd);
  if(-1==msgid)
    return ld->ld_errno;
  
  if ( ldap_result( ld, msgid, 1, NULL, &result ) == -1 ) {
    return( ld->ld_errno );	/* ldap_result sets ld_errno */
  }
  
  msgid = ldap_result2error(ld, result, 1);
  
  return msgid;
}

int ldap_bind (LDAP *ld, LDAP_CONST char *dn, LDAP_CONST char *cred, int method){
  int msgid;
  struct berval bvcred;
  if(NULL==ld) {
    return -1;
  }
  if(LDAP_AUTH_SIMPLE==method){
    if(NULL==cred)
      cred="";
    bvcred.bv_len=strlen(cred);
    bvcred.bv_val=strndup(cred,bvcred.bv_len);
    if(LDAP_SUCCESS!=ldap_sasl_bind(ld,dn,LDAP_SASL_SIMPLE,&bvcred,NULL,NULL,&msgid)){
      ldap_memfree(bvcred.bv_val);
      return (-1);
    }
  }else {
    ld->ld_errno=LDAP_AUTH_METHOD_NOT_SUPPORTED;
    return (-1);
  }
  ldap_memfree(bvcred.bv_val);
  return msgid;
}
    
int ldap_bind_s (LDAP *ld, LDAP_CONST char *dn, LDAP_CONST char *cred, int method){
  int msgid;
  LDAPMessage *result;
  if(NULL==ld)
    return LDAP_PARAM_ERROR;
  msgid=ldap_bind(ld, dn, cred,method);
  if(-1==msgid)
    return ld->ld_errno;
  
  if (ldap_result( ld, msgid, 1, NULL, &result ) == -1) {
    return( ld->ld_errno );	/* ldap_result sets ld_errno */
  }
  
  msgid = ldap_result2error(ld, result, 1);
  
  return msgid;
}

/* depricated ...
int ldap_kerberos_bind (LDAP *ld, LDAP_CONST char *dn);

int ldap_kerberos_bind_s (LDAP *ld, LDAP_CONST char *dn);
*/
 
 
int ldap_unbind_ext (LDAP *ld, LDAPControl *serverctrls, LDAPControl *clientctrls) {
  LDAP_LDAPMessage msg;

  msg.m_messageID=++(ld->msg_id);
  msg.m_protocolOp=LDAP_LDAPMessage_protocolOp(LDAP_LDAPMessage_protocolOp::e_unbindRequest);
  
  if(ld->socket->IsOpen()) {
    PBER_Stream encoding_stream;
    msg.Encode(encoding_stream);
    encoding_stream.Write(*ld->socket);
    ld->socket->Shutdown(PChannel::ShutdownReadAndWrite);
  }
  ldap_free(ld);
  return LDAP_SUCCESS;
}

int ldap_unbind (LDAP *ld){
  return ldap_unbind_ext(ld, NULL, NULL);
}

int ldap_unbind_s (LDAP *ld){
  return ldap_unbind_ext(ld, NULL, NULL);
}

//
// End of bind.cxx
//

