/*
 * gkacct.cxx
 *
 * Accounting modules for GNU Gatekeeper. Provides generic
 * support for accounting to the gatekeeper.
 *
 * Copyright (c) 2003, Quarcom FHU, Michal Zygmuntowicz
 *
 * This work is published under the GNU Public License (GPL)
 * see file COPYING for details.
 * We also explicitely grant the right to link this code
 * with the OpenH323 library.
 *
 * $Log: gkacct.cxx,v $
 * Revision 1.1.2.9  2003/12/26 14:00:54  zvision
 * Fixed VC6 warning about too long browser/debug symbols
 *
 * Revision 1.1.2.8  2003/12/21 12:20:35  zvision
 * Fixed conditional compilation
 *
 * Revision 1.1.2.7  2003/11/01 10:35:47  zvision
 * Fixed missing semicolon. Thanks to Hu Yuxin
 *
 * Revision 1.1.2.6  2003/10/27 20:27:52  zvision
 * Improved handling of multiple accounting modules and better tracing
 *
 * Revision 1.1.2.5  2003/10/07 15:22:28  zvision
 * Added support for accounting session updates
 *
 * Revision 1.1.2.4  2003/09/28 14:09:48  zvision
 * Microsecond field added back to h323-xxx-time attributes
 *
 * Revision 1.1.2.3  2003/09/18 01:18:24  zvision
 * Merged accounting code parts from 2.2
 *
 * Revision 1.1.2.2  2003/09/14 21:13:06  zvision
 * Added new FileAcct logger from Tamas Jalsovszky. Thanks!
 * Fixed module stacking. API redesigned.
 *
 * Revision 1.1.2.1  2003/06/19 15:36:04  zvision
 * Initial generic accounting support for GNU GK.
 *
 */
#ifdef HAS_ACCT

#if (_MSC_VER >= 1200)
#pragma warning( disable : 4786 ) // warning about too long debug symbol off
#endif

#include "gk_const.h"
#include "h323util.h"
#include "Toolkit.h"
#include "RasTbl.h"
#include "gkacct.h"

extern const char* CallTableSection;
/// name of the config file section for accounting configuration
const char* GkAcctSectionName = "Gatekeeper::Acct";


PString GkAcctLogger::AsString(
	const PTime& tm
	)
{
	struct tm _tm;
	struct tm* tmptr = &_tm;
	time_t t;
	
	if( (time(&t) != (time_t)(-1))
#ifndef WIN32
		&& (localtime_r(&t,tmptr) == tmptr) )
#else
		&& ((tmptr = localtime(&t)) != NULL) )
#endif
	{
		char buf[10];
		
		buf[0] = 0;
		if( strftime(buf,sizeof(buf),"%Z",tmptr) < 10 )
		{
			buf[9] = 0;
			const PString tzname(buf);
			if( !tzname.IsEmpty() )
			{
				PString s = tm.AsString( "hh:mm:ss.uuu @@@ www MMM d yyyy" );
				s.Replace("@@@",tzname);
				return s;
			}
		}
	}
	
	return tm.AsString( "hh:mm:ss.uuu z www MMM d yyyy" );
}

PString GkAcctLogger::AsString(
	const time_t& tm
	)
{
	struct tm _tm;
	struct tm* tmptr = &_tm;
	time_t t = tm;
	
#ifndef WIN32
	if( localtime_r(&t,tmptr) == tmptr ) {
		char buf[48];
		size_t sz = strftime(buf,sizeof(buf),"%T.000 %Z %a %b %d %Y",tmptr);
#else
	if( (tmptr = localtime(&t)) != NULL ) {
		char buf[96];
		size_t sz = strftime(buf,sizeof(buf),"%H:%M:%S.000 %Z %a %b %d %Y",tmptr);
#endif
		if( sz < sizeof(buf) && sz > 0 )
			return buf;
	}
	
	return PTime(tm).AsString( "hh:mm:ss.uuu z www MMM d yyyy" );
}

GkAcctLogger::GkAcctLogger(
	PConfig& cfg, 
	const PString& moduleName,
	const char* cfgSecName
	) 
	: 
	controlFlag(Required),
	defaultStatus(Fail),
	enabledEvents(AcctAll),
	supportedEvents(AcctNone),
	name(moduleName),
	configSectionName(cfgSecName)
{
	if( configSectionName.IsEmpty() )
		configSectionName = moduleName;
		
	const PStringArray control( 
		cfg.GetString( GkAcctSectionName, moduleName, "" ).Tokenise(";,")
		);

	if( control.GetSize() < 1 )
		PTRACE(1,"GKACCT\tEmpty config entry for module "<<moduleName);
	else if( moduleName *= "default" ) {
		controlFlag = Required;
		defaultStatus = Toolkit::AsBool(control[0]) ? Ok : Fail;
		supportedEvents = AcctAll;
	} else if (control[0] *= "optional")
		controlFlag = Optional;
	else if (control[0] *= "sufficient")
		controlFlag = Sufficient;
	else if (control[0] *= "alternative")
		controlFlag = Alternative;
	
	if( control.GetSize() > 1 )
		enabledEvents = GetEvents(control);
	
	PTRACE(1,"GKACCT\tCreated module "<<moduleName<<" with event mask "
		<<PString(PString::Unsigned,(long)enabledEvents,16)
		);
}

GkAcctLogger::~GkAcctLogger()
{
	PTRACE(1,"GKACCT\tDestroyed module "<<name);
}

int GkAcctLogger::GetEvents(
	const PStringArray& tokens
	) const
{
	int mask = 0;
	
	for( PINDEX i = 1; i < tokens.GetSize(); i++ )
	{
		const PString& token = tokens[i];
		if( token *= "start" )
			mask |= AcctStart;
		else if( token *= "stop" )
			mask |= AcctStop;
		else if( token *= "update" )
			mask |= AcctUpdate;
		else if( token *= "on" )
			mask |= AcctOn;
		else if( token *= "off" )
			mask |= AcctOff;
	}
	
	return mask;
}

GkAcctLogger::Status GkAcctLogger::LogAcctEvent(
	AcctEvent evt, /// accounting event to log
	callptr& call /// additional data for the event
	)
{
	return (evt & enabledEvents & supportedEvents) ? defaultStatus : Next;
}


FileAcct::FileAcct( 
	PConfig& cfg,
	const PString& moduleName,
	const char* cfgSecName
	)
	:
	GkAcctLogger( cfg, moduleName, cfgSecName ),
	cdrFile(NULL)
{
	SetSupportedEvents( FileAcctEvents );	
	
	cdrFilename = cfg.GetString(GetConfigSectionName(),"DetailFile","");
	rotateCdrFile = Toolkit::AsBool(cfg.GetString(
		GetConfigSectionName(),"Rotate","0"
		));

	Rotate();
	if( cdrFile && cdrFile->IsOpen() )
		PTRACE(2,"GKACCT\t"<<GetName()<<" CDR file: "<<cdrFile->GetFilePath());
}

FileAcct::~FileAcct()
{
	PWaitAndSignal lock(cdrFileMutex);
	if( cdrFile ) {
		cdrFile->Close();
		delete cdrFile;
	}
}

GkAcctLogger::Status FileAcct::LogAcctEvent(
	GkAcctLogger::AcctEvent evt, 
	callptr& call
	)
{
	if( (evt & GetEnabledEvents() & GetSupportedEvents()) == 0 )
		return Next;
		
	if( (evt & (AcctStart|AcctUpdate|AcctStop)) && (!call) ) {
		PTRACE(1,"GKACCT\t"<<GetName()<<" - missing call info for event"<<evt);
		return Fail;
	}
	
	PString cdrString;
	
	if( !GetCDRText(cdrString,evt,call) ) {
		PTRACE(2,"GKACCT\t"<<GetName()<<" - unable to get CDR text for event "<<evt
			<<", call no. "<<call->GetCallNumber()
			);
		return Fail;
	}

	PWaitAndSignal lock(cdrFileMutex);
	
	if( cdrFile && cdrFile->IsOpen() ) {
		if( cdrFile->WriteLine(PString(cdrString)) ) {
			PTRACE(5,"GKACCT\t"<<GetName()<<" - CDR string for event "<<evt
				<<", call no. "<<call->GetCallNumber()<<": "<<cdrString
				);
			return Ok;
		} else
			PTRACE(1,"GKACCT\t"<<GetName()<<" - write CDR text for event "<<evt
				<<", call no. "<<call->GetCallNumber()<<" failed: "<<cdrFile->GetErrorText()
				);
	} else
		PTRACE(1,"GKACCT\t"<<GetName()<<" - write CDR text for event "<<evt
			<<", for call no. "<<call->GetCallNumber()<<" failed: CDR file is closed"
			);
		
	return Fail;
}

bool FileAcct::GetCDRText(
	PString& cdrString,
	AcctEvent evt,
	callptr& call
	)
{
	if( (evt & AcctStop) && call ) {
		cdrString = call->GenerateCDR();
		return !cdrString.IsEmpty();
	}
	
	return false;	
}

void FileAcct::Rotate()
{
	PWaitAndSignal lock(cdrFileMutex);

	if( cdrFile ) {
		if( cdrFile->IsOpen() )
			if( rotateCdrFile )
				cdrFile->Close();
			else
				return;
		delete cdrFile;
		cdrFile = NULL;
	}
	
	const PFilePath fn = cdrFilename;
	
	if( rotateCdrFile && PFile::Exists(fn) )
		if( !PFile::Rename(fn,fn.GetFileName() + PTime().AsString(".yyyyMMdd-hhmmss")) )
			PTRACE(1,"GKACCT\t"<<GetName()<<" rotate failed - could not rename"
				" the log file: "<<cdrFile->GetErrorText()
				);
	
	cdrFile = new PTextFile(fn,PFile::WriteOnly, PFile::Create | PFile::DenySharedWrite);
	if (!cdrFile->IsOpen()) {
   	    PTRACE(1,"GKACCT\t"<<GetName()<<" could not open file"
			" required for plain text accounting \""
			<<fn<<"\" :"<<cdrFile->GetErrorText()
			);
		delete cdrFile;
		cdrFile = NULL;
	    return;
	}
	cdrFile->SetPermissions(PFileInfo::UserRead|PFileInfo::UserWrite);
	cdrFile->SetPosition(cdrFile->GetLength());
}


GkAcctFactoryBase::GkAcctFactoryArray* GkAcctFactoryBase::acctLoggerFactories = NULL;

GkAcctFactoryBase::GkAcctFactoryBase(
	const PString& moduleName
	) 
	: 
	name(moduleName)
{
	static GkAcctFactoryArray factories;
	acctLoggerFactories = &factories;
	
	PINDEX i;
	
	for( i = 0; i < acctLoggerFactories->GetSize(); i++ )
	{
		GkAcctFactoryBase *const factory = acctLoggerFactories->GetAt(i);
		if( factory && ((factory == this) || (factory->GetName() == moduleName)) )
		{
			PTRACE(1,"GKACCT\tGlobal list of modules already contains"
				" a factory for "<<name
				);
			break;
		}
	}
	
	if( i >= acctLoggerFactories->GetSize() )
	{
		acctLoggerFactories->SetAt(i,this);
		PTRACE(3,"GKACCT\tRegistered factory for "<<name);
	}
}

GkAcctFactoryBase::~GkAcctFactoryBase()
{
}

PObject::Comparison GkAcctFactoryBase::Compare( 
	const PObject& obj
	) const
{
	if( !obj.IsDescendant(GkAcctFactoryBase::Class()) )
		return LessThan;
	else
		return name.Compare( ((const GkAcctFactoryBase&)obj).name );
}

GkAcctFactoryBase* GkAcctFactoryBase::FindFactoryFor(
	const PString& moduleName
	)
{
	if( acctLoggerFactories != NULL )
		for( PINDEX i = 0; i < acctLoggerFactories->GetSize(); i++ )
		{
			GkAcctFactoryBase *const factory = acctLoggerFactories->GetAt(i);
			if( factory && (factory->GetName() == moduleName) )
				return factory;
		}
	return NULL;
}

GkAcctLoggers::GkAcctLoggers(
	PConfig& cfg
	)
{
	const PStringArray acctLoggerNames( 
		cfg.GetKeys(GkAcctSectionName) 
		);

	for( PINDEX i = 0; i < acctLoggerNames.GetSize(); i++ ) 
	{
		GkAcctFactoryBase* acctFactory 
			= GkAcctFactoryBase::FindFactoryFor(acctLoggerNames[i]);

		if( acctFactory != NULL )
			Append(acctFactory->CreateAcctLogger(cfg));
		else
			PTRACE(1,"GKACCT\tUnknown accounting logger module name "
				<<acctLoggerNames[i]<< " - not created"
				);
	}
	
	acctUpdateInterval = cfg.GetInteger(CallTableSection,"AcctUpdateInterval",0);
}

GkAcctLoggers::~GkAcctLoggers()
{
}

BOOL GkAcctLoggers::LogAcctEvent( 
	GkAcctLogger::AcctEvent evt, /// accounting event to log
	callptr& call, /// additional data for the event
	const time_t now
	)
{
	// check if the acct update logging is enabled 
	// and update interval has passed
	if( evt & GkAcctLogger::AcctUpdate ) {
		if( !call )
			return FALSE;
		if( acctUpdateInterval == 0 
			|| (now - call->GetLastAcctUpdateTime()) < acctUpdateInterval )
			return TRUE;
		else
			call->SetLastAcctUpdateTime(now);
	}
	
	BOOL finalResult = TRUE;
	GkAcctLogger::Status status = GkAcctLogger::Ok;

	// log the event with all configured modules
	for( PINDEX i = 0; i < GetSize(); i++ ) {
		GkAcctLogger* logger = (GkAcctLogger*)GetAt(i);
		if( logger == NULL ) {
			PTRACE(1,"GKACCT\tNULL logger on the list");
			return FALSE;
		}
		
		if( (evt & logger->GetEnabledEvents() & logger->GetSupportedEvents()) == 0 )
			continue;
			
		switch( status = logger->LogAcctEvent( evt, call ) )
		{
		case GkAcctLogger::Ok:
#if PTRACING
			if( PTrace::CanTrace(3) ) {
				ostream& strm = PTrace::Begin(3,__FILE__,__LINE__);
				strm<<"GKACCT\t"<<logger->GetName()<<" logged event "<<evt;
				if( call )
					strm<<" for call no. "<<call->GetCallNumber();
				PTrace::End(strm);
			}
#endif
			break;
			
		default:
			// required and sufficient rules always determine 
			// status of the request
			if( logger->GetControlFlag() == GkAcctLogger::Required
				|| logger->GetControlFlag() == GkAcctLogger::Sufficient ) {
#if PTRACING
				if( PTrace::CanTrace(3) ) {
					ostream& strm = PTrace::Begin(3,__FILE__,__LINE__);
					strm<<"GKACCT\t"<<logger->GetName()<<" failed to log event "<<evt;
					if( call )
						strm<<" for call no. "<<call->GetCallNumber();
					PTrace::End(strm);
				}
#endif
				finalResult = FALSE;
			}
		}
		
		// sufficient and alternative are terminal rules (on log success)
		if( status == GkAcctLogger::Ok
			&& (logger->GetControlFlag() == GkAcctLogger::Sufficient
			|| logger->GetControlFlag() == GkAcctLogger::Alternative) )
			break;
	}

	// a last rule determine status of the the request
	if( finalResult && status != GkAcctLogger::Ok )
		finalResult = FALSE;
		
#if PTRACING
	if( PTrace::CanTrace(2) ) {
		ostream& strm = PTrace::Begin(2,__FILE__,__LINE__);
		strm<<"GKACCT\t"<<(finalResult?"Successfully logged event ":"Failed to log event ")
			<<evt;
		if( call )
			strm<<" for call no. "<<call->GetCallNumber();
		PTrace::End(strm);
	}
#endif
	return finalResult;
}

// append the "default" rule for accounting
GkAcctFactory<GkAcctLogger> DEFAULT_ACCT("default");
// append file based accounting logger to the global list of loggers
GkAcctFactory<FileAcct> FILE_ACCT("FileAcct");

#endif /* HAS_ACCT */
