//////////////////////////////////////////////////////////////////
//
// MulticastGRQ.cxx thread for multicast gatekeeper discovery
//
// This work is published under the GNU Public License (GPL)
// see file COPYING for details.
// We also explicitely grant the right to link this code
// with the OpenH323 library.
//
// History:
// 	990904	initial version (Jan Willamowius)
// 	990924	bugfix: insert GK RAS adress (Jan Willamowius)
// 	990924	bugfix: join multicast group after listen() (Jan Willamowius)
//  000807  bugfix: GRQ multicast replies now go to specified RAS port, 
//					not source port (Denver Trouton)
//
//////////////////////////////////////////////////////////////////


#if (_MSC_VER >= 1200)
#pragma warning( disable : 4291 ) // warning about no matching operator delete
#pragma warning( disable : 4786 ) // warning about too long debug symbol off
#pragma warning( disable : 4800 ) // warning about forcing value to bool
#endif

#include "MulticastGRQ.h"
#include "RasSrv.h"
#include "gk_const.h"
#include "h323util.h"
#include "Toolkit.h"
#ifdef WIN32
#include <winsock.h>
#else
#include <netinet/in.h>
#endif

MulticastGRQ::MulticastGRQ(const PIPSocket::Address & _GKHome, H323RasSrv * _RasSrv)
	: PThread(1000, NoAutoDeleteThread)
{
	GKHome = _GKHome;
	RasSrv = _RasSrv;

	Resume();
}

MulticastGRQ::~MulticastGRQ()
{
}

void MulticastGRQ::Main(void)
{
	PTRACE(1, "GK\tMulticast listener started");

	ReadLock cfglock(ConfigReloadMutex);

	// set socket to multicast
	struct ip_mreq mreq;
	mreq.imr_multiaddr.s_addr = inet_addr(GkConfig()->GetString("MulticastGroup", GK_DEF_MULTICAST_GROUP));
	mreq.imr_interface.s_addr = GKHome;
	// queue length is useless for UDP socket
	MulticastListener.Listen(GKHome, 0, WORD(GkConfig()->GetInteger("MulticastPort", GK_DEF_MULTICAST_PORT)));

	if (setsockopt(MulticastListener.GetHandle(), IPPROTO_IP, IP_ADD_MEMBERSHIP, (char *)&mreq, sizeof(mreq)) < 0)
	{
		PTRACE(1, "GK\tCan't join multicast group.");
		MulticastListener.Close();
		//Suspend();
	} else {
		MulticastListener.SetWriteTimeout(PTimeInterval(500));
	}
	while (MulticastListener.IsOpen())
	{ 
		WORD rx_port;
		PIPSocket::Address rx_addr;
		PBYTEArray rdbuf(4096);
		PPER_Stream rdstrm(rdbuf);

		ConfigReloadMutex.EndRead();
		int iResult = MulticastListener.ReadFrom(rdstrm.GetPointer(), rdstrm.GetSize(), rx_addr, rx_port);
		ConfigReloadMutex.StartRead();

		if (!iResult) {
	    		PTRACE(1, "GK\tMulticast thread: Read error: " << MulticastListener.GetErrorText());
			continue;
		}

		PTRACE(2, "GK\tRead from " << rx_addr << ':' << rx_port);
    
		H225_RasMessage obj_req;   
		if (!obj_req.Decode(rdstrm)) {
			PTRACE(1, "GK\tCouldn't decode message!");
			continue;
		}
#if PTRACING
		if (PTrace::CanTrace(3))
			PTRACE(3, "GK\n" << setprecision(2) << obj_req);
		else
			PTRACE(2, "GK\tReceived " << obj_req.GetTagName());
#endif
		H225_RasMessage obj_rpl;
		switch (obj_req.GetTag())
		{
		case H225_RasMessage::e_gatekeeperRequest:
			PTRACE(1, "GK\tMulticast GRQ Received");
			if ( RasSrv->OnGRQ( rx_addr, obj_req, obj_rpl ) ) {
				H225_GatekeeperRequest & obj_grq = obj_req;
				RasSrv->SendRas( obj_rpl, obj_grq.m_rasAddress, MulticastListener );
			}
			break;
		case H225_RasMessage::e_locationRequest:
			PTRACE(1, "GK\tMulticast LRQ Received");
			if ( RasSrv->OnLRQ( rx_addr, obj_req, obj_rpl ) ) {
				H225_LocationRequest & obj_lrq = obj_req;
				RasSrv->SendRas( obj_rpl, obj_lrq.m_replyAddress, MulticastListener );
			}
			break;
		default:
			PTRACE(1, "GK\tUnknown RAS message received");
			break;      
		}
	}
}

void MulticastGRQ::Close(void)
{
	MulticastListener.Close();
	
	// terminate thread
//	Terminate();
}

