//////////////////////////////////////////////////////////////////
//
// GkClient.cxx
//
// Copyright (c) Citron Network Inc. 2001-2002
//
// This work is published under the GNU Public License (GPL)
// see file COPYING for details.
// We also explicitely grant the right to link this code
// with the OpenH323 library.
//
// initial author: Chih-Wei Huang <cwhuang@linux.org.tw>
// initial version: 02/27/2002
//
//////////////////////////////////////////////////////////////////

#if (_MSC_VER >= 1200)
#pragma warning( disable : 4291 ) // warning about no matching operator delete
#pragma warning( disable : 4786 ) // warning about too long debug symbol off
#pragma warning( disable : 4800 ) // warning about forcing value to bool
#endif

#include "GkClient.h"
#include "gk_const.h"
#include "RasSrv.h"
#include "ProxyChannel.h"
#include <h323pdu.h> 

const char *EndpointSection = "Endpoint";
const char *RewriteE164Section = "Endpoint::RewriteE164";

class GKPendingList : public PendingList {
public:
	GKPendingList(H323RasSrv *rs, int ttl) : PendingList(rs, ttl) {}

	bool Insert(const H225_AdmissionRequest &, const endptr &, int, long callDurationLimit = -1);
	bool ProcessACF(const H225_RasMessage &, int);
	bool ProcessARJ(H225_CallIdentifier &, int);
};

bool GKPendingList::Insert(const H225_AdmissionRequest & arq_ras, const endptr & reqEP, int reqNum, long callDurationLimit)
{
	PWaitAndSignal lock(usedLock);
	arqList.push_back(new PendingARQ(reqNum, arq_ras, reqEP, 0, callDurationLimit));
	return true;
}

bool GKPendingList::ProcessACF(const H225_RasMessage & acf_ras, int reqNum)
{
	PWaitAndSignal lock(usedLock);
	iterator Iter = FindBySeqNum(reqNum);
	if (Iter != arqList.end()) {
		endptr called = RegistrationTable::Instance()->InsertRec(const_cast<H225_RasMessage &>(acf_ras));
		if (called) {
			(*Iter)->DoACF(RasSrv, called);
			Remove(Iter);
		} else {
			PTRACE(2, "GKC\tUnable to add EP for this ACF!");
		}
		return true;
	}
	return false;
}

bool GKPendingList::ProcessARJ(H225_CallIdentifier & callid, int reqNum)
{
	PWaitAndSignal lock(usedLock);
	iterator Iter = FindBySeqNum(reqNum);
	if (Iter != arqList.end()) {
		H225_AdmissionRequest arq;
		endptr ep;
		(*Iter)->GetRequest(arq, ep);
		if (arq.HasOptionalField(H225_AdmissionRequest::e_callIdentifier))
			callid = arq.m_callIdentifier;
		// try neighbors...
		if (!RasSrv->SendLRQ(arq, ep))
			(*Iter)->DoARJ(RasSrv);
		Remove(Iter);
		return true;
	}
	return false;
}

class AlternateGKs {
public:
	AlternateGKs(const PIPSocket::Address &, WORD);
	void Set(const H225_ArrayOf_AlternateGK &);
	bool Get(PIPSocket::Address &, WORD &);
	
private:
	typedef std::multimap<int, H225_TransportAddress> GKList;
	GKList AltGKs;
	GKList::iterator index;
	PIPSocket::Address pgkaddr, pgkport;
};

using std::make_pair;

AlternateGKs::AlternateGKs(const PIPSocket::Address & gkaddr, WORD gkport)
{
	pgkaddr = gkaddr, pgkport = gkport;
}

void AlternateGKs::Set(const H225_ArrayOf_AlternateGK & agk)
{
	AltGKs.clear();
	for (PINDEX i = 0; i < agk.GetSize(); ++i) {
		const H225_AlternateGK & gk = agk[i];
		AltGKs.insert(make_pair(int(gk.m_priority), gk.m_rasAddress));
	}
	index = AltGKs.begin();
}

bool AlternateGKs::Get(PIPSocket::Address & gkaddr, WORD & gkport)
{
	if (AltGKs.size() == 0)
		return false;

	if (index == AltGKs.end()) {
		index = AltGKs.begin();
		// switch back to original GK
		gkaddr = pgkaddr, gkport = pgkport;
		return true;
	}

	const H225_TransportAddress & rasAddress = (index++)->second;
	if (!GetIPAndPortFromTransportAddr(rasAddress, gkaddr, gkport)) {
		PTRACE(3, "GKC\tInvalid AlternateGK Address!" );
		return false;
	}
	return true;
}

class ReceivedLCF {
public:
	ReceivedLCF(PSyncPoint & s) : sync(s) {}
	void OnReceived(const H225_LocationConfirm & lcf) { signalAdr = lcf.m_callSignalAddress, sync.Signal(); }
	H225_TransportAddress GetSignalAdr() const { return signalAdr; }
	
private:
	H225_TransportAddress signalAdr;
	PSyncPoint & sync;
};

class NATHandlingThread : public MyPThread {
public:
	PCLASSINFO ( NATHandlingThread, MyPThread )

	NATHandlingThread(GkClient *, const H225_TransportAddress *);

	// override from class MyPThread
	virtual void Close();
	virtual void Exec();

private:
	bool DetectIncomingCall();

	GkClient *gkClient;
	PIPSocket::Address gkip;
	WORD gkport;
	ReceivedLCF rlcf;
	int retryInterval;
	CallSignalSocket *socket;
};

NATHandlingThread::NATHandlingThread(GkClient *gc, const H225_TransportAddress * addr) : gkClient(gc), gkport(0), rlcf(sync)
{
	if (addr)
		GetIPAndPortFromTransportAddr(*addr, gkip, gkport);
	retryInterval = GkConfig()->GetInteger(EndpointSection, "NATRetryInterval", 60) * 1000;
	socket = 0;
	Resume();
}

void NATHandlingThread::Close()
{
	MyPThread::Close();
	if (socket) {
		gkClient->SendInfo(socket, Q931::CallState_DisconnectRequest);
		socket->Close();
	}
}

void NATHandlingThread::Exec()
{
	if (!gkport) {
		// try to get the signalling address of parent GK by sending an LRQ
		// note the parent GK must use routed mode
		gkClient->SendLRQ(PString(), &rlcf);
		if (!sync.Wait(retryInterval) || !isOpen)
			return;
		H225_TransportAddress addr = rlcf.GetSignalAdr();
		if (!GetIPAndPortFromTransportAddr(addr, gkip, gkport)) {
			sync.Wait(retryInterval);
			return;
		}
	}	
	socket = new CallSignalSocket;
	socket->SetPort(gkport);
	if (socket->Connect(PIPSocket::Address(gkip))) {
		PTRACE(2, "GKC\t" << AsString(gkip, gkport) << " connected, wainting for incoming call");
		while (socket->IsOpen() && isOpen) {
			gkClient->SendInfo(socket, Q931::CallState_IncomingCallProceeding);
			if (DetectIncomingCall()) {
				PTRACE(3, "GKC\tIncoming call detected");
				// incoming call
				RasThread->GetSignalHandler()->Insert(socket);
				socket = 0;
				return;
			}
		}
	}
	delete socket;
	socket = 0;
	if (isOpen)
		sync.Wait(retryInterval);
}

bool NATHandlingThread::DetectIncomingCall()
{
	int retry = GkConfig()->GetInteger(EndpointSection, "NATKeepaliveInterval", 86400); // one day
	while (isOpen && retry-- > 0)
		if (socket->IsReadable(1000)) // one second
			return isOpen;
	return false;
}

// class GkClient
GkClient::GkClient(H323RasSrv *rasSrv) : m_rasSrv(rasSrv)
{
	// initialize these variable for safe
	m_ttl = 0;
	m_callAddr = m_rasAddr = 0;
	m_arqPendingList = 0;
	m_gkList = 0;
	m_rewriteInfo = 0;
	m_natThread = 0;
	m_lcfHook = 0;
	m_password = GkConfig()->GetString(EndpointSection, "Password", "");
	PString gk(GkConfig()->GetString(EndpointSection, "Gatekeeper", "no"));
	if (gk == "no")
		return;
	if (!GetTransportAddress(gk, GK_DEF_UNICAST_RAS_PORT, m_gkaddr, m_gkport)) {
		// unresolvable?
		PTRACE(1, "GKC\tWarning: Can't resolve parent GK " << gk);
		return;
	}

	m_retry = GkConfig()->GetInteger(EndpointSection, "RRQRetryInterval", 10);
	m_gkfailtime = ((m_resend = m_retry) << 6);
	m_arqPendingList = new GKPendingList(rasSrv, GkConfig()->GetInteger(EndpointSection, "ARQTimeout", 2));
	m_gkList = new AlternateGKs(m_gkaddr, m_gkport);
	m_useAltGKPermanent = false;
	m_rewriteInfo = new Toolkit::RewriteData(GkConfig(), RewriteE164Section);

	SendRRQ();
}

GkClient::~GkClient()
{
	RemoveNATThread();

	if (IsRegistered() && Toolkit::AsBool(GkConfig()->GetString(EndpointSection, "UnregisterOnReload", "0")))
		SendURQ();
	delete m_callAddr;
	delete m_rasAddr;
	delete m_arqPendingList;
	delete m_gkList;
	delete m_rewriteInfo;
}

inline void GkClient::SendRas(const H225_RasMessage & ras)
{
	m_rasSrv->SendRas(ras, m_gkaddr, m_gkport);
}

bool GkClient::CheckGKIPVerbose(const PIPSocket::Address & gkip)
{
	if (gkip != m_gkaddr) {
		PTRACE(2, "GKC\tReceived RAS not from my GK? ignore!");
		return false;
	}
	return true;
}

PString GkClient::GetParent() const
{
	return IsRegistered() ?
		AsString(m_gkaddr, m_gkport) + '\t' + m_endpointId :
		PString("not registered");
}

void GkClient::CheckRegistration()
{
	if (m_ttl > 0 && (PTime() - m_registeredTime) > m_ttl)
		SendRRQ();
	if (m_arqPendingList)
		m_arqPendingList->Check();
}

void GkClient::BuildFullRRQ(H225_RegistrationRequest & rrq)
{
	rrq.m_discoveryComplete = FALSE;

	rrq.m_callSignalAddress.SetSize(1);
	rrq.m_callSignalAddress[0] = *m_callAddr;

	rrq.m_terminalType.IncludeOptionalField(H225_EndpointType::e_gatekeeper);

	PINDEX as, p;
	PString t(GkConfig()->GetString(EndpointSection, "Type", "gateway").ToLower());
	if (t[0] == 't') {
		rrq.m_terminalType.IncludeOptionalField(H225_EndpointType::e_terminal);
	} else if (t[0] == 'g') {
		rrq.m_terminalType.IncludeOptionalField(H225_EndpointType::e_gateway);
		PString prefix(GkConfig()->GetString(EndpointSection, "Prefix", ""));
		PStringArray prefixes=prefix.Tokenise(",;", FALSE);
		as = prefixes.GetSize();
		if (as > 0) {
			rrq.m_terminalType.m_gateway.IncludeOptionalField(H225_GatewayInfo::e_protocol);
			rrq.m_terminalType.m_gateway.m_protocol.SetSize(1);
			H225_SupportedProtocols & protocol = rrq.m_terminalType.m_gateway.m_protocol[0];
			protocol.SetTag(H225_SupportedProtocols::e_voice);
			H225_VoiceCaps & voicecap = (H225_VoiceCaps &)protocol;
			voicecap.m_supportedPrefixes.SetSize(as);
			for (PINDEX p = 0; p < as; ++p)
				H323SetAliasAddress(prefixes[p], voicecap.m_supportedPrefixes[p].m_prefix);
		}
		rrq.IncludeOptionalField(H225_RegistrationRequest::e_multipleCalls);
		rrq.m_multipleCalls = TRUE;
	} // else what?

	rrq.IncludeOptionalField(H225_RegistrationRequest::e_terminalAlias);
	PString h323id(GkConfig()->GetString(EndpointSection, "H323ID", m_rasSrv->GetGKHome()));
	PStringArray h323ids=h323id.Tokenise(" ,;\t", FALSE);
	as = h323ids.GetSize();
	rrq.m_terminalAlias.SetSize(as);
	for (p = 0; p < as; ++p)
		H323SetAliasAddress(h323ids[p], rrq.m_terminalAlias[p]);
	m_h323Id = h323ids[0];

	PString e164(GkConfig()->GetString(EndpointSection, "E164", ""));
	PStringArray e164s=e164.Tokenise(" ,;\t", FALSE);
	PINDEX s = e164s.GetSize() + as;
	rrq.m_terminalAlias.SetSize(s);
	for (p = as; p < s; ++p)
		H323SetAliasAddress(e164s[p-as], rrq.m_terminalAlias[p]);
	m_e164 = e164s[0];

	int ttl = GkConfig()->GetInteger(EndpointSection, "TimeToLive", 0);
	if (ttl > 0) {
		rrq.IncludeOptionalField(H225_RegistrationRequest::e_timeToLive);
		rrq.m_timeToLive = ttl;
	}

	H225_VendorIdentifier & vendor = rrq.m_endpointVendor;
	vendor.IncludeOptionalField(H225_VendorIdentifier::e_productId);
	vendor.m_productId = PString(PString::Printf, "GNU Gatekeeper on %s %s %s, %s %s", (const unsigned char*)(PProcess::GetOSName()), (const unsigned char*)(PProcess::GetOSHardware()), (const unsigned char*)(PProcess::GetOSVersion()) ,__DATE__, __TIME__);
	vendor.IncludeOptionalField(H225_VendorIdentifier::e_versionId);
	vendor.m_versionId = "Version " + PProcess::Current().GetVersion();

	rrq.m_keepAlive = FALSE;
}

void GkClient::BuildLightWeightRRQ(H225_RegistrationRequest & rrq)
{
	rrq.m_discoveryComplete = TRUE;

	rrq.IncludeOptionalField(H225_RegistrationRequest::e_endpointIdentifier);
	rrq.m_endpointIdentifier = m_endpointId;
	rrq.IncludeOptionalField(H225_RegistrationRequest::e_gatekeeperIdentifier);
	rrq.m_gatekeeperIdentifier = m_gatekeeperId;
	rrq.m_keepAlive = TRUE;
}

void GkClient::SendRRQ()
{
	PWaitAndSignal lock(rasMutex);
	if (m_resend > m_gkfailtime) { // try alternate gatekeepers
		m_resend = m_gkfailtime;
		if (m_gkList->Get(m_gkaddr, m_gkport)) {
			Unregister(); // always re-register
			m_resend = m_retry;
			PString altgk = AsString(m_gkaddr, m_gkport);
			if (m_useAltGKPermanent)
				Toolkit::Instance()->SetConfig(1, EndpointSection, "Gatekeeper", altgk);
			PTRACE(2, "GKC\tUse Alternate GK " << altgk << (m_useAltGKPermanent ? " permanently" : " temporarily"));
		}
	}
	m_ttl = m_resend * 1000;
	m_resend *= 2;

	if (!IsRegistered()) {
		delete m_callAddr;
		delete m_rasAddr;
		Toolkit::Instance()->GetRouteTable()->InitTable();
		m_callAddr = new H225_TransportAddress(m_rasSrv->GetCallSignalAddress(m_gkaddr));
		m_rasAddr = new H225_TransportAddress(m_rasSrv->GetRasAddress(m_gkaddr));
	}

	H225_RasMessage rrq_ras;
	rrq_ras.SetTag(H225_RasMessage::e_registrationRequest);
	H225_RegistrationRequest & rrq = rrq_ras;

	rrq.m_requestSeqNum = m_rasSrv->GetRequestSeqNum();
	rrq.m_protocolIdentifier.SetValue(H225_ProtocolID);
	rrq.m_discoveryComplete = FALSE;
        rrq.m_rasAddress.SetSize(1);
	rrq.m_rasAddress[0] = *m_rasAddr;
	rrq.IncludeOptionalField(H225_RegistrationRequest::e_supportsAltGK);

	IsRegistered() ? BuildLightWeightRRQ(rrq) : BuildFullRRQ(rrq);
	SetPassword(rrq);
	m_registeredTime = PTime();
	SendRas(rrq_ras);
}

bool GkClient::OnRCF(const H225_RegistrationConfirm & rcf, const PIPSocket::Address & gkip)
{
	if (!CheckGKIPVerbose(gkip))
		return false;

	if (!IsRegistered()) {
		PTRACE(2, "GKC\tRegister successfully to GK " << m_gkaddr);
		m_endpointId = rcf.m_endpointIdentifier;
		m_gatekeeperId = rcf.m_gatekeeperIdentifier;
	}
	if (rcf.HasOptionalField(H225_RegistrationConfirm::e_alternateGatekeeper))
		m_gkList->Set(rcf.m_alternateGatekeeper);
	m_ttl = rcf.HasOptionalField(H225_RegistrationConfirm::e_timeToLive) ?
		(rcf.m_timeToLive - m_retry) * 1000 : 0;
	m_resend = m_retry;

	// NAT handling
	if (rcf.HasOptionalField(H225_RegistrationConfirm::e_nonStandardData))
		if (rcf.m_nonStandardData.m_data.AsString().Find("NAT=") == 0)
			if (!m_natThread)
				m_natThread = new NATHandlingThread(this, (rcf.m_callSignalAddress.GetSize() > 0) ? &rcf.m_callSignalAddress[0] : 0);

	return true;
}

bool GkClient::OnRRJ(const H225_RegistrationReject & rrj, const PIPSocket::Address & gkip)
{
	if (!CheckGKIPVerbose(gkip))
		return false;

	PTRACE(1, "GKC\tRegistration Rejected: " << rrj.m_rejectReason.GetTagName());

	if (rrj.HasOptionalField(H225_RegistrationReject::e_altGKInfo)) {
		Unregister();
		m_gkList->Set(rrj.m_altGKInfo.m_alternateGatekeeper);
		m_useAltGKPermanent = rrj.m_altGKInfo.m_altGKisPermanent;
		m_resend = m_gkfailtime + 1;
		SendRRQ();
	} else if (rrj.m_rejectReason.GetTag() == H225_RegistrationRejectReason::e_fullRegistrationRequired) {
		SendURQ();
		SendRRQ();
	} else
		Unregister();
	return true;
}

void GkClient::SendURQ()
{
	H225_RasMessage urq_ras;
	urq_ras.SetTag(H225_RasMessage::e_unregistrationRequest);
	H225_UnregistrationRequest & urq = urq_ras;
	urq.m_requestSeqNum = m_rasSrv->GetRequestSeqNum();
	urq.IncludeOptionalField(H225_UnregistrationRequest::e_gatekeeperIdentifier);
	urq.m_gatekeeperIdentifier = m_gatekeeperId;
	urq.IncludeOptionalField(H225_UnregistrationRequest::e_endpointIdentifier);
	urq.m_endpointIdentifier = m_endpointId;
	urq.m_callSignalAddress.SetSize(1);
	urq.m_callSignalAddress[0] = *m_callAddr;
	SetPassword(urq);

	Unregister();
	SendRas(urq_ras);
}

bool GkClient::OnURQ(const H225_UnregistrationRequest & urq, const PIPSocket::Address & gkip)
{
	if (gkip != m_gkaddr || m_endpointId != urq.m_endpointIdentifier.GetValue()) // not me?
		return false;

	Unregister();

	bool useAltGK = false;
	if (urq.HasOptionalField(H225_UnregistrationRequest::e_alternateGatekeeper)) {
		m_gkList->Set(urq.m_alternateGatekeeper);
		useAltGK = true;
	}

	switch (urq.m_reason.GetTag())
	{
		case H225_UnregRequestReason::e_reregistrationRequired:
		case H225_UnregRequestReason::e_ttlExpired:
			SendRRQ();
			break;

		default:
			m_registeredTime = PTime();
			m_ttl = m_retry * 1000;
			if (useAltGK)
				m_resend = m_gkfailtime + 1;
			break;
	}
	return true;
}

int GkClient::BuildARQ(H225_AdmissionRequest & arq)
{
	// Don't set call model, let the GK decide it
	arq.RemoveOptionalField(H225_AdmissionRequest::e_callModel); 

	arq.m_endpointIdentifier = m_endpointId;

	arq.IncludeOptionalField(H225_AdmissionRequest::e_srcCallSignalAddress);
	arq.m_srcCallSignalAddress = *m_callAddr;

	arq.IncludeOptionalField(H225_AdmissionRequest::e_gatekeeperIdentifier);
	arq.m_gatekeeperIdentifier = m_gatekeeperId;
	arq.RemoveOptionalField(H225_AdmissionRequest::e_cryptoTokens);
	SetPassword(arq);

	return (arq.m_requestSeqNum = m_rasSrv->GetRequestSeqNum());
}

void GkClient::SendARQ(const H225_AdmissionRequest & arq, const endptr & reqEP, long callDurationLimit )
{
	H225_RasMessage arq_ras;
	arq_ras.SetTag(H225_RasMessage::e_admissionRequest);
	H225_AdmissionRequest & arq_obj = arq_ras;
	arq_obj = arq; // copy and then modify
	int reqNum = BuildARQ(arq_obj);

	RewriteE164(arq_obj.m_srcInfo, true);

	m_arqPendingList->Insert(arq, reqEP, reqNum,callDurationLimit);
	SendRas(arq_ras);
}

void GkClient::SendARQ(const H225_Setup_UUIE & setup, unsigned crv, const callptr & call)
{
	H225_RasMessage arq_ras;
	arq_ras.SetTag(H225_RasMessage::e_admissionRequest);
	H225_AdmissionRequest & arq = arq_ras;
	int reqNum = BuildARQ(arq);

	bool answer = call->GetCalledParty();
	arq.m_callReferenceValue = crv;
	arq.m_conferenceID = setup.m_conferenceID;
	if (setup.HasOptionalField(H225_Setup_UUIE::e_callIdentifier)) {
		arq.IncludeOptionalField(H225_AdmissionRequest::e_callIdentifier);
		arq.m_callIdentifier = setup.m_callIdentifier;
	}
	if (setup.HasOptionalField(H225_Setup_UUIE::e_sourceAddress)) {
		arq.m_srcInfo = setup.m_sourceAddress;
		if (!answer)
			RewriteE164(arq.m_srcInfo, true);
	} else {
		// no sourceAddress privided in Q.931 Setup?
		// since srcInfo is mandatory, set my aliases as the srcInfo
		arq.m_srcInfo.SetSize(1);
		H323SetAliasAddress(m_h323Id, arq.m_srcInfo[0]);
		if (!m_e164) {
			arq.m_srcInfo.SetSize(2);
			H323SetAliasAddress(m_e164, arq.m_srcInfo[1]);
		}
	}
	if (setup.HasOptionalField(H225_Setup_UUIE::e_destinationAddress)) {
		arq.IncludeOptionalField(H225_AdmissionRequest::e_destinationInfo);
		arq.m_destinationInfo = setup.m_destinationAddress;
		RewriteE164(arq.m_destinationInfo, true);
	}
	arq.m_answerCall = answer;
	arq.m_bandWidth = call->GetBandWidth();

	m_arqAnsweredList[reqNum] = call;
	SendRas(arq_ras);
}

void GkClient::OnACF(const H225_RasMessage & acf_ras, const PIPSocket::Address & gkip)
{
	if (!CheckGKIPVerbose(gkip))
		return;

	const H225_AdmissionConfirm & acf = acf_ras;
	int reqNum = acf.m_requestSeqNum.GetValue();
	if (m_arqPendingList->ProcessACF(acf_ras, reqNum))
		return;

	// an ACF to an answer ARQ
	iterator Iter = m_arqAnsweredList.find(reqNum);
	if (Iter == m_arqAnsweredList.end()) {
		PTRACE(2, "GKC\tUnknown ACF, ignore!");
		return;
	}
	callptr call = Iter->second;
	if (!call->GetCalledParty()) {
		endptr called = RegistrationTable::Instance()->InsertRec(const_cast<H225_RasMessage &>(acf_ras));
		call->SetCalled(called);
	}
	m_arqAnsweredList.erase(Iter);
}

void GkClient::OnARJ(const H225_RasMessage & arj_ras, const PIPSocket::Address & gkip)
{
	if (!CheckGKIPVerbose(gkip))
		return;

	const H225_AdmissionReject & arj = arj_ras;
	if (arj.HasOptionalField(H225_AdmissionReject::e_altGKInfo)) {
		m_gkList->Set(arj.m_altGKInfo.m_alternateGatekeeper);
		m_useAltGKPermanent = arj.m_altGKInfo.m_altGKisPermanent;
		m_resend = m_gkfailtime + 1;
		SendRRQ();
	} else if (arj.m_rejectReason.GetTag() == H225_AdmissionRejectReason::e_callerNotRegistered) { // reregister again
		Unregister();
		SendRRQ();
	}
	
	int reqNum = arj.m_requestSeqNum.GetValue();
	H225_CallIdentifier callid;
	if (m_arqPendingList->ProcessARJ(callid, reqNum))
		return;

	// an ARJ to an answer ARQ
	iterator Iter = m_arqAnsweredList.find(reqNum);
	if (Iter != m_arqAnsweredList.end()) {
		callptr call = Iter->second;
		PTRACE(2, "GKC\tGot ARJ for call " << call->GetCallNumber() << ", reason " << arj.m_rejectReason.GetTagName());
		// TODO: routeCallToGatekeeper
		call->Disconnect(true);
		m_arqAnsweredList.erase(Iter);
	} else {
		PTRACE(2, "GKC\tUnknown ARJ, ignore!");
	}
}

bool GkClient::OnDRQ(const H225_DisengageRequest & drq, const PIPSocket::Address & gkip)
{
	if (m_gkaddr == gkip && drq.m_endpointIdentifier.GetValue() == m_endpointId) {
		if (callptr call = drq.HasOptionalField(H225_DisengageRequest::e_callIdentifier) ? CallTable::Instance()->FindCallRec(drq.m_callIdentifier) : CallTable::Instance()->FindCallRec(drq.m_callReferenceValue))
			call->Disconnect(true);
		return true;
	}
	return false;
}

void GkClient::SendDRQ(H225_RasMessage & drq_ras)
{
	H225_DisengageRequest & drq = drq_ras;
	drq.IncludeOptionalField(H225_DisengageRequest::e_gatekeeperIdentifier);
	drq.m_gatekeeperIdentifier = m_gatekeeperId;
	drq.m_endpointIdentifier = m_endpointId;
	SetPassword(drq);
	SendRas(drq_ras);
}

void GkClient::SendLRQ(const PString & alias, ReceivedLCF *rlcf)
{
	H225_RasMessage lrq_ras;
	lrq_ras.SetTag(H225_RasMessage::e_locationRequest);
	H225_LocationRequest & lrq = lrq_ras;
	lrq.m_requestSeqNum.SetValue(m_rasSrv->GetRequestSeqNum());
	lrq.m_replyAddress = m_rasSrv->GetRasAddress(m_gkaddr);
	lrq.m_destinationInfo.SetSize(1);
	H323SetAliasAddress((alias.IsEmpty() ? m_h323Id : alias), lrq.m_destinationInfo[0]);
	lrq.IncludeOptionalField(H225_LocationRequest::e_endpointIdentifier);
	lrq.m_endpointIdentifier = m_endpointId;
	lrq.IncludeOptionalField(H225_LocationRequest::e_gatekeeperIdentifier);
	lrq.m_gatekeeperIdentifier = m_gatekeeperId;
	lrq.IncludeOptionalField(H225_LocationRequest::e_hopCount);
	lrq.m_hopCount = 1; // don't forward it again
	SetPassword(lrq);
	SendRas(lrq_ras);
	m_lcfHook = rlcf;
}

bool GkClient::OnLCF(const H225_LocationConfirm & lcf)
{
	if (m_lcfHook) {
		m_lcfHook->OnReceived(lcf);
		m_lcfHook = 0;
		return true;
	}
	return false;
}

bool GkClient::RewriteE164(H225_AliasAddress & alias, bool fromInternal)
{
	if (alias.GetTag() != H225_AliasAddress::e_dialedDigits) 
		return false;
		        
	PString e164 = H323GetAliasAddressString(alias);

	bool changed = RewriteString(e164, fromInternal);
	if (changed)
		H323SetAliasAddress(e164, alias);

	return changed;
}

bool GkClient::RewriteE164(H225_ArrayOf_AliasAddress & aliases, bool fromInternal)
{
	bool changed = false;
	for (PINDEX i = 0; i < aliases.GetSize(); ++i)
		if (RewriteE164(aliases[i], fromInternal))
			changed = true;
	return changed;
}

bool GkClient::RewriteE164(Q931 & SetupMesg, H225_Setup_UUIE & Setup, bool fromInternal)
{
	unsigned plan, type;
	PString Number;

	bool result = false;
	if (fromInternal) {
		bool r1 = SetupMesg.GetCallingPartyNumber(Number, &plan, &type);
		if (r1 && (result = RewriteString(Number, true)))
			SetupMesg.SetCallingPartyNumber(Number, plan, type);
		if ((!r1 || result) && Setup.HasOptionalField(H225_Setup_UUIE::e_sourceAddress))
			result = RewriteE164(Setup.m_sourceAddress, true) || result;
	} else {
		bool r1 = SetupMesg.GetCalledPartyNumber(Number, &plan, &type);
		if (r1 && (result = RewriteString(Number, false)))
			SetupMesg.SetCalledPartyNumber(Number, plan, type);
		if ((!r1 || result) && Setup.HasOptionalField(H225_Setup_UUIE::e_destinationAddress))
			result = RewriteE164(Setup.m_destinationAddress, false) || result;
	}
	return result;
}

bool GkClient::RewriteString(PString & alias, bool fromInternal) const
{
	if (!m_rewriteInfo)
		return false;
	for (PINDEX i = 0; i < m_rewriteInfo->Size(); ++i) {
		PString prefix, insert;
		if (fromInternal) {
			insert = m_rewriteInfo->Key(i);
			prefix = m_rewriteInfo->Value(i);
		} else {
			prefix = m_rewriteInfo->Key(i);
			insert = m_rewriteInfo->Value(i);
		}
		int len = prefix.GetLength();
		if (len == 0 || strncmp(prefix, alias, len) == 0){
			PString result = insert + alias.Mid(len);
			PTRACE(2, "GKC\tRewritePString: " << alias << " to " << result);
			alias = result;
			return true;
		}
	}
	return false;
}

void GkClient::SetClearTokens(H225_ArrayOf_ClearToken & clearTokens, const PString & id)
{
#ifdef OPENH323_NEWVERSION
	clearTokens.RemoveAll();
	H235AuthCAT auth;
	// avoid copying for thread-safely
	auth.SetLocalId((const char *)id);
	auth.SetPassword((const char *)m_password);
	H225_ArrayOf_CryptoH323Token dumbTokens;
	auth.PrepareTokens(clearTokens, dumbTokens);
#endif
}

void GkClient::SetCryptoTokens(H225_ArrayOf_CryptoH323Token & cryptoTokens, const PString & id)
{
//	auth.SetLocalId(m_h323Id);
	auth.SetLocalId((const char*)id);
	auth.SetPassword((const char*)m_password);
#ifdef OPENH323_NEWVERSION
	H225_ArrayOf_ClearToken dumbTokens;
	auth.PrepareTokens(dumbTokens, cryptoTokens);
#else
	auth.Prepare(cryptoTokens); 
#endif
}

bool GkClient::SendInfo(CallSignalSocket *socket, int state)
{
	Q931 information;
	information.BuildInformation(0, false);
	PBYTEArray buf, epid(m_endpointId, m_endpointId.GetLength(), false);
	information.SetIE(Q931::FacilityIE, epid);
	information.SetCallState(Q931::CallStates(state));
	information.Encode(buf);
	return socket->TransmitData(buf);
}

void GkClient::Unregister()
{
	m_endpointId = PString();
	RemoveNATThread();
}

void GkClient::RemoveNATThread()
{
	if (m_natThread) {
		m_natThread->Destroy();
		m_natThread = 0;
	}
}
