//////////////////////////////////////////////////////////////////
//
// BroadcastListen.cxx thread for listening to broadcasts
// (only needed on some OSs)
//
// This work is published under the GNU Public License (GPL)
// see file COPYING for details.
// We also explicitely grant the right to link this code
// with the OpenH323 library.
//
// History:
// 	991016	initial version (Jan Willamowius)
//
//////////////////////////////////////////////////////////////////


#include "BroadcastListen.h"
#include "RasSrv.h"
#include "Toolkit.h"
#include "gk_const.h"
#ifdef WIN32
#include <winsock.h>
#else
#include <netinet/in.h>
#endif

BroadcastListen::BroadcastListen(H323RasSrv * _RasSrv)
	: PThread(1000, NoAutoDeleteThread)
{
	RasSrv = _RasSrv;

	Resume();
}

BroadcastListen::~BroadcastListen()
{
}

void BroadcastListen::Main(void)
{
	PTRACE(1, "GK\tBroadcast listener started");

	ReadLock cfglock(ConfigReloadMutex);

	// queue length is useless for UDP socket
	BroadcastListener.Listen(0,
		WORD(GkConfig()->GetInteger("UnicastRasPort", GK_DEF_UNICAST_RAS_PORT)),
		PSocket::CanReuseAddress);

	if (BroadcastListener.IsOpen()) {
		BroadcastListener.SetWriteTimeout(PTimeInterval(500));
	} else {
		PTRACE(1,"GK\tBind to broadcast listener failed!");
	}
	while (BroadcastListener.IsOpen())
	{ 
		WORD rx_port;
		PIPSocket::Address rx_addr;

		PBYTEArray rdbuf(4096);
		PPER_Stream rdstrm(rdbuf);
		
		ConfigReloadMutex.EndRead();
		int iResult = BroadcastListener.ReadFrom(rdstrm.GetPointer(), rdstrm.GetSize(), rx_addr, rx_port);
		ConfigReloadMutex.StartRead();

		if (!iResult) {
	    		PTRACE(1, "GK\tBroadcast thread: Read error: " << BroadcastListener.GetErrorText());
			continue;
		}

		PTRACE(2, "GK\tRead from " << rx_addr << ':' << rx_port);
    
		H225_RasMessage obj_req;   
		if (!obj_req.Decode(rdstrm)) {
			PTRACE(1, "GK\tCouldn't decode message!");
			continue;
		}
#if PTRACING
		if (PTrace::CanTrace(3))
			PTRACE(3, "GK\n" << setprecision(2) << obj_req);
		else
			PTRACE(2, "GK\tReceived " << obj_req.GetTagName());
#endif
		H225_RasMessage obj_rpl;

		switch (obj_req.GetTag())
		{
		case H225_RasMessage::e_gatekeeperRequest:    
			PTRACE(1, "GK\tBroadcast GRQ Received");
			if ( RasSrv->OnGRQ( rx_addr, obj_req, obj_rpl ) ) {
				H225_GatekeeperRequest & obj_grq = obj_req;
				RasSrv->SendRas( obj_rpl, obj_grq.m_rasAddress, BroadcastListener );
			}
			break;
		// It is said Cisco Call Manager sends RRQ via broadcast
		// But is it valid to use broadcasting RRQ?
		case H225_RasMessage::e_registrationRequest:    
			PTRACE(1, "GK\tBroadcast RRQ Received");
			if ( RasSrv->OnRRQ( rx_addr, obj_req, obj_rpl ) )
				RasSrv->SendReply( obj_rpl, rx_addr, rx_port, BroadcastListener );
			break;
		case H225_RasMessage::e_locationRequest :
			PTRACE(1, "GK\tBroadcast LRQ Received");
			if ( RasSrv->OnLRQ( rx_addr, obj_req, obj_rpl ) ) {
				H225_LocationRequest & obj_lrq = obj_req;
				RasSrv->SendRas( obj_rpl, obj_lrq.m_replyAddress, BroadcastListener );
			}
			break;
		default:
			PTRACE(1, "GK\tUnknown RAS message broadcast received");
			break;      
		}
	}
}

void BroadcastListen::Close(void)
{
	BroadcastListener.Close();
	
	// terminate thread
//	Terminate();
}

