#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2004 Free Software Foundation
#
# FILE:
# GRRun.py
#
# DESCRIPTION:
# Class that loads and populates a report
#
# NOTES:
#


from gnue.reports import VERSION
from gnue.common.apps import GDebug
from gnue.common.apps.GClientApp import GClientApp
from gnue.common.rpc import client
from gnue.reports.base.GREngine import GREngine
from gnue.reports.base import GRFilters, GRExceptions
from GRRunUI import GRRunUI
from gnue.reports.base.GRReportInfo import GRReportInfo
from gnue.reports.base.GRConfig import ConfigOptions
import string, sys
import tempfile


def quietprint(*message):
  print string.join(message)

def quietprint_nl(*message):
  print string.join(message),

def _quietprintSuppress(*message):
  GDebug.printMesg(1,string.join(message))


class GRRun(GClientApp):

  #
  # GClientApp() overrides
  #
  VERSION = VERSION
  COMMAND = "gnue-reports"
  NAME = "GNUe Reports Client"
  COMMAND_OPTIONS = [
      [ 'destination_type', 'D', 'destination-type', 1, 'file', 'type',
          'This specifies how the report should be output. The currently '
          'supported values for <type> are file [default], printer, email, '
          'and fax. Note that printer, email, and fax are sent via the '
          'server\'s machine, not the client\'s machine.  To '
          '\nNOTE: Only file, printer, and email are currently implemented!' ],
      [ 'destination', 'd', 'destination', 1, '-', 'dest',
          'Where should the report be output to?  The value of this '
          'depends on the destination type (e.g., if sending to printer, '
          'then -d specifies the printer name; if sending via email, then '
          '-d specifies the email address.) If <dest> is "-", then output is '
          'sent to stdout -- NOTE: when sending to stdout, also use the -q '
          '[--quiet] option or you may get junk in your output stream. '
          'NOTE: Currently the default value is "-" -- this may change once GNUe '
          'Reports is formally released!'],
      [ 'destination_options', None, 'destination-options', 1, '', 'opts',
          'Options to pass to the destination process. Available options are '
          'specific to the type of destination. '
          'Example: \'--destination-options "-o nobanner" \''],
      [ 'filter', 'f', 'filter', 1, 'raw', 'filt',
          'Select the filter to be used to process report output.  <filt> '
          'is the name of the filtering process as defined on the Report '
          'Server machine. If not specified, the "raw" filter is used '
          '(i.e., no filtering takes place.)'],
      [ 'list_filters', None, 'list-filters', 0, 0, None,
          'List the available [predefined] filters available to GNUe Reports'],
      [ 'filter_options', 'F', 'filter-options', 1, '', 'opts',
          'Options to pass to the filter process. Available options are '
          'specific to the filter.  --list-filters will list available filters '
          'and their options.  Example: \'--filter-options "paper=letter margin=1,1,1,1" \''],
      [ 'sortoption', 's', 'sort', 1, '', 'sort',
          'Select the "sort-option" used to sort the report. '],
      [ 'batch', 'b', 'batch', 0, 0, None,
          'Run GNUe Reports in batch mode -- i.e., no login information or '
          'runtime parameters will be prompted.  If required parameters are '
          'not provided, then Reports will terminate with an error'],
      [ 'quiet', 'q', 'quiet', 0, 0, None,
          'Run GNUe Reports in quiet mode -- i.e., display no output. '
          'NOTE: if --debug-level is specified, then suppressed text will '
          'be output as debugging information (at debug level 1)'],
      [ 'ui', 'U', 'ui', 1, 'text', 'interface',
          'Run GNUe Reports using a specific UI.  Possible UI\'s are text and gui.'],
      [ 'comment', 'C', 'comment', 0, 0, None,
          'Include structural comments in the XML output stream. '
          'Useful for debugging .grd files.'],
      [ 'standalone', 'S', 'standalone', 0, 0, None,
          'Create a standalone, single-use server instance.  Use this option in a '
          'non-client/server environment or in a debugging/development environment. '
          'NOTE: Until the Reports Server is mature this flag has the opposite meaning'],
      [ 'exclude_xml', 'X', 'exclude-xml', 0, 0, None,
          'Do not output GNUe Report''s runtime XML markup information.  If specified, '
          'then the GRD''s layout section will be processed and output '
          'as-is; i.e., without any additional information added by '
          'GNUe Reports.'], ]
  SUMMARY = \
     "GNUe Reports is the primary reporting agent of the GNU Enterprise system."
  USAGE = GClientApp.USAGE + " file [param1=val] [param2=val] ..."
  USE_DATABASE_OPTIONS = 1

  #
  # Initialize the class
  #
  def __init__(self):
    GClientApp.__init__(self, application='reports',defaults=ConfigOptions)
    self.configurationManager.registerAlias('gConfigReports', 'reports')


  def run(self):

    #
    # Are we doing a simple help-like operation?
    #

    # assign report file from 1st free argument
    try:
      reportfile = self.ARGUMENTS[0]
    except:
      self.handleStartupError ("No Report Definition File Specified.")

    # ...list available filters?
    if self.OPTIONS["list_filters"]:
      self.listFilters(reportfile)
      sys.exit()


    #
    # Get the user supplied parameters
    #
    userParameters = self.getCommandLineParameters(self.ARGUMENTS[1:])
    

    #
    # Get the user supplied sortoption
    #
    if len(self.OPTIONS["sortoption"]):
      sortoption = self.OPTIONS["sortoption"]
    else:
      sortoption = None


    filter = self.OPTIONS['filter']
    destination = self.OPTIONS["destination"]
    destinationType = self.OPTIONS["destination_type"]
    destinationOptions = self.OPTIONS["destination_options"]


    #
    # Set quiet mode
    #
    if self.OPTIONS["quiet"]:
      global quietprint
      global quietprint_nl
      quietprint = _quietprintSuppress
      quietprint_nl = _quietprintSuppress


    #
    # Will we do a UI engine
    #
    if not self.OPTIONS['batch']:
      runui = GRRunUI(self,reportInfo=self.getReportInformation(reportfile),
                      ui = self.OPTIONS['ui'],
                      userParameters=userParameters,
                      destination=destination,
                      destinationType=destinationType,
                      filter=filter,
                      sortoption=sortoption
                      )

      lh = runui.getLoginHandler()
      self.getConnectionManager().setLoginHandler(lh)

      runui.run()

      userParameters = runui.getParameters()
      destination = runui.getDestination()
      destinationType = runui.getDestinationType()
      filter = runui.getFilter()
      sortoption = runui.getSortOption()

    else:
      lh = NoLoginHandler()
      self.getConnectionManager().setLoginHandler(lh)

    #
    # Now, run an engine instance
    #

    # HACK: for now standalone means server
    if not self.OPTIONS["standalone"]:
     try:
       rep_engine = GREngine(self.getConnectionManager())
       rep_engine.processReport(reportfile,
                                destination = destination,
                                destinationType = destinationType,
                                destinationOptions = destinationOptions,
                                filter = filter,
                                batch = self.OPTIONS['batch'],
                                filterOptions = self.OPTIONS['filter_options'],
                                parameters=userParameters,
                                sortoption=sortoption,
                                includeStructuralComments=self.OPTIONS["comment"],
                                omitGNUeXML=self.OPTIONS["exclude_xml"])
       
     except GRExceptions.ReportMarkupException, msg:
       self.handleStartupError (msg)
     except GRExceptions.ReportUnreadable, msg:
       self.handleStartupError (msg)
     except IOError, msg:
       self.handleStartupError (msg)
       
    else:
      try:
        # TODO: load this paramter from somewhere
        params = {'host': 'localhost',
                  'port': 8766,
                  'transport': 'http' }
        interface = 'xmlrpc'
        
        adapter    = client.attach(interface, params)
        rep_server = adapter.request("GReportServer")
        rep_engine = rep_server.requestReportEngine()
        rep_engine.setDestination(destination, destinationType, destinationOptions)
        rep_engine.setFilter(filter, self.OPTIONS['filter_options'])

        # HACK:
        if sortoption==None:
          sortoption=0
        # HACKEND:
        
        rep_engine.processReport(reportfile,                                 
                                 userParameters,
                                 sortoption,
                                 self.OPTIONS["comment"],
                                 self.OPTIONS["exclude_xml"])
        rep_engine.clear()
        
      except GRExceptions.ReportMarkupException, msg:
        self.handleStartupError (msg)
      except GRExceptions.ReportUnreadable, msg:
        self.handleStartupError (msg)
      except IOError, msg:
        self.handleStartupError (msg)
       
    sys.exit()

  #
  # List filters available on the server
  #
  def listFilters(self, location):

    self.printVersion()

    reportInfo = self.getReportInformation(location)
    namespace = reportInfo.getNamespace()
    
    if namespace:
      filters = reportInfo.getFilters()
      filters.sort()
    else: 
      filters = []

    filters.insert(0,('raw',{'description':'Do not process the XML with a filter. Emit raw XML.',
                       'engine': 'raw' }))

    processed = []
    maxlen = 0
    for filter, properties in filters:
      filter = filter.split(':')[-1]
      maxlen = max(maxlen, len(filter))
      try:
        descr = properties['description']
        if not len(descr):
          raise KeyError
      except KeyError:
        descr = "%s output filter (no description available)" % filter
      processed.append((filter, descr))




    print 'Available filters for "%s" reports:' % namespace and namespace.split(':')[-1] or 'Raw'
    print
    for filter, descr in processed:
      print "  %s  %s" % (filter.ljust(maxlen+2), descr)
    print


  #
  # Get a report's information
  #
  # TODO: This needs to support GNUe Reports Server
  # TODO: when the time is right.
  def getReportInformation(self, location):
    return GRReportInfo(location)



#
# Login handler for --batch mode
#
from gnue.common.datasources import GLoginHandler
class NoLoginHandler(GLoginHandler.LoginHandler):

  defaults = {'_username': None, '_password': None}

  def getLogin(self, loginData, error):
    val = {}
    for prompt in loginData[2]:
       if self.defaults.has_key(prompt[0])  and self.defaults[prompt[0]]:
         val[prompt[0]] = self.defaults[prompt[0]]
       else:
         val[prompt[0]] = None
    return val

  def destroyLoginDialog(self):
    pass

if __name__ == '__main__':
  GRRun().run()

