#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2004 Free Software Foundation
#
# FILE:
# ToolBar.py
#
# DESCRIPTION:
# A generic UIdriver base for common message-based GUI toolkits.
#
# NOTES:
#

import string
from types import StringType

from gnue.forms.uidrivers._base import UserActions
  
class ToolBar:


  ####################################################################
  #
  # These methods should be superclassed
  # by the toolkit-specific ToolBar
  #
  
  def init(self): 
    pass

  # Add a menu item (action)
  def addAction(self, name, parent, userAction):
    pass
    
  # Add a separator
  def addSeparator(self, parent): 
    pass
    
  # Enable a menu item
  def enableItem(self, item):
    pass
  
  # Disable a menu item
  def disableItem(self, item): 
    pass

  def startingItem(self, item): 
    pass

  def endingItem(self, item): 
    pass
        

  ####################################################################
  #
  # These methods should not be touched
  # by the toolkit-specific ToolBar.
  # 
  # Or, in the words of the great poet, 
  # M.C. Hammer,  "Can't touch this!"
  #
      
  def __init__(self, uidriver, container, form): 
    self.driver = uidriver
    self.container = container
    self.form = form
    self.enabled = 1
    self.__actionMap = {}
    
    self.finalize()
    
  def disable(self): 
    self.enabled = 0
    
  def finalize(self): 
    if self.enabled: 

      # Let specific drivers create themselves
      self.init()
      
      for item in _DEFAULT_TOOLBAR: 
        if item == None: 
          self.addSeparator()
        else: 
          name, contents = item
          userAction = UserActions.userActionMap[contents]
          self.__actionMap[userAction.event] = self.addAction(name, userAction)
          eventListeners = { 
                  'can%s' %  userAction.event: self.__canRequestEvent, 
                  'cannot%s' %  userAction.event: self.__cannotRequestEvent }
                  
          if userAction.canToggle: 
            eventListeners.update( {
                    'begin%s' %  userAction.event: self.__beginRequestEvent, 
                    'end%s' %  userAction.event: self.__endRequestEvent } )
                    
          self.driver.registerEventListeners(eventListeners)
        
  def __canRequestEvent(self, event): 
    item = self.__actionMap[event.__event__[3:]]
    self.enableItem(item)

  def __cannotRequestEvent(self, event): 
    e = event.__event__[6:]
    item = self.__actionMap[e]
    userAction = UserActions.userActionMap[e]
    if userAction.canToggle: 
      return
    self.disableItem(item)
        
  def __beginRequestEvent(self, event): 
    item = self.__actionMap[event.__event__[5:]]
    self.startingItem(item)

  def __endRequestEvent(self, event): 
    item = self.__actionMap[event.__event__[3:]]
    self.endingItem(item)

      
      
          
####################################################################
#
#

_DEFAULT_TOOLBAR =  (
    (_('Save Changes'), "COMMIT"), 
    (_('Insert New Record'),"NEWRECORD"), 
    (_('Delete Current Record'),'MARKFORDELETE'), 
    None, 
    (_('First Record'), 'FIRSTRECORD'), 
    (_('Previous Record'), "PREVRECORD"), 
    (_('Next Record'), "NEXTRECORD"), 
    (_('Last Record'), 'LASTRECORD'), 
    (_('Go to record...'), 'JUMPPROMPT'), 
    None, 
    (_('Enter Search Mode'), "ENTERQUERY"), 
    (_('Perform Search'), "EXECQUERY"), 
    None,       
    (_('Revert Changes'), "ROLLBACK"), 
    None, 
    (_('Close'), "EXIT") )
       
