#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2002-2004 Free Software Foundation
#
# FILE:
# GFKeyMapper.py
#
# DESCRIPTION:
"""
Handles physical to logical key mapping for GNUe Forms.
Also performs logical key to Events mapping.
"""
#
# NOTES:
#


import string, sys
from gnue.common.apps import GDebug


class InvalidKeystrokeName(StandardError):
  """
  Exception class used by the loadUserKeyMap function.
  """
  pass

 
##
##
##
class BaseKeyMapper:
  """
  A basic key mapper. This will normally
  serve most UI's input needs.
  """
  
  def __init__(self, userKeyMap):
    self.__functionMap = userKeyMap
    self.__keyTranslations = {}

  #
  # This should be called by the UI class to set their
  # key mappings (i.e., we need to know what wxPython
  # thinks F1 is, etc)
  #
  def setUIKeyMap(self, keyset):
    self.__keyTranslations = keyset
    self._translateUserKeyMap()

  #
  # Install a key/event mapping.
  # This can be called by the app,
  # but more than likely, the app
  # will call loadUserKeyMap which
  # calls this.
  #
  def setUserKeyMap(self, keymap):
    self.__functionMap = keymap
    self._translateUserKeyMap()


  #
  # Given a hash of the form:
  # { 'PREVBLOCK': 'Ctrl-F1' }
  # decode the key events and save into our usermap
  #
  def loadUserKeyMap(self, dict):
    usermap = {}
    
    for event in dict.keys():
      val = string.upper(dict[event])

      # Save any actual '+' keystrokes
      if val[:1] == '-':
        val = 'NEG' + val[1:]
      if val[-1:] == '-':
        val = val[:-2] + 'NEG'
      val = string.replace(string.replace(val,' ',''),'--','-NEG')

      keys = string.split(val,'-')

      base = None
      shifted = False
      meta = False
      ctrl = False

      for key in keys:
        if key in ('CTRL','CONTROL'):
          ctrl = True
        elif key in ('META','ALT'):
          meta = True
        elif key in ('SHFT','SHIFT'):
          shifted = True
        elif vk.__dict__.has_key(key):
          base = vk.__dict__[key]
        elif len(key) == 1:
          # TODO: This might not be necessary
          key = ord(key)
        else:
          raise InvalidKeystrokeName, \
            _("Invalid keystroke id '%s' in keymap for '%s'") % (key,event)

      if base is None:
        raise InvalidKeystrokeName, \
          _("Invalid keystroke combination '%s' in keymap for '%s'") % \
              (dict[event],event)

      usermap[(base, shifted, ctrl, meta)] = string.upper(event)


    # Now, load any default keys they forgot to bind
    for key in DefaultMapping.keys():
      if  DefaultMapping[key] not in usermap.values() and \
          not usermap.has_key(key):
        usermap[key] = DefaultMapping[key]

    # Just in case...
    usermap.update( {
          (vk.TAB,      False, False, False) : 'NEXTENTRY',
          (vk.ENTER,    False, False, False) : 'NEXTENTRY',
          (vk.RETURN,   False, False, False) : 'NEXTENTRY'} )

    self.setUserKeyMap(usermap)


  #
  # Return the (virtual) keystroke assigned to
  # a given event. (e.g., if event = 'PREVBLOCK'
  # then return vk.PAGEUP)
  #
  def getEventKeystroke(self, event):
    for key in self.__functionMap.keys():
      if self.__functionMap[key] == event:
        return key
    return None


  #
  # Same as getEventKeystroke except that
  # a text representation is return ('F1')
  # instead of the numerical code. Useful
  # for building "Menus" or "Help Screens"
  #
  def getEventKeystrokeRepr(self, event, metamappings={}, separator="+"):
     keystroke = self.getEventKeystroke(event)

     if keystroke is None:
       return None

     base, shifted, ctrl, meta = keystroke
     rv = ""
     if ctrl:
       rv += metamappings.get("CONTROL","Ctrl") + separator
     if meta:
       rv += metamappings.get("META","Alt") + separator
     if shifted:
       rv += metamappings.get("SHIFT","Shift") + separator

     v = reverseLookup(base)

     return "%s%s" % (rv, metamappings.get(v,v))


  #
  # Translate a keystroke into an event.
  # (keystroke is the UI-specific keystroke,
  # not our virtual keys.)  Returns None if
  # the keystroke isn't tied to an event.
  #
  # This needs to stay as simple as possible
  # as it gets called for each keystroke
  #
  def getEvent(self, basekey, shift=False, ctrl=False, meta=False):
    try:
      return self._translatedUserKeyMap[(basekey, shift, ctrl, meta)]
    except KeyError:
      return None

  #
  # Used internally to create a quick lookup
  # hash for time-sensitive methods.
  #
  def _translateUserKeyMap(self):
    self._translatedUserKeyMap = {}
    for keys in self.__functionMap.keys():
      try:
        base, sh, ctrl, meta = keys
        self._translatedUserKeyMap[(self.__keyTranslations[base],
                   sh, ctrl, meta)] = self.__functionMap[keys]
      except KeyError:
        pass



#####################################################################
#
#
class _VirtualKeys:
  """
  Create a container class for the
  Virtual Key definitions... this
  is to keep our namespace clean.
  """
  
  def __init__(self):
    self.F1 = -999
    self.F2 = -998
    self.F3 = -997
    self.F4 = -996
    self.F5 = -995
    self.F6 = -994
    self.F7 = -993
    self.F8 = -992
    self.F9 = -991
    self.F10 = -990
    self.F11 = -989
    self.F12 = -988
    self.INSERT = -987
    self.DELETE = -986
    self.HOME = -985
    self.END = -984
    self.PAGEUP = -983
    self.PAGEDOWN = -982
    self.UP = -981
    self.DOWN = -980
    self.LEFT = -979
    self.RIGHT = -978
    self.TAB = -977
    self.ENTER = -976
    self.RETURN = -975
    self.BACKSPACE = -974
    self.X = -973
    self.V = -972
    self.C = -971
    self.A = -970


#
# ..and the application will only
#   need one instance, so create one.
#
vk = _VirtualKeys()


#
# Given a keycode value (e.g., -999), return
# the text representation as a string (e.g., 'F1')
#
def reverseLookup(keyvalue):
  """
  Given a keycode value (e.g., -999), return
  the text representation as a string (e.g., 'F1')
  """

  # This is done for efficiency of real-time lookups;
  # i.e., we don't often do reverseLookups, but a
  # regular lookup must happen as efficiently as
  # possible as a lookup happens each time a key is
  # pressed!

  for key in dir(vk):
    if getattr(vk,key) == keyvalue:
      return key
  return None


#####################################################################
#
# Default event mappings
#
DefaultMapping = {

      # (Key, Shifted, Ctrl'd, Meta/Alt'd)
      (vk.A,        False, True, False) : 'SELECTALL',
      (vk.C,        False, True, False) : 'COPY',
      (vk.V,        False, True, False) : 'PASTE',
      (vk.X,        False, True, False) : 'CUT',
      (vk.PAGEUP,   True,  False, False) : 'JUMPROWSUP',
      (vk.PAGEDOWN, True,  False, False) : 'JUMPROWSDOWN',
      (vk.PAGEUP,   False, True,  False) : 'PREVPAGE',
      (vk.PAGEDOWN, False, True,  False) : 'NEXTPAGE',
      (vk.PAGEUP,   False, False, False) : 'PREVBLOCK',
      (vk.PAGEDOWN, False, False, False) : 'NEXTBLOCK',
      (vk.TAB,      False, False, False) : 'NEXTENTRY',
      (vk.ENTER,    False, False, False) : 'NEXTENTRY',
      (vk.RETURN,   False, False, False) : 'NEXTENTRY',
      (vk.TAB,      True,  False, False) : 'PREVENTRY',
      (vk.LEFT,     False, False, False) : 'CURSORLEFT',
      (vk.RIGHT,    False, False, False) : 'CURSORRIGHT',
      (vk.END,      False, False, False) : 'CURSOREND',
      (vk.HOME,     False, False, False) : 'CURSORHOME',
      (vk.LEFT,     True,  False, False) : 'SELECTLEFT',
      (vk.RIGHT,    True,  False, False) : 'SELECTRIGHT',
      (vk.END,      True,  False, False) : 'SELECTTOEND',
      (vk.HOME,     True,  False, False) : 'SELECTTOHOME',
      (vk.BACKSPACE,False, False, False) : 'BACKSPACE',
      (vk.INSERT,   False, False, False) : 'MODETOGGLE',
      (vk.DELETE,   False, False, False) : 'DELETE',
      (vk.UP,       False, False, False) : 'PREVRECORD',
      (vk.DOWN,     False, False, False) : 'NEXTRECORD',
      (vk.UP,       False, True,  False) : 'FIRSTRECORD',
      (vk.DOWN,     False, True,  False) : 'LASTRECORD',
      (vk.F2,       False, False, False) : 'JUMPPROMPT',
      (vk.F5,       False, False, False) : 'MARKFORDELETE',
      (vk.F6,       False, False, False) : 'COMMIT',
      (vk.F8,       False, False, False) : 'ENTERQUERY',
      (vk.F8,       True,  False, False) : 'COPYQUERY',
      (vk.F9,       False, False, False) : 'EXECQUERY',
      (vk.F9,       True,  False, False) : 'CANCELQUERY',
      (vk.F11,      False, False, False) : 'ROLLBACK',
      (vk.F12,      False, False, False) : 'NEWRECORD',
      (vk.ENTER,    True,  False, False) : 'NEWLINE',
   }


#
# The application will only
# need one instance, so create one.
#
KeyMapper = BaseKeyMapper(DefaultMapping)
