# GNU Enterprise Forms - UI Driver - Base class for user interfaces
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: UIdriver.py 7064 2005-02-25 13:55:58Z kilo $

import sys, os, dircache

from gnue.common import events
from gnue.common.definitions.GObjects import *
from gnue.common.definitions.GRootObj import GRootObj
from gnue.common.utils.FileUtils import dyn_import
from gnue.common.apps import errors

from gnue.forms.GFForm import *

# =============================================================================
# Exceptions
# =============================================================================

class ImplementationError (errors.SystemError):
  def __init__ (self, drivername, method):
    msg = u_("The UI-Driver %(name)s has no implementation of %(method)s") \
          % {'name': drivername,
             'method': method}
    errors.SystemError.__init__ (self, msg)


# -----------------------------------------------------------------------------
# Guess a default iconset for the current environment
# -----------------------------------------------------------------------------

def guessDefaultIconset ():
  """
  Guess the default iconset to use in this environment
  """

  # TODO: This is *very* crude logic
  try:
    import mac
    return 'macosx'
  except ImportError:
    try:
      import _winreg
      # TODO: Is there a registry key that tells the version of windows?
      # TODO: Windows 95/98 should use 'default', not 'winxp'
      return 'winxp'
    except ImportError:
      try:
        import posix
        if os.environ.get('KDE_FULL_SESSION',''):
          return 'kde3'
        ds = os.environ.get('DESKTOP_SESSION','')
        if ds[:3] == 'kde':
          return 'kde3'
        elif ds[:5] == 'gnome':
          return 'gnome'
      except ImportError:
        return 'default'



# =============================================================================
# Base class for UI drivers
# =============================================================================

class GFUserInterfaceBase (GRootObj, events.EventAware):

  default_iconset = guessDefaultIconset ()

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__(self, eventHandler, name = "Undefined", disableSplash = None,
               parentContainer = None, moduleName = None):
    """
    Constructors of descendants must bring their UI into a state where it can
    perform simple tasks like showMessage () or showException ().
    """
    # moduleName is here only for Designer to be able to pass it in
    #when debugging a form.

    GRootObj.__init__(self, 'uiDriver', None, None)

    self.name           = name
    self._type          = 'UIDriver'
    self._disableSplash = disableSplash

    # Used when forms are embedded in navigator. What parentContainer is
    # depends on the individual UIdriver and Navigator.
    self._parentContainer = parentContainer

    events.EventAware.__init__(self, eventHandler)

    # Mapping - internal record status : displayed on status bar
    self.statusValues = {'saved'     :'OK',
                         'modified'  :'MOD',
                         'deleted'   :'DEL',
                         'query'     :'QRY',
                         }

    # So other modules can get to the ui
    __builtins__ ['forms_ui']  = self

    # Multi-form support
    self._formNameToUIForm = {}             # Holds links the to the top level
                                            # UIForm objects held in memory
    self._formNameToGFForm = {}             # Holds links the to the top level
                                            # UIForm objects held in memory
    self._gfObjToUIWidget = {}              # dictionary of driver specific 
                                            # UIfoo widgets with the GFObj used
                                            # as the key

    self._gfObjToToolkitWidgets = {}        # the GFObj to UI widget cross ref


    # Our local "clipboard"
    self.__clipboard = None

    # Import and register supported widgets in UI driver
    self._supportedWidgets = {}

    if moduleName:  #We are within Designer, running a form
      basedir = os.path.dirname(sys.modules[moduleName].__file__)
    else:   #Normal case
      basedir  = os.path.dirname (sys.modules [self.__module__].__file__)
    
    uiDriver = os.path.basename (basedir)
    basedir +='/widgets/'
    for widgetName in dircache.listdir (basedir):
      try:
        # _xxx are abstract classes, .xxx are hidden dirs (like ".svn")
        if widgetName [0] != '_' and widgetName [0] != '.':
          if os.path.isdir (basedir + widgetName):
            # Directories: import as module
            widget = dyn_import ('gnue.forms.uidrivers.%s.widgets.%s' \
                                 % (uiDriver, widgetName))
          elif os.path.isfile (basedir + widgetName):
            # Files: import only .py (not .pyc or .pyo or anything else)
            (widgetName, ext) = os.path.splitext (widgetName)
            if ext == '.py':
              widget = dyn_import ('gnue.forms.uidrivers.%s.widgets.%s' \
                                   % (uiDriver, widgetName))

          try:
            self._supportedWidgets [widget.configuration ['provides']] = widget
          except Exception, mesg:
            raise ImportError, mesg

      except ImportError, mesg:
        gDebug (1, "%s.widgets.%s doesn't appear to be a valid ui widget" \
                   % (uiDriver, widgetName))
        gDebug (1, ' --> %s' % mesg)

    self._uiFocusWidget = None

    # Dialogs required by base
    #
    # TODO: Once <dialog> works we may be able to do away with these
    self._DIALOGS = {}


  # ---------------------------------------------------------------------------
  # Initialize
  # ---------------------------------------------------------------------------

  def initialize (self):
    """
    This function gets called on activation of a GFInstance. It performs the
    last steps needed for handling forms. Descendants of this class must put
    their special init stuff into _initialize (), which gets called
    automatically from this function.
    """

    # register incomming events
    self.registerEventListeners ({
            'gotoENTRY'        : self.switchFocus,
            'updateENTRY'      : self.updateEntry,
            'updateEntryEditor': self.updateEntryEditor,
            'gotoPAGE'         : self.gotoPage,
            'exitApplication'  : self.exitApplication,
            'setTitle'         : self.setTitle,

            # User feedback events from the virtual form to the UI
            'beginWAIT'        : self.beginWait,
            'endWAIT'          : self.endWait,
            'formALERT'        : self.formAlert,
            'uiUPDATESTATUS'   : self.updateStatusBar,

            # Clipboard contents
            'setCLIPBOARD'     : self.setClipboardContents,
            'getCLIPBOARD'     : self.getClipboardContents,

            # Printout
            'uiPRINTOUT'       : self.printout,
          })

    # now call the implementation specific initialization
    self._initialize ()


  # ---------------------------------------------------------------------------
  # Driver-specific initializations
  # ---------------------------------------------------------------------------

  def _initialize (self):
    """
    Descendants of this class will override this function to do all
    initializations.
    """
    pass



  # ---------------------------------------------------------------------------
  # Build the user interface
  # ---------------------------------------------------------------------------

  def _buildUI (self, object, formName):
    try:
      #
      # Look up the parent GObj of the current obj
      #
      if object._type == "GFPage":
        parent = object.findParentOfType (None)
      else:
        parent = object.getParent ()

      # find the ui widget that corrosponds with that parent
      uiParent = self
      if parent and self._gfObjToUIWidget.has_key (parent):
        uiParent = self._gfObjToUIWidget [parent]

      event = self._updateCreateEvent (events.Event ('CreateUIWidget', None,
                   object       = object,
                   parent       = uiParent,
                   #container   = self.currentWidget[0],
                   textWidth    = self.textWidth,
                   textHeight   = self.textHeight,
                   widgetWidth  = self.widgetWidth,
                   widgetHeight = self.widgetHeight,
                   interface    = self,
                   initialize   = 1))

      supported = self._supportedWidgets [object._type]
      uiWidget  = supported.configuration ['baseClass'] (event)
      uiWidget._form = self._form

      #
      # Add to the cross refernce
      #
      self._gfObjToUIWidget [object] = uiWidget

      #
      # If the current object is a GFForm then add it to the
      # dictionary.
      #

      if object._type == 'GFForm':
        self._formNameToUIForm [formName] = uiWidget
        self._formNameToGFForm [formName] = object

      #
      # GFObject to list of widget set specific widgets
      # associated with it
      #
      # Note: The list would be empty on hidden fields
      if not uiWidget.widgets == []:
        self._gfObjToToolkitWidgets [object] = uiWidget.widgets

      gDebug (5, "Widget is %s" % uiWidget.widgets)

    except KeyError:
      gDebug(4, "KeyError in _buildUI")


  #############################################################################
  #
  # Public Interface
  #
  # The interface exposed to the forms backend
  #
  #


  # ---------------------------------------------------------------------------
  # Build a form from a GObj tree
  # ---------------------------------------------------------------------------

  def buildForm (self, form, formName):

    self._form = form

    # Create the UI from the GFForm passed in
    form.walk (self._buildUI, formName = formName)
    self._gfObjToUIWidget [form].phaseInit ()

    self._formNameToUIForm [formName]._gfObjToToolkitWidgets = \
        self._gfObjToToolkitWidgets
    self._formNameToUIForm [formName]._gfObjToUIWidget = self._gfObjToUIWidget
    self._formNameToUIForm [formName]._form = form


  # ---------------------------------------------------------------------------
  # Activate a form
  # ---------------------------------------------------------------------------

  def activateForm (self, formName, modal = 0):

    self._form = self._formNameToGFForm[formName]
    self._UIform = self._formNameToUIForm[formName]
    self._activateForm(self._UIform, modal)



  #############################################################################
  #
  # EVENT FUNCTIONS
  #
  # Handles incoming events and calls UI instance specific functions to
  # execute the actions.  These events come from the forms back end.
  #


  # ---------------------------------------------------------------------------
  # Moves the focus to a specific UI widget
  # ---------------------------------------------------------------------------

  def switchFocus (self, event):

    object = event.object
    if object: # Some pages might not have any widgets that can be active
      if self._uiFocusWidget:
        self._uiFocusWidget.loseFocus ()
      self._uiFocusWidget = self._gfObjToUIWidget [object]
      self._uiFocusIndex = object._visibleIndex
      self._uiFocusWidget.indexedFocus (object._visibleIndex)
      self.dispatchEvent ('beginEDITMODE', object, _form = object._form)


  # ---------------------------------------------------------------------------
  # Update all visible toolkit widgets tied to a specific GFObject
  # ---------------------------------------------------------------------------

  def updateEntry (self, event):

    if event.data.hidden or event.data._type == 'GFButton':
      return

    entry   = event.data
    field   = entry._field
    handler = entry._displayHandler
    prehandling     = handler.editing
    handler.editing = 0
    index = entry._visibleIndex
    block = entry._block
    currentRecord = block._resultSet.getRecordNumber ()

    # Fill the prior spots
    for count in range (index):
      if currentRecord - (index - count) < 0:
        continue

      record = block._resultSet.getRecord (currentRecord - (index - count))
      value  = handler.getDisplayFiller (record.getField (field.field))

      gDebug (5, "UPD-ENTRY %s prior: '%s' (%s)" % (entry, value, count))
      self._gfObjToUIWidget [entry].setValue (value, count)

    # Fill current spot
    value = handler.getDisplayFiller (entry.getValue ())
    gDebug (5, "UPD-ENTRY %s current: '%s' (%s)" % (entry, value, index))
    self._gfObjToUIWidget [entry].setValue (value, index)

    # Fill trailing spots
    #
    # You must skip the matching index but you do not want to just add 1 to
    # count as the formulas would then be off
    count = index

    lastRow = block._resultSet.getRecordCount ()
    while count < int (entry._rows):
      if count != index:
        cr  = currentRecord + (count - index)
        rec = block._resultSet.getRecord (cr)

        if rec is None:
          # Blank the displayed widget
          value = handler.getDisplayFiller (None)
          # Don't ask... but it's needed
          lastRow -= 1
        else:
          value = handler.getDisplayFiller (rec.getField (field.field))

        gDebug (5, "UPD-ENTRY %s trail: '%s' (%s)" % (entry, value, count))
        self._gfObjToUIWidget [entry].setValue (value, count, cr <= lastRow)

      count += 1

    handler.editing = prehandling

  # ---------------------------------------------------------------------------
  # Update the displayed value and set the cursor position for a GFObject
  # ---------------------------------------------------------------------------

  def updateEntryEditor (self, event):

    index  = event.object._visibleIndex
    widget = self._gfObjToUIWidget [event.object]
    gDebug (5, "UPD-ENTRY-ED %s '%s' %s" % (event.object, event.display, index))
    widget.setValue (event.display, index)
    widget.setCursorPosition (event.cursor, index)

    if event.selection != None:
      selection1, selection2 = event.selection
      widget.setSelectedArea (selection1, selection2, index)


  # ---------------------------------------------------------------------------
  # Get the clipboard contents
  # ---------------------------------------------------------------------------

  def getClipboardContents (self, event):

    gDebug (5, "Getting clipboard '%s'" % self.__clipboard)
    event.__result__ = "%s" % self.__clipboard


  # ---------------------------------------------------------------------------
  # Set the clipboard contents
  # ---------------------------------------------------------------------------

  def setClipboardContents (self, event):

    gDebug (5, "Setting clipboard '%s'" % event.text)
    self.__clipboard = "%s" % event.text


  #############################################################################
  #
  # Optional Functions
  #
  # UIDrivers can override the following functions
  #


  # ---------------------------------------------------------------------------
  # Enter wait state
  # ---------------------------------------------------------------------------

  def beginWait (self, event):
    """
    Called whenever forms goes into a "wait" state in which user cannot
    interact with interface (e.g., while waiting for a query or a commit)
    """
    pass


  # ---------------------------------------------------------------------------
  # Leave wait state
  # ---------------------------------------------------------------------------

  def endWait (self, event):
    """
    Called whenever forms leaves a wait state
    """
    pass


  # ---------------------------------------------------------------------------
  # Form has told the application to close so call the UIs private exit routine
  # ---------------------------------------------------------------------------

  def exitApplication (self, event):
    self._exit (event._formName)


  # ---------------------------------------------------------------------------
  # Update the widget creation event
  # ---------------------------------------------------------------------------

  def _updateCreateEvent(self, event):
    """
    Can be used by UI drivers to add more attributes to the event that creates
    a widget. Called by the _stdConstructor during the building of the UI
    """
    return event


  # ---------------------------------------------------------------------------
  # Perform the default printout function
  # ---------------------------------------------------------------------------

  def printout(self, event):
    """
    Perform the default printout/"screen print" function for this uidriver
    """
    pass


  # ---------------------------------------------------------------------------
  # Show the last exception
  # ---------------------------------------------------------------------------

  def showException (self, group = None, name = None, message = None,
      detail = None):
    """
    This function shows the last exception raised. Exceptions of the user group
    are delegated to the showMessage-function.
    """
    if (group, name, message, detail) == (None, None, None, None):
      (group, name, message, detail) = errors.getException ()
    if group == 'user':
      self.showMessage (message, kind = 'Error', title = _("GNUe Message"))
    else:
      self._showException (group, name, message, detail)

  # ---------------------------------------------------------------------------
  # Show a message of a given kind
  # ---------------------------------------------------------------------------

  def showMessage (self, message, kind = 'Info', title = None, cancel = False):
    """
    This function calls the UI driver's implementation of the _showMessage
    method. See _showMessage () for a detailed doc.
    """
    return self._showMessage (message, kind, title, cancel)


  # ---------------------------------------------------------------------------
  # gotoPage
  # ---------------------------------------------------------------------------

  def gotoPage (self, event):

    uiForm = self._gfObjToUIWidget [event._form]._uiForm
    uiPage = self._gfObjToUIWidget [event.data]
    uiForm.gotoPage (uiPage)


  # ---------------------------------------------------------------------------
  # Calls the UIForm's _setStatusBar function to update status
  # ---------------------------------------------------------------------------

  def updateStatusBar (self, event):

    status = None
    if event.recordStatus != None:
      status = self.statusValues [event.recordStatus]

    insertValue = None
    if event.insertMode:
      insertValue = event.insertMode and 'INS' or 'OVR'

    self._gfObjToUIWidget [event._form]._uiForm._setStatusBar (event.tip,
                                                           status,
                                                           insertValue,
                                                           event.currentRecord,
                                                           event.maxRecord,
                                                           event.currentPage,
                                                           event.maxPage)



  #############################################################################
  #
  # Required Functions
  #
  # UIDrivers must implement the following features


  # ---------------------------------------------------------------------------
  # Activate a form
  # ---------------------------------------------------------------------------

  def _activateForm (self, form, modal):

    raise ImplementationError, (self.name, '_activateForm')


  # ---------------------------------------------------------------------------
  # Exit the application
  # ---------------------------------------------------------------------------

  def _exit(self):

    raise ImplementationError, (self.name, '_exit')


  # ---------------------------------------------------------------------------
  # Set the form's displayed title
  # ---------------------------------------------------------------------------

  def setTitle (self, event):
    """
    set the form's displayed title
    """
    pass


  # ---------------------------------------------------------------------------
  # Abstract: make sure a descendant of this class has a _showException method
  # ---------------------------------------------------------------------------

  def _showException (self, group, name, message, detail):
    """
    This function must be overriden by a descendant UI driver class. It get's
    called for all exceptions except UserErrors. It's purpose is to display
    an exception. 
    @param group: Group of the exception like 'system', 'admin', 'application'
    @param name: Name of the exception, i.e. 'KeyError', 'FoobarError', ...
    @param message: Message of the exception
    @param detail: Detail of the exception, i.e. a traceback
    """
    raise ImplementationError, (self.name, '_showException')


  # ---------------------------------------------------------------------------
  # Implementation of showMessage ()
  # ---------------------------------------------------------------------------
  
  def _showMessage (self, message, kind = 'Info', title = None, cancel = False):
    """
    This function must be implemented by a UI driver class and shows a messages
    of a given kind.

    @param message: the text of the message
    @param kind: type of the message. Valid types are 'Info', 'Warning',
        'Question', 'Error'
    @param title: title of the message
    @param cancel: If True something like a cancel button should be available
    @return: True if the Ok-, Close-, or Yes-button was pressed, False if the
        No-button was pressed or None if the Cancel-button was pressed.
    """
    raise ImplementationError, (self.name, '_showMessage')
