#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# FILE:
# pyro/ServerAdapter.py
#
# DESCRIPTION:
# Set of classes that implement the Pyro server driver for GNUe Comm.
#
# NOTES:
# Requires pyro from pyro.sf.net
#
# Server Parameters:
#
#    port        The port that the service is located on
#


from gnue.common.apps import errors, GDebug
from gnue.common.rpc import server
from gnue.common.rpc.drivers import Base
from gnue.common.rpc.drivers._helpers import ObjectLibrarian, ObjectEnabler

import string, sys, os, posixpath, urllib, socket

def __initplugin__ ():
  try:
    import Pyro.naming
    import Pyro.core
    from Pyro.protocol import getHostname
    from Pyro.errors import PyroError,NamingError

  except ImportError:
    tmsg = _("\nUnable to load Pyro.  To use the Pyro interface, \n"
             "please install pyro from:\n    http://pyro.sf.net/")
    raise GComm.AdapterInitializationError, tmsg


# Mapping from GRPC's datatype to XML-RPC datatypes
_datatypeMap = {
  'integer': 'int',
  'string': 'string',
  'boolean': 'boolean',
  'date': 'dateTime.iso8601',
  'number': 'double',
  'base64': 'base64',
  'binary': 'base64'
}

##############################################################################
#
# ServerAdapter
#
class ServerAdapter(Base.Server):

  def __init__(self, rpcdef, bindings, params):
    try:
      self._port = params['port']
    except KeyError:
      pass

    if params.has_key('bindto'):
      self._bindto = params['bindto']
    else:
      self._bindto = '' # bind to all interfaces

    if params.has_key('allowed_hosts'):
      # TODO: Remove spaces, etc.
      self._allowed_hosts = string.split(params['allowed_hosts'],',')
    else:
      self._allowed_hosts = [''] # allow access from all hosts

    if params.has_key('loglevel'):
      self._loglevel = params['loglevel']
    else:
      self._loglevel = 0

    self.pyro_group = ':GComm'
    # TODO: set pyro_group to other name, depending on grpc

    # initialize pyro server
    Pyro.core.initServer()
    Pyro.config.PYRO_NS_DEFAULTGROUP=self.pyro_group
    self.daemon = Pyro.core.Daemon()

    # locate the name server
    if not hasattr(self,'ns'):
      locator = Pyro.naming.NameServerLocator()
      GDebug.printMesg(9,'PYRO: searching for Naming Service...')
      self.ns = locator.getNS()
      GDebug.printMesg(9,'Naming Service found at %s (%s) %s.' % \
                       (self.ns.URI.address,
                        (Pyro.protocol.getHostname(self.ns.URI.address) or '??'),
                        self.ns.URI.port))

    # make sure our namespace group exists
    try:
      self.ns.createGroup(self.pyro_group)
    except NamingError:
      pass

    self.daemon.useNameServer(self.ns)

    self.mapObjects(rpcdef,bindings)
    

  #
  # Return an exception
  #
  def raiseException(self, exception, message, event=None):
    pass


  #                                                                        #
  #                                                                        #
  ##########################################################################
  def serve(self):

    # enter the service loop.
    GDebug.printMesg(9,'Starting pyro daemon loop.')
    
    try:
      # daemon.setTimeout(5)
      self.daemon.requestLoop()
    except KeyboardInterrupt:
      print 'shutting down gracefully.'
      # todo: disconnect all obj.
      #daemon.disconnect(obj)
      self.daemon.shutdown()
      print 'Exiting.'


  #
  # Create an internal "service directory"
  #
  def mapObjects(self, object, bindings, parent=None):

    # For servicable objects, maintain a complete "path" for reference
    if object._type in ('RpService','RpMethod','RpObject'):
      if parent and hasattr(parent,'_path'):
        object._path = "%s.%s" % (parent._path, object.name)
      else:
        object._path = object.name


    ##    
    ## Add binding informations to the objects
    ##
    ## services are static objects and
    ## objects  are dynamic ones
    ##
    if hasattr(object,'binding'):      

      # the direct mapping
      if bindings.has_key(object.binding):
        GDebug.printMesg(9,'GNURPC Binding Information:');
        GDebug.printMesg(9,' * %s is bound to \n * %s' % \
                         (object.binding,bindings[object.binding]))

        # for services create an "static" object 
        if object._type == 'RpService':
          object._realbinding=bindings[object.binding]()
          # TODO: Clearup this "static" object

        else:
          
          # in all other cases just save the binding information
          object._realbinding=bindings[object.binding]          
                
      else:
        # RpObject normaly don't need binding information, because
        # they are bound dynamicly
        if object._type != 'RpObject':
          
          print u_("Missing Binding information. Please add binding "
                   "information for %s") % object.binding

        
    # care for bindings in all Services
    if object._type == 'RpService':
      if hasattr(object,'_realbinding'):
        pass  # nothing to do
      else:
        if parent._type == 'RpService':
          try:
            object._realbinding=getattr(parent._realbinding,\
                                         object.name)

            GDebug.printMesg(9,'* %s is bound to \n * %s' % \
                             (object._path,object._realbinding))
          except:
            tmsg = u_("GNURPC cannot bind service '%(name)s' to service "
                      "'%(destination)s'") \
                   % {'name'       : object.name,
                      'destination': parent.name}
            raise AttributeError, tmsg
        elif parent._type == 'RpGnuRpc':
          pass
        else:
          tmsg = u_("GNURPC cannot bind service '%(name)s' to service "
                    "'%(destination)s'") \
                 % {'name'       : object.name,
                    'destination': parent._type}
          raise AttributeError, tmsg

      # create binding for service
      try:
        self.ns.deleteGroup(object._path)
      except:
        pass
      
      self.ns.createGroup(object._path)
      
      self._bindDelegateto(object._path+'.self',object._realbinding)

          

    # Compute binding for methods and for attributs
    # both are direct lins to the specific object
    # 
    # the dispatcher has to distinguish methods and
    # objects by testing if they are callable
    if (object._type == 'RpMethod') or \
       (object._type == 'RpAttribute'):
      
      # check for the binding
      if hasattr(object,'_realbinding'):
        bindto=object._realbinding
      else:
        if parent._type == 'RpService':
          try:
            bindto=getattr(parent._realbinding,object.name)
            GDebug.printMesg(9,'* %s is bound to \n * %s' % \
                             (object._path,bindto))
          except:
            tmsg = u_("GNURPC cannot bind method/attribut '%(name)s' to "
                      "service '%(service)s'") \
                   % {'name'   : object.name,
                      'service': parent.name}
            raise AttributeError, tmsg
            pass
        else:
          bindto=None

    if object._type == 'RpMethod':    
      self.addRpMethod(object,parent,bindto)

    #
    # Add all attribute methods to our directory..
    # XML-RPC doesn't support "Attributes", so an
    # attribute is exposed as a pair of get_<name>
    # and set_<name> methods.
    #
    if object._type == 'RpAttribute':
      self.addRpAttribut(object,parent,bindto)


    # Now, map our children
    for child in object._children:
      self.mapObjects(child, bindings, object)

  def addRpMethod(self,object,parent,binding):
    if binding!=None:
      self._bindDelegateto(object._path,binding)
    
  def _bindDelegateto(self,name,binding):
    # unregister method
    try:
      self.ns.unregister(name)
    except NamingError:
      pass
    except:
      pass

    # register using delegation
    proxy=Pyro.core.ObjBase()
    proxy.delegateTo(binding)
    #print 'bindto: %s (%s,%s)' % (name,binding, type(binding))
    self.daemon.connect(proxy,name)

  def addRpAttribut(self,object,parent,binding):

    return
    if not object.readonly:
      # Add the set_* directory entry
      self.daemon.connect({'%s.set_%s' % \
                              (parent._path, object.name):\
                              binding})
    if not object.writeonly:
      # Add the get_* directory entry
          self.server.addMethods({'%s.get_%s' % \
                              (parent._path, object.name):\
                              binding})

  #
  # Call the requested method
  #
  def call(self, method, params):
    if self._loglevel>0:
      print _("Dispatching: "), method, params
    

    ## Check if the Method is part of a service or a pointer to a
    ## single object
    ##
    ## Call to an object:  method="_Management:235423456_.getAttr"
    ##                     params=""
    ## Call to an method: (of a service=one object)
    ##                     method="DonutPlace.Management.Restart"

    if method[0]=='[':
      # call to an object
      # 1. get the object from the objectlibrarian
      # 2. check, if the object is supported by the gfd
      try:
        i=string.index(method,']',1)
        objhandle=method[1:i]
        method=method[i+2:]
      except ValueError:
        tmsg = u_("Wrong format of object handle in method call %s") % method
        raise AttributeError, tmsg
      # TODO check in service dir, if obj is supported or not
      o=ObjectLibrarian.retrieveObject(objhandle)
      try:
        server_method=getattr(o,method)
        server_attribute=None
      except AttributeError:
        server_method=None
        try:
          server_attribute=getattr(o,method[4:])
        except AttributeError:
          
          if method != "_close":
            msg = u_("Internal XMLRPC server error: method %s can be "
                     "found in the directory (build out of a .grpc file), "
                     "but the object doesn't contain this method/attribut. "
                     "Please check you .grpc file for wrong return types.") \
                  % method
            
            raise AttributeError, msg
        
      if method!="_close":
        direntry = self.getMethodDirEntry(o._type+"."+method)
        signature=direntry['signature']
      else:
        signature= ('string',)                

    else:

      # call to a service method or a helper call (get/set) for
      # a service attribut
      try:
        direntry = self.getMethodDirEntry(method)
        server_method = direntry['binding']
        server_attribute = None
        
        # check if it is an real method (binding -> method)
        # or an get/set method for an attribut (binding-> attribut)
        if (type(server_method)!=type(self.call)):
          server_attribute = server_method
          server_method=None
          
        signature=direntry['signature']

        if (server_method==None) and (server_attribute==None):
          tmsg = u_("Server XML-RPC method '%s' is not bound to real method") \
                 % method
          raise AttributeError, tmsg
      except KeyError:
        tmsg = u_("Server does not have XML-RPC procedure %s") % method
        raise AttributeError, tmsg
    try:
      #
      pass
        # TODO:  Compare submitted attributs with signature
    except KeyError:
      tmsg = u_("Server XML-RPC procedure %(method)s accepts just %(attr)s "
                "as attributs") \
             % {'method': method,
                'attr'  : attr}
      raise AttributeError, tmsg
    

    # replace object handles in param with the real object
    counter=0
    while counter<len(params):
      p=params[counter]
      if type(p)==type(""):
        if (len(p)==42) and (p[0]=="[") and (p[41]=="]"):
          try:
            p=p[1:41]
            obj=ObjectLibrarian.retrieveObject(p)
            newp=params[0:counter-1]+(obj,)+params[counter+1:]
            params=newp
          except:
            pass
      counter=counter+1;

    # check if it is an real method (binding -> method)
    # or an get/set method for an attribut (binding-> attribut)

    if (server_method!=None):
            
      # call method with params
      result=server_method(*params)

    # check if it's the close object method
    elif (method=="_close"):
      
      result=self.releaseDynamicObject(o)
      
    else:
      
      ## check wether its the set or the get method for the attribut
      mparts=string.splitfields(method,'.')
      mparts.reverse()
      calltype=mparts[0]
      calltype=calltype[:4]
      GDebug.printMesg(9,'method %s has calling type %s' %\
                       (method,calltype))
      if calltype=='set_':
        # setAttribut method
        server_attribute=params[0]
      elif calltype=='get_':
        # getAttribut method
        result=server_attribute
      else:
        tmsg = u_("Internal Server XML-RPC error: method type (get/set "
                  "attribute) couldn't be detected (method %s)") % method
        raise AttributeError, tmsg
    

    # replace real object in param with an object handle
    if type(result)==type(self):  ## both should be instances
       ObjectLibrarian.archiveObject(result)

       # get the type of the result
       rtype=signature[0]
       # delete the surrounding brackets < >
       rtype=rtype[1:len(rtype)-1]
       # store typeinfo in new object
       result._type=rtype

       result=ObjectLibrarian.getObjectReference(result)
       self.registerDynamicObject(result,rtype)

    # check for empty results (not allowed for XMLRPC)
    if (result==None) or (result==[None]):
      GDebug.printMesg(9,'Transform result None into 1')
      result=1
      
    return result
