#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# $Id: language.py 7460 2005-04-22 14:53:23Z reinhard $

import string

from gnue.common.apps import errors
from gnue.common.utils.FileUtils import dyn_import

adapters = {}


# =============================================================================
# Exceptions
# =============================================================================

# -----------------------------------------------------------------------------
# Python module not available
# -----------------------------------------------------------------------------

class AdapterNotFoundError (errors.AdminError):
  """
  The language adapter for the requested language cannot be imported.
  """
  def __init__ (self, language):
    msg = u_("No adapter available for language '%s'") % language
    errors.AdminError.__init__ (self, msg)


# -----------------------------------------------------------------------------
# Abstract method not implemented
# -----------------------------------------------------------------------------

class ImplementationError (errors.SystemError):
  """
  Exception raised if an abstract method isn't implemented by a descendant.
  """
  def __init__ (self, classname, method):
    msg = u_("The class '%(class)s' has no implementation for '%(method)s'") \
           % {"class" : classname,
              "method": method}
    errors.SystemError.__init__ (self, msg)

# -----------------------------------------------------------------------------
# Code failed on compilation
# -----------------------------------------------------------------------------

class CompileError (errors.RemoteError):
  pass

# -----------------------------------------------------------------------------
# Code failed on execution
# -----------------------------------------------------------------------------

class RuntimeError (errors.RemoteError):
  pass

# -----------------------------------------------------------------------------
# Abort the current execution
# -----------------------------------------------------------------------------

class AbortRequest (errors.UserError):
  pass

# -----------------------------------------------------------------------------
# Get or create an instance of a given language adapter
# -----------------------------------------------------------------------------

def getLanguageAdapter (language):
  """
  This function returns an execution context factory for the given language.
  AdapterNotFoundError will be raised, if the language adapter cannot be
  imported.
  """
  global adapters
  lang = str(language.lower ())
  
  if not adapters.has_key (lang):
    try:
      adapter = dyn_import ('gnue.common.logic.adapters.%s' % lang)
      adapters [lang] = adapter.LanguageAdapter ()

    except ImportError:
      raise AdapterNotFoundError, language

  return adapters [lang]


# =============================================================================
# Self test code
# =============================================================================

if __name__ == '__main__':

  print "Creating language adapter for 'python' ..."
  adapter = getLanguageAdapter ('python')

  print "Creating new execution environment ..."
  environ = adapter.createNewContext ()
  environ.shortname   = "testing"
  environ.description = "Execution context for self testing"

  code = """
print "Hello World!"
print "My name is %s." % name
print foobar
return value * 2
"""

  print "Creating a new virtual code object ..."
  method = environ.buildFunction ('myFunctionName', code,
                        {'name': "", 'value': 0})

  params = {'name': 'foo', 'value': 'bar'}
  print "Calling function with: %s" % params
  res = method (**params)
  print "   result:", repr (res)

  params = {'name': 'fooBar', 'value': 4}
  print "Calling function with: %s" % params
  res = method (**params)
  print "   result:", repr (res)

  print "Thank's for playing."
